﻿using System;
using System.Collections.Generic;
using System.Text;

using System.Diagnostics;

using System.Threading;
using System.Net.Sockets;
using System.Xml.Serialization;
using System.Security.Cryptography;
using System.IO;

namespace LocalMessenger
{
    public class ContactMember
    {
        // メッセージを受け取った時のイベント
        public event ReceivedHandler received;

        private Member member;
        public Member Member
        {
            get { return member; }
            set { member = value; }
        }

        private Manager manager;
        private RijndaelManaged aes;
        private byte[] key;
        private byte[] iv;

        private TcpClient client;
        private NetworkStream ns;

        private Thread thread;
        private XmlSerializer serializer;

        /// <summary>
        /// メンバーに接続する。
        /// </summary>
        public ContactMember(Manager manager, Member member)
        {
            this.manager = manager;
//            this.member = manager.GetMember(member.Name, member.Address);
            this.member = member;
            serializer = new XmlSerializer(typeof(Packet));
            aes = new RijndaelManaged();
        }

        private void sendKey(RSAParameters publicKey)
        {
            RSACryptoServiceProvider rsa = new RSACryptoServiceProvider();
            rsa.ImportParameters(publicKey);
            key = rsa.Encrypt(aes.Key, false);  // 相手の公開鍵で暗号化した暗号鍵
            iv = rsa.Encrypt(aes.IV, false);    // 相手の公開鍵で暗号化した初期ベクタ

            byte[] size;

            size = BitConverter.GetBytes(key.Length);
            ns.Write(size, 0, size.Length);     // key のサイズの送信
            ns.Write(key, 0, key.Length);       // key 本体の送信

            size = BitConverter.GetBytes(iv.Length);
            ns.Write(size, 0, size.Length);     // iv のサイズの送信
            ns.Write(iv, 0, iv.Length);         // iv 本体の送信

            key = aes.Key;                      // 暗号化前の暗号鍵
            iv = aes.IV;                        // 暗号化前の初期ベクタ
        }

        private void recvKey()
        {
            int count = 0;

            byte[] size = BitConverter.GetBytes(count);
            byte[] buf;

            try
            {
                ns.Read(size, 0, size.Length);
                count = BitConverter.ToInt32(size, 0);
                buf = new byte[count];
                ns.Read(buf, 0, count);
                key = manager.Rsa.Decrypt(buf, false);   // AES 鍵の取得

                ns.Read(size, 0, size.Length);
                count = BitConverter.ToInt32(size, 0);
                buf = new byte[count];
                ns.Read(buf, 0, count);
                iv = manager.Rsa.Decrypt(buf, false);    //AES 初期化ベクタの取得
            }
            catch (CryptographicException e)
            {
                // 復号化できないデータを受信したら切断する。
                Debug.WriteLine(e);
                client.GetStream().Close();
                return;
            }
        }

        /// <summary>
        /// 接続し、キーを送信した後メッセージ受信を開始する。
        /// </summary>
        public void Connect()
        {
            client = new TcpClient(member.Address,
                Properties.Settings.Default.PortNumber);
            ns = client.GetStream();
            sendKey(member.PublicKey);
            thread = new Thread(listen);
            thread.Start();
        }

        /// <summary>
        /// 接続された時、キーを受信した後メッセージ受信を開始する。
        /// </summary>
        public void Connect(TcpClient client)
        {
            this.client = client;
            ns = client.GetStream();
            recvKey();
            thread = new Thread(listen);
            thread.Start();
        }

        /// <summary>
        /// メッセージ受信を停止して接続を切断する。
        /// </summary>
        public void Disconnect()
        {
            ns.Close();
            thread.Abort();
        }

        public void Send(Packet packet)
        {

        
        
        
        }

        public void sendCommand(TcpClient client, Command command, RSAParameters publicKey)
        {
            using (MemoryStream ms = new MemoryStream())
            {
                ICryptoTransform transform =
                    aes.CreateEncryptor(command.Key, command.IV);
                CryptoStream cs =
                    new CryptoStream(ms, transform, CryptoStreamMode.Write);

                serializer.Serialize(cs, command);
                cs.FlushFinalBlock();

                byte[] sendBytes = new byte[ms.Length];
                ms.Seek(0, SeekOrigin.Begin);
                int count = ms.Read(sendBytes, 0, (int)ms.Length);
                ms.Close();
                cs.Close();

                byte[] size = BitConverter.GetBytes(count);
                ns.Write(size, 0, size.Length);
                ns.Write(sendBytes, 0, count);
                ns.Flush();
            }   // stream オブジェクトを明示的に破棄
        }

        private void listen()
        {
            while (true)
            {
                int count = 0;
                byte[] size = BitConverter.GetBytes(count);
                ns.Read(size, 0, size.Length);
                count = BitConverter.ToInt32(size, 0);

                byte[] receiveBytes = new byte[count];
                int len = ns.Read(receiveBytes, 0, count);

                if ((len == 0) || (len != count))
                    Disconnect();         // おかしなパケット受信時はスレッド終了

                using (MemoryStream ms = new MemoryStream())
                {
                    ms.Write(receiveBytes, 0, count);
                    ms.Seek(0, SeekOrigin.Begin);

                    ICryptoTransform transform = aes.CreateDecryptor(aes.Key, aes.IV);
                    CryptoStream cs =
                        new CryptoStream(ms, transform, CryptoStreamMode.Read);

                    Message message = (Message)serializer.Deserialize(cs);

                    if (received != null)
                        received(message);

                    ms.Close();
                    cs.Close();
                }   // MemoryStream オブジェクトを明示的に破棄
            }
        }
    }
}
