﻿using System;
using System.Collections.Generic;
using System.Text;

using System.Diagnostics;

using System.Threading;
using System.Net.Sockets;
using System.IO;
using System.Xml.Serialization;
using System.Security.Cryptography;

namespace LocalMessenger
{
    public delegate void ReceivedHandler(Message message);

    public class ChatMember
    {
        // メッセージを受け取った時のイベント
        public event ReceivedHandler received;

        private Member member;
        public Member Member
        {
            get { return member; }
            set { member = value; }
        }

        private byte[] key;                     // 暗号化キー
        private byte[] iv;                      // 暗号化初期ベクタ

        private Thread thread;
        private TcpClient client;
        private NetworkStream ns;
        private RijndaelManaged aes;
        private XmlSerializer serializer;

        private Object lockThis = new Object();

        public ChatMember(Member member)
        {
            Member = member;
        }

        /// <summary>
        /// メッセージ受信を開始する。
        /// </summary>
        public void Start(TcpClient client, byte[] key, byte[] iv)
        {
            lock (lockThis)
            {
                this.client = client;
                this.key = key;
                this.iv = iv;

                serializer = new XmlSerializer(typeof(Message));
                aes = new RijndaelManaged();

                if (client != null)
                {
                    ns = client.GetStream();
                    thread = new Thread(listen);
                    thread.Start();
                }
            }
        }

        /// <summary>
        /// メッセージ受信を停止して接続を切断する。
        /// </summary>
        public void Stop()
        {
            lock (lockThis)
            {
                if (client != null)
                {   // 接続されていたら切断
                    if (client.Connected)
                        ns.Close();

                    client = null;
                }

                thread.Abort();
            }
        }

        private void listen()
        {
            try
            {
                while (true)
                {
                    int count = 0;
                    byte[] size = BitConverter.GetBytes(count);
                    ns.Read(size, 0, size.Length);
                    count = BitConverter.ToInt32(size, 0);
                    byte[] receiveBytes = new byte[count];
                    int len = ns.Read(receiveBytes, 0, count);

                    if ((len == 0) || (len != count))
                        break;

                    using (MemoryStream ms = new MemoryStream())
                    {
                        ms.Write(receiveBytes, 0, count);
                        ms.Seek(0, SeekOrigin.Begin);

                        ICryptoTransform transform = aes.CreateDecryptor(key, iv);
                        CryptoStream cs =
                            new CryptoStream(ms, transform, CryptoStreamMode.Read);

                        try
                        {
                            Message message = (Message)serializer.Deserialize(cs);

                            if (received != null)
                                received(message);
                        }
                        catch (System.Xml.XmlException e)
                        {
                            Debug.WriteLine(e.ToString());
                        }
                        catch (IndexOutOfRangeException e)
                        {
                            Debug.WriteLine(e.ToString());
                        }

                        ms.Close();
                        cs.Close();
                    }   // MemoryStream オブジェクトを明示的に破棄
                }
            }
            catch (IOException e)
            {
                Debug.WriteLine(e);
            }
            catch (InvalidOperationException e)
            {
                Debug.WriteLine(e);
            }
            catch (NullReferenceException e)
            {
                Debug.WriteLine(e);
            }
            finally
            {
                Stop();
            }
        }

        /// <summary>
        /// メッセージを送る
        /// </summary>
        /// <param name="message">送るメッセージ</param>
        public void SendMessage(Message message)
        {
            if (client == null)
            {
                Debug.WriteLine("SendMessage(): TCPClient is null.");
                return;
            }

            using (MemoryStream ms = new MemoryStream())
            {
                ICryptoTransform transform = aes.CreateEncryptor(key, iv);
                CryptoStream cs =
                    new CryptoStream(ms, transform, CryptoStreamMode.Write);

                serializer.Serialize(cs, message);
                cs.FlushFinalBlock();

                byte[] sendBytes = new byte[ms.Length];
                ms.Seek(0, SeekOrigin.Begin);
                int count = ms.Read(sendBytes, 0, (int)ms.Length);
                ms.Close();
                cs.Close();

                NetworkStream ns = client.GetStream();
                byte[] size = BitConverter.GetBytes(count);
                ns.Write(size, 0, size.Length);
                ns.Write(sendBytes, 0, count);
                ns.Flush();
            }   // stream オブジェクトを明示的に破棄
        }
    }
}
