/*
 * Copyright (C) 2014 kgto.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
/*
 * $Id: SearchDataRW.java 116 2015-01-14 16:20:40Z tuna_p $
 */

package webScraping.utility;

import webScraping.core.SearchData;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * 検索データ読込・保存.
 * @author kgto
 */
public class SearchDataRW {
    /* ---------------------------------------------------------------------- *
     *   フィールド
     * ---------------------------------------------------------------------- */
    private String UrlAdress;
    
    DocumentBuilder builder;
    public Document document;
    Element root;
    
    /* ---------------------------------------------------------------------- *
     *   コンストラクタ
     * ---------------------------------------------------------------------- */
    public SearchDataRW() {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            builder = factory.newDocumentBuilder();
            
        } catch (ParserConfigurationException ex) {
            Logger.getLogger(SearchDataRW.class.getName()).log(Level.SEVERE, null, ex);
        }
    }
    
    /* ---------------------------------------------------------------------- *
     *   Setter
     * ---------------------------------------------------------------------- */
    public void seturl(String UrlAdress) {
        this.UrlAdress = UrlAdress;
    }
    
    /* ---------------------------------------------------------------------- *
     *   Getter
     * ---------------------------------------------------------------------- */
    public String geturl() {
        return UrlAdress;
    }
    
    /* ---------------------------------------------------------------------- *
     *   メソッド
     * ---------------------------------------------------------------------- */
    /**
     * 保存.
     * @param file 
     */
    public void save(File file) {
        saveUrl(UrlAdress);
        saveSearchList();
        write(file);
    }
    
    /**
     * 読込.
     * @param file 
     */
    public void load(File file) {
        read(file);
        loadUrl();
        loadSearchList();
    }
    
    /* ---------------------------------------------------------------------- */
    
    void loadUrl() {
        NodeList nodelist = root.getElementsByTagName("url");
        Node node = nodelist.item(0);
        UrlAdress = node.getFirstChild().getNodeValue();
    }
    
    public void loadSearchList() {
        SearchData.clear();
        
        NodeList nodelist = root.getElementsByTagName("searchlist");
        for(int i = 0; i < nodelist.getLength(); i++) {
            Node childnode = nodelist.item(i);
            
            boolean sdatflg = false;
            SearchData sdat = new SearchData();
            for (Node child = childnode.getFirstChild(); child != null; child = child.getNextSibling()) {
                if(child.getNodeType() == Node.ELEMENT_NODE) {
                    String tag = child.getNodeName();
                    String rtn = "";
                    if(child.getFirstChild() != null) {
                        rtn = child.getFirstChild().getNodeValue();
                    }
                    switch (tag) {
                        case "item" :
                            sdat.setitem(rtn);
                            sdatflg = true;
                            break;
                        case "htmltag" :
                            sdat.setHtmltag(rtn);
                            sdatflg = true;
                            break;
                        case "htmlid" :
                            sdat.setHtmlid(rtn);
                            sdatflg = true;
                            break;
                        case "htmlclass" :
                            sdat.setHtmlclass(rtn);
                            sdatflg = true;
                            break;
                        case "around" :
                            sdat.setaround(rtn);
                            sdatflg = true;
                            break;
                        case "regexp" :
                            sdat.setregexp(rtn);
                            sdatflg = true;
                            break;
                    }
                }
            }
            if(sdatflg) SearchData.add(sdat);
        }
    }
    
    public String loadMsg404() {
        StringBuilder strbuf = new StringBuilder();
        NodeList nodelist = root.getElementsByTagName("msg404");
        for(int i = 0; i < nodelist.getLength(); i++) {
            Node childnode = nodelist.item(i);
            String str = childnode.getFirstChild().getNodeValue();
            if(strbuf.length() > 0) {
                strbuf.append("\n");
            }
            strbuf.append(str);
        }
        return strbuf.toString();
    }
    
    public Element loadElement(String elementTagName) {
        NodeList nodelist = root.getElementsByTagName(elementTagName);
        Element element = (Element)nodelist.item(0);
        
        return element;
    }
    
    /* ---------------------------------------------------------------------- */
    
    void saveUrl(String urladdress) {
        checkdoc();
        removeElement("url");    // 既にElementが存在してた場合、一度削除
        
        Element url = document.createElement("url");
        url.appendChild(document.createTextNode(urladdress));
        root.appendChild(url);
    }
    
    void saveSearchList() {
        checkdoc();
        removeElement("searchlist");    // 既にElementが存在してた場合、一度削除
        
        int count = 0;
        for(int i = 0; i < SearchData.size(); i++) {
            SearchData sdat = SearchData.get(i);
            
            Element cslist    = document.createElement("searchlist");
            cslist.setAttribute("listNo", String.valueOf(++count));
            
            addChild(cslist, "item", sdat.getitem());
            addChild(cslist, "htmltag", sdat.getHtmltag());
            addChild(cslist, "htmlid", sdat.getHtmlid());
            addChild(cslist, "htmlclass", sdat.getHtmlclass());
            addChild(cslist, "around", sdat.getaround());
            addChild(cslist, "regexp", sdat.getregexp());

            root.appendChild(cslist);
        }
    }
    
    void saveMsg404(String msg) {
        checkdoc();
        removeElement("msg404");    // 既にElementが存在してた場合、一度削除
        
        String[] msgs = msg.split("\n");
        int count = 0;
        for(String msgOne : msgs) {
            Element msgElement = document.createElement("msg404");
            msgElement.setAttribute("No", String.valueOf(++count));
            msgElement.appendChild(document.createTextNode(msgOne));
            
            root.appendChild(msgElement);
        }
    }
    
    public void saveElement(Element element) {
        checkdoc();
        removeElement(element.getTagName());    // 既にElementが存在してた場合、一度削除

        root.appendChild(element);
    }
    
    /* ---------------------------------------------------------------------- */
    
    private void addChild(Element cslist, String keyword, String data) {
        if(!data.isEmpty()) {
            Element element = document.createElement(keyword);
            element.appendChild(document.createTextNode(data));
            cslist.appendChild(element);
        }
    }
    
    private void removeElement(String elementTagName) {
        int nodeSize;
        do {
            NodeList nodelist = document.getElementsByTagName(elementTagName);
            nodeSize = nodelist.getLength();
            for(int i = 0; i < nodelist.getLength(); i++) {
                Node node = nodelist.item(i);
                root.removeChild(node);
            }
        } while(nodeSize > 0);
    }
    
    /**
     * ドキュメントチェック.
     * 新規の場合やＸＭＬファイルの読込みが行われていない状態時、新たにルートエレメントを作成する。
     * 既読の場合、ルートエレメントの取得を行う。
     */
    public void checkdoc() {
        if(document == null) {
            DOMImplementation domImpl = builder.getDOMImplementation();
            document = domImpl.createDocument("","searchdata",null);
        }
        root = document.getDocumentElement();
    }
    
    /**
     * XML読込み.
     * @param file 
     */
    public void read(File file) {
        try {
            document = builder.parse(file);
            root = document.getDocumentElement();
            
        } catch (SAXException | IOException ex) {
            Logger.getLogger(SearchDataRW.class.getName()).log(Level.SEVERE, null, ex);
        }
    }
    
    /**
     * XML書込み.
     * @param file 
     */
    public void write(File file) {
        try {
            TransformerFactory transFactory = TransformerFactory.newInstance();
            Transformer transformer = transFactory.newTransformer();
            
            DOMSource source = new DOMSource(document);
            FileOutputStream os = new FileOutputStream(file);
            StreamResult result = new StreamResult(os);
            transformer.transform(source, result);
            
        } catch (TransformerConfigurationException ex) {
            Logger.getLogger(SearchDataRW.class.getName()).log(Level.SEVERE, null, ex);
        } catch (FileNotFoundException | TransformerException ex) {
            Logger.getLogger(SearchDataRW.class.getName()).log(Level.SEVERE, null, ex);
        }
    }
    
}
