/*
 * CAN/LIN Checker for RL78/F13,F14
 *
 * Target: QB-R5F10PPJ-TB (RL78/F14, 100pin, 256KB ROM, 20KB RAM)
 * Author: Yasushi Tanaka
 *
 * [ OhCo ]
 */

#include "common.h"
#include "cpu.h"
#include "timer.h"
#include "uart.h"
#include "log.h"

/*
 * O
 * obt@TCY`
 */
#define LOG_STACK_SIZE			((u1)(0x50))
									/* X^bNɊmۂTCY */

/*
 * O
 * R[h`
 */
#define LOG_CODE_CR				((u1)(0x0d))
									/* CR */
#define LOG_CODE_LF				((u1)(0x0a))
									/* LF */
#define LOG_CODE_TERM			((u1)(0x00))
									/* I[ */

/*
 * O
 * O[oϐ
 */
static SADDR u1* log_buffer;
 									/* Oobt@ */
static SADDR u2  log_size;
									/* Oobt@̗LoCg */
static SADDR u2  log_readp;
									/* ǂݍ݃|Cg */
static SADDR u2  log_writep;
									/* ݃|Cg */
static SADDR u2  log_num;
									/* OLoCg */
static SADDR u2  log_txnum;
									/* OMoCg */
static SADDR u1  log_enabled;
									/* Oo͗L */

/*
 * O
 * 
 */
void log_init(void)
{
	/* Oobt@̃|C^ƃTCY(DTCݒ肷) */
	log_buffer = NULL;
	log_size = 0;

	/* O͗L */
	log_enabled = U1_TRUE;

	/* Oobt@ */
	log_readp = 0;
	log_writep = 0;
	log_num = 0;

	/* M̃oCg */
	log_txnum = 0;
}

/*
 * O
 * obt@ݒ
 */
void log_set_buf(u1* buf, u2 size)
{
	/* DTCʒm */
	log_buffer = buf;
	log_size = size;
}

/*
 * O
 * o͋֎~
 */
void log_disable(void)
{
	log_enabled = U1_FALSE;
}

/*
 * O
 * o͋
 */
void log_enable(void)
{
	log_enabled = U1_TRUE;
}

/*
 * O
 * o͋Ԏ擾
 */
u1 log_is_enabled(void)
{
	return log_enabled;
}

/*
 * O
 * UARTM
 *
 * 荞݋֎~ԂŌĂ΂
 */
static void log_uart_tx(void)
{
	/* log_txnum */
	if ((u2)(log_readp + log_num) > log_size)
	{
		/* obt@̏I[܂ */
		log_txnum = (u2)(log_size - log_readp);
	}
	else
	{
		/* obt@̗Lׂ */
		log_txnum = log_num;
	}

	/* 0x0080𒴂Ă΁A0x007fɉ */
	if (log_txnum >= 0x0080)
	{
		log_txnum = 0x007f;
	}

	/* 0ȊOł΁AUARToRőM */
	if (0 != log_txnum)
	{
		uart_tx_start(LOG_UART_CH, &log_buffer[log_readp], (u1)log_txnum);
	}
}

/*
 * O
 * o(^CX^vAsȂ)
 */
void log_output(u1* msg)
{
	u2  len1;
	u2  len2;
	u2  count;
	u1* ptr;
	u1  psw;

	/* bZ[W̒𐔂 */
	len1 = 0;
	ptr = msg;
	while (0x00 != *ptr++)
	{
		len1++;
	}

	/* 荞݋֎~ */
	psw = cpu_di();

	/* ݂̃OLoCglenrāAI[o[t[h */
	if ((u2)(len1 + log_num) > log_size)
	{
		len1 = (u2)(log_size - log_num);
	}

	/* obt@܂肵Alen2쐬 */
	if ((u2)(log_writep + len1) > log_size)
	{
		/* ܂ */
		len2 = (u2)((log_writep + len1) - log_size);
		len1 -= len2;
	}
	else
	{
		/* ܂Ȃ */
		len2 = 0;
	}

	/* ŏ̃Rs[ */
	ptr = &log_buffer[log_writep];
	for (count=0; count<len1; count++)
	{
		*ptr++ = *msg++;
	}

	/* log_numXV */
	log_num += len1;

	/* log_writep̉Zƃ[v */
	log_writep += len1;
	if (log_writep == log_size)
	{
		log_writep = 0;
		ptr = &log_buffer[0];
	}

	/* 2ڂ̃Rs[ */
	if (0 != len2)
	{
		for (count=0; count<len2; count++)
		{
			*ptr++ = *msg++;
		}

		/* log_numXV */
		log_num += len2;

		/* log_writep̉Z([v͕sv) */
		log_writep += len2;
	}

	/* ݃OM`FbN */
	if (0 == log_txnum)
	{
		/* M */
		log_uart_tx();
	}

	/* 荞ݕA */
	cpu_ei(psw);
}

/*
 * O
 * 10i3ݒ
 */
static void log_u2_3dec(u1* buf, u2 value)
{
	u2 div;

	/* 100̌Zbg */
	div = value / 100;
	value %= 100;
	buf[0] = (u1)(div + '0');

	/* 10̌Zbg */
	div = value / 10;
	value %= 10;
	buf[1] = (u1)(div + '0');

	/* 1̌Zbg */
	buf[2] = (u1)(value + '0');
}

/*
 * O
 * 10i4ݒ
 */
static void log_u2_4dec(u1* buf, u2 value)
{
	u2 div;

	/* 1000̌Zbg */
	div = value / 1000;
	value %= 1000;
	*buf++ = (u1)(div + '0');

	/* log_u2_3dec()Ƌ */
	log_u2_3dec(buf, value);
}

/*
 * O
 * 16i2ݒ
 */
static void log_u1_2hex(u1* buf, u1 value)
{
	u1 nibble;

	/* 4bit */
	nibble = (u1)(value >> 4);
	nibble += 0x30;
	if (nibble > 0x39)
	{
		nibble += 7;
	}
	*buf++ = nibble;

	/* 4bit */
	value &= 0x0f;
	value += 0x30;
	if (value > 0x39)
	{
		value += 7;
	}
	*buf = value;
}

/*
 * O
 * 16i4ݒ
 */
void log_u2_4hex(u1* buf, u2 value)
{
	/* ʃoCg */
	log_u1_2hex(buf, (u1)(value >> 8));

	/* ʃoCg */
	log_u1_2hex(&buf[2], (u1)(value));
}

/*
 * O
 * ݒ
 */
static u1 log_time_stamp(u1* buf)
{
	timer_info info;
	u1 ret;

	/* ߂lsec[1]=0^Cvŏ */
	/* [9999.999.999]̂悤ȃX^Cł邽߁A14 */
	ret = 14;

	/* 擪f[^ */
	*buf++ = (u1)'[';

	/* ^C}玞擾 */
	timer_getinfo(&info);

	/* sec[1]0ȊOł+4ǉ */
	if (0 != info.sec[1])
	{
		log_u2_4dec(buf, info.sec[1]);
		buf += 4;
		ret += 4;
	}

	/* sec[0] */
	log_u2_4dec(buf, info.sec[0]);
	buf += 4;

	/* sIh */
	*buf++ = (u1)'.';

	/* ms */
	log_u2_3dec(buf, info.ms);
	buf += 3;

	/* sIh */
	*buf++ = (u1)'.';

	/* us */
	log_u2_3dec(buf, info.us);
	buf += 3;

	/* I[f[^ */
	*buf = (u1)']';

	/* 񂾃oCgԂ */
	return ret;
}

/*
 * O
 * o
 */
void log_msg(char* msg)
{
	u1  buf[LOG_STACK_SIZE];
	u1  stamp;
	u1  len;
	u1* ptr;

	/* Oo͋̏ꍇ̂ */
	if (U1_TRUE == log_enabled)
	{
		/* ݒ */
		stamp = log_time_stamp(buf);

		/* ^CX^vƉsEI[coCgZo */
		len = sizeof(buf) - stamp - (2 + 1);

		/* |C^ݒ */
		ptr = &buf[stamp];

		/* I[ɒB邩Ac肪ȂȂ܂Ń[v */
		while (('\0' != *msg) && (len > 0))
		{
			/* f[^Rs[ */
			*ptr++ = (u1)*msg++;
			len--;
		}

		/* sƏI[Zbg */
		*ptr++ = LOG_CODE_CR;
		*ptr++ = LOG_CODE_LF;
		*ptr = LOG_CODE_TERM;

		/* o */
		log_output(buf);
	}
}

/*
 * O
 * o(u1l)
 */
void log_u1(char* msg, u1 value)
{
	u1  buf[LOG_STACK_SIZE];
	u1  stamp;
	u1  len;
	u1* ptr;

	/* Oo͋̏ꍇ̂ */
	if (U1_TRUE == log_enabled)
	{
		/* ݒ */
		stamp = log_time_stamp(buf);

		/* ^CX^vu1lEsEI[coCgZo */
		len = sizeof(buf) - stamp - (4 + 2 + 1);

		/* |C^ݒ */
		ptr = &buf[stamp];

		/* I[ɒB邩Ac肪ȂȂ܂Ń[v */
		while (('\0' != *msg) && (len > 0))
		{
			/* f[^Rs[ */
			*ptr++ = (u1)*msg++;
			len--;
		}

		/* u1lZbg */
		*ptr++ = (u1)'(';
		log_u1_2hex(ptr, value);
		ptr += 2;
		*ptr++ = (u1)')';

		/* sƏI[Zbg */
		*ptr++ = LOG_CODE_CR;
		*ptr++ = LOG_CODE_LF;
		*ptr = LOG_CODE_TERM;

		/* o */
		log_output(buf);
	}
}

/*
 * O
 * o(u2l)
 */
void log_u2(char* msg, u2 value)
{
	u1  buf[LOG_STACK_SIZE];
	u1  stamp;
	u1  len;
	u1* ptr;

	/* Oo͋̏ꍇ̂ */
	if (U1_TRUE == log_enabled)
	{
		/* ݒ */
		stamp = log_time_stamp(buf);

		/* ^CX^vu2lEsEI[coCgZo */
		len = sizeof(buf) - stamp - (6 + 2 + 1);

		/* |C^ݒ */
		ptr = &buf[stamp];

		/* I[ɒB邩Ac肪ȂȂ܂Ń[v */
		while (('\0' != *msg) && (len > 0))
		{
			/* f[^Rs[ */
			*ptr++ = (u1)*msg++;
			len--;
		}

		/* u2lZbg */
		*ptr++ = (u1)'(';
		log_u2_4hex(ptr, value);
		ptr += 4;
		*ptr++ = (u1)')';

		/* sƏI[Zbg */
		*ptr++ = LOG_CODE_CR;
		*ptr++ = LOG_CODE_LF;
		*ptr = LOG_CODE_TERM;

		/* o */
		log_output(buf);
	}
}

/*
 * O
 * M
 *
 * 荞݋֎~ԂŌĂ΂
 */
void log_isr(u1 ch)
{
	/* `l */
	if (LOG_UART_CH == ch)
	{
		/* OM̂ŁAreadpXV */
		log_readp += log_txnum;
		if (log_readp == log_size)
		{
			log_readp = 0;
		}

		/* OM̂ŁAnumtxnumXV */
		log_num -= log_txnum;
		log_txnum = 0;

		/* log_num΁A𑗐M */
		if (0 != log_num)
		{
			log_uart_tx();
		}
	}
}
