/*
 * bzmpd.cpp
 *
 *  Created on: 2012/03/02
 *      Author: tanaka
 */
#include <stdlib.h>
#include <stdarg.h>
#include <pthread.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <malloc.h>
#include <syslog.h>
#include <sys/stat.h>

#include "bzmpd.h"
#include "controller.h"

const static char *myname = "bzmpd";

BZMPD::BZMPD()
{
	sig_term	= 0;
	sig_hup		= 0;
	sig_int		= 0;
	abortFlag	= false;
	controller	= NULL;
	memset(&config, 0, sizeof(config));
}

BZMPD::~BZMPD()
{
	waitThreads();
	freeConf(&config);
}

bool BZMPD::init(const char *confFile)
{
	strcpy(this->confFile, confFile);
	if( !loadConf(confFile) ) return false;

	return true;
}

void BZMPD::destroy()
{
	delete this;
}

void BZMPD::log(int logLevel, const char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
	if( logLevel == LOG_DEBUG )
		if( config.logLevel > LogLevelDebug ) return;
	if( logLevel == LOG_INFO )
		if( config.logLevel > LogLevelVerbose ) return;
	if( logLevel == LOG_NOTICE )
		if( config.logLevel > LogLevelNotice ) return;
	if( logLevel == LOG_WARNING )
		if( config.logLevel > LogLevelWarning ) return;
#ifndef DAEMON
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
#else
	vsyslog(logLevel, fmt, ap);
#endif
}

bool BZMPD::mainLoop()
{
	log(LOG_INFO, "bzmpd start");
	sigset_t ss;
	sigemptyset(&ss);
	sigaddset(&ss, SIGINT);
	sigaddset(&ss, SIGHUP);
	sigaddset(&ss, SIGTERM);
	if( sigprocmask(SIG_BLOCK, &ss, NULL) != 0 ) {
		log(LOG_ERR, "sigprocmask");
		return false;
	}
	if( !startThreads() ) {
		log(LOG_ERR, "bzmpd main loop exit");
		return false;
	}
	int sn;
	struct timespec ts={5,0};
	bool doRun=true;
	while(doRun) {
		log(LOG_INFO, "watch signal");
		siginfo_t info;
		if( sigtimedwait(&ss, &info, &ts) > 0) {
			if( info.si_signo == SIGTERM ) {
				sig_term	= 1;
				log(LOG_INFO, "accept SIGTERM");
				waitThreads();
				sig_term	= 0;
				break;
			}
			if( info.si_signo == SIGINT ) {
				sig_int		= 1;
				log(LOG_INFO, "accept SIGINT");
				waitThreads();
				sig_int		= 0;
				doRun = false;
				break;
			}
			if( info.si_signo == SIGHUP ) {
				sig_hup		= 1;
				log(LOG_INFO, "accept SIGHUP");
				waitThreads();
				sig_hup		= 0;
				BZMPDCONF newconf;
				memset(&newconf, 0, sizeof(newconf));
				if( !loadConf(confFile, &newconf) ) {
					freeConf(&newconf);
					continue;
				}
				confLock.writeLock();
				freeConf(&config);
				config = newconf;
				confLock.unlock();
				if( !startThreads() ) {
					return false;
				}
				log(LOG_INFO, "bzmpd restart");
			}
		} else {
			if( isAborting() ) {
				log(LOG_INFO, "accept Abort");
				waitThreads();
				break;
			}
		}
	}
	log(LOG_INFO, "bzmpd terminate");
	return true;
}

void BZMPD::doAbort()
{
	abortFlag = true;
	log(LOG_INFO, "accept abort");
}

void BZMPD::freeConf(BZMPDCONF *conf)
{
	if( !conf ) return;
	if( conf->conf ) free(conf->conf);
	if( conf->musicFolder )	free(conf->musicFolder);
	if( conf->importFilder ) free(conf->importFilder);
	if( conf->firmPort ) free(conf->firmPort);
	if( conf->displayPort ) free(conf->displayPort);
	if( conf->mpdPort )	free(conf->mpdPort);
	if( conf->rmcPort0 ) free(conf->rmcPort0);
	if( conf->rmcPort1 ) free(conf->rmcPort1);
	if( conf->smallFont ) free(conf->smallFont);
	if( conf->mediumFont ) free(conf->mediumFont);
	if( conf->largeFont ) free(conf->largeFont);
	return;
}

bool BZMPD::loadConf(const char *confFile)
{
	strcpy(this->confFile, confFile);
	if( !loadConf(confFile, &config) ) return false;
	return true;
}

bool BZMPD::loadConf(const char *confFile, BZMPDCONF *conf)
{
	log(LOG_INFO, "loading %s", confFile);

	const char *p = confFile;
	const char *f = NULL;
	while( *p ) {
		if( *p == '/' ) f = p;
		p++;
	}

	conf->conf			= strdup(confFile);
	conf->confFolder	= strdup(confFile);
	if( f ) {
		conf->confFolder[f-confFile] = 0;
	}
	conf->logLevel		= LogLevelError;
	conf->musicFolder	= strdup(MUSIC_FOLDER);
	conf->importFilder	= strdup(IMPORT_FOLDER);
	conf->firmPort		= NULL;
	conf->displayPort	= NULL;
	conf->mpdPort		= NULL; //MPD_PORT;
	conf->rmcPort0		= NULL; // TCP_PORT;
	conf->rmcPort1		= NULL;//strdup(BT_PORT);
	conf->screenTimeout	= SCREENTIMEOUT;
	conf->smallFont		= NULL;
	conf->mediumFont	= NULL;
	conf->largeFont		= NULL;

	FILE *fp = fopen(confFile, "rt");
	if( fp == NULL ) {
		log(LOG_ERR, "bzmpd.conf can't open");
		return false;
	}
	bool hasError=false;
	int line=1;
	char buff[1024];
	char key[256];
	char value[1024];
	while( !feof(fp) ) {
		if( fgets(buff, sizeof(buff), fp) == NULL ) break;
		char *p = buff;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p == 0 || *p == '\r' || *p == '\n' || *p == '#') {
			line++;
			continue;
		}
		char *pWord = p;
		while( *p != 0 && *p != '\r' && *p != '\n' && *p != '\t' && *p != ' ' && *p != '=') p++;
		if( p-pWord > sizeof(key)-1 ) {
			log(LOG_ERR, "%s:%d keyword length is too long.", confFile, line);
			hasError = true;
			line++;
			continue;
		}
		strncpy(key, pWord, p-pWord);
		key[p-pWord] = 0;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p != '=' ) {
			log(LOG_ERR, "%s:%d syntax error.", confFile, line);
			hasError = true;
			line++;
			continue;
		}
		p++;
		while( *p == ' ' || *p == '\t' ) p++;
		char em = 0;
		if( *p == '"' ) {
			em = '"';
			p++;
		}
		pWord = p;
		while( *p != 0 ) {
			if( em ) {
				if( *p == em ) break;
			} else {
				if( *p == '\t' || *p == ' ' || *p == '\r' || *p == '\n' ) break;
			}
			p++;
		}
		if( p-pWord > sizeof(key)-1 ) {
			log(LOG_ERR, "%s:%d value length is too long.", confFile, line);
			hasError = true;
			line++;
			continue;
		}
		strncpy(value, pWord, p-pWord);
		value[p-pWord] = 0;
		if( strcmp(key, "musicFolder") == 0 ) {
			if( conf->musicFolder ) free(conf->musicFolder);
			conf->musicFolder = strdup(value);
		} else if( strcmp(key, "importFilder") == 0 ) {
			if( conf->importFilder ) free(conf->importFilder);
			conf->importFilder = strdup(value);
		} else if( strcmp(key, "firmPort") == 0 ) {
			if( conf->firmPort ) free(conf->firmPort);
			conf->firmPort	= strdup(value);
		} else if( strcmp(key, "displayPort") == 0 ) {
			if( conf->displayPort ) free(conf->displayPort);
			conf->displayPort	= strdup(value);
		} else if( strcmp(key, "mpdPort") == 0 ) {
			if( conf->mpdPort ) free(conf->mpdPort);
			conf->mpdPort	= strdup(value);
		} else if( strcmp(key, "rmcPort0") == 0 ) {
			if( conf->rmcPort0 ) free(conf->rmcPort0);
			conf->rmcPort0	= strdup(value);
		} else if( strcmp(key, "rmcPort1") == 0 ) {
			if( conf->rmcPort1 ) free(conf->rmcPort1);
			conf->rmcPort1	= strdup(value);
		} else if( strcmp(key, "screenTimeout") == 0 ) {
			conf->screenTimeout	= atoi(value);
		} else if( strcmp(key, "smallFont") == 0 ) {
			conf->smallFont	= strdup(value);
		} else if( strcmp(key, "mediumFont") == 0 ) {
			conf->mediumFont	= strdup(value);
		} else if( strcmp(key, "largeFont") == 0 ) {
			conf->largeFont		= strdup(value);
		} else if( strcmp(key, "powerOnDelay") == 0 ) {
		} else if( strcmp(key, "powerOffDelay") == 0 ) {
		} else if( strcmp(key, "keyRepeatFirst") == 0 ) {
		} else if( strcmp(key, "keyRepeatNext") == 0 ) {
		} else if( strcmp(key, "logLevel") == 0 ) {
			if( strcmp(value, "debug") == 0 )
				conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "verbose") == 0 )
				conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "notice") == 0 )
				conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "warning") == 0 )
				conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "error") == 0 )
				conf->logLevel	= LogLevelDebug;
			else {
				log(LOG_ERR, "%s:%d bad parameter. (debug, verbose, notice, warning or error)", confFile, line);
				hasError = true;
			}
		} else {
			log(LOG_ERR, "%s:%d unknown keyword error. (%s)", confFile, line, key);
			hasError = true;
			line++;
			continue;
		}
		line++;
	}
	fclose(fp);
	/*
	log(LOG_INFO, "firmPort: %s", conf->firmPort);
	log(LOG_INFO, "musicFolder: %s", conf->musicFolder);
	log(LOG_INFO, "importFilder: %s", conf->importFilder);
	log(LOG_INFO, "mpdPort: %d", conf->mpdPort);
	log(LOG_INFO, "tcpPort: %d", conf->tcpPort);
	log(LOG_INFO, "btPort: %s", conf->btPort);
	log(LOG_INFO, "screenTimeout: %d", conf->screenTimeout);
	log(LOG_INFO, "firmSpeed: %d", conf->firmSpeed);
	*/
	return (hasError ? false : true);
}

void BZMPD::freeRegionList(REGIONLIST *pList)
{
	int i;
	for( i = 0; i < pList->nRegions; i++ ) {
		if( pList->pRegions[i].regionName ) free(pList->pRegions[i].regionName);
	}
	if( pList->pRegions ) free(pList->pRegions);
	free(pList);
}

STATIONLIST * BZMPD::getFMStationLList(int regionId)
{
	char fn[1024];
	sprintf(fn, "%s/fm_l%2.2d.conf", config.confFolder, regionId);
	return getStationList(fn);
}

STATIONLIST * BZMPD::getCFMStationLList(int regionId)
{
	char fn[1024];
	sprintf(fn, "%s/cfm_l%2.2d.conf", config.confFolder, regionId);
	return getStationList(fn);
}

REGIONLIST * BZMPD::getRegionList()
{
	char fn[1024];
	sprintf(fn, "%s/region.conf", config.confFolder);
	FILE *fp = fopen(fn, "rb");
	if( fp == NULL ) {
		log(LOG_ERR, "%s can't open", fn);
		return NULL;
	}
	bool hasError=false;
	int line=1;
	int nRegions=0;
	char buff[1024];
	char key[256];
	char value[1024];
	while( !feof(fp) ) {
		if( fgets(buff, sizeof(buff), fp) == NULL ) break;
		char *p = buff;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p == 0 || *p == '\r' || *p == '\n' || *p == '#') {
			line++;
			continue;
		}
		char *pWord = p;
		while( *p != 0 && *p != '\r' && *p != '\n' && *p != '\t' && *p != ' ' && *p != '=') p++;
		if( p-pWord > sizeof(key)-1 ) {
			log(LOG_ERR, "%s:%d keyword length is too long.", fn, line);
			hasError = true;
			line++;
			continue;
		}
		strncpy(key, pWord, p-pWord);
		key[p-pWord] = 0;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p != '=' ) {
			log(LOG_ERR, "%s:%d syntax error.", fn, line);
			hasError = true;
			line++;
			continue;
		}
		p++;
		while( *p == ' ' || *p == '\t' ) p++;
		char em = 0;
		if( *p == '"' ) {
			em = '"';
			p++;
		}
		pWord = p;
		while( *p != 0 ) {
			if( em ) {
				if( *p == em ) break;
			} else {
				if( *p == '\t' || *p == ' ' || *p == '\r' || *p == '\n' ) break;
			}
			p++;
		}
		if( p-pWord > sizeof(key)-1 ) {
			log(LOG_ERR, "%s:%d value length is too long.", fn, line);
			hasError = true;
			line++;
			continue;
		}
		strncpy(value, pWord, p-pWord);
		value[p-pWord] = 0;

		if( key[0] != 'L' ||
			strlen(key) != 3 ||
			key[1] < '0' ||
			key[1] > '9' ||
			key[2] < '0' ||
			key[2] > '9' ||
			strlen(value) == 0 ) {
			hasError = true;
			log(LOG_ERR, "%s:%d key name error.", fn, line);
			line++;
			continue;
		}
		nRegions++;
		line++;
	}

	if( hasError ) {
		fclose(fp);
		return NULL;
	}

	fseek(fp, 0, SEEK_SET);

	REGIONLIST *pList = (REGIONLIST*)malloc(sizeof(REGIONLIST));
	pList->nRegions	= nRegions;
	pList->pRegions = (REGION*)malloc(sizeof(REGION)*nRegions);
	memset(pList->pRegions, 0, sizeof(REGION)*nRegions);
	REGION *pReg = pList->pRegions;

	while( !feof(fp) ) {
		if( fgets(buff, sizeof(buff), fp) == NULL ) break;
		char *p = buff;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p == 0 || *p == '\r' || *p == '\n' || *p == '#') {
			continue;
		}
		char *pWord = p;
		while( *p != 0 && *p != '\r' && *p != '\n' && *p != '\t' && *p != ' ' && *p != '=') p++;
		strncpy(key, pWord, p-pWord);
		key[p-pWord] = 0;
		while( *p == ' ' || *p == '\t' ) p++;
		p++;
		while( *p == ' ' || *p == '\t' ) p++;
		char em = 0;
		if( *p == '"' ) {
			em = '"';
			p++;
		}
		pWord = p;
		while( *p != 0 ) {
			if( em ) {
				if( *p == em ) break;
			} else {
				if( *p == '\t' || *p == ' ' || *p == '\r' || *p == '\n' ) break;
			}
			p++;
		}
		strncpy(value, pWord, p-pWord);
		value[p-pWord] = 0;
		pReg->id	= atoi(&key[1]);
		pReg->regionName	= strdup(value);
		pReg++;
	}
	fclose(fp);
	return pList;
}


STATIONLIST * BZMPD::getStationList(const char *fn)
{
	FILE *fp = fopen(fn, "rb");
	if( fp == NULL ) {
		return NULL;
	}

	int nStation = 0;

	bool hasError=false;
	int line=1;
	char buff[1024];
	char key[256];
	char value[1024];

	while( !feof(fp) ) {
		if( fgets(buff, sizeof(buff), fp) == NULL ) break;
		char *p = buff;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p == 0 || *p == '\r' || *p == '\n' || *p == '#') {
			line++;
			continue;
		}
		char *pWord = p;
		while( *p != 0 && *p != '\r' && *p != '\n' && *p != '\t' && *p != ' ' && *p != '=') p++;
		if( p-pWord > sizeof(key)-1 ) {
			log(LOG_ERR, "%s:%d keyword length is too long.", fn, line);
			hasError = true;
			line++;
			continue;
		}
		strncpy(key, pWord, p-pWord);
		key[p-pWord] = 0;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p != '=' ) {
			log(LOG_ERR, "%s:%d syntax error.", fn, line);
			hasError = true;
			line++;
			continue;
		}
		p++;
		while( *p == ' ' || *p == '\t' ) p++;
		char em = 0;
		if( *p == '"' ) {
			em = '"';
			p++;
		}
		pWord = p;
		while( *p != 0 ) {
			if( em ) {
				if( *p == em ) break;
			} else {
				if( *p == '\t' || *p == ' ' || *p == '\r' || *p == '\n' ) break;
			}
			p++;
		}
		if( p-pWord > sizeof(key)-1 ) {
			log(LOG_ERR, "%s:%d value length is too long.", fn, line);
			hasError = true;
			line++;
			continue;
		}
		if( strcmp(key, "NAME") == 0 ) {
			nStation++;
		} else if( strcmp(key, "CALL") == 0 ) {
		} else if( strcmp(key, "FREQ") == 0 ) {
		} else {
			log(LOG_ERR, "%s:%d unknown keyword error. (%s)", fn, line, key);
			hasError = true;
			line++;
			continue;
		}
		line++;
	}
	if( hasError ) {
		fclose(fp);
		return NULL;
	}

	STATIONLIST *pList = (STATIONLIST*)malloc(sizeof(STATIONLIST));
	pList->nStations	= nStation;
	pList->pStations	= (STATION*)malloc(sizeof(STATION)*nStation);
	memset(pList->pStations, 0, sizeof(STATION)*nStation);
	STATION *pPtr = pList->pStations;
	fseek(fp, 0, SEEK_SET);

	while( !feof(fp) ) {
		if( fgets(buff, sizeof(buff), fp) == NULL ) break;
		char *p = buff;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p == 0 || *p == '\r' || *p == '\n' || *p == '#') {
			line++;
			continue;
		}
		char *pWord = p;
		while( *p != 0 && *p != '\r' && *p != '\n' && *p != '\t' && *p != ' ' && *p != '=') p++;
		strncpy(key, pWord, p-pWord);
		key[p-pWord] = 0;
		while( *p == ' ' || *p == '\t' ) p++;
		p++;
		while( *p == ' ' || *p == '\t' ) p++;
		char em = 0;
		if( *p == '"' ) {
			em = '"';
			p++;
		}
		pWord = p;
		char *lw = NULL;
		while( *p != 0 ) {
			if( em ) {
				if( *p == em ) break;
			} else {
				if( *p == ' ' ) lw = p;
				if( *p == '\r' || *p == '\n' ) break;
				lw = NULL;
			}
			p++;
		}
		if( lw ) p = lw;
		strncpy(value, pWord, p-pWord);
		value[p-pWord] = 0;

		if( strcmp(key, "NAME") == 0 ) {
			if( pPtr->stationName != NULL ) {
				pPtr++;
			}
			pPtr->stationName	= strdup(value);
		} else if( strcmp(key, "CALL") == 0 ) {
			pPtr->callName	= strdup(value);
		} else if( strcmp(key, "FREQ") == 0 ) {
			char frn[256];
			int  frq=0;
			p = value;
			char *np = frn;
			while( *p && *p != ':' ) {
				*np++ = *p++;
			}
			*np = 0;
			if( *p == ':' ) {
				p++;
			} else {
				frn[0] = 0;
				p = value;
			}
			while( '0' <= *p && *p <= '9' ) {
				frq = frq * 10 + (*p - '0');
				p++;
			}
			if( pPtr->nFreq == 0 ) {
				FREQ *freq = (FREQ*)malloc(sizeof(FREQ));
				freq->freq	= frq;
				freq->freqName	= strdup(frn);
				pPtr->nFreq = 1;
				pPtr->pFreq	= freq;
			} else if( pPtr->nFreq == 1 ) {
				FREQ *pSave = pPtr->pFreq;
				pPtr->pFreq = (FREQ*)malloc(sizeof(FREQ)*4);
				memset(pPtr->pFreq, 0, sizeof(FREQ*)*4);
				pPtr->pFreq[0].freq		= pSave->freq;
				pPtr->pFreq[0].freqName	= pSave->freqName;
				pPtr->pFreq[1].freq		= frq;
				pPtr->pFreq[1].freqName	= strdup(frn);
				pPtr->nFreq	= 2;
				free(pSave);
			} else {
				int bNode = ((pPtr->nFreq / 4)+1)*4;
				if( pPtr->nFreq + 1 >= bNode) {
					bNode = bNode + 4;
					pPtr->pFreq = (FREQ*)realloc(pPtr->pFreq, sizeof(FREQ)*bNode);
				}
				pPtr->pFreq[pPtr->nFreq].freq		= frq;
				pPtr->pFreq[pPtr->nFreq].freqName	= strdup(frn);
				pPtr->nFreq++;
			}
		}
		line++;
	}
	fclose(fp);
	return pList;
}

void BZMPD::freeStationList(STATIONLIST *pList)
{
	int i;
	for(i = 0; i < pList->nStations; i++ ) {
		free(pList->pStations[i].stationName);
		int n;
		for( n = 0; n < pList->pStations[i].nFreq; n++ ) {
			if( pList->pStations[i].pFreq[n].freqName )
				free(pList->pStations[i].pFreq[n].freqName);
		}
		if( pList->pStations[i].pFreq )
			free(pList->pStations[i].pFreq);
		i++;
	}
	free(pList->pStations);
	free(pList);
}

void BZMPD::waitThreads()
{
	if( controller && controller->isRunning() ) {
		controller->stop();
	}
	delete controller;
	controller	= NULL;
}

bool BZMPD::startThreads()
{
	controller	= new Controller();
	if( !controller->initalize(this) ) {
		log(LOG_ERR, "failed initalize controller");
		return false;
	}

	if( !controller->run() ) {
		log(LOG_ERR, "failed start %s thread", controller->getThreadName());
		return false;
	}
	return true;
}

#define CONF_FOLDER		"/etc/bzmpd"
#define CONF_FILENAME	"bzmpd.conf"
#define REGION_FILENAME	"region.conf"

void dumpStations(BZMPD *bz, STATIONLIST *pList, const char *hd)
{
	int i;
	for( i = 0; i < pList->nStations; i++) {
		bz->log(LOG_INFO, "%s%s:%s", hd, pList->pStations[i].stationName, pList->pStations[i].callName);
		int n;
		for( n = 0; n < pList->pStations[i].nFreq; n++) {
			bz->log(LOG_INFO, "%s:%d", pList->pStations[i].pFreq[n].freqName,pList->pStations[i].pFreq[n].freq);
		}
	}
	bz->freeStationList(pList);
}

int main(int argc, char *argv[])
{
	char confFile[1024];
	strcpy(confFile, CONF_FOLDER "/" CONF_FILENAME);

	openlog(myname, LOG_PID, LOG_DAEMON);
	BZMPD *bzmpdPtr = new BZMPD();

#ifdef DAEMON
	pid_t pid, sid;
	pid = fork();
	if ( pid < 0 ) {
		bzmpdPtr->log(LOG_ERR, "fork");
		exit(EXIT_FAILURE);
	}
	if ( pid > 0 ) exit(EXIT_SUCCESS);
	sid = setsid();
	if ( sid < 0 ) {
		bzmpdPtr->log(LOG_ERR, "setsid");
		exit(EXIT_FAILURE);
	}
	if ( chdir("/") < 0 ) {
		bzmpdPtr->log(LOG_ERR, "chdir");
		exit(EXIT_FAILURE);
	}
	umask(0);
	close(STDIN_FILENO);
	close(STDOUT_FILENO);
	close(STDERR_FILENO);
#endif

	if( !bzmpdPtr->loadConf(confFile) ) {
		bzmpdPtr->log(LOG_ERR, "bzmpd abort");
		exit(EXIT_FAILURE);
	}
#if 0
	REGIONLIST *pList = bzmpdPtr->getRegionList();
	if( pList ) {
		int i;
		for( i = 0; i < pList->nRegions; i++) {
			bzmpdPtr->log(LOG_INFO, "=== %d:%s ===", pList->pRegions[i].id, pList->pRegions[i].regionName);
			STATIONLIST *pFM = bzmpdPtr->getFMStationLList(pList->pRegions[i].id);
			if( pFM ) dumpStations(bzmpdPtr, pFM, "■");
			STATIONLIST *pCFM = bzmpdPtr->getCFMStationLList(pList->pRegions[i].id);
			if( pCFM ) dumpStations(bzmpdPtr, pCFM, "◇");
		}
		bzmpdPtr->freeRegionList(pList);
	}
#endif
	if( !bzmpdPtr->mainLoop() ) {
		bzmpdPtr->log(LOG_ERR, "bzmpd abort");
		exit(EXIT_FAILURE);
	}
	bzmpdPtr->destroy();
	closelog();
	exit(EXIT_SUCCESS);
}



