/*
 * bzmpd.h
 *
 *  Created on: 2012/03/02
 *      Author: tanaka
 */

#ifndef BZMPD_H_
#define BZMPD_H_

#include "lock.h"
#include "port.h"
#include "device.h"
#include <csignal>

#define CONF_FOLDER		"/etc/bzmpd"
#define CONF_FILENAME	"bzmpd.conf"
#define REGION_FILENAME	"region.conf"
#define STATUS_FILENAME	"/var/lib/bzmpd/status"
#define MUSIC_FOLDER	"/var/lib/mpd/music"
#define IMPORT_FOLDER	"/media/sda"
#define FIRM_PORT		"/dev/ttyUSB0"
#define MPD_PORT		6600
#define TCP_PORT		0
#define BT_PORT			"/dev/rfcomm0"
#define SCREENTIMEOUT	0
#define FIRM_SPEED		115200

enum LogLevel {
	LogLevelDebug = 0,
	LogLevelVerbose = 1,
	LogLevelNotice = 2,
	LogLevelWarning = 3,
	LogLevelError = 4,
};

class Controller;

typedef struct _BZMPDCONF {
	char *conf;	// bzmpd.conf filename. default is confFileName.
	char *confFolder;
	LogLevel logLevel;
	int	powerOnDelay;
	int	powerOffDelay;
	int	keyRepeatFirst;
	int	keyRepeatNext;
	char *musicFolder;	// MPD music folder. default is "/var/lib/mpd/music"
	char *importFilder;	// music file import folder. SDCARD etc...
	char *firmPort;		// firmware communication serial port. default is "COM:ttyUSB0:115200"
	char *displayPort;	// firmware communication serial port. default is "COM:ttyUSB0:115200"
	char *mpdPort;		// MPD listen port. default is 6600. "TCP/C:6600"
	char *rmcPort0;		// BZMPD remocon listen port #0. "TCP/S:6601" default 0 is no listen mode.
	char *rmcPort1;		// BZMPD remocon listen port #1. "BT:rfcomm0" default 0 is no listen mode.
	int screenTimeout;	// screen timeout second, default 0 is no timeout.
	char *smallFont;
	char *mediumFont;
	char *largeFont;
} BZMPDCONF;

typedef struct _FREQ {
	char *	freqName;
	int	freq;
} FREQ;
typedef struct _STATION {
	char *	stationName;
	char *	callName;
	int		nFreq;
	FREQ *	pFreq;
} STATION;

typedef struct _STATIONLIST {
	int	nStations;
	STATION *pStations;
} STATIONLIST;

typedef struct _REGION {
	int id;
	char *regionName;
} REGION;

typedef struct _REGIONLIST {
	int	nRegions;
	REGION *pRegions;
} REGIONLIST;

typedef struct _VOLUMES {
	char *	volumeName;
	int		vol[6];
} VOLUMES;

typedef struct _SOURCE {
	char *	sourceName;
	int		sourceId;
} SOURCE;

typedef struct _BZMPDSTAT {
	int				nSources;
	SOURCE *		pSources;
	int				curentSource;

	VOLUMES			masterVolumes[3];
	int				currentVolumes;
	bool			muting;
	unsigned int	volumeStep;

	int				nStations;
	STATION			pStations;
	int				currentStation;
	unsigned int	tunneStep;

} BZMPDSTAT;

class EventListener
{
public:
	EventListener() {}
	virtual ~EventListener() {}
	virtual void onEvent(size_t dataSize, const char *data) = 0;
};

class BZMPD {
public:
	BZMPD();
	void	log(int logLevel, const char *fmt, ...);
	bool	init(const char *confFile);
	bool	loadConf(const char *confFile);
	const BZMPDCONF *getConfig() { return &config; }
	LockObject &getConfigLock() { return confLock; }
	Controller *	getController() { return controller; }
	REGIONLIST *	getRegionList();
	void			freeRegionList(REGIONLIST *pList);
	STATIONLIST *	getFMStationLList(int regionId);
	STATIONLIST *	getCFMStationLList(int regionId);
	void			freeStationList(STATIONLIST *pList);

	bool	hasSigTerm() { return sig_term != 0; }
	bool	hasSigHup() { return sig_hup != 0; }
	bool	hasSigInt() { return sig_int != 0; }
	bool	isAborting() { return abortFlag; }
	bool	hasSignal() { return hasSigTerm() || hasSigInt() || hasSigHup() || isAborting() ;}

	bool	mainLoop();
	void	doAbort();
	void	destroy();

	void freeConf(BZMPDCONF *conf);
	bool loadConf(const char *confFile, BZMPDCONF *conf);

protected:
	~BZMPD();
	STATIONLIST *	getStationList(const char *fn);

	bool	startThreads();
	void	waitThreads();

	char		confFile[512];
	LockObject	confLock;
	BZMPDCONF	config;
	Controller	*controller;

	volatile sig_atomic_t sig_term;
	volatile sig_atomic_t sig_hup;
	volatile sig_atomic_t sig_int;

	volatile bool abortFlag;

};

#endif /* BZMPD_H_ */
