/*
 * controller.cpp
 *
 *  Created on: 2012/03/09
 *      Author: tanaka
 */

#include <time.h>
#include <syslog.h>
#include <pthread.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <malloc.h>

#include "controller.h"
#include "bzmpd.h"
#include "port232c.h"
#include "portBluetooth.h"
#include "portUSB.h"
#include "portSock.h"
#include "portServerSock.h"
#include "devFirm.h"
#include "devMPD.h"
#include "devRemocon.h"

#include "screen.h"
#include "scrTopmenu.h"
#include "scrSelector.h"
#include "scrFmStation.h"
#include "scrFmFreq.h"
#include "scrArtist.h"
#include "scrAlbum.h"
#include "scrTrack.h"
#include "scrPlaylist.h"
#include "scrPlaylistTrack.h"
#include "scrMediaPlay.h"

#include "../../ssg/include/ssg.h"

Controller::Controller()
{
	kcode	= 0;

	currentMode		= MD_BOOTUP;
	doStop			= false;
	threadHandlePtr	= NULL;
	masterVolume	= 0;
	int i;
	for( i=0; i<6; i++ ) {
		deviceOffsets[i] = 0;
	}



	// fm station
	fmRegion		= 0;
	fmStationList	= NULL;

	// fm tunner
	fmFreq			= 0;
	fmTunned		= false;
	fmScanning		= 0;

	// media
	curArtist		= 0;
	curAlbum		= 0;
	curTrack		= 0;
	curPlaylist		= 0;
	mdArtist.buffers	= 0;
	mdArtist.count		= 0;
	mdArtist.metadatas	= NULL;
	mdAlbum.buffers		= 0;
	mdAlbum.count		= 0;
	mdAlbum.metadatas	= NULL;
	mdTrack.buffers		= 0;
	mdTrack.count		= 0;
	mdTrack.metagroup	= NULL;
	mdPlaylist.buffers	= 0;
	mdPlaylist.count	= 0;
	mdPlaylist.metadatas= NULL;
	cacheArtist		= false;
	cacheAlbum		= false;
	cacheTrack		= false;
	cachePlaylist	= false;

	mediaSource		= SRC_FM;
	nPorts			= 0;
	bPorts			= 0;
	pPorts			= NULL;
	firmReady		= false;

	// screen
	scrCurrent		= NULL;
	nScreen			= 0;
	bScreen			= 0;
	pScreen			= NULL;
	nScrHistory		= 0;
	bScrHistory		= 0;
	pScrHistory		= NULL;
	fontSmall		= NULL;
	fontMedium		= NULL;
	fontLarge		= NULL;
	painter			= NULL;
	screenBmp		= NULL;
}

Controller::~Controller()
{
	if( isRunning() ) stop();

	for( int i = 0; i < nPorts; i++ ) {
		pPorts[i]->close();
		delete pPorts[i];
	}
	free(pPorts);
}

Port *Controller::createPort(const char *def)
{
	// COM:name:bps
	// TCP:C:port
	// TCP:S:port
	// BT:name
	char dev[16];
	const char *p0 = def;
	char *p1 = dev;
	while(*p0 && *p0 != ':') {
		*p1++ = *p0++;
	}
	*p1 = 0;
	if( *p0 == ':' ) p0++;
	Port *port = NULL;
	if( strcmp(dev, "com") == 0 ) {
		bzmpdPtr->log(LOG_INFO, "com port = %s", def);
		port = new Port232c(bzmpdPtr);
	} else if( strcmp(dev, "tcp/s") == 0 ) {
		bzmpdPtr->log(LOG_INFO, "tcp server port = %s", def);
		port = new PortServerSock(bzmpdPtr);
	} else if( strcmp(dev, "tcp/c") == 0 ) {
		bzmpdPtr->log(LOG_INFO, "tcp client port = %s", def);
		port = new PortSock(bzmpdPtr);
	} else if( strcmp(dev, "usb") == 0 ) {
		bzmpdPtr->log(LOG_INFO, "usb port = %s", def);
		port = new PortUSB(bzmpdPtr);
	} else if( strcmp(dev, "bt") == 0 ) {
		bzmpdPtr->log(LOG_INFO, "bluetooth port = %s", def);
		port = new PortBluetooth(bzmpdPtr);
	} else {
		bzmpdPtr->log(LOG_ERR, "unknown port = %s", def);
		return NULL;
	}
	if( !port->open(p0) ) {
		delete port;
		return NULL;
	}
	return port;
}

bool Controller::initalize(BZMPD *bzmpd)
{
	bzmpdPtr		= bzmpd;
	devFirm.initalize(bzmpd);
	devRemocon.initalize(bzmpd);
	devMPD.initalize(bzmpd);

	readyListener.setController(this);
	powerDownListener.setController(this);
	panelListener.setController(this);

	const BZMPDCONF *conf = bzmpd->getConfig();
	if( conf->firmPort != NULL ) {
		Port *port = createPort(conf->firmPort);
		if( !port ) {
			return false;
		}
		if( !port->setReceiveDevice(&devFirm) ) {
			delete port;
			return false;
		}
		devFirm.setDevicePort(port);
		devFirm.setFirmReadyListener(&readyListener);
		devFirm.setPowerDownListener(&powerDownListener);
		devFirm.setSwitchListener(&panelListener);
		addConnection(port);
	}
	if( conf->displayPort != NULL ) {
		Port *port = createPort(conf->displayPort);
		if( !port ) {
			return false;
		}
		devFirm.setDisplayPort(port);
	}
	if( conf->rmcPort0 != NULL ) {
		Port *port = createPort(conf->rmcPort0);
		if( !port ) {
			return false;
		}
		if( !port->setReceiveDevice(&devRemocon) ) {
			delete port;
			return false;
		}
		addConnection(port);
	}
	if( conf->rmcPort1 != NULL ) {
		Port *port = createPort(conf->rmcPort1);
		if( !port ) {
			return false;
		}
		if( !port->setReceiveDevice(&devRemocon) ) {
			delete port;
			return false;
		}
		addConnection(port);
	}
	if( conf->mpdPort != NULL ) {
		Port *port = createPort(conf->mpdPort);
		if( !port ) {
			return false;
		}
		/*
		if( !port->setReceiveDevice(&devMPD) ) {
			delete port;
			return false;
		}
		*/
		if( !devMPD.setDevicePort(port) ) {
			delete port;
			return false;
		}
		//addConnection(port);
	}

	firmReady	= false;
	scrCurrent		= NULL;
	nScreen			= 0;
	nScrHistory		= 0;

	if( fontSmall ) delete fontSmall;
	if( fontMedium ) delete fontMedium;
	if( fontLarge ) delete fontLarge;
	fontSmall 	 = NULL;
	fontMedium 	 = NULL;
	fontLarge 	 = NULL;
	if( conf->smallFont ) {
		fontSmall = ssg::Font::loadA(conf->smallFont);
		if( fontSmall == NULL ) {
			bzmpdPtr->log(LOG_ERR, "font %s can't load", conf->smallFont);
		}
	}
	if( conf->mediumFont ) {
		fontMedium = ssg::Font::loadA(conf->mediumFont);
		if( fontMedium == NULL ) {
			bzmpdPtr->log(LOG_ERR, "font %s can't load", conf->mediumFont);
		}
	}
	if( conf->largeFont ) {
		fontLarge = ssg::Font::loadA(conf->largeFont);
		if( fontLarge == NULL ) {
			bzmpdPtr->log(LOG_ERR, "font %s can't load", conf->largeFont);
		}
	}
	if( painter == NULL ) {
		painter = new ssg::Painter();
	}
	if( screenBmp == NULL ) {
		screenBmp = ssg::Bitmap::create(256, 64, ssg::SC_HORZ);
	}
	addScreen( new ScrTopmenu(this) );
	addScreen( new ScrSelector(this) );
	addScreen( new ScrArtist(this) );
	addScreen( new ScrAlbum(this) );
	addScreen( new ScrTrack(this) );
	addScreen( new ScrFmStation(this) );
	addScreen( new ScrFmFrequency(this) );
	addScreen( new ScrPlaylist(this) );
	addScreen( new ScrPlaylistTrack(this) );
	addScreen( new ScrMediaPlay(this) );
	return true;
}

bool Controller::addScreen(Screen *scr)
{
	if( nScreen +1 >= bScreen ) {
		int bNew = bScreen + 16;
		Screen **pNew = (Screen**)realloc(pScreen, sizeof(Screen*)*bNew);
		if( pNew == NULL ) return false;
		pScreen	= pNew;
		bScreen	= bNew;
	}
	pScreen[nScreen++] = scr;
	return true;
}

bool Controller::addConnection(Port *port)
{
	if( nPorts + 1 >= bPorts ) {
		int bNew = bPorts + 8;
		Port **pNew = (Port**)realloc(pPorts, sizeof(Port*)*bNew);
		if( !pNew ) return false;
		bPorts	= bNew;
		pPorts	= pNew;
	}
	pPorts[nPorts++] = port;
	return true;
}

void Controller::removeConnection(Port *port)
{
	int i;
	for( i = 0; i < nPorts; i++ ) {
		if( pPorts[i] == port ) {
			for( int n = i; n < nPorts-1; n++ ) {
				pPorts[n] = pPorts[n+1];
			}
			break;
		}
	}
}

bool Controller::addClientConnection(Port *port)
{
	if( !addConnection(port) ) {
		return false;
	}
	port->setReceiveDevice(&devRemocon);
	return true;
}

bool Controller::run()
{
	if( threadHandlePtr != NULL ) {
		bzmpdPtr->log(LOG_ERR, "thread %s is already strated", getThreadName());
	}
	if( pthread_create( &threadHandle, NULL, threadEntry, this) != 0 ) {
		bzmpdPtr->log(LOG_ERR, "start thread failed");
		return false;
	}
	threadHandlePtr	= &threadHandle;
	doStop	= false;
	return true;
}

void *Controller::threadEntry(void *ptr)
{
	void *ret = ((Controller*)ptr)->mainLoop();
	((Controller*)ptr)->threadHandlePtr	= NULL;
	return ret;
}

bool Controller::stop()
{
	doStop	= true;
	if( threadHandlePtr ) {
		void *ret = NULL;
		pthread_join(threadHandle, &ret);
		threadHandlePtr = NULL;
	}
	return true;
}

bool Controller::isRunning()
{
	return threadHandlePtr != NULL;
}

bool Controller::dispatch()
{
	if( bzmpdPtr->hasSignal() ) {
		if( bzmpdPtr->hasSigTerm() ) {
			onEvent(EC_POWEROFF, 0);
		}
		return false;
	}
	if( doStop ) return false;

	int i;
	int maxfd = 0;
	fd_set rfds;
	FD_ZERO(&rfds);
	for( i = 0; i < nPorts; i++ ) {
		int fd = pPorts[i]->getDesciptor();
		FD_SET(fd, &rfds);
		if( fd > maxfd ) maxfd = fd;
	}
	struct timeval tv;
	tv.tv_sec	= 0;
	tv.tv_usec	= 500000;
	if( kcode ) tv.tv_usec = 50000;
	int retval = select(maxfd+1, &rfds, NULL, NULL, &tv);
	if( retval > 0 ) {
		bzmpdPtr->log(LOG_INFO, "select %d", retval);
		for( i = 0; i < nPorts; i++ ) {
			if( FD_ISSET(pPorts[i]->getDesciptor(), &rfds) ) {
				if( !pPorts[i]->process() ) {
					bzmpdPtr->log(LOG_ERR, "process error %s", i);
				}
			}
		}
	}
	onTimerProc();
	return true;
}

ssg::Font *Controller::getSmallFont()
{
	if( fontSmall ) return fontSmall;
	if( fontMedium ) return fontMedium;
	if( fontLarge ) return fontLarge;
	return NULL;
}

ssg::Font *Controller::getMediumFont()
{
	if( fontMedium ) return fontMedium;
	if( fontSmall ) return fontSmall;
	if( fontLarge ) return fontLarge;
	return NULL;
}

ssg::Font *Controller::getLargeFont()
{
	if( fontLarge ) return fontLarge;
	if( fontMedium ) return fontMedium;
	if( fontSmall ) return fontSmall;
	return NULL;
}

ssg::Bitmap *Controller::getScreenBitmap()
{
	return screenBmp;
}

ssg::Painter *Controller::getPainter()
{
	return painter;
}

void *Controller::mainLoop()
{
	bzmpdPtr->log(LOG_INFO, "Controller start");
	currentMode	= MD_NORMAL;
	devFirm.clearDisplay();
	goNoHistory(SC_TOPMENU);
	if( scrCurrent && scrCurrent->needPaint() ) {
		scrCurrent->doPaint();
	}
	while(!doStop) {
		if( !dispatch() ) break;
	}
	return NULL;
}

// station
REGIONLIST *Controller::getFmRegionList()
{
	return bzmpdPtr->getRegionList();
}

int Controller::getFmRegion()
{
	return fmRegion;
}

bool Controller::setFmRegion(int region)
{
	STATIONLIST *list = bzmpdPtr->getFMStationLList(region);
	if( list == NULL ) return false;
	if( fmStationList ) bzmpdPtr->freeStationList(fmStationList);
	fmRegion		= region;
	fmStationList	= list;
	return true;
}

STATIONLIST *Controller::getFmStationList()
{
	return fmStationList;
}

STATION *Controller::findFmStation(int freq)
{
	int f;
	for( f = 0; f < fmStationList->nStations; f++ ) {
		int n;
		for( n = 0; n < fmStationList->pStations[f].nFreq; n++ ) {
			if( fmStationList->pStations[f].pFreq[n].freq == freq ) {
				return &fmStationList->pStations[f];
			}
		}
	}
	return NULL;
}
// music source
bool Controller::cmdSetMediaSource(SourceType st)
{
	bzmpdPtr->log(LOG_DEBUG, "cmdSetMediaSource(%d)", st);

	if( st != SRC_FM && st != SRC_MP3 ) {
		bzmpdPtr->log(LOG_INFO, "cmdSetMediaSource: Bad Channel");
		return false;
	}
	if( mediaSource == st ) {
		bzmpdPtr->log(LOG_INFO, "cmdSetMediaSource: Channel is not changed");
	} else {
		mediaSource = st;
		devFirm.setMediaSource(mediaSource);
	}
	return true;
}

// fm
bool Controller::cmdSetFmFrequency(int frequency)
{
	if( !devFirm.setFreq(frequency) ) return false;
	return true;
}

int Controller::cmdGetFmFrequency()
{
	int freq=0;
	if( !devFirm.getFreq(freq) ) return 0;
	return freq;
}

bool Controller::cmdGetFmStatus(TUNESTATUS &st)
{
	if( !devFirm.getTuneStatus(st) ) return false;
	return true;
}

bool Controller::cmdScanFmStatus(int direction)
{
	if( direction < 0 ) {
		if( !devFirm.searchLower(5) ) return false;
	} else {
		if( !devFirm.searchUpper(5) ) return false;
	}
	return true;
}

// media artist
const MPD_METADATA_LIST	*Controller::cmdListArtist()
{
	bzmpdPtr->log(LOG_INFO, "cmdListArtist");
	if( !cacheArtist ) {
		bzmpdPtr->log(LOG_INFO, "cmdListArtist1");
		if( !devMPD.dbList("artist", NULL, mdArtist) ) return NULL;
		devMPD.sortMetaDataList(mdArtist, false);
		bzmpdPtr->log(LOG_INFO, "cmdListArtist2");

		curArtist	= 0;
		cacheArtist = true;
		if( cacheAlbum ) {
			bzmpdPtr->log(LOG_INFO, "cmdListArtist3");
			devMPD.clearMetaDataList(mdAlbum);
		}
		bzmpdPtr->log(LOG_INFO, "cmdListArtist4");
		cacheAlbum	= false;
		curAlbum	= 0;
		if( cacheTrack ) {
			bzmpdPtr->log(LOG_INFO, "cmdListArtist5");
			devMPD.clearMetaGroupList(mdTrack);
		}
		bzmpdPtr->log(LOG_INFO, "cmdListArtist6");
		cacheTrack	= false;
		curTrack	= 0;
	}
	bzmpdPtr->log(LOG_INFO, "cmdListArtist");
	return &mdArtist;
}

bool Controller::cmdSetArtist(int id)
{
	if( !cacheArtist ) {
		return false;
	}
	if( id < 0 || mdArtist.count <= id ) {
		return false;
	}
	curArtist	= id;
	if( cacheAlbum ) {
		devMPD.clearMetaDataList(mdAlbum);
	}
	cacheAlbum	= false;
	curAlbum	= 0;
	if( cacheTrack ) {
		devMPD.clearMetaGroupList(mdTrack);
	}
	cacheTrack	= false;
	curTrack	= 0;
	return true;
}

int Controller::cmdGetArtist()
{
	return curArtist;
}
const char *Controller::cmdGetArtistName()
{
	if( !cacheArtist ||
		curArtist < 0 || mdArtist.count <= curArtist ) {
		return "- unknown -";
	}
	return mdArtist.metadatas[curArtist].value;
}

bool Controller::cmdPlayArtist()
{
	if( !cacheArtist ) {
		return false;
	}
	if( curArtist < 0 || mdArtist.count <= curArtist ) {
		return false;
	}
	if( !devMPD.pbStop() ) return false;
	if( !devMPD.plClear() ) return false;
	if( !cacheAlbum ) {
		if( !cmdListAlbum() ) return false;
	}
	int ai;
	for( ai = 0; ai < mdAlbum.count; ai++ ) {
		if( !cmdSetAlbum(ai) ) return false;
		if( !cmdListTrack() ) return false;
		int ti;
		for( ti = 0; ti < mdTrack.count; ti++ ) {
			cmdAddPlaylistTrack(mdTrack.metagroup[ti]);
		}
	}
	if( !devMPD.pbPlayId(0) ) return false;
	return true;
}

// media album
const MPD_METADATA_LIST	*Controller::cmdListAlbum()
{
	if( !cacheArtist ) {
		return NULL;
	}
	if( curArtist < 0 || mdArtist.count <= curArtist ) {
		return NULL;
	}
	if( !cacheAlbum ) {
		char query[1024];
		const char *artist = mdArtist.metadatas[curArtist].value;
		sprintf(query, "artist \"%s\"", artist);
		if( !devMPD.dbList("album", query, mdAlbum) ) return NULL;
		devMPD.sortMetaDataList(mdAlbum, false);

		curAlbum	= 0;
		cacheAlbum	= true;
		if( cacheTrack ) {
			devMPD.clearMetaGroupList(mdTrack);
		}
		cacheTrack	= false;
		curTrack	= 0;
	}
	return &mdAlbum;
}

bool Controller::cmdSetAlbum(int id)
{
	if( !cacheArtist ) {
		return false;
	}
	if( curArtist < 0 || mdArtist.count <= curArtist ) {
		return false;
	}
	if( !cacheAlbum ) {
		return false;
	}
	if( id < 0 || mdAlbum.count <= id ) {
		return false;
	}
	curAlbum	= id;
	if( cacheTrack ) {
		devMPD.clearMetaGroupList(mdTrack);
	}
	cacheTrack	= false;
	curTrack	= 0;
	return true;
}
int Controller::cmdGetAlbum()
{
	return curAlbum;
}

const char *Controller::cmdGetAlbumName()
{
	if( !cacheArtist ||
		curArtist < 0 || mdArtist.count <= curArtist ||
		!cacheAlbum ||
		curAlbum < 0 || mdAlbum.count <= curAlbum ) {
		return "- unknown -";
	}
	return mdAlbum.metadatas[curAlbum].value;
}

bool Controller::cmdPlayAlbum()
{
	if( !cacheArtist ) {
		return false;
	}
	if( curArtist < 0 || mdArtist.count <= curArtist ) {
		return false;
	}
	if( !cacheAlbum ) {
		return false;
	}
	if( curAlbum < 0 || mdAlbum.count <= curAlbum ) {
		return false;
	}
	if( !devMPD.pbStop() ) return false;
	if( !devMPD.plClear() ) return false;
	int ai;
	for( ai = 0; ai < mdAlbum.count; ai++ ) {
		if( !cmdSetAlbum(ai) ) return false;
		if( !cmdListTrack() ) return false;
		int ti;
		for( ti = 0; ti < mdTrack.count; ti++ ) {
			cmdAddPlaylistTrack(mdTrack.metagroup[ti]);
		}
	}
	if( !devMPD.pbPlay(0) ) return false;
	return true;
}
// media track
const MPD_METAGROUP_LIST *Controller::cmdListTrack()
{
	if( !cacheArtist ) {
		return false;
	}
	if( curArtist < 0 || mdArtist.count <= curArtist ) {
		return false;
	}
	if( !cacheAlbum ) {
		return false;
	}
	if( curAlbum < 0 || mdAlbum.count <= curAlbum ) {
		return false;
	}
	if( !cacheTrack ) {
		const char *artist = mdArtist.metadatas[curArtist].value;
		const char *album = mdAlbum.metadatas[curAlbum].value;
		char query[1024];
		sprintf(query, "artist \"%s\" album \"%s\"", artist, album);
		if( !devMPD.dbFind(query, mdTrack) ) return NULL;
		devMPD.sortMetaGroupList(mdTrack, devMPD.getTagId("Track"), true);
		curTrack	= 0;
		cacheTrack	= true;
	}
	return &mdTrack;
}
bool Controller::cmdSetTrack(int id)
{
	if( !cacheArtist ) {
		return false;
	}
	if( curArtist < 0 || mdArtist.count <= curArtist ) {
		return false;
	}
	if( !cacheAlbum ) {
		return false;
	}
	if( curAlbum < 0 || mdAlbum.count <= curAlbum ) {
		return false;
	}
	if( !cacheTrack ) {
		return false;
	}
	if( id < 0 || mdTrack.count <= id ) {
		return false;
	}
	curTrack	= id;
	return true;
}

int Controller::cmdGetTrack()
{
	return curTrack;
}

const char *Controller::cmdGetTrackName()
{
	if( !cacheArtist ||
		curArtist < 0 || mdArtist.count <= curArtist ||
		!cacheAlbum ||
		curAlbum < 0 || mdAlbum.count <= curAlbum ||
		!cacheTrack ||
		curTrack < 0 || mdTrack.count <= curTrack ) {
		return "- unknown -";
	}
	int id = devMPD.getTagId("title");
	int n = devMPD.findMetaData(*mdTrack.metagroup[curTrack], id);
	if( n == -1 )
		return "- unknown -";
	return mdTrack.metagroup[curTrack]->metadatas[n].value;
}

bool Controller::cmdPlayTrack()
{
	if( !cacheArtist ) {
		return false;
	}
	if( curArtist < 0 || mdArtist.count <= curArtist ) {
		return false;
	}
	if( !cacheAlbum ) {
		return false;
	}
	if( curAlbum < 0 || mdAlbum.count <= curAlbum ) {
		return false;
	}
	if( !cacheTrack ) {
		return false;
	}
	if( curTrack < 0 || mdTrack.count <= curTrack ) {
		return false;
	}
	if( !devMPD.pbStop() ) return false;
	if( !devMPD.plClear() ) return false;
	int ai;
	for( ai = 0; ai < mdAlbum.count; ai++ ) {
		if( !cmdSetAlbum(ai) ) return false;
		if( !cmdListTrack() ) return false;
		int ti;
		for( ti = 0; ti < mdTrack.count; ti++ ) {
			cmdAddPlaylistTrack(mdTrack.metagroup[ti]);
		}
	}
	if( !devMPD.pbPlay(curTrack) ) return false;
	return true;
}

// playlist
const MPD_METADATA_LIST * Controller::cmdListPlaylist()
{
	if( !cachePlaylist ) {
		devMPD.clearMetaDataList(mdPlaylist);
		MPD_METAGROUP_LIST m;
		m.buffers	= 0;
		m.count		= 0;
		m.metagroup	= NULL;
		if( !devMPD.dbListInfo(NULL, m) ) return false;
		int tagId = devMPD.getTagId("playlist");
		int i;
		for( i = 0; i < m.count; i++ ) {
			int n;
			for( n = 0; n < m.metagroup[i]->count; n++) {
				if( m.metagroup[i]->metadatas[n].type == tagId ) {
					devMPD.addMetaData(mdPlaylist, tagId, m.metagroup[i]->metadatas[n].value);
				}
			}
		}
		curPlaylist		= 0;
		cachePlaylist	= true;
	}
	return &mdPlaylist;
}

int Controller::cmdGetPlaylist()
{
	return curPlaylist;
}

bool Controller::cmdSetPlaylist(int id)
{
	if( !cachePlaylist ) {
		return false;
	}
	if( id < 0 || mdPlaylist.count <= id ) {
		return false;
	}
	curPlaylist	= id;
	return true;
}

bool Controller::cmdLoadPlaylist(const char *name)
{
	return devMPD.plLoad(name);
}

bool Controller::cmdDeletePlaylist(const char *name)
{
	return devMPD.plRemove(name);
}

bool Controller::cmdSavePlaylist(const char *name)
{
	return devMPD.plSave(name);
}
bool Controller::cmdRenamePlaylist(const char *oldName, const char *newName)
{
	return devMPD.plRename(oldName, newName);
}

bool Controller::cmdClearPlaylistTrack()
{
	return devMPD.plClear();
}
bool Controller::cmdAddPlaylistTrack(MPD_METADATA_LIST *track)
{
	int tagId = devMPD.getTagId("file");
	int n = devMPD.findMetaData(*track, tagId);
	if( n < 0 ) return false;
	if( !devMPD.plAdd(track->metadatas[n].value) ) return false;
	return true;
}

bool Controller::cmdRemovePlaylistTrack(int n)
{
	return devMPD.plDelete(n);
}

const MPD_METAGROUP_LIST *Controller::cmdListPlaylistTrack()
{
	return NULL;
//	devMPD.clearMetaGroupList(list);
//	return devMPD.plPlaylistInfo(-1, list);
}

int Controller::cmdGetPlaylistTrack()
{
	return 0;
//	devMPD.clearMetaDataList(list);
//	return devMPD.plCurrentSong(list);
}

bool Controller::cmdSetPlaylistTrack(int n)
{
	return devMPD.pbPlay(n);
}

// playback control
bool Controller::cmdPlay()
{
	return devMPD.pbPlay(0);
}

bool Controller::cmdPause()
{
	return devMPD.pbPause(true);
}

bool Controller::cmdStop()
{
	return devMPD.pbStop();
}

bool Controller::cmdNext()
{
	return devMPD.pbNext();
}

bool Controller::cmdPrev()
{
	return devMPD.pbPrevious();
}

void Controller::onEvent(EventCode ec, int data)
{
	EventCode ec2=EC_NONE;
	int data2 = 0;

	switch(ec) {
	case EC_POWERON:
		currentMode	= MD_NORMAL;
		cmdInitDisplay();
		break;
	case EC_POWEROFF:
		if( currentMode != MD_SHUTDOWN ) {
			currentMode = MD_SHUTDOWN;
			devFirm.powerOff();
			system("poweroff");
		}
		break;
	case EC_POWERDOWN:
		if( currentMode != MD_SHUTDOWN ) {
			currentMode = MD_SHUTDOWN;
		}
		system("poweroff");
		break;
	case EC_TIMER:
		break;
	case EC_BUTTONDOWN:
		printf("EC_BUTTONDOWN %d\n", data);
		if( data == KC_SELECTSOURCE ) {
			if( mediaSource == SRC_FM )
				cmdSetMediaSource(SRC_MP3);
			else
				cmdSetMediaSource(SRC_FM);
			break;
		}
		if( data == KC_PLAY ) {
			cmdPlay();
			break;
		}
		if( data == KC_PAUSE ) {
			cmdPause();
			break;
		}
		if( data == KC_STOP ) {
			cmdStop();
			break;
		}
		break;
	case EC_BUTTONUP:
		printf("EC_BYTTONUP %d\n", data);
		break;
	case EC_KEY:
		printf("EC_KEY %d\n", data);
		if( data == KC_VOLUMEDOWN ) {
			cmdVolumeMinus();
			break;
		}
		if( data == KC_VOLUMEUP) {
			cmdVolumePlus();
			break;
		}
		break;
	}
	if( ec2 != EC_NONE ) {
		onEvent(ec2, data2);
	}
	if( scrCurrent && scrCurrent->needPaint() ) {
		scrCurrent->doPaint();
	}
}

bool Controller::goScreen(int id)
{
	Screen *pNext = NULL;
	int i;
	for( i = 0; i < nScreen; i++ ) {
		if( pScreen[i]->getId() == id ) {
			pNext	= pScreen[i];
			break;
		}
	}
	if( pNext == NULL ) return false;

	if( scrCurrent ) {
		scrCurrent->onInactivate();
		if( nScrHistory+1 >= bScrHistory ) {
			int bNew = bScrHistory + 4;
			Screen **pNew = (Screen**)realloc(pScrHistory, sizeof(Screen*)*bNew);
			if( pNew == NULL ) return false;
			pScrHistory	= pNew;
			bScrHistory	= bNew;
		}
		pScrHistory[nScrHistory++] = scrCurrent;
	}
	scrCurrent	= pNext;
	scrCurrent->onCreate();
	scrCurrent->onActivate();
	return true;
}

bool Controller::goNoHistory(int id)
{
	Screen *pNext = NULL;
	int i;
	for( i = 0; i < nScreen; i++ ) {
		if( pScreen[i]->getId() == id ) {
			pNext	= pScreen[i];
			break;
		}
	}
	if( pNext == NULL ) return false;

	if( scrCurrent ) {
		scrCurrent->onInactivate();
	}
	scrCurrent	= pNext;
	scrCurrent->onCreate();
	scrCurrent->onActivate();
	return true;
}

bool Controller::gobackScreen()
{
	Screen *pPrev = NULL;
	if( nScrHistory == 0 ) {
		return false;
	}
	pPrev = pScrHistory[--nScrHistory];
	if( scrCurrent ) {
		scrCurrent->onInactivate();
		scrCurrent->onDestory();
	}
	scrCurrent	= pPrev;
	scrCurrent->onActivate();
	return true;
}

bool Controller::cmdInitDisplay()
{
	devFirm.clearDisplay();
	goScreen(0);
	return true;
}

bool Controller::cmdVolumePlus()
{
	if( masterVolume < 80 ) {
		masterVolume++;
		int vol[6];
		int i;
		for( i = 0; i < 6; i++ ) {
			if( deviceOffsets[i] + masterVolume < 80 )
				vol[i] += masterVolume;
			else
				vol[i] = 80;
		}
		// send firmware
		{
			devFirm.setVolume(vol);
		}
	}
	return true;
}

bool Controller::cmdVolumeMinus()
{
	if( masterVolume > 0 ) {
		masterVolume--;
		int vol[6];
		int i;
		for( i = 0; i < 6; i++ ) {
			if( deviceOffsets[i] + masterVolume < 80 )
				vol[i] += masterVolume;
			else
				vol[i] = 80;
		}
		// send firmware
		{
			devFirm.setVolume(vol);
		}
	}
	return true;
}

void Controller::onTimerProc()
{
	if( scrCurrent )
		scrCurrent->onEvent(EC_TIMER, 0);
	else
		onEvent(EC_TIMER, 0);

	if( kcode ) {
		struct timespec ts;
		clock_gettime(CLOCK_REALTIME, &ts);
		if( rWait.tv_sec < ts.tv_sec || (rWait.tv_sec == ts.tv_sec && rWait.tv_nsec <= ts.tv_nsec) ) {
			if( scrCurrent )
				scrCurrent->onEvent(EC_KEY, kcode);
			else
				onEvent(EC_KEY, kcode);

			if( rWait.tv_nsec >= 900000000 ) {
				rWait.tv_sec 	+= 1;
				rWait.tv_nsec	-= 900000000;
			} else {
				rWait.tv_nsec	+= 100000000;
			}
		}
	}
}

void Controller::onPowerOnProc()
{
	if( scrCurrent )
		scrCurrent->onEvent(EC_POWERON, 0);
	else
		onEvent(EC_POWERON,0);
}

void Controller::onPowerOffProc()
{
	if( scrCurrent )
		scrCurrent->onEvent(EC_POWEROFF, 0);
	else
		onEvent(EC_POWEROFF, 0);
}

void Controller::onFirmReadyProc()
{
	if( scrCurrent )
		scrCurrent->onEvent(EC_FIRMREADY, 0);
	else
		onEvent(EC_FIRMREADY, 0);
}

void Controller::onPowerDownProc()
{
	if( scrCurrent )
		scrCurrent->onEvent(EC_POWERDOWN, 0);
	else
		onEvent(EC_POWERDOWN, 0);
}

void Controller::onButtonDownProc(int data)
{
	bzmpdPtr->log(LOG_INFO, "Controller::onButtonDownProc d=%d", data);
	if( scrCurrent )
		scrCurrent->onEvent(EC_BUTTONDOWN, data);
	else
		onEvent(EC_BUTTONDOWN, data);

	kcode	= data;
	clock_gettime(CLOCK_REALTIME, &rWait);
	if( rWait.tv_nsec >= 800000000 ) {
		rWait.tv_sec 	+= 1;
		rWait.tv_nsec	-= 800000000;
	} else {
		rWait.tv_nsec	+= 200000000;
	}
	if( scrCurrent )
		scrCurrent->onEvent(EC_KEY, kcode);
	else
		onEvent(EC_KEY, kcode);
}

void Controller::onButtonUpProc(int data)
{
	kcode	= 0;
	bzmpdPtr->log(LOG_INFO, "Controller::onButtonUpProc d=%d", data);
	if( scrCurrent )
		scrCurrent->onEvent(EC_BUTTONUP, data);
	else
		onEvent(EC_BUTTONUP, data);
}

void Controller::onRotateProc(int data)
{
	kcode	= 0;

	if( scrCurrent )
		scrCurrent->onEvent(EC_ROTATE, data);
	else
		onEvent(EC_ROTATE, data);
}
