/*
 * controller.h
 *
 *  Created on: 2012/03/09
 *      Author: tanaka
 */

#ifndef CONTROLLER_H_
#define CONTROLLER_H_
#include <pthread.h>
#include <string.h>
#include <syslog.h>

class BZMPD;
#include "bzmpd.h"
#include "devFirm.h"
#include "devRemocon.h"
#include "devMPD.h"

class Screen;
namespace ssg {
	class Font;
	class Bitmap;
	class Painter;
}

enum SystemMode {
	MD_BOOTUP=0,
	MD_SHUTDOWN,
	MD_SLEEP,
	MD_NORMAL,
	MD_ERROR,
};

enum EventCode {
	EC_NONE=0,
	EC_POWERON=1,
	EC_POWEROFF=2,
	EC_POWERDOWN=3,
	EC_FIRMREADY=4,
	EC_BUTTONDOWN=5,
	EC_BUTTONUP=6,
	EC_KEY=7,
	EC_ROTATE=8,
	EC_TIMER=9,
};

enum KeyCode {
	KC_SELECTSOURCE = 1,
	KC_VOLUMEDOWN = 2,
	KC_VOLUMEUP = 3,
	KC_STOP = 4,
	KC_PLAY = 5,
	KC_MENU = 6,
	KC_ESC = 7,
	KC_ENTER = 8,
	KC_PAUSE = 9,
	EC_PLAYSTATUS = 10,
};

enum SourceType {
	SRC_FM 	= 0,
	SRC_MP3	= 1,
};

class Controller
{
public:
	Controller();
	~Controller();

	void *mainLoop();
	const char *getThreadName() { return "Controller"; }
	bool run();
	bool stop();
	bool isRunning();
	pthread_t *getThreadHandle();

	static void *threadEntry(void *);

	Port *createPort(const char *def);
	bool initalize(BZMPD *bzmpd);
	bool addClientConnection(Port *port);
	bool addConnection(Port *port);
	void removeConnection(Port *port);

	BZMPD *		getBZMPD() { return bzmpdPtr; }
	DevMPD *	getMPD() { return &devMPD; }
	DevFirm *	getFirm() { return &devFirm; }

	// screen control
	bool goScreen(int id);
	bool goNoHistory(int id);
	bool gobackScreen();
	bool dispatch();

	// graphics
	ssg::Font *getSmallFont();
	ssg::Font *getMediumFont();
	ssg::Font *getLargeFont();
	ssg::Bitmap *getScreenBitmap();
	ssg::Painter *getPainter();

	// fm station
	REGIONLIST *getFmRegionList();
	int			getFmRegion();
	bool		setFmRegion(int region);
	STATIONLIST *getFmStationList();
	STATION *	findFmStation(int freq);

	// music source
	bool	cmdSetMediaSource(SourceType st);

	// fm tunner
	bool	cmdSetFmFrequency(int frequency);
	int		cmdGetFmFrequency();
	bool	cmdGetFmStatus(TUNESTATUS &st);
	bool	cmdScanFmStatus(int direction);

	// media artist
	const MPD_METADATA_LIST	*cmdListArtist();
	bool	cmdSetArtist(int id);
	int		cmdGetArtist();
	const char *cmdGetArtistName();
	bool	cmdPlayArtist();

	// media album
	const MPD_METADATA_LIST	*cmdListAlbum();
	bool	cmdSetAlbum(int id);
	int		cmdGetAlbum();
	const char *cmdGetAlbumName();
	bool	cmdPlayAlbum();

	// media track
	const MPD_METAGROUP_LIST *cmdListTrack();
	bool	cmdSetTrack(int id);
	int		cmdGetTrack();
	const char *cmdGetTrackName();
	bool	cmdPlayTrack();

	// playlist
	const MPD_METADATA_LIST	*cmdListPlaylist();
	int		cmdGetPlaylist();
	bool	cmdSetPlaylist(int id);

	const MPD_METAGROUP_LIST * cmdListPlaylistTrack();
	int		cmdGetPlaylistTrack();
	bool	cmdSetPlaylistTrack(int id);

	bool	cmdLoadPlaylist(const char *name);
	bool	cmdDeletePlaylist(const char *name);
	bool	cmdSavePlaylist(const char *name);
	bool	cmdRenamePlaylist(const char *oldName, const char *newName);

	bool	cmdClearPlaylistTrack();
	bool	cmdAddPlaylistTrack(MPD_METADATA_LIST *track);
	bool	cmdRemovePlaylistTrack(int n);

	// playback control
	bool	cmdPlay();
	bool	cmdPause();
	bool	cmdStop();
	bool	cmdNext();
	bool	cmdPrev();

	//
	bool cmdInitDisplay();
	bool cmdVolumePlus();
	bool cmdVolumeMinus();

	void onTimerProc();
	void onPowerOnProc();
	void onPowerOffProc();
	void onPowerDownProc();
	void onFirmReadyProc();
	void onButtonDownProc(int data);
	void onButtonUpProc(int data);
	void onRotateProc(int data);

	void onEvent(EventCode ec, int data);

	class ReadyListner : public EventListener {
	public:
		ReadyListner() {
		}
		void setController(Controller *ctrl) {
			controller = ctrl;
		}
		virtual void onEvent(size_t dataSize, const char *data) {
			controller->onFirmReadyProc();
		}
		virtual ~ReadyListner() {}
	protected:
		Controller *controller;
	};

	class PowerDownListner : public EventListener {
	public:
		PowerDownListner() {
		}
		void setController(Controller *ctrl) {
			controller = ctrl;
		}
		virtual void onEvent(size_t dataSize, const char *data) {
			controller->onPowerDownProc();
		}
		virtual ~PowerDownListner() {}
	protected:
		Controller *controller;
	};

	class SwitchPanelListner : public EventListener {
	public:
		SwitchPanelListner() {
		}
		void setController(Controller *ctrl) {
			controller = ctrl;
		}
		virtual void onEvent(size_t dataSize, const char *data) {
			controller->getBZMPD()->log(LOG_INFO, "SwitchPanelListner::onEvent len=%d data=%s", dataSize, data);
			if( dataSize == 1) {
				if( strcmp(data,"L") == 0 )
					return controller->onRotateProc(-1);
				if( strcmp(data,"R") == 0 )
					return controller->onRotateProc(1);
			}

			if( dataSize == 2 ) {
				if( strcmp(data,"RL") == 0 )
					return controller->onRotateProc(-1);
				if( strcmp(data,"RR") == 0 )
					return controller->onRotateProc(1);
				if( strcmp(data,"D1") == 0 )
					return controller->onButtonDownProc(KC_SELECTSOURCE);
				if( strcmp(data,"D2") == 0 )
					return controller->onButtonDownProc(KC_VOLUMEDOWN);
				if( strcmp(data,"D3") == 0 )
					return controller->onButtonDownProc(KC_VOLUMEUP);
				if( strcmp(data,"D4") == 0 )
					return controller->onButtonDownProc(KC_PLAY);
				if( strcmp(data,"D5") == 0 )
					return controller->onButtonDownProc(KC_STOP);
				if( strcmp(data,"D6") == 0 )
					return controller->onButtonDownProc(KC_MENU);
				if( strcmp(data,"D7") == 0 )
					return controller->onButtonDownProc(KC_ENTER);

				if( strcmp(data,"U1") == 0 )
					return controller->onButtonUpProc(KC_SELECTSOURCE);
				if( strcmp(data,"U2") == 0 )
					return controller->onButtonUpProc(KC_VOLUMEDOWN);
				if( strcmp(data,"U3") == 0 )
					return controller->onButtonUpProc(KC_VOLUMEUP);
				if( strcmp(data,"U4") == 0 )
					return controller->onButtonUpProc(KC_PLAY);
				if( strcmp(data,"U5") == 0 )
					return controller->onButtonUpProc(KC_STOP);
				if( strcmp(data,"U6") == 0 )
					return controller->onButtonUpProc(KC_MENU);
				if( strcmp(data,"U7") == 0 )
					return controller->onButtonUpProc(KC_ENTER);
			}
			controller->getBZMPD()->log(LOG_INFO, "SwitchPanelListner::onEvent unknown");
		}

		virtual ~SwitchPanelListner() {}
	protected:
		Controller *controller;
	};

protected:
	bool	addScreen(Screen *scr);

	BZMPD *		bzmpdPtr;
	bool		doStop;
	pthread_t 	threadHandle;
	pthread_t 	*threadHandlePtr;
	SystemMode	currentMode;
	bool		firmReady;

	int			kcode;
	struct timespec		rWait;

	// screen
	Screen *	scrCurrent;
	int			nScreen;
	int			bScreen;
	Screen **	pScreen;
	int			nScrHistory;
	int			bScrHistory;
	Screen **	pScrHistory;

	// graphics
	ssg::Font	*fontSmall;
	ssg::Font	*fontMedium;
	ssg::Font	*fontLarge;
	ssg::Painter *painter;
	ssg::Bitmap *screenBmp;

	ReadyListner		readyListener;
	PowerDownListner	powerDownListener;
	SwitchPanelListner	panelListener;

	int		masterVolume;
	int		deviceOffsets[6];

	// fm station
	int		fmRegion;
	STATIONLIST *fmStationList;

	// fm tunner
	int		fmFreq;
	bool	fmTunned;
	int		fmScanning;

	// media info
	bool	cacheArtist;
	int		curArtist;
	MPD_METADATA_LIST	mdArtist;
	bool	cacheAlbum;
	int		curAlbum;
	MPD_METADATA_LIST	mdAlbum;
	bool	cacheTrack;
	int		curTrack;
	MPD_METAGROUP_LIST	mdTrack;

	bool	cachePlaylist;
	int		curPlaylist;
	MPD_METADATA_LIST mdPlaylist;
	//
	SourceType	mediaSource;

	// volume
	int		volumes[6];
	int		mediaVolumes[6];

	DevMPD		devMPD;
	DevFirm		devFirm;
	DevRemocon	devRemocon;

	int		bPorts;
	int		nPorts;
	Port **	pPorts;
};

#endif /* CONTROLLER_H_ */
