/*
 * devMPD.cpp
 *
 *  Created on: 2012/03/11
 *      Author: tanaka
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include "bzmpd.h"
#include "devMPD.h"
#include "port.h"

DevMPD::DevMPD()
{
	tagList.buffers	= 0;
	tagList.count	= 0;
	tagList.tags	= NULL;
}
DevMPD::~DevMPD()
{
	clearTagType(tagList);
}

bool DevMPD::setDevicePort(Port *port)
{
	char data[256];
	size_t rs;
	if( !port->recvData(256, data, rs) ) {
		bzmpdPtr->log(LOG_ERR, "mpd sock can't recive");
		return false;
	}
	bzmpdPtr->log(LOG_INFO, "mpd response = %s", data);
	if( rs < 7 ) {
		bzmpdPtr->log(LOG_ERR, "mpd conection error");
		return false;
	}
	if( strncmp(data, "OK MPD", 6) != 0 ) {
		bzmpdPtr->log(LOG_ERR, "mpd conection error");
		return false;
	}
	strcpy(mpd_version, &data[7]);
	if( !Device::setDevicePort(port) ) return false;
	if( !adTagtypes(tagList) ) {
		bzmpdPtr->log(LOG_ERR, "mpd tagtype error");
		return false;
	}
	if( !addTagType(tagList, "Id") ||
		!addTagType(tagList, "Time") ||
		!addTagType(tagList, "Last-Modified") ||
		!addTagType(tagList, "file") ) {
		return false;
	}
	return true;
}

bool DevMPD::process(Port *port)
{
	char buff[256];
	size_t rs;
	port->recvData(256, buff, rs);
	bzmpdPtr->log(LOG_WARNING, "MPD Recive %s", buff);
	return true;
}

bool DevMPD::connect(Port *port)
{
	char res[256];
	sprintf(res, "OK MPD %s\n", mpd_version);
	return port->sendData(strlen(res), res);
}

bool DevMPD::sendCmd(const char *cmd)
{
	return sendCmd(cmd, strlen(cmd));
}

bool DevMPD::sendCmd(const char *cmd, size_t len)
{
	bzmpdPtr->log(LOG_INFO, "sendCmd 1 %s", cmd);
	if( devicePort == NULL ) {
		bzmpdPtr->log(LOG_INFO, "sendCmd 2 %s", cmd);
		return false;
	}
	if( !devicePort->sendData(len, cmd) ) {
		bzmpdPtr->log(LOG_INFO, "sendCmd 3 %s", cmd);
		return false;
	}
	bzmpdPtr->log(LOG_INFO, "sendCmd 4 %s", cmd);
	return true;
}

bool DevMPD::recvRes(size_t buffSize, char *buff, size_t &readed)
{
	if( devicePort == NULL ) {
		bzmpdPtr->log(LOG_ERR, "DevMPD::recvRes: devicePort is null");
		return false;
	}
	if( !devicePort->recvData(buffSize, buff, readed) ) {
		bzmpdPtr->log(LOG_ERR, "DevMPD::recvRes: devicePort read error");
		return false;
	}
	bzmpdPtr->log(LOG_INFO, "DevMPD::recvRes: readed=%d %s", readed, buff);
	return true;

}

bool DevMPD::recvResult()
{
	char b[256];
	size_t rs;
	if( !recvRes(sizeof(b), b, rs) ) return false;
	if( strcmp(b, "OK") != 0 ) return false;
	return true;
}

bool DevMPD::recvMetaData(MPD_METADATA_LIST &t)
{
	clearMetaDataList(t);
	char buff[2048];
	size_t rs;
	while(1) {
		if( !recvRes(sizeof(buff), buff, rs) ) return false;
		if( rs == 0 ) {
			bzmpdPtr->log(LOG_ERR, "DevMPD::recvMetaData: read error");
			return false;
		}
		if( strcmp(buff,"OK") == 0 ){
			break;
		} else if( strncmp(buff,"ACK ", 4) == 0 ){
			bzmpdPtr->log(LOG_ERR, "DevMPD::recvMetaData: %s", buff);
			return false;
		} else {
			if( !addMetaData(t, buff) ) return false;
		}
	}
	return true;
}

bool DevMPD::recvMetaGroupData(MPD_METAGROUP_LIST &metalist)
{
	clearMetaGroupList(metalist);
	MPD_METADATA_LIST *list = (MPD_METADATA_LIST *)malloc(sizeof(MPD_METADATA_LIST));
	list->buffers	= 0;
	list->count		= 0;
	list->metadatas	= NULL;
	char buff[2048];
	size_t rs;
	while(1) {
		if( !recvRes(sizeof(buff), buff, rs) ) return false;
		if( rs == 0 ) {
			return false;
		}
		if( strcmp(buff,"OK") == 0 ){
			break;
		} else if( strncmp(buff,"ACK ", 4) == 0 ){
			return false;
		} else {
			char *v = buff;
			while( *v == ' ' ) v++;
			char *k = v;
			while( *v && *v != ':' ) v++;
			if( *v == ':' ) {
				*v++ = 0;
			}
			while( *v == ' ' ) v++;
			char *d = v;

			int tagId = getTagId(k);
			int find = findMetaData(*list, tagId);
			if( find >= 0 ) {
				addMetaGroup(metalist, list);
				list = (MPD_METADATA_LIST *)malloc(sizeof(MPD_METADATA_LIST));
				list->buffers	= 0;
				list->count		= 0;
				list->metadatas	= NULL;
			}
			if( !addMetaData(*list, tagId, d) ) return false;
		}
	}
	if( list->count != 0 ) {
		addMetaGroup(metalist, list);
	} else {
		free(list);
	}
	return true;
}

bool DevMPD::addTagType(MPD_TAG_LIST &t, const char *v)
{
	if( t.count+1 >= t.buffers ) {
		int nb = t.buffers + 20;
		MPD_TAG *np = (MPD_TAG*)realloc(t.tags, nb*sizeof(MPD_TAG));
		if( np == NULL ) return false;
		t.buffers	= nb;
		t.tags		= np;
	}
	t.tags[t.count++].type = strdup(v);
	return true;
}

void DevMPD::clearTagType(MPD_TAG_LIST &t)
{
	int i;
	for( i = 0; i < t.count; i++ ) {
		free(t.tags[i].type);
	}
	if( t.buffers ) {
		free(t.tags);
	}
	t.count		= 0;
	t.buffers	= 0;
	t.tags		= NULL;
}

bool DevMPD::addUrlHandler(MPD_URLHANDLER_LIST &t, const char *v)
{
	if( t.count+1 >= t.buffers ) {
		int nb = t.buffers + 20;
		MPD_URLHANDLER *np = (MPD_URLHANDLER*)realloc(t.handlers, nb*sizeof(MPD_URLHANDLER));
		if( np == NULL ) return false;
		t.buffers	= nb;
		t.handlers		= np;
	}
	t.handlers[t.count++].url	= strdup(v);
	return true;
}

void DevMPD::clearUrlHandler(MPD_URLHANDLER_LIST &t)
{
	int i;
	for( i = 0; i < t.count; i++ ) {
		free(t.handlers[i].url);
	}
	if( t.buffers ) {
		free(t.handlers);
	}
	t.count		= 0;
	t.buffers	= 0;
	t.handlers	= NULL;
}

int	DevMPD::findMetaData(MPD_METADATA_LIST &t, int tagId)
{
	int i;
	for( i = 0; i < t.count; i++ ) {
		if( t.metadatas[i].type == tagId ) return i;
	}
	return -1;
}

bool DevMPD::addMetaData(MPD_METADATA_LIST &t, int tagId, const char *value)
{
	if( t.count+1 >= t.buffers ) {
		int nb = t.buffers + 20;
		MPD_METADATA *np = (MPD_METADATA*)realloc(t.metadatas, nb*sizeof(MPD_METADATA));
		if( np == NULL ) {
			bzmpdPtr->log(LOG_ERR, "DevMPD::addMetaData: memory error");
			return false;
		}
		t.buffers	= nb;
		t.metadatas	= np;
	}
	t.metadatas[t.count].type	= tagId;
	t.metadatas[t.count].value	= strdup(value);
	t.count++;
	return true;
}

bool DevMPD::addMetaData(MPD_METADATA_LIST &t, const char *v)
{
	const char *k;
	size_t	kn = 0;
	while( *v == ' ' ) v++;
	k = v;
	while( *v && *v != ':' ) v++;
	kn = v-k;
	if( *v == ':' ) v++;

	int tt;
	for( tt = 0; tt < tagList.count; tt++ ) {
		if( strncmp(k, tagList.tags[tt].type, kn) == 0 && tagList.tags[tt].type[kn] == 0 ) {
			break;
		}
	}
	if( tt >= tagList.count ) {
		char *tag = (char*)malloc(kn+1);
		strncpy(tag, k, kn);
		tag[kn] = 0;
		addTagType(tagList, tag);
		free(tag);
	}

	const char *d;
	size_t	dn = 0;
	while( *v == ' ' ) v++;
	d = v;
	while( *v ) v++;
	dn = v-k;
	if( kn == 0 ) return false;

	if( t.count+1 >= t.buffers ) {
		int nb = t.buffers + 20;
		MPD_METADATA *np = (MPD_METADATA*)realloc(t.metadatas, nb*sizeof(MPD_METADATA));
		if( np == NULL ) return false;
		t.buffers	= nb;
		t.metadatas	= np;
	}
	t.metadatas[t.count].type	= tt;
	t.metadatas[t.count].value	= (char *)malloc(dn+1);
	strncpy(t.metadatas[t.count].value, d, dn);
	t.metadatas[t.count].value[dn]	= 0;
	t.count++;
	return true;
}

void DevMPD::clearMetaDataList(MPD_METADATA_LIST &t)
{
	int i;
	for( i = 0; i < t.count; i++ ) {
		free(t.metadatas[i].value);
	}
	if( t.buffers ) {
		free(t.metadatas);
	}
	t.count		= 0;
	t.buffers	= 0;
	t.metadatas	= NULL;
}
static int sortKey;
static bool sortNumericCompaire;

static int metadataCmp(const void *p0, const void *p1)
{
	MPD_METADATA *m0 = (MPD_METADATA*)p0;
	MPD_METADATA *m1 = (MPD_METADATA*)p1;
	if( sortNumericCompaire  ) {
		int v0 = atoi(m0->value);
		int v1 = atoi(m1->value);
		if( v0 < v1 ) return -1;
		if( v0 > v1 ) return 1;
		return 0;
	} else {
		return strcmp(m0->value, m1->value);
	}
}

void DevMPD::sortMetaDataList(MPD_METADATA_LIST &t, bool numericCompaire)
{
	sortNumericCompaire = numericCompaire;
	qsort(t.metadatas, t.count, sizeof(MPD_METADATA), metadataCmp);
}

bool DevMPD::addMetaGroup(MPD_METAGROUP_LIST &t, MPD_METADATA_LIST *v)
{
	if( t.count+1 >= t.buffers ) {
		int nb = t.buffers + 20;
		MPD_METADATA_LIST **np = (MPD_METADATA_LIST**)realloc(t.metagroup, nb*sizeof(MPD_METADATA_LIST*));
		if( np == NULL ) return false;
		t.buffers	= nb;
		t.metagroup	= np;
	}
	t.metagroup[t.count++]	= v;
	return true;
}

void DevMPD::clearMetaGroupList(MPD_METAGROUP_LIST &t)
{
	int i;
	for( i = 0; i < t.count; i++ ) {
		clearMetaDataList(*t.metagroup[i]);
		free(t.metagroup[i]);
	}
	if( t.buffers ) {
		free(t.metagroup);
	}
	t.count		= 0;
	t.buffers	= 0;
	t.metagroup	= NULL;
}

static int metagroupCmp(const void *p0, const void *p1)
{
	MPD_METADATA_LIST *m0 = *(MPD_METADATA_LIST**)p0;
	MPD_METADATA_LIST *m1 = *(MPD_METADATA_LIST**)p1;

	MPD_METADATA *k0 = NULL;
	MPD_METADATA *k1 = NULL;
	int i;
	for( i = 0; i < m0->count; i++) {
		if( m0->metadatas[i].type == sortKey) {
			k0 = &m0->metadatas[i];
			break;
		}
	}
	for( i = 0; i < m1->count; i++) {
		if( m1->metadatas[i].type == sortKey) {
			k1 = &m1->metadatas[i];
			break;
		}
	}
	if( k0 && k1 ) {
		if( sortNumericCompaire ) {
			int v0 = atoi(k0->value);
			int v1 = atoi(k1->value);
			if( v0 < v1 ) return -1;
			if( v0 > v1 ) return 1;
			return 0;
		} else {
			return strcmp(k0->value, k1->value);
		}
	}
	if( k0 ) return -1;
	if( k1 ) return 1;
	return 0;
}

void DevMPD::sortMetaGroupList(MPD_METAGROUP_LIST &t, int tagId, bool numericCompaire)
{
	sortKey = tagId;
	sortNumericCompaire = numericCompaire;
	qsort(t.metagroup, t.count, sizeof(MPD_METADATA_LIST*), metagroupCmp);
}

char * DevMPD::seekValue(const char *key, char *ptr)
{
	char *p = ptr;
	while( *key == *p ) {
		key++; p++;
	}
	if( *p != ':' ) return NULL;
	p++;
	while( *p == ' ' ) p++;
	if( *p == 0 ) return NULL;
	return p;
}

bool DevMPD::isConnected()
{
	return devicePort && devicePort->isConnect();
}

Port * DevMPD::getPort()
{
	return devicePort;
}

int DevMPD::getTagId(const char *tagName)
{
	int tt;
	for( tt = 0; tt < tagList.count; tt++) {
		if( strcmp(tagName, tagList.tags[tt].type) == 0 ) {
			return tt;
		}
	}
	addTagType(tagList, tagName);
	return tt;
}

// ex command
bool DevMPD::exSend(const char *sd, size_t sd_size)
{
	if( !isConnected() ) return false;
	return sendCmd(sd, sd_size);
}

bool DevMPD::exRecv(char *rd, size_t rd_size, size_t &readed)
{
	readed = 0;
	size_t	rds=0;
	while(1) {
		if( !recvRes(rd_size, &rd[readed], rds) ) return false;
		if( rds == 0 ) break;
		if( strcmp(&rd[readed],"OK") == 0 ){
			break;
		} else if( strncmp(&rd[readed],"ACK ", 4) == 0 ){
			break;
		}
		readed	+= rds;
		rd_size	-= rds;
		if( rd_size <= 1 ) break;
	}
	return true;
}

// admin command
bool DevMPD::adConnect()
{
	if( isConnected() ) {
		return false;
	}
	if( devicePort == NULL ) {
		return false;
	}
	const BZMPDCONF *cf = bzmpdPtr->getConfig();
	if( !devicePort->open(cf->mpdPort) ) {
		return false;
	}
	return true;
}

void DevMPD::adDisconnect()
{
	if( devicePort ) {
		devicePort->close();
	}
}

bool DevMPD::adDisableOutput()
{
	if( !sendCmd("disableoutput\n") ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::adEnableOutput()
{
	if( !sendCmd("enableoutput\n") ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::adKill()
{
	if( !sendCmd("kill\n") ) return false;
	return true;
}

bool DevMPD::adUpdate(const char *path, int &jobId)
{
	jobId = 0;
	char cmd[2048];
	strcpy(cmd, "update");
	if( path ) {
		if( strlen(path) > 2040 ) return false;
		strcat(cmd, " ");
		strcat(cmd, path);
	}
	strcat(cmd, "\n");
	if( !sendCmd(cmd) ) return false;
	size_t rs;
	if( !recvRes(sizeof(cmd), cmd, rs) ) return false;
	char *v;
	if( (v = seekValue("updating_db:", cmd)) == NULL ) {
		return false;
	}
	jobId	= atoi(v);
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::adStatus(MPD_STATUS &st)
{
	if( !sendCmd("status\n") ) return false;
	char buff[2048];
	size_t rs;
	char *p;
	while(1) {
		if( !recvRes(sizeof(buff), buff, rs) ) return false;
		if( rs == 0 ) break;
		if( (p = seekValue("volume", buff)) != NULL  ) {
			st.volume	= atoi(p);
		} else if( (p = seekValue("repeat", buff)) != NULL ) {
			st.repeat	= atoi(p) == 0 ? false :  true;
		} else if( (p = seekValue("random", buff)) != NULL ) {
			st.random	= atoi(p) == 0 ? false :  true;
		} else if( (p = seekValue("playlist", buff)) != NULL ) {
			st.playlist 	= atoi(p);
		} else if( ( p = seekValue("playlistlength", buff)) != NULL ) {
			st.playlistlength 	= atoi(p);
		} else if( (p = seekValue("xfade", buff)) != NULL ) {
			st.xfade	= atoi(p);
		} else if( (p = seekValue("state", buff)) != NULL ) {
			if( strncmp(p, "play", 4) == 0 ) st.state	= ST_PLAY;
			else if( strncmp(p, "stop", 4) == 0 ) st.state	= ST_STOP;
			else if( strncmp(p, "pause", 5) == 0 ) st.state	= ST_PAUSE;
			else st.state = ST_UNKNOWN;
		} else if( (p = seekValue("song", buff)) != NULL ) {
			st.song	= atoi(p);
		} else if( (p = seekValue("songid", buff)) != NULL ) {
			st.songId	= atoi(p);
		} else if( (p = seekValue("elapsed_time", buff)) != NULL ) {
			st.elapsedTime	= atoi(p);
		} else if( (p = seekValue("total_time", buff)) != NULL ) {
			st.totalTime = atoi(p);
		} else if( (p = seekValue("bitrate", buff)) != NULL ) {
			st.bitRate	= atoi(p);
		} else if( (p = seekValue("sampleRate", buff)) != NULL ) {
			st.sampleRate	= atoi(p);
		} else if( (p = seekValue("bits", buff)) != NULL ) {
			st.bits	= atoi(p);
		} else if( (p = seekValue("channels", buff)) != NULL ) {
			st.channels	= atoi(p);
		} else if( (p = seekValue("updating_db", buff)) != NULL ) {
			st.updatingDb	= atoi(p);
		} else if( (p = seekValue("error", buff)) != NULL ) {
			st.error	= strdup(p);
		} else if( (p = seekValue("nextsong", buff)) != NULL ) {
			st.nextSong	= atoi(p);
		} else if( (p = seekValue("nextsongid", buff)) != NULL ) {
			st.nextSongId	= atoi(p);
		} else if( strcmp(buff,"OK") == 0 ){
			break;
		} else if( strncmp(buff,"ACK ", 4) == 0 ){
			return false;
		} else {
			return false;
		}
	}
	return true;
}

bool DevMPD::adOutputs(MPD_OUTPUTS &o)
{
	if( !sendCmd("outputs\n") ) return false;
	if( !recvResult() ) return false;
	char buff[2048];
	size_t rs;
	char *p;
	while(1) {
		if( !recvRes(sizeof(buff), buff, rs) ) return false;
		if( rs == 0 ) break;
		if( (p = seekValue("outputid", buff)) != NULL ) {
			o.outputId	= atoi(p);
		} else if( (p = seekValue("outputname", buff)) != NULL ) {
			o.outputName	= strdup(p);
		} else if( (p = seekValue("outputenabled", buff)) != NULL ) {
			o.outputEnabled	= atoi(p) == 0 ? false :  true;
		} else if( strcmp(buff,"OK") == 0 ){
			break;
		} else if( strncmp(buff,"ACK ", 4) == 0 ){
			return false;
		} else {
			return false;
		}
	}
	return true;
}

bool DevMPD::adTagtypes(MPD_TAG_LIST &t)
{
	clearTagType(t);
	if( !sendCmd("tagtypes\n") ) return false;
//	if( !recvResult() ) return false;
	char buff[2048];
	size_t rs;
	char *p;
	while(1) {
		if( !recvRes(sizeof(buff), buff, rs) ) return false;
		if( rs == 0 ) break;
		if( (p = seekValue("tagtype", buff)) != NULL ) {
			if( !addTagType(t, p) ) return false;
		} else if( strcmp(buff,"OK") == 0 ){
			break;
		} else if( strncmp(buff,"ACK ", 4) == 0 ){
			return false;
		} else {
			return false;
		}
	}
	return true;
}

bool DevMPD::adUrlHandlers(MPD_URLHANDLER_LIST &uhl)
{
	if( !sendCmd("urlhandlers\n") ) return false;
//	if( !recvResult() ) return false;
	clearUrlHandler(uhl);
	char buff[2048];
	size_t rs;
	char *p;
	while(1) {
		if( !recvRes(sizeof(buff), buff, rs) ) return false;
		if( rs == 0 ) break;
		if( (p = seekValue("handler", buff)) != NULL ) {
			if( !addUrlHandler(uhl, p) ) return false;
		} else if( strcmp(buff,"OK") == 0 ){
			break;
		} else if( strncmp(buff,"ACK ", 4) == 0 ){
			return false;
		} else {
			return false;
		}
	}
	return true;
}


// database command
bool DevMPD::dbFind(const char *query, MPD_METAGROUP_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "find %s\n", query);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaGroupData(t) ) return false;
	return true;
}

bool DevMPD::dbList(const char *tag, const char *query, MPD_METADATA_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "list %s", tag);
	if( query ) {
		strcat(cmd, " ");
		strcat(cmd, query);
	}
	strcat(cmd, "\n");
	bzmpdPtr->log(LOG_INFO, "DevMPD::dbList %s", cmd);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaData(t) ) {
		return false;
	}
	return true;
}

bool DevMPD::dbListAll(const char *path, MPD_METADATA_LIST &t)
{
	char cmd[2048];
	strcpy(cmd, "listall");
	if( path ) {
		strcat(cmd, " \"");
		strcat(cmd, path);
		strcat(cmd, "\"");
	}
	strcat(cmd, "\n");

	if( !sendCmd(cmd) ) return false;
	if( !recvMetaData(t) ) return false;
	return true;
}

bool DevMPD::dbListAllInfo(const char *path, MPD_METAGROUP_LIST &t)
{
	char cmd[2048];
	strcpy(cmd, "listallinfo");
	if( path ) {
		strcat(cmd, " \"");
		strcat(cmd, path);
		strcat(cmd, "\"");
	}
	strcat(cmd, "\n");

	if( !sendCmd(cmd) ) return false;
	if( !recvMetaGroupData(t) ) return false;
	return true;
}

bool DevMPD::dbListInfo(const char *path, MPD_METAGROUP_LIST &t)
{
	char cmd[2048];
	strcpy(cmd, "lsinfo");
	if( path ) {
		strcat(cmd, " \"");
		strcat(cmd, path);
		strcat(cmd, "\"");
	}
	strcat(cmd, "\n");

	if( !sendCmd(cmd) ) return false;
	if( !recvMetaGroupData(t) ) return false;
	return true;
}

bool DevMPD::dbSearch(const char *query, MPD_METAGROUP_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "search %s\n", query);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaGroupData(t) ) return false;
	return true;
}

bool DevMPD::dbCount(const char *query, MPD_METADATA_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "count %s\n", query);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaData(t) ) return false;
	return true;
}

// playlist command
bool DevMPD::plAdd(const char *song)
{
	char cmd[2048];
	sprintf(cmd, "add \"%s\"\n", song);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plAddId(const char *song, int &plid)
{
	char cmd[2048];
	sprintf(cmd, "addid \"%s\"\n", song);
	if( !sendCmd(cmd) ) return false;
	MPD_METADATA_LIST md;
	md.buffers	= 0;
	md.count	= 0;
	md.metadatas= NULL;
	if( !recvMetaData(md) ) return false;
	if( md.count != 1 || getTagId("Id") != md.metadatas->type ) {
		return false;
	}
	plid = atoi(md.metadatas[0].value);
	clearMetaDataList(md);
	return true;
}

bool DevMPD::plClear()
{
	if( !sendCmd("clear\n") ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plCurrentSong(MPD_METADATA_LIST &t)
{
	if( !sendCmd("currentsong\n") ) return false;
	clearMetaDataList(t);
	if( !recvMetaData(t) ) return false;
	return true;
}

bool DevMPD::plDelete(int song)
{
	char cmd[2048];
	sprintf(cmd, "delete \"%d\"\n", song);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plDeleteId(int songId)
{
	char cmd[2048];
	sprintf(cmd, "deleteid \"%d\"\n", songId);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plLoad(const char *name)
{
	char cmd[2048];
	sprintf(cmd, "load \"%s\"\n", name);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plRename(const char *oldName, const char *newName)
{
	char cmd[2048];
	sprintf(cmd, "rename \"%s\" \"%s\"\n", oldName, newName);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plMove(int fromNum, int toNum)
{
	char cmd[2048];
	sprintf(cmd, "move \"%d\" \"%d\"\n", fromNum, toNum);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plMoveId(int fromId, int toId)
{
	char cmd[2048];
	sprintf(cmd, "moveid \"%d\" \"%d\"\n", fromId, toId);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plPlaylistInfo(int song, MPD_METAGROUP_LIST &t)
{
	char cmd[2048];
	if( song >= 0 ) {
		sprintf(cmd, "playlistinfo %d\n", song);
	} else {
		strcpy(cmd, "playlistinfo\n");
	}
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaGroupData(t) ) return false;
	return true;
}

bool DevMPD::plPlaylistId(int songId, MPD_METAGROUP_LIST &t)
{
	char cmd[2048];
	if( songId >= 0 ) {
		sprintf(cmd, "playlistid %d\n", songId);
	} else {
		strcpy(cmd, "playlistid\n");
	}
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaGroupData(t) ) return false;
	return true;
}

bool DevMPD::plChanges(int ver, MPD_METADATA_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "plchanges %d\n", ver);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaData(t) ) return false;
	return true;
}

bool DevMPD::plChangesPositionId(int ver, MPD_METADATA_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "plchangesposid %d\n", ver);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaData(t) ) return false;
	return true;
}

bool DevMPD::plRemove(const char *name)
{
	char cmd[2048];
	sprintf(cmd, "rm \"%s\"\n", name);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plSave(const char *name)
{
	char cmd[2048];
	sprintf(cmd, "save \"%s\"\n", name);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plShuffle()
{
	if( !sendCmd("shuffle\n") ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plSwap(int song1, int song2)
{
	char cmd[2048];
	sprintf(cmd, "swap %d %d\n", song1, song2);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plSwapId(int songId1, int songId2)
{
	char cmd[2048];
	sprintf(cmd, "swapid %d %d\n", songId1, songId2);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plListPlaylist(const char *name, MPD_METADATA_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "listplaylist \"%s\"\n", name);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaData(t) ) return false;
	return true;
}

bool DevMPD::plListPlaylistInfo(const char *name, MPD_METADATA_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "listplaylistinfo \"%s\"\n", name);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaData(t) ) return false;
	return true;
}

bool DevMPD::plPlaylistAdd(const char *name, const char *path)
{
	char cmd[2048];
	sprintf(cmd, "playlistadd \"%s\" \"%s\"\n", name, path);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plPlaylistClear(const char *name)
{
	char cmd[2048];
	sprintf(cmd, "playlistclear \"%s\"\n", name);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plPlaylistDelete(const char *name, int songId)
{
	char cmd[2048];
	sprintf(cmd, "playlistdelete \"%s\" %d\n", name, songId);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plPlaylistMove(const char *name, int songId, int pos)
{
	char cmd[2048];
	sprintf(cmd, "playlistmove \"%s\" %d %d\n", name, songId, pos);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::plPlaylistFind(const char *scope, const char *query, MPD_METADATA_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "playlistfind \"%s\" \"%s\"\n", scope, query);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaData(t) ) return false;
	return true;
}

bool DevMPD::plPlaylistSearch(const char *scope, const char *query, MPD_METADATA_LIST &t)
{
	char cmd[2048];
	sprintf(cmd, "playlistsearch \"%s\" \"%s\"\n", scope, query);
	if( !sendCmd(cmd) ) return false;
	if( !recvMetaData(t) ) return false;
	return true;
}


// playback command
bool DevMPD::pbCrossfade(int sec)
{
	char cmd[2048];
	sprintf(cmd, "crossfade %d\n", sec);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbNext()
{
	if( !sendCmd("next\n") ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbPause(bool state)
{
	char cmd[2048];
	sprintf(cmd, "pause %d\n", state ? 1 : 0);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbPlay(int song)
{
	char cmd[2048];
	if( song != -1 )
		sprintf(cmd, "play %d\n", song);
	else
		sprintf(cmd, "play\n");
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbPlayId(int songId)
{
	char cmd[2048];
	if( songId != -1 )
		sprintf(cmd, "playid %d\n", songId);
	else
		sprintf(cmd, "playid\n");
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbPrevious()
{
	if( !sendCmd("previous\n") ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbRandom(bool state)
{
	char cmd[2048];
	sprintf(cmd, "random %d\n", state ? 1 : 0);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbRepeat(bool state)
{
	char cmd[2048];
	sprintf(cmd, "repeat %d\n", state ? 1 : 0);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbSeek(int song, int time)
{
	char cmd[2048];
	sprintf(cmd, "seek %d %d\n", song, time);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbSeekId(int songId, int time)
{
	char cmd[2048];
	sprintf(cmd, "seekid %d %d\n", songId, time);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbSetVol(int vol)
{
	char cmd[2048];
	sprintf(cmd, "setvol %d\n", vol);
	if( !sendCmd(cmd) ) return false;
	if( !recvResult() ) return false;
	return true;
}

bool DevMPD::pbStop()
{
	if( !sendCmd("stop\n") ) return false;
	if( !recvResult() ) return false;
	return true;
}
