/*
 * devMPD.h
 *
 *  Created on: 2012/03/11
 *      Author: tanaka
 */

#ifndef DEVMPD_H_
#define DEVMPD_H_

#include "device.h"
class BZMPD;

enum MPD_STATE {
	ST_UNKNOWN=0,
	ST_PLAY,
	ST_STOP,
	ST_PAUSE
};

enum MPD_SCOPE {
	MPD_ARTIST=1,	// artist - name of the Artist
	MPD_ALBUM,		// album - name of the Album from which the song is taken
	MPD_TITLE,		// title - title of the song
	MPD_TRACK,		// track - track number on the album
	MPD_NAME,		// name - name of an audio stream
	MPD_GENRE,		// genre - keywords that place the song in one or more aesthetic baskets
	MPD_DATE,		// date - date the song was recorded (not when it was ripped ;-)
	MPD_COMPOSER,	// composer - who wrote the song
	MPD_PERFORMER,	// performer - principal soloist or performer
	MPD_COMMENT,	// comment - typically something attached by the ripper
	MPD_DISC,		// disc - disc number (in a set). Derived from the ID3 TPOS tag.
};

typedef struct _MPD_STATUS {
	int		volume; // (0-100)
	bool	repeat;	// (0 or 1)
	bool	random;	// (0 or 1)
	unsigned int	playlist;	// (31-bit unsigned integer, the playlist version number)
	int		playlistlength;	// (integer, the length of the playlist)
	int		xfade;	// <int seconds> (crossfade in seconds)
	MPD_STATE	state;	// ("play", "stop", or "pause")
	int		song;	// (current song stopped on or playing, playlist song number)
	int		songId;	// (current song stopped on or playing, playlist songid)
	unsigned int	elapsedTime;	// <int elapsed>:<time total> (of current playing/paused song)
	unsigned int	totalTime;
	unsigned int	bitRate;	// <int bitrate> (instantaneous bitrate in kbps)
	unsigned int	sampleRate;	// audio: <int sampleRate>:<int bits>:<int channels>
	unsigned int	bits;		// audio: <int sampleRate>:<int bits>:<int channels>
	unsigned int	channels;	// audio: <int sampleRate>:<int bits>:<int channels>
	unsigned int	updatingDb;	// <int job id>
	char *error;	// if there is an error, returns message here
	int	nextSong;	// (next song, playlist song number)
	int	nextSongId;	// (next song, playlist songid)[1]
} MPD_STATUS;

typedef struct _MPD_OUTPUTS {
	int		outputId;		// <int output> the output number
	char *	outputName;		// <str name> the name as defined in the MPD configuration file
	bool	outputEnabled;	// <int enabled> 1 if enabled, 0 if disabled
} MPD_OUTPUTS;

typedef struct _MPD_TAG {
	char	* type;	// tag name
} MPD_TAG;

typedef struct _MPD_TAG_LIST {
	int	count;
	int buffers;
	MPD_TAG	*tags;
} MPD_TAG_LIST;

typedef struct _MPD_URLHANDLER {
	char	* url;	// prefix
} MPD_URLHANDLER;

typedef struct _MPD_URLHANDLER_LIST {
	int	count;
	int buffers;
	MPD_URLHANDLER	*handlers;
} MPD_URLHANDLER_LIST;

typedef struct _MPD_METADATA {
	int	type;
	char *value;
} MPD_METADATA;
typedef struct _MPD_METADATA_LIST {
	int	count;
	int buffers;
	MPD_METADATA	*metadatas;
} MPD_METADATA_LIST;
typedef struct _MPD_METAGROUP_LIST {
	int	count;
	int buffers;
	MPD_METADATA_LIST	**metagroup;
} MPD_METAGROUP_LIST;

class DevMPD : public Device
{
public:
	DevMPD();
	~DevMPD();

	virtual const char *getName() { return "mpd"; }
	virtual bool setDevicePort(Port *port);
	virtual bool process(Port *port);
	virtual bool connect(Port *port);

	bool	addTagType(MPD_TAG_LIST &t, const char *v);
	void	clearTagType(MPD_TAG_LIST &t);
	bool	addUrlHandler(MPD_URLHANDLER_LIST &t, const char *v);
	void	clearUrlHandler(MPD_URLHANDLER_LIST &t);
	int		findMetaData(MPD_METADATA_LIST &t, int tagId);
	bool	addMetaData(MPD_METADATA_LIST &t, const char *v);
	bool	addMetaData(MPD_METADATA_LIST &t, int tagId, const char *value);
	void	clearMetaDataList(MPD_METADATA_LIST &t);
	void	sortMetaDataList(MPD_METADATA_LIST &t, bool numericCompaire);

	bool	addMetaGroup(MPD_METAGROUP_LIST &t, MPD_METADATA_LIST *v);
	void	clearMetaGroupList(MPD_METAGROUP_LIST &t);
	void	sortMetaGroupList(MPD_METAGROUP_LIST &t, int tagId, bool numericCompaire);

	char *	seekValue(const char *key, char *ptr);
	bool	isConnected();
	Port *	getPort();
	int		getTagId(const char *tagName);

	// extend command
	bool	exSend(const char *sd, size_t sd_size);
	bool	exRecv(char *rd, size_t rd_size, size_t &readed);

	// admin command
	bool	adConnect();
	void	adDisconnect();
	bool	adDisableOutput();
	bool	adEnableOutput();
	bool	adKill();
	bool	adUpdate(const char *path, int &jobId);

	// Informational Commands
	bool	adStatus(MPD_STATUS &st);
	bool	adOutputs(MPD_OUTPUTS &outputs);
	bool	adTagtypes(MPD_TAG_LIST &tags);
	bool	adUrlHandlers(MPD_URLHANDLER_LIST &uhl);

	// database command
	bool	dbFind(const char *query, MPD_METAGROUP_LIST &t);
	bool	dbList(const char *tag, const char *query, MPD_METADATA_LIST &t);
	bool	dbListAll(const char *path, MPD_METADATA_LIST &t);
	bool	dbListAllInfo(const char *path, MPD_METAGROUP_LIST &metalist);
	bool	dbListInfo(const char *path, MPD_METAGROUP_LIST &metalist);
	bool	dbSearch(const char *query, MPD_METAGROUP_LIST &metalist);
	bool	dbCount(const char *query, MPD_METADATA_LIST &metalist);

	// playlist command
	bool	plAdd(const char *song);
	bool	plAddId(const char *song, int &plid);
	bool	plClear();
	bool	plCurrentSong(MPD_METADATA_LIST &metalist);
	bool	plDelete(int song);
	bool	plDeleteId(int songId);
	bool	plLoad(const char *name);
	bool	plRename(const char *oldName, const char *newName);
	bool	plMove(int fromNum, int toNum);
	bool	plMoveId(int fromId, int toId);
	bool	plPlaylistInfo(int song, MPD_METAGROUP_LIST &metalist);
	bool	plPlaylistId(int songId, MPD_METAGROUP_LIST &metalist);
	bool	plChanges(int ver, MPD_METADATA_LIST &metalist);
	bool	plChangesPositionId(int ver, MPD_METADATA_LIST &metalist);
	bool	plRemove(const char *name);
	bool	plSave(const char *name);
	bool	plShuffle();
	bool	plSwap(int song1, int song2);
	bool	plSwapId(int songId1, int songId2);
	bool	plListPlaylist(const char *name, MPD_METADATA_LIST &metalist);
	bool	plListPlaylistInfo(const char *name, MPD_METADATA_LIST &metalist);
	bool	plPlaylistAdd(const char *name, const char *path);
	bool	plPlaylistClear(const char *name);
	bool	plPlaylistDelete(const char *name, int songId);
	bool	plPlaylistMove(const char *name, int songId, int pos);
	bool	plPlaylistFind(const char *scope, const char *query, MPD_METADATA_LIST &metalist);
	bool	plPlaylistSearch(const char *scope, const char *query, MPD_METADATA_LIST &metalist);

	// playback command
	bool	pbCrossfade(int sec);
	bool	pbNext();
	bool	pbPause(bool state);
	bool	pbPlay(int song);
	bool	pbPlayId(int songId);
	bool	pbPrevious();
	bool	pbRandom(bool state);
	bool	pbRepeat(bool state);
	bool	pbSeek(int song, int time);
	bool	pbSeekId(int songId, int time);
	bool	pbSetVol(int vol);
	bool	pbStop();

protected:
	bool	sendCmd(const char *cmd, size_t len);
	bool	sendCmd(const char *cmd);
	bool	recvRes(size_t buffSize, char *buff, size_t &readed);
	bool	recvResult();
	bool	recvMetaData(MPD_METADATA_LIST &metalist);
	bool	recvMetaGroupData(MPD_METAGROUP_LIST &metalist);

	char	mpd_version[16];
	MPD_TAG_LIST tagList;
};

#endif /* DEVMPD_H_ */
