/*
 * Port232c.cpp
 *
 *  Created on: 2012/03/02
 *      Author: tanaka
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <malloc.h>
#include <syslog.h>
#include <termios.h>
#include <fcntl.h>
#include <errno.h>
#include "bzmpd.h"
#include "port232c.h"

Port232c::Port232c(BZMPD *bzmpd) : Port(bzmpd)
{
	chunkRead	= 0;
	chunkSize	= 0;
	fd			= 0;
	endFile		= false;
}

Port232c::~Port232c()
{
	if( isConnect() )
		close();
}

bool Port232c::open(const char *conf)
{
	char dev[256];
	const char *p0 = conf;
	char *p1 = dev;
	while( *p0 && *p0 != ':' ) {
		*p1++ = *p0++;
	}
	*p1 = 0;
	if( *p0 == ':' ) p0++;
	int baudrate = B9600;
	if( strcmp(p0, "9600") == 0 )
		baudrate	= B9600;
	else if( strcmp(p0, "19200") == 0 )
		baudrate	= B19200;
	else if( strcmp(p0, "38400") == 0 )
		baudrate	= B38400;
	else if( strcmp(p0, "57600") == 0 )
		baudrate	= B57600;
	else if( strcmp(p0, "115200") == 0 )
		baudrate	= B115200;

	int f	= ::open(dev, O_RDWR|O_NOCTTY);
	if( f < 0 ) {
		bzmpdPtr->log(LOG_ERR, "serial port %s can't open", dev);
		return false;
	}

	struct termios tioNew;
	memset(&tioNew, 0, sizeof(tioNew));

	tcgetattr(fd, &tioSave);
	tioNew.c_cflag = (baudrate | CS8 | CLOCAL | CREAD);
	tioNew.c_iflag = (IGNPAR | ICRNL);
	tioNew.c_oflag = 0;
	tioNew.c_lflag = ICANON;
#if 0
	tioNew.c_cc[VINTR]    = 0;     /* Ctrl-c */
	tioNew.c_cc[VQUIT]    = 0;     /* Ctrl-\ */
	tioNew.c_cc[VERASE]   = 0;     /* del */
	tioNew.c_cc[VKILL]    = 0;     /* @ */
	tioNew.c_cc[VEOF]     = 4;     /* Ctrl-d */
	tioNew.c_cc[VTIME]    = 0;     /* キャラクタ間タイマを使わない */
	tioNew.c_cc[VMIN]     = 1;     /* 1文字来るまで，読み込みをブロックする */
	tioNew.c_cc[VSWTC]    = 0;     /* '\0' */
	tioNew.c_cc[VSTART]   = 0;     /* Ctrl-q */
	tioNew.c_cc[VSTOP]    = 0;     /* Ctrl-s */
	tioNew.c_cc[VSUSP]    = 0;     /* Ctrl-z */
	tioNew.c_cc[VEOL]     = 0;     /* '\0' */
	tioNew.c_cc[VREPRINT] = 0;     /* Ctrl-r */
	tioNew.c_cc[VDISCARD] = 0;     /* Ctrl-u */
	tioNew.c_cc[VWERASE]  = 0;     /* Ctrl-w */
	tioNew.c_cc[VLNEXT]   = 0;     /* Ctrl-v */
	tioNew.c_cc[VEOL2]    = 0;     /* '\0' */
#endif
	tcflush(f, TCIFLUSH);
	tcsetattr(f,TCSANOW,&tioNew);
	fd	= f;
	chunkRead	= 0;
	chunkSize	= 0;
	endFile		= false;
	return true;
}

void Port232c::close()
{
	if( fd ) {
		::close(fd);
		fd	= 0;
	}
}

bool Port232c::isConnect()
{
	return fd != 0;
}

bool Port232c::isDataEnd()
{
	return endFile;
}

int Port232c::getDesciptor()
{
	return fd;
}

bool Port232c::sendData(size_t size, const char *data)
{
	size_t wsize = ::write(fd, data, size);
	if( wsize != size ) return false;
	return true;
}

bool Port232c::recvData(size_t size, char *data, size_t &readed)
{
	readed = 0;
	char *bp = data;
	while(1) {
		if( chunkSize ) {
			char *p = &chunk[chunkRead];
			while( chunkRead < chunkSize && size > 0 ) {
				if( *p == '\r' || *p == '\n' ) {
					*bp = 0;
					if( *p == '\r' ) { p++; chunkRead++; }
					if( *p == '\n' ) { p++; chunkRead++; }
					readed = bp - data;
					return true;
				}
				*bp++ = *p++;
				size--;
				chunkRead++;
			}
			if( size == 0 ) {
				readed = bp - data;
				return true;
			}
		}

		chunkRead	= 0;
		while(1) {
			size_t recvs = ::read(fd,chunk, sizeof(chunk));
			if(recvs == 0) {
				endFile = true;
				return false;
			}
			if( ((int)recvs) == -1) {
				if( errno == EAGAIN ) {
					readed	= bp - data;
					return true;
				} else {
					return false;
				}
			}
			chunkSize = recvs;
		}
	}
	return true;
}
