/*
 * PortUSB.cpp
 *
 *  Created on: 2012/03/02
 *      Author: tanaka
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <malloc.h>
#include <syslog.h>
#include <usb.h>
#include <fcntl.h>
#include <errno.h>
#include <pthread.h>
#include "bzmpd.h"
#include "portUSB.h"

#define VENDOR_ID	0x03EB
#define PRODUCT_ID	0x2013

PortUSB::PortUSB(BZMPD *bzmpd) : Port(bzmpd)
{
	chunkRead	= 0;
	chunkSize	= 0;
	pipe_fd[0]	= 0;
	pipe_fd[1]	= 0;
	uh			= NULL;
	endFile		= false;
	doStop		= false;
	threadHandlePtr	= NULL;
}

PortUSB::~PortUSB()
{
	if( isConnect() )
		close();
}

const char *parseInt(const char *p, int &d)
{
	d = 0;
	if( *p == '0' && (*(p+1) == 'x' || *(p+1) == 'X')) {
		p += 2;
		while( *p && *p != ':' ) {
			int a = 0;
			if( '0' <= *p && *p <= '9' )
				a = *p - '0';
			else if( 'a' <= *p && *p <= 'f' )
				a = *p - 'a' + 10;
			else if( 'A' <= *p && *p <= 'F' )
				a = *p - 'A' + 10;
			else
				return p;
			d = d * 16 + a;
			p++;
		}
	} else {
		while( *p && *p != ':' ) {
			int a = 0;
			if( '0' <= *p && *p <= '9' )
				a = *p - '0';
			else
				return p;
			d = d * 10 + a;
			p++;
		}
	}
	return p;
}

struct usb_device *PortUSB::findUSBDevice(int vid, int pid)
{
	struct usb_bus *bus;
	struct usb_device *dev;
	for( bus = usb_get_busses(); bus; bus = bus->next ) {
		for( dev = bus->devices; dev; dev = dev->next ) {
			bzmpdPtr->log(LOG_DEBUG, "USB Device VID:%x PID:%x", dev->descriptor.idVendor, dev->descriptor.idProduct);
			if( dev->descriptor.idVendor == vid &&
				dev->descriptor.idProduct == pid) {
				return dev;
			}
		}
	}
	return NULL;
}
bool PortUSB::open(const char *conf)
{
	int	vid=-1;
	int pid=-1;
	int ep_in=-1;
	int ep_out=-1;
	const char *p0 = conf;
	while( *p0 == ':') p0++;
	while( *p0 ) {
		if( strncmp(p0, "vid:", 4) == 0 ) {
			p0 = parseInt(p0+4, vid);
		} else if( strncmp(p0, "pid:", 4) == 0 ) {
			p0 = parseInt(p0+4, pid);
		} else if( strncmp(p0, "in:", 3) == 0 ) {
			p0 = parseInt(p0+3, ep_in);
		} else if( strncmp(p0, "out:", 4) == 0 ) {
			p0 = parseInt(p0+4, ep_out);
		} else {
			bzmpdPtr->log(LOG_ERR, "USB port defnition error %d:%s", p0-conf, p0);
			return false;
		}
		if( *p0 == ':' ) {
			p0++;
			continue;
		}
		if( *p0 ) {
			bzmpdPtr->log(LOG_ERR, "USB port defnition error %d:%s", p0-conf, p0);
			return false;
		}
	}
	if( vid == -1 || pid == -1 || ep_in == -1 || ep_out == -1 ) {
		bzmpdPtr->log(LOG_ERR, "USB port defnition error");
		return false;
	}
	bzmpdPtr->log(LOG_DEBUG, "USB config VID=%x PID=%x in=%d out=%d", vid, pid, ep_in, ep_out);

	usb_init();
	usb_find_busses();
	usb_find_devices();

	struct usb_device *dev = findUSBDevice(vid, pid);
	if( !dev ) {
		bzmpdPtr->log(LOG_ERR, "USB Device VID:%x PID:%x can't found", vid, pid);
		return false;
	}

	bzmpdPtr->log(LOG_INFO, "USB Device VID:%x PID:%x found", vid, pid);

	struct usb_dev_handle *dh;
	if( (dh=usb_open(dev))==NULL ){
		bzmpdPtr->log(LOG_ERR, "USB Device VID:%x PID:%x can't open", vid, pid);
		return false;
	}

	if(usb_set_configuration(dh,dev->config->bConfigurationValue)<0){
	       if(usb_detach_kernel_driver_np(dh,dev->config->interface->altsetting->bInterfaceNumber)<0 ){
				bzmpdPtr->log(LOG_ERR, "usb_set_configuration Error.\n");
				bzmpdPtr->log(LOG_ERR, "usb_detach_kernel_driver_np Error.(%s)\n",usb_strerror());
				usb_release_interface(dh, 0);
	    		usb_close(dh);
	    		return false;
	       }
	}
	if(usb_claim_interface(dh,dev->config->interface->altsetting->bInterfaceNumber)<0 ){
	       if(usb_detach_kernel_driver_np(dh,dev->config->interface->altsetting->bInterfaceNumber)<0 ){
				bzmpdPtr->log(LOG_ERR, "usb_claim_interface Error.\n");
				bzmpdPtr->log(LOG_ERR, "usb_detach_kernel_driver_np Error.(%s)\n",usb_strerror());
				usb_release_interface(dh, 0);
	    		usb_close(dh);
				return false;
	       }
	   }

	if(usb_claim_interface(dh,dev->config->interface->altsetting->bInterfaceNumber)<0 ){
		bzmpdPtr->log(LOG_ERR, "usb_claim_interface Error.(%s)\n",usb_strerror());
		usb_release_interface(dh, 0);
		usb_close(dh);
		return false;
	}
	u_in	= ep_in;
	u_out	= ep_out;
	uh		= dh;

	doStop		= false;
	chunkRead	= 0;
	chunkSize	= 0;
	endFile		= false;
	pipe(pipe_fd);
	if( threadHandlePtr != NULL ) {
		bzmpdPtr->log(LOG_ERR, "thread 'USB Monitor' is already strated");
	}
	if( pthread_create( &threadHandle, NULL, threadEntry, this) != 0 ) {
		bzmpdPtr->log(LOG_ERR, "start thread failed");
		return false;
	}
	threadHandlePtr	= &threadHandle;
	return true;
}

void PortUSB::close()
{
	bzmpdPtr->log(LOG_DEBUG, "USB do close");
	doStop	= true;
	endFile	= true;
	if( uh ) {
		usb_release_interface(uh, 0);
		usb_close(uh);
	}
	if( pipe_fd[0] )
		::close(pipe_fd[0]);
	if( pipe_fd[1] )
		::close(pipe_fd[1]);

	if( threadHandlePtr ) {
		void *ret = NULL;
		pthread_join(threadHandle, &ret);
		threadHandlePtr = NULL;
		doStop	= false;
	}
	uh	= NULL;
	chunkRead	= 0;
	chunkSize	= 0;
	pipe_fd[0]	= 0;
	pipe_fd[1]	= 0;
	bzmpdPtr->log(LOG_DEBUG, "USB close done");
}

bool PortUSB::isConnect()
{
	return pipe_fd[0] != 0 || uh != NULL;
}

bool PortUSB::isDataEnd()
{
	return endFile;
}

int PortUSB::getDesciptor()
{
	return pipe_fd[0];
}

bool PortUSB::sendPacket(size_t size, const char *data)
{
#ifndef DAEMON
	char msg[32];
	int i;
	for( i = 0; i < size; i++ ) {
		msg[i] = data[i];
	}
	msg[i] = 0;
	bzmpdPtr->log(LOG_DEBUG, "USB send %d bytes <%s>", size, msg);
#endif
	return sendData(size, data);
}

bool PortUSB::recvPacket(size_t size, char *data, size_t &readed)
{
	return recvData(size, data, readed);
}

#define TIMEOUT 	900
bool PortUSB::sendData(size_t size, const char *data)
{
	int wsize = usb_interrupt_write(uh, u_out, data, size, TIMEOUT);
	if( wsize < 0 ) {
		bzmpdPtr->log(LOG_ERR, "USB write error %s", usb_strerror());
		return false;
	}
	bzmpdPtr->log(LOG_INFO, "USB send %d bytes ok", size);
	if( (size_t)wsize != size ) return false;
	return true;
}

bool PortUSB::recvData(size_t size, char *data, size_t &readed)
{
	bzmpdPtr->log(LOG_DEBUG, "USB: recvData %d", size);
	char dummy[2];
	read(pipe_fd[0], dummy, 1);
	readed = 0;
	char *bp = data;
	while(!doStop && !endFile) {
		{
			bzmpdPtr->log(LOG_DEBUG, "USB: Wait chunk buffer for read");
			WriteLock lc(lockObject);
			bzmpdPtr->log(LOG_DEBUG, "USB: Read chunk buffer. bufferSize=%d chunkSize=%d chunkRead=%d", sizeof(chunk), chunkSize, chunkRead);
			if( chunkSize ) {
				char *p = &chunk[chunkRead];
				while( chunkRead < chunkSize && size > 0 ) {
					*bp++ = *p++;
					size--;
					chunkRead++;
				}
				if( size == 0 ) {
					readed = bp - data;
					bzmpdPtr->log(LOG_DEBUG, "USB: recvData ok. bufferSize=%d chunkSize=%d chunkRead=%d", sizeof(chunk), chunkSize, chunkRead);
					return true;
				}
			}
		}
		bzmpdPtr->log(LOG_DEBUG, "USB: Wait for data recive");
		usleep(10000);
	}
	return true;
}

void *PortUSB::threadEntry(void *ptr)
{
	void *ret = ((PortUSB*)ptr)->mainLoop();
	((PortUSB*)ptr)->threadHandlePtr	= NULL;
	return ret;
}

#define TIMEOUT 	900
#define PACKETSIZE	8
void *PortUSB::mainLoop()
{
	bzmpdPtr->log(LOG_INFO, "USB Monitor start");
	while(!doStop && !endFile) {
		int r,i;
		char answer[PACKETSIZE];
		r = usb_interrupt_read(uh, u_in, answer, PACKETSIZE, TIMEOUT);
		if( r > 0 ) {
			bzmpdPtr->log(LOG_DEBUG, "USB: Recive from device size=%d %c%c", r, answer[0], answer[1]);
			while(!doStop && !endFile){
				{
					bzmpdPtr->log(LOG_DEBUG, "USB: Wait chunk buffer for write");
					WriteLock lc(lockObject);
					bzmpdPtr->log(LOG_DEBUG, "USB: Write chunk buffer. bufferSize=%d chunkSize=%d chunkRead=%d", sizeof(chunk), chunkSize, chunkRead);
					size_t insize = chunkSize-chunkRead;
					if( sizeof(chunk) - insize < r ) {
						bzmpdPtr->log(LOG_DEBUG, "USB: Wait chunk buffer free(insize=%d, free=%d)", insize, sizeof(chunk) - insize);
						::usleep(10000);
						continue;
					}
					memcpy(chunk, &chunk[chunkRead], insize);
					memcpy(&chunk[insize], answer, r);
					chunkSize = insize + r;
					chunkRead = 0;
				}
				write(pipe_fd[1], "0", 1);
				bzmpdPtr->log(LOG_DEBUG, "USB: read ok. bufferSize=%d chunkSize=%d chunkRead=%d", sizeof(chunk), chunkSize, chunkRead);
				break;
			}
		}
	}
	return NULL;
}

