/*
 * codegen.cpp
 *
 *  Created on: 2012/01/25
 *      Author: tanaka
 */

#include "codegen.h"
#include "executeService.h"
#include "variant.h"
#include "parser.h"

namespace es {

#define ED_RESULT(RE, PC, CODE, EA) \
		if( (EA).mode != EM_IGNR ) { \
			if( (EA).mode == EM_ST ) { \
				if( !(CODE).add(IC_TEST, (RE)) ) { \
					setError(PC); \
					return false; \
				} \
			} else { \
				if( (EA).mode == EM_NA ) { \
					if( !getTempFolder((EA)) ) { \
						setError(PC); \
						return false; \
					} \
					if( !code.add(IC_MOVE, (RE), (EA)) ) { \
						setError(PC); \
						return false; \
					} \
				} else if( ((EA).mode & EM_MASK) == EM_OFFSET ) { \
					if( !code.add(IC_MOVEV, (RE), (EA)) ) { \
						setError(PC); \
						return false; \
					} \
				} else if( !code.add(IC_MOVE, (RE), (EA)) ) { \
					setError(PC); \
					return false; \
				} \
			} \
		}

#define ED_RESULT_R0(PC, CODE, EA) \
	if( (EA).mode != EM_IGNR ) { \
		if( (EA).mode == EM_ST ) { \
			if( !(CODE).add(IC_TEST, ED_R0()) ) { \
				setError((PC)); \
				return false; \
			} \
		} else if( (EA).mode != EM_R0 ) { \
			if( (EA).mode == EM_NA ) { \
				if( !getTempFolder(EA) ) { \
					setError(PC); \
					return false; \
				} \
				if( !(CODE).add(IC_MOVE, ED_R0(), (EA)) ) { \
					setError(PC); \
					return false; \
				} \
			} else if( ((EA).mode & EM_MASK) == EM_OFFSET ) { \
				if( !(CODE).add(IC_MOVEV, ED_R0(), (EA)) ) { \
					setError(PC); \
					return false; \
				} \
			} else if( !(CODE).add(IC_MOVE, ED_R0(), (EA)) ) { \
				setError(PC); \
				return false; \
			} \
		} \
	}


Bytecode::Bytecode()
{
	scopeBase = NULL;
}

Bytecode::~Bytecode()
{
}

void Bytecode::setScopeBase(const Bytecode *base)
{
	scopeBase	= base;
}

BYTE * Bytecode::addCode(SIZE size)
{
	SIZE p = code.size();
	try {
		code.resize(code.size()+size);
	}
	catch(Error &) {
		return NULL;
	}
	return code.ptr() + p;
}

bool Bytecode::add(WORD d)
{
	try {
		code.add(sizeof(WORD), (const BYTE *)&d);
	}
	catch(Error &) {
		return false;
	}
	return true;
}

bool Bytecode::add(INT d)
{
	try {
		code.add(sizeof(INT), (const BYTE *)&d);
	}
	catch(Error &) {
		return false;
	}
	return true;
}

bool Bytecode::add(SIZE d)
{
	try {
		code.add(sizeof(SIZE), (const BYTE *)&d);
	}
	catch(Error &) {
		return false;
	}
	return true;
}

bool Bytecode::add(ICode ic)
{
	try {
		code.add(sizeof(BYTE), (const BYTE *)&ic);
	}
	catch(Error &) {
		return false;
	}
	return true;
}

bool Bytecode::add(const ED &ea)
{
	BYTE *p;
	INT size;
	BYTE type = (ea.mode & EM_MASK);
	switch(type) {
	case EM_REG:
		p = addCode(sizeof(BYTE));
		if( !p ) return false;
		*p++	= (BYTE)ea.mode;
		break;
	case EM_OFFSET:
		p = addCode(sizeof(BYTE)+sizeof(INT));
		if( !p ) return false;
		*p++		= (BYTE)ea.mode;
		*(INT*)p	= ea.offset;
		break;
	case EM_IMD:
		size = imdSize(ea.mode);
		p = addCode(sizeof(BYTE)+size);
		if( !p ) return false;
		*p++	= (BYTE)ea.mode;
		copyMemory(p, (const BYTE*)&ea.offset, size);
		break;
	}
	return true;
}

bool Bytecode::add(ICode ic, const ED &ea)
{
	if( !add(ic) || !add(ea) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, const ED &ea, WORD d)
{
	if( !add(ic) || !add(ea) || !add(d) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, const ED &ea, WORD d, INT d2)
{
	if( !add(ic) || !add(ea) || !add(d) || !add(d2) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, const ED &ea1, const ED &ea2)
{
	if( !add(ic) || !add(ea1) || !add(ea2) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, const ED &ea1, const ED &ea2, const ED &ea3)
{
	if( !add(ic) || !add(ea1) || !add(ea2) || !add(ea3) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, INT oa)
{
	if( !add(ic) || !add(oa) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, INT oa, INT size)
{
	if( !add(ic) || !add(oa) || !add(size) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, WORD d)
{
	if( !add(ic) || !add(d) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, WORD sid, WORD arg)
{
	if( !add(ic) || !add(sid) || !add(arg) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, WORD sid, INT offset)
{
	if( !add(ic) || !add(sid) || !add(offset) ) return false;
	return true;
}

bool Bytecode::add(ICode ic, WORD sid, INT minArgs, INT maxArgs, INT offset)
{
	if( !add(ic) || !add(sid) || !add(minArgs) || !add(maxArgs) || !add(offset) ) return false;
	return true;
}

bool Bytecode::add(const Bytecode &d)
{
	try {
		INT top_addr = (INT)code.size();
		code += d.code;
		SIZE i;
		for( i = 0; i < d.labelTableRef.size(); i++ ) {
			const CodeLabel *ref = d.labelTableRef.ptr(i);
			CodeLabel *hit = d.findCodeLabel(&ref->label);
			INT ref_addr = top_addr + ref->addr;

			if( !hit ) {
				labelTableRef.add(&ref->label, ref_addr);
			} else if( hit->addr == CodeLabel::CODE_NOTSET ) {
				hit->setAddr(ref_addr);
			} else {
				INT def_addr = top_addr + hit->addr;
				BYTE *p = &code.ptr()[ref_addr];
				*(INT*)p = def_addr - ref_addr;
			}
		}
		for( i = 0; i < d.labelTable.size(); i++ ) {
			const CodeLabel *ptr = d.labelTable.ptr(i);
			if( ptr->addr == CodeLabel::CODE_NOTDEF ||
				ptr->addr == CodeLabel::CODE_NOTSET ) {
				labelTable.add(&ptr->label, ptr->addr);
			} else {
				labelTable.add(&ptr->label, ptr->addr+top_addr);
			}
		}
		link();
	}
	catch(Error &) {
		return false;
	}
	return true;
}

CodeLabel * Bytecode::findCodeLabel(const SYMBOL * label) const
{
	return labelTable.find(label);
}

bool Bytecode::addCodeLabel(const SYMBOL * label)
{
	CodeLabel *hit = labelTable.find(label);
	if( !hit ) {
		return labelTable.add(label, CodeLabel::CODE_NOTSET);
	}
	return true;
}

bool Bytecode::addCodeLabelRef(const SYMBOL *label, INT addr)
{
	return labelTableRef.add(label, addr);
}

SIZE Bytecode::saveCodeLabel() const
{
	return labelTable.size();
}

bool Bytecode::restoreCodeLabel(SIZE savePos)
{
	return labelTable.resize(savePos);
}

bool Bytecode::link()
{
	SIZE i = 0;
	while( i < labelTableRef.size() ) {
		const CodeLabel *ptr = labelTableRef.ptr(i);
		CodeLabel *hit = findCodeLabel(&ptr->label);
		if( hit && hit->addr != CodeLabel::CODE_NOTDEF && hit->addr != CodeLabel::CODE_NOTSET ) {
			BYTE *p = &code.ptr()[ptr->addr];
			INT	la = hit->addr - ptr->addr;
			*(INT*)p = la;
			labelTableRef.erase(i);
		} else {
			i++;
		}
	}
	return labelTableRef.empty();
}

SIZE Bytecode::size()
{
	return code.size();
}

CodeGenerator::CodeGenerator(IContext *ctx)
{
	this->ctx		= ctx;
	this->es		= ctx->getExecuteService();
	this->tm		= ctx->getTypeManager();
	this->pcProgram	= NULL;
}

CodeGenerator::~CodeGenerator()
{
}

void CodeGenerator::setError(const PARSEINF * pc)
{
	es->throwException(thisScope, ERROR_LOGICAL, NULL);
}

bool CodeGenerator::generate(BYTES *result, const IScope *scope, const PARSEINF *pInf)
{
	if( pInf == NULL || pInf->type != ST_PROGRAM ) {
		es::setError(ERROR_LOGICAL);
		return false;
	}

	thisScope = es->createStack(scope, this);
	pcProgram = &pInf->dProgram;

	DWORD mc = MAGICCODE;
	const CHAR *url_ptr	= getStringReadPtr(&pcProgram->URL);
	SIZE url_len		= getStringSize(&pcProgram->URL);
	SIZE minCodeSize = sizeof(mc) +
					   sizeof(SIGNATURE_SECTION) + url_len +
					   sizeof(CDATA_SECTION) +
					   sizeof(XREF_SECTION) +
					   sizeof(CODE_SECTION);
	clearBytes(result);
	reserveBytes(result, minCodeSize);

	// magic
	if( !addByteSequence(result, (const BYTE*)&mc, sizeof(mc)) ) {
		return false;
	}

	// signature section
	SIGNATURE_SECTION sig_sec;
	sig_sec.head.type		= SEC_SIGNATURE;
	sig_sec.head.size		= sizeof(SIGNATURE_SECTION)+url_len*sizeof(CHAR);
	sig_sec.codeSetVersion	= 0x0100;
	sig_sec.genTime			= getDateTime();
	sig_sec.compilerVersion[0]	= 1;
	sig_sec.compilerVersion[1]	= 0;
	sig_sec.compilerVersion[2]	= 0;
	sig_sec.compilerVersion[3]	= 0;
	sig_sec.optionFlags		= 0;
	sig_sec.sourceURLSize	= url_len;
	addByteSequence(result, (const BYTE *)&sig_sec, sizeof(SIGNATURE_SECTION)-sizeof(CHAR));
	if( url_len ) addByteSequence(result, (const BYTE*)url_ptr, (url_len+1) * sizeof(CHAR));

	tempSeq			= 0;
	lastLine		= 0;
	usedRegs		= 0;
	protectedRegs	= 0;
	initScope();
	Bytecode inner_code;
	SIZE	nCodeFragment=0;
	if( !codeGen(pcProgram->innerCode, inner_code, nCodeFragment) ) {
		return false;
	}

	// cdata section
	if( !cdata.empty() ) {
		SIZE cnt  = cdata.size();
		SIZE sec_size = sizeof(CDATA_SECTION) - sizeof(CDATA);
		SIZE i;
		for( i = 0; i < cnt; i++ ) {
			BYTES * b = cdata.get(i);
			sec_size += sizeof(CDATA) - sizeof(BYTE);
			sec_size += getBytesSize(b);
		}
		CDATA_SECTION *sec = (CDATA_SECTION*)allocMemory(sec_size);
		if( !sec ) {
			freeMemory(sec);
			return false;
		}

		sec->head.type	= SEC_CDATA;
		sec->head.size	= sec_size;
		sec->count		= cnt;

		BYTE *p = (BYTE*)&sec->data[0];
		for( i = 0; i < cnt; i++ ) {
			BYTES *b = cdata.get(i);
			CDATA *d = (CDATA*)p;
			d->size	= getBytesSize(b);
			copyMemory(d->value, getBytesReadPtr(b), d->size);
			MethodRefArray::Node *ref = methodRef.find((WORD)i);
			if( ref ) {
				CDEF_METHOD *method = (CDEF_METHOD*)d->value;
				method->procAddr	= (INT)ref->offset;
			}
			p += sizeof(CDATA) - sizeof(BYTE) + d->size;
		}
		if( !addByteSequence(result, (BYTE*)sec, sec_size) ) {
			freeMemory(sec);
			return false;
		}
		freeMemory(sec);
	}

	// symbol section
	{
		SIZE cnt  = symbol.size();
		if( cnt > 0 ) {
			SIZE sec_size = sizeof(SYMBOL_SECTION) - sizeof(SYMBOL);
			SIZE i;
			for( i = 0; i < cnt; i++ ) {
				STRING * b = symbol.get(i);
				sec_size += sizeof(SYMBOLT) + getStringSize(b) * sizeof(CHAR);
			}
			SYMBOL_SECTION *sec = (SYMBOL_SECTION*)allocMemory(sec_size);
			if( !sec ) {
				es->throwException(thisScope,getErrorCode(), NULL);
				return false;
			}

			sec->head.type	= SEC_SYMBOL;
			sec->head.size	= sec_size;
			sec->count	= cnt;

			BYTE *p = (BYTE*)&sec->symbol[0];
			for( i = 0; i < cnt; i++ ) {
				STRING *b = symbol.get(i);
				SYMBOLT *d = (SYMBOLT*)p;
				d->hash	= calcSymbolHash(getStringReadPtr(b));
				d->len	= (SHORT)getStringSize(b);
				copyChars(d->symbol, getStringReadPtr(b));
				p += sizeof(SYMBOLT) + d->len * sizeof(CHAR);
			}
			if( !addByteSequence(result, (const BYTE*)sec, sec_size) ) {
				es->throwException(thisScope,getErrorCode(), NULL);
				freeMemory(sec);
				return false;
			}
			freeMemory(sec);
		}
	}

	// xref section
	{
		if( !stackXRef.empty() ) {
			Bytes sectionData;
			sectionData.resize(sizeof(XREF_SECTION) - sizeof(XREF_LIST));
			SIZE i;
			for( i = 0; i < stackXRef.size(); i++ ) {
				const BYTES *data = stackXRef.getXRefData(i);
				sectionData.add(data);
			}
			XREF_SECTION *sec = (XREF_SECTION*)sectionData.ptr();
			sec->head.type	= SEC_XREF;
			sec->head.size	= sectionData.size();
			sec->count		= stackXRef.size();
			if( !addBytes(result, sectionData) ) {
				es->throwException(thisScope, getErrorCode(), NULL);
				return false;
			}
		}
	}

	// code section
	{
		if( !inner_code.code.empty() ) {
			CODE_SECTION cs;
			cs.head.type = SEC_CODE;
			cs.head.size = sizeof(CODE_SECTION) - sizeof(BYTE) + inner_code.code.size();
			cs.nFragment = nCodeFragment;
			if( !addByteSequence(result, (BYTE*)&cs, sizeof(CODE_SECTION) - sizeof(BYTE)) ) {
				es->throwException(thisScope, getErrorCode(), NULL);
				return false;
			}
			if( !addBytes(result, inner_code.code) ) {
				es->throwException(thisScope,getErrorCode(), NULL);
				return false;
			}
		}
	}
	return true;
}

DWORD CodeGenerator::addCData(const BYTES * bin)
{
	SIZE i;
	SIZE len = getBytesSize(bin);
	const BYTE *ptr = getBytesReadPtr(bin);
	for( i = 0; i < cdata.size(); i++ ) {
		BYTES * p = cdata.get(i);
		SIZE n = getBytesSize(p);
		if( n == len ) {
			if( memcmp(p, ptr, len) ) return (WORD)i;
		}
	}
	if( !cdata.add(bin) )
		return INVALID_SDI;
	return (WORD)i;
}

DWORD CodeGenerator::addCData(const BYTE * pByte, SIZE len)
{
	Bytes b(pByte, len);
	return addCData(b);
}

DWORD CodeGenerator::addCData(const CHAR * str)
{
	SIZE ptrLen = getCharsLength(str);
	return addCData((const BYTE *)str, sizeof(CHAR)*ptrLen);
}

DWORD CodeGenerator::addCData(const STRING * str)
{
	return addCData((const BYTE*)getStringReadPtr(str), getStringSize(str));
}

DWORD CodeGenerator::addCData(const VARIANT * val)
{
	if( val->vt == VT_STRING )
		return addCData(&val->dString);
	if( val->vt == VT_BYTES )
		return addCData(&val->dBytes);
	return (WORD)INVALID_SDI;
}

DWORD CodeGenerator::addCData(const PARSEINF * pc)
{
	return addCData(&pc->dExpression.data.v);
}

DWORD CodeGenerator::addXRef(VariableXRefList &xrefList)
{
	if( !stackXRef.add(xrefList) ) {
		return INVALID_SDI;
	}
	return stackXRef.size()-1;
}

WORD CodeGenerator::addSymbol(const PARSEINF * pc)
{
	const STRING *name = &pc->dExpression.data.v.dString;
	INT i = 0;
	if( !symbol.empty() ) {
		for( i = 0; i < (INT)symbol.size(); i++ ) {
			const STRING *s = symbol.get(i);
			if( isEqualString(s, name) ) {
				return i;
			}
		}
	}
	try {
		symbol.add(name);
	}
	catch(Error &) {
		return (WORD)INVALID_SDI;
	}
	return i;
}

WORD CodeGenerator::addSymbol(const CHAR * sym)
{
	String name(sym);
	INT i = 0;
	if( !symbol.empty() ) {
		for( i = 0; i < (INT)symbol.size(); i++ ) {
			const STRING *s = symbol.get(i);
			if( isEqualString(s, name) ) {
				return i;
			}
		}
	}
	try {
		symbol.add(name);
	}
	catch(Error &) {
		return (WORD)INVALID_SDI;
	}
	return i;
}

bool CodeGenerator::initScope()
{
	scope.defs		= 0;
	scope.xref.clear();
	scope.frame		= 0;
	scope.scopeTop	= 0;
	scope.blockTop	= 0;
	return true;
}

bool CodeGenerator::beginScope(VariableScope &st)
{
	st.defs			= variable.size();
	st.xref			= scope.xref;
	st.frame		= scope.frame;
	st.scopeTop		= scope.scopeTop;
	st.blockTop		= scope.blockTop;

	scope.defs		= 0;
	scope.xref.clear();
	scope.frame		= 0;
	scope.scopeTop	= variable.size();
	scope.blockTop	= variable.size();
	return true;
}

bool CodeGenerator::beginScopeBlock(VariableScope &st)
{
	st.defs			= variable.size();
	st.xref			= scope.xref;
	st.frame		= scope.frame;
	st.scopeTop		= scope.scopeTop;
	st.blockTop		= scope.blockTop;
	scope.blockTop	= variable.size();
	return true;
}

bool CodeGenerator::endScopeBlock(VariableScope &st)
{
	if( variable.size() > st.defs ) {
		try {
			variable.resize(st.defs);
		}
		catch(Error &) {
			return false;
		}
	}
	SIZE i;
	for( i = st.xref.size(); i < scope.xref.size(); i++ ) {
		VariableXRef *xref = scope.xref.get(i);
		if( xref->end == 0 ) xref->end = lastLine+1;
	}
	scope.blockTop	= st.blockTop;
	return true;
}

bool CodeGenerator::endScope(VariableScope &st)
{
	if( variable.size() > st.defs ) {
		try {
			variable.resize(st.defs);
		}
		catch(Error &) {
			return false;
		}
	}
	scope.xref		= st.xref;
	scope.frame		= st.frame;
	scope.scopeTop	= st.scopeTop;
	scope.blockTop	= st.blockTop;
	return true;
}

INT CodeGenerator::addVariable(const PARSEINF * pc)
{
	String name;
	if( pc->dExpression.exprType == ET_DATA )
		name = pc->dExpression.data.v.dString;
	else
		name = pc->dExpression.expPara.name;

	if( !variable.empty() ) {
		INT i;
		for( i = (INT)variable.size()-1; i >= 0 && i >= (INT)scope.blockTop; i-- ) {
			STRING *s = variable.get(i);
			if( isEqualString(s, name) ) {
				if( scope.frame <= (INT)i )
					return (INT)i-scope.frame+1;
				else
					return (INT)i-scope.frame-1;
			}
		}
	}
	try {
		variable.add(name);
	}
	catch(Error &) {
		return (INT)INVALID_VARIABLE;
	}
	INT id = (INT)variable.size()-scope.frame;
	scope.xref.add(new VariableXRef(id, name, pc->line));
	return id;
}

INT CodeGenerator::findVariable(const PARSEINF *pc)
{
	const STRING *name = &pc->dExpression.data.v.dString;
	if( !variable.empty() ) {
		INT i;
		for( i = (INT)variable.size()-1; i >= 0 && i >= (INT)scope.scopeTop; i-- ) {
			STRING *s = variable.get(i);
			if( isEqualString(s, name) ) {
				if( scope.frame <= (INT)i )
					return (INT)i-scope.frame+1;
				else
					return (INT)i-scope.frame;
			}
		}
	}
	return (INT)INVALID_VARIABLE;
}

bool CodeGenerator::addMethodRef(WORD cid, SIZE fn)
{
	return methodRef.add(cid, fn);
}

bool CodeGenerator::setImd(const PARSEINF * pc, ED &ea)
{
	return setImd(&pc->dExpression.data.v, ea);
}

bool CodeGenerator::setImd(const VARIANT * v, ED &ea)
{
	switch(v->vt) {
	case VT_NULL:
		ea.mode		= EM_NULL;
		break;
	case VT_BOOL:
		ea.mode		= EM_BOOL;
		ea.dBool	= v->dBool;
		break;
	case VT_CHAR:
		ea.mode		= EM_CHAR;
		ea.dChar	= v->dChar;
		break;
	case VT_BYTE:
		ea.mode		= EM_BYTE;
		ea.dByte	= v->dByte;
		break;
	case VT_WORD:
		ea.mode		= EM_WORD;
		ea.dWord	= v->dWord;
		break;
	case VT_DWORD:
		ea.mode		= EM_DWORD;
		ea.dDword	= v->dDword;
		break;
	case VT_SHORT:
		ea.mode		= EM_SHORT;
		ea.dShort	= v->dShort;
		break;
	case VT_INT:
		ea.mode		= EM_INT;
		ea.dInt		= v->dInt;
		break;
	case VT_DECIMAL:
		ea.mode		= EM_DECIMAL;
		ea.dDecimal	= v->dDecimal;
		break;
	case VT_FLOAT:
		ea.mode		= EM_FLOAT;
		ea.dFloat	= v->dFloat;
		break;
	case VT_BYTES:
		ea.mode		= EM_BYTES;
		ea.dBytes	= addCData(v);
		if( ea.dBytes == INVALID_SDI ) {
			es->throwException(thisScope, getErrorCode(), NULL);
			return false;
		}
		break;
	case VT_STRING:
		ea.mode		= EM_STRING;
		ea.dString	= addCData(v);
		if( ea.dString == INVALID_SDI ) {
			es->throwException(thisScope, getErrorCode(), NULL);
			return false;
		}
		break;
	default:
		es->throwException(thisScope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

bool CodeGenerator::getImd(VARIANT * v, ED &ea)
{
	VariantPtr r(v);
	switch(ea.mode) {
	case EM_NULL:		r.clear();			break;
	case EM_BOOL:		r = ea.dBool;		break;
	case EM_CHAR:		r = ea.dChar;		break;
	case EM_BYTE:		r = ea.dByte;		break;
	case EM_WORD:		r = ea.dWord;		break;
	case EM_DWORD:		r = ea.dDword;		break;
	case EM_SHORT:		r = ea.dShort;		break;
	case EM_INT:		r = ea.dInt;		break;
	case EM_DECIMAL:	r = ea.dDecimal;	break;
	case EM_FLOAT:		r = ea.dFloat;		break;
	case EM_BYTES:
		{
			const BYTES *p = cdata.get(ea.dString);
			if( !p ) {
				es->throwException(thisScope, ERROR_LOGICAL, NULL);
				return false;
			}
			r = cdata.get(ea.dBytes);
		}
		break;
	case EM_STRING:
		{
			const BYTES *p = cdata.get(ea.dString);
			if( !p ) {
				es->throwException(thisScope, ERROR_LOGICAL, NULL);
				return false;
			}
			r = (CHAR*)getBytesReadPtr(p);
		}
		break;
	default:
		es->throwException(thisScope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

bool CodeGenerator::findFreeReg(ED &r)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::protectReg(EffectiveMode r)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::freeReg(Bytecode &code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::clearRegs(Bytecode &code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::getTempFolder(ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::isLiteralExpression(const PARSEINF * pc)
{
	if( !pc ) return true;
	if( pc->type != ST_EXPRESSION ) return false;
	if( pc->dExpression.accessType == ACT_WRITE ) return false;
	switch( pc->dExpression.exprType ) {
	case ET_OPR1:
		if( !isLiteralExpression(pc->dExpression.exp1.expresion) ) return false;
		break;
	case ET_OPR2:
		if( !isLiteralExpression(pc->dExpression.exp2.expresion1) ) return false;
		if( !isLiteralExpression(pc->dExpression.exp2.expresion2) ) return false;
		break;
	case ET_OPR3:
		if( !isLiteralExpression(pc->dExpression.exp3.expresion1) ) return false;
		if( !isLiteralExpression(pc->dExpression.exp3.expresion2) ) return false;
		if( !isLiteralExpression(pc->dExpression.exp3.expresion3) ) return false;
		break;
	case ET_DATA:
		if( pc->dExpression.token == TID_SYMBOL ) return false;
		break;
	default:
		return false;
	}
	return true;
}

bool CodeGenerator::evaluateLiteral(const PARSEINF * pc, VARIANT * v)
{
	if( pc->dExpression.exprType == ET_DATA ) {
		if( !copyVariant(v, &pc->dExpression.data.v) ) {
			setError(pc);
			return false;
		}
		return true;
	}

	if( pc->dExpression.exprType == ET_OPR1 ) {
		Variant d;
		if( !evaluateLiteral(pc->dExpression.exp1.expresion, d) ) return false;
		bool r;
		switch(pc->dExpression.token) {
		case TID_FPLUS:		r = true;	break;
		case TID_FMINUS:	r = es->opNeg(v, thisScope, d);		break;
		case TID_NOT:		r = es->opNot(v, thisScope, d);		break;
		case TID_BITNOT:	r = es->opBitCPL(v, thisScope, d);	break;
		default:
			setError(pc);
			return false;
		}
		return r;
	}
	if( pc->dExpression.exprType == ET_OPR2 ) {
		const PARSEINF *lh;
		const PARSEINF *rh;
		if( pc->dExpression.associate == AM_LEFT ) {
			lh = pc->dExpression.exp2.expresion1;
			rh = pc->dExpression.exp2.expresion2;
		} else {
			rh = pc->dExpression.exp2.expresion1;
			lh = pc->dExpression.exp2.expresion2;
		}
		Variant lv;
		Variant rv;
		if( !evaluateLiteral(lh, lv) ) return false;
		if( !evaluateLiteral(rh, rv) ) return false;
		bool r;
		switch(pc->dExpression.token) {
		case TID_MUL:		r = es->opMul(v, thisScope, lv, rv);	break;
		case TID_DIV:		r = es->opDiv(v, thisScope, lv, rv);	break;
		case TID_MOD:		r = es->opMod(v, thisScope, lv, rv);	break;
		case TID_PLUS:		r = es->opAdd(v, thisScope, lv, rv);	break;
		case TID_MINUS:		r = es->opSub(v, thisScope, lv, rv);	break;
		case TID_AND:		r = es->opBitAND(v, thisScope, lv, rv);	break;
		case TID_XOR:		r = es->opBitXOR(v, thisScope, lv, rv);	break;
		case TID_OR:		r = es->opBitOR(v, thisScope, lv, rv);	break;
		case TID_LSHIFT:	r = es->opLShift(v, thisScope, lv, rv);	break;
		case TID_RSHIFT:	r = es->opRShift(v, thisScope, lv, rv);	break;
		default:
			setError(pc);
			return false;
		}
		return r;
	}
	setError(pc);
	return false;
}

bool CodeGenerator::codeGen(const PARSEINF * pc, Bytecode &code, SIZE &nCodeFragment)
{
	nCodeFragment = 0;
	Bytecode seg;
	Bytecode bin;
	if( !bin.addCodeLabel(Symbol(ES_T("@return"))) ) {
		es->throwException(thisScope, getErrorCode(), NULL);
		return false;
	}
	if( !codeStatementList(pc, bin) ) {
		return false;
	}
	if( !scope.xref.empty() ) {
		WORD cid = addXRef(scope.xref);
		if( cid == INVALID_SDI ) return false;
		if( !seg.add(IC_LINK, (WORD)scope.xref.size(), cid) ) {
			es->throwException(thisScope, getErrorCode(), NULL);
			return false;
		}
	}
	if( !seg.add(bin) ) {
		es->throwException(thisScope, getErrorCode(), NULL);
		return false;
	}
	CodeLabel *retAddr = seg.findCodeLabel(Symbol(ES_T("@return")));
	if( !retAddr ) {
		es->throwException(thisScope, getErrorCode(), NULL);
	}
	retAddr->setAddr((INT)seg.size());
	if( !scope.xref.empty() ) {
		if( !seg.add(IC_UNLINK) ) {
			es->throwException(thisScope, getErrorCode(), NULL);
			return false;
		}
		variable.clear();
	}
	if( !seg.add(IC_RET) ) {
		es->throwException(thisScope, getErrorCode(), NULL);
		return false;
	}
	if( !seg.link() ) {
		es->throwException(thisScope, getErrorCode(), NULL);
		return false;
	}
	// add code segment
	if( !code.add(seg.size()) ||
		!code.add(bin) ) {
		es->throwException(thisScope, getErrorCode(), NULL);
		return false;
	}

	for( SIZE f = 0; f < funcs.size(); f++ ) {
		Bytecode *fcode = funcs.ptr(f);
		// add code segment
		if( !code.add(fcode->size()) ||
			!code.add(*fcode) ) {
			es->throwException(thisScope, getErrorCode(), NULL);
			return false;
		}
	}
	nCodeFragment = funcs.size()+1;
	funcs.clear();
	return true;
}

bool CodeGenerator::codeStatementList(const PARSEINF * pc, Bytecode &code)
{
	while(pc) {
		if( !codeStatement(pc, code) ) {
			return false;
		}
		if( pc ) pc = pc->next;
	}
	return true;
}

bool CodeGenerator::codeStatement(const PARSEINF * pc, Bytecode& code)
{
	if( lastLine != pc->line ) {
		if( !code.add(IC_LINE, pc->line) ) {
			setError(pc);
			return false;
		}
		lastLine	= pc->line;
	}

	switch(pc->type) {
	case ST_NOP:		return true;
	case ST_END:		return true;
	case ST_BLOCK:		return codeBlock(pc, code);
	case ST_IMPORT:		return codeImport(pc, code);
	case ST_USE:		return codeUse(pc, code);
	case ST_PACKAGE:	return codePackage(pc, code);
	case ST_CLASS:		return codeClass(pc, code);
	case ST_INTERFACE:	return codeInterface(pc, code);
	case ST_TEMPLATE:	return codeTemplate(pc, code);
	case ST_OBJECT:		return codeObject(pc, code);
	case ST_IF:			return codeIf(pc, code);
	case ST_RETURN:		return codeReturn(pc, code);
	case ST_FOR:		return codeFor(pc, code);
	case ST_FORIN:		return codeForIn(pc, code);
	case ST_WHILE:		return codeWhile(pc, code);
	case ST_DO:			return codeDo(pc, code);
	case ST_SWITCH:		return codeSwitch(pc, code);
	case ST_CONTNUE:	return codeContinue(pc, code);
	case ST_BREAK:		return codeBreak(pc, code);
	case ST_TRY:		return codeTry(pc, code);
	case ST_THROW:		return codeThrow(pc, code);
	case ST_EXPRESSION:	return codeExpression(pc, code);
	}
	setError(pc);
	return false;

}

bool CodeGenerator::codeBlock(const PARSEINF * pc, Bytecode& code)
{
	VariableScope vs;
	beginScopeBlock(vs);
	PARSEINF *bpc = pc->dBlock.innerCode;
	if( !codeStatementList(bpc, code) ) {
		return false;
	}
	endScopeBlock(vs);
	return true;
}

bool CodeGenerator::codeImport(const PARSEINF * pc, Bytecode& code)
{
	WORD uix = addCData(&pc->dImport.sourceURL);
	if( uix == INVALID_SDI ) {
		setError(pc);
		return false;
	}
	if( !code.add(IC_IMPORT, uix) ) {
		setError(pc);
		return false;
	}
	return true;
}

bool CodeGenerator::codeUse(const PARSEINF * pc, Bytecode& code)
{
	const PARSEINF *addr = pc->dUse.addr;
	ED_NA ed;
	if( !codeExpression(addr, code, ed) ) return false;
	if( !code.add(IC_USE, ed) ) {
		setError(pc);
		return false;
	}
	return true;
}

bool CodeGenerator::codePackage(const PARSEINF * pc, Bytecode& code)
{
	WORD subclasses=0;
	WORD imports=0;
	WORD attrs=0;
	PARSEINF * prop = pc->dPackage.innerCode->dBlock.innerCode;
	while( prop ) {
		switch( prop->type ) {
		case ST_CLASS:
		case ST_INTERFACE:
		case ST_TEMPLATE:
			subclasses++;
			break;
		case ST_IMPORT:
			imports++;
			break;
		case ST_PROPERTY:
		case ST_FUNCTION:
			attrs++;
			break;
		case ST_END:
			break;
		default:
			setError(prop);
			return false;
		}
		prop = prop->next;
	}

	WORD sid = addSymbol(getStringReadPtr(&pc->dPackage.name));
	if( sid == INVALID_SDI ) {
		setError(pc);
		return false;
	}

	SIZE defcls_size = sizeof(CDEF_CLASS)+sizeof(WORD)*(attrs-1);
	CDEF_CLASS *defcls = (CDEF_CLASS*)allocMemory(defcls_size);
	if( !defcls ) {
		setError(pc);
		return false;
	}
	defcls->cdef.name		= sid;
	defcls->cdef.line		= pc->line;
	defcls->cdef.acm		= ACM_EMPTY;
	defcls->cdef.attrType	= AT_PACKAGE;
	defcls->attrs			= attrs;

	WORD *	cid = defcls->addr;
	Bytes 	pdef;

	if( !codeAttrDef(cid,pc->dPackage.innerCode->dBlock.innerCode) ) {
		freeMemory(defcls);
		return false;
	}

	WORD defcls_cid = addCData((const BYTE *)defcls, defcls_size);
	if( defcls_cid == INVALID_SDI ) {
		setError(pc);
		return false;
	}

	freeMemory(defcls);
	if( !code.add(IC_PAKG, defcls_cid) ) {
		setError(pc);
		return false;
	}
	if( subclasses || imports ) {
		if( !code.add(IC_PUSH, ED_RX()) ||
			!code.add(IC_MOVE, ED_R0(), ED_RX()) ) {
			setError(pc);
			return false;
		}
		prop = pc->dPackage.innerCode->dBlock.innerCode;
		while( prop ) {
			switch( prop->type ) {
			case ST_CLASS:
				if( !codeClass(prop, code)) return false;
				break;
			case ST_INTERFACE:
				if( !codeInterface(prop, code)) return false;
				break;
			case ST_TEMPLATE:
				if( !codeTemplate(prop, code)) return false;
				break;
			case ST_IMPORT:
				if( !codeImport(prop, code)) return false;
				break;
			case ST_END:
				break;
			default:
				setError(prop);
				return false;
			}
			prop = prop->next;
		}

		if( !code.add(IC_PULL, ED_RX()) ) {
			setError(pc);
			return false;
		}
	}
	return true;
}

bool CodeGenerator::codeClass(const PARSEINF * pc, Bytecode& code)
{
	WORD baseclasses=0;
	WORD interfaces=0;
	WORD attrs=0;
	if( pc->dClass.basePath != NULL ) {
		baseclasses = 1;
	}
	PARSEINF * ipc = pc->dClass.interfaces;
	while(ipc) {
		ipc = ipc->next;
		interfaces++;
	}
	PARSEINF * prop = pc->dClass.innerCode->dBlock.innerCode;
	while( prop ) {
		switch( prop->type ) {
		case ST_PROPERTY:
		case ST_FUNCTION:
			attrs++;
			break;
		case ST_END:
			break;
		default:
			setError(prop);
			return false;
		}
		prop = prop->next;
	}

	WORD sid = addSymbol(getStringReadPtr(&pc->dClass.name));
	if( sid == INVALID_SDI ) {
		setError(pc);
		return false;
	}

	SIZE defcls_size = sizeof(CDEF_CLASS)+sizeof(WORD)*(baseclasses+interfaces+attrs-1);
	CDEF_CLASS *defcls = (CDEF_CLASS*)allocMemory(defcls_size);
	if( !defcls ) {
		setError(pc);
		return false;
	}
	defcls->cdef.name		= sid;
	defcls->cdef.line		= pc->line;
	defcls->cdef.acm		= pc->dClass.acm;
	defcls->cdef.attrType	= AT_CLASS;
	defcls->attrs			= baseclasses+interfaces+attrs;

	WORD *	cid = defcls->addr;
	INT		n=0;
	if( pc->dClass.basePath ) {
		Bytes 	pdef;
		if( !codePDef(pc->dClass.basePath, NULL, pdef) ) {
			freeMemory(defcls);
			return false;
		}
		cid[n] = addCData(pdef);
		n++;
	}

	if( !codeInterfaceDef(&cid[n],pc->dClass.interfaces) ) {
		freeMemory(defcls);
		return false;
	}
	n += interfaces;

	if( !codeAttrDef(&cid[n],pc->dClass.innerCode->dBlock.innerCode) ) {
		freeMemory(defcls);
		return false;
	}

	WORD defcls_cid = addCData((const BYTE *)defcls, defcls_size);
	if( defcls_cid == INVALID_SDI ) {
		setError(pc);
		return false;
	}
	freeMemory(defcls);
	if( !code.add(IC_CLASS, defcls_cid) ) {
		setError(pc);
		return false;
	}
	return true;
}

bool CodeGenerator::codeInterface(const PARSEINF * pc, Bytecode& code)
{
	WORD interfaces=0;
	WORD attrs=0;
	PARSEINF * ipc = pc->dInterface.interfaces;
	while(ipc) {
		ipc = ipc->next;
		interfaces++;
	}
	PARSEINF * prop = pc->dInterface.innerCode->dBlock.innerCode;
	while( prop ) {
		switch( prop->type ) {
		case ST_PROPERTY:
		case ST_FUNCTION:
			attrs++;
			break;
		case ST_END:
			break;
		default:
			setError(prop);
			return false;
		}
		prop = prop->next;
	}

	WORD sid = addSymbol(getStringReadPtr(&pc->dInterface.name));
	if( sid == INVALID_SDI ) {
		setError(pc);
		return false;
	}

	SIZE defcls_size = sizeof(CDEF_CLASS)+sizeof(WORD)*(interfaces+attrs-1);
	CDEF_CLASS *defcls = (CDEF_CLASS*)allocMemory(defcls_size);
	if( !defcls ) {
		setError(pc);
		return false;
	}
	defcls->cdef.name		= sid;
	defcls->cdef.line		= pc->line;
	defcls->cdef.acm		= pc->dInterface.acm;
	defcls->cdef.attrType	= AT_INTERFACE;
	defcls->attrs			= interfaces+attrs;

	WORD *	cid = defcls->addr;
	INT		n=0;
	if( pc->dInterface.interfaces ) {
		Bytes 	pdef;
		if( !codeInterfaceDef(&cid[n],pc->dInterface.interfaces) ) {
			freeMemory(defcls);
			return false;
		}
		n += interfaces;
	}
	if( !codeAttrDef(&cid[n],pc->dInterface.innerCode->dBlock.innerCode) ) {
		freeMemory(defcls);
		return false;
	}
	WORD defcls_cid = addCData((const BYTE *)defcls, defcls_size);
	if( defcls_cid == INVALID_SDI ) {
		setError(pc);
		return false;
	}
	freeMemory(defcls);
	if( !code.add(IC_INTF, defcls_cid) ) {
		setError(pc);
		return false;
	}
	return true;
}

bool CodeGenerator::codeTemplate(const PARSEINF * pc, Bytecode& code)
{
	WORD attrs=0;
	WORD interfaces=0;
	if( pc->dTemplate.basePath != NULL ) {
		attrs++;
	}
	if( pc->dTemplate.typePath != NULL ) {
		attrs++;
	}
	PARSEINF * ipc = pc->dTemplate.interfaces;
	while(ipc) {
		interfaces++;
		ipc = ipc->next;
	}
	PARSEINF * prop = pc->dTemplate.innerCode->dBlock.innerCode;
	while( prop ) {
		switch( prop->type ) {
		case ST_PROPERTY:
		case ST_FUNCTION:
			attrs++;
			break;
		case ST_END:
			break;
		default:
			setError(prop);
			return false;
		}
		prop = prop->next;
	}

	WORD sid = addSymbol(getStringReadPtr(&pc->dTemplate.name));
	if( sid == INVALID_SDI ) {
		setError(pc);
		return false;
	}

	SIZE defcls_size = sizeof(CDEF_CLASS)+sizeof(WORD)*(interfaces+attrs-1);
	CDEF_CLASS *defcls = (CDEF_CLASS*)allocMemory(defcls_size);
	if( !defcls ) {
		setError(pc);
		return false;
	}
	defcls->cdef.name		= sid;
	defcls->cdef.line		= pc->line;
	defcls->cdef.acm		= pc->dTemplate.acm;
	defcls->cdef.attrType	= AT_CLASS;
	defcls->attrs			= interfaces+attrs;

	WORD *	cid = defcls->addr;
	INT		n=0;
	if( pc->dTemplate.basePath ) {
		Bytes 	pdef;
		if( !codePDef(pc->dTemplate.basePath, NULL, pdef) ) {
			freeMemory(defcls);
			return false;
		}
		cid[n] = addCData(pdef);
		n++;
	}
	if( pc->dTemplate.typePath ) {
		Bytes 	pdef;
		if( !codePDef(pc->dTemplate.typePath, NULL, pdef) ) {
			freeMemory(defcls);
			return false;
		}
		cid[n] = addCData(pdef);
		n++;
	}

	if( !codeAttrDef(&cid[n],pc->dTemplate.interfaces) ) {
		freeMemory(defcls);
		return false;
	}
	n += interfaces;

	if( !codeAttrDef(&cid[n],pc->dTemplate.innerCode->dBlock.innerCode) ) {
		freeMemory(defcls);
		return false;
	}

	WORD defcls_cid = addCData((const BYTE *)defcls, defcls_size);
	if( defcls_cid == INVALID_SDI ) {
		setError(pc);
		return false;
	}
	freeMemory(defcls);
	if( !code.add(IC_CLASS, defcls_cid) ) {
		setError(pc);
		return false;
	}
	return true;
}

bool CodeGenerator::codeInterfaceDef(WORD *cid, const PARSEINF * pc)
{
	const PARSEINF * ipc = pc;
	SIZE n=0;

	while(ipc) {
		Bytes pdef;
		if( !codePDef(ipc, NULL, pdef) ) {
			return false;
		}
		cid[n] = addCData(pdef);
		n++;
		ipc = ipc->next;
	}
	return true;
}

bool CodeGenerator::codeAttrDef(WORD *cid, const PARSEINF * pc)
{
	const PARSEINF * prop = pc;
	SIZE n=0;

	prop = pc;
	while( prop ) {
		WORD id;
		if( prop->type == ST_PROPERTY )	{
			if( !codePropertyDef(id, prop) ) {
				return false;
			}
		} else if( prop->type == ST_METHOD ) {
			if( !codeMethodDef(id, prop) ) {
				return false;
			}
		} else {
			es->throwException(thisScope, ERROR_LOGICAL, NULL);
		}
		cid[n++] = id;
		prop = prop->next;
	}
	return true;
}

bool CodeGenerator::codePropertyDef(WORD &id, const PARSEINF * pc)
{
	WORD si = addSymbol(getStringReadPtr(&pc->dProperty.name));
	if( si == INVALID_SDI ) {
		es->throwException(thisScope, ERROR_LOGICAL, NULL);
		return false;
	}

	Variant val;
	if( pc->dProperty.initCode ) {
		if( !evaluateLiteral(pc->dProperty.initCode, val) ) {
			return false;
		}
	}

	Bytes pdef;
	if( !codePDef(pc->dProperty.path, pc->dProperty.array, pdef) ) return false;

	CDEF_PROPERTY	def;
	def.cdef.name	= si;
	def.cdef.line	= pc->line;
	def.cdef.acm	= pc->dProperty.acm;
	def.addr		= addCData(pdef);
	def.initVT		= (WORD)val->vt;
	if( pc->dProperty.initCode ) {
		switch(val->vt) {
		case VT_NULL:		break;
		case VT_BOOL:		def.initValue.dBool		= val->dBool;		break;
		case VT_CHAR:		def.initValue.dChar		= val->dChar;		break;
		case VT_BYTE:		def.initValue.dByte		= val->dByte;		break;
		case VT_WORD:		def.initValue.dWord		= val->dWord;		break;
		case VT_DWORD:		def.initValue.dDword	= val->dDword;		break;
		case VT_SHORT:		def.initValue.dShort	= val->dShort;		break;
		case VT_INT:		def.initValue.dInt		= val->dInt;		break;
		case VT_DECIMAL:	def.initValue.dDecimal	= val->dDecimal;	break;
		case VT_FLOAT:		def.initValue.dFloat	= val->dFloat;		break;
		case VT_BYTES:		def.initValue.dBytes	= addCData(val);	break;
		case VT_STRING:		def.initValue.dString	= addCData(val);	break;
		}
	}
	id	= addCData((const BYTE *)&def, sizeof(CDEF_PROPERTY));
	return true;
}

bool CodeGenerator::codeMethodDef(WORD &id, const PARSEINF * pc)
{
	WORD si = addSymbol(getStringReadPtr(&pc->dMethod.name));
	if( si == INVALID_SDI ) {
		setError(pc);
		return false;
	}
	VariableScope vs;
	beginScope(vs);

	INT minArgs = 0;
	INT maxArgs = 0;
	INT nArgs = 0;
	if( !codeFunctionArgs(pc->dMethod.params, minArgs, maxArgs, nArgs) ) return false;
	if( pc->dMethod.innerCode ) {
		Bytecode inner_code;
		if( !codeFunctionCode(pc->dMethod.innerCode, inner_code) ) return false;

		WORD xid = INVALID_SDI;
		if( !scope.xref.empty() ) {
			INT endLine = 0;
			if( pc->next ) {
				endLine = pc->next->line;
			}
			for(INT i = 0; i < nArgs; i++ ) {
				VariableXRef *xRef = scope.xref.get(i);
				xRef->end	= endLine;
				xRef->id	= xRef->id - nArgs -1;
			}
			xid = addXRef(scope.xref);
			if( xid == INVALID_SDI ) {
				setError(pc);
				return false;
			}
		}

		Bytecode *f_code = new Bytecode();
		INT maxId = scope.xref.maxId();
		INT vars = (INT)scope.xref.size() - nArgs;
		if( maxId != 0x80000000 || nArgs || xid != INVALID_SDI ) {
			if( !f_code->add(IC_LINK, (WORD)vars, xid) ) {
				setError(pc);
				delete f_code;
				return false;
			}
		}
		if( !f_code->add(inner_code) ) {
			setError(pc);
			delete f_code;
			return false;
		}
		if( maxId != 0x80000000 || nArgs || xid != INVALID_SDI ) {
			if( !f_code->add(IC_UNLINK) ) {
				setError(pc);
				delete f_code;
				return false;
			}
		}
		if( !f_code->add(IC_RET) ) {
			setError(pc);
			delete f_code;
			return false;
		}

		if( !funcs.add(f_code) ) {
			setError(pc);
			delete f_code;
			return false;
		}
	} else {
		if( !pc->parent ||
			!pc->parent->parent ||
			pc->parent->parent->type != ST_INTERFACE ) {
			es->throwException(thisScope, ERROR_SYNTAX, NULL);
			return false;
		}
	}

	endScope(vs);

	CDEF_METHOD def;
	def.cdef.name		= si;
	def.cdef.line		= pc->line;
	def.cdef.acm		= pc->dMethod.acm;
	def.cdef.attrType	= AT_METHOD;
	def.minArgs			= minArgs;
	def.maxArgs			= maxArgs;
	def.hasResult		= pc->dMethod.hasResult;
	if( pc->dMethod.innerCode) {
		def.procAddr	= (INT)funcs.size();
	} else {
		def.procAddr	= 0;
	}
	id = addCData((const BYTE *)&def, sizeof(def));
	if( id == INVALID_SDI ) {
		setError(pc);
		return false;
	}
	if( pc->dMethod.innerCode) {
		if( !addMethodRef(id, funcs.size()) ) {
			setError(pc);
			return false;
		}
	}
	return true;
}


bool CodeGenerator::codeObject(const PARSEINF * pc, Bytecode& code)
{
	ED_NA ic_ea;
	if( pc->dObject.initExpression ) {
		if(pc->dObject.initExpression->dExpression.token == TID_EQ ) {
			if( !codeExpression(pc->dObject.initExpression->dExpression.exp2.expresion1, code, ic_ea) ) {
				return false;
			}
		} else {
			es->throwException(thisScope, ERROR_LOGICAL, NULL);
			return false;
		}
	}

	if( pc->dObject.array ) {
		ED_NA ea_adr;
		ED_R0 ea_arr;
		ED ea_zero;
		ea_zero.mode	= EM_INT;
		ea_zero.dInt	= 0;
		if( !codePathExpression(pc->dObject.path, code, ea_adr) ) return false;
		if( pc->dObject.array->type == ST_EXPRESSION &&
				pc->dObject.array->dExpression.exprType == ET_NOP ) {
			ea_arr.mode = EM_INT;
			ea_arr.dInt	= 0;
		} else {
			if(	!codeExpression(pc->dObject.array, code, ea_arr) ) return false;
		}
		if( !code.add(IC_VSCLS, ea_adr, ea_arr) ||
			!code.add(IC_NEW, ED_R0(), ea_zero) ) {
			setError(pc);
			return false;
		}
		freeReg(code, ea_adr);
		freeReg(code, ea_arr);
	} else {
		ED_R0 ea_adr;
		if( !codePathExpression(pc->dObject.path, code, ea_adr) ) return false;
		ED ea_zero;
		ea_zero.mode	= EM_INT;
		ea_zero.dInt	= 0;
		if( !code.add(IC_NEW, ea_adr, ea_zero) ) {
			setError(pc);
			return false;
		}
	}
	WORD cid	= addSymbol(getStringReadPtr(&pc->dObject.name));
	if( cid == INVALID_SDI ) {
		setError(pc);
		return false;
	}
	if( !code.add(IC_TLINK, ED_R0(), (WORD)cid) ) {
		setError(pc);
		return false;
	}

	if( pc->dObject.initExpression ) {
		if(pc->dObject.initExpression->dExpression.token == TID_EQ ) {
			if( !code.add(IC_SET, ic_ea, ED_R0()) ) {
				setError(pc);
				return false;
			}
		} else {
			es->throwException(thisScope, ERROR_LOGICAL, NULL);
			return false;
		}
		freeReg(code, ic_ea);
	}
	if( !clearRegs(code) ) return false;

	if( pc->dObject.innerCode ) {
		if( !code.add(IC_PUSH, ED_RX()) ||
			!code.add(IC_MOVE, ED_R0(), ED_RX()) ) {
			setError(pc);
			return false;
		}
		if( !codeStatement(pc->dObject.innerCode, code) ) return false;
		if( !code.add(IC_PULL, ED_RX()) ) {
			setError(pc);
			return false;
		}
	}
	return true;
}

bool CodeGenerator::codeFunction(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeFunctionArgs(const PARSEINF * pc, INT &minArgs, INT &maxArgs, INT &nArgs)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeFunctionCode(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}


bool CodeGenerator::codeReturn(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeIf(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeSwitch(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeCase(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeFor(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeForIn(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeWhile(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeDo(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeBreak(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeContinue(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeTry(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeCatch(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeFinal(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeThrow(const PARSEINF * pc, Bytecode& code)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeExpression(const PARSEINF * pc, Bytecode& code)
{
	ED_IGNR ea;
	if( !codeExpression(pc, code, ea) ) {
		return false;
	}
	if( !clearRegs(code) ) return false;
	return true;
}

bool CodeGenerator::codeExpression(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	bool ret = false;
	switch(pc->dExpression.exprType) {
	case ET_NOP:
		ret = true;
		break;
	case ET_DATA:
		ret = codeData(pc, code, ea);
		break;
	case ET_OPR1:
		ret = codeOperator1(pc, code, ea);
		break;
	case ET_OPR2:
		ret = codeOperator2(pc, code, ea);
		break;
	case ET_OPR3:
		ret = codeOperator3(pc, code, ea);
		break;
	case ET_OPRS:
		if( pc->dExpression.token == TID_CALL ) {
			ret = codeOperatorCall(pc, code, ea);
			break;
		}
		if( pc->dExpression.token == TID_INDEX ) {
			ret = codeOperatorIndex(pc, code, ea);
			break;
		}
		break;
	case ET_OPRE:
		ret = codeOperatorE(pc, code, ea);
		break;
	case ET_PATH:
		ret = codePath(pc, code, ea);
		break;
	default:
		setError(pc);
	}
	if( !ret ) return ret;
	if( ea.mode == EM_ST || ea.mode == EM_IGNR ) {
		clearRegs(code);
	}
	return true;
}

bool CodeGenerator::codeParamExpression(const PARSEINF * pc, Bytecode& code, INT &argc)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codePathExpression(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codePath(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeData(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	if( (pc->dExpression.token & TT_MASK) == TT_LITERAL ) {
		return codeLiteral(pc, code, ea);
	} else if( (pc->dExpression.token & TT_MASK) == TT_SYMBOL ) {
		return codeSymbol(pc, code, ea);
	}
	setError(pc);
	return false;
}

bool CodeGenerator::codeLiteral(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	if( pc->dExpression.accessType != ACT_READ ) {
		setError(pc);
		return false;
	}
	if( ea.mode == EM_IGNR ) return true;
	if( ea.mode == EM_NA || ea.mode == EM_R0) {
		if( !setImd(pc, ea) ) return false;
	} else {
		ED_NA lea;
		if( !setImd(pc, lea) ) return false;
		ED_RESULT(lea, pc, code, ea);
	}
	return true;
}

bool CodeGenerator::codeSymbol(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	if( ea.mode == EM_IGNR ) return true;

	if( pc->dExpression.token == TID_THIS ) {
		if( ea.mode == EM_NA ) {
			if( !getTempFolder(ea) ) {
				setError(pc);
				return false;
			}
			if( !code.add(IC_REFX, ea) ) {
				setError(pc);
				return false;
			}
		} else if( ea.mode == EM_ST ) {
			if( !code.add(IC_REFX, ED_R0()) ||
				!code.add(IC_TEST, ED_R0()) ) {
				setError(pc);
				return false;
			}
		} else {
			if( !code.add(IC_REFX, ea) ) {
				setError(pc);
				return false;
			}
		}
		return true;
	}

	if( pc->dExpression.token == TID_SUPER ) {
		if( !code.add(IC_SUPR) ) {
			setError(pc);
			return false;
		}
		ED_RESULT(ED_R0(), pc, code, ea);
		return true;
	}
	if( pc->dExpression.token == TID_SYMBOL) {
		INT vid = findVariable(pc);
		if( vid == INVALID_VARIABLE ) {
			WORD sid = addSymbol(pc);
			if( sid == INVALID_SDI ) {
				setError(pc);
				return false;
			}
			if( !code.add(IC_REF, sid) ) {
				setError(pc);
				return false;
			}
			ED_RESULT_R0(pc, code, ea);
		} else {
			const PARSEINF * p = pc->parent;
			while(p) {
				if( p->type != ST_EXPRESSION ) break;
				if( p->dExpression.token == TID_ANDEQ ) {
					setError(pc);
					return false;
				}
				p = p->parent;
			}
			if( ea.mode == EM_NA || ea.mode == EM_R0 ) {
				ea = ED_OFFSET(EM_OFP,vid);
			} else {
				ED_RESULT(ED_OFFSET(EM_OFP,vid), pc, code, ea);
			}
		}
		return true;
	}
	setError(pc);
	return false;
}

bool CodeGenerator::codeOperator1(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	if( pc->dExpression.token == TID_VAR )
		return codeVarDefiniton(pc->dExpression.exp1.expresion, code, ea);

	if( pc->dExpression.token == TID_NEW )
		return codeOperatorNew(pc->dExpression.exp1.expresion, code, ea);

	const PARSEINF * d = pc->dExpression.exp1.expresion;

	ED_R0 ea1;
	if( !codeExpression(d, code, ea1) ) return false;
	if( (ea1.mode & EM_MASK ) == EM_IMD ) {
		Variant vd;
		if( getImd(vd, ea1) ) {
			Variant v;
			bool r = false;
			IScope *expScope = es->createStack(thisScope, this);
			if( es->enterTryBlock(expScope) ) {
				r = es->op1(v, expScope, pc->dExpression.token, vd);
				es->leaveTryBlock(expScope);
			}
			//expScope->release();
			if( r ) {
				if( ea.mode == EM_IGNR ) return true;
				if( ea.mode == EM_NA || ea.mode == EM_R0) {
					if( !setImd(v, ea) ) {
						setError(pc);
						return false;
					}
				} else {
					ED_NA lea;
					if( !setImd(v, lea) ) {
						setError(pc);
						return false;
					}
					ED_RESULT(lea, pc, code, ea);
				}
				return true;
			}
		}
	}

	ICode ic;
	VariantType vt;

	switch(pc->dExpression.token) {
	case TID_PLUSLEFT:		ic = IC_INCL;	break;
	case TID_MINUSLEFT:		ic = IC_DECL;	break;
	case TID_NOT:			ic = IC_NOT;	break;
	case TID_BITNOT:		ic = IC_BITCPL;	break;
	case TID_FPLUS:			ea	= ea1;		return true;
	case TID_FMINUS:		ic = IC_NEG;	break;
	case TID_PLUSRIGHT:		ic = IC_INCR;	break;
	case TID_MINUSRIGHT:	ic = IC_DECR;	break;
	case TID_SIZEOF:		ic = IC_SZOF;	break;
	case TID_CAST_BOOL:		ic = IC_CAST;	vt = VT_BOOL;	break;
	case TID_CAST_CHAR:		ic = IC_CAST;	vt = VT_CHAR;	break;
	case TID_CAST_BYTE:		ic = IC_CAST;	vt = VT_BYTE;	break;
	case TID_CAST_WORD:		ic = IC_CAST;	vt = VT_WORD;	break;
	case TID_CAST_DWORD:	ic = IC_CAST;	vt = VT_DWORD;	break;
	case TID_CAST_SHORT:	ic = IC_CAST;	vt = VT_SHORT;	break;
	case TID_CAST_INT:		ic = IC_CAST;	vt = VT_INT;	break;
	case TID_CAST_DECIMAL:	ic = IC_CAST;	vt = VT_DECIMAL;break;
	case TID_CAST_FLOAT:	ic = IC_CAST;	vt = VT_FLOAT;	break;
	case TID_CAST_BYTES:	ic = IC_CAST;	vt = VT_BYTES;	break;
	case TID_CAST_STRING:	ic = IC_CAST;	vt = VT_STRING;	break;
	default:
		setError(pc);
		return false;
	}

	if( ic == IC_CAST ) {
		if( !code.add(ic, ea1, (WORD)vt) ) {
			setError(pc);
			return false;
		}
	} else {
		if( !code.add(ic, ea1) ) {
			setError(pc);
			return false;
		}
	}
	ED_RESULT_R0(pc, code, ea);
	return true;
}

bool CodeGenerator::codeOperator2(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeOperator3(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeOperatorCall(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeOperatorIndex(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeOperatorE(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeOperatorPath(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeOperatorNew(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codeVarDefiniton(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	if( pc->dExpression.exprType == ET_OPR2 ) {
		PARSEINF * lh = pc->dExpression.exp2.expresion1;
		PARSEINF * rh = pc->dExpression.exp2.expresion2;
		if( pc->dExpression.token == TID_EQ &&
			lh->dExpression.exprType == ET_DATA && lh->dExpression.token == TID_SYMBOL ) {
			ED_OFFSET vea(EM_OFP, addVariable(lh));
			if( !codeExpression(rh, code, vea) ) return false;
			if( ea.mode != EM_IGNR ) {
				ea = vea;
			}
			return true;
		} else if( pc->dExpression.token == TID_COM ) {
			ED_IGNR vea;
			if( !codeVarDefiniton(lh, code, ea) ) return false;
			if( !codeVarDefiniton(rh, code, vea) ) return false;
			return true;
		} else if( pc->dExpression.token == TID_ANDEQ ) {
			setError(pc);
			return false;
		}
	} else if( pc->dExpression.exprType == ET_DATA && pc->dExpression.token == TID_SYMBOL ) {
		ED_OFFSET vea(EM_OFP, addVariable(pc));
		if( ea.mode != EM_IGNR ) {
			ea = vea;
		}
		return true;
	}
	setError(pc);
	return false;
}

bool CodeGenerator::codeFunctionParam(const PARSEINF * pc, Bytecode& code, ED &ea)
{
	es->throwException(thisScope, ERROR_NOTIMPL, NULL);
	return false;
}

bool CodeGenerator::codePDef(const PARSEINF * pc, const PARSEINF * arr, Bytes &pdef)
{
	PDefCode pcode;
	WORD r;
	if( pc->dExpression.token == TID_SYMBOL ) {
		if( !PDefSymbol(pc, pcode) ) return false;
	} else if( pc->dExpression.token == TID_DOT ) {
		if( !PDefPath(pc, pcode) ) return false;
	} else if( pc->dExpression.token == TID_SPC ) {
		if( !PDefSpecialize(pc, pcode, r) ) return false;
	} else if( pc->dExpression.token == TID_INDEX ) {
		if( !PDefVectorize(pc, pcode, r) ) return false;
	} else {
		setError(pc);
		return false;
	}
	while(arr) {
		if( !isLiteralExpression(arr) ) {
			setError(arr);
			return false;
		}
		Variant vArrNum;
		if( !evaluateLiteral(arr, vArrNum) ) {
			return false;
		}
		if( vArrNum->vt != VT_INT ) {
			setError(arr);
			return false;
		}
		pcode.code.add((BYTE)PDC_VECT);
		pcode.code.add(r);
		pcode.code.add((DWORD)vArrNum->dInt);
		r = ++(pcode.r);
		arr	= arr->next;
	}
	pdef.add((WORD)pcode.code.size());
	pdef.add(pcode.code);
	return true;
}

bool CodeGenerator::PDefSymbol(const PARSEINF * pc, PDefCode &pcode)
{
	WORD sid = addSymbol(pc);
	if( sid == INVALID_SDI ) {
		return false;
	}
	pcode.code.add((BYTE)PDC_PATH);
	pcode.code.add((WORD)1);
	pcode.code.add(sid);
	return true;
}

bool CodeGenerator::PDefPath(const PARSEINF * pc, PDefCode &pcode)
{
	PDefCode pcodetmp;
	const PARSEINF * lh = pc->dExpression.exp2.expresion1;
	const PARSEINF * rh = pc->dExpression.exp2.expresion2;

	if( lh->dExpression.token == TID_SYMBOL ) {
		if( !PDefSymbol(lh, pcodetmp) ) return false;
	} else if( lh->dExpression.token == TID_DOT ) {
		if( !PDefPath(lh, pcodetmp) ) return true;
	} else {
		setError(pc);
		return false;
	}
	if( rh->dExpression.token == TID_SYMBOL ) {
		if( !PDefSymbol(rh, pcodetmp) ) return false;
	} else {
		setError(pc);
		return false;
	}
	pcode.code.add((BYTE)PDC_PATH);
	pcode.code.add((WORD)(pcodetmp.code.size()/sizeof(WORD)));
	pcode.code.add(pcodetmp.code);

	return true;
}

bool CodeGenerator::PDefSpecialize(const PARSEINF * pc, PDefCode &pcode, WORD &r)
{
	const PARSEINF * lh = pc->dExpression.exp2.expresion1;
	const PARSEINF * rh = pc->dExpression.exp2.expresion2;

	WORD lr, rr;
	if( rh->dExpression.token == TID_SYMBOL ) {
		if( !PDefSymbol(rh, pcode) ) return false;
		rr = ++(pcode.r);
	} else if( rh->dExpression.token == TID_DOT ) {
		if( !PDefPath(rh, pcode) ) return false;
		rr = ++(pcode.r);
	} else if( rh->dExpression.token == TID_SPC ) {
		if( !PDefSpecialize(rh, pcode, rr) ) return false;
	} else if( rh->dExpression.token == TID_INDEX ) {
		if( !PDefVectorize(rh, pcode, rr) ) return false;
	} else {
		setError(pc);
		return false;
	}

	if( lh->dExpression.token == TID_SYMBOL ) {
		if( !PDefSymbol(lh, pcode) ) return false;
		lr = ++(pcode.r);
	} else if( lh->dExpression.token == TID_DOT ) {
		PDefCode pcode1;
		if( !PDefPath(lh, pcode1) ) return false;
		lr = ++(pcode.r);
	} else {
		setError(pc);
		return false;
	}
	pcode.code.add((BYTE)PDC_TSCLS);
	pcode.code.add(lr);
	pcode.code.add(rr);
	r = ++(pcode.r);
	return true;
}

bool CodeGenerator::PDefVectorize(const PARSEINF * pc, PDefCode &pcode, WORD &r)
{
	const PARSEINF * lh = pc->dExpression.exp2.expresion1;
	const PARSEINF * rh = pc->dExpression.exp2.expresion2;

	WORD rr;
	if( rh->dExpression.token == TID_SYMBOL ) {
		if( !PDefSymbol(rh, pcode) ) return false;
		rr = ++(pcode.r);
	} else if( rh->dExpression.token == TID_DOT ) {
		if( !PDefPath(rh, pcode) ) return false;
		rr = ++(pcode.r);
	} else if( rh->dExpression.token == TID_SPC ) {
		if( !PDefSpecialize(rh, pcode, rr) ) return false;
	} else if( rh->dExpression.token == TID_INDEX ) {
		if( !PDefVectorize(rh, pcode, rr) ) return false;
	} else {
		setError(pc);
		return false;
	}

	if( !isLiteralExpression(lh) ) {
		setError(lh);
		return false;
	}
	Variant vArrNum=0;
	if( lh ) {
		if( !evaluateLiteral(lh, vArrNum) ) {
			return false;
		}
	}
	if( vArrNum->vt != VT_INT ) {
		setError(lh);
		return false;
	}
	pcode.code.add((BYTE)PDC_VECT);
	pcode.code.add(rr);
	pcode.code.add((DWORD)vArrNum->dInt);
	r = ++(pcode.r);
	return true;
}

} // es
