/*
 * codegen.h
 *
 *  Created on: 2012/01/25
 *      Author: tanaka
 */

#ifndef CODEGEN_H_
#define CODEGEN_H_

#include "../include/elise.h"
#include "platform.h"
#include "string.h"
#include "bytes.h"
#include "symbol.h"
#include "executeService.h"

namespace es {

class CodeLabel {
public:
	enum {
		CODE_NOTDEF = 0x7fffffff,
		CODE_NOTSET = 0x7ffffffe,
		FUNC_NOTSET = 0x7ffffffd,
	};
	CodeLabel() {
		initSymbol(&label);
		this->addr 		= CODE_NOTSET;
		this->next		= NULL;
	}
	CodeLabel(const CodeLabel &d) {
		initSymbol(&label);
		copy(d);
	}
	CodeLabel(const SYMBOL *label) {
		initSymbol(&this->label);
		copySymbol(&this->label, label);
		this->addr 		= CODE_NOTSET;
		this->next		= NULL;
	}
	CodeLabel(const SYMBOL *label, INT addr) {
		initSymbol(&this->label);
		copySymbol(&this->label, label);
		this->addr 		= addr;
		this->next		= NULL;
	}
	~CodeLabel() {
		clearSymbol(&label);
	}
	CodeLabel &copy(const CodeLabel &d) {
		if( &d != this ) {
			copySymbol(&label, &d.label);
			addr 		= d.addr;
			next		= d.next;
		}
		return *this;
	}
	void	setAddr(INT addr) {
		this->addr = addr;
	}
	void	setNext(CodeLabel *next) {
		this->next = next;
	}
	CodeLabel *getNext() const {
		return this->next;
	}
	
	SYMBOL		label;
	INT			addr;
	CodeLabel *	next;
};

class CodeLabelTable {
public:
	CodeLabelTable() {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
		for(int i=0; i<64; i++) hashDic[i] = NULL;
	}
	~CodeLabelTable() {
		clear();
	}

	bool add(const SYMBOL *label, INT addr) {
		CodeLabel *hit = find(label);
		if( hit ) {
			hit->setAddr(addr);
			return true;
		}

		if( nNodes+1 > bNodes ) {
			SIZE bNodesNew = bNodes + 10;
			CodeLabel** pNodesNew = (CodeLabel**)allocMemory(sizeof(CodeLabel*)*bNodesNew);
			if( pNodesNew == NULL ) return false;
			pNodes	= pNodesNew;
			bNodes	= bNodesNew;
		}
		CodeLabel *p = new CodeLabel(label, addr);
		pNodes[nNodes++] = p;
		DWORD hash = getSymbolHash(label);
		INT n = hash % 64;
		CodeLabel *np = hashDic[n];
		p->setNext(np);
		hashDic[n] = p;
		return true;
	}
	CodeLabel * find(const SYMBOL *label) const {
		if( nNodes == 0 ) return NULL;
		DWORD hash = getSymbolHash(label);
		INT n = hash % 64;
		CodeLabel *np = hashDic[n];
		while( np ) {
			if( isEqualSymbols(label, &np->label) ) return np;
			np = np->getNext();
		}
		return NULL;
	}
	bool resize(SIZE newSize) {
		if( nNodes == newSize ) return true;
		if( nNodes < newSize ) {
			setError(ERROR_LOGICAL);
			return false;
		}
		SIZE i;
		for( i = newSize; i < nNodes; i++ ) {
			CodeLabel *p = pNodes[i];
			DWORD hash = getSymbolHash(&p->label);
			INT n = hash/64;
			CodeLabel *pp = NULL;
			CodeLabel *np = hashDic[n];
			while( np ) {
				if( np == p ) {
					np = p->getNext();
					if( pp )
						pp->setNext(np);
					else
						hashDic[n] = np;
					break;
				}
				pp = np;
				np = np->getNext();
			}
			delete pNodes[i];
		}
		nNodes = newSize;
		return true;
	}
	void clear() {
		SIZE i;
		for( i = 0; i < nNodes; i++ ) {
			delete pNodes[i];
		}
		freeMemory(pNodes);
		for(int i=0; i<64; i++) hashDic[i] = NULL;
		pNodes = NULL;
		nNodes = 0;
		bNodes = 0;
	}
	
	void erase(SIZE n) {
		CodeLabel *p = pNodes[n];
		DWORD hash = getSymbolHash(&p->label);
		INT nx = hash/64;
		CodeLabel *pp = NULL;
		CodeLabel *np = hashDic[nx];
		while( np ) {
			if( np == p ) {
				np = p->getNext();
				if( pp )
					pp->setNext(np);
				else
					hashDic[n] = np;
				break;
			}
			pp = np;
			np = np->getNext();
		}
		delete p;
		SIZE i;
		for( i = n; i < nNodes-1; i++ ) {
			pNodes[i] = pNodes[i+1];
		}
		nNodes--;
	}
	CodeLabel *	ptr(SIZE n) {
		return pNodes[n];
	}
	const CodeLabel * ptr(SIZE n) const {
		return pNodes[n];
	}
	SIZE size() const {
		return nNodes;
	}
	bool empty() const {
		return nNodes==0;
	}
	SIZE			nNodes;
	SIZE			bNodes;
	CodeLabel**		pNodes;
	CodeLabel*		hashDic[64];
};

class Bytecode {
public:
	Bytecode();
	~Bytecode();

	void	setScopeBase(const Bytecode *base);
	BYTE *	addCode(SIZE size);
	bool	add(WORD d);
	bool	add(INT d);
	bool	add(SIZE d);
	bool	add(ICode ic);
	bool	add(const ED &ea);
	bool	add(ICode ic, const ED &ea);
	bool	add(ICode ic, const ED &ea, WORD d);
	bool	add(ICode ic, const ED &ea, WORD d, INT d2);
	bool	add(ICode ic, const ED &ea1, const ED &ea2);
	bool	add(ICode ic, const ED &ea1, const ED &ea2, const ED &ea3);
	bool	add(ICode ic, INT oa);
	bool	add(ICode ic, INT oa, INT size);
	bool	add(ICode ic, WORD d);
	bool	add(ICode ic, WORD sid, WORD arg);
	bool	add(ICode ic, WORD sid, INT offset);
	bool	add(ICode ic, WORD sid, INT minArgs, INT maxArgs, INT offset);
	bool	add(const Bytecode &d);
	CodeLabel *findCodeLabel(const SYMBOL * label) const;
	bool	addCodeLabel(const SYMBOL * label);
	bool	addCodeLabelRef(const SYMBOL *label, INT addr);
	bool	setCodeLabelAddr(const SYMBOL * label, INT addr);
	SIZE	saveCodeLabel() const;
	bool	restoreCodeLabel(SIZE savePos);
	bool	link();
	SIZE	size();
	static INT imdSize(EffectiveMode mode) {
		if( (mode & EM_MASK) == EM_IMD ) {
			switch(mode) {
			case EM_NULL:
				return 0;
			case EM_BOOL:
				return sizeof(BYTE);
			case EM_CHAR:
				return sizeof(CHAR);
			case EM_BYTE:
				return sizeof(BYTE);
			case EM_WORD:
				return sizeof(WORD);
			case EM_DWORD:
				return sizeof(DWORD);
			case EM_SHORT:
				return sizeof(SHORT);
			case EM_INT:
				return sizeof(INT);
			case EM_DECIMAL:
				return sizeof(DECIMAL);
			case EM_FLOAT:
				return sizeof(FLOAT);
			case EM_BYTES:
			case EM_STRING:
				return sizeof(DWORD);
			}
		}
		return 0;
	}

	Bytes			code;
	CodeLabelTable	labelTable;
	CodeLabelTable	labelTableRef;
	const Bytecode *scopeBase;
};

class BytecodeArray
{
public:
	BytecodeArray() {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	~BytecodeArray() {
		clear();
	}
	void clear() {
		SIZE i;
		for( i = 0; i < nNodes; i++ ) {
			delete pNodes[i];
		}
		if( pNodes ) freeMemory(pNodes);
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	bool add(Bytecode *code) {
		if( nNodes+1 > bNodes ) {
			SIZE bNodesNew = bNodes + 10;
			Bytecode** pNodesNew = (Bytecode**)resizeMemory(pNodes, sizeof(Bytecode*)*bNodesNew);
			if( pNodesNew == NULL ) return false;
			pNodes	= pNodesNew;
			bNodes	= bNodesNew;
		}
		pNodes[nNodes++] = code;
		return true;
	}
	Bytecode *ptr(SIZE n) {
		return pNodes[n];
	}

	SIZE size() const {
		return nNodes;
	}
	SIZE empty() const {
		return nNodes==0;
	}
	SIZE		nNodes;
	SIZE		bNodes;
	Bytecode**	pNodes;
};

class MethodRefArray
{
public:
	typedef struct _Node {
		WORD	cid;
		SIZE	fn;
		SIZE	offset;
	} Node;
	MethodRefArray() {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	~MethodRefArray() {
		clear();
	}
	void clear() {
		if( pNodes ) freeMemory(pNodes);
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	bool add(WORD cid, SIZE fn) {
		if( nNodes+1 > bNodes ) {
			SIZE bNodesNew = bNodes + 10;
			Node* pNodesNew = (Node*)resizeMemory(pNodes, sizeof(Node)*bNodesNew);
			if( pNodesNew == NULL ) return false;
			pNodes	= pNodesNew;
			bNodes	= bNodesNew;
		}
		pNodes[nNodes].cid = cid;
		pNodes[nNodes++].fn	= fn;
		return true;
	}
	Node *find(WORD cid) {
		for( SIZE n = 0; n < nNodes; n++ ) {
			if( pNodes[n].cid == cid ) return &pNodes[n];
		}
		return NULL;
	}
	Node *find(SIZE fn) {
		for( SIZE n = 0; n < nNodes; n++ ) {
			if( pNodes[n].fn == fn ) return &pNodes[n];
		}
		return NULL;
	}
	SIZE size() const {
		return nNodes;
	}
	SIZE empty() const {
		return nNodes==0;
	}
	SIZE		nNodes;
	SIZE		bNodes;
	Node*		pNodes;
};

class VariableXRef {
public:
	VariableXRef(INT id, STRING * symbol) {
		initString(&this->symbol);
		this->start		= 0;
		this->end		= 0;
		this->id		= id;
		copyString(&this->symbol, symbol);
	}
	VariableXRef(INT id, STRING * symbol, INT start) {
		initString(&this->symbol);
		this->start		= start;
		this->end		= 0;
		this->id		= id;
		copyString(&this->symbol, symbol);
	}
	VariableXRef(const VariableXRef &d) {
		initString(&this->symbol);
		this->start		= d.start;
		this->end		= d.end;
		this->id		= d.id;
		copyString(&this->symbol, &d.symbol);
	}
	VariableXRef &operator=(const VariableXRef &d) {
		if( this != &d ) {
			this->start		= d.start;
			this->end		= d.end;
			this->id		= d.id;
			copyString(&this->symbol, &d.symbol);
		}
		return *this;
	}
	~VariableXRef() {
		clearString(&this->symbol);
	}
	INT		start;
	INT		end;
	INT		id;
	STRING	symbol;
};

class VariableXRefList {
public:
	VariableXRefList() {
		nMax	= 0x80000000;
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	~VariableXRefList() {
		clear();
	}
	void clear() {
		if( pNodes ) {
			SIZE i;
			for( i = 0; i < nNodes; i++ ) {
				delete pNodes[i];
			}
			freeMemory(pNodes);
		}
		nMax	= 0x80000000;
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	bool add(VariableXRef *ptr) {
		if( nNodes+1 > bNodes ) {
			SIZE bNodesNew = bNodes + 20;
			VariableXRef** pNodesNew = (VariableXRef**)resizeMemory(pNodes, sizeof(VariableXRef*)*bNodesNew);
			if( pNodesNew == NULL ) return false;
			pNodes	= pNodesNew;
			bNodes	= bNodesNew;
		}
		pNodes[nNodes++] = ptr;
		if( nMax < ptr->id ) nMax = ptr->id;
		return true;
	}
	SIZE size() const {
		return nNodes;
	}
	INT maxId() const {
		return nMax;
	}
	SIZE empty() const {
		return nNodes==0;
	}
	VariableXRef *get(SIZE i) {
		if( i < 0 || nNodes <= i ) return NULL;
		return pNodes[i];
	}
	VariableXRefList &operator=(const VariableXRefList &d) {
		if( &d != this ) {
			if( nNodes ) clear();
			pNodes = (VariableXRef**)allocMemory(sizeof(VariableXRef*)*d.bNodes);
			bNodes = d.bNodes;
			for( nNodes = 0; nNodes < d.nNodes; nNodes++ ) {
				pNodes[nNodes] = new VariableXRef(*d.pNodes[nNodes]);
				if( nMax < pNodes[nNodes]->id ) nMax = pNodes[nNodes]->id;
			}
		}
		return *this;
	}
	INT				nMax;
	SIZE			nNodes;
	SIZE			bNodes;
	VariableXRef**	pNodes;
};

class StackXRefArray {
public:
	class Node {
	public:
		Node() {
		}
		bool load(VariableXRefList &xrefList) {
			if( xrefList.empty() ) {
				return true;
			}
			try {
				SIZE count = xrefList.size();
				xrefData.resize(sizeof(XREF_LIST) - sizeof(XREF));
				SIZE i;
				for( i = 0; i < count; i++ ) {
					Bytes ent;
					VariableXRef *ptr = xrefList.get(i);
					WORD len = (WORD)getStringSize(&ptr->symbol);
					ent.resize(sizeof(XREF)+len*sizeof(CHAR));
					XREF * pRef = (XREF *)ent.ptr();
					pRef->startLine	= ptr->start;
					pRef->endLine	= ptr->end;
					pRef->offset	= ptr->id;
					pRef->size		= len;
					copyChars(pRef->symbol, getStringReadPtr(&ptr->symbol));
					xrefData.add(ent);
				}
				XREF_LIST * list = (XREF_LIST*)xrefData.ptr();
				list->size	= xrefData.size();
				list->count	= count;
			}
			catch(...) {
				return false;
			}
			return true;
		}
		Bytes			xrefData;
	};
	StackXRefArray() {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	~StackXRefArray() {
		clear();
	}
	void clear() {
		if( pNodes ) {
			SIZE i;
			for( i = 0; i < nNodes; i++ ) delete pNodes[i];
			freeMemory(pNodes);
		}
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	bool add(VariableXRefList &xref) {
		if( nNodes+1 > bNodes ) {
			SIZE bNodesNew = bNodes + 10;
			Node** pNodesNew = (Node**)resizeMemory(pNodes, sizeof(Node*)*bNodesNew);
			if( pNodesNew == NULL ) return false;
			pNodes	= pNodesNew;
			bNodes	= bNodesNew;
		}
		Node *n = new Node();
		if( !n->load(xref) ) return false;
		pNodes[nNodes++] = n;
		return true;
	}
	SIZE size() const {
		return nNodes;
	}
	const BYTES * getXRefData(SIZE i) const {
		if( i < 0 || nNodes <= i ) return NULL;
		return pNodes[i]->xrefData;
	}
	bool empty() const {
		return nNodes==0;
	}
	SIZE		nNodes;
	SIZE		bNodes;
	Node**			pNodes;
};

enum {
	INVALID_SDI		 = 0x8000,
	INVALID_VARIABLE = 0x80000000,
};

class ED_NA : public ED {
public:
	ED_NA() {
		this->mode		= EM_NA;
	}
};

class ED_IGNR : public ED {
public:
	ED_IGNR() {
		this->mode		= EM_IGNR;
	}
};

class ED_R0 : public ED {
public:
	ED_R0() {
		this->mode		= EM_R0;
	}
};
class ED_RX : public ED {
public:
	ED_RX() {
		this->mode		= EM_RX;
	}
};
class ED_ST : public ED {
public:
	ED_ST() {
		this->mode		= EM_ST;
	}
};
class ED_EX : public ED {
public:
	ED_EX() {
		this->mode		= EM_EX;
	}
};
class ED_SP : public ED {
public:
	ED_SP(INT offset) {
		this->mode		= EM_OSP;
		this->offset	= offset;
	}
};
class ED_FP : public ED {
public:
	ED_FP(INT offset) {
		this->mode		= EM_OFP;
		this->offset	= offset;
	}
};
class ED_OFFSET : public ED {
public:
	ED_OFFSET(EffectiveMode em, INT offset) {
		this->mode		= em;
		this->offset	= offset;
	}
};
class ED_NULL : public ED {
public:
	ED_NULL() {
		this->mode		= EM_NULL;
	}
};
class ED_CDATA : public ED {
public:
	ED_CDATA(EffectiveMode em, DWORD id) {
		this->mode		= em;
		this->dString	= id;
	}
};

class VariableScope {
public:
	VariableScope() {
		defs		= 0;
		frame		= 0;
		scopeTop	= 0;
		blockTop	= 0;
	}
	~VariableScope() {

	}
	SIZE			defs;
//		SIZE			vars;
	VariableXRefList	xref;
	INT				frame;
	SIZE			scopeTop;
	SIZE			blockTop;
};

class PDefCode {
public:
	PDefCode() { r = 0; }
	WORD	r;
	Bytes	code;
};


class CodeGenerator
{
public:
	CodeGenerator(IContext *ctx);
	~CodeGenerator();
	
	bool generate(BYTES *result, const IScope *scope, const PARSEINF *pInf);

protected:
	enum {
		INVALID_SDI		 = 0x8000,
		INVALID_VARIABLE = 0x80000000,
	};

	void setError(const PARSEINF * pc);

	DWORD addCData(const BYTES * bin);
	DWORD addCData(const BYTE * pByte, SIZE len);
	DWORD addCData(const CHAR * str);
	DWORD addCData(const STRING * str);
	DWORD addCData(const VARIANT * val);
	DWORD addCData(const PARSEINF * pc);
	DWORD addXRef(VariableXRefList &xrefList);
	bool addMethodRef(WORD cid, SIZE fn);
	bool setImd(const PARSEINF * pc, ED &ea);
	bool setImd(const VARIANT * v, ED &ea);
	bool getImd(VARIANT * v, ED &ea);
	bool findFreeReg(ED &r);
	bool protectReg(EffectiveMode r);
	bool freeReg(Bytecode &code, ED &ea);
	bool clearRegs(Bytecode &code);
	bool getTempFolder(ED &ea);
	bool evaluateLiteral(const PARSEINF * pc, VARIANT * v);

	WORD addSymbol(const PARSEINF * pc);
	WORD addSymbol(const CHAR * sym);

	bool initScope();
	bool beginScope(VariableScope &st);
	bool beginScopeBlock(VariableScope &st);
	bool endScopeBlock(VariableScope &st);
	bool endScope(VariableScope &st);
	INT	addVariable(const PARSEINF * pc);
	INT	findVariable(const PARSEINF * pc);
	const PARSEINF * findLastCode(const PARSEINF * pc);
	bool isLiteralExpression(const PARSEINF * pc);

	bool codeGen(const PARSEINF * pc, Bytecode &code, SIZE &nCodeFragment);
	bool codeStatementList(const PARSEINF * pc, Bytecode &code);
	bool codeStatement(const PARSEINF * pc, Bytecode& code);
	bool codeBlock(const PARSEINF * pc, Bytecode& code);
	bool codeImport(const PARSEINF * pc, Bytecode& code);
	bool codeUse(const PARSEINF * pc, Bytecode& code);
	bool codePackage(const PARSEINF * pc, Bytecode& code);
	bool codeClass(const PARSEINF * pc, Bytecode& code);
	bool codeInterface(const PARSEINF * pc, Bytecode& code);
	bool codeTemplate(const PARSEINF * pc, Bytecode& code);
	bool codeInterfaceDef(WORD *cid, const PARSEINF * pc);
	bool codeAttrDef(WORD *cid, const PARSEINF * pc);
	bool codePropertyDef(WORD &id, const PARSEINF * pc);
	bool codeMethodDef(WORD &id, const PARSEINF * pc);
	bool codeObject(const PARSEINF * pc, Bytecode& code);
	bool codeFunction(const PARSEINF * pc, Bytecode& code);
	bool codeFunctionArgs(const PARSEINF * pc, INT &minArgs, INT &maxArgs, INT &nArgs);
	bool codeFunctionCode(const PARSEINF * pc, Bytecode& code);
	bool codeReturn(const PARSEINF * pc, Bytecode& code);
	bool codeIf(const PARSEINF * pc, Bytecode& code);
	bool codeSwitch(const PARSEINF * pc, Bytecode& code);
	bool codeCase(const PARSEINF * pc, Bytecode& code);
	bool codeFor(const PARSEINF * pc, Bytecode& code);
	bool codeForIn(const PARSEINF * pc, Bytecode& code);
	bool codeWhile(const PARSEINF * pc, Bytecode& code);
	bool codeDo(const PARSEINF * pc, Bytecode& code);
	bool codeBreak(const PARSEINF * pc, Bytecode& code);
	bool codeContinue(const PARSEINF * pc, Bytecode& code);
	bool codeTry(const PARSEINF * pc, Bytecode& code);
	bool codeCatch(const PARSEINF * pc, Bytecode& code);
	bool codeFinal(const PARSEINF * pc, Bytecode& code);
	bool codeThrow(const PARSEINF * pc, Bytecode& code);
	bool codeExpression(const PARSEINF * pc, Bytecode& code);

	bool codeExpression(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeParamExpression(const PARSEINF * pc, Bytecode& code, INT &argc);
	bool codePathExpression(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codePath(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeData(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeLiteral(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeSymbol(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeOperator1(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeOperator2(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeOperator3(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeOperatorCall(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeOperatorIndex(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeOperatorE(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeOperatorPath(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeOperatorNew(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeVarDefiniton(const PARSEINF * pc, Bytecode& code, ED &ea);
	bool codeFunctionParam(const PARSEINF * pc, Bytecode& code, ED &ea);

	bool codePDef(const PARSEINF * pc, const PARSEINF * arr, Bytes &pdef);
	bool PDefSymbol(const PARSEINF * pc, PDefCode &pcode);
	bool PDefPath(const PARSEINF * pc, PDefCode &pcode);
	bool PDefSpecialize(const PARSEINF * pc, PDefCode &pcode, WORD &r);
	bool PDefVectorize(const PARSEINF * pc, PDefCode &pcode, WORD &r);

	IContext *			ctx;

	IExecuteService*	es;
	ITypeManager *		tm;

	const PINF_PROGRAM *pcProgram;
	IScope * 			thisScope;
	SIGNATURE_SECTION	pSig;
	StringArray			variable;
	StringArray			symbol;
	VariableScope		scope;
	BytesArray			cdata;
	BytecodeArray		funcs;
	MethodRefArray		methodRef;
	StackXRefArray		stackXRef;
	WORD				protectedRegs;
	WORD				usedRegs;
	INT					tempSeq;
	INT					lastLine;
	IURL *				pSourceURL;
};

} // es

#endif /* CODEGEN_H_ */
