/*
 * datetime.cpp
 *
 *  Created on: 2012/01/25
 *      Author: tanaka
 */

#include "datetime.h"

namespace es {

static int daysOfYear[] = {
/* 1800-1809 */	365,365,365,365,366,365,365,365,366,365,
/* 1810-1819 */	365,365,366,365,365,365,366,365,365,365,
/* 1820-1829 */	366,365,365,365,366,365,365,365,366,365,
/* 1830-1839 */	365,365,366,365,365,365,366,365,365,365,
/* 1840-1849 */	366,365,365,365,366,365,365,365,366,365,
/* 1850-1859 */	365,365,366,365,365,365,366,365,365,365,
/* 1860-1869 */	366,365,365,365,366,365,365,365,366,365,
/* 1870-1879 */	365,365,366,365,365,365,366,365,365,365,
/* 1880-1889 */	366,365,365,365,366,365,365,365,366,365,
/* 1890-1899 */	365,365,366,365,365,365,366,365,365,365,
/* 1900-1909 */	365,365,365,365,366,365,365,365,366,365,
/* 1910-1919 */	365,365,366,365,365,365,366,365,365,365,
/* 1920-1929 */	366,365,365,365,366,365,365,365,366,365,
/* 1930-1939 */	365,365,366,365,365,365,366,365,365,365,
/* 1940-1949 */	366,365,365,365,366,365,365,365,366,365,
/* 1950-1959 */	365,365,366,365,365,365,366,365,365,365,
/* 1960-1969 */	366,365,365,365,366,365,365,365,366,365,
/* 1970-1979 */	365,365,366,365,365,365,366,365,365,365,
/* 1980-1989 */	366,365,365,365,366,365,365,365,366,365,
/* 1990-1999 */	365,365,366,365,365,365,366,365,365,365,
/* 2000-2009 */	366,365,365,365,366,365,365,365,366,365,
/* 2010-2019 */	365,365,366,365,365,365,366,365,365,365,
/* 2020-2029 */	366,365,365,365,366,365,365,365,366,365,
/* 2030-2039 */	365,365,366,365,365,365,366,365,365,365,
/* 2040-2049 */	366,365,365,365,366,365,365,365,366,365,
/* 2050-2059 */	365,365,366,365,365,365,366,365,365,365,
/* 2060-2069 */	366,365,365,365,366,365,365,365,366,365,
/* 2070-2079 */	365,365,366,365,365,365,366,365,365,365,
/* 2080-2089 */	366,365,365,365,366,365,365,365,366,365,
/* 2090-2099 */	365,365,366,365,365,365,366,365,365,365,
/* 2100-2109 */	365,365,365,365,366,365,365,365,366,365,
/* 2110-2119 */	365,365,366,365,365,365,366,365,365,365,
/* 2120-2129 */	366,365,365,365,366,365,365,365,366,365,
/* 2130-2139 */	365,365,366,365,365,365,366,365,365,365,
/* 2140-2149 */	366,365,365,365,366,365,365,365,366,365,
/* 2150-2159 */	365,365,366,365,365,365,366,365,365,365,
/* 2160-2169 */	366,365,365,365,366,365,365,365,366,365,
/* 2170-2179 */	365,365,366,365,365,365,366,365,365,365,
/* 2180-2189 */	366,365,365,365,366,365,365,365,366,365,
/* 2190-2199 */	365,365,366,365,365,365,366,365,365,365,
/* 2200-2209 */	365,365,365,365,366,365,365,365,366,365,
/* 2210-2219 */	365,365,366,365,365,365,366,365,365,365,
/* 2220-2229 */	366,365,365,365,366,365,365,365,366,365,
/* 2230-2239 */	365,365,366,365,365,365,366,365,365,365,
/* 2240-2249 */	366,365,365,365,366,365,365,365,366,365,
/* 2250-2259 */	365,365,366,365,365,365,366,365,365,365,
/* 2260-2269 */	366,365,365,365,366,365,365,365,366,365,
/* 2270-2279 */	365,365,366,365,365,365,366,365,365,365,
/* 2280-2289 */	366,365,365,365,366,365,365,365,366,365,
/* 2290-2299 */	365,365,366,365,365,365,366,365,365,365,
/* 2300-2309 */	365,365,365,365,366,365,365,365,366,365,
};

static INT daysOfMonth[12] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

static const CHAR * weekName3[] = {
	ES_T("Sun"), ES_T("Mon"), ES_T("Tue"), ES_T("Wed"), ES_T("Thu"), ES_T("Fri"), ES_T("Sat")
};

static const CHAR * monthName3[] = {
	ES_T("Jan"), ES_T("Feb"), ES_T("Mar"), ES_T("Apr"), ES_T("May"), ES_T("Jun"), ES_T("Jul"), ES_T("Aug"), ES_T("Sep"), ES_T("Oct"), ES_T("Nov"), ES_T("Dec")
};

#define SPLITDATE(DT) (((DT)>>32))
#define SPLITTIME(DT) ((DT)&0xffffffff)
#define MAKEDATETIME(D,T) (((D)<<32)|((T)&0xffffffff))
#define RANGE_FROM 0x0000000100000000
#define RANGE_TO  0x0002c95d0001517f

DATETIME getDateRangeFrom()
{
	return (DATETIME)RANGE_FROM;
}

DATETIME getDateRangeTo()
{
	return (DATETIME)RANGE_TO;
}

DATETIME getDateTime()
{
	SYSTEMTIME	st;
	GetSystemTime(&st);
	DATETIMEELEMENTS dte;
	dte.date.year	= st.wYear;
	dte.date.month	= st.wMonth;
	dte.date.date	= st.wDay;
	dte.time.hour	= st.wHour;
	dte.time.minute	= st.wMinute;
	dte.time.second	= st.wSecond;
	return makeDateTime(&dte);
}

DATEMARKER makeDateMarker1(INT month, INT day, const TIMEELEMENTS * timeElements)
{
	DATETIMEELEMENTS elm;
	elm.date.year	= 1800;
	elm.date.month	= month;
	elm.date.date	= day;
	if( timeElements ) {
		elm.time		= *timeElements;
	} else {
		elm.time.hour	= 0;
		elm.time.minute	= 0;
		elm.time.second	= 0;
	}
	return (DATEMARKER)makeDateTime(&elm);
}

DATEMARKER makeDateMarker2(INT month, INT dayOfWeek, INT nWeek, const TIMEELEMENTS * timeElements)
{
	if( dayOfWeek < 0 || 6 < dayOfWeek ) {
		setError(ERROR_PARAMETER);
		return 0;
	}
	if( nWeek < 0 || 5 < nWeek ) {
		setError(ERROR_PARAMETER);
		return 0;
	}

	DATETIMEELEMENTS elm;
	elm.date.year	= 1801 + dayOfWeek;
	elm.date.month	= month;
	elm.date.date	= nWeek;
	if( timeElements ) {
		elm.time		= *timeElements;
	} else {
		elm.time.hour	= 0;
		elm.time.minute	= 0;
		elm.time.second	= 0;
	}
	return (DATEMARKER)makeDateTime(&elm);
}

DATETIME makeMarkerDate(DATEMARKER dateMarker, INT year)
{
	if( year < 1800 || 2300 < year ) {
		setError(ERROR_PARAMETER);
		return 0;
	}
	if( dateMarker == 0 ) {
		setError(ERROR_PARAMETER);
		return 0;
	}

	DATETIMEELEMENTS dateTimeElements;
	if( !splitDateTime(dateMarker, &dateTimeElements) ) return 0;

	if( dateTimeElements.date.year == 1800 ) {
		dateTimeElements.date.year	= year;
	} else {
		INT dayOfweek	= dateTimeElements.date.year - 1801;
		INT nWeek		= dateTimeElements.date.date;
		DATETIME date	= getWeekDate(year, dateTimeElements.date.month, dayOfweek, nWeek);
		if( date == 0 ) {
			return 0;
		}
		DATETIMEELEMENTS dateElements;
		if( !splitDateTime(date, &dateElements) ) return 0;
		dateTimeElements.date	= dateElements.date;
	}
	return makeDateTime(&dateTimeElements);
}

DATETIME makeDate(const DATEELEMENTS * dateElements)
{
	INT year	= dateElements->year;
	INT month	= dateElements->month;
	INT day		= dateElements->date;

	if( (year  < 1800 || 2300 <= year) ||
		(month < 1    ||   12 < month) ||
		(day   < 1    ||   31 <   day) ) {
		setError(ERROR_PARAMETER);
		return 0;
	}
	if( getDaysOfMonth(year, month) < day ) {
		setError(ERROR_PARAMETER);
		return 0;
	}

	// Normalize
	year	-= 1800;
	month	-= 1;
	day		-= 1;

	// day count from 1800/1/1
	DATETIME   dates = 0;
	const INT * doy = daysOfYear;
	for( INT yi = 0; yi < year; yi++ ) {
		dates += *doy++;
	}
	for( INT i = 0; i < month; i++ ) dates += daysOfMonth[i];
	if( dateElements->month >= 2 && isLeapYear(year) ) {
		dates++;
	}
	dates += day;

	DATETIME d = MAKEDATETIME(dates,0);
	return d;
}

DATETIME makeDateTime(const DATETIMEELEMENTS * dateTimeElements)
{
	INT year	= dateTimeElements->date.year;
	INT month	= dateTimeElements->date.month;
	INT day		= dateTimeElements->date.date;

	if( (year  < 1800 || 2300 <= year) ||
		(month < 1    ||   12 < month) ||
		(day   < 1    ||   31 <   day) ) {
		setError(ERROR_PARAMETER);
		return 0;
	}
	if( getDaysOfMonth(year, month) < day ) {
		setError(ERROR_PARAMETER);
		return 0;
	}

	INT hour	= dateTimeElements->time.hour;
	INT min		= dateTimeElements->time.minute;
	INT sec		= dateTimeElements->time.second;
	if( (hour < 0 || 23 < hour) ||
		(min  < 0 || 59 < min ) ||
		(sec  < 0 || 59 < sec ) ) {
		setError(ERROR_PARAMETER);
		return 0;
	}

	// Normalize
	year	-= 1800;
	month	-= 1;
	day		-= 1;

	// day count from 1800/1/1
	DATETIME   dates = 0;
	const INT * doy = daysOfYear;
	for( INT yi = 0; yi < year; yi++ ) {
		dates += *doy++;
	}

	for( INT i = 0; i < month; i++ ) dates += daysOfMonth[i];
	if( dateTimeElements->date.month >= 2 && isLeapYear(year) ) {
		dates++;
	}
	dates += day;

	// time
	DATETIME times = hour * 60 * 60 + min * 60 + sec;

	DATETIME d = MAKEDATETIME(dates,times);
	return d;
}

DATETIME makeDateTimeNC(const DATETIMEELEMENTS * dateTimeElements)
{
	INT year	= dateTimeElements->date.year;
	INT month	= dateTimeElements->date.month;
	INT day		= dateTimeElements->date.date;

	if( (year  < 1800 || 2300 <= year) ||
		(month < 1    ||   12 < month) ||
		(day   < 1    ||   31 <   day) ) {
		setError(ERROR_PARAMETER);
		return 0;
	}
	if( getDaysOfMonth(year, month) < day ) {
		setError(ERROR_PARAMETER);
		return 0;
	}

	INT hour	= dateTimeElements->time.hour;
	INT min		= dateTimeElements->time.minute;
	INT sec		= dateTimeElements->time.second;

	// Normalize
	year	-= 1800;
	month	-= 1;
	day		-= 1;

	// day count from 1800/1/1
	DATETIME   dates = 0;
	const INT * doy = daysOfYear;
	for( INT yi = 0; yi < year; yi++ ) {
		dates += *doy++;
	}

	for( INT i = 0; i < month; i++ ) dates += daysOfMonth[i];
	if( dateTimeElements->date.month >= 2 && isLeapYear(year) ) {
		dates++;
	}
	dates += day;

	// time
	DATETIME times = hour * 60 * 60 + min * 60 + sec;

	DATETIME d = MAKEDATETIME(dates,times);
	return d;
}

bool splitDateTime(DATETIME dateTime, DATETIMEELEMENTS *dateTimeElements)
{
	if( !splitDate(dateTime, &dateTimeElements->date) ||
		!splitTime(dateTime, &dateTimeElements->time) ) {
		return false;
	}
	return true;
}

bool splitDate(DATETIME dateTime, DATEELEMENTS *dateElements)
{
	dateElements->year		= 0;
	dateElements->month		= 0;
	dateElements->date		= 0;
	dateElements->dayOfWeek	= 0;

	if( dateTime <= 0 ) {
		setError(ERROR_PARAMETER);
		return false;
	}

	TIMESPAN date = SPLITDATE(dateTime);

	// year
	TIMESPAN pass=0;
	const INT * doy = daysOfYear;
	while( pass + *doy <= date ) {
		pass += *doy++;
		dateElements->year++;
	}

	// month
	TIMESPAN yday = date - pass;
	TIMESPAN mdays=0;
	while( mdays + daysOfMonth[dateElements->month] <= yday) {
		mdays += daysOfMonth[dateElements->month++];
	}
	if( dateElements->month >= 2 && isLeapYear(dateElements->year) ) {
		mdays++;
	}

	// day
	dateElements->date = (INT)(yday - mdays);

	// Normalize
	dateElements->year  += 1800;
	dateElements->month += 1;
	dateElements->date  += 1;

	// week
	TIMESPAN zy = dateElements->month <= 2 ? dateElements->year - 1   : dateElements->year;
	TIMESPAN zm = dateElements->month <= 2 ? dateElements->month + 12 : dateElements->month;
	dateElements->dayOfWeek = (zy + zy/4 - zy/100 + zy/400 + (13*zm+8)/5 + dateElements->date)%7;
	return true;
}

bool splitTime(DATETIME dateTime, TIMEELEMENTS *timeElements)
{
	timeElements->hour		= 0;
	timeElements->minute	= 0;
	timeElements->second	= 0;
	if( dateTime <= 0 ) {
		setError(ERROR_PARAMETER);
		return false;
	}

	INT time = SPLITTIME(dateTime);
	if( time != 0 ) {
		timeElements->hour		= time / 3600;
		timeElements->minute	= time % 3600 / 60;
		timeElements->second	= time % 60;
	}
	return true;
}

INT getYear(DATETIME dateTime)
{
	DATEELEMENTS dateElements;
	if( dateTime <= 0 ) return 0;
	splitDate(dateTime, &dateElements);
	return dateElements.year;
}

INT getMonth(DATETIME dateTime)
{
	DATEELEMENTS dateElements;
	if( dateTime <= 0 ) return 0;
	splitDate(dateTime, &dateElements);
	return dateElements.month;
}

INT getDay(DATETIME dateTime)
{
	DATEELEMENTS dateElements;
	if( dateTime <= 0 ) return 0;
	splitDate(dateTime, &dateElements);
	return dateElements.date;
}

INT getDayOfWeek(DATETIME dateTime)
{
	DATEELEMENTS dateElements;
	if( dateTime <= 0 ) return 0;
	splitDate(dateTime, &dateElements);
	return dateElements.dayOfWeek;
}

INT getHour(DATETIME dateTime)
{
	TIMEELEMENTS timeElements;
	if( dateTime <= 0 ) return 0;
	splitTime(dateTime, &timeElements);
	return timeElements.hour;
}

INT getMinute(DATETIME dateTime)
{
	TIMEELEMENTS timeElements;
	if( dateTime <= 0 ) return 0;
	splitTime(dateTime, &timeElements);
	return timeElements.minute;
}

INT getSecond(DATETIME dateTime)
{
	TIMEELEMENTS timeElements;
	if( dateTime <= 0 ) return 0;
	splitTime(dateTime, &timeElements);
	return timeElements.second;
}

INT getDayOfYear(DATETIME dateTime)
{
	DATEELEMENTS dateElements;
	if( dateTime <= 0 ) return 0;
	splitDate(dateTime, &dateElements);
	INT days = 0;
	INT m;
	for( m = 1; m < dateElements.month; m++ ) {
		days += getDaysOfMonth(dateElements.year, m);
	}
	days += dateElements.date;
	return days;
}

TIMESPAN diffTime(DATETIME dt0, DATETIME dt1)
{
	if( dt0 <= 0 || dt1 <= 0 ) {
		setError(ERROR_PARAMETER);
		return 0;
	}
	TIMESPAN d0 = SPLITDATE(dt0);
	TIMESPAN t0 = SPLITTIME(dt0);

	TIMESPAN d1 = SPLITDATE(dt1);
	TIMESPAN t1 = SPLITTIME(dt1);

	TIMESPAN s0 = d0*24*60*60+t0;
	TIMESPAN s1 = d1*24*60*60+t1;
	return s0-s1;
}

DATETIME addTime(DATETIME baseTime, TIMESPAN span)
{
	if( span < 0 ) return subTime(baseTime, -span);
	if( baseTime <= 0 ) {
		setError(ERROR_PARAMETER);
		return 0;
	}

	TIMESPAN bd = SPLITDATE(baseTime);
	TIMESPAN bt = SPLITTIME(baseTime);

	TIMESPAN sd = span / (24*60*60);
	TIMESPAN st = span % (24*60*60);

	TIMESPAN rt = (bt + st) % (24*60*60);
	if( rt >= (24*60*60) ) {
		rt -= (24*60*60);
		sd += 1;
	}
	TIMESPAN rd = (bd + sd) + (bt + st) / (24*60*60);
	if( rd <= 0 || 0x2c95c < rd ) {
		setError(ERROR_DATERANGE);
		return 0;
	}
	DATETIME r = MAKEDATETIME(rd, rt);
	if( r < RANGE_FROM || RANGE_TO < r ) {
		setError(ERROR_DATERANGE);
		return 0;
	}
	return r;
}

DATETIME subTime(DATETIME baseTime, TIMESPAN span)
{
	if( span < 0 ) return addTime(baseTime, -span);
	if( baseTime <= 0 ) {
		setError(ERROR_PARAMETER);
		return 0;
	}

	TIMESPAN bd = SPLITDATE(baseTime);
	TIMESPAN bt = SPLITTIME(baseTime);

	TIMESPAN sd = span / (24*60*60);
	TIMESPAN st = span % (24*60*60);

	TIMESPAN rt = (bt - st) % (24*60*60);
	if( rt < 0L ) {
		rt += (24*60*60);
		sd += 1;
	}
	TIMESPAN rd = (bd - sd) + (bt - st) / (24*60*60);
	if( rd <= 0 || 0x2c95c < rd ) {
		setError(ERROR_DATERANGE);
		return false;
	}
	DATETIME r = MAKEDATETIME(rd, rt);
	if( r < RANGE_FROM || RANGE_TO < r ) {
		setError(ERROR_DATERANGE);
		return false;
	}
	return r;
}

TIMESPAN addTimeSpan(TIMESPAN ts0, TIMESPAN ts1)
{
	return ts0 + ts1;
}

TIMESPAN subTimeSpan(TIMESPAN ts0, TIMESPAN ts1)
{
	return ts0 - ts1;
}

INT getDaySpan(TIMESPAN span)
{
	return (INT)(span / (24*60*60));
}

INT getHourSpan(TIMESPAN span)
{
	return (INT)(span / (60*60));
}

INT getMinuteSpan(TIMESPAN span)
{
	return (INT)(span / 60);
}

INT getSecondSpan(TIMESPAN span)
{
	return (INT)span;
}

bool isLeapYear(INT year)
{
	if( year < 1800 || 2300 < year ) return false;
	return daysOfYear[year-1800] == 366;
}

INT getDaysOfYear(INT year)
{
	if( year < 1800 || 2300 < year ) {
		setError(ERROR_PARAMETER);
		return 0;
	}
	return daysOfYear[year-1800];
}

INT getDaysOfMonth(INT year, INT month)
{
	if( month < 1 || 12 < month ) {
		setError(ERROR_PARAMETER);
		return 0;
	}
	if( month == 2 && isLeapYear(year) ) {
		return daysOfMonth[month-1] + 1;
	} else {
		return daysOfMonth[month-1];
	}
}

DATETIME getWeekDate(INT year, INT month, INT week, INT n)
{
	DATEELEMENTS de;
	de.year		= year;
	de.month	= month;
	de.date		= 1;
	DATETIME d = makeDate(&de);
	if( d == 0 ) return 0;
	if( !splitDate(d, &de) ) return 0;
	if( week < 0 || 6 < week ) {
		setError(ERROR_PARAMETER);
		return 0;
	}
	INT nDay = 0;
	if( n < 1 ) {
		setError(ERROR_PARAMETER);
		return 0;
	}

	if( n < 5 ) {
		nDay = (n-1)*7 - de.dayOfWeek + (week < de.dayOfWeek ? week + 7 : week);
	} else {
		INT nw;
		INT cd;
		INT daysOfMonth = getDaysOfMonth(year, month);
		for( nw = 1; nw <= n; nw++ ) {
			cd = (nw-1)*7 - de.dayOfWeek + (week < de.dayOfWeek ? week + 7 : week);
			if( cd >= daysOfMonth ) break;
			nDay = cd;
		}
	}
	de.date	+= nDay;
	return makeDate(&de);
}

bool rfc822ToDate(const CHAR * rfc822, DATETIME & dateTime)
{
	// Mon, 02 Nov 2009 03:58:28 +0900 (JST)
	// 02 Nov 2009 03:58:28 +0900 (JST)
	INT	year=0, month=0, day=0, yday=0, hour=0, min=0, sec=0;

	typedef struct _tzStruct {
		INT				len;
		const CHAR *	sym;
		INT				bias;
	} tzStruct;
	static const tzStruct tzs[] = {
		{2,ES_T("UT"),  0},{3,ES_T("UTC"), 0},{3,ES_T("GMT"), 0},
		{3,ES_T("EST"),-5},{3,ES_T("EDT"),-4},
		{3,ES_T("CST"),-6},{3,ES_T("CDT"),-5},
		{3,ES_T("MST"),-7},{3,ES_T("MDT"),-6},
		{3,ES_T("PST"),-8},{3,ES_T("PDT"),-7},
		{1,ES_T("Z"), 0},
		{1,ES_T("A"), -1},{1,ES_T("B"), -2},{1,ES_T("C"), -3},{1,ES_T("D"), -4},
		{1,ES_T("E"), -5},{1,ES_T("F"), -6},{1,ES_T("G"), -7},{1,ES_T("H"), -8},
		{1,ES_T("I"), -9},{1,ES_T("K"),-10},{1,ES_T("L"),-11},{1,ES_T("M"),-12},
		{1,ES_T("N"),  1},{1,ES_T("O"),  2},{1,ES_T("P"),  3},{1,ES_T("Q"),  4},
		{1,ES_T("R"),  5},{1,ES_T("S"),  6},{1,ES_T("T"),  7},{1,ES_T("U"),  8},
		{1,ES_T("V"),  9},{1,ES_T("W"), 10},{1,ES_T("X"), 11},{1,ES_T("Y"), 12},
		{0,NULL,0},
	};

	const CHAR * p_str = rfc822;
	if( !p_str ) return 0;

	// Week
	while( *p_str == ES_T(' ') ) p_str++;
	for( INT w = 0; w < 7; w++ ) {
		if( isEqualCharsNoCaseLen(p_str, weekName3[w], 3) ) {
			p_str += 3;
			if( *p_str == ES_T(',') ) p_str++;
			break;
		}
	}

	// Day of Month
	while( *p_str == ES_T(' ') ) p_str++;
	for( INT d = 0; d < 2; d++ ) {
		if( ES_T('0') <= *p_str && *p_str <= ES_T('9') ) {
			day = day * 10 + (*p_str - ES_T('0'));
			p_str++;
		} else {
			break;
		}
	}
	if( day < 0 || 31 < day ) return 0;

	// Month
	while( *p_str == ES_T(' ') ) p_str++;
	for( INT m = 0; m < 12; m++ ) {
		if(isEqualCharsNoCaseLen(p_str, monthName3[m], 3) ) {
			month = m+1;
			p_str += 3;
			break;
		}
	}
	if( month == 0 ) return 0;

	// year
	while( *p_str == ES_T(' ') ) p_str++;
	for( INT y = 0; y < 4; y++ ) {
		if( ES_T('0') <= *p_str && *p_str <= ES_T('9') ) {
			year = year * 10 + (*p_str - ES_T('0'));
			p_str++;
		} else {
			break;
		}
	}
	if( year <= 0 || 9999 <= year ) return 0;

	// hour
	while( *p_str == ES_T(' ') ) p_str++;
	for( INT h = 0; h < 2; h++ ) {
		if( ES_T('0') <= *p_str && *p_str <= ES_T('9') ) {
			hour = hour*10+(*p_str-ES_T('0'));
			p_str++;
		} else {
			break;
		}
	}
	if( hour > 24 ) return 0;
	if( *p_str == ES_T(':') ) p_str++;

	// min
	while( *p_str == ES_T(' ') ) p_str++;
	for( INT m = 0; m < 2; m++ ) {
		if( ES_T('0') <= *p_str && *p_str <= ES_T('9') ) {
			min = min*10+(*p_str-ES_T('0'));
			p_str++;
		} else {
			break;
		}
	}
	if( min > 59 ) return 0;
	if( *p_str == ES_T(':') ) p_str++;

	// sec
	while( *p_str == ES_T(' ') ) p_str++;
	for( INT s = 0; s < 2; s++ ) {
		if( ES_T('0') <= *p_str && *p_str <= ES_T('9') ) {
			sec = sec*10+(*p_str-ES_T('0'));
			p_str++;
		} else {
			break;
		}
	}
	if( sec > 59 ) return 0;

	// time zone
	FLOAT bias=0.0;
	while( *p_str == ES_T(' ') ) p_str++;
	const tzStruct *tp = tzs;
	while(tp->len) {
		if(isEqualCharsNoCaseLen(p_str, tp->sym, tp->len) ) break;
		tp++;
	}
	if(tp->len != 0) {
		bias = tp->bias;
		p_str += tp->len;
	} else {
		while((ES_T('A') <= *p_str && *p_str <= ES_T('Z')) || (ES_T('a') <= *p_str && *p_str <= ES_T('z'))) p_str++;
		if( *p_str == ES_T('-') || *p_str == ES_T('+') ) {
			FLOAT sign = (*p_str == ES_T('-')) ? 1.0 : -1.0;
			p_str++;
			INT tzh = 0;
			INT tzm = 0;
			for( INT h = 0; h < 2; h++ ) {
				if( ES_T('0') <= *p_str && *p_str <= ES_T('9') ) {
					tzh += tzh * 10 + (*p_str-ES_T('0'));
					p_str++;
				} else {
					break;
				}
			}
			for( INT m = 0; m < 2; m++ ) {
				if( ES_T('0') <= *p_str && *p_str <= ES_T('9') ) {
					tzm += tzm * 10 + (*p_str-ES_T('0'));
					p_str++;
				} else {
					break;
				}
			}
			bias = (tzh + tzm / 60.0) * sign;
		}
	}

	TIMESPAN tzSpan = (TIMESPAN)(bias * 60 * 60);
	DATETIMEELEMENTS lcDaleTimeElements;
	lcDaleTimeElements.date.year	= year;
	lcDaleTimeElements.date.month	= month;
	lcDaleTimeElements.date.date	= day;
	lcDaleTimeElements.time.hour	= hour;
	lcDaleTimeElements.time.minute	= min;
	lcDaleTimeElements.time.second	= sec;

	DATETIME lcDateTime;
	lcDateTime = makeDateTime(&lcDaleTimeElements);
	if( lcDateTime == 0 ) {
		return false;
	}
	if( tzSpan != 0 ) {
		// global time
		dateTime = addTime(lcDateTime, tzSpan);
	} else {
		dateTime	= lcDateTime;
	}
	return true;
}

bool dateToRfc822(DATETIME dateTime, STRING * rfc822)
{
	DATETIMEELEMENTS dateTimeElements;

	if( !splitDateTime(dateTime, &dateTimeElements) ) {
		return false;
	}
	formatString(rfc822, ES_T("%1, %2i %3 %4i %5i:%6i:%7i GMT"),
		weekName3[dateTimeElements.date.dayOfWeek],
		dateTimeElements.date.date,
		monthName3[dateTimeElements.date.month-1],
		dateTimeElements.date.year,
		dateTimeElements.time.hour,
		dateTimeElements.time.minute,
		dateTimeElements.time.second);
	return true;
}

} // es

