/*
 * lex.cpp
 *
 *  Created on: 2012/01/17
 *      Author: tanaka
 */

#include "lexer.h"
#include "variant.h"
#include "bytes.h"
#include "string.h"
#include "sstring.h"

#ifdef ELISE_SRC_UTF16
//#	include "char_utf16.h"
#elif  ELISE_SRC_MBCS
#	include "char_mbcs.h"
#else // utf8
#	include "char_utf8.h"
#endif

namespace es {

typedef BYTE (*ReaderProc)(Lexer *lex);
typedef bool  (*DigitPerser)(VARIANT &v, INT digit, BYTE *src);

#define SOURCE_CHUNK_SIZE 64
SourceChunk::SourceChunk()
{
	reader			= NULL;
	readPosition	= 0;
	filePosition	= 0;
	ugReadPosition	= 0;
	ugFilePosition	= 0;
	chunkSize		= 0;
	bSavePosition	= 0;
	nSavePosition	= 0;
	pSavePosition	= NULL;
	initSString(&chunkBuffer);
}
SourceChunk::~SourceChunk()
{
	clearSString(&chunkBuffer);
	if( pSavePosition ) freeMemory(pSavePosition);
}

bool SourceChunk::setSource(ISourceReader *r)
{
	if( pSavePosition ) freeMemory(pSavePosition);
	reader	= r;
	readPosition	= 0;
	filePosition	= 0;
	ugReadPosition	= 0;
	ugFilePosition	= 0;
	chunkSize		= 0;
	bSavePosition	= 0;
	nSavePosition	= 0;
	pSavePosition	= NULL;
	initSString(&chunkBuffer);
	return true;
}

SCHAR SourceChunk::getChar()
{
	ugReadPosition	= readPosition;
	ugFilePosition	= filePosition;
	if( readPosition == chunkSize ) {
		SSTRING nc;
		initSString(&nc);
		if( !reader->readString(SOURCE_CHUNK_SIZE, &nc) ) {
			clearSString(&nc);
			return READ_ERROR;
		}
		if( getSStringSize(&nc) == 0 ) {
			clearSString(&nc);
			return 0;
		}
		if( nSavePosition != 0 ) {
			addSString(&chunkBuffer, &nc);
		} else if( readPosition > 0 ){
			const SCHAR *p = getSStringReadPtr(&chunkBuffer);
			SCHAR tmp;
			tmp = p[readPosition-1];
			resizeSString(&chunkBuffer,1);
			SCHAR *d = getSStringWritePtr(&chunkBuffer);
			*d	= tmp;
			addSString(&chunkBuffer, &nc);
			readPosition = 1;
			ugReadPosition	= 1;
		} else {
			copySString(&chunkBuffer, &nc);
			readPosition = 0;
			ugReadPosition	= 0;
		}
		clearSString(&nc);
		chunkSize = getSStringSize(&chunkBuffer);
	}
	filePosition++;
	return getSStringReadPtr(&chunkBuffer)[readPosition++];
}

SCHAR SourceChunk::nextChar()
{
	if( readPosition == chunkSize ) {
		SSTRING nc;
		initSString(&nc);
		if( !reader->readString(SOURCE_CHUNK_SIZE, &nc) ) {
			clearSString(&nc);
			return READ_ERROR;
		}
		if( getSStringSize(&nc) == 0 ) {
			clearSString(&nc);
			return 0;
		}
		addSString(&chunkBuffer, &nc);
		clearSString(&nc);
		chunkSize = getSStringSize(&chunkBuffer);
	}
	return getSStringReadPtr(&chunkBuffer)[readPosition];
}

const SCHAR * SourceChunk::getBufferPtr(SIZE &maxChars)
{
	while( readPosition+maxChars >= chunkSize ) {
		SSTRING nc;
		initSString(&nc);
		if( !reader->readString(SOURCE_CHUNK_SIZE, &nc) ) {
			clearSString(&nc);
			return NULL;
		}
		if( getSStringSize(&nc) == 0 ) {
			clearSString(&nc);
			break;
		}
		addSString(&chunkBuffer, &nc);
		clearSString(&nc);
		chunkSize = getSStringSize(&chunkBuffer);
	}
	maxChars = chunkSize - readPosition;
	return &getSStringReadPtr(&chunkBuffer)[readPosition];
}

bool SourceChunk::movePosition(INT offset)
{
	SIZE newPosition = readPosition + offset;
	if( 0 <= newPosition && newPosition <= chunkSize ) {
		readPosition	= newPosition;
		filePosition	+= offset;
		ugReadPosition	= readPosition-1;
		ugFilePosition	= filePosition-1;
		return true;
	}
	return false;
}

SIZE SourceChunk::getPosition()
{
	return filePosition;
}

bool SourceChunk::savePoint()
{
	if( nSavePosition+4 > bSavePosition ) {
		SIZE nBlock = bSavePosition + 16;
		SIZE *nPtr = (SIZE*)resizeMemory(pSavePosition, sizeof(SIZE)*nBlock);
		if( nPtr == NULL ) return false;
		bSavePosition	= nBlock;
		pSavePosition	= nPtr;
	}
	pSavePosition[nSavePosition++] = readPosition;
	pSavePosition[nSavePosition++] = filePosition;
	pSavePosition[nSavePosition++] = ugReadPosition;
	pSavePosition[nSavePosition++] = ugFilePosition;
	return true;
}

bool SourceChunk::clearPoint()
{
	if( nSavePosition < 4 ) {
		setError(ERROR_LOGICAL);
		return false;
	}
	nSavePosition -= 4;
	return true;
}

bool SourceChunk::rewindPoint()
{
	if( nSavePosition < 4 ) {
		setError(ERROR_LOGICAL);
		return false;
	}
	ugFilePosition	= pSavePosition[--nSavePosition];
	ugReadPosition	= pSavePosition[--nSavePosition];
	filePosition	= pSavePosition[--nSavePosition];
	readPosition	= pSavePosition[--nSavePosition];
	return true;
}

Lexer::Lexer(IContext *ctx)
{
	this->ctx	= ctx;
	url			= NULL;
	resourceURL	= NULL;
	initState(state);
}

Lexer::~Lexer()
{
	clearVariant(&state.value);
}

void Lexer::setSource(const IURL *pURL, ISourceReader *src)
{
	url		= pURL;
	source.setSource(src);
}

SIZE Lexer::getTrailCharSize(SCHAR c)
{
#ifdef ELISE_SRC_UTF16
	return 0;
#elif  ELISE_SRC_MBCS
	return getTrailCharSizeMBCS(c);
#else // utf8
	return getTrailCharSizeUTF8(c);
#endif
}

bool Lexer::convertEncode(STRING *d, SSTRING *s)
{
#ifdef ELISE_SRC_UTF16
	return copyString(d, (STRING*)s);
#elif  ELISE_SRC_MBCS
	return convertMBCS(s, s);
#else // utf8
	return convertUTF8(d, s);
#endif
}

bool Lexer::getCharIf(SCHAR next)
{
	if( source.nextChar() == next ) {
		source.getChar();
		return true;
	}
	return false;
}

SCHAR Lexer::nextChar()
{
	return source.nextChar();
}

SCHAR Lexer::getChar()
{
	SCHAR c = source.getChar();
	if( c == SourceChunk::READ_ERROR ) {
		return TID_ERROR;
	}
	if( c == 0 ) return 0;
	if( state.line == 0 )
		state.line = 1;
	state.position = source.getPosition();
	if( c == ES_ST('\r') || c == ES_ST('\n') ) {
		state.line++;
		if( c == ES_ST('\r') && source.nextChar() == ES_ST('\n') ) {
			source.getChar();
		}
		c = ES_ST('\n');
	}
	return c;
}

SCHAR Lexer::seekWSPC()
{
	SCHAR c=0;
	while( (c = getChar()) ) {
		// white space
		while( c == ES_ST(' ') || c == ES_ST('\n') || c == ES_ST('\r') || c == ES_ST('\t') ) {
			c = getChar();
		}
		// comment
		if( c == ES_ST('/') ) {
			if( getCharIf(ES_ST('/')) ) {
				// line comment
				while( (c = getChar()) ) {
					SIZE ts = getTrailCharSize(c);
					if( ts ) {
						while(ts) {
							c = getChar();
							ts--;
						};
						continue;
					}
					if( c == ES_ST('\n') ) break;
				}
				continue;
			} else if( getCharIf(ES_ST('*')) ) {
				// block comment
				while( (c = getChar()) ) {
					SIZE ts = getTrailCharSize(c);
					if( ts ) {
						while(ts) {
							c = getChar();
							ts--;
						};
						continue;
					}
					if( c == ES_ST('*') && getCharIf(ES_ST('/')) ) break;
				}
				continue;
			}
		}
		break;
	}
	return c;
}


static BYTE	BinReader(Lexer *lex)
{
	if( !lex ) return 2;	// return digit
	SCHAR c = lex->nextChar();
	if( ES_ST('0') <= c && c <= ES_T('1') ) return lex->getChar() - ES_ST('0');
	return 0xff;
}

static BYTE DecReader(Lexer *lex)
{
	if( !lex ) return 10;	// return digit
	CHAR c = lex->nextChar();
	if( ES_ST('0') <= c && c <= ES_ST('9') ) return lex->getChar() - ES_ST('0');
	return 0xff;
}

static BYTE HexReader(Lexer *lex)
{
	if( !lex ) return 16;	// return digit
	CHAR c = lex->nextChar();
	if( ES_ST('0') <= c && c <= ES_ST('9') ) return lex->getChar() - ES_ST('0');
	if( ES_ST('a') <= c && c <= ES_ST('f') ) return lex->getChar() - ES_ST('a') + 10;
	if( ES_ST('A') <= c && c <= ES_ST('F') ) return lex->getChar() - ES_ST('A') + 10;
	return 0xff;
}

bool ByteParser(VARIANT &v, INT digit, BYTE *buff)
{
	BYTE val=0;
	WORD valt=0;
	INT	n;
	while( ( n = *buff) != 0xff ) {
		valt = valt * digit + n;
		val = (BYTE)valt;
		if( val != valt ) return false;
		buff++;
	}
	VariantPtr token(&v);
	token = val;
	return true;
}

bool WordParser(VARIANT &v, INT digit, BYTE *buff)
{
	WORD val=0;
	DWORD valt=0;
	INT	n;
	while( ( n = *buff) != 0xff ) {
		valt = valt * digit + n;
		val = valt;
		if( val != valt ) return false;
		buff++;
	}
	VariantPtr token(&v);
	token = val;
	return true;
}

bool DWordParser(VARIANT &v, INT digit, BYTE *buff)
{
	DWORD val=0;
	unsigned long long valt=0;
	INT	n;
	while( ( n = *buff) != 0xff ) {
		valt = valt * digit + n;
		val = (DWORD)valt;
		if( val != valt ) return false;
		buff++;
	}
	VariantPtr token(&v);
	token = val;
	return true;
}

bool ShortParser(VARIANT &v, INT digit, BYTE *buff)
{
	SHORT val=0;
	INT	valt=0;
	INT	n;
	while( ( n = *buff) != 0xff ) {
		valt = valt * digit + n;
		val = valt;
		if( val != valt ) return false;
		buff++;
	}
	VariantPtr token(&v);
	token = val;
	return true;
}

bool IntParser(VARIANT &v, INT digit, BYTE *buff)
{
	INT 	val=0;
	INT64	valt=0;
	INT	n;
	while( ( n = *buff) != 0xff ) {
		valt = valt * digit + n;
		val = (INT)valt;
		if( val != valt ) return false;
		buff++;
	}
	VariantPtr token(&v);
	token = val;
	return true;
}

bool FloatParser(VARIANT &v, INT digit, BYTE *buff)
{
	FLOAT 	val=0;
	FLOAT	valt=0;
	INT	n;
	while( ( n = *buff) != 0xff && n != 0xf0 ) {
		valt = val;
		val = val * digit + n;
		if( val < valt ) return false;
		buff++;
	}
	if( *buff == 0xf0 ) {
		buff++;
		FLOAT pval = 0;
		FLOAT pvalt = 0;
		INT pn=1;
		INT pnt=1;
		while( ( n = *buff) != 0xff ) {
			pvalt	= pval;
			pval	= pval * digit + n;
			if( pval < pvalt ) return false;
			pnt	= pn;
			pn	= pn * digit;
			if( pn < pnt ) return false;
			buff++;
		}
		val = val + pval / pn;
	}
	VariantPtr token(&v);
	token = val;
	return true;
}

bool DecimalParser(VARIANT &v, INT digit, BYTE *buff)
{
	INT64 val=0;
	INT64 valt=0;
	INT	n;
	while( ( n = *buff) != 0xff && n != 0xf0 ) {
		valt = val;
		val = val * digit + n;
		if( val < valt ) return false;
		buff++;
	}
	INT64 pval=0;
	if( *buff == 0xf0 ) {
		buff++;
		INT pn=1;
		INT pv=1000;
		while( ( n = *buff) != 0xff ) {
			pval = pval * digit + n;
			pn++;
			pv /= digit;
			buff++;
		}
		if( pv < 1 ) return false;
		pval *= pv;
	}
	valt = val;
	val = val * 10000;
	if( val / 10000 != valt ) return false;
	val += pval;

	DECIMAL fx;
	fx.dv	= val;
	VariantPtr token(&v);
	token = fx;
	return true;
}

static bool EscapeCharValueProc(Lexer *lex, SCHAR &c)
{
	c = lex->getChar();
	if( !c ) return false;

	switch(c) {
	case ES_ST('r'):	c = ES_ST('\r');	break;
	case ES_ST('n'):	c = ES_ST('\n');	break;
	case ES_ST('t'):	c = ES_ST('\t');	break;
	case ES_ST('x'):
	case ES_ST('X'):
		{
			lex->getChar();
			INT n;
			INT charSize = sizeof(SCHAR);
			c = 0;
			for( n = 0; n < charSize; n++ ) {
				INT d = HexReader(lex);
				if( d == 0xff ) break;
				c = c * 16 + c;
			}
		}
		break;
	}
	return true;
}

TokenId Lexer::getToken()
{
	TokenId		token = TID_EOF;
	SCHAR		c;
	LEXERSTATE	st;

	c = seekWSPC();
	if( c == TID_ERROR ) return TID_ERROR;
	if( c == 0 ) return token;

	switch( c ) {
	case ES_ST('&'):
		if( getCharIf(ES_ST('&')) )		token	= TID_ANDAND;
		else if( getCharIf(ES_ST('=')))	token	= TID_ANDEQ;
		else							token	= TID_AND;
		break;
	case ES_ST('|'):
		if( getCharIf(ES_ST('|')) ) 	token	= TID_OROR;
		else if( getCharIf(ES_ST('|')))	token	= TID_OREQ;
		else							token	= TID_OR;
		break;
	case ES_ST('+'):
		if( getCharIf(ES_ST('+')) ) 	token	= TID_PLUSPLUS;
		else if( getCharIf(ES_ST('=')))	token	= TID_PLUSEQ;
		else							token	= TID_PLUS;
		break;
	case ES_ST('-'):
		if( getCharIf(ES_ST('-')) )		token	= TID_MINUSMINUS;
		else if( getCharIf(ES_ST('=')))	token	= TID_MINUSEQ;
		else							token	= TID_MINUS;
		break;
	case ES_ST('*'):
		if( getCharIf(ES_ST('=')) )		token	= TID_MULEQ;
		else							token	= TID_MUL;
		break;
	case ES_ST('/'):
		if( getCharIf(ES_ST('=')) )		token	= TID_DIVEQ;
		else							token	= TID_DIV;
		break;
	case ES_ST('%'):
		if( getCharIf(ES_ST('=')) )		token	= TID_MODEQ;
		else							token	= TID_MOD;
		break;
	case ES_ST('='):
		if( getCharIf(ES_ST('=')) )		token	= TID_EQEQ;
		else							token	= TID_EQ;
		break;
	case ES_ST('!'):
		if( getCharIf(ES_ST('=')) ) 	token	= TID_NE;
		else							token	= TID_NOT;
		break;
	case ES_ST('<'):
		if( getCharIf(ES_ST('<')) ) 	token	= TID_LSHIFT;
		else if( getCharIf(ES_ST('=')))	token	= TID_LE;
		else							token	= TID_LT;
		break;
	case ES_ST('>'):
		if( getCharIf(ES_ST('>')) ) 	token	= TID_RSHIFT;
		else if( getCharIf(ES_ST('=')))	token	= TID_GE;
		else							token	= TID_GT;
		break;
	case ES_ST('^'):
		token	= TID_XOR;
		break;
	case ES_ST(':'):
		if( getCharIf(ES_ST(':')) )		token	= TID_SPC;
		else							token	= TID_COL;
		break;
	case ES_ST('?'):	token	= TID_QU;			break;
	case ES_ST('['):	token	= TID_INDEX;		break;
	case ES_ST(']'):	token	= TID_EDA;			break;
	case ES_ST('('):	token	= TID_STC;			break;
	case ES_ST(')'):	token	= TID_EDC;			break;
	case ES_ST('{'):	token	= TID_BEGINBLOCK;	break;
	case ES_ST('}'):	token	= TID_ENDBLOCK;		break;
	case ES_ST(','):	token	= TID_COM;			break;
	case ES_ST('.'):	token	= TID_DOT;			break;
	case ES_ST('~'):	token	= TID_BITNOT;		break;
	case ES_ST(';'):	token	= TID_EOS;			break;
	case ES_ST('#'):	token	= TID_SHARP;		break;

	case ES_ST('0'):
	case ES_ST('1'):
	case ES_ST('2'):
	case ES_ST('3'):
	case ES_ST('4'):
	case ES_ST('5'):
	case ES_ST('6'):
	case ES_ST('7'):
	case ES_ST('8'):
	case ES_ST('9'):
		{
			bool has_point = false;
			ReaderProc reader = NULL;
			DigitPerser parser = NULL;
			BYTE buff[256];
			BYTE *buff_ptr = buff;

			if( c == ES_ST('0') ) {
				if( getCharIf(ES_ST('x')) ) {
					reader = HexReader;
				} else if( getCharIf(ES_ST('b')) ) {
					reader = BinReader;
				} else {
					*buff_ptr++ = (BYTE)(c - ES_ST('0'));
				}
			}
			if( reader == NULL )
				reader = DecReader;

			BYTE n;
			while( (n = reader(this)) != 0xff ) {
				*buff_ptr++ = n;
			};
			if( getCharIf(ES_ST('.')) ) {
				*buff_ptr++ = 0xf0;
				has_point = true;
				while( (n = reader(this)) != 0xff ) {
					*buff_ptr++ = n;
				}
			}
			*buff_ptr++ = 0xff;
			if( has_point ) {
				if( getCharIf(ES_ST('f')) ) {
					parser	= DecimalParser;
					token	= TID_DECIMAL;
				} else {
					parser	= FloatParser;
					token	= TID_FLOAT;
				}
			} else {
				if( getCharIf(ES_ST('s'))  ) {
					parser	= ShortParser;
					token	= TID_SHORT;
				} else if( getCharIf(ES_ST('h')) ) {
					parser	= ByteParser;
					token	= TID_BYTE;
				} else if( getCharIf(ES_ST('w')) ) {
					parser	= WordParser;
					token	= TID_WORD;
				} else if( getCharIf(ES_ST('u')) ) {
					parser	= DWordParser;
					token	= TID_DWORD;
				} else if( getCharIf(ES_ST('f')) ) {
					parser	= DecimalParser;
					token	= TID_DECIMAL;
				} else {
					parser	= IntParser;
					token	= TID_INTEGER;
				}
			}
			if( !parser(state.value, reader(NULL), buff) ) {
				return TID_ERROR;
			}
		}
		break;

	case ES_ST('"'):
		{
			SString str;
			c = getChar();
			while(1) {
				SString line; //const CHAR * sp = state.ptr-1;
				while( c ) {
					SIZE ts = getTrailCharSize(c);
					if( ts ) {
						line += c;
						while(ts) {
							line += getChar();
							ts--;
						}
					} else if( c == ES_ST('\\') ) {
						if( !EscapeCharValueProc(this, c) ) {
							return TID_ERROR;
						}
						line += c;
					} else if( c == ES_ST('"') ) {
						break;
					}
					c = getChar();
				}
				str += line;
				saveState(st);
				c = seekWSPC();
				if( c != ES_ST('"') ) {
					restoreState(st);
					break;
				}
				freeState(st);
				c	= getChar();
			}
			VariantPtr v(&state.value);
			String ustr;
			convertEncode(ustr, str);
			v	= ustr;
			token = TID_STRING;
		}
		break;

	case ES_ST('\''):
		{
			c = getChar();
			if( c == ES_ST('\'') ) {
				c	= 0;
			} else if( c == ES_ST('\\') ) {
				if( !EscapeCharValueProc(this, c) ) {
					return TID_ERROR;
				}
				if( getChar() != ES_ST('\'') )  {
					return TID_ERROR;
				}
			} else if( getChar() != ES_ST('\'') ) {
				return TID_ERROR;
			}
			VariantPtr v(&state.value);
			SString str = c;
			String ustr;
			convertEncode(ustr, str);
			v = ustr[0];
			token = TID_CHAR;
		}
		break;

	case ES_ST('x'):
	case ES_ST('X'):
		if( nextChar() == ES_ST('"') ) {
			Bytes bin;
			while(1) {
				getChar(); // read '"'
				Bytes line;
				BYTE byteData=0;
				INT nBit=0;
				while( 1 ) {
					INT n = HexReader(this);
					if( n != 0xff ) {
						byteData = (byteData << 4) + n;
						nBit++;
						if( nBit == 2 ) {
							line		+= byteData;
							byteData	= 0;
							nBit		= 0;
						}
					} else {
						if( nBit ) {
							line		+= byteData;
							byteData	= 0;
							nBit		= 0;
						}
						c = nextChar();
						if( findSChar(ES_ST("-_ \t\n,"), c) != NOSIZE ) {
							getChar();
							continue;
						}
						break;
					}
				}
				if( getChar() != ES_ST('"') ) {
					return TID_ERROR;
				}
				bin += line;
				saveState(st);
				c = seekWSPC();
				if( c != ES_ST('x') && c != ES_ST('X') ) {
					restoreState(st);
					break;
				}
				if( nextChar() != ES_ST('"') ) {
					restoreState(st);
					break;
				}
				freeState(st);
			}

			VariantPtr v(&state.value);
			v	= bin;
			token = TID_BYTES;
			break;
		}
		// no break;
	default:
		if( (ES_ST('a') <= c && c <= ES_ST('z')) ||
			(ES_ST('A') <= c && c <= ES_ST('Z')) ||
			c == ES_ST('_') ||
			c == ES_ST('$') ) {

			SString sym;
			String usym;
			bool resKey=false;
			if( c == ES_ST('$') && getCharIf(ES_ST('$')) ) {
				resKey = true;
				getChar();
			} else {
				sym += c;
			}

			SIZE ptrSize=13; // maximum reserved words("synchronized") length+1
			const SCHAR *ptr = source.getBufferPtr(ptrSize);
			if( !ptr ) return TID_ERROR;
			const SCHAR *sp = ptr;

			while(1) {
				while( ptrSize ) {
					if( (ES_ST('a') <= *sp && *sp <= ES_ST('z')) ||
						(ES_ST('A') <= *sp && *sp <= ES_ST('Z')) ||
						(ES_ST('0') <= *sp && *sp <= ES_ST('9')) ||
						*sp == ES_ST('_') ) {
						sp++;
						ptrSize--;
					} else {
						break;
					}
				}
				if( sp-ptr ) {
					sym.add(ptr, sp-ptr);
					source.movePosition(sp-ptr);
				}
				if( ptrSize ) {
					c = nextChar();
					break;
				}
				ptrSize = 24; // mostly enough 
				ptr = source.getBufferPtr(ptrSize);
				if( !ptr ) return TID_ERROR;
			}

			VariantPtr v(&state.value);
			if( resKey ) {
				while( c == ES_ST('.') ) {
					sym += getChar();
					ptrSize = 24;
					ptr = source.getBufferPtr(ptrSize);
					if( !ptr ) return TID_ERROR;
					const SCHAR *sp = ptr;
					while(1) {
						while( ptrSize ) {
							if( (ES_ST('a') <= *sp && *sp <= ES_ST('z')) ||
								(ES_ST('A') <= *sp && *sp <= ES_ST('Z')) ||
								(ES_ST('0') <= *sp && *sp <= ES_ST('9')) ||
								*sp == ES_ST('_') ) {
								sp++;
								ptrSize--;
							} else {
								break;
							}
						}
						if( sp-ptr ) {
							sym.add(ptr, sp-ptr);
							source.movePosition(sp-ptr);
						}
						if( ptrSize ) {
							c = nextChar();
							break;
						}
						ptrSize = 24;
						ptr = source.getBufferPtr(ptrSize);
						if( !ptr ) return TID_ERROR;
					}
				}

				if( findResource(sym, v) ) {
					switch(v->vt) {
					case VT_NULL:		return TID_NULL;
					case VT_BOOL:		return v->dBool ? TID_TRUE : TID_FALSE;
					case VT_CHAR:		return TID_CHAR;
					case VT_BYTE:		return TID_BYTE;
					case VT_WORD:		return TID_WORD;
					case VT_DWORD:		return TID_DWORD;
					case VT_SHORT:		return TID_SHORT;
					case VT_INT:		return TID_INTEGER;
					case VT_DECIMAL:	return TID_DECIMAL;
					case VT_FLOAT:		return TID_FLOAT;
					case VT_BYTES:		return TID_BYTES;
					case VT_STRING:		return TID_STRING;
					}
					return TID_ERROR;
				}
			}
			// reserved words
			token = findReserved(sym);
			if( token != TID_EOF ) {
				switch(token) {
				case TID_TRUE:
					v	= true;
					break;
				case TID_FALSE:
					v	= false;
					break;
				case TID_NULL:
					v.clear();
					break;
				}
				break;
			}

			convertEncode(usym, sym);
			v	= usym;
			token	= TID_SYMBOL;
			break;
		}
		token = TID_ERROR;
		break;
	}
	return token;
}

bool Lexer::getTokenValue(VARIANT &value) const
{
	return copyVariant(&value, &state.value);
}

const IURL* Lexer::getSourceURL() const
{
	return url;
}

INT Lexer::getCurrentPos() const
{
	return state.position;
}

INT Lexer::getCurrentLine() const
{
	return state.line;
}

INT Lexer::getTokenStartPos() const
{
	return state.position;
}

void Lexer::initState(LEXERSTATE &st)
{
	st.position	= 0;
	st.line		= 0;
	initVariant(&st.value);
}

void Lexer::saveState(LEXERSTATE &st)
{
	st.position	= state.position;
	st.line		= state.line;
	initVariant(&st.value);
	copyVariant(&st.value, &state.value);
	source.savePoint();
}

void Lexer::restoreState(LEXERSTATE &st)
{
	state.position	= st.position;
	state.line		= st.line;
	copyVariant(&state.value, &st.value);
	clearVariant(&st.value);
	source.rewindPoint();
}

void Lexer::freeState(LEXERSTATE &st)
{
	clearVariant(&st.value);
	source.clearPoint();
}

bool Lexer::findResource(const SSTRING *word, VARIANT *v)
{
	return false;
}

TokenId	Lexer::findReserved(const SSTRING *word)
{
	const SCHAR *p = getSStringReadPtr(word);
	SIZE ws = getSStringSize(word);
	switch(*p++) {
	case ES_ST('a'):
		if( ws == 2 && isEqualSChars(p, ES_ST("s")))		return TID_RESERVED;
		if( ws == 8 && isEqualSChars(p, ES_ST("bstract")))	return TID_RESERVED;
		break;
	case ES_ST('b'):
		if( ws == 5 && isEqualSChars(p, ES_ST("reak")))		return TID_BREAK;
		break;
	case ES_ST('c'):
		if( ws == 4 && isEqualSChars(p, ES_ST("ase")))		return TID_CASE;
		if( ws == 5 ) {
			if( isEqualSChars(p, ES_ST("atch")))			return TID_CATCH;
			if( isEqualSChars(p, ES_ST("lass")))			return TID_CLASS;
			if( isEqualSChars(p, ES_ST("onst")))			return TID_CONST;
		}
		if( ws == 8 && isEqualSChars(p, ES_ST("ontinue")))	return TID_CONTINUE;
		break;
	case ES_ST('d'):
		if( ws == 2 && isEqualSChars(p, ES_ST("o")))		return TID_DO;
		if( ws == 6 && isEqualSChars(p, ES_ST("elete")))	return TID_RESERVED;
		if( ws == 7 && isEqualSChars(p, ES_ST("efault")))	return TID_DEFAULT;
		if( ws == 8 && isEqualSChars(p, ES_ST("ebugger")))	return TID_RESERVED;
		break;
	case ES_ST('e'):
		if( ws == 4 && isEqualSChars(p, ES_ST("lse")))		return TID_ELSE;
		if( ws == 6 && isEqualSChars(p, ES_ST("xport")))	return TID_RESERVED;
		if( ws == 7 && isEqualSChars(p, ES_ST("xtends")))	return TID_EXTENDS;
		break;
	case ES_ST('f'):
		if( ws == 3 && isEqualSChars(p, ES_ST("or")))		return TID_FOR;
		if( ws == 5 ) {
			if( isEqualSChars(p, ES_ST("alse")))			return TID_FALSE;
			if( isEqualSChars(p, ES_ST("inal")))			return TID_FINAL;
		}
		if( ws == 7 && isEqualSChars(p, ES_ST("inally")))	return TID_RESERVED;
		if( ws == 8 && isEqualSChars(p, ES_ST("unction")))	return TID_FUNCTION;
		break;
	case ES_ST('i'):
		if( ws == 2 ) {
			if( isEqualSChars(p, ES_ST("f")))				return TID_IF;
			if( isEqualSChars(p, ES_ST("n")))				return TID_IN;
			if( isEqualSChars(p, ES_ST("s")))				return TID_RESERVED;
		}
		if( ws == 6 && isEqualSChars(p, ES_ST("mport")))	return TID_IMPORT;
		if( ws == 8 && isEqualSChars(p, ES_ST("nternal")))	return TID_RESERVED;
		if( ws == 9 && isEqualSChars(p, ES_ST("nterface")))	return TID_INTERFACE;
		if( ws == 10 ) {
			if( isEqualSChars(p, ES_ST("mplements")))		return TID_IMPLEMENTS;
			if( isEqualSChars(p, ES_ST("nstanceof")))		return TID_INSTANCEOF;
		}
		break;
	case ES_ST('m'):
		if( ws == 6 && isEqualSChars(p, ES_ST("ethod")))	return TID_METHOD;
		break;
	case ES_ST('n'):
		if( ws == 3 && isEqualSChars(p, ES_ST("ew")))		return TID_NEW;
		if( ws == 4 && isEqualSChars(p, ES_ST("ull")))		return TID_NULL;
		if( ws == 6 && isEqualSChars(p, ES_ST("ative")))	return TID_RESERVED;
		if( ws == 9 && isEqualSChars(p, ES_ST("amespace")))	return TID_NAMESPACE;
		break;
	case ES_ST('p'):
		if( ws == 6 && isEqualSChars(p, ES_ST("ublic")))	return TID_RESERVED;
		if( ws == 7 ) {
			if( isEqualSChars(p, ES_ST("ackage")))			return TID_PACKAGE;
			if( isEqualSChars(p, ES_ST("rivate")))			return TID_PRIVATE;
		}
		if( ws == 8 && isEqualSChars(p, ES_ST("roperty")))	return TID_PROPERTY;
		if( ws == 9 && isEqualSChars(p, ES_ST("rotected")))	return TID_PROTECTED;
		break;
	case ES_ST('r'):
		if( ws == 3 && isEqualSChars(p, ES_ST("aw")))		return TID_RAWSET;
		if( ws == 6 && isEqualSChars(p, ES_ST("eturn")))	return TID_RETURN;
		if( ws == 9 && isEqualSChars(p, ES_ST("epresent")))	return TID_REPRESENT;
		break;
	case ES_ST('s'):
		if( ws == 5 && isEqualSChars(p, ES_ST("uper")))		return TID_SUPER;
		if( ws == 6 ) {
			if( isEqualSChars(p, ES_ST("witch")))			return TID_SWITCH;
			if( isEqualSChars(p, ES_ST("tatic")))			return TID_STATIC;
			if( isEqualSChars(p, ES_ST("izeof")))			return TID_SIZEOF;
			if( isEqualSChars(p, ES_ST("truct")))			return TID_RESERVED;
		}
		if( ws == 12 && isEqualSChars(p, ES_ST("ynchronized")))	return TID_RESERVED;
		break;
	case ES_ST('t'):
		if( ws == 3 && isEqualSChars(p, ES_ST("ry")))		return TID_TRY;
		if( ws == 4 ) {
			if( isEqualSChars(p, ES_ST("rue")))				return TID_TRUE;
			if( isEqualSChars(p, ES_ST("his")))				return TID_THIS;
		}
		if( ws == 5 && isEqualSChars(p, ES_ST("hrow")))		return TID_THROW;
		if( ws == 6 && isEqualSChars(p, ES_ST("ypeof")))	return TID_TYPEOF;
		if( ws == 8 && isEqualSChars(p, ES_ST("emplate")))	return TID_TEMPLATE;
		if( ws == 9 && isEqualSChars(p, ES_ST("ransient")))	return TID_RESERVED;
		break;
	case ES_ST('u'):
		if( ws == 3 && isEqualSChars(p, ES_ST("se")))		return TID_USE;
		break;
	case ES_ST('v'):
		if( ws == 3 && isEqualSChars(p, ES_ST("ar")))		return TID_VAR;
		if( ws == 4 && isEqualSChars(p, ES_ST("oid")))		return TID_RESERVED;
		if( ws == 8 && isEqualSChars(p, ES_ST("olatile")))	return TID_RESERVED;
		break;
	case ES_ST('w'):
		if( ws == 4 && isEqualSChars(p, ES_ST("ith")))		return TID_RESERVED;
		if( ws == 5 && isEqualSChars(p, ES_ST("hile")))		return TID_WHILE;
		break;
	}
	return TID_EOF;
}


} // es
