/*
 * parser.cpp
 *
 *  Created on: 2012/01/19
 *      Author: tanaka
 */

#include "parser.h"
#include "variant.h"

// #define ELISE_BLOCKPARSE


namespace es {

ParseInfoWalker::ParseInfoWalker(IContext *ctx, PARSEINF *root)
{
	this->refCount	= 1;
	this->ctx		= ctx;
	this->top		= root;
	this->cur		= NULL;
}

ParseInfoWalker::~ParseInfoWalker()
{
}

void ParseInfoWalker::addRef() const
{
	refCount++;
}

bool ParseInfoWalker::releaseRef() const
{
	if( --refCount == 0 ) {
		delete this;
		return true;
	}
	return false;
}

IContext * ParseInfoWalker::getContext() const
{
	return ctx;
}

PARSEINF * ParseInfoWalker::seekDown()
{
	PARSEINF * p = cur;
	while(p) {
		switch(p->type) {
		case ST_PROGRAM:
			if( p->dProgram.innerCode ) return p->dProgram.innerCode;
			break;
		case ST_EXPRESSION:
			{
				switch(p->dExpression.exprType) {
				case ET_OPR1:
					if( p->dExpression.exp1.expresion ) return p->dExpression.exp1.expresion;
					break;
				case ET_OPR2:
					if( p->dExpression.exp2.expresion1 ) return p->dExpression.exp2.expresion1;
					if( p->dExpression.exp2.expresion2 ) return p->dExpression.exp2.expresion2;
					break;
				case ET_OPR3:
					if( p->dExpression.exp3.expresion1 ) return p->dExpression.exp3.expresion1;
					if( p->dExpression.exp3.expresion2 ) return p->dExpression.exp3.expresion2;
					if( p->dExpression.exp3.expresion3 ) return p->dExpression.exp3.expresion3;
					break;
				case ET_OPRS:
					if( p->dExpression.token == TID_CALL ) {
						if( p->dExpression.expCall.addr ) return p->dExpression.expCall.addr;
						if( p->dExpression.expCall.argv ) return p->dExpression.expCall.argv;
					}
					if( p->dExpression.token == TID_INDEX ) {
						if( p->dExpression.expIndex.addr ) return p->dExpression.expIndex.addr;
						if( p->dExpression.expIndex.index ) return p->dExpression.expIndex.index;
					}
					break;
				case ET_PATH:
					if( p->dExpression.expPath.symbol ) return p->dExpression.expPath.symbol;
					break;
				case ET_TYPE:
					if( p->dExpression.expType.defaultTypePath ) return p->dExpression.expType.defaultTypePath;
					break;
				}
			}
			break;
		case ST_BLOCK:
			if( p->dBlock.innerCode ) return p->dBlock.innerCode;
			break;
		case ST_USE:
			if( p->dUse.addr ) return p->dUse.addr;
			break;
		case ST_PACKAGE:
			if( p->dPackage.innerCode ) return p->dPackage.innerCode;
			break;
		case ST_CLASS:
			if( p->dClass.basePath ) return p->dClass.basePath;
			if( p->dClass.interfaces) return p->dClass.interfaces;
			if( p->dClass.innerCode ) return p->dClass.innerCode;
			break;
		case ST_TEMPLATE:
			if( p->dTemplate.typePath ) return p->dTemplate.typePath;
			if( p->dTemplate.basePath ) return p->dTemplate.basePath;
			if( p->dTemplate.interfaces ) return p->dTemplate.interfaces;
			if( p->dTemplate.innerCode ) return p->dTemplate.innerCode;
			break;
		case ST_INTERFACE:
			if( p->dInterface.interfaces ) return p->dInterface.interfaces;
			if( p->dInterface.innerCode ) return p->dInterface.innerCode;
			break;
		case ST_PROPERTY:
			if( p->dProperty.path ) return p->dProperty.path;
			if( p->dProperty.array ) return p->dProperty.array;
			if( p->dProperty.initCode ) return p->dProperty.initCode;
			break;
		case ST_METHOD:
			if( p->dMethod.params ) return p->dMethod.params;
			if( p->dMethod.innerCode ) return p->dMethod.innerCode;
			break;
		case ST_FUNCTION:
			if( p->dFunction.params ) return p->dFunction.params;
			if( p->dFunction.innerCode ) return p->dFunction.innerCode;
			break;
		case ST_OBJECT:
			if( p->dObject.path ) return p->dObject.path;
			if( p->dObject.array ) return p->dObject.array;
			if( p->dObject.initExpression ) return p->dObject.initExpression;
			if( p->dObject.innerCode ) return p->dObject.innerCode;
			break;
		case ST_RETURN:
			if( p->dReturn.resultValue ) return p->dReturn.resultValue;
			break;
		case ST_IF:
			if( p->dIf.expression ) return p->dIf.expression;
			if( p->dIf.trueCode ) return p->dIf.trueCode;
			if( p->dIf.falseCode ) return p->dIf.falseCode;
			break;
		case ST_SWITCH:
			if( p->dSwitch.expression ) return p->dSwitch.expression;
			if( p->dSwitch.caseList ) return p->dSwitch.caseList;
			break;
		case ST_CASE:
			if( p->dCase.expression ) return p->dCase.expression;
			if( p->dCase.caseCode ) return p->dCase.caseCode;
			break;
		case ST_DEFAULT:
			if( p->dDefault.caseCode ) return p->dDefault.caseCode;
			break;
		case ST_FOR:
			if( p->dFor.initExpression ) return p->dFor.initExpression;
			if( p->dFor.condExpression ) return p->dFor.condExpression;
			if( p->dFor.nextExpression ) return p->dFor.nextExpression;
			if( p->dFor.innerCode ) return p->dFor.innerCode;
			break;
		case ST_FORIN:
			if( p->dForin.variableExpression ) return p->dForin.variableExpression;
			if( p->dForin.targetExpression ) return p->dForin.targetExpression;
			if( p->dForin.innerCode ) return p->dForin.innerCode;
			break;
		case ST_WHILE:
			if( p->dWhile.expression ) return p->dWhile.expression;
			if( p->dWhile.innerCode ) return p->dWhile.innerCode;
			break;
		case ST_DO:
			if( p->dDo.innerCode ) return p->dDo.innerCode;
			if( p->dDo.expression ) return p->dDo.expression;
			break;
		case ST_TRY:
			if( p->dTry.innerCode ) return p->dTry.innerCode;
			if( p->dTry.catchParam ) return p->dTry.catchParam;
			if( p->dTry.catchCode ) return p->dTry.catchCode;
			break;
		case ST_THROW:
			return p->dThrow.exceptionCode;
		}
		p	= p->next;
	}
	return NULL;
}

PARSEINF * ParseInfoWalker::seekUp()
{
	return cur->parent;
}

PARSEINF * ParseInfoWalker::seekNext()
{
	PARSEINF *p = cur->parent;
	if( !p ) return NULL;

	switch(p->type) {
	case ST_EXPRESSION:
		{
			switch(p->dExpression.exprType) {
			case ET_OPR2:
				if( p->dExpression.exp2.expresion1 == cur )
					if( p->dExpression.exp2.expresion2 ) return p->dExpression.exp2.expresion2;
				break;
			case ET_OPR3:
				if( p->dExpression.exp3.expresion1 == cur ) {
					if( p->dExpression.exp3.expresion2 ) return p->dExpression.exp3.expresion2;
					if( p->dExpression.exp3.expresion3 ) return p->dExpression.exp3.expresion3;
				}
				if( p->dExpression.exp3.expresion2 == cur ) {
					if( p->dExpression.exp3.expresion3 ) return p->dExpression.exp3.expresion3;
				}
				break;
			case ET_OPRS:
				if( p->dExpression.token == TID_CALL ) {
					if( p->dExpression.expCall.addr == cur ) {
						if( p->dExpression.expCall.argv ) return p->dExpression.expCall.argv;
					}
				}
				if( p->dExpression.token == TID_INDEX ) {
					if( p->dExpression.expIndex.addr == cur ) {
						if( p->dExpression.expIndex.index ) return p->dExpression.expIndex.index;
					}
				}
				break;
			}
		}
		break;
	case ST_CLASS:
		if( p->dClass.basePath == cur ) {
			if( p->dClass.interfaces ) return p->dClass.interfaces;
			if( p->dClass.innerCode ) return p->dClass.innerCode;
		}
		if( p->dClass.interfaces == cur ) {
			if( p->dClass.innerCode ) return p->dClass.innerCode;
		}
		break;
	case ST_TEMPLATE:
		if( p->dTemplate.typePath == cur ) {
			if( p->dTemplate.basePath ) return p->dTemplate.basePath;
			if( p->dTemplate.interfaces ) return p->dTemplate.interfaces;
			if( p->dTemplate.innerCode ) return p->dTemplate.innerCode;
		}
		if( p->dTemplate.basePath == cur ) {
			if( p->dTemplate.interfaces ) return p->dTemplate.interfaces;
			if( p->dTemplate.innerCode ) return p->dTemplate.innerCode;
		}
		if( p->dTemplate.interfaces == cur ) {
			if( p->dTemplate.innerCode ) return p->dTemplate.innerCode;
		}
		break;
	case ST_INTERFACE:
		if( p->dInterface.interfaces == cur ) {
			if( p->dInterface.innerCode ) return p->dInterface.innerCode;
		}
		break;
	case ST_PROPERTY:
		if( p->dProperty.path == cur ) {
			if( p->dProperty.array ) return p->dProperty.array;
			if( p->dProperty.initCode ) return p->dProperty.initCode;
		}
		if( p->dProperty.array == cur ) {
			if( p->dProperty.initCode ) return p->dProperty.initCode;
		}
		break;
	case ST_METHOD:
		if( p->dMethod.params == cur ) {
			if( p->dMethod.innerCode ) return p->dMethod.innerCode;
		}
		break;
	case ST_FUNCTION:
		if( p->dFunction.params == cur ) {
			if( p->dFunction.innerCode ) return p->dFunction.innerCode;
		}
		break;
	case ST_OBJECT:
		if( p->dObject.path == cur ) {
			if( p->dObject.array ) return p->dObject.array;
			if( p->dObject.initExpression ) return p->dObject.initExpression;
			if( p->dObject.innerCode ) return p->dObject.innerCode;
		}
		if( p->dObject.array == cur ) {
			if( p->dObject.initExpression ) return p->dObject.initExpression;
			if( p->dObject.innerCode ) return p->dObject.innerCode;
		}
		if( p->dObject.initExpression == cur ) {
			if( p->dObject.innerCode ) return p->dObject.innerCode;
		}
		break;
	case ST_IF:
		if( p->dIf.expression == cur ) {
			if( p->dIf.trueCode ) return p->dIf.trueCode;
			if( p->dIf.falseCode ) return p->dIf.falseCode;
		}
		if( p->dIf.trueCode == cur ) {
			if( p->dIf.falseCode ) return p->dIf.falseCode;
		}
		break;
	case ST_SWITCH:
		if( p->dSwitch.expression == cur ) {
			if( p->dSwitch.caseList ) return p->dSwitch.caseList;
		}
		break;
	case ST_CASE:
		if( p->dCase.expression == cur ) {
			if( p->dCase.caseCode ) return p->dCase.caseCode;
		}
		break;
	case ST_FOR:
		if( p->dFor.initExpression == cur ) {
			if( p->dFor.condExpression ) return p->dFor.condExpression;
			if( p->dFor.nextExpression ) return p->dFor.nextExpression;
			if( p->dFor.innerCode ) return p->dFor.innerCode;
		}
		if( p->dFor.condExpression == cur ) {
			if( p->dFor.nextExpression ) return p->dFor.nextExpression;
			if( p->dFor.innerCode ) return p->dFor.innerCode;
		}
		if( p->dFor.nextExpression == cur ) {
			if( p->dFor.innerCode ) return p->dFor.innerCode;
		}
		break;
	case ST_FORIN:
		if( p->dForin.variableExpression == cur ) {
			if( p->dForin.targetExpression) return p->dForin.targetExpression;
			if( p->dForin.innerCode ) return p->dForin.innerCode;
		}
		if( p->dForin.targetExpression == cur ) {
			if( p->dForin.innerCode ) return p->dForin.innerCode;
		}
		break;
	case ST_WHILE:
		if( p->dWhile.expression == cur ) {
			if( p->dWhile.innerCode ) return p->dWhile.innerCode;
		}
		break;
	case ST_DO:
		if( p->dDo.innerCode == cur ) {
			if( p->dDo.expression ) return p->dDo.expression;
		}
		break;
	case ST_TRY:
		if( p->dTry.innerCode == cur ) {
			if( p->dTry.catchParam ) return p->dTry.catchParam;
			if( p->dTry.catchCode ) return p->dTry.catchCode;
		}
		if( p->dTry.catchParam == cur ) {
			if( p->dTry.catchCode ) return p->dTry.catchCode;
		}
		break;
	}
	if( cur->next ) return cur->next;
	return NULL;
}

PARSEINF * ParseInfoWalker::next()
{
	if( cur == NULL ) {
		cur	= top;
		return top;
	}

	PARSEINF * next	= seekDown();
	if( next ) {
		cur	= next;
		return next;
	}

	next	= seekNext();
	if( next ) {
		cur	= next;
		return next;
	}

	while(cur) {
		if( cur == top ) {
			cur	= NULL;
			return NULL;
		}

		// up
		next		= seekUp();
		if( !next || next == top ) {
			cur	= NULL;
			return NULL;
		}
		cur		= next;
		next	= seekNext();
		if( next ) {
			cur	= next;
			return next;
		}
	}
	cur = NULL;
	return NULL;
}

PARSEINF * ParseInfoWalker::nextStatement()
{
	if( cur == NULL ) {
		cur	= top;
		return top;
	}

	// seek(2) next
	PARSEINF * next = seekNext();
	if( next ) {
		cur	= next;
		return next;
	}

	while(cur) {
		if( cur == top ) {
			cur	= NULL;
			return NULL;
		}
		next = seekUp();
		if( !next ) {
			cur	= NULL;
			return NULL;
		}
		cur		= next;
		next	= seekNext();
		if( next ) {
			cur	= next;
			return next;
		}
	}
	cur = NULL;
	return NULL;
}

Parser::Parser(IContext *ctx) {
	this->ctx			= ctx;
	this->lex			= NULL;
#ifdef ELISE_BLOCKPARSE
	this->parseBlock	= NULL;
#endif
}

Parser::~Parser()
{

}

PARSEINF * Parser::parse(Lexer * lex)
{
#ifdef ELISE_BLOCKPARSE
	if( parseBlock ) {
		freeParseBlock((PARSEINF *)((BYTE*)parseBlock)+sizeof(SIZE)*2+sizeof(BYTE**));
	}
#endif
	this->lex = lex;
	PARSEINF *pgm = createParseInfo(ST_PROGRAM);

	PARSEINF * prev=NULL;
	TokenId token;
	while( (token = lex->getToken()) ) {
		if( token == TID_EOF ) break;
		if( token == TID_ERROR ) {
			releaseParseInfo(pgm);
			setError(ERROR_TOKEN, lex->getSourceURL(), lex->getCurrentLine());
			return NULL;
		}
		if( token == TID_EOS ) continue;
#ifdef _DEBUG
		int lastLine = lex->getCurrentLine();
		if( lastLine == 1584 ) {
			int a=0;
		}
#endif
		PARSEINF * code = NULL;
//		if( mode == PARSE_MODE_IMPORT ) {
//			code = StageClassBlockStatement(token);
//		} else {
			code = stageStatement(token);
//		}
		if( code == NULL ) {
			releaseParseInfo(pgm);
			return NULL;
		}
		if( prev ) {
			prev->next	= code;
			code->prev	= prev;
		} else {
			pgm->dProgram.innerCode = code;
		}
		code->parent	= pgm;
		prev = code;
	}
	pgm->size	= lex->getCurrentPos() - pgm->position;
#ifdef ELISE_BLOCKPARSE
	parseBlock = NULL;
#endif
	return pgm;
}

IParseInfoWalker *Parser::createParseInfoWalker(PARSEINF* info)
{
	return new ParseInfoWalker(ctx, info);
}

#ifdef ELISE_BLOCKPARSE
PARSEINF *Parser::allocNode(SIZE size)
{
	BYTE *pSeg  = parseBlock;
	BYTE *pLast = NULL;
	while(pSeg) {
		BYTE *ptr = pSeg;
		SIZE *nSize = (SIZE*)ptr;	ptr += sizeof(SIZE);
		SIZE *nUsed = (SIZE*)ptr;	ptr += sizeof(SIZE);
		BYTE **ppNext = (BYTE**)ptr;	ptr += sizeof(BYTE**);
		if( *nSize-*nUsed>size ) {
			PARSEINF *info = (PARSEINF*)(pSeg+*nUsed);
			*nUsed += size;
			return info;
		}
		pLast = pSeg;
		pSeg = *ppNext;
	}
	SIZE newSize = 1024*512;
	BYTE *pNewSeg = (BYTE*)allocMemory(newSize);
	BYTE *ptr = pNewSeg;
	*(SIZE*)ptr = newSize;	ptr += sizeof(SIZE);
	*(SIZE*)ptr = sizeof(SIZE)*2+sizeof(BYTE**)+size;	ptr += sizeof(SIZE);
	*(BYTE**)ptr = NULL;	ptr += sizeof(BYTE**);
	if( pLast ) {
		BYTE *ptr = pLast;
		*(BYTE**)(ptr+sizeof(SIZE)*2) = pNewSeg;
	} else {
		parseBlock = pNewSeg;
	}
	return (PARSEINF*)ptr;
}

void Parser::freeParseBlock(PARSEINF *pgm)
{
	BYTE *pTopSeg = (BYTE*)pgm;
	pTopSeg -= sizeof(SIZE)*2+sizeof(BYTE**);

	PARSEINF def;
	BYTE *ptr = pTopSeg;
	while(ptr) {
		BYTE *pSeg = ptr;
		SIZE nSize = *(SIZE*)ptr;	ptr += sizeof(SIZE);
		SIZE nUsed = *(SIZE*)ptr;	ptr += sizeof(SIZE);
		BYTE *pNext = *(BYTE**)ptr;	ptr += sizeof(BYTE**);
		SIZE nSeek = 0;
		while(nSeek < nUsed) {
			PARSEINF *info = (PARSEINF*)ptr;
			SIZE infoSize=0;
			SIZE infoHeaderSize = sizeof(def) - ((BYTE*)(&def.dProgram)-(BYTE*)&def);

			BYTE *buff = (BYTE*)allocMemory(infoHeaderSize+infoSize);
			switch(info->type) {
			case ST_END:
				infoSize	= 0;
				break;
			case ST_PROGRAM:
				infoSize	= sizeof(PINF_PROGRAM);
				clearString(&info->dProgram.URL);
				break;
			case ST_NOP:
				infoSize	= 0;
				break;
			case ST_EXPRESSION:
				infoSize	= sizeof(PINF_EXPRESSION);
				switch(info->dExpression.exprType) {
				case ET_DATA:
					clearVariant(&info->dExpression.data.v);
					break;
				case ET_PARA:
					clearString(&info->dExpression.expPara.name);
					break;
				case ET_TYPE:
					clearString(&info->dExpression.expType.typeName);
					break;
				}
				break;
			case ST_BLOCK:
				infoSize	= sizeof(PINF_BLOCK);
				break;
			case ST_IMPORT:
				infoSize	= sizeof(PINF_IMPORT);
				clearString(&info->dImport.sourceURL);
				break;
			case ST_USE:
				infoSize	= sizeof(PINF_USE);
				break;
			case ST_PACKAGE:
				infoSize	= sizeof(PINF_PACKAGE);
				clearString(&info->dPackage.name);
				break;
			case ST_CLASS:
				infoSize	= sizeof(PINF_CLASS);
				clearString(&info->dClass.name);
				break;
			case ST_TEMPLATE:
				infoSize	= sizeof(PINF_TEMPLATE);
				clearString(&info->dTemplate.name);
				break;
			case ST_INTERFACE:
				infoSize	= sizeof(PINF_INTERFACE);
				clearString(&info->dInterface.name);
				break;
			case ST_PROPERTY:
				infoSize	= sizeof(PINF_PROPERTY);
				clearString(&info->dProperty.name);
				break;
			case ST_METHOD:
				infoSize	= sizeof(PINF_METHOD);
				clearString(&info->dMethod.name);
				break;
			case ST_FUNCTION:
				infoSize	= sizeof(PINF_FUNCTION);
				clearString(&info->dFunction.name);
				break;
			case ST_OBJECT:
				infoSize	= sizeof(PINF_OBJECT);
				clearString(&info->dObject.name);
				break;
			case ST_RETURN:
				infoSize	= sizeof(PINF_RETURN);
				break;
			case ST_IF:
				infoSize	= sizeof(PINF_IF);
				break;
			case ST_SWITCH:
				infoSize	= sizeof(PINF_SWITCH);
				break;
			case ST_CASE:
				infoSize	= sizeof(PINF_CASE);
				break;
			case ST_DEFAULT:
				infoSize	= sizeof(PINF_DEFAULT);
				break;
			case ST_FOR:
				infoSize	= sizeof(PINF_FOR);
				break;
			case ST_FORIN:
				infoSize	= sizeof(PINF_FORIN);
				break;
			case ST_WHILE:
				infoSize	= sizeof(PINF_WHILE);
				break;
			case ST_DO:
				infoSize	= sizeof(PINF_DO);
				break;
			case ST_BREAK:
				break;
			case ST_CONTNUE:
				break;
			case ST_TRY:
				infoSize	= sizeof(PINF_TRY);
				break;
			case ST_THROW:
				infoSize	= sizeof(PINF_THROW);
				break;
			}
			SIZE nNodeSize = infoHeaderSize+infoSize;
			ptr += nNodeSize;
			nSeek += nNodeSize;
		}
		freeMemory(pSeg);
		ptr = pNext;
	}
}
#endif

void Parser::releaseParseInfo(PARSEINF* info)
{
#ifdef ELISE_BLOCKPARSE
//	freeParseBlock(info);
	return;
#else
	//////////////
	PARSEINF * n;
	while( info ) {
		n = info->next;
		switch(info->type) {
		case ST_END:
			break;
		case ST_PROGRAM:
			releaseParseInfo(info->dProgram.innerCode);
			clearString(&info->dProgram.URL);
			break;
		case ST_NOP:
			break;
		case ST_EXPRESSION:
			switch(info->dExpression.exprType) {
			case ET_NOP:
				break;
			case ET_DATA:
				clearVariant(&info->dExpression.data.v);
				break;
			case ET_OPR1:
				releaseParseInfo(info->dExpression.exp1.expresion);
				break;
			case ET_OPR2:
				releaseParseInfo(info->dExpression.exp2.expresion1);
				releaseParseInfo(info->dExpression.exp2.expresion2);
				break;
			case ET_OPR3:
				releaseParseInfo(info->dExpression.exp3.expresion1);
				releaseParseInfo(info->dExpression.exp3.expresion2);
				releaseParseInfo(info->dExpression.exp3.expresion3);
				break;
			case ET_OPRS:
				if( info->dExpression.token == TID_CALL ) {
					releaseParseInfo(info->dExpression.expCall.addr);
					releaseParseInfo(info->dExpression.expCall.argv);
				}
				if( info->dExpression.token == TID_INDEX ) {
					releaseParseInfo(info->dExpression.expIndex.addr);
					releaseParseInfo(info->dExpression.expIndex.index);
				}
				break;
			case ET_OPRE:
				break;
			case ET_PARA:
				clearString(&info->dExpression.expPara.name);
				break;
			case ET_PATH:
				releaseParseInfo(info->dExpression.expPath.symbol);
				break;
			case ET_TYPE:
				clearString(&info->dExpression.expType.typeName);
				releaseParseInfo(info->dExpression.expType.defaultTypePath);
				break;
			}
			break;
		case ST_BLOCK:
			releaseParseInfo(info->dBlock.innerCode);
			break;
		case ST_IMPORT:
			clearString(&info->dImport.sourceURL);
			break;
		case ST_USE:
			releaseParseInfo(info->dUse.addr);
			break;
		case ST_PACKAGE:
			clearString(&info->dPackage.name);
			releaseParseInfo(info->dPackage.innerCode);
			break;
		case ST_CLASS:
			clearString(&info->dClass.name);
			releaseParseInfo(info->dClass.basePath);
			releaseParseInfo(info->dClass.interfaces);
			releaseParseInfo(info->dClass.innerCode);
			break;
		case ST_TEMPLATE:
			clearString(&info->dTemplate.name);
			releaseParseInfo(info->dTemplate.typePath);
			releaseParseInfo(info->dTemplate.basePath);
			releaseParseInfo(info->dTemplate.interfaces);
			releaseParseInfo(info->dTemplate.innerCode);
			break;
		case ST_INTERFACE:
			clearString(&info->dInterface.name);
			releaseParseInfo(info->dInterface.interfaces);
			releaseParseInfo(info->dInterface.innerCode);
			break;
		case ST_PROPERTY:
			clearString(&info->dProperty.name);
			releaseParseInfo(info->dProperty.path);
			releaseParseInfo(info->dProperty.array);
			releaseParseInfo(info->dProperty.initCode);
			break;
		case ST_METHOD:
			clearString(&info->dMethod.name);
			releaseParseInfo(info->dMethod.params);
			releaseParseInfo(info->dMethod.innerCode);
			break;
		case ST_FUNCTION:
			clearString(&info->dFunction.name);
			releaseParseInfo(info->dFunction.params);
			releaseParseInfo(info->dFunction.innerCode);
			break;
		case ST_OBJECT:
			clearString(&info->dObject.name);
			releaseParseInfo(info->dObject.path);
			releaseParseInfo(info->dObject.array);
			releaseParseInfo(info->dObject.initExpression);
			releaseParseInfo(info->dObject.innerCode);
			break;
		case ST_RETURN:
			releaseParseInfo(info->dReturn.resultValue);
			break;
		case ST_IF:
			releaseParseInfo(info->dIf.expression);
			releaseParseInfo(info->dIf.trueCode);
			releaseParseInfo(info->dIf.falseCode);
			break;
		case ST_SWITCH:
			releaseParseInfo(info->dSwitch.expression);
			releaseParseInfo(info->dSwitch.caseList);
			break;
		case ST_CASE:
			releaseParseInfo(info->dCase.expression);
			releaseParseInfo(info->dCase.caseCode);
			break;
		case ST_DEFAULT:
			releaseParseInfo(info->dDefault.caseCode);
			break;
		case ST_FOR:
			releaseParseInfo(info->dFor.initExpression);
			releaseParseInfo(info->dFor.condExpression);
			releaseParseInfo(info->dFor.nextExpression);
			releaseParseInfo(info->dFor.innerCode);
			break;
		case ST_FORIN:
			releaseParseInfo(info->dForin.variableExpression);
			releaseParseInfo(info->dForin.targetExpression);
			releaseParseInfo(info->dForin.innerCode);
			break;
		case ST_WHILE:
			releaseParseInfo(info->dWhile.expression);
			releaseParseInfo(info->dWhile.innerCode);
			break;
		case ST_DO:
			releaseParseInfo(info->dDo.expression);
			releaseParseInfo(info->dDo.innerCode);
			break;
		case ST_BREAK:
		case ST_CONTNUE:
			break;
		case ST_TRY:
			releaseParseInfo(info->dTry.innerCode);
			releaseParseInfo(info->dTry.catchParam);
			releaseParseInfo(info->dTry.catchCode);
			break;
		case ST_THROW:
			releaseParseInfo(info->dThrow.exceptionCode);
			break;
		}
		freeMemory(info);
		info	= n;
	}
	return;
#endif
}

PARSEINF * Parser::createParseInfo(StatementType st)
{
	PARSEINF def;
	SIZE infoSize=0;
	SIZE infoHeaderSize = sizeof(def) - ((BYTE*)(&def.dProgram)-(BYTE*)&def);
	switch(st) {
	case ST_END:		infoSize	= 0;	break;
	case ST_PROGRAM:	infoSize	= sizeof(PINF_PROGRAM);	break;
	case ST_NOP:		infoSize	= 0;	break;
	case ST_EXPRESSION:	infoSize	= sizeof(PINF_EXPRESSION);break;
	case ST_BLOCK:		infoSize	= sizeof(PINF_BLOCK);	break;
	case ST_IMPORT:		infoSize	= sizeof(PINF_IMPORT);	break;
	case ST_PACKAGE:	infoSize	= sizeof(PINF_PACKAGE);	break;
	case ST_CLASS:		infoSize	= sizeof(PINF_CLASS);	break;
	case ST_TEMPLATE:	infoSize	= sizeof(PINF_TEMPLATE);break;
	case ST_INTERFACE:	infoSize	= sizeof(PINF_INTERFACE);break;
	case ST_PROPERTY:	infoSize	= sizeof(PINF_PROPERTY);break;
	case ST_METHOD:		infoSize	= sizeof(PINF_METHOD);	break;
	case ST_FUNCTION:	infoSize	= sizeof(PINF_FUNCTION);break;
	case ST_OBJECT:		infoSize	= sizeof(PINF_OBJECT);	break;
	case ST_RETURN:		infoSize	= sizeof(PINF_RETURN);	break;
	case ST_IF:			infoSize	= sizeof(PINF_IF);		break;
	case ST_SWITCH:		infoSize	= sizeof(PINF_SWITCH);	break;
	case ST_CASE:		infoSize	= sizeof(PINF_CASE);	break;
	case ST_DEFAULT:	infoSize	= sizeof(PINF_DEFAULT);	break;
	case ST_FOR:		infoSize	= sizeof(PINF_FOR);		break;
	case ST_FORIN:		infoSize	= sizeof(PINF_FORIN);	break;
	case ST_WHILE:		infoSize	= sizeof(PINF_WHILE);	break;
	case ST_DO:			infoSize	= sizeof(PINF_DO);		break;
	case ST_BREAK:		infoSize	= 0;	break;
	case ST_CONTNUE:	infoSize	= 0;	break;
	case ST_TRY:		infoSize	= sizeof(PINF_TRY);		break;
	case ST_THROW:		infoSize	= sizeof(PINF_THROW);	break;
	case ST_USE:		infoSize	= sizeof(PINF_USE);		break;
	default:	return NULL;
	}

#ifdef ELISE_BLOCKPARSE
	BYTE *buff = (BYTE*)allocNode(infoHeaderSize+infoSize);
#else
	BYTE *buff = (BYTE*)allocMemory(infoHeaderSize+infoSize);
#endif

	if( !buff ) return NULL;
	fillMemory(buff, 0, infoHeaderSize+infoSize);

	PARSEINF* info = (PARSEINF*)buff;
	info->type 		= st;
	info->line		= lex->getCurrentLine();
	info->position	= lex->getTokenStartPos();
	info->size		= lex->getCurrentPos() - info->position;
	return info;
}

PARSEINF * Parser::createExpression(ExpressionType et, TokenId token)
{
	PARSEINF * pac = NULL;
	TokenType tt = (TokenType)(token & TT_MASK);

	switch(tt) {
	case TT_LITERAL:
		pac = createLiteralExpression(token);
		break;

	case TT_SYMBOL:
		pac = createSymbolExpression(token);
		break;

	case TT_OPERATOR:

		switch(token) {
		case TID_STC:
			if( et == ET_DATA ) token	= TID_CALL;
			break;
		case TID_PLUSPLUS:
			if( et == ET_DATA ) {
				token	= TID_PLUSLEFT;
			} else {
				token	= TID_PLUSRIGHT;
			}
			break;
		case TID_MINUSMINUS:
			if( et == ET_DATA ) {
				token	= TID_MINUSLEFT;
			} else {
				token	= TID_MINUSRIGHT;
			}
			break;
		case TID_PLUS:
		case TID_MINUS:
			if( et == ET_NOP  || et == ET_OPR1 ||
				et == ET_OPR2 || et == ET_OPR3 || et == ET_OPRS ) {
				if( token == TID_PLUS ) {
					token = TID_FPLUS;
				} else {
					token = TID_FMINUS;
				}
			}
			break;
		}
		switch(token) {
		case TID_DOT:			pac = createOperator(ET_OPR2, 1, AM_LEFT,  ACT_READ, token);	break;
		case TID_STC:			pac = createOperator(ET_OPRS, 1, AM_LEFT,  ACT_READ, token);	break;
		case TID_EDC:			pac = createOperator(ET_OPRE, 1, AM_LEFT,  ACT_READ, token);	break;
		case TID_EDA:			pac = createOperator(ET_OPRE, 1, AM_LEFT,  ACT_READ, token);	break;

		case TID_SPC:			pac = createOperator(ET_OPR2, 2, AM_RIGHT,  ACT_READ, token);	break;

		case TID_INDEX:			pac = createOperator(ET_OPRS, 3, AM_LEFT,  ACT_READ, token);	break;
		case TID_CALL:			pac = createOperator(ET_OPRS, 3, AM_LEFT,  ACT_READ, token);	break;

		case TID_PLUSLEFT:		pac = createOperator(ET_OPR1, 4, AM_LEFT,  ACT_WRITE, token);	break;
		case TID_MINUSLEFT:		pac = createOperator(ET_OPR1, 4, AM_LEFT,  ACT_WRITE, token);	break;
		case TID_PLUSRIGHT:		pac = createOperator(ET_OPR1, 5, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_MINUSRIGHT:	pac = createOperator(ET_OPR1, 5, AM_RIGHT, ACT_WRITE, token);	break;

		case TID_NOT:			pac = createOperator(ET_OPR1, 5, AM_RIGHT, ACT_READ, token);	break;
		case TID_BITNOT:		pac = createOperator(ET_OPR1, 5, AM_RIGHT, ACT_READ, token);	break;
		case TID_FPLUS:			pac = createOperator(ET_OPR1, 5, AM_RIGHT, ACT_READ, token);	break;
		case TID_FMINUS:		pac = createOperator(ET_OPR1, 5, AM_RIGHT, ACT_READ, token);	break;
		case TID_NEW:			pac = createOperator(ET_OPR1, 5, AM_RIGHT, ACT_READ, token);	break;

		case TID_MUL:			pac = createOperator(ET_OPR2, 6, AM_LEFT,  ACT_READ, token);	break;
		case TID_DIV:			pac = createOperator(ET_OPR2, 6, AM_LEFT,  ACT_READ, token);	break;
		case TID_MOD:			pac = createOperator(ET_OPR2, 6, AM_LEFT,  ACT_READ, token);	break;

		case TID_PLUS:			pac = createOperator(ET_OPR2, 7, AM_LEFT,  ACT_READ, token);	break;
		case TID_MINUS:			pac = createOperator(ET_OPR2, 7, AM_LEFT,  ACT_READ, token);	break;

		case TID_LSHIFT:		pac = createOperator(ET_OPR2, 8, AM_LEFT,  ACT_READ, token);	break;
		case TID_RSHIFT:		pac = createOperator(ET_OPR2, 8, AM_LEFT,  ACT_READ, token);	break;
		case TID_RRSHIFT:		pac = createOperator(ET_OPR2, 8, AM_LEFT,  ACT_READ, token);	break;

		case TID_GT:			pac = createOperator(ET_OPR2, 9, AM_LEFT,  ACT_READ, token);	break;
		case TID_LT:			pac = createOperator(ET_OPR2, 9, AM_LEFT,  ACT_READ, token);	break;
		case TID_GE:			pac = createOperator(ET_OPR2, 9, AM_LEFT,  ACT_READ, token);	break;
		case TID_LE:			pac = createOperator(ET_OPR2, 9, AM_LEFT,  ACT_READ, token);	break;
		case TID_INSTANCEOF:	pac = createOperator(ET_OPR2, 9, AM_LEFT,  ACT_READ, token);	break;
		case TID_TYPEOF:		pac = createOperator(ET_OPR2, 9, AM_LEFT,  ACT_READ, token);	break;
		case TID_SIZEOF:		pac = createOperator(ET_OPR1, 9, AM_RIGHT, ACT_READ, token);	break;

		case TID_EQEQ:			pac = createOperator(ET_OPR2, 10, AM_LEFT,  ACT_READ, token);	break;
		case TID_NE:			pac = createOperator(ET_OPR2, 10, AM_LEFT,  ACT_READ, token);	break;

		case TID_AND:			pac = createOperator(ET_OPR2,11, AM_LEFT,  ACT_READ, token);	break;
		case TID_XOR:			pac = createOperator(ET_OPR2,12, AM_LEFT,  ACT_READ, token);	break;
		case TID_OR:			pac = createOperator(ET_OPR2,13, AM_LEFT,  ACT_READ, token);	break;

		case TID_ANDAND:		pac = createOperator(ET_OPR2,14, AM_LEFT,  ACT_READ, token);	break;
		case TID_OROR:			pac = createOperator(ET_OPR2,15, AM_LEFT,  ACT_READ, token);	break;

		case TID_COL:			pac = createOperator(ET_OPR3,16, AM_RIGHT, ACT_READ, token);	break;
		case TID_QU:			pac = createOperator(ET_OPR3,16, AM_RIGHT, ACT_READ, token);	break;

		case TID_EQ:			pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_MULEQ:			pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_DIVEQ:			pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_MODEQ:			pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_PLUSEQ:		pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_MINUSEQ:		pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_LSHIFTEQ:		pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_RSHIFTEQ:		pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_ANDEQ:			pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_XOREQ:			pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_OREQ:			pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_REFEQ:			pac = createOperator(ET_OPR2,17, AM_RIGHT, ACT_WRITE, token);	break;
		case TID_COM:			pac = createOperator(ET_OPR2,18, AM_LEFT,  ACT_READ, token);	break;
		case TID_VAR:			pac = createOperator(ET_OPR1,19, AM_RIGHT, ACT_READ, token);	break;
			break;
		default:
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			break;
		}
		break;
	}
	return pac;
}

PARSEINF * Parser::createLiteralExpression(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_EXPRESSION);
	pc->dExpression.exprType	= ET_DATA;
	pc->dExpression.token		= token;
	pc->dExpression.accessType	= ACT_READ;
	lex->getTokenValue(pc->dExpression.data.v);
	return pc;
}

PARSEINF * Parser::createSymbolExpression(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_EXPRESSION);
	pc->dExpression.exprType	= ET_DATA;
	pc->dExpression.token		= token;
	lex->getTokenValue(pc->dExpression.data.v);
	return pc;
}

PARSEINF * Parser::createOperator(ExpressionType et, INT prec, AssociateMode ascm, AccessType acc, TokenId tid)
{
	PARSEINF * pc = createParseInfo(ST_EXPRESSION);
	pc->dExpression.exprType	= et;
	pc->dExpression.token		= tid;
	pc->dExpression.precedence	= prec;
	pc->dExpression.associate	= ascm;
	pc->dExpression.accessType	= acc;
	switch(et) {
	case ET_OPR1:
		pc->dExpression.exp1.expresion		= NULL;
		break;

	case ET_OPR2:
		pc->dExpression.exp2.expresion1	= NULL;
		pc->dExpression.exp2.expresion2	= NULL;
		break;

	case ET_OPR3:
		pc->dExpression.exp3.expresion1	= NULL;
		pc->dExpression.exp3.expresion2	= NULL;
		pc->dExpression.exp3.expresion3	= NULL;
		break;

	case ET_OPRS:
	case ET_OPRE:
	case ET_PARA:
		break;
	default:
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	return pc;
}

PARSEINF * Parser::createPathExpression(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_EXPRESSION);
	pc->dExpression.exprType	= ET_PATH;
	pc->dExpression.token		= token;
	return pc;
}

PARSEINF * Parser::createParaExpression(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_EXPRESSION);
	pc->dExpression.exprType	= ET_PARA;
	pc->dExpression.token		= token;
	Variant v;
	lex->getTokenValue(*v);
	copyString( &pc->dExpression.expPara.name, &v->dString);
	pc->dExpression.expPara.array = -1;
	return pc;
}

PARSEINF * Parser::createTypeExpression(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_EXPRESSION);
	pc->dExpression.exprType	= ET_TYPE;
	pc->dExpression.token		= token;
	Variant v;
	lex->getTokenValue(*v);
	copyString( &pc->dExpression.expType.typeName, &v->dString);
	pc->dExpression.expType.defaultTypePath	= NULL;
	return pc;
}

bool Parser::stackToken(ExpressionType &prev_type, PARSEINF * pac, ParseStack &cs, ParseStack &ds)
{
	if( isDataExpression(pac) ) {
		if( isDataExpression(prev_type) ) {
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			return false;
		}
		ds.push(pac);
		prev_type = pac->dExpression.exprType;
		return true;
	}

	if( !isOperatorExpression(pac) ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return false;
	}

	PARSEINF * op = pac;
	if( op->dExpression.exprType == ET_OPRE ) {
		if( !reduceC(op, cs, ds) ) {
			releaseParseInfo(op);
			return false;
		}
		if( op->dExpression.exprType == ET_OPRE ) {
			releaseParseInfo(op);
			prev_type = ET_DATA;
		} else {
			prev_type = op->dExpression.exprType;
		}
	} else {
		if( !reduceCheck(op, cs, ds) ) {
			return false;
		}
		prev_type = op->dExpression.exprType;
	}
	return true;
}

bool Parser::reduceCheck(PARSEINF * pac, ParseStack &cs, ParseStack &ds)
{
	if( cs.size() == 0 ) {
		cs.push(pac);
		return true;
	}

	INT r0 = pac->dExpression.precedence;
	PARSEINF * op;
	while(cs.size() != 0) {
		op	= cs.top();
		if( !isOperatorExpression(op) ) {
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			return false;
		}
		INT r1 = op->dExpression.precedence;
		//
		if( op->dExpression.exprType == ET_OPRS ) break;

		if( r0 > r1) {
			cs.pop();
			if( !reduce(op, cs, ds) ) {
				return false;
			}
		} else if( r0 == r1 ) {
			if( op->dExpression.associate == AM_LEFT ) {
				cs.pop();
				if( !reduce(op, cs, ds) ) {
					return false;
				}
			} else {
				break;
			}
		} else {
			break;
		}
	}
	cs.push(pac);
	return true;
}

bool Parser::reduce(PARSEINF * pac, ParseStack &cs, ParseStack &ds)
{
	switch( pac->dExpression.exprType ) {
	case ET_OPRE:	return reduceC(pac, cs, ds);
	case ET_OPR1:	return reduce1(pac, cs, ds);
	case ET_OPR2:	return reduce2(pac, cs, ds);
	case ET_OPR3:	return reduce3(pac, cs, ds);
	}
	setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
	return false;
}

bool Parser::reduce1(PARSEINF * pac, ParseStack &cs, ParseStack &ds)
{
	if( ds.empty() ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return false;
	}
	PARSEINF * dw0	= ds.pop();
	if( pac->dExpression.exp1.expresion ) {
		releaseParseInfo(pac->dExpression.exp1.expresion);
	}
	dw0->dExpression.accessType = pac->dExpression.accessType;
	pac->dExpression.exp1.expresion	= dw0;
	dw0->parent	= pac;
	ds.push(pac);
	return true;
}

bool Parser::reduce2(PARSEINF * pac, ParseStack &cs, ParseStack &ds)
{
	if( ds.size() < 2 ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return false;
	}
	PARSEINF * rhd	= ds.pop();
	PARSEINF * lhd	= ds.pop();
	if( pac->dExpression.exp2.expresion1 ) {
		releaseParseInfo(pac->dExpression.exp2.expresion1);
	}
	if( pac->dExpression.exp2.expresion2 ) {
		releaseParseInfo(pac->dExpression.exp2.expresion2);
	}
	if( pac->dExpression.accessType == ACT_WRITE ) {
		if( pac->dExpression.associate == AM_RIGHT ) {
			lhd->dExpression.accessType = ACT_WRITE;
		} else if( pac->dExpression.associate == AM_LEFT ) {
			rhd->dExpression.accessType = ACT_WRITE;
		}
	}
	pac->dExpression.exp2.expresion1 = lhd;
	pac->dExpression.exp2.expresion2 = rhd;
	lhd->parent = pac;
	rhd->parent	= pac;
	ds.push(pac);
	return true;
}

bool Parser::reduce3(PARSEINF * pac, ParseStack &cs, ParseStack &ds)
{
	if( cs.size() < 1 || ds.size() < 3 ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return false;
	}
	PARSEINF * rhd	= ds.pop();

	PARSEINF * op = NULL;
	while(!cs.empty()) {
		op = cs.pop();
		if( op->dExpression.token  == TID_QU ) break;
		if( !reduce(op, cs, ds) ) return false;
	}
	if( !op || op->dExpression.token != TID_QU ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return false;
	}
	releaseParseInfo(op);

	// データスタックから1,2項をPOP
	if( ds.size() < 2 ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return false;
	}
	PARSEINF * lhd	= ds.pop();
	PARSEINF * cnd	= ds.pop();
	if( pac->dExpression.exp3.expresion1 ) {
		releaseParseInfo(pac->dExpression.exp3.expresion1);
	}
	if( pac->dExpression.exp3.expresion2 ) {
		releaseParseInfo(pac->dExpression.exp3.expresion2);
	}
	if( pac->dExpression.exp3.expresion3 ) {
		releaseParseInfo(pac->dExpression.exp3.expresion3);
	}
	pac->dExpression.exp3.expresion1	= cnd;
	pac->dExpression.exp3.expresion2	= lhd;
	pac->dExpression.exp3.expresion3	= rhd;
	cnd->parent	= pac;
	lhd->parent	= pac;
	rhd->parent	= pac;
	ds.push(pac);
	return true;
}

bool Parser::reduceC(PARSEINF * pac, ParseStack &cs, ParseStack &ds)
{
	INT argc=0;
	PARSEINF * cd;
	// 閉じ括弧は、開き括弧まで REDUCE
	while(1) {
		if( cs.empty() ) {
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			return false;
		}
		cd = cs.pop();
		if( pac->dExpression.token== TID_EDC ) {
			if( cd->dExpression.token == TID_STC ) {
				if( !ds.empty() ) {
					PARSEINF * ccd = ds.top();
					if( cd->position <= ccd->position  ) {
						if( ccd->dExpression.exprType == ET_OPR2 && ccd->dExpression.token == TID_COM) {
							setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
							return false;
						}
						if( ccd->dExpression.exprType == ET_DATA && ccd->dExpression.token == TID_SYMBOL ) {
							VariantType vt = ctx->getTypeManager()->isPrimitiveName(getStringReadPtr(&ccd->dExpression.data.v.dString));
							if( vt != VT_NULL && vt != VT_REF ) {
								switch(vt) {
								case VT_BOOL:	pac->dExpression.token = TID_CAST_BOOL;		break;
								case VT_CHAR:	pac->dExpression.token = TID_CAST_CHAR;		break;
								case VT_BYTE:	pac->dExpression.token = TID_CAST_BYTE;		break;
								case VT_WORD:	pac->dExpression.token = TID_CAST_WORD;		break;
								case VT_DWORD:	pac->dExpression.token = TID_CAST_DWORD;	break;
								case VT_SHORT:	pac->dExpression.token = TID_CAST_SHORT;	break;
								case VT_INT:	pac->dExpression.token = TID_CAST_INT;		break;
								case VT_DECIMAL:pac->dExpression.token = TID_CAST_DECIMAL;	break;
								case VT_FLOAT:	pac->dExpression.token = TID_CAST_FLOAT;	break;
								case VT_BYTES:	pac->dExpression.token = TID_CAST_BYTES;	break;
								case VT_STRING:	pac->dExpression.token = TID_CAST_STRING;	break;
								default:
									setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
									return false;
								}
								pac->dExpression.exprType		= ET_OPR1;
								pac->dExpression.accessType		= ACT_READ;
								pac->dExpression.associate		= AM_RIGHT;
								pac->dExpression.precedence		= 5;
								pac->dExpression.exp1.expresion	= NULL;
								ds.pop();
								releaseParseInfo(ccd);
								cs.push(pac);
								return true;
							}
						}
					}
				}
				break;
			}
			if( cd->dExpression.token == TID_CALL ) {
				if( ds.empty() ) {
					setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
					return false;
				}
				PARSEINF * addr = ds.pop();
				PARSEINF * argv = NULL;
				if( cd->position <= addr->position ) {
					if( ds.empty() ) {
						setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
						return false;
					}
					argv = addr;
					addr = ds.pop();
				}
				if( argv != NULL ) {
					argc++;
					PARSEINF * ccd = argv;
					while( ccd->dExpression.exprType == ET_OPR2 && ccd->dExpression.token == TID_COM) {
						argc++;
						if( ccd->dExpression.exp2.expresion1->dExpression.token == TID_COM ) {
							ccd	= ccd->dExpression.exp2.expresion1;
						} else if( ccd->dExpression.exp2.expresion2->dExpression.token == TID_COM ) {
							ccd	= ccd->dExpression.exp2.expresion2;
						} else {
							break;
						}
					}
				}
				cd->dExpression.expCall.addr	= addr;
				cd->dExpression.expCall.argc	= argc;
				cd->dExpression.expCall.argv	= argv;
				addr->parent	= cd;
				if( argv ) argv->parent		= cd;
				ds.push(cd);
				break;
			}
		} else if( pac->dExpression.token == TID_EDA ) {
			if( cd->dExpression.token == TID_INDEX ) {
				if( !ds.empty() ) {
					PARSEINF * ccd = ds.top();
					if( cd->position <= ccd->position  ) {
						if( ccd->dExpression.exprType == ET_OPR2 && ccd->dExpression.token == TID_COM) {
							setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
							return false;
						}
					}
				}
				if( ds.empty() ) {
					setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
					return false;
				}
				PARSEINF * addr = ds.pop();
				PARSEINF * indx = NULL;
				if( cd->position <= addr->position ) {
					if( ds.empty() ) {
						setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
						return false;
					}
					indx = addr;
					addr = ds.pop();
				}
				cd->dExpression.expIndex.addr	= addr;
				cd->dExpression.expIndex.index	= indx;
				addr->parent	= cd;
				if( indx ) indx->parent	= cd;
				if( indx != NULL ) {
					if( indx->dExpression.token == TID_COM ) {
						setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
						return false;
					}
				}
				ds.push(cd);
				break;
			}
		}
		if( !reduce(cd, cs, ds )) {
			return false;
		}
	}
	return true;
}

bool Parser::isDataExpression(PARSEINF * pc)
{
	if( pc->type != ST_EXPRESSION ) return false;
	return isDataExpression(pc->dExpression.exprType);
}

bool Parser::isDataExpression(ExpressionType et)
{
	return (et == ET_DATA /*|| et == ET_LITERAL || et == ET_SYMBOL*/);
}

bool Parser::isOperatorExpression(PARSEINF * pc)
{
	if( pc->type != ST_EXPRESSION ) return false;
	return isOperatorExpression(pc->dExpression.exprType);
}

bool Parser::isOperatorExpression(ExpressionType et)
{
	return (et == ET_OPR1 || et == ET_OPR2 || et == ET_OPR3 ||
		et == ET_OPRS ||	et == ET_OPRE || et == ET_PARA );
}

bool Parser::isLiteralExpression(PARSEINF * expr)
{
	if( !expr ) return true;
	if( expr->dExpression.accessType == ACT_WRITE ) return false;
	switch( expr->dExpression.exprType ) {
	case ET_OPR1:
		if( !isLiteralExpression(expr->dExpression.exp1.expresion) ) return false;
		break;
	case ET_OPR2:
		if( !isLiteralExpression(expr->dExpression.exp2.expresion1) ) return false;
		if( !isLiteralExpression(expr->dExpression.exp2.expresion2) ) return false;
		break;
	case ET_OPR3:
		if( !isLiteralExpression(expr->dExpression.exp3.expresion1) ) return false;
		if( !isLiteralExpression(expr->dExpression.exp3.expresion2) ) return false;
		if( !isLiteralExpression(expr->dExpression.exp3.expresion3) ) return false;
		break;
	case ET_DATA:
		if( expr->dExpression.token == TID_SYMBOL ) return false;
		break;
	default:
		return false;
	}
	return true;
}

bool Parser::uniqMemberCheck(PARSEINF * pc)
{
	while( pc && pc->next ) {
		const STRING * pName = NULL;
		switch(pc->type) {
		case ST_PROPERTY:	pName = &pc->dProperty.name;		break;
		case ST_FUNCTION:	pName = &pc->dFunction.name;	break;
		case ST_CLASS:		pName = &pc->dClass.name;		break;
		case ST_PACKAGE:	pName = &pc->dPackage.name;		break;
		case ST_INTERFACE:	pName = &pc->dInterface.name;	break;
		case ST_TEMPLATE:	pName = &pc->dTemplate.name;	break;
		}
		if( pName ) {
			PARSEINF * fp = pc->next;
			while( fp ) {
				const STRING * fName = NULL;
				switch(fp->type) {
				case ST_PROPERTY:	fName = &fp->dProperty.name;		break;
				case ST_FUNCTION:	fName = &fp->dFunction.name;	break;
				case ST_CLASS:		fName = &fp->dClass.name;		break;
				case ST_PACKAGE:	fName = &fp->dPackage.name;		break;
				case ST_INTERFACE:	fName = &fp->dInterface.name;	break;
				case ST_TEMPLATE:	fName = &fp->dTemplate.name;	break;
				}
				if( fName ) {
					if( isEqualString(pName, fName) ) {
						setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
						return false;
					}
				}
				fp = fp->next;
			}
		}
		pc = pc->next;
	}
	return true;
}

bool Parser::hasReturnCode(PARSEINF * pc)
{
	if( pc->type == ST_RETURN ) {
		return pc->dReturn.resultValue != NULL;
	}
	PARSEINF * p;
	PARSEINF * last = NULL;
	switch(pc->type) {
	case ST_BLOCK:
		p = pc->dBlock.innerCode;
		while(p) {
			last = p;
			p = p->next;
		}
		if( !last ) {
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			return false;
		}
		return hasReturnCode(last);

	case ST_IF:
		if( pc->dIf.trueCode == NULL || pc->dIf.falseCode == NULL ) {
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			return false;
		}
		p = pc->dIf.trueCode;
		while(p) {
			last = p;
			p = p->next;
		}
		if( !hasReturnCode(last) ) return false;
		p = pc->dIf.falseCode;
		while(p) {
			last = p;
			p = p->next;
		}
		return hasReturnCode(last);

	case ST_SWITCH:
		p = pc->dSwitch.caseList;
		while(p) {
			if( p->type == ST_DEFAULT ) {
				return hasReturnCode(p->dDefault.caseCode);
			}
			p = p->next;
		}
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return false;

	case ST_TRY:
		p = pc->dTry.innerCode;
		while(p) {
			last = p;
			p = p->next;
		}
		if( !last ) {
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			return false;
		}
		if( !hasReturnCode(last) ) return false;
		p = pc->dTry.catchCode;
		while(p) {
			last = p;
			p = p->next;
		}
		if( !last ) {
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			return false;
		}
		return hasReturnCode(last);
	}
	setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
	return false;
}

bool Parser::parseACM(TokenId token, AccessModifier &acm)
{
	acm = ACM_EMPTY;
	LEXERSTATE st;
	bool stateSvaed=false;
	lex->initState(st);
	do {
		AccessModifier p = ACM_EMPTY;
		switch(token) {
		case TID_PUBLIC:	p = ACM_EMPTY;		break;
		case TID_PROTECTED:	p = ACM_PROTECTED;	break;
		case TID_PRIVATE:	p = ACM_PRIVATE;	break;
		case TID_STATIC:	p = ACM_STATIC;		break;
		case TID_CONST:		p = ACM_CONST;		break;
		case TID_REPRESENT:	p = ACM_REPRESENT;	break;
		default:
			setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
			if( stateSvaed ) lex->freeState(st);
			return false;
		}
		if( (acm & ACM_SCOPEMASK) && (p & ACM_SCOPEMASK) ) {
			setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
			if( stateSvaed ) lex->freeState(st);
			return false;
		}
		acm = (AccessModifier)(acm|p);
		if( stateSvaed ) lex->freeState(st);
		lex->saveState(st);
		stateSvaed = true;
		token = lex->getToken();
	} while( (token & TT_MASK) == TT_MODIFIER );
	lex->restoreState(st);
	return true;
}

PARSEINF * Parser::stageStatement(TokenId token)
{
	PARSEINF * pc;
	switch(token) {
	case TID_EOS:			return stageEOS(token);
	case TID_IMPORT:		pc = stageImport(token); break;
	case TID_USE:			return stageUse(token);
	case TID_BEGINBLOCK:	return stageBlockStatement(token);
	case TID_RETURN:		pc = stageReturn(token); break;
	case TID_IF:			return stageIf(token);
	case TID_FOR:			return stageFor(token);
	case TID_WHILE:			return stageWhile(token);
	case TID_DO:			return stageDo(token);
	case TID_SWITCH:		return stageSwitch(token);
	case TID_BREAK:			pc = stageBreak(token); break;
	case TID_CONTINUE:		pc = stageContinue(token); break;
	case TID_TRY:			return stageTry(token);
	case TID_CATCH:			return stageCatch(token);
	case TID_THROW:			pc = stageThrow(token); break;
	case TID_PACKAGE:		return stagePackage(token);
	case TID_CLASS:			return stageClass(token, ACM_DEFAULT);
	case TID_INTERFACE:		return stageInterface(token, ACM_DEFAULT);
	case TID_TEMPLATE:		return stageTemplate(token, ACM_DEFAULT);
	case TID_FUNCTION:		return stageFunction(token);
	case TID_RESERVED:		return stageReserved(token);
	default:				return stageExprStatement(token);
	}
	if( pc == NULL ) {
		return NULL;
	}
	token = lex->getToken();
	if( token != TID_EOS ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	return pc;
}

PARSEINF * Parser::stageEOS(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_NOP);
	pc->size	= 0;
	return pc;
}

PARSEINF * Parser::stageExpression(TokenId token)
{
	PARSEINF * pc = NULL;
	bool	end_expr	= false;
	INT  cnest	= 0;
	INT  tnest	= 0;
	INT  qnest	= 0;
	ParseStack	cs;
	ParseStack	ds;
	ExpressionType prev_token = ET_NOP;
	INT startPos = lex->getTokenStartPos();
	bool varExpression = (token == TID_VAR);
	bool callExpression = false;
	bool stateSaved = false;

	LEXERSTATE st;
	lex->initState(st);

	while( token && !end_expr ) {
		switch( token ) {
		case TID_ERROR:
			setError(ERROR_TOKEN, lex->getSourceURL(), lex->getCurrentLine());
			if( !stateSaved ) lex->freeState(st);
			return NULL;
		case TID_VAR:
			if( startPos != lex->getTokenStartPos() ) {
				setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
				if( !stateSaved ) lex->freeState(st);
				return NULL;
			}
			break;
		case TID_EOS:
			end_expr = true;
			break;
		case TID_STC:
			cnest++;
			break;
		case TID_EDC:
			if( --cnest < 0 ) {
				end_expr = true;
			}
			break;
		case TID_INDEX:
			tnest++;
			break;
		case TID_EDA:
			if( --tnest < 0 ) {
				end_expr = true;
			}
			break;
		case TID_COM:
			if( !callExpression && !varExpression ) {
				end_expr = true;
			}
			break;
		case TID_QU:
			qnest++;
			break;
		case TID_COL:
			if( --qnest < 0 ) {
				end_expr = true;
			}
			break;
		}
		if( !end_expr && (pc = createExpression(prev_token, token)) == NULL ) {
			end_expr = true;
		}
		if( end_expr ) {
			if( stateSaved ) lex->restoreState(st);
			stateSaved = false;
			break;
		}
		if( pc->dExpression.token == TID_CALL ) {
			callExpression = true;
		}
		if( !stackToken(prev_token, pc, cs, ds) ) {
			if( stateSaved ) lex->freeState(st);
			return NULL;
		}
		if( stateSaved ) lex->freeState(st);
		lex->saveState(st);
		token = lex->getToken();
		stateSaved = true;
	}
	if( stateSaved ) lex->freeState(st);

	if( !end_expr ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	if( startPos == lex->getTokenStartPos() && cs.empty() && ds.empty() ) {
		pc = createParseInfo(ST_EXPRESSION);
		pc->size	= 0;
		pc->dExpression.exprType	= ET_NOP;
		pc->dExpression.token		= token;
		return pc;
	}
	while( cs.size() ) {
		pc	= cs.pop();
		if( !reduce(pc, cs, ds )) return NULL;
	}
	if( ds.size() != 1 ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	pc	= ds.pop();
	pc->size	= lex->getCurrentPos() -  startPos;
	return pc;
}

PARSEINF * Parser::stageLiteralExpression(TokenId token)
{
	PARSEINF * expr = stageExpression(token);
	if( expr == NULL ) return NULL;

	if( !isLiteralExpression(expr) ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(expr);
		return NULL;
	}
	return expr;
}

PARSEINF * Parser::stageEvaluateExpression(TokenId token)
{
	PARSEINF * expr = stageExpression(token);
	if( expr == NULL ) return NULL;

	if( expr->type != ST_EXPRESSION ||
		expr->dExpression.exprType == ET_NOP ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(expr);
		return NULL;
	}
	return expr;
}

PARSEINF * Parser::stagePathExpression(TokenId token)
{
	PARSEINF * pc = NULL;
	bool	end_expr = false;
	ParseStack	cs;
	ParseStack	ds;
	ExpressionType	prev_token = ET_NOP;
	LEXERSTATE		st;
	INT	startPos = lex->getTokenStartPos();
	INT cnest=0;
	INT tnest=0;
	bool stateSaved = false;

	lex->initState(st);

	while( token && !end_expr ) {
		switch( token ) {
		case TID_ERROR:
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			if( stateSaved ) lex->freeState(st);
			return NULL;
		case TID_VAR:
		case TID_QU:
		case TID_NEW:
		case TID_GT:
		case TID_LT:
		case TID_GE:
		case TID_LE:
		case TID_INSTANCEOF:
		case TID_TYPEOF:
		case TID_SIZEOF:
		case TID_EQEQ:
		case TID_NE:
		case TID_AND:
		case TID_XOR:
		case TID_OR:
		case TID_ANDAND:
		case TID_OROR:
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			if( stateSaved ) lex->freeState(st);
			return NULL;
		case TID_EOS:
		case TID_COM:
			end_expr = true;
			break;
		case TID_STC:
			cnest++;
			break;
		case TID_EDC:
			if( --cnest < 0 ) {
				end_expr = true;
			}
			break;
		case TID_INDEX:
			tnest++;
			break;
		case TID_EDA:
			if( --tnest < 0 ) {
				end_expr = true;
			}
			break;
		case TID_COL:
			token = TID_SPC;
			break;
		}
		if( !end_expr && (pc = createExpression(prev_token, token)) == NULL ) {
			end_expr = true;
			if( pc ) releaseParseInfo(pc);
		}
		if( !end_expr ) {
			if( isDataExpression(pc) && isDataExpression(prev_token) ) {
				end_expr = true;
//#pragma message (_CRS_WARNING("dimension [] error"))
	//			} else if( IsOperatorExpression(pc) && IsOperatorExpression(prev_token) ) {
	//				end_expr = true;
			}
		}
		if( end_expr ) {
			if( stateSaved ) lex->restoreState(st);
			stateSaved = false;
			break;
		}
		if( pc->dExpression.accessType != 0 &&
			pc->dExpression.accessType != ACT_READ ) {
			setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
			if( stateSaved ) lex->freeState(st);
			if( pc ) releaseParseInfo(pc);
			return NULL;
		}
		if( !stackToken(prev_token, pc, cs, ds) ) {
			if( stateSaved ) lex->freeState(st);
			if( pc ) releaseParseInfo(pc);
			return NULL;
		}
		if( stateSaved ) lex->freeState(st);
		lex->saveState(st);
		stateSaved	= true;
		token = lex->getToken();
	}
	if( stateSaved ) lex->freeState(st);

	if( !end_expr ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	while( cs.size() ) {
		pc	= cs.pop();
		if( !reduce(pc, cs, ds )) return NULL;
	}
	if( ds.size() != 1 ) {
		setError(ERROR_EXPRESSION, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	pc	= ds.pop();
	pc->size	= lex->getCurrentPos() -  startPos;
	return pc;
}

PARSEINF * Parser::stageImport(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_IMPORT);

	String url_string;
	token = lex->getToken();
	if( token == TID_SYMBOL ) {
		Variant v;
		lex->getTokenValue(*v);
		url_string = String(&v->dString);
	} else if( token == TID_STRING ) {
		Variant v;
		lex->getTokenValue(*v);
		url_string = String(&v->dString);
	} else {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}

	copyString(&pc->dImport.sourceURL, url_string);
	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageUse(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_USE);

	token = lex->getToken();
	PARSEINF *addr = stagePathExpression(token);
	if( addr == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dUse.addr	= addr;
	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageBlockStatement(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_BLOCK);
	PARSEINF * prev = NULL;
	token = lex->getToken();
	while( token ) {
		if( token == TID_ENDBLOCK ) {
			pc->size	= lex->getCurrentPos() - pc->position;
			return pc;
		}
		PARSEINF * spc = stageStatement(token);
		if( spc == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		if( prev ) {
			prev->next	= spc;
			spc->prev	= prev;
		} else {
			pc->dBlock.innerCode = spc;
		}
		spc->parent	= pc;
		prev		= spc;
		token = lex->getToken();
	}
	releaseParseInfo(pc);
	setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
	return NULL;
}

PARSEINF * Parser::stageReturn(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_RETURN);
	LEXERSTATE st;
	lex->initState(st);
	lex->saveState(st);
	token = lex->getToken();
	if( token != TID_EOS ) {
		lex->freeState(st);
		PARSEINF * expr = stageExpression(token);
		if( expr == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dReturn.resultValue	= expr;
		expr->parent= pc;
	} else {
		lex->restoreState(st);
	}
	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageIf(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_IF);

	if( lex->getToken() != TID_STC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	token = lex->getToken();
	PARSEINF * expr = stageEvaluateExpression(token);
	if( expr == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dIf.expression	= expr;
	expr->parent = pc;
	if( lex->getToken() != TID_EDC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}

	LEXERSTATE st;
	lex->initState(st);
	bool stateSave = false;
	token = lex->getToken();
	if( token != TID_ELSE ) {
		PARSEINF * trueCode = stageStatement(token);
		if( trueCode == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dIf.trueCode	= trueCode;
		trueCode->parent	= pc;
		lex->saveState(st);
		stateSave	= true;
		token = lex->getToken();
	}

	if( token == TID_ELSE ) {
		if( stateSave ) {
			lex->freeState(st);
			stateSave	= false;
		}
		token = lex->getToken();
		PARSEINF * falseCode = stageStatement(token);
		if( falseCode == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dIf.falseCode	= falseCode;
		falseCode->parent	= pc;
	} else {
		if( stateSave ) {
			lex->restoreState(st);
		}
	}
	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;

}

PARSEINF * Parser::stageFor(TokenId token)
{
	INT pos = lex->getTokenStartPos();

	PARSEINF * pc = NULL;
	if( lex->getToken() != TID_STC ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	token = lex->getToken();
	PARSEINF * expr = stageExpression(token);
	if( expr == NULL ) {
		return NULL;
	}
	if( expr->dExpression.exprType != ET_NOP )
		token = lex->getToken();
	if( token == TID_IN ) {
		pc = createParseInfo(ST_FORIN);
		pc->dForin.variableExpression	= expr;
		expr->parent	= pc;
		token = lex->getToken();
		if( (expr = stageExpression(token)) == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dForin.targetExpression	= expr;
		expr->parent	= pc;
		token = lex->getToken();

	} else {
		pc = createParseInfo(ST_FOR);
		pc->dFor.initExpression	= expr;
		expr->parent	= pc;
		if( token != TID_EOS ) {
			releaseParseInfo(pc);
			setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
			return NULL;
		}
		token = lex->getToken();
		if( (expr = stageExpression(token)) == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dFor.condExpression	= expr;
		expr->parent	= pc;

		if( expr->dExpression.exprType != ET_NOP )
			token = lex->getToken();

		if( token != TID_EOS ) {
			releaseParseInfo(pc);
			setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
			return NULL;
		}
		token = lex->getToken();
		if( (expr = stageExpression(token)) == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dFor.nextExpression	= expr;
		expr->parent	= pc;
		if( expr->dExpression.exprType != ET_NOP )
			token = lex->getToken();
	}
	if( token != TID_EDC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	token = lex->getToken();
	PARSEINF * code;
	if( (code = stageStatement(token)) == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	if( pc->type == ST_FOR )
		pc->dFor.innerCode		= code;
	else
		pc->dForin.innerCode	= code;
	code->parent	= pc;
	pc->position	= pos;
	pc->size		= lex->getCurrentPos() - pos;
	return pc;
}

PARSEINF * Parser::stageWhile(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_WHILE);

	if( lex->getToken() != TID_STC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	PARSEINF * expr;
	token = lex->getToken();
	if( (expr = stageEvaluateExpression(token)) == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dWhile.expression	= expr;
	expr->parent	= pc;
	if( lex->getToken() != TID_EDC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	PARSEINF * code;
	token = lex->getToken();
	if( (code = stageStatement(token)) == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dWhile.innerCode	= code;
	code->parent	= pc;
	pc->size		= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageDo(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_DO);

	token = lex->getToken();
	PARSEINF * code = stageStatement(token);
	if( code == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dDo.innerCode	= code;
	code->parent	= pc;

	if( lex->getToken() != TID_WHILE ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	if( lex->getToken() != TID_STC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	token = lex->getToken();
	PARSEINF * expr = stageEvaluateExpression(token);
	if( expr == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dDo.expression	= expr;
	expr->parent	= pc;
	if( lex->getToken() != TID_EDC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	pc->size		= lex->getCurrentPos() - pc->position;
	return pc;

}

PARSEINF * Parser::stageSwitch(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_SWITCH);
	if( lex->getToken() != TID_STC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	token = lex->getToken();
	PARSEINF * expr = stageEvaluateExpression(token);
	if( expr == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dSwitch.expression	= expr;
	expr->parent	= pc;
	if( lex->getToken() != TID_EDC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	if( lex->getToken() != TID_BEGINBLOCK ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	token = lex->getToken();
	PARSEINF * case_default_prev = NULL;
	while( token == TID_CASE || token == TID_DEFAULT ) {
		PARSEINF * case_default;
		if( token == TID_CASE ) {
			case_default = createParseInfo(ST_CASE);
			if( case_default_prev ) {
				case_default_prev->next	= case_default;
				case_default->prev		= case_default_prev;
			} else {
				pc->dSwitch.caseList	= case_default;
			}

			token = lex->getToken();
			PARSEINF * expr = stageEvaluateExpression(token);
			if( expr == NULL ) {
				releaseParseInfo(pc);
				return NULL;
			}
			case_default->dCase.expression	= expr;
			expr->parent	= case_default;

		} else {
			case_default = createParseInfo(ST_DEFAULT);
			if( case_default_prev ) {
				case_default_prev->next		= case_default;
				case_default->prev			= case_default_prev;
			} else {
				pc->dSwitch.caseList	= case_default;
			}
		}
		case_default->parent	= pc;
		case_default_prev		= case_default;

		if( lex->getToken() != TID_COL ) {
			releaseParseInfo(pc);
			setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
			return NULL;
		}
		token = lex->getToken();
		PARSEINF * code_prev = NULL;
		while( token != TID_ENDBLOCK &&
			   token != TID_CASE &&
			   token != TID_DEFAULT ) {
			PARSEINF  * code = stageStatement(token);
			if( code == NULL ) {
				releaseParseInfo(pc);
				return NULL;
			}
			if( code_prev ) {
				code_prev->next	= code;
				code->prev		= code_prev;
			} else {
				if( case_default->type == ST_CASE )
					case_default->dCase.caseCode	= code;
				else
					case_default->dDefault.caseCode	= code;
			}
			code->parent	= case_default;
			code_prev		= code;
			token = lex->getToken();
		}
	}
	if( token  != TID_ENDBLOCK ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageBreak(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_BREAK);
	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageContinue(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_CONTNUE);
	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageTry(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_TRY);
	token = lex->getToken();
	if( token != TID_BEGINBLOCK ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	PARSEINF * code = stageBlockStatement(token);
	if( code == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dTry.innerCode	= code;
	code->parent	= pc;

	if( lex->getToken() != TID_CATCH ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	if( lex->getToken() != TID_STC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	if( lex->getToken() == TID_SYMBOL ) {
		PARSEINF * sym = createSymbolExpression(TID_SYMBOL);
		if( sym == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dTry.catchParam	= sym;
		sym->parent		= pc;
	}
	if( lex->getToken() != TID_EDC ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	if( lex->getToken() != TID_BEGINBLOCK ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}
	PARSEINF * block = stageBlockStatement(token);
	if( block == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dTry.catchCode	= block;
	block->parent	= pc;

	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageCatch(TokenId token)
{
	setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
	return NULL;
}

PARSEINF * Parser::stageThrow(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_THROW);
	LEXERSTATE st;
	lex->initState(st);
	lex->saveState(st);
	token = lex->getToken();
	if( token != TID_EOS ) {
		lex->freeState(st);
		PARSEINF * expr = stageExpression(token);
		if( expr == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dThrow.exceptionCode = expr;
		expr->parent	= pc;
	} else {
		lex->restoreState(st);
	}
	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageFunction(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_FUNCTION);
	token = lex->getToken();
	if( token != TID_SYMBOL ) {
		releaseParseInfo(pc);
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}

	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dFunction.name, &v->dString);
	if( isEqualChars(getStringReadPtr(&pc->dFunction.name), ES_T("constructor"))) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	if( isEqualChars(getStringReadPtr(&pc->dFunction.name), ES_T("finalizer"))) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	if( lex->getToken() != TID_STC ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	PARSEINF * prev = NULL;
	token = lex->getToken();
	if( token != TID_EDC ) {
		while( token ) {
			if( token != TID_SYMBOL ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}

			PARSEINF * arg = createParaExpression(token);
			if( prev ) {
				PARSEINF * p0;
				p0 = pc->dFunction.params;
				while(p0->next) {
					if( isEqualString(&p0->dExpression.expPara.name, &arg->dExpression.expPara.name) ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
						releaseParseInfo(pc);
						return NULL;
					}
					p0 = p0->next;
				}
				prev->next	= arg;
				arg->prev	= prev;
			} else {
				pc->dFunction.params = arg;
			}
			arg->parent	= pc;
			prev		= arg;

			token = lex->getToken();
			if( token == TID_INDEX ) {
				token = lex->getToken();
				if( token == TID_INTEGER ) {
					Variant v;
					lex->getTokenValue(*v);
					arg->dExpression.expPara.array	= v->dInt;
					token = lex->getToken();
				} else {
					arg->dExpression.expPara.array	= 0;
				}
				if( token != TID_EDA ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
					releaseParseInfo(pc);
					return NULL;
				}
				token = lex->getToken();
			}

			if( token == TID_COM ) {
				token = lex->getToken();
			} else if( token == TID_EDC ) {
				break;
			} else {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
		}
	}
	if( token != TID_EDC ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	token = lex->getToken();

	if( token != TID_BEGINBLOCK ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	PARSEINF * block = stageBlockStatement(token);
	if( block == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dFunction.innerCode	= block;
	block->parent	= pc;
	pc->size		= lex->getCurrentPos() - pc->position;

	IParseInfoWalker *piw = createParseInfoWalker(block);
	if( piw ) {
		PARSEINF * base=NULL;
		PARSEINF * p = piw->next();
		while( p != NULL ) {
			if( p->type == ST_CLASS ||
				p->type == ST_PACKAGE ||
				p->type == ST_TEMPLATE ||
				p->type == ST_INTERFACE ||
				p->type == ST_FUNCTION ) {
				p = piw->nextStatement();
				continue;
			}
			if( p->type == ST_RETURN ) {
				if( pc->dFunction.hasResult ) {
					if( !p->dReturn.resultValue ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), p->line);
						releaseParseInfo(pc);
						return NULL;
					}
				}
				if( base ) {
					if( base->dReturn.resultValue && !p->dReturn.resultValue ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), p->line);
						releaseParseInfo(pc);
						return NULL;
					}
					if( !base->dReturn.resultValue &&  p->dReturn.resultValue ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), p->line);
						releaseParseInfo(pc);
						return NULL;
					}
				} else {
					base	= p;
				}
			}
			p = piw->next();
		}
		if( !pc->dFunction.hasResult && base && base->dReturn.resultValue ) {
			pc->dFunction.hasResult = 1;
		}
		if( pc->dFunction.hasResult ) {
			PARSEINF * last=NULL;
			p = block->dBlock.innerCode;
			while(p) {
				last = p;
				p = p->next;
			}
			if( !hasReturnCode(last) ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), last->line);
				releaseParseInfo(pc);
				return NULL;
			}
		}
	}
	return pc;
}

PARSEINF * Parser::stagePackage(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_PACKAGE);

	token = lex->getToken();
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dPackage.name, &v->dString);

	token = lex->getToken();
	if( token != TID_BEGINBLOCK ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	PARSEINF * block = stagePackageBlock(token);
	if( block == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dPackage.innerCode	= block;
	block->parent	= pc;
	pc->size		= lex->getCurrentPos() - pc->position;

	PARSEINF *mp = block->dBlock.innerCode;
	while( mp ) {
		if( mp->type == ST_METHOD ) {
			if( mp->dMethod.acm & ACM_CONSTRUCTOR ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
				releaseParseInfo(pc);
				return NULL;
			}
			if( isEqualString(&mp->dMethod.name, &pc->dPackage.name) ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
				releaseParseInfo(pc);
				return NULL;
			}
			if( mp->dMethod.acm & ACM_FINALIZER ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
				releaseParseInfo(pc);
				return NULL;
			}
		}
		mp	= mp->next;
	}

	if( !uniqMemberCheck(block->dBlock.innerCode) ) {
		releaseParseInfo(pc);
		return NULL;
	}
	return pc;

}

PARSEINF * Parser::stagePackageBlock(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_BLOCK);
	PARSEINF * prev = NULL;
	token = lex->getToken();
	do {
		if( token == TID_EOS ) {
			token = lex->getToken();
			continue;
		}
		if( token == TID_ENDBLOCK ) {
			break;
		}

		PARSEINF *def = NULL;
		const STRING *defName = NULL;
		AccessModifier acm = ACM_EMPTY;
		if( (token & TT_MASK) == TT_MODIFIER ) {
			if( !parseACM(token, acm)) {
				return NULL;
			}
			token = lex->getToken();
		}
		switch(token) {
		case TID_CLASS:
			if( (acm & (ACM_CONST|ACM_STATIC|ACM_REPRESENT)) != 0 ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			def = stageClass(token, acm);
			if( def ) defName = &def->dClass.name;
			break;
		case TID_INTERFACE:
			if( (acm & (ACM_CONST|ACM_STATIC|ACM_REPRESENT)) != 0 ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			def = stageInterface(token, acm);
			if( def ) defName = &def->dInterface.name;
			break;
		case TID_TEMPLATE:
			if( (acm & (ACM_CONST|ACM_STATIC|ACM_REPRESENT)) != 0 ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			def = stageTemplate(token, acm);
			if( def ) defName = &def->dTemplate.name;
			break;
		case TID_FUNCTION:
			if( (acm & ACM_REPRESENT) != 0 ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			def = stageMethod(token, (AccessModifier)(acm|ACM_STATIC));
			if( def ) defName = &def->dMethod.name;
			break;
		case TID_SYMBOL:
			if( (acm & ACM_REPRESENT) != 0 ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			def = stageProperty(token, (AccessModifier)(acm|ACM_STATIC));
			if( def ) defName = &def->dProperty.name;
			break;
		case TID_IMPORT:
			if( acm != ACM_EMPTY ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			def = stageImport(token);
			break;
		default:
			setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
			releaseParseInfo(pc);
			return NULL;
		}

		if( def == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		if( prev ) {
			if( defName ) {
				PARSEINF *p = pc->dBlock.innerCode;
				while( p ) {
					const STRING *name = NULL;
					switch( p->type ) {
					case ST_CLASS:		defName = &p->dClass.name;		break;
					case ST_INTERFACE:	defName = &p->dInterface.name;	break;
					case ST_TEMPLATE:	defName = &p->dTemplate.name;	break;
					case ST_METHOD:		defName = &p->dMethod.name;		break;
					case ST_PROPERTY:	defName = &p->dProperty.name;	break;
					}
					if( name && isEqualString(name, defName) ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
						releaseParseInfo(pc);
					}
					p = p->next;
				}
			}
			prev->next	= def;
			def->prev	= prev;
		} else {
			pc->dBlock.innerCode	= def;
		}
		def->parent	= pc;
		prev		= def;
	} while( (token = lex->getToken()) != TID_ENDBLOCK );
	return pc;
}

PARSEINF * Parser::stageClass(TokenId token, AccessModifier acm)
{
	PARSEINF * pc = createParseInfo(ST_CLASS);
	token = lex->getToken();

	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dClass.name, &v->dString);

	token	= lex->getToken();
	if( token == TID_EXTENDS ) {
		token = lex->getToken();
		if( token != TID_SYMBOL ) {
			setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
			releaseParseInfo(pc);
			return NULL;
		}
		PARSEINF *base = stagePathExpression(token);
		if( base == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dClass.basePath	= base;
		base->parent	= pc;
		token = lex->getToken();
	}

	if( token == TID_IMPLEMENTS ) {
		PARSEINF * path_prev = NULL;
		do {
			token = lex->getToken();
			if( token != TID_SYMBOL ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			PARSEINF * path = stagePathExpression(token);
			if( path == NULL ) {
				releaseParseInfo(pc);
				return NULL;
			}
			if( path_prev ) {
				path_prev->next	= path;
				path->prev		= path_prev;
			} else {
				pc->dClass.interfaces	= path;
			}
			path->parent	= pc;
			path_prev		= path;
		} while( (token = lex->getToken()) == TID_COM );
	}

	if( token != TID_BEGINBLOCK ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	PARSEINF * block = stageClassBlock(token);
	if( block == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dClass.innerCode	= block;
	block->parent		= pc;
	pc->size	= lex->getCurrentPos() - pc->position;

	PARSEINF *mp = block->dBlock.innerCode;
	PARSEINF *cf = NULL;
	PARSEINF *ff = NULL;
	while( mp ) {
		if( mp->type == ST_METHOD ) {
			if( mp->dMethod.acm & ACM_CONSTRUCTOR ) {
				if( cf ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
					releaseParseInfo(pc);
					return NULL;
				}
				cf	= mp;
			}
			if( isEqualString(&mp->dMethod.name, &pc->dClass.name) ) {
				if( cf ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
					releaseParseInfo(pc);
					return NULL;
				}
				cf	= mp;
				mp->dMethod.acm = (AccessModifier)((INT)mp->dMethod.acm | ACM_CONSTRUCTOR);
			}
			if( mp->dMethod.acm & ACM_FINALIZER ) {
				if( ff ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
					releaseParseInfo(pc);
					return NULL;
				}
				ff	= mp;
				mp->dMethod.acm = (AccessModifier)(mp->dMethod.acm|ACM_FINALIZER);
			}
		}
		mp	= mp->next;
	}

	if( cf ) {
		IParseInfoWalker *piw = createParseInfoWalker(cf);
		if( piw ) {
			PARSEINF * base=NULL;
			PARSEINF * p;
			while( (p = piw->next()) != NULL ) {
				if( p->type == ST_EXPRESSION &&
					p->dExpression.exprType == ET_OPRS &&
					p->dExpression.token == TID_CALL &&
					p->dExpression.expCall.addr &&
					p->dExpression.expCall.addr->dExpression.token == TID_SUPER ) {
					break;
				}
			}
			if( !p ) {
				pc->dClass.acm = (AccessModifier)(pc->dClass.acm|ACM_AUTOCALL);
			}
		}
	}

	if( !uniqMemberCheck(block->dBlock.innerCode) ) {
		releaseParseInfo(pc);
		return NULL;
	}
	return pc;
}

PARSEINF * Parser::stageClassBlock(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_BLOCK);
	PARSEINF * prev = NULL;
	token = lex->getToken();
	do {
		if( token == TID_EOS ) {
			token = lex->getToken();
			continue;
		}
		if( token == TID_ENDBLOCK ) {
			break;
		}

		PARSEINF *def = NULL;
		const STRING *defName = NULL;
		AccessModifier acm = ACM_EMPTY;
		if( (token & TT_MASK) == TT_MODIFIER ) {
			if( !parseACM(token, acm)) {
				return NULL;
			}
			token = lex->getToken();
		}
		switch(token) {
		case TID_FUNCTION:
			if( (acm & ACM_REPRESENT) != 0 ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			def = stageMethod(token, acm);
			if( def ) defName = &def->dMethod.name;
			break;
		case TID_SYMBOL:
			def = stageProperty(token, acm);
			if( def ) defName = &def->dProperty.name;
			break;
		default:
			setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
			releaseParseInfo(pc);
			return NULL;
		}

		if( def == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		if( prev ) {
			if( defName ) {
				PARSEINF *p = pc->dBlock.innerCode;
				while( p ) {
					const STRING *name = NULL;
					if( p->type == ST_PROPERTY ) {
						name = &p->dProperty.name;
					} else if( p->type == ST_METHOD ) {
						name = &p->dMethod.name;
					}
					if( name && isEqualString(name, defName) ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
						releaseParseInfo(pc);
					}
					p = p->next;
				}
			}
			prev->next	= def;
			def->prev	= prev;
		} else {
			pc->dBlock.innerCode	= def;
		}
		def->parent	= pc;
		prev		= def;
	} while( (token = lex->getToken()) != TID_ENDBLOCK );
	return pc;
}

PARSEINF * Parser::stageInterface(TokenId token, AccessModifier acm)
{
	PARSEINF * pc = createParseInfo(ST_INTERFACE);
	token = lex->getToken();
	pc->dInterface.acm	= acm;

	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dInterface.name, &v->dString);
#ifdef _DEBUG
	if( lex->getCurrentLine() == 1379 ) {
		int a=0;
	}
#endif
	token	= lex->getToken();
	if( token == TID_EXTENDS ) {
		PARSEINF * base_prev = NULL;
		do {
			token = lex->getToken();
			if( token != TID_SYMBOL ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			PARSEINF * base = stagePathExpression(token);
			if( base == NULL ) {
				releaseParseInfo(pc);
				return NULL;
			}
			if( base_prev ) {
				base_prev->next	= base;
				base->prev		= base_prev;
			} else {
				pc->dInterface.interfaces = base;
			}
			base->parent	= pc;
			base_prev		= base;
		} while( (token = lex->getToken()) == TID_COM );
	}

	if( token != TID_BEGINBLOCK ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	PARSEINF * block = stageInterfaceBlock(token);
	if( block == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dInterface.innerCode	= block;
	pc->size	= lex->getCurrentPos() - pc->position;
	block->parent			= pc;

	PARSEINF * mp = block->dBlock.innerCode;
	while( mp ) {
		if( mp->type == ST_METHOD ) {
			if( mp->dMethod.acm & ACM_CONSTRUCTOR ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
				releaseParseInfo(pc);
				return NULL;
			}
			if( isEqualString(&mp->dMethod.name, &pc->dTemplate.name) ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
				releaseParseInfo(pc);
				return NULL;
			}
			if( mp->dMethod.acm & ACM_FINALIZER ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
				releaseParseInfo(pc);
				return NULL;
			}
		}
		mp	= mp->next;
	}

	if( !uniqMemberCheck(block->dBlock.innerCode) ) {
		releaseParseInfo(pc);
		return NULL;
	}
	return pc;
}

PARSEINF * Parser::stageInterfaceBlock(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_BLOCK);
	PARSEINF * prev = NULL;
	token = lex->getToken();
	do {
		if( token == TID_EOS ) {
			token = lex->getToken();
			continue;
		}
		if( token == TID_ENDBLOCK ) {
			break;
		}

		PARSEINF *def = NULL;
		const STRING *defName = NULL;
		AccessModifier acm = ACM_EMPTY;
		if( (token & TT_MASK) == TT_MODIFIER ) {
			if( !parseACM(token, acm)) {
				return NULL;
			}
			token = lex->getToken();
		}
		switch(token) {
		case TID_FUNCTION:
			if( (acm & (ACM_PRIVATE|ACM_PROTECTED|ACM_STATIC|ACM_REPRESENT)) != 0 ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			def = stageIMethod(token, acm);
			if( def ) defName = &def->dMethod.name;
			break;
		case TID_SYMBOL:
			if( (acm & (ACM_PRIVATE|ACM_PROTECTED)) != 0 ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			def = stageIProperty(token, acm);
			if( def ) defName = &def->dProperty.name;
			break;
		default:
			setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
			releaseParseInfo(pc);
			return NULL;
		}

		if( def == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		if( prev ) {
			if( defName ) {
				PARSEINF *p = pc->dBlock.innerCode;
				while( p ) {
					const STRING *name = NULL;
					if( p->type == ST_PROPERTY ) {
						name = &p->dProperty.name;
					} else if( p->type == ST_METHOD ) {
						name = &p->dMethod.name;
					}
					if( name && isEqualString(name, defName) ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
						releaseParseInfo(pc);
					}
					p = p->next;
				}
			}
			prev->next	= def;
			def->prev	= prev;
		} else {
			pc->dBlock.innerCode	= def;
		}
		def->parent	= pc;
		prev		= def;
	} while( (token = lex->getToken()) != TID_ENDBLOCK );
	return pc;
}

PARSEINF * Parser::stageTemplate(TokenId token, AccessModifier acm)
{
	PARSEINF * pc = createParseInfo(ST_TEMPLATE);
	token = lex->getToken();

	pc->dTemplate.acm	= acm;

	if( token != TID_CLASS ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	token	= lex->getToken();
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dTemplate.name, &v->dString);

	token	= lex->getToken();
	if( token != TID_SPC ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	token	= lex->getToken();
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	PARSEINF *type = createTypeExpression(token);

	Variant t;
	lex->getTokenValue(*t);
	copyString(&type->dExpression.expType.typeName, &t->dString);

	pc->dTemplate.typePath = type;
	type->parent = pc;

	token	= lex->getToken();
	if( token == TID_EQ ) {
		token	= lex->getToken();
		if( token != TID_SYMBOL ) {
			setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
			releaseParseInfo(pc);
			return NULL;
		}
		PARSEINF *path = stagePathExpression(token);
		if(  path == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		type->dExpression.expType.defaultTypePath = path;
		path->parent = type;
		token	= lex->getToken();
	}

	if( token != TID_EXTENDS ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	token = lex->getToken();
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	PARSEINF *base = stagePathExpression(token);
	if( base == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dTemplate.basePath = base;
	base->parent	= pc;

	token = lex->getToken();
	if( token == TID_IMPLEMENTS ) {
		PARSEINF * path_prev = NULL;
		do {
			token = lex->getToken();
			if( token != TID_SYMBOL ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			PARSEINF *path = stagePathExpression(token);
			if( path == NULL ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
			if( path_prev ) {
				path_prev->next	= path;
				path->prev		= path_prev;
			} else {
				pc->dTemplate.interfaces	= path;
			}
			path->parent	= pc;
			path_prev		= path;
		} while( (token = lex->getToken()) == TID_COM );
	}

	if( token != TID_BEGINBLOCK ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	PARSEINF *block = stageClassBlock(token);
	if( block == NULL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dTemplate.innerCode	= block;
	block->parent	= pc;
	pc->size	= lex->getCurrentPos() - pc->position;

	PARSEINF * mp = block->dBlock.innerCode;
	PARSEINF * cf = NULL;
	PARSEINF * ff = NULL;
	while( mp ) {
		if( mp->type == ST_METHOD ) {
			if( mp->dMethod.acm & ACM_CONSTRUCTOR ) {
				if( cf ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
					releaseParseInfo(pc);
					return NULL;
				}
				cf	= mp;
			}
			if( mp->dMethod.acm & ACM_FINALIZER ) {
				if( ff ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), mp->line);
					releaseParseInfo(pc);
					return NULL;
				}
				ff	= mp;
			}
		}
		mp	= mp->next;
	}

	if( cf ) {
		IParseInfoWalker *piw = createParseInfoWalker(cf);
		if( piw ) {
			PARSEINF * base=NULL;
			PARSEINF * p;
			while( (p = piw->next()) != NULL ) {
				if( p->type == ST_EXPRESSION &&
					p->dExpression.exprType == ET_OPRS &&
					p->dExpression.token == TID_CALL &&
					p->dExpression.expCall.addr &&
					p->dExpression.expCall.addr->dExpression.token == TID_SUPER ) {
					break;
				}
			}
			if( !p ) {
				pc->dTemplate.acm = (AccessModifier)(pc->dTemplate.acm|ACM_AUTOCALL);
			}
		}
	}

	if( !uniqMemberCheck(block->dBlock.innerCode) ) {
		releaseParseInfo(pc);
		return NULL;
	}
	return pc;
}

PARSEINF * Parser::stageExprStatement(TokenId token)
{
	INT nextStage = 0;
	if( token == TID_SYMBOL ) {
		LEXERSTATE st;
		lex->initState(st);
		lex->saveState(st);
		TokenId nextToken = lex->getToken();
		while(nextToken == TID_DOT || nextToken == TID_SPC ) {
			nextToken = lex->getToken();
			if( nextToken != TID_SYMBOL ) break;
			nextToken = lex->getToken();
		}
		if( nextToken == TID_SYMBOL ) {
			nextStage	= 1;
		}
		lex->restoreState(st);
	}

	PARSEINF * pc;
	if( nextStage ) {
		pc = stageObject(token);
	} else {
#ifdef _DEBUG
		if( lex->getCurrentLine() == 1467 ) {
			int a=0;
		}
#endif
		pc = stageExpression(token);
		if( !pc ) return NULL;
		token = lex->getToken();
		if( token != TID_EOS ) {
			setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
			releaseParseInfo(pc);
			return NULL;
		}
	}
	return pc;
}

PARSEINF * Parser::stageObject(TokenId token)
{
	PARSEINF * pc = createParseInfo(ST_OBJECT);
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	PARSEINF * path = stagePathExpression(token);
	if( path == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dObject.path	= path;
	path->parent		= pc;

	token = lex->getToken();
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dObject.name, &v->dString);

	token	= lex->getToken();
	if( token == TID_INDEX ) {
		token = lex->getToken();
		PARSEINF *expr = stageExpression(token);
		if( expr == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dObject.array	= expr;
		expr->parent	= pc;

		if( expr->dExpression.exprType != ET_NOP )
			token	= lex->getToken();
		if( token != TID_EDA ) {
			setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
			releaseParseInfo(pc);
			return NULL;
		}
		token	= lex->getToken();
	}

	if( token == TID_BEGINBLOCK ) {
		PARSEINF * block = stageBlockStatement(token);
		if( block == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dObject.innerCode	= block;
		block->parent	= pc;
	} else {
		if( (token & TT_MASK) == TT_OPERATOR &&
			token != TID_EOS ) {
			PARSEINF * op = createExpression(ET_DATA, token);
			if( op == NULL ) {
				releaseParseInfo(pc);
				return NULL;
			}
			pc->dObject.initExpression	= op;
			op->parent	= pc;
			if( op->type != ST_EXPRESSION ||
				op->dExpression.exprType != ET_OPR2 ) {
				setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}

			token = lex->getToken();
			PARSEINF * code = stageEvaluateExpression(token);
			if( code == NULL ) {
				releaseParseInfo(pc);
				return NULL;
			}
			pc->dObject.initExpression->dExpression.exp2.expresion1 = code;
			code->parent	= pc->dObject.initExpression;
			token	= lex->getToken();
		}
		if( token != TID_EOS ) {
			setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
			releaseParseInfo(pc);
			return NULL;
		}
	}
	pc->size	= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageMethod(TokenId token, AccessModifier acm)
{
	PARSEINF * pc = createParseInfo(ST_METHOD);
	token = lex->getToken();
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dMethod.name, &v->dString);
	if( isEqualChars(getStringReadPtr(&pc->dMethod.name), ES_T("constructor")) ) {
		acm = (AccessModifier)(acm|ACM_CONSTRUCTOR);
	}
	if( isEqualChars(getStringReadPtr(&pc->dMethod.name), ES_T("finalizer")) ) {
		acm = (AccessModifier)(acm|ACM_FINALIZER);
	}
	pc->dMethod.acm	= acm;

	if( lex->getToken() != TID_STC ) {
		setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	PARSEINF * prev = NULL;
	token = lex->getToken();
	if( token != TID_EDC ) {
		while( token ) {
			if( token != TID_SYMBOL ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}

			PARSEINF * arg = createParaExpression(token);
			if( prev ) {
				PARSEINF * p0;
				p0 = pc->dMethod.params;
				while(p0->next) {
					if( isEqualString(&p0->dExpression.expPara.name, &arg->dExpression.expPara.name) ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
						releaseParseInfo(pc);
						return NULL;
					}
					p0 = p0->next;
				}
				prev->next	= arg;
				arg->prev	= prev;
			} else {
				pc->dMethod.params = arg;
			}
			arg->parent	= pc;
			prev		= arg;

			token = lex->getToken();
			if( token == TID_INDEX ) {
				token = lex->getToken();
				if( token == TID_INTEGER ) {
					Variant v;
					lex->getTokenValue(*v);
					arg->dExpression.expPara.array	= v->dInt;
					token = lex->getToken();
				} else {
					arg->dExpression.expPara.array	= 0;
				}
				if( token != TID_EDA ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
					releaseParseInfo(pc);
					return NULL;
				}
				token = lex->getToken();
			}

			if( token == TID_COM ) {
				token = lex->getToken();
			} else if( token == TID_EDC ) {
				break;
			} else {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
		}
	}
	if( token != TID_EDC ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	token = lex->getToken();
	if( token != TID_BEGINBLOCK ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	PARSEINF * block = stageBlockStatement(token);
	if( block == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dMethod.innerCode	= block;
	block->parent	= pc;
	pc->size		= lex->getCurrentPos() - pc->position;

	IParseInfoWalker *piw = createParseInfoWalker(block);
	if( piw ) {
		PARSEINF * base=NULL;
		PARSEINF * p = piw->next();
		while( p != NULL ) {
			if( p->type == ST_CLASS ||
				p->type == ST_PACKAGE ||
				p->type == ST_TEMPLATE ||
				p->type == ST_INTERFACE ||
				p->type == ST_FUNCTION ) {
				p = piw->nextStatement();
				continue;
			}
			if( p->type == ST_RETURN ) {
				if( pc->dMethod.hasResult ) {
					if( !p->dReturn.resultValue ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), p->line);
						releaseParseInfo(pc);
						return NULL;
					}
				}
				if( base ) {
					if( base->dReturn.resultValue && !p->dReturn.resultValue ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), p->line);
						releaseParseInfo(pc);
						return NULL;
					}
					if( !base->dReturn.resultValue &&  p->dReturn.resultValue ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), p->line);
						releaseParseInfo(pc);
						return NULL;
					}
				} else {
					base	= p;
				}
			}
			p = piw->next();
		}
		if( !pc->dMethod.hasResult && base && base->dReturn.resultValue ) {
			pc->dMethod.hasResult = 1;
		}
		if( pc->dMethod.hasResult ) {
			PARSEINF * last=NULL;
			p = block->dBlock.innerCode;
			while(p) {
				last = p;
				p = p->next;
			}
			if( !hasReturnCode(last) ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), last->line);
				releaseParseInfo(pc);
				return NULL;
			}
		}
	}
	return pc;
}

PARSEINF * Parser::stageProperty(TokenId token, AccessModifier acm)
{
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}

	PARSEINF * path = stagePathExpression(token);
	if( path == NULL ) {
		return NULL;
	}
	PARSEINF * pc = createParseInfo(ST_PROPERTY);
	pc->dProperty.acm	= acm;
	pc->dProperty.path	= path;
	path->parent	= pc;

	token = lex->getToken();
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dProperty.name, &v->dString);

	token = lex->getToken();
	if( token == TID_INDEX ) {
		PARSEINF * arr_prev = NULL;
		do {
			token = lex->getToken();
			PARSEINF * expr = stageExpression(token);
			if( expr == NULL ) {
				releaseParseInfo(pc);
				return NULL;
			}
			if( arr_prev ) {
				arr_prev->next	= expr;
				expr->prev		= arr_prev;
			} else {
				pc->dProperty.array	=	expr;
			}
			expr->parent = pc;
			arr_prev	= expr;

			if( expr->dExpression.exprType != ET_NOP ) {
				if( !isLiteralExpression(expr) ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
					releaseParseInfo(pc);
					return NULL;
				}
			}
			token	= lex->getToken();
			if( token != TID_EDA ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
		} while( (token = lex->getToken()) == TID_INDEX );
	}

	if( token == TID_EQ ) {
		token	= lex->getToken();
		PARSEINF * code = stageLiteralExpression(token);
		if( code == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dProperty.initCode	= code;
		code->parent		= pc;
		token	= lex->getToken();
	}

	if( token != TID_EOS ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	pc->size		= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageIMethod(TokenId token, AccessModifier acm)
{
	PARSEINF * pc = createParseInfo(ST_METHOD);
	token = lex->getToken();
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dMethod.name, &v->dString);
	if( isEqualChars(getStringReadPtr(&pc->dMethod.name), ES_T("constructor")) ) {
		acm = (AccessModifier)(acm|ACM_CONSTRUCTOR);
	}
	if( isEqualChars(getStringReadPtr(&pc->dMethod.name), ES_T("finalizer")) ) {
		acm = (AccessModifier)(acm|ACM_FINALIZER);
	}
	pc->dMethod.acm	= acm;

	if( lex->getToken() != TID_STC ) {
		setError(ERROR_SYNTAX,lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	PARSEINF * prev = NULL;
	token = lex->getToken();
	if( token != TID_EDC ) {
		while( token ) {
			if( token != TID_SYMBOL ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}

			PARSEINF * arg = createParaExpression(token);
			if( prev ) {
				PARSEINF * p0;
				p0 = pc->dMethod.params;
				while(p0->next) {
					if( isEqualString(&p0->dExpression.expPara.name, &arg->dExpression.expPara.name) ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
						releaseParseInfo(pc);
						return NULL;
					}
					p0 = p0->next;
				}
				prev->next	= arg;
				arg->prev	= prev;
			} else {
				pc->dMethod.params = arg;
			}
			arg->parent	= pc;
			prev		= arg;

			token = lex->getToken();
			if( token == TID_INDEX ) {
				token = lex->getToken();
				if( token == TID_INTEGER ) {
					Variant v;
					lex->getTokenValue(*v);
					arg->dExpression.expPara.array	= v->dInt;
					token = lex->getToken();
				} else {
					arg->dExpression.expPara.array	= 0;
				}
				if( token != TID_EDA ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
					releaseParseInfo(pc);
					return NULL;
				}
				token = lex->getToken();
			}

			if( token == TID_COM ) {
				token = lex->getToken();
			} else if( token == TID_EDC ) {
				break;
			} else {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
		}
	}

	token = lex->getToken();

	if( token == TID_EOS ) {
		return pc;
	}
	if( token != TID_BEGINBLOCK ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	PARSEINF * block = stageBlockStatement(token);
	if( block == NULL ) {
		releaseParseInfo(pc);
		return NULL;
	}
	pc->dMethod.innerCode	= block;
	block->parent	= pc;
	pc->size		= lex->getCurrentPos() - pc->position;

	IParseInfoWalker *piw = createParseInfoWalker(block);
	if( piw ) {
		PARSEINF * base=NULL;
		PARSEINF * p = piw->next();
		while( p != NULL ) {
			if( p->type == ST_CLASS ||
				p->type == ST_PACKAGE ||
				p->type == ST_TEMPLATE ||
				p->type == ST_INTERFACE ||
				p->type == ST_FUNCTION ) {
				p = piw->nextStatement();
				continue;
			}
			if( p->type == ST_RETURN ) {
				if( pc->dMethod.hasResult ) {
					if( !p->dReturn.resultValue ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), p->line);
						releaseParseInfo(pc);
						return NULL;
					}
				}
				if( base ) {
					if( base->dReturn.resultValue && !p->dReturn.resultValue ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), p->line);
						releaseParseInfo(pc);
						return NULL;
					}
					if( !base->dReturn.resultValue &&  p->dReturn.resultValue ) {
						setError(ERROR_SYNTAX, lex->getSourceURL(), p->line);
						releaseParseInfo(pc);
						return NULL;
					}
				} else {
					base	= p;
				}
			}
			p = piw->next();
		}
		if( !pc->dMethod.hasResult && base && base->dReturn.resultValue ) {
			pc->dMethod.hasResult = 1;
		}
		if( pc->dMethod.hasResult ) {
			PARSEINF * last=NULL;
			p = block->dBlock.innerCode;
			while(p) {
				last = p;
				p = p->next;
			}
			if( !hasReturnCode(last) ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), last->line);
				releaseParseInfo(pc);
				return NULL;
			}
		}
	}
	return pc;
}

PARSEINF * Parser::stageIProperty(TokenId token, AccessModifier acm)
{
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		return NULL;
	}

	PARSEINF * path = stagePathExpression(token);
	if( path == NULL ) {
		return NULL;
	}
	PARSEINF * pc = createParseInfo(ST_PROPERTY);
	pc->dProperty.acm	= acm;
	pc->dProperty.path	= path;
	path->parent	= pc;

	token = lex->getToken();
	if( token != TID_SYMBOL ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}

	Variant v;
	lex->getTokenValue(*v);
	copyString(&pc->dProperty.name, &v->dString);

	token = lex->getToken();
	if( token == TID_INDEX ) {
		PARSEINF * arr_prev = NULL;
		do {
			token = lex->getToken();
			PARSEINF * expr = stageExpression(token);
			if( expr == NULL ) {
				releaseParseInfo(pc);
				return NULL;
			}
			if( arr_prev ) {
				arr_prev->next	= expr;
				expr->prev		= arr_prev;
			} else {
				pc->dProperty.array	=	expr;
			}
			expr->parent = pc;
			arr_prev	= expr;

			if( expr->dExpression.exprType != ET_NOP ) {
				if( !isLiteralExpression(expr) ) {
					setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
					releaseParseInfo(pc);
					return NULL;
				}
			}
			token	= lex->getToken();
			if( token != TID_EDA ) {
				setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
				releaseParseInfo(pc);
				return NULL;
			}
		} while( (token = lex->getToken()) == TID_INDEX );
	}

	if( token == TID_EQ ) {
		token	= lex->getToken();
		PARSEINF * code = stageLiteralExpression(token);
		if( code == NULL ) {
			releaseParseInfo(pc);
			return NULL;
		}
		pc->dProperty.initCode	= code;
		code->parent		= pc;
		token	= lex->getToken();
	}

	if( token != TID_EOS ) {
		setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
		releaseParseInfo(pc);
		return NULL;
	}
	pc->size		= lex->getCurrentPos() - pc->position;
	return pc;
}

PARSEINF * Parser::stageReserved(TokenId token)
{
	setError(ERROR_SYNTAX, lex->getSourceURL(), lex->getCurrentLine());
	return NULL;
}

} // es
