/*
 * sstring.cpp
 *
 *  Created on: 2012/01/27
 *      Author: tanaka
 */
#include <stdlib.h>
#include <wchar.h>
#include <float.h>
#include <math.h>
#include "../include/elise.h"
#include "platform.h"
#include "bytes.h"
#include "sstring.h"

namespace es {

void initializeSString()
{
}

void terminateSString()
{
}

bool copySChars(SCHAR *s1, const SCHAR *s2)
{
	while( *s2 ) *s1++ = *s2++;
	*s1 = 0;
	return true;
}

bool copySCharsLen(SCHAR *s1, const SCHAR *s2, SIZE len)
{
	if( !s1 ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( s2 ) {
		while( *s2 && len ) { *s1++ = *s2++; len--; }
	}
	*s1 = 0;
	return true;
}

SIZE getSCharsLength(const SCHAR *s1)
{
	SIZE n=0;
	if( s1 ) {
		while(*s1) {
			n++;
			s1++;
		}
	}
	return n;
}

bool isEqualSChars(const SCHAR *s1, const SCHAR *s2)
{
	if( s1 == s2 ) return true;
	if( !s1 || !s2 ) return false;
	while(*s1 && *s2) {
		if( *s1 != *s2 ) return false;
		s1++; s2++;
	}
	if( *s1 || *s2 ) return false;
	return true;
}

bool isEqualSCharsLen(const SCHAR *s1, const SCHAR *s2, SIZE chars)
{
	if( chars == 0 || s1 == s2 ) return true;
	if( !s1 || !s2 ) return false;
	while(*s1 && *s2 && chars > 0 ) {
		if( *s1 != *s2 ) return false;
		s1++; s2++; chars--;
	}
	if( chars != 0 ) return false;
	return true;
}

bool isEqualSCharsNoCase(const SCHAR *s1, const SCHAR *s2)
{
	if( s1 == s2 ) return true;
	if( !s1 || !s2 ) return false;
	while(*s1 && *s2) {
		SCHAR c1 = *s1;
		SCHAR c2 = *s2;
		if( ES_ST('A') <= c1 && c1 <= ES_ST('Z') ) c1 = c1 - ES_ST('A') + ES_ST('a');
		if( ES_ST('A') <= c2 && c2 <= ES_ST('Z') ) c2 = c2 - ES_ST('A') + ES_ST('a');
		if( c1 != c2 ) return false;
		s1++; s2++;
	}
	if( *s1 || *s2 ) return false;
	return true;
}

bool isEqualSCharsNoCaseLen(const SCHAR *s1, const SCHAR *s2, SIZE chars)
{
	if( chars == 0 || s1 == s2 ) return true;
	if( !s1 || !s2 ) return false;
	while(*s1 && *s2 && chars > 0 ) {
		SCHAR c1 = *s1;
		SCHAR c2 = *s2;
		if( ES_ST('A') <= c1 && c1 <= ES_ST('Z') ) c1 = c1 - ES_ST('A') + ES_ST('a');
		if( ES_ST('A') <= c2 && c2 <= ES_ST('Z') ) c2 = c2 - ES_ST('A') + ES_ST('a');
		if( c1 != c2 ) return false;
		s1++; s2++; chars--;
	}
	if( chars != 0 ) return false;
	return true;
}
INT compareSChars(const SCHAR* s1, const SCHAR * s2)
{
	if( s1 == s2 ) return 0;
	if( s1 == NULL ) return -1;
	if( s2 == NULL ) return 1;
	while(*s1 && *s2) {
		INT d = *s1++ - *s2++;
		if( d != 0 ) return d;
	}
	if( *s1 != 0 ) return 1;
	if( *s2 != 0 ) return -1;
	return 0;
}

INT compareSCharsNoCase(const SCHAR* s1, const SCHAR * s2)
{
	if( s1 == s2 ) return 0;
	if( s1 == NULL ) return -1;
	if( s2 == NULL ) return 1;
	while(*s1 && *s2) {
		SCHAR c1 = *s1++;
		SCHAR c2 = *s2++;
		if( ES_ST('A') <= c1 && c1 <= ES_ST('Z') ) c1 = c1 - ES_ST('A') + ES_ST('a');
		if( ES_ST('A') <= c2 && c2 <= ES_ST('Z') ) c2 = c2 - ES_ST('A') + ES_ST('a');
		INT d = c1 - c2;
		if( d != 0 ) return d;
	}
	if( *s1 != 0 ) return 1;
	if( *s2 != 0 ) return -1;
	return 0;
}

SIZE findSChar(const SCHAR* s, SCHAR k)
{
	const SCHAR *p = s;
	if( p ) {
		while(*p) {
			if(*p == k) return (p-s)/sizeof(SCHAR);
			p++;
		}
	}
	return NOSIZE;
}

SIZE findSChars(const SCHAR* s, const SCHAR *k)
{
	const SCHAR *p = s;
	if( p && k ) {
		while(*p) {
			if(*p == *k) {
				const SCHAR *c1 = p;
				const SCHAR *c2 = k;
				while(*c1 && *c1 == *c2) {
					c1++; c2++;
				}
				if( *c2 == 0 ) {
					return (p-s)/sizeof(SCHAR);
				}
			}
			p++;
		}
	}
	return NOSIZE;
}

SIZE findSCharSet(const SCHAR* s, const SCHAR* k)
{
	const SCHAR *p = s;
	if( p && k ) {
		while(*p) {
			const SCHAR *q = k;
			while(*q ) {
				if( *p == *q ) return (p-s)/sizeof(SCHAR);
				q++;
			}
			p++;
		}
	}
	return NOSIZE;
}

SIZE findLastSChar(const SCHAR* s, SCHAR k)
{
	if( s && k ) {
		SIZE sn = getSCharsLength(s);
		const SCHAR *p = s+sn-1;
		while( s <= p ) {
			if(*p == k) return (p-s)/sizeof(SCHAR);
			p--;
		}
	}
	return NOSIZE;
}

SIZE findLastSChars(const SCHAR* s, const SCHAR* k)
{
	if( s && k ) {
		SIZE sn = getSCharsLength(s);
		const SCHAR *p = s+sn-1;
		while(s <= p) {
			if(*p == *k) {
				const SCHAR *c1 = p;
				const SCHAR *c2 = k;
				while(*c1 && *c1 == *c2) {
					c1++; c2++;
				}
				if( *c2 == 0 ) {
					return (p-s)/sizeof(SCHAR);
				}
			}
			p--;
		}
	}
	return NOSIZE;
}

SIZE findLastSCharSet(const SCHAR* s, const SCHAR* k)
{
	if( s && k ) {
		SIZE sn = getSCharsLength(s);
		const SCHAR *p = s+sn-1;
		while(s <= p) {
			const SCHAR *q = k;
			while(*q ) {
				if( *p == *q ) {
					return (p-s)/sizeof(SCHAR);
				}
				q++;
			}
			p--;
		}
	}
	return NOSIZE;
}

SIZE findNotSChar(const SCHAR* s, SCHAR k)
{
	const SCHAR *p = s;
	if( p ) {
		while(*p) {
			if(*p != k) return (p-s)/sizeof(SCHAR);
			p++;
		}
	}
	return NOSIZE;
}

SIZE findNotSChars(const SCHAR* s, const SCHAR *k)
{
	const SCHAR *p = s;
	if( p && k ) {
		while(*p) {
			if(*p == *k) {
				const SCHAR *c1 = p;
				const SCHAR *c2 = k;
				while(*c1 && *c1 == *c2) {
					c1++; c2++;
				}
				if( *c2 == 0 ) {
					p++;
					continue;
				}
			}
			return (p-s)/sizeof(SCHAR);
		}
	}
	return NOSIZE;
}

SIZE findNotSCharSet(const SCHAR* s, const SCHAR* k)
{
	const SCHAR *p = s;
	if( p && k ) {
		while(*p) {
			const SCHAR *q = k;
			while(*q ) {
				if( *p == *q ) break;
				q++;
			}
			if( *q == 0 ) {
				return (p-s)/sizeof(SCHAR);
			}
			p++;
		}
	}
	return NOSIZE;
}


SIZE findLastNotSChar(const SCHAR* s, SCHAR k)
{
	if( s && k ) {
		SIZE sn = getSCharsLength(s);
		const SCHAR *p = s + sn - 1;;
		while(s <= p) {
			if(*p != k) return (p-s)/sizeof(SCHAR);
			p--;
		}
	}
	return NOSIZE;
}

SIZE findLastNotSCharSet(const SCHAR* s, const SCHAR* k)
{
	if( s && k ) {
		SIZE sn = getSCharsLength(s);
		const SCHAR *p = s + sn - 1;;
		while(s <= p) {
			const SCHAR *q = k;
			while(*q ) {
				if( *p == *q ) break;
				q++;
			}
			if( *q == 0 ) {
				return (p-s)/sizeof(SCHAR);
			}
			p--;
		}
	}
	return NOSIZE;
}

SSTRING * allocSString(SIZE chars)
{
	return (SSTRING*)allocBytes(chars * sizeof(SCHAR) + sizeof(SCHAR));
}

void initSString(SSTRING *ptr)
{
	initBytes((BYTES*)ptr);
}

bool resizeSString(SSTRING *s, SIZE chars)
{
	return resizeBytes((BYTES*)s, chars * sizeof(SCHAR) + sizeof(SCHAR));
}

bool extendSString(SSTRING *s, SIZE chars)
{
	return extendBytes((BYTES*)s, chars * sizeof(SCHAR) + sizeof(SCHAR));
}

void clearSString(SSTRING *s)
{
	clearBytes((BYTES *)s);
}

void freeSString(SSTRING *s)
{
	freeBytes((BYTES*)s);
}

bool copySString(SSTRING * dest, const SSTRING *src)
{
	return copyBytes((BYTES*)dest, (const BYTES *)src);
}

bool isEmptySString(const SSTRING *s)
{
	return isEmptyBytes((const BYTES*)s);
}

bool isEqualSString(const SSTRING * s, const SSTRING * d)
{
	if( s == d ) return true;
	if( !s || !d ) return false;
	if( s->shptr == d->shptr ) return true;
	return isEqualSChars((SCHAR*)s->shptr->_char, (SCHAR*)d->shptr->_char);
}

SIZE getSStringSize(const SSTRING *s)
{
	SIZE size = getBytesSize((const BYTES*)s);
	if( size < sizeof(SCHAR) ) return 0;
	return size  / sizeof(SCHAR) - 1;
}

SIZE getSStringLength(const SSTRING *s)
{
	SIZE size = 0;
	if( s ) {
		size = getSCharsLength(getSStringReadPtr(s));
	}
	return size;
}

SIZE getSStringCapacity(const SSTRING *s)
{
	SIZE size = getBytesCapacity((const BYTES*)s);
	if( size < sizeof(SCHAR) ) return 0;
	return size  / sizeof(SCHAR) - 1;
}

const SCHAR * getSStringReadPtr(const SSTRING *s)
{
	return (const SCHAR *)getBytesReadPtr((const BYTES *)s);
}

SCHAR * getSStringWritePtr(SSTRING *s)
{
	return (SCHAR *)getBytesWritePtr((BYTES *)s);
}

bool setSChars(SSTRING * s, const SCHAR * d)
{
	clearSString(s);
	SIZE n = getSCharsLength(d);
	if( n ) {
		if( !resizeSString(s, n) ) return false;
		copySChars(getSStringWritePtr(s), d);
	}
	return true;
}

bool setSCharsLen(SSTRING * s, const SCHAR * d, SIZE chars)
{
	clearSString(s);
	SIZE n = getSCharsLength(d);
	if( chars < n ) n = chars;
	if( n ) {
		if( !resizeSString(s, n) ) return false;
		copySCharsLen(getSStringWritePtr(s), d, n);
	}
	return true;
}

bool addSChar(SSTRING * s, SCHAR d)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d ) {
		SIZE n = getSStringSize(s);
		SIZE l = getSStringLength(s);
		if( l+1 >= n ) {
			if( !resizeSString(s, l+1) ) return false;
		}
		SCHAR *ptr = getSStringWritePtr(s) + l;
		*ptr++ = d;
		*ptr++ = 0;
	}
	return true;
}

bool addSChars(SSTRING * s, const SCHAR* d)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d && *d ) {
		SIZE a = getSCharsLength(d);
		SIZE n = getSStringSize(s);
		if( !resizeSString(s, n+a) ) return false;
		SCHAR *ptr = getSStringWritePtr(s) + n;
		copySChars(ptr, d);
	}
	return true;
}

bool addSCharsLen(SSTRING * s, const SCHAR* d, SIZE len)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d && *d ) {
		SIZE a = getSCharsLength(d);
		if( len < a ) a = len;
		SIZE n = getSStringSize(s);
		if( !resizeSString(s, n+a) ) return false;
		SCHAR *ptr = getSStringWritePtr(s) + n;
		copySCharsLen(ptr, d, a);
	}
	return true;
}

bool addSString(SSTRING * s, const SSTRING *d)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d ) {
		SIZE addSize = getSStringSize(d);
		if( addSize == 0 ) return true;
		SIZE curSize = getSStringSize(s);
		if( !resizeSString(s, curSize+addSize) ) return false;
		SCHAR *ptr = getSStringWritePtr(s) + curSize;
		copySChars(ptr, getSStringReadPtr(d));
	}
	return true;
}

bool cropSString(SSTRING * s, const SSTRING *d, SIZE pos, SIZE len)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	SString t;
	if( d && len > 0) {
		SIZE a = getSStringLength(d);
		if( pos < a ) {
			if( a < pos + len ) len = a - pos;
			const SCHAR *ptr = getSStringReadPtr(d) + pos;
			if( !resizeSString(t, len) ) return false;
			if( !copySCharsLen(getSStringWritePtr(t), ptr, len) ) return false;
		}
	}
	SStringPtr ptr(s);
	ptr = t;
	return true;
}

bool trimLeftSString(SSTRING * s, const SSTRING *d, const SCHAR *trimCharSet)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d ) {
		const SCHAR *ptr = getSStringReadPtr(d);
		if( ptr ) {
			SString wsp = ES_ST(" ");
			if( trimCharSet && *trimCharSet )
				wsp += trimCharSet;
			const SCHAR *cset = wsp.ptr();
			SIZE pos = findSCharSet(ptr, cset);
			if( pos == NOSIZE ) {
				copySString(s,d);
			} else {
				setSChars(s, ptr+pos);
			}
		}
	}
	return true;
}

bool trimRightSString(SSTRING * s, const SSTRING *d, const SCHAR *trimCharSet)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d ) {
		const SCHAR *ptr = getSStringReadPtr(d);
		if( ptr ) {
			SString wsp = ES_ST(" ");
			if( trimCharSet && *trimCharSet )
				wsp += trimCharSet;
			const SCHAR *cset = wsp.ptr();
			SIZE pos = findLastNotSCharSet(ptr, cset);
			if( pos == NOSIZE ) {
				copySString(s,d);
			} else {
				cropSString(s, d, 0, pos+1);
			}
		}
	}
	return true;
}

}; // es
