/*
 * sstring.h
 *
 *  Created on: 2012/01/27
 *      Author: tanaka
 */

#ifndef ES_SSTRING_H_
#define ES_SSTRING_H_

#include "../include/elise.h"
#include "platform.h"
#include "error.h"

namespace es {

class SString
{
public:
	SString() {
		initSString(&s);
	}
	SString(const SSTRING * d) {
		s.shptr = NULL;
		if( !copySString(&s, d) )
			throw Error();
	}
	SString(const SSTRING &d) {
		s.shptr = NULL;
		if( !copySString(&s, &d) )
			throw Error();
	}
	SString(const SCHAR * d) {
		s.shptr = NULL;
		if( !setSChars(&s, d) )
			throw Error();
	}
	SString(const SCHAR d) {
		s.shptr = NULL;
		SCHAR v[2];
		v[0] = d;
		v[1] = 0;
		if( !setSChars(&s, v) )
			throw Error();
	}
	SString(const SString &d) {
		s.shptr = NULL;
		if( !copySString(&s, d.sstringPtr()) )
			throw Error();
	}
	SString& operator=(const SCHAR *d) {
		if( !setSChars(&s, d) ) {
			throw Error();
		}
		return *this;
	}
	SString& operator=(SCHAR d) {
		SCHAR v[2];
		v[0] = d;
		v[1] = 0;
		if( !setSChars(&s, v) ) {
			throw Error();
		}
		return *this;
	}
	SString& operator=(const SString &d) {
		if( this != &d ) {
			copy(d.sstringPtr());
		}
		return *this;
	}
	virtual ~SString() {
		clear();
	}
	void resize(SIZE size) {
		if( !resizeSString(&s, size) )
			throw Error();
	}
	void clear() {
		clearSString(&s);
	}

	bool empty() const {
		return isEmptySString(&s);
	}
	SIZE length() const {
		if( !empty() ) {
			return getSCharsLength(getSStringReadPtr(&s));
		}
		return 0;
	}
	SIZE capacity() const {
		return getSStringCapacity(&s);
	}
	SIZE size() const {
		return getSStringSize(&s);
	}
	const SCHAR * ptr() const {
		return getSStringReadPtr(&s);
	}
	SCHAR * ptr() {
		return getSStringWritePtr(&s);
	}
	const SSTRING * sstringPtr() const {
		return &s;
	}
	SSTRING * sstringPtr() {
		return &s;
	}
	SCHAR at(SIZE pos) const {
		return ptr()[pos];
	}
	void set(const SCHAR * set) {
		if( !setSChars(sstringPtr(), set) )
			throw Error();
	}
	void set(const SCHAR * set, SIZE len) {
		if( !setSCharsLen(sstringPtr(), set, len) )
			throw Error();
	}
	void copy(const SSTRING * src) {
		if( !copySString(sstringPtr(), src) )
			throw Error();
	}
	void add(const SSTRING * src) {
		if( !addSString(sstringPtr(), src) )
			throw Error();
	}
	void add(const SCHAR * src) {
		if( !addSChars(sstringPtr(), src) )
			throw Error();
	}
	void add(const SCHAR * src, SIZE len) {
		if( !addSCharsLen(sstringPtr(), src, len) )
			throw Error();
	}
	void add(SCHAR src) {
		if( !addSChar(sstringPtr(), src) )
			throw Error();
	}
	SString	crop(SIZE pos, SIZE size) const {
		SString ret;
		if( !cropSString(ret, sstringPtr(), pos, size) )
			throw Error();
		return ret;
	}

	SString	left(SIZE pos) const {
		return crop(0,pos);
	}
	SString	right(SIZE pos) const {
		return crop(pos,size()-pos);
	}
	SString	trimLeft(const SCHAR *trimCharSet=NULL) const {
		SString ret;
		if( !trimLeftSString(ret, sstringPtr(), trimCharSet) )
			throw Error();
		return ret;
	}
	SString	trimRight(const SCHAR *trimCharSet=NULL) const {
		SString ret;
		if( !trimRightSString(ret, sstringPtr(), trimCharSet) )
			throw Error();
		return ret;
	}
	SString	trim(const SCHAR *trimCharSet=NULL) const {
		return trimLeft(trimCharSet).trimRight(trimCharSet);
	}
	bool equals(const SCHAR * cmp) const {
		return isEqualSChars(ptr(), cmp);
	}
	bool equals(const SSTRING * cmp) const {
		return isEqualSString(sstringPtr(), cmp);
	}
	bool equalsNocase(const SCHAR * cmp) const {
		return isEqualSCharsNoCase(ptr(), cmp);
	}
	bool equalsNocase(const SSTRING * cmp) const {
		return isEqualSCharsNoCase(ptr(), getSStringReadPtr(cmp));
	}
	INT compare(const SCHAR * cmp) const {
		return compareSChars(ptr(), cmp);
	}
	INT compare(const SSTRING * cmp) const {
		return compareSChars(ptr(), getSStringReadPtr(cmp));
	}
	INT compareNocase(const SCHAR * cmp) const {
		return compareSCharsNoCase(ptr(), cmp);
	}
	INT compareNocase(const SSTRING * cmp) const {
		return compareSCharsNoCase(ptr(), getSStringReadPtr(cmp));
	}

	SIZE find(SIZE pos, SCHAR find) const {
		return findSChar(ptr()+pos, find);
	}
	SIZE find(SIZE pos, const SCHAR * find) const {
		return findSChars(ptr()+pos, find);
	}
	SIZE find(SIZE pos, const SSTRING * find) const {
		return findSChars(ptr()+pos, getSStringReadPtr(find));
	}
	SIZE findLast(SIZE pos, SCHAR find) const {
		return findLastSChar(ptr()+pos, find);
	}
	SIZE findLast(SIZE pos, const SCHAR * find) const {
		return findLastSChars(ptr()+pos, find);
	}
	SIZE findLast(SIZE pos, const SSTRING * find) const {
		return findLastSChars(ptr()+pos, getSStringReadPtr(find));
	}
	SIZE findCharSet(SIZE pos, const SCHAR * find) const {
		return es::findSCharSet(ptr()+pos, find);
	}
	operator SSTRING * () {
		return sstringPtr();
	}
	operator const SSTRING * () const {
		return sstringPtr();
	}
	SSTRING*	operator->() {
		return sstringPtr();
	}
	const SSTRING* operator->() const {
		return sstringPtr();
	}
	const SCHAR & operator[](INT n) const {
		return ptr()[n];
	}
	SCHAR &	operator[](INT n) {
		return ptr()[n];
	}
	const SCHAR & operator[](SIZE n) const {
		return ptr()[n];
	}
	SCHAR &	operator[](SIZE n) {
		return ptr()[n];
	}
	bool operator==(const SCHAR * d) const {
		return equals(d);
	}
	bool operator==(const SString &d) const {
		return equals(d);
	}
	bool operator!=(const SCHAR * d) const {
		return !equals(d);
	}
	SString	operator+(const SSTRING * add) const {
		SString ret(sstringPtr());
		ret.add(add);
		return ret;
	}
	SString	operator+(const SCHAR * add) const {
		SString ret(sstringPtr());
		ret.add(add);
		return ret;
	}
	SString	operator+(SCHAR add) const {
		SString ret(sstringPtr());
		ret.add(add);
		return ret;
	}
	SString&	operator+=(const SSTRING * d) {
		add(d);
		return *this;
	}
	SString&	operator+=(const SCHAR * d) {
		add(d);
		return *this;
	}
	SString&	operator+=(SCHAR d) {
		add(d);
		return *this;
	}

protected:
	SSTRING s;
};

class SStringPtr
{
protected:
	SStringPtr();

public:
	SStringPtr(SSTRING * d) {
		s = d;
	}
	~SStringPtr() {
	}
	SStringPtr& operator=(const SStringPtr &d) {
		if( this != &d ) {
			copy(d);
		}
		return *this;
	}
	SStringPtr& operator=(const SCHAR *d) {
		if( !setSChars(s, d) ) {
			throw Error();
		}
		return *this;
	}
	SStringPtr& operator=(SCHAR d) {
		SCHAR v[2];
		v[0] = d;
		v[1] = 0;
		if( !setSChars(s, v) ) {
			throw Error();
		}
		return *this;
	}
	SStringPtr& operator=(const SString &d) {
		if( this->s != d.sstringPtr() ) {
			copy(d.sstringPtr());
		}
		return *this;
	}
	void resize(SIZE size) {
		if( !resizeSString(sstringPtr(), size) )
			throw Error();
	}
	void clear() {
		clearSString(sstringPtr());
	}

	bool empty() const {
		return isEmptySString(sstringPtr());
	}
	SIZE length() const {
		if( !empty() ) {
			return getSCharsLength(getSStringReadPtr(sstringPtr()));
		}
		return 0;
	}
	SIZE capacity() const {
		return getSStringCapacity(sstringPtr());
	}
	SIZE size() const {
		return getSStringSize(sstringPtr());
	}
	const SCHAR * ptr() const {
		return getSStringReadPtr(sstringPtr());
	}
	SCHAR * ptr() {
		return getSStringWritePtr(sstringPtr());
	}
	const SSTRING * sstringPtr() const {
		return s;
	}
	SSTRING * sstringPtr() {
		return s;
	}
	SCHAR at(SIZE pos) const {
		return ptr()[pos];
	}
	void set(const SCHAR * set) {
		if( !setSChars(sstringPtr(), set) )
			throw Error();
	}
	void copy(const SSTRING * src) {
		if( !copySString(sstringPtr(), src) )
			throw Error();
	}
	void add(const SSTRING * src) {
		if( !addSString(sstringPtr(), src) )
			throw Error();
	}
	void add(const SCHAR * src) {
		if( !addSChars(sstringPtr(), src) )
			throw Error();
	}
	void add(SCHAR src) {
		if( !addSChar(sstringPtr(), src) )
			throw Error();
	}
	SString	crop(SIZE pos, SIZE size) const {
		SString ret;
		if( !cropSString(ret, sstringPtr(), pos, size) )
			throw Error();
		return ret;
	}

	SString	left(SIZE pos) const {
		return crop(0,pos);
	}
	SString	right(SIZE pos) const {
		return crop(pos,size()-pos);
	}
	SString	trimLeft(const SCHAR *trimCharSet=NULL) const {
		SString ret;
		if( !trimLeftSString(ret, sstringPtr(), trimCharSet) )
			throw Error();
		return ret;
	}
	SString	trimRight(const SCHAR *trimCharSet=NULL) const {
		SString ret;
		if( !trimRightSString(ret, sstringPtr(), trimCharSet) )
			throw Error();
		return ret;
	}
	SString	trim(const SCHAR *trimCharSet=NULL) const {
		return trimLeft(trimCharSet).trimRight(trimCharSet);
	}
	bool equals(const SCHAR * cmp) const {
		return isEqualSChars(ptr(), cmp);
	}
	bool equals(const SSTRING * cmp) const {
		return isEqualSString(sstringPtr(), cmp);
	}
	bool equalsNocase(const SCHAR * cmp) const {
		return isEqualSCharsNoCase(ptr(), cmp);
	}
	bool equalsNocase(const SSTRING * cmp) const {
		return isEqualSCharsNoCase(ptr(), getSStringReadPtr(cmp));
	}
	INT compare(const SCHAR * cmp) const {
		return compareSChars(ptr(), cmp);
	}
	INT compare(const SSTRING * cmp) const {
		return compareSChars(ptr(), getSStringReadPtr(cmp));
	}
	INT compareNocase(const SCHAR * cmp) const {
		return compareSCharsNoCase(ptr(), cmp);
	}
	INT compareNocase(const SSTRING * cmp) const {
		return compareSCharsNoCase(ptr(), getSStringReadPtr(cmp));
	}

	SIZE find(SIZE pos, SCHAR find) const {
		return findSChar(ptr()+pos, find);
	}
	SIZE find(SIZE pos, const SCHAR * find) const {
		return findSChars(ptr()+pos, find);
	}
	SIZE find(SIZE pos, const SSTRING * find) const {
		return findSChars(ptr()+pos, getSStringReadPtr(find));
	}
	SIZE findLast(SIZE pos, SCHAR find) const {
		return findLastSChar(ptr()+pos, find);
	}
	SIZE findLast(SIZE pos, const SCHAR * find) const {
		return findLastSChars(ptr()+pos, find);
	}
	SIZE findLast(SIZE pos, const SSTRING * find) const {
		return findLastSChars(ptr()+pos, getSStringReadPtr(find));
	}
	SIZE findCharSet(SIZE pos, const SCHAR * find) const {
		return es::findSCharSet(ptr()+pos, find);
	}
	operator SSTRING * () {
		return sstringPtr();
	}
	operator const SSTRING * () const {
		return sstringPtr();
	}
	SSTRING*	operator->() {
		return sstringPtr();
	}
	const SSTRING* operator->() const {
		return sstringPtr();
	}
	const SCHAR & operator[](INT n) const {
		return ptr()[n];
	}
	SCHAR &	operator[](INT n) {
		return ptr()[n];
	}
	const SCHAR & operator[](SIZE n) const {
		return ptr()[n];
	}
	SCHAR &	operator[](SIZE n) {
		return ptr()[n];
	}
	bool operator==(const SCHAR * d) const {
		return equals(d);
	}
	bool operator==(const SString &d) const {
		return equals(d);
	}
	bool operator!=(const SCHAR * d) const {
		return !equals(d);
	}
	SString	operator+(const SSTRING * add) const {
		SString ret(sstringPtr());
		ret.add(add);
		return ret;
	}
	SString	operator+(const SCHAR * add) const {
		SString ret(sstringPtr());
		ret.add(add);
		return ret;
	}
	SString	operator+(SCHAR add) const {
		SString ret(sstringPtr());
		ret.add(add);
		return ret;
	}
	SStringPtr&	operator+=(const SSTRING * d) {
		add(d);
		return *this;
	}
	SStringPtr&	operator+=(const SCHAR * d) {
		add(d);
		return *this;
	}
	SStringPtr&	operator+=(SCHAR d) {
		add(d);
		return *this;
	}

protected:
	SSTRING *s;
};

class SStringArray
{
public:
	enum {
		BLOCK_SIZE = 10,
	};
	SStringArray() {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	SStringArray(INT size) {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
		while(size--) add((SCHAR *)NULL);
	}
	SStringArray(const SStringArray &d) {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
		copy(d);
	}
	~SStringArray() {
		clear();
	}
	SStringArray &operator=(const SStringArray &d) {
		if( this != &d ) copy(d);
		return *this;
	}
	void copy(const SStringArray &d) {
		if( this == &d ) return;
		clear();
		resize(d.size());
		SIZE i;
		for( i = 0; i < d.size(); i++ ) {
			copySString(&pNodes[i], &d.pNodes[i]);
		}
	}

	void add(const SSTRING * pSString) {
		if( nNodes+1 > bNodes ) {
			SIZE bNodesNew = bNodes + BLOCK_SIZE;
			SSTRING * pNodesNew = (SSTRING *)resizeMemory(pNodes, sizeof(SSTRING)*bNodesNew);
			if( pNodesNew == NULL ) throw Error();
			for(SIZE i = bNodes; i < bNodesNew; i++ )
				initSString(&pNodesNew[i]);
			pNodes	= pNodesNew;
			bNodes	= bNodesNew;
		}
		copySString(&pNodes[nNodes++], pSString);
	}
	void add(const SCHAR * pSString) {
		SString t(pSString);
		return add(t);
	}
	void clear() {
		SIZE i;
		for( i = 0; i < nNodes; i++ ) {
			clearSString(&pNodes[i]);
		}
		if( pNodes ) freeMemory(pNodes);
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	void resize(SIZE n) {
		SIZE i;
		for( i = n; i < nNodes; i++ ) {
			clearSString(&pNodes[i]);
		}
		nNodes = n;
	}
	SSTRING * get(SIZE n) {
		return &pNodes[n];
	}
	SIZE	size() const {
		return nNodes;
	}
	SIZE	empty() const {
		return nNodes==0;
	}
	SStringPtr operator[](INT index) {
		return SStringPtr(&pNodes[index]);
	}
	const SStringPtr operator[](INT index) const {
		return SStringPtr(&pNodes[index]);
	}
	SIZE		nNodes;
	SIZE		bNodes;
	SSTRING *	pNodes;
};

}

#endif /* SSTRING_H_ */
