/*
 * string.cpp
 *
 *  Created on: 2011/12/23
 *      Author: tanaka
 */
#include <stdlib.h>
#include <wchar.h>
#include <float.h>
#include <math.h>
#include "../include/elise.h"
#include "platform.h"
#include "bytes.h"
#include "string.h"

namespace es {

void initializeString()
{
}

void terminateString()
{
}

bool copyChars(CHAR *s1, const CHAR *s2)
{
	while( *s2 ) *s1++ = *s2++;
	*s1 = 0;
	return true;
}

bool copyCharsLen(CHAR *s1, const CHAR *s2, SIZE len)
{
	if( !s1 ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( s2 ) {
		while( *s2 && len ) { *s1++ = *s2++; len--; }
	}
	*s1 = 0;
	return true;
}

SIZE getCharsLength(const CHAR *s1)
{
	SIZE n=0;
	if( s1 ) {
		while(*s1) {
			n++;
			s1++;
		}
	}
	return n;
}

bool isEqualChars(const CHAR *s1, const CHAR *s2)
{
	if( s1 == s2 ) return true;
	if( !s1 || !s2 ) return false;
	while(*s1 && *s2) {
		if( *s1 != *s2 ) return false;
		s1++; s2++;
	}
	if( *s1 || *s2 ) return false;
	return true;
}

bool isEqualCharsLen(const CHAR *s1, const CHAR *s2, SIZE chars)
{
	if( chars == 0 || s1 == s2 ) return true;
	if( !s1 || !s2 ) return false;
	while(*s1 && *s2 && chars > 0 ) {
		if( *s1 != *s2 ) return false;
		s1++; s2++; chars--;
	}
	if( chars != 0 ) return false;
	return true;
}

bool isEqualCharsNoCase(const CHAR *s1, const CHAR *s2)
{
	if( s1 == s2 ) return true;
	if( !s1 || !s2 ) return false;
	while(*s1 && *s2) {
		CHAR c1 = *s1;
		CHAR c2 = *s2;
		if( ES_T('A') <= c1 && c1 <= ES_T('Z') ) c1 = c1 - ES_T('A') + ES_T('a');
		if( ES_T('A') <= c2 && c2 <= ES_T('Z') ) c2 = c2 - ES_T('A') + ES_T('a');
		if( c1 != c2 ) return false;
		s1++; s2++;
	}
	if( *s1 || *s2 ) return false;
	return true;
}

bool isEqualCharsNoCaseLen(const CHAR *s1, const CHAR *s2, SIZE chars)
{
	if( chars == 0 || s1 == s2 ) return true;
	if( !s1 || !s2 ) return false;
	while(*s1 && *s2 && chars > 0 ) {
		CHAR c1 = *s1;
		CHAR c2 = *s2;
		if( ES_T('A') <= c1 && c1 <= ES_T('Z') ) c1 = c1 - ES_T('A') + ES_T('a');
		if( ES_T('A') <= c2 && c2 <= ES_T('Z') ) c2 = c2 - ES_T('A') + ES_T('a');
		if( c1 != c2 ) return false;
		s1++; s2++; chars--;
	}
	if( chars != 0 ) return false;
	return true;
}
INT compareChars(const CHAR* s1, const CHAR * s2)
{
	if( s1 == s2 ) return 0;
	if( s1 == NULL ) return -1;
	if( s2 == NULL ) return 1;
	while(*s1 && *s2) {
		INT d = *s1++ - *s2++;
		if( d != 0 ) return d;
	}
	if( *s1 != 0 ) return 1;
	if( *s2 != 0 ) return -1;
	return 0;
}

INT compareCharsNoCase(const CHAR* s1, const CHAR * s2)
{
	if( s1 == s2 ) return 0;
	if( s1 == NULL ) return -1;
	if( s2 == NULL ) return 1;
	while(*s1 && *s2) {
		CHAR c1 = *s1++;
		CHAR c2 = *s2++;
		if( ES_T('A') <= c1 && c1 <= ES_T('Z') ) c1 = c1 - ES_T('A') + ES_T('a');
		if( ES_T('A') <= c2 && c2 <= ES_T('Z') ) c2 = c2 - ES_T('A') + ES_T('a');
		INT d = c1 - c2;
		if( d != 0 ) return d;
	}
	if( *s1 != 0 ) return 1;
	if( *s2 != 0 ) return -1;
	return 0;
}

void makeUpperCase(CHAR* s)
{
	if( s ) {
		while(*s) {
			if( ES_T('a') <= *s && *s <= ES_T('z') ) {
				*s = *s - ES_T('a') + ES_T('A');
			}
			s++;
		}
	}
}

void makeLowerCase(CHAR* s)
{
	if( s ) {
		while(*s) {
			if( ES_T('A') <= *s && *s <= ES_T('Z') ) {
				*s = *s - ES_T('A') + ES_T('a');
			}
			s++;
		}
	}
}

SIZE findChar(const CHAR* s, CHAR k)
{
	const CHAR *p = s;
	if( p ) {
		while(*p) {
			if(*p == k) return (p-s)/sizeof(CHAR);
			p++;
		}
	}
	return NOSIZE;
}

SIZE findChars(const CHAR* s, const CHAR *k)
{
	const CHAR *p = s;
	if( p && k ) {
		while(*p) {
			if(*p == *k) {
				const CHAR *c1 = p;
				const CHAR *c2 = k;
				while(*c1 && *c1 == *c2) {
					c1++; c2++;
				}
				if( *c2 == 0 ) {
					return (p-s)/sizeof(CHAR);
				}
			}
			p++;
		}
	}
	return NOSIZE;
}

SIZE findCharSet(const CHAR* s, const CHAR* k)
{
	const CHAR *p = s;
	if( p && k ) {
		while(*p) {
			const CHAR *q = k;
			while(*q ) {
				if( *p == *q ) return (p-s)/sizeof(CHAR);
				q++;
			}
			p++;
		}
	}
	return NOSIZE;
}

SIZE findLastChar(const CHAR* s, CHAR k)
{
	if( s && k ) {
		SIZE sn = getCharsLength(s);
		const CHAR *p = s+sn-1;
		while( s <= p ) {
			if(*p == k) return (p-s)/sizeof(CHAR);
			p--;
		}
	}
	return NOSIZE;
}

SIZE findLastChars(const CHAR* s, const CHAR* k)
{
	if( s && k ) {
		SIZE sn = getCharsLength(s);
		const CHAR *p = s+sn-1;
		while(s <= p) {
			if(*p == *k) {
				const CHAR *c1 = p;
				const CHAR *c2 = k;
				while(*c1 && *c1 == *c2) {
					c1++; c2++;
				}
				if( *c2 == 0 ) {
					return (p-s)/sizeof(CHAR);
				}
			}
			p--;
		}
	}
	return NOSIZE;
}

SIZE findLastCharSet(const CHAR* s, const CHAR* k)
{
	if( s && k ) {
		SIZE sn = getCharsLength(s);
		const CHAR *p = s+sn-1;
		while(s <= p) {
			const CHAR *q = k;
			while(*q ) {
				if( *p == *q ) {
					return (p-s)/sizeof(CHAR);
				}
				q++;
			}
			p--;
		}
	}
	return NOSIZE;
}

SIZE findNotChar(const CHAR* s, CHAR k)
{
	const CHAR *p = s;
	if( p ) {
		while(*p) {
			if(*p != k) return (p-s)/sizeof(CHAR);
			p++;
		}
	}
	return NOSIZE;
}

SIZE findNotChars(const CHAR* s, const CHAR *k)
{
	const CHAR *p = s;
	if( p && k ) {
		while(*p) {
			if(*p == *k) {
				const CHAR *c1 = p;
				const CHAR *c2 = k;
				while(*c1 && *c1 == *c2) {
					c1++; c2++;
				}
				if( *c2 == 0 ) {
					p++;
					continue;
				}
			}
			return (p-s)/sizeof(CHAR);
		}
	}
	return NOSIZE;
}

SIZE findNotCharSet(const CHAR* s, const CHAR* k)
{
	const CHAR *p = s;
	if( p && k ) {
		while(*p) {
			const CHAR *q = k;
			while(*q ) {
				if( *p == *q ) break;
				q++;
			}
			if( *q == 0 ) {
				return (p-s)/sizeof(CHAR);
			}
			p++;
		}
	}
	return NOSIZE;
}


SIZE findLastNotChar(const CHAR* s, CHAR k)
{
	if( s && k ) {
		SIZE sn = getCharsLength(s);
		const CHAR *p = s + sn - 1;;
		while(s <= p) {
			if(*p != k) return (p-s)/sizeof(CHAR);
			p--;
		}
	}
	return NOSIZE;
}

SIZE findLastNotCharSet(const CHAR* s, const CHAR* k)
{
	if( s && k ) {
		SIZE sn = getCharsLength(s);
		const CHAR *p = s + sn - 1;;
		while(s <= p) {
			const CHAR *q = k;
			while(*q ) {
				if( *p == *q ) break;
				q++;
			}
			if( *q == 0 ) {
				return (p-s)/sizeof(CHAR);
			}
			p--;
		}
	}
	return NOSIZE;
}

INT	parseInt(const CHAR* s)
{
	INT n=0;
	if( s ) {
		while(*s) {
			if( *s < ES_T('0') || ES_T('9') < *s ) break;
			n = n * 10 + (*s - ES_T('0'));
			s++;
		}
	}
	return n;
}

STRING * allocString(SIZE chars)
{
	return (STRING*)allocBytes(chars * sizeof(CHAR) + sizeof(CHAR));
}

void initString(STRING *ptr)
{
	initBytes((BYTES*)ptr);
}

bool resizeString(STRING *s, SIZE chars)
{
	return resizeBytes((BYTES*)s, chars * sizeof(CHAR) + sizeof(CHAR));
}

bool extendString(STRING *s, SIZE chars)
{
	return extendBytes((BYTES*)s, chars * sizeof(CHAR) + sizeof(CHAR));
}

void clearString(STRING *s)
{
	clearBytes((BYTES *)s);
}

void freeString(STRING *s)
{
	freeBytes((BYTES*)s);
}

bool copyString(STRING * dest, const STRING *src)
{
	return copyBytes((BYTES*)dest, (const BYTES *)src);
}

bool isEmptyString(const STRING *s)
{
	return isEmptyBytes((const BYTES*)s);
}

bool isEqualString(const STRING * s, const STRING * d)
{
	if( s == d ) return true;
	if( !s || !d ) return false;
	if( s->shptr == d->shptr ) return true;
	return isEqualChars(s->shptr->_wchar, d->shptr->_wchar);
}

SIZE getStringSize(const STRING *s)
{
	SIZE size = getBytesSize((const BYTES*)s);
	if( size < sizeof(CHAR) ) return 0;
	return size  / sizeof(CHAR) - 1;
}

SIZE getStringLength(const STRING *s)
{
	SIZE size = 0;
	if( s ) {
		size = getCharsLength(getStringReadPtr(s));
	}
	return size;
}

SIZE getStringCapacity(const STRING *s)
{
	SIZE size = getBytesCapacity((const BYTES*)s);
	if( size < sizeof(CHAR) ) return 0;
	return size  / sizeof(CHAR) - 1;
}

const CHAR * getStringReadPtr(const STRING *s)
{
	return (const CHAR *)getBytesReadPtr((const BYTES *)s);
}

CHAR * getStringWritePtr(STRING *s)
{
	return (CHAR *)getBytesWritePtr((BYTES *)s);
}

bool setChars(STRING * s, const CHAR * d)
{
	clearString(s);
	SIZE n = getCharsLength(d);
	if( n ) {
		if( !resizeString(s, n) ) return false;
		copyChars(getStringWritePtr(s), d);
	}
	return true;
}

bool setCharsLen(STRING * s, const CHAR * d, SIZE chars)
{
	clearString(s);
	SIZE n = getCharsLength(d);
	if( chars < n ) n = chars;
	if( n ) {
		if( !resizeString(s, n) ) return false;
		copyCharsLen(getStringWritePtr(s), d, n);
	}
	return true;
}

bool addChar(STRING * s, CHAR d)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d ) {
		SIZE n = getStringSize(s);
		SIZE l = getStringLength(s);
		if( l+1 >= n ) {
			if( !resizeString(s, l+1) ) return false;
		}
		CHAR *ptr = getStringWritePtr(s) + l;
		*ptr++ = d;
		*ptr++ = 0;
	}
	return true;
}

bool addChars(STRING * s, const CHAR* d)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d && *d ) {
		SIZE a = getCharsLength(d);
		SIZE n = getStringSize(s);
		if( !resizeString(s, n+a) ) return false;
		CHAR *ptr = getStringWritePtr(s) + n;
		copyChars(ptr, d);
	}
	return true;
}

bool addCharsLen(STRING * s, const CHAR* d, SIZE len)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d && *d ) {
		SIZE a = getCharsLength(d);
		if( len < a ) a = len;
		SIZE n = getStringSize(s);
		if( !resizeString(s, n+a) ) return false;
		CHAR *ptr = getStringWritePtr(s) + n;
		copyCharsLen(ptr, d, a);
	}
	return true;
}

bool addString(STRING * s, const STRING *d)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d ) {
		SIZE addSize = getStringSize(d);
		if( addSize == 0 ) return true;
		SIZE curSize = getStringSize(s);
		if( !resizeString(s, curSize+addSize) ) return false;
		CHAR *ptr = getStringWritePtr(s) + curSize;
		copyChars(ptr, getStringReadPtr(d));
	}
	return true;
}

bool cropString(STRING * s, const STRING *d, SIZE pos, SIZE len)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	String t;
	if( d && len > 0) {
		SIZE a = getStringLength(d);
		if( pos < a ) {
			if( a < pos + len ) len = a - pos;
			const CHAR *ptr = getStringReadPtr(d) + pos;
			if( !resizeString(t, len) ) return false;
			if( !copyCharsLen(getStringWritePtr(t), ptr, len) ) return false;
		}
	}
	StringPtr ptr(s);
	ptr = t;
	return true;
}

bool trimLeftString(STRING * s, const STRING *d, const CHAR *trimCharSet)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d ) {
		const CHAR *ptr = getStringReadPtr(d);
		if( ptr ) {
			String wsp = ES_T(" ");
			if( trimCharSet && *trimCharSet )
				wsp += trimCharSet;
			const CHAR *cset = wsp.ptr();
			SIZE pos = findCharSet(ptr, cset);
			if( pos == NOSIZE ) {
				copyString(s,d);
			} else {
				setChars(s, ptr+pos);
			}
		}
	}
	return true;
}

bool trimRightString(STRING * s, const STRING *d, const CHAR *trimCharSet)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	if( d ) {
		const CHAR *ptr = getStringReadPtr(d);
		if( ptr ) {
			String wsp = ES_T(" ");
			if( trimCharSet && *trimCharSet )
				wsp += trimCharSet;
			const CHAR *cset = wsp.ptr();
			SIZE pos = findLastNotCharSet(ptr, cset);
			if( pos == NOSIZE ) {
				copyString(s,d);
			} else {
				cropString(s, d, 0, pos+1);
			}
		}
	}
	return true;
}

bool boolToString(STRING *s, bool v)
{
	if( v )
		return setChars(s, ES_T("true"));
	else
		return setChars(s, ES_T("false"));
}

static const CHAR *hexChar=ES_T("0123456789abcdef");
bool byteToString(STRING *s, BYTE v)
{
	StringPtr str(s);
	try {
		str += hexChar[(v>>4)&0x0f];
		str += hexChar[(v&0x0f)];
	}
	catch(Error &) {
		return false;
	}
	catch(...) {
		setError(ERROR_RUNTIME);
		return false;
	}
	return true;
}

bool wordToString(STRING *s, WORD v)
{
	StringPtr str(s);
	try {
		str += hexChar[(v>>12)&0x0f];
		str += hexChar[(v>>8)&0x0f];
		str += hexChar[(v>>4)&0x0f];
		str += hexChar[(v&0x0f)];
	}
	catch(Error &) {
		return false;
	}
	catch(...) {
		setError(ERROR_RUNTIME);
		return false;
	}
	return true;
}

bool dwordToString(STRING *s, DWORD v)
{
	StringPtr str(s);
	try {
		str += hexChar[(v>>28)&0x0f];
		str += hexChar[(v>>24)&0x0f];
		str += hexChar[(v>>20)&0x0f];
		str += hexChar[(v>>16)&0x0f];
		str += hexChar[(v>>12)&0x0f];
		str += hexChar[(v>>8)&0x0f];
		str += hexChar[(v>>4)&0x0f];
		str += hexChar[(v&0x0f)];
	}
	catch(Error &) {
		return false;
	}
	catch(...) {
		setError(ERROR_RUNTIME);
		return false;
	}
	return true;
}

static const CHAR *decChar=ES_T("0123456789");
bool intToString(STRING *s, INT64 v)
{
	CHAR b[16];
	CHAR *ptr = b;
	if( v < 0 ) {
		*ptr++ = ES_T('-');
		v = -v;
	}
	do {
		*ptr++ = decChar[v%10];
		v /= 10;
	} while(v);
	*ptr-- = 0;
	CHAR *swp = b;
	while(swp<ptr) {
		CHAR c = *ptr;
		*ptr = *swp;
		*swp = c;
		ptr--;
		swp++;
	}
	return setChars(s, b);
}

bool shortToString(STRING *s, SHORT v)
{
	return intToString(s, v);
}

bool decimalToString(STRING *s, DECIMAL v)
{
	STRING	istr;
	STRING	nstr;
	STRING	fstr;
	initString(&istr);
	initString(&nstr);
	initString(&fstr);

	INT64	idata = v.dv / (INT64)10000;
	INT64	pdata = v.dv % (INT64)10000;

	SIZE	slen;
	SIZE	ilen;
	SIZE	flen;
	SIZE	plen;
	if( idata == (INT64)0 )  {
		slen = 0;
		ilen = 1;
		setChars(&istr, ES_T("0"));
	} else if( idata < (INT64)0 )  {
		slen = 1;
		intToString(&istr, -idata);
		ilen = getStringLength(&istr);
	} else {
		slen = 0;
		intToString(&istr, idata);
		ilen = getStringLength(&istr);
	}

	if( pdata == (INT64)0 ) {
		flen = 0;
	} else if( pdata < (INT64)0 ) {
		slen = 1;
		intToString(&nstr, -pdata);
		flen = getStringLength(&nstr);
	} else {
		intToString(&nstr, pdata);
		flen = getStringLength(&nstr);
	}

	setChars(&fstr, ES_T("0000"));
	SIZE setPos = 4-flen<0?0:4-flen;
	CHAR *fbuf = getStringWritePtr(&fstr);
	copyCharsLen(&fbuf[setPos], getStringReadPtr(&nstr), 4-setPos);
	for( flen=4; flen > 0; flen--) {
		if( fbuf[flen-1] != ES_T('0') ) break;
	}
	resizeString(&fstr, flen);
	plen = flen ? 1 : 0;

	clearString(s);
	if( slen ) {
		if( !addChar(s, ES_T('-')) ) {
			return false;
		}
	}
	if( !addString(s, &istr) ) {
		return false;
	}
	if( flen ) {
		if( !addChar(s, ES_T('.')) ||
			!addString(s, &fstr) ) {
			return false;
		}
	}
	return true;
}

bool floatToString(STRING *s, FLOAT v)
{
	FLOAT idata=0;
	modf(v, &idata);
	INT	  ilen=0;
	if( idata == 0.0 ) {
		ilen = 1;
	} else {
		ilen = (INT)(log10(fabs(idata))+1);
	}

	INT flen = DBL_DIG - ilen;
	if( flen < 0 ) flen	= 0;

	INT	sign = 0;
	INT dec	= 0;
	const char *cvtchar = fcvt(v, flen, &dec, &sign);
	if( cvtchar == NULL ) {
		setError(ERROR_RUNTIME);
		return false;
	}
	SIZE cvtlen=0;
	CHAR cvtdigit[512];
	CHAR *cvtcnv = cvtdigit;
	while(*cvtchar) {
		cvtlen++;
		*cvtcnv++ = decChar[*cvtchar++ -'0'];
	}
	*cvtcnv = 0;

	String istr;
	String fstr;
	try {
		if( dec <= 0 ) {
			istr = ES_T("0");
			if( dec < 0 ) {
				INT zn = (INT)abs(dec);
				for(INT i=0; i<zn; i++) fstr += ES_T('0');
			}
			fstr += cvtdigit;
		} else {
			CHAR *s = cvtdigit;
			for(INT i=0; i<dec; i++) {
				istr += *s++;
			}
			INT zn = cvtlen - dec;
			if( zn > 0 ) {
				fstr = &cvtdigit[dec];
			}
		}

		if( !fstr.empty() ) {
			CHAR *ptr = fstr.ptr();
			CHAR *p = ptr;
			CHAR *lastNotZero = NULL;
			while( *p ) {
				if( *p != '0' ) lastNotZero = p;
				p++;
			}
			if( lastNotZero ) {
				SIZE cutPos = lastNotZero - ptr;
				SIZE size = fstr.size();
				if( cutPos < size ) fstr = fstr.left(cutPos);
			} else {
				fstr.clear();
			}
		}

		StringPtr ptr(s);
		ptr.clear();
		if( v < 0 ) {
			ptr += ES_T('-');
		}
		ptr += istr;
		if( !fstr.empty() ) {
			ptr += ES_T('.');
			ptr += fstr;
		}
	}
	catch(Error &) {
		return false;
	}
	return true;
}

bool bytesToString(STRING *s, BYTES v)
{
	StringPtr str(s);
	SIZE n	= getBytesSize(&v);
	const BYTE *p = getBytesReadPtr(&v);
	try {
		while(n) {
			BYTE b = *p++;
			str += hexChar[(b>>4)&0x0f];
			str += hexChar[b&0x0f];
			n--;
		}
	}
	catch(Error &) {
		return false;
	}
	return true;
}

bool formatStringVa(STRING * s, const CHAR * format, va_list args)
{
	if( !s ) {
		setError(ERROR_PARAMETER);
		return false;
	}
	const CHAR *fp = format;
	String arg_type;
	SIZE typelen = 0;
	while(*fp) {
		if( *fp == ES_T('%') ) {
			fp++;
			if( *fp == ES_T('%') ) {
				fp++;
				continue;
			}
			SIZE n = 0;
			while(ES_T('0') <= *fp && *fp <= ES_T('9') ) {
				n = n * 10 + (*fp - ES_T('0'));
				fp++;
			}
			if( n == 0 ) {
				continue;
			}
			while( typelen <= n ) {
				arg_type += ES_T(' ');
				typelen++;
			}
			CHAR type = ES_T('S');		// default type is string
			if( *fp != ES_T(';') ) {	// ';' is terminate character
				type = *fp++;
			}
			if( arg_type[n] != type && arg_type[n] != ES_T(' ') && type != ES_T('s') ) {
				setError(ERROR_PARAMETER);
				return false;
			}
			arg_type[n] = type;
			if( *fp == ES_T(';') ) {	// ';' is terminate character
				fp++;
			}
		} else {
			fp++;
		}
	}

	try {
		StringArray arg_text(arg_type.size());
		for( SIZE i=1; i<arg_type.size(); i++ ) {
			String tv;
			CHAR t = arg_type[i];
			switch(t) {
			case ES_T('L'):	// bool
				if( !boolToString(tv, va_arg(args, int)?true:false) ) return false;
				break;
			case ES_T('C'):	// char
				tv = va_arg(args, int);
				break;
			case ES_T('B'):	// byte
				if( !byteToString(tv, (BYTE)va_arg(args, int)) ) return false;
				break;
			case ES_T('W'):	// word
				if( !wordToString(tv, (WORD)va_arg(args, int)) ) return false;
				break;
			case ES_T('D'):	// dword
				if( !dwordToString(tv, va_arg(args, DWORD)) ) return false;
				break;
			case ES_T('s'):	// short
				if( !shortToString(tv, (SHORT)va_arg(args, int)) ) return false;
				break;
			case ES_T('i'):	// int
				if( !intToString(tv, va_arg(args, INT)) ) return false;
				break;
			case ES_T('d'):	// decimal
				if( !decimalToString(tv, va_arg(args, DECIMAL)) ) return false;
				break;
			case ES_T('f'):	// float
				if( !floatToString(tv, va_arg(args, FLOAT)) ) return false;
				break;
			case ES_T('H'):	// bytes
				if( !bytesToString(tv, va_arg(args, BYTES)) ) return false;
				break;
			case ES_T('S'):	// string
				tv = va_arg(args, STRING);
				break;
			default:
				setError(ERROR_TYPE);
				return false;
			}
			arg_text[i] = tv;
		}
		fp = format;
		StringPtr buf(s);
		buf.clear();
		while(*fp) {
			if( *fp == ES_T('%') ) {
				fp++;
				if( *fp == ES_T('%') ) {
					fp++;
					buf += ES_T('%');
					continue;
				}
				SIZE n = 0;
				while(ES_T('0') <= *fp && *fp <= ES_T('9') ) {
					n = n * 10 + (*fp - ES_T('0'));
					fp++;
				}
				if( n == 0 ) {
					buf += ES_T('%');
					continue;
				}
				buf += arg_text[n];
				if( *fp != ES_T(';') ) {	// ';' is terminate character
					fp++;
				}
			} else {
				buf += *fp++;
			}
		}
	}
	catch(Error &) {
		return false;
	}
	return true;
}

bool formatString(STRING * s, const CHAR * format, ...)
{
	va_list	args;
	va_start(args, format);
	bool r = formatStringVa(s, format, args);
	va_end(args);
	return r;
}

} // es
