/*
 * string.h
 *
 *  Created on: 2011/12/24
 *      Author: tanaka
 */

#ifndef ES_STRING_H_
#define ES_STRING_H_

#include "../include/elise.h"
#include "platform.h"
#include "error.h"

namespace es {

class String
{
public:
	String() {
		initString(&s);
	}
	String(const STRING * d) {
		s.shptr = NULL;
		if( !copyString(&s, d) )
			throw Error();
	}
	String(const STRING &d) {
		s.shptr = NULL;
		if( !copyString(&s, &d) )
			throw Error();
	}
	String(const CHAR * d) {
		s.shptr = NULL;
		if( !setChars(&s, d) )
			throw Error();
	}
	String(const CHAR d) {
		s.shptr = NULL;
		CHAR v[2];
		v[0] = d;
		v[1] = 0;
		if( !setChars(&s, v) )
			throw Error();
	}
	String(const String &d) {
		s.shptr = NULL;
		if( !copyString(&s, d.stringPtr()) )
			throw Error();
	}
	String& operator=(const CHAR *d) {
		if( !setChars(&s, d) ) {
			throw Error();
		}
		return *this;
	}
	String& operator=(CHAR d) {
		CHAR v[2];
		v[0] = d;
		v[1] = 0;
		if( !setChars(&s, v) ) {
			throw Error();
		}
		return *this;
	}
	String& operator=(const String &d) {
		if( this != &d ) {
			copy(d.stringPtr());
		}
		return *this;
	}
	virtual ~String() {
		clear();
	}
	void resize(SIZE size) {
		if( !resizeString(&s, size) )
			throw Error();
	}
	void clear() {
		clearString(&s);
	}

	bool empty() const {
		return isEmptyString(&s);
	}
	SIZE length() const {
		if( !empty() ) {
			return getCharsLength(getStringReadPtr(&s));
		}
		return 0;
	}
	SIZE capacity() const {
		return getStringCapacity(&s);
	}
	SIZE size() const {
		return getStringSize(&s);
	}
	const CHAR * ptr() const {
		return getStringReadPtr(&s);
	}
	CHAR * ptr() {
		return getStringWritePtr(&s);
	}
	const STRING * stringPtr() const {
		return &s;
	}
	STRING * stringPtr() {
		return &s;
	}
	CHAR at(SIZE pos) const {
		return ptr()[pos];
	}
	void set(const CHAR * set) {
		if( !setChars(stringPtr(), set) )
			throw Error();
	}
	void set(const CHAR * set, SIZE len) {
		if( !setCharsLen(stringPtr(), set, len) )
			throw Error();
	}
	void copy(const STRING * src) {
		if( !copyString(stringPtr(), src) )
			throw Error();
	}
	void add(const STRING * src) {
		if( !addString(stringPtr(), src) )
			throw Error();
	}
	void add(const CHAR * src) {
		if( !addChars(stringPtr(), src) )
			throw Error();
	}
	void add(const CHAR * src, SIZE len) {
		if( !addCharsLen(stringPtr(), src, len) )
			throw Error();
	}
	void add(CHAR src) {
		if( !addChar(stringPtr(), src) )
			throw Error();
	}
	String	crop(SIZE pos, SIZE size) const {
		String ret;
		if( !cropString(ret, stringPtr(), pos, size) )
			throw Error();
		return ret;
	}

	String	left(SIZE pos) const {
		return crop(0,pos);
	}
	String	right(SIZE pos) const {
		return crop(pos,size()-pos);
	}
	String	trimLeft(const CHAR *trimCharSet=NULL) const {
		String ret;
		if( !trimLeftString(ret, stringPtr(), trimCharSet) )
			throw Error();
		return ret;
	}
	String	trimRight(const CHAR *trimCharSet=NULL) const {
		String ret;
		if( !trimRightString(ret, stringPtr(), trimCharSet) )
			throw Error();
		return ret;
	}
	String	trim(const CHAR *trimCharSet=NULL) const {
		return trimLeft(trimCharSet).trimRight(trimCharSet);
	}
	String	upperCase() const {
		String ret(stringPtr());
		makeUpperCase(ret.ptr());
		return ret;
	}
	String	lowerCase() const {
		String ret(stringPtr());
		makeLowerCase(ret.ptr());
		return ret;
	}
	void format(const CHAR * format, ...) {
		va_list	ap;
		va_start(ap, format);
		clear();
		if( !formatStringVa(stringPtr(), format, ap) )
			throw Error();
	}
	bool equals(const CHAR * cmp) const {
		return isEqualChars(ptr(), cmp);
	}
	bool equals(const STRING * cmp) const {
		return isEqualString(stringPtr(), cmp);
	}
	bool equalsNocase(const CHAR * cmp) const {
		return isEqualCharsNoCase(ptr(), cmp);
	}
	bool equalsNocase(const STRING * cmp) const {
		return isEqualCharsNoCase(ptr(), getStringReadPtr(cmp));
	}
	INT compare(const CHAR * cmp) const {
		return compareChars(ptr(), cmp);
	}
	INT compare(const STRING * cmp) const {
		return compareChars(ptr(), getStringReadPtr(cmp));
	}
	INT compareNocase(const CHAR * cmp) const {
		return compareCharsNoCase(ptr(), cmp);
	}
	INT compareNocase(const STRING * cmp) const {
		return compareCharsNoCase(ptr(), getStringReadPtr(cmp));
	}

	SIZE find(SIZE pos, CHAR find) const {
		return findChar(ptr()+pos, find);
	}
	SIZE find(SIZE pos, const CHAR * find) const {
		return findChars(ptr()+pos, find);
	}
	SIZE find(SIZE pos, const STRING * find) const {
		return findChars(ptr()+pos, getStringReadPtr(find));
	}
	SIZE findLast(SIZE pos, CHAR find) const {
		return findLastChar(ptr()+pos, find);
	}
	SIZE findLast(SIZE pos, const CHAR * find) const {
		return findLastChars(ptr()+pos, find);
	}
	SIZE findLast(SIZE pos, const STRING * find) const {
		return findLastChars(ptr()+pos, getStringReadPtr(find));
	}
	SIZE findCharSet(SIZE pos, const CHAR * find) const {
		return es::findCharSet(ptr()+pos, find);
	}
	operator STRING * () {
		return stringPtr();
	}
	operator const STRING * () const {
		return stringPtr();
	}
	STRING*	operator->() {
		return stringPtr();
	}
	const STRING* operator->() const {
		return stringPtr();
	}
	const CHAR & operator[](INT n) const {
		return ptr()[n];
	}
	CHAR &	operator[](INT n) {
		return ptr()[n];
	}
	const CHAR & operator[](SIZE n) const {
		return ptr()[n];
	}
	CHAR &	operator[](SIZE n) {
		return ptr()[n];
	}
	bool operator==(const CHAR * d) const {
		return equals(d);
	}
	bool operator==(const String &d) const {
		return equals(d);
	}
	bool operator!=(const CHAR * d) const {
		return !equals(d);
	}
	String	operator+(const STRING * add) const {
		String ret(stringPtr());
		ret.add(add);
		return ret;
	}
	String	operator+(const CHAR * add) const {
		String ret(stringPtr());
		ret.add(add);
		return ret;
	}
	String	operator+(CHAR add) const {
		String ret(stringPtr());
		ret.add(add);
		return ret;
	}
	String&	operator+=(const STRING * d) {
		add(d);
		return *this;
	}
	String&	operator+=(const CHAR * d) {
		add(d);
		return *this;
	}
	String&	operator+=(CHAR d) {
		add(d);
		return *this;
	}

protected:
	STRING s;
};

class StringPtr
{
protected:
	StringPtr();

public:
	StringPtr(STRING * d) {
		s = d;
	}
	~StringPtr() {
	}
	StringPtr& operator=(const StringPtr &d) {
		if( this != &d ) {
			copy(d);
		}
		return *this;
	}
	StringPtr& operator=(const CHAR *d) {
		if( !setChars(s, d) ) {
			throw Error();
		}
		return *this;
	}
	StringPtr& operator=(CHAR d) {
		CHAR v[2];
		v[0] = d;
		v[1] = 0;
		if( !setChars(s, v) ) {
			throw Error();
		}
		return *this;
	}
	StringPtr& operator=(const String &d) {
		if( this->s != d.stringPtr() ) {
			copy(d.stringPtr());
		}
		return *this;
	}
	void resize(SIZE size) {
		if( !resizeString(stringPtr(), size) )
			throw Error();
	}
	void clear() {
		clearString(stringPtr());
	}

	bool empty() const {
		return isEmptyString(stringPtr());
	}
	SIZE length() const {
		if( !empty() ) {
			return getCharsLength(getStringReadPtr(stringPtr()));
		}
		return 0;
	}
	SIZE capacity() const {
		return getStringCapacity(stringPtr());
	}
	SIZE size() const {
		return getStringSize(stringPtr());
	}
	const CHAR * ptr() const {
		return getStringReadPtr(stringPtr());
	}
	CHAR * ptr() {
		return getStringWritePtr(stringPtr());
	}
	const STRING * stringPtr() const {
		return s;
	}
	STRING * stringPtr() {
		return s;
	}
	CHAR at(SIZE pos) const {
		return ptr()[pos];
	}
	void set(const CHAR * set) {
		if( !setChars(stringPtr(), set) )
			throw Error();
	}
	void copy(const STRING * src) {
		if( !copyString(stringPtr(), src) )
			throw Error();
	}
	void add(const STRING * src) {
		if( !addString(stringPtr(), src) )
			throw Error();
	}
	void add(const CHAR * src) {
		if( !addChars(stringPtr(), src) )
			throw Error();
	}
	void add(CHAR src) {
		if( !addChar(stringPtr(), src) )
			throw Error();
	}
	String	crop(SIZE pos, SIZE size) const {
		String ret;
		if( !cropString(ret, stringPtr(), pos, size) )
			throw Error();
		return ret;
	}

	String	left(SIZE pos) const {
		return crop(0,pos);
	}
	String	right(SIZE pos) const {
		return crop(pos,size()-pos);
	}
	String	trimLeft(const CHAR *trimCharSet=NULL) const {
		String ret;
		if( !trimLeftString(ret, stringPtr(), trimCharSet) )
			throw Error();
		return ret;
	}
	String	trimRight(const CHAR *trimCharSet=NULL) const {
		String ret;
		if( !trimRightString(ret, stringPtr(), trimCharSet) )
			throw Error();
		return ret;
	}
	String	trim(const CHAR *trimCharSet=NULL) const {
		return trimLeft(trimCharSet).trimRight(trimCharSet);
	}
	String	upperCase() const {
		String ret(stringPtr());
		makeUpperCase(ret.ptr());
		return ret;
	}
	String	lowerCase() const {
		String ret(stringPtr());
		makeLowerCase(ret.ptr());
		return ret;
	}
	void format(const CHAR * format, ...) {
		va_list	ap;
		va_start(ap, format);
		clear();
		if( !formatStringVa(stringPtr(), format, ap) )
			throw Error();
	}
	bool equals(const CHAR * cmp) const {
		return isEqualChars(ptr(), cmp);
	}
	bool equals(const STRING * cmp) const {
		return isEqualString(stringPtr(), cmp);
	}
	bool equalsNocase(const CHAR * cmp) const {
		return isEqualCharsNoCase(ptr(), cmp);
	}
	bool equalsNocase(const STRING * cmp) const {
		return isEqualCharsNoCase(ptr(), getStringReadPtr(cmp));
	}
	INT compare(const CHAR * cmp) const {
		return compareChars(ptr(), cmp);
	}
	INT compare(const STRING * cmp) const {
		return compareChars(ptr(), getStringReadPtr(cmp));
	}
	INT compareNocase(const CHAR * cmp) const {
		return compareCharsNoCase(ptr(), cmp);
	}
	INT compareNocase(const STRING * cmp) const {
		return compareCharsNoCase(ptr(), getStringReadPtr(cmp));
	}

	SIZE find(SIZE pos, CHAR find) const {
		return findChar(ptr()+pos, find);
	}
	SIZE find(SIZE pos, const CHAR * find) const {
		return findChars(ptr()+pos, find);
	}
	SIZE find(SIZE pos, const STRING * find) const {
		return findChars(ptr()+pos, getStringReadPtr(find));
	}
	SIZE findLast(SIZE pos, CHAR find) const {
		return findLastChar(ptr()+pos, find);
	}
	SIZE findLast(SIZE pos, const CHAR * find) const {
		return findLastChars(ptr()+pos, find);
	}
	SIZE findLast(SIZE pos, const STRING * find) const {
		return findLastChars(ptr()+pos, getStringReadPtr(find));
	}
	SIZE findCharSet(SIZE pos, const CHAR * find) const {
		return es::findCharSet(ptr()+pos, find);
	}
	operator STRING * () {
		return stringPtr();
	}
	operator const STRING * () const {
		return stringPtr();
	}
	STRING*	operator->() {
		return stringPtr();
	}
	const STRING* operator->() const {
		return stringPtr();
	}
	const CHAR & operator[](INT n) const {
		return ptr()[n];
	}
	CHAR &	operator[](INT n) {
		return ptr()[n];
	}
	const CHAR & operator[](SIZE n) const {
		return ptr()[n];
	}
	CHAR &	operator[](SIZE n) {
		return ptr()[n];
	}
	bool operator==(const CHAR * d) const {
		return equals(d);
	}
	bool operator==(const String &d) const {
		return equals(d);
	}
	bool operator!=(const CHAR * d) const {
		return !equals(d);
	}
	String	operator+(const STRING * add) const {
		String ret(stringPtr());
		ret.add(add);
		return ret;
	}
	String	operator+(const CHAR * add) const {
		String ret(stringPtr());
		ret.add(add);
		return ret;
	}
	String	operator+(CHAR add) const {
		String ret(stringPtr());
		ret.add(add);
		return ret;
	}
	StringPtr&	operator+=(const STRING * d) {
		add(d);
		return *this;
	}
	StringPtr&	operator+=(const CHAR * d) {
		add(d);
		return *this;
	}
	StringPtr&	operator+=(CHAR d) {
		add(d);
		return *this;
	}

protected:
	STRING *s;
};

class StringArray
{
public:
	enum {
		BLOCK_SIZE = 10,
	};
	StringArray() {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	StringArray(INT size) {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
		while(size--) add((CHAR *)NULL);
	}
	StringArray(const StringArray &d) {
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
		copy(d);
	}
	~StringArray() {
		clear();
	}
	StringArray &operator=(const StringArray &d) {
		if( this != &d ) copy(d);
		return *this;
	}
	void copy(const StringArray &d) {
		if( this == &d ) return;
		clear();
		resize(d.size());
		SIZE i;
		for( i = 0; i < d.size(); i++ ) {
			copyString(&pNodes[i], &d.pNodes[i]);
		}
	}

	void add(const STRING * pString) {
		if( nNodes+1 > bNodes ) {
			SIZE bNodesNew = bNodes + BLOCK_SIZE;
			STRING * pNodesNew = (STRING *)resizeMemory(pNodes, sizeof(STRING)*bNodesNew);
			if( pNodesNew == NULL ) throw Error();
			for(SIZE i = bNodes; i < bNodesNew; i++ )
				initString(&pNodesNew[i]);
			pNodes	= pNodesNew;
			bNodes	= bNodesNew;
		}
		copyString(&pNodes[nNodes++], pString);
	}
	void add(const CHAR * pString) {
		String t(pString);
		return add(t);
	}
	void clear() {
		SIZE i;
		for( i = 0; i < nNodes; i++ ) {
			clearString(&pNodes[i]);
		}
		if( pNodes ) freeMemory(pNodes);
		nNodes	= 0;
		bNodes	= 0;
		pNodes	= NULL;
	}
	void resize(SIZE n) {
		SIZE i;
		for( i = n; i < nNodes; i++ ) {
			clearString(&pNodes[i]);
		}
		nNodes = n;
	}
	STRING * get(SIZE n) {
		return &pNodes[n];
	}
	SIZE	size() const {
		return nNodes;
	}
	SIZE	empty() const {
		return nNodes==0;
	}
	StringPtr operator[](INT index) {
		return StringPtr(&pNodes[index]);
	}
	const StringPtr operator[](INT index) const {
		return StringPtr(&pNodes[index]);
	}
	SIZE		nNodes;
	SIZE		bNodes;
	STRING *	pNodes;
};

} // es

#endif /* ES_STRING_H_ */
