/*
 * typeManager.cpp
 *
 *  Created on: 2012/01/06
 *      Author: tanaka
 */

#include "typeManager.h"
#include "symbol.h"
#include "variant.h"
#include "bytes.h"
#include "ref.h"

namespace es {

static Symbol symNull;
static Symbol symRef;
static Symbol symBool;
static Symbol symChar;
static Symbol symByte;
static Symbol symWord;
static Symbol symDWord;
static Symbol symShort;
static Symbol symInt;
static Symbol symDecimal;
static Symbol symFloat;
static Symbol symBytes;
static Symbol symChars;

TypeManager::TypeManager(IContext *ctx)
{
	this->ctx	= ctx;
	symNull = ES_T("null");
	symRef=ES_T("reference");
	symBool=ES_T("bool");
	symChar=ES_T("char");
	symByte=ES_T("byte");
	symWord=ES_T("word");
	symDWord=ES_T("dword");
	symShort=ES_T("short");
	symInt=ES_T("int");
	symDecimal=ES_T("decimal");
	symFloat=ES_T("float");
	symBytes=ES_T("bytes");
	symChars=ES_T("chars");
}
TypeManager::~TypeManager()
{

}

bool TypeManager::isReference(const VARIANT * v) const
{
	return !isPrimitive(v);
}

bool TypeManager::isPrimitive(const VARIANT * v) const
{
	switch( v->vt ) {
	case VT_NULL:
	case VT_BOOL:
	case VT_CHAR:
	case VT_BYTE:
	case VT_WORD:
	case VT_DWORD:
	case VT_SHORT:
	case VT_INT:
	case VT_DECIMAL:
	case VT_FLOAT:
	case VT_BYTES:
	case VT_STRING:
		return true;
	case VT_REF:
		return false;
	}
	return false;
}

VariantType TypeManager::isPrimitiveName(const CHAR *name) const
{
	VariantType vt = VT_NULL;
	if( isEqualSymbol(symBool, name) )			vt	= VT_BOOL;
	else if( isEqualSymbol(symChar,name) )		vt	= VT_CHAR;
	else if( isEqualSymbol(symByte,name) )		vt	= VT_BYTE;
	else if( isEqualSymbol(symWord,name) )		vt	= VT_WORD;
	else if( isEqualSymbol(symDWord,name) )		vt	= VT_DWORD;
	else if( isEqualSymbol(symShort,name) )		vt	= VT_SHORT;
	else if( isEqualSymbol(symInt,name) )		vt	= VT_INT;
	else if( isEqualSymbol(symDecimal,name) )	vt	= VT_DECIMAL;
	else if( isEqualSymbol(symFloat,name) )		vt	= VT_FLOAT;
	else if( isEqualSymbol(symBytes,name) )		vt	= VT_BYTES;
	else if( isEqualSymbol(symChars,name) )		vt	= VT_STRING;
	else if( isEqualSymbol(symRef,name) )		vt	= VT_REF;
	return vt;
}

VariantType TypeManager::isPrimitiveName(const SYMBOL *name) const
{
	VariantType vt = VT_NULL;
	if( isEqualSymbols(symBool, name) )			vt	= VT_BOOL;
	else if( isEqualSymbols(symChar,name) )		vt	= VT_CHAR;
	else if( isEqualSymbols(symByte,name) )		vt	= VT_BYTE;
	else if( isEqualSymbols(symWord,name) )		vt	= VT_WORD;
	else if( isEqualSymbols(symDWord,name) )	vt	= VT_DWORD;
	else if( isEqualSymbols(symShort,name) )	vt	= VT_SHORT;
	else if( isEqualSymbols(symInt,name) )		vt	= VT_INT;
	else if( isEqualSymbols(symDecimal,name) )	vt	= VT_DECIMAL;
	else if( isEqualSymbols(symFloat,name) )	vt	= VT_FLOAT;
	else if( isEqualSymbols(symBytes,name) )	vt	= VT_BYTES;
	else if( isEqualSymbols(symChars,name) )	vt	= VT_STRING;
	else if( isEqualSymbols(symRef,name) )		vt	= VT_REF;
	return vt;
}


SIZE TypeManager::getPrimitiveSize(VariantType vt) const
{
	switch( vt ) {
	case VT_NULL:		return sizeof(REF);
	case VT_BOOL:		return sizeof(bool);
	case VT_CHAR:		return sizeof(CHAR);
	case VT_BYTE:		return sizeof(BYTE);
	case VT_WORD:		return sizeof(WORD);
	case VT_DWORD:		return sizeof(DWORD);
	case VT_SHORT:		return sizeof(SHORT);
	case VT_INT:		return sizeof(INT);
	case VT_DECIMAL:	return sizeof(DECIMAL);
	case VT_FLOAT:		return sizeof(FLOAT);
	case VT_BYTES:		return sizeof(BYTES);
	case VT_STRING:		return sizeof(STRING);
	case VT_REF:		return sizeof(REF);
	}
	return 0;
}

const CHAR * TypeManager::getTypeName(VariantType vt) const
{
	const Symbol *sp = NULL;
	switch( vt ) {
	case VT_NULL:		sp = &symNull;	break;
	case VT_BOOL:		sp = &symBool;	break;
	case VT_CHAR:		sp = &symChar;	break;
	case VT_BYTE:		sp = &symByte;	break;
	case VT_WORD:		sp = &symWord;	break;
	case VT_DWORD:		sp = &symDWord;	break;
	case VT_SHORT:		sp = &symShort;	break;
	case VT_INT:		sp = &symInt;	break;
	case VT_DECIMAL:	sp = &symDecimal;	break;
	case VT_FLOAT:		sp = &symFloat;	break;
	case VT_BYTES:		sp = &symBytes;	break;
	case VT_STRING:		sp = &symChars;	break;
	case VT_REF:		sp = &symRef;	break;
	}
	if( sp ) return sp->ptr();
	return NULL;
}

bool TypeManager::normalize(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	Variant tmp;
	Variant res = t;
	while( res.isReference() && res->dRef.prop ) {
		const ATTR *attr = cm->findAttr(&res->dRef);
		if( attr->type == AT_PROPERTY ) {
			if( !es->getProperty(tmp, scope, &res->dRef) ) return false;
			res = tmp;
		} else if(  attr->type == AT_CLASS ||
					attr->type == AT_PACKAGE ||
					attr->type == AT_INTERFACE ||
					attr->type == AT_TEMPLATE ) {
			Ref r;
			if( !cm->normalizeReference(r, &res->dRef) ) return false;
			res = r;
		} else {
			break;
		}
	}
	VariantPtr r(v);
	r = res;
	return true;
}

bool TypeManager::changeType(VARIANT * v, const IScope *scope, VariantType type, const VARIANT * t) const
{
	if( type == VT_NULL ) {
		clearVariant(v);
		return true;
	}
	if( type == VT_REF ) {
		if( t->vt == VT_NULL ) {
			clearVariant(v);
			return true;
		}
		if( !toObject(v, scope, t) ) return false;
		return true;
	}

	switch(type) {
	case VT_BOOL:		return toBool(v, scope, t);
	case VT_CHAR:		return toChar(v, scope, t);
	case VT_BYTE:		return toByte(v, scope, t);
	case VT_WORD:		return toWord(v, scope, t);
	case VT_DWORD:		return toDWord(v, scope, t);
	case VT_SHORT:		return toShort(v, scope, t);
	case VT_INT:		return toInt(v, scope, t);
	case VT_DECIMAL:	return toDecimal(v, scope, t);
	case VT_FLOAT:		return toFloat(v, scope, t);
	case VT_BYTES:		return toBytes(v, scope, t);
	case VT_STRING:		return toString(v, scope, t);
	}
	es->throwException(scope, ERROR_TYPE, NULL);
	return false;
}

bool TypeManager::toBool(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	VariantPtr val(v);

	switch( t->vt ) {
	case VT_NULL:		val	= false;								break;
	case VT_REF:		val = objectToBool(scope, &v->dRef);		break;
	case VT_BOOL:		val	= t->dBool;								break;
	case VT_CHAR:		val = charToBool(scope, v->dChar);			break;
	case VT_BYTE:		val = byteToBool(scope, v->dByte);			break;
	case VT_WORD:		val = wordToBool(scope, v->dWord);			break;
	case VT_DWORD:		val = dwordToBool(scope, v->dDword);		break;
	case VT_SHORT:		val = shortToBool(scope, v->dShort);		break;
	case VT_INT:		val = intToBool(scope, v->dInt);			break;
	case VT_DECIMAL:	val = decimalToBool(scope, v->dDecimal);	break;
	case VT_FLOAT:		val = floatToBool(scope, v->dFloat);		break;
	case VT_BYTES:		val = bytesToBool(scope, &v->dBytes);		break;
	case VT_STRING:		val = stringToBool(scope, &v->dString);		break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

// bool
bool TypeManager::charToBool(const IScope *scope, CHAR d) const
{
	return d ? true : false;
}

bool TypeManager::byteToBool(const IScope *scope, BYTE d) const
{
	return d ? true : false;
}

bool TypeManager::wordToBool(const IScope *scope, WORD d) const
{
	return d ? true : false;
}

bool TypeManager::dwordToBool(const IScope *scope, DWORD d) const
{
	return d ? true : false;
}

bool TypeManager::shortToBool(const IScope *scope, SHORT d) const
{
	return d ? true : false;
}

bool TypeManager::intToBool(const IScope *scope, INT d) const
{
	return d ? true : false;
}

bool TypeManager::decimalToBool(const IScope *scope, DECIMAL d) const
{
	return d.dv != 0L ? true : false;
}

bool TypeManager::floatToBool(const IScope *scope, FLOAT d) const
{
	return d != 0.0 ? true : false;
}

bool TypeManager::bytesToBool(const IScope *scope, const BYTES * d) const
{
	WORD v = 0;
	SIZE n = getBytesSize(d);
	if( n != sizeof(v) && scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("bytes"), ES_T("bool") );
	}
	const BYTE *ptr = getBytesReadPtr(d);
	BYTE *dsp = (BYTE*)&v;
	for( SIZE i=0; i<sizeof(v) && n>0; i++, n-- ) {
		*dsp++ = *ptr++;
	}
	return v ? true : false;
}

bool TypeManager::stringToBool(const IScope *scope, const STRING * d) const
{
	return (d && !isEmptyString(d)) ? true : false;
}

bool TypeManager::objectToBool(const IScope *scope, const REF * d) const
{
	if( d->prop && d->prop->type == AT_PROPERTY && ((PROPERTYINF*)d->prop)->classPath ) {
		return (d && (d->object != NULL || d->cls != NULL)) ? true : false;
	}
	const Variant s = d;
	Variant p;
	if( !toPrimitive(p, scope, s) ) return false;
	Variant v;
	if( !toBool(v, scope, p) ) return false;
	return v->dBool;
}

bool TypeManager::toChar(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toChar(v, scope, p);
	}
	VariantPtr val(v);

	switch( t->vt ) {
	case VT_NULL:		val	= (CHAR)ES_T('0');						break;
	case VT_BOOL:		val	= boolToChar(scope, t->dBool);			break;
	case VT_CHAR:		val = t->dChar;								break;
	case VT_BYTE:		val = byteToChar(scope, t->dByte);			break;
	case VT_WORD:		val = wordToChar(scope, t->dWord);			break;
	case VT_DWORD:		val = dwordToChar(scope, t->dDword);		break;
	case VT_SHORT:		val = shortToChar(scope, t->dShort);		break;
	case VT_INT:		val = intToChar(scope, t->dInt);			break;
	case VT_DECIMAL:	val = decimalToChar(scope, t->dDecimal);	break;
	case VT_FLOAT:		val = floatToChar(scope, t->dFloat);		break;
	case VT_BYTES:		val = bytesToChar(scope, &t->dBytes);		break;
	case VT_STRING:		val = stringToChar(scope, &t->dString);		break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

CHAR TypeManager::boolToChar(const IScope *scope, bool d) const
{
	return d ? ES_T('y') : ES_T('n');
}

CHAR TypeManager::byteToChar(const IScope *scope, BYTE d) const
{
#ifdef ELISE_UNICODE
	return wordToChar(scope, byteToWord(scope, d));
#else
	return d;
#endif
}

CHAR TypeManager::wordToChar(const IScope *scope, WORD d) const
{
#ifdef ELISE_UNICODE
	return d;
#else
	return byteToChar(scope,wordToByte(scope,d));
#endif
}

CHAR TypeManager::dwordToChar(const IScope *scope, DWORD d) const
{
#ifdef ELISE_UNICODE
	return wordToChar(scope, dwordToWord(scope,d));
#else
	return byteToChar(scope, dwordToByte(scope,d));
#endif
}

CHAR TypeManager::shortToChar(const IScope *scope, SHORT d) const
{
#ifdef ELISE_UNICODE
	return wordToChar(scope, shortToWord(scope,d));
#else
	return byteToChar(scope, shortToByte(scope,d));
#endif
}

CHAR TypeManager::intToChar(const IScope *scope, INT d) const
{
#ifdef ELISE_UNICODE
	return wordToChar(scope, intToWord(scope,d));
#else
	return byteToChar(scope, intToByte(scope,d));
#endif
}

CHAR TypeManager::decimalToChar(const IScope *scope, DECIMAL d) const
{
#ifdef ELISE_UNICODE
	return wordToChar(scope, decimalToWord(scope,d));
#else
	return byteToChar(scope, decimalToByte(scope,d));
#endif
}

CHAR TypeManager::floatToChar(const IScope *scope, FLOAT d) const
{
#ifdef ELISE_UNICODE
	return wordToChar(scope, floatToWord(scope,d));
#else
	return byteToChar(scope, floatToByte(scope,d));
#endif
}

CHAR TypeManager::bytesToChar(const IScope *scope, const BYTES * d) const
{
	CHAR v = 0;
	SIZE n = getBytesSize(d);
	if( n != sizeof(v) && scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("bytes"), ES_T("char") );
	}
	const BYTE *ptr = getBytesReadPtr(d);
	BYTE *dsp = (BYTE*)&v;
	for( SIZE i=0; i<sizeof(v) && n>0; i++, n-- ) {
		*dsp++ = *ptr++;
	}
	return v;
}

CHAR TypeManager::stringToChar(const IScope *scope, const STRING * d) const
{
	return floatToChar(scope, stringToFloat(scope,d));
}

bool TypeManager::toByte(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toByte(v, scope, p);
	}
	VariantPtr val(v);

	switch( t->vt ) {
	case VT_NULL:		val	= (BYTE)0;								break;
	case VT_BOOL:		val	= boolToByte(scope, t->dBool);			break;
	case VT_CHAR:		val = charToByte(scope, t->dChar);			break;
	case VT_BYTE:		val = t->dByte;								break;
	case VT_WORD:		val = wordToByte(scope, t->dWord);			break;
	case VT_DWORD:		val = dwordToByte(scope, t->dDword);		break;
	case VT_SHORT:		val = shortToByte(scope, t->dShort);		break;
	case VT_INT:		val = intToByte(scope, t->dInt);			break;
	case VT_DECIMAL:	val = decimalToByte(scope, t->dDecimal);	break;
	case VT_FLOAT:		val = floatToByte(scope, t->dFloat);		break;
	case VT_BYTES:		val = bytesToByte(scope, &t->dBytes);		break;
	case VT_STRING:		val = stringToByte(scope, &t->dString);		break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

BYTE TypeManager::boolToByte(const IScope *scope, bool d) const
{
	return d ? 1 : 0;
}

BYTE TypeManager::charToByte(const IScope *scope, CHAR d) const
{
#ifdef ELISE_UNICODE
	return wordToByte(scope, charToWord(scope,d));
#else
	return d;
#endif
}

BYTE TypeManager::wordToByte(const IScope *scope, WORD d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATABITLOST, ES_T("word"), ES_T("byte") );
	}
	return (BYTE)d;
}

BYTE TypeManager::dwordToByte(const IScope *scope, DWORD d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATABITLOST, ES_T("dword"), ES_T("byte") );
	}
	return d;
}

BYTE TypeManager::shortToByte(const IScope *scope, SHORT d) const
{
	return wordToByte(scope, shortToWord(scope, d));
}

BYTE TypeManager::intToByte(const IScope *scope, INT d) const
{
	return dwordToByte(scope, intToDWord(scope, d));
}

BYTE TypeManager::decimalToByte(const IScope *scope, DECIMAL d) const
{
	return intToByte(scope, decimalToInt(scope, d));
}

BYTE TypeManager::floatToByte(const IScope *scope, FLOAT d) const
{
	return intToByte(scope, floatToInt(scope, d));
}

BYTE TypeManager::bytesToByte(const IScope *scope, const BYTES * d) const
{
	BYTE v = 0;
	SIZE n = getBytesSize(d);
	if( n != sizeof(v) && scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("bytes"), ES_T("byte") );
	}
	const BYTE *ptr = getBytesReadPtr(d);
	BYTE *dsp = (BYTE*)&v;
	for( SIZE i=0; i<sizeof(v) && n>0; i++, n-- ) {
		*dsp++ = *ptr++;
	}
	return v;
}

BYTE TypeManager::stringToByte(const IScope *scope, const STRING * d) const
{
	return floatToByte(scope, stringToFloat(scope,d));
}

bool TypeManager::toWord(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toWord(v, scope, p);
	}
	VariantPtr val(v);

	switch( t->vt ) {
	case VT_NULL:		val	= (WORD)0;								break;
	case VT_BOOL:		val	= boolToWord(scope, t->dBool);			break;
	case VT_CHAR:		val = charToWord(scope, t->dChar);			break;
	case VT_BYTE:		val = byteToWord(scope, t->dByte);			break;
	case VT_WORD:		val = t->dWord;			break;
	case VT_DWORD:		val = dwordToWord(scope, t->dDword);		break;
	case VT_SHORT:		val = shortToWord(scope, t->dShort);		break;
	case VT_INT:		val = intToWord(scope, t->dInt);			break;
	case VT_DECIMAL:	val = decimalToWord(scope, t->dDecimal);	break;
	case VT_FLOAT:		val = floatToWord(scope, t->dFloat);		break;
	case VT_BYTES:		val = bytesToWord(scope, &t->dBytes);		break;
	case VT_STRING:		val = stringToWord(scope, &t->dString);		break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

WORD TypeManager::boolToWord(const IScope *scope, bool d) const
{
	return d ? 1 : 0;
}

WORD TypeManager::charToWord(const IScope *scope, CHAR d) const
{
#ifdef ELISE_UNICODE
	return d;
#else
	return byteToChar(scope, charToByte(scope,d));
#endif
}

WORD TypeManager::byteToWord(const IScope *scope, BYTE d) const
{
	return d;
}

WORD TypeManager::dwordToWord(const IScope *scope, DWORD d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATABITLOST, ES_T("dword"), ES_T("word") );
	}
	return d;
}

WORD TypeManager::shortToWord(const IScope *scope, SHORT d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATASIGNLOST, ES_T("short"), ES_T("word") );
	}
	return d;

}

WORD TypeManager::intToWord(const IScope *scope, INT d) const
{
	return dwordToWord(scope, intToDWord(scope,d));
}

WORD TypeManager::decimalToWord(const IScope *scope, DECIMAL d) const
{
	return intToWord(scope, decimalToInt(scope,d));
}

WORD TypeManager::floatToWord(const IScope *scope, FLOAT d) const
{
	return intToWord(scope, floatToInt(scope,d));
}

WORD TypeManager::bytesToWord(const IScope *scope, const BYTES * d) const
{
	WORD v = 0;
	SIZE n = getBytesSize(d);
	if( n != sizeof(v) && scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("bytes"), ES_T("word") );
	}
	const BYTE *ptr = getBytesReadPtr(d);
	BYTE *dsp = (BYTE*)&v;
	for( SIZE i=0; i<sizeof(v) && n>0; i++, n-- ) {
		*dsp++ = *ptr++;
	}
	return v;
}

WORD TypeManager::stringToWord(const IScope *scope, const STRING * d) const
{
	return floatToWord(scope, stringToFloat(scope,d));
}

bool TypeManager::toDWord(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toDWord(v, scope, p);
	}
	VariantPtr val(v);

	switch( t->vt ) {
	case VT_NULL:		val	= (DWORD)0;								break;
	case VT_BOOL:		val	= boolToDWord(scope, t->dBool);			break;
	case VT_CHAR:		val = charToDWord(scope, t->dChar);			break;
	case VT_BYTE:		val = byteToDWord(scope, t->dByte);			break;
	case VT_WORD:		val = wordToDWord(scope, t->dWord);			break;
	case VT_DWORD:		val = t->dDword;		break;
	case VT_SHORT:		val = shortToDWord(scope, t->dShort);		break;
	case VT_INT:		val = intToDWord(scope, t->dInt);			break;
	case VT_DECIMAL:	val = decimalToDWord(scope, t->dDecimal);	break;
	case VT_FLOAT:		val = floatToDWord(scope, t->dFloat);		break;
	case VT_BYTES:		val = bytesToDWord(scope, &t->dBytes);		break;
	case VT_STRING:		val = stringToDWord(scope, &t->dString);	break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

DWORD TypeManager::boolToDWord(const IScope *scope, bool d) const
{
	return d ? 1 : 0;
}

DWORD TypeManager::charToDWord(const IScope *scope, CHAR d) const
{
#ifdef ELISE_UNICODE
	return wordToDWord(scope, charToWord(scope, d));
#else
	return byteToDWord(scope, charToByte(scope, d));
#endif
}

DWORD TypeManager::byteToDWord(const IScope *scope, BYTE d) const
{
	return d;
}

DWORD TypeManager::wordToDWord(const IScope *scope, WORD d) const
{
	return d;
}

DWORD TypeManager::shortToDWord(const IScope *scope, SHORT d) const
{
	return wordToDWord(scope, shortToWord(scope,d));
}

DWORD TypeManager::intToDWord(const IScope *scope, INT d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATASIGNLOST, ES_T("int"), ES_T("dword") );
	}
	return d;
}

DWORD TypeManager::decimalToDWord(const IScope *scope, DECIMAL d) const
{
	return intToDWord(scope, decimalToInt(scope,d));
}

DWORD TypeManager::floatToDWord(const IScope *scope, FLOAT d) const
{
	return intToDWord(scope, floatToInt(scope,d));
}

DWORD TypeManager::bytesToDWord(const IScope *scope, const BYTES * d) const
{
	DWORD v = 0;
	SIZE n = getBytesSize(d);
	if( n != sizeof(v) && scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("bytes"), ES_T("dword") );
	}
	const BYTE *ptr = getBytesReadPtr(d);
	BYTE *dsp = (BYTE*)&v;
	for( SIZE i=0; i<sizeof(v) && n>0; i++, n-- ) {
		*dsp++ = *ptr++;
	}
	return v;
}

DWORD TypeManager::stringToDWord(const IScope *scope, const STRING * d) const
{
	return floatToDWord(scope, stringToFloat(scope,d));
}

bool TypeManager::toShort(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toShort(v, scope, p);
	}
	VariantPtr val(v);

	switch( t->vt ) {
	case VT_NULL:		val	= (SHORT)0;								break;
	case VT_BOOL:		val	= boolToShort(scope, t->dBool);			break;
	case VT_CHAR:		val = charToShort(scope, t->dChar);			break;
	case VT_BYTE:		val = byteToShort(scope, t->dByte);			break;
	case VT_WORD:		val = wordToShort(scope, t->dWord);			break;
	case VT_DWORD:		val = dwordToShort(scope, t->dDword);		break;
	case VT_SHORT:		val = t->dShort;		break;
	case VT_INT:		val = intToShort(scope, t->dInt);			break;
	case VT_DECIMAL:	val = decimalToShort(scope, t->dDecimal);	break;
	case VT_FLOAT:		val = floatToShort(scope, t->dFloat);		break;
	case VT_BYTES:		val = bytesToShort(scope, &t->dBytes);		break;
	case VT_STRING:		val = stringToShort(scope, &t->dString);	break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

SHORT TypeManager::boolToShort(const IScope *scope, bool d) const
{
	return d ? 1 : 0;
}

SHORT TypeManager::charToShort(const IScope *scope, CHAR d) const
{
#ifdef ELISE_UNICODE
	return wordToShort(scope, charToWord(scope, d));
#else
	return byteToShort(scope, charToByte(scope, d));
#endif
}

SHORT TypeManager::byteToShort(const IScope *scope, BYTE d) const
{
	return d;
}

SHORT TypeManager::wordToShort(const IScope *scope, WORD d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("word"), ES_T("short") );
	}
	return d;
}

SHORT TypeManager::dwordToShort(const IScope *scope, DWORD d) const
{
	return intToShort(scope, dwordToInt(scope,d));
}

SHORT TypeManager::intToShort(const IScope *scope, INT d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("int"), ES_T("short") );
	}
	return d;
}

SHORT TypeManager::decimalToShort(const IScope *scope, DECIMAL d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("decimal"), ES_T("short") );
	}
	return es::decimalToInt(d);
}

SHORT TypeManager::floatToShort(const IScope *scope, FLOAT d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("float"), ES_T("short") );
	}
	return (BYTE)d;
}

SHORT TypeManager::bytesToShort(const IScope *scope, const BYTES * d) const
{
	SHORT v = 0;
	SIZE n = getBytesSize(d);
	if( n != sizeof(v) && scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("bytes"), ES_T("short") );
	}
	const BYTE *ptr = getBytesReadPtr(d);
	BYTE *dsp = (BYTE*)&v;
	for( SIZE i=0; i<sizeof(v) && n>0; i++, n-- ) {
		*dsp++ = *ptr++;
	}
	return v;
}

SHORT TypeManager::stringToShort(const IScope *scope, const STRING * d) const
{
	return floatToShort(scope, stringToFloat(scope,d));
}


bool TypeManager::toInt(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toInt(v, scope, p);
	}
	VariantPtr val(v);

	switch( t->vt ) {
	case VT_NULL:		val	= (INT)0;							break;
	case VT_BOOL:		val	= boolToInt(scope, t->dBool);		break;
	case VT_CHAR:		val = charToInt(scope, t->dChar);		break;
	case VT_BYTE:		val = byteToInt(scope, t->dByte);		break;
	case VT_WORD:		val = wordToInt(scope, t->dWord);		break;
	case VT_DWORD:		val = dwordToInt(scope, t->dDword);		break;
	case VT_SHORT:		val = shortToInt(scope, t->dShort);		break;
	case VT_INT:		val = t->dInt;			break;
	case VT_DECIMAL:	val = decimalToInt(scope, t->dDecimal);	break;
	case VT_FLOAT:		val = floatToInt(scope, t->dFloat);		break;
	case VT_BYTES:		val = bytesToInt(scope, &t->dBytes);	break;
	case VT_STRING:		val = stringToInt(scope, &t->dString);	break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

INT TypeManager::boolToInt(const IScope *scope, bool d) const
{
	return d ? 1 : 0;
}

INT TypeManager::charToInt(const IScope *scope, CHAR d) const
{
	return d;
}

INT TypeManager::byteToInt(const IScope *scope, BYTE d) const
{
	return dwordToInt(scope, byteToDWord(scope,d));
}

INT TypeManager::wordToInt(const IScope *scope, WORD d) const
{
	return dwordToInt(scope, wordToDWord(scope,d));
}

INT TypeManager::dwordToInt(const IScope *scope, DWORD d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("dword"), ES_T("int") );
	}
	return d;
}

INT TypeManager::shortToInt(const IScope *scope, SHORT d) const
{
	return d;
}

INT TypeManager::decimalToInt(const IScope *scope, DECIMAL d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("decimal"), ES_T("int") );
	}
	return  es::decimalToInt(d);
}

INT TypeManager::floatToInt(const IScope *scope, FLOAT d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("float"), ES_T("int") );
	}
	return (BYTE)d;
}

INT TypeManager::bytesToInt(const IScope *scope, const BYTES * d) const
{
	INT v = 0;
	SIZE n = getBytesSize(d);
	if( n != sizeof(v) && scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("bytes"), ES_T("int") );
	}
	const BYTE *ptr = getBytesReadPtr(d);
	BYTE *dsp = (BYTE*)&v;
	for( SIZE i=0; i<sizeof(v) && n>0; i++, n-- ) {
		*dsp++ = *ptr++;
	}
	return v;
}

INT TypeManager::stringToInt(const IScope *scope, const STRING * d) const
{
	return floatToInt(scope, stringToFloat(scope,d));
}

bool TypeManager::toDecimal(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toDecimal(v, scope, p);
	}
	VariantPtr val(v);
	DECIMAL dv = { 0 };
	switch( t->vt ) {
	case VT_NULL:		val	= dv;							break;
	case VT_BOOL:		val	= boolToDecimal(scope, t->dBool);		break;
	case VT_CHAR:		val = charToDecimal(scope, t->dChar);		break;
	case VT_BYTE:		val = byteToDecimal(scope, t->dByte);		break;
	case VT_WORD:		val = wordToDecimal(scope, t->dWord);		break;
	case VT_DWORD:		val = dwordToDecimal(scope, t->dDword);		break;
	case VT_SHORT:		val = shortToDecimal(scope, t->dShort);		break;
	case VT_INT:		val = intToDecimal(scope, t->dInt);			break;
	case VT_DECIMAL:	val = t->dDecimal;	break;
	case VT_FLOAT:		val = floatToDecimal(scope, t->dFloat);		break;
	case VT_BYTES:		val = bytesToDecimal(scope, &t->dBytes);	break;
	case VT_STRING:		val = stringToDecimal(scope, &t->dString);	break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

DECIMAL TypeManager::boolToDecimal(const IScope *scope, bool d) const
{
	DECIMAL v;
	v = es::intToDecimal(boolToInt(scope,d));
	return v;
}

DECIMAL TypeManager::charToDecimal(const IScope *scope, CHAR d) const
{
	DECIMAL v;
	v = es::intToDecimal(charToInt(scope,d));
	return v;
}

DECIMAL TypeManager::byteToDecimal(const IScope *scope, BYTE d) const
{
	DECIMAL v;
	v = es::intToDecimal(byteToInt(scope,d));
	return v;
}

DECIMAL TypeManager::wordToDecimal(const IScope *scope, WORD d) const
{
	DECIMAL v;
	v = es::intToDecimal(wordToInt(scope,d));
	return v;
}

DECIMAL TypeManager::dwordToDecimal(const IScope *scope, DWORD d) const
{
	DECIMAL v;
	v = es::intToDecimal(dwordToInt(scope,d));
	return v;
}

DECIMAL TypeManager::shortToDecimal(const IScope *scope, SHORT d) const
{
	DECIMAL v;
	v = es::intToDecimal(d);
	return v;
}

DECIMAL TypeManager::intToDecimal(const IScope *scope, INT d) const
{
	DECIMAL v;
	v = es::intToDecimal(d);
	return v;
}

DECIMAL TypeManager::floatToDecimal(const IScope *scope, FLOAT d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("float"), ES_T("decimal") );
	}
	DECIMAL v = es::floatToDecimal(d);
	return v;
}

DECIMAL TypeManager::bytesToDecimal(const IScope *scope, const BYTES * d) const
{
	DECIMAL v = {0};
	SIZE n = getBytesSize(d);
	if( n != sizeof(v) && scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("bytes"), ES_T("decimal") );
	}
	const BYTE *ptr = getBytesReadPtr(d);
	BYTE *dsp = (BYTE*)&v;
	for( SIZE i=0; i<sizeof(v) && n>0; i++, n-- ) {
		*dsp++ = *ptr++;
	}
	return v;
}

DECIMAL TypeManager::stringToDecimal(const IScope *scope, const STRING * d) const
{
	return es::floatToDecimal(stringToFloat(scope, d));
}

bool TypeManager::toFloat(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toFloat(v, scope, p);
	}
	VariantPtr val(v);

	switch( t->vt ) {
	case VT_NULL:		val	= (FLOAT)0.0;							break;
	case VT_BOOL:		val	= boolToFloat(scope, t->dBool);			break;
	case VT_CHAR:		val = charToFloat(scope, t->dChar);			break;
	case VT_BYTE:		val = byteToFloat(scope, t->dByte);			break;
	case VT_WORD:		val = wordToFloat(scope, t->dWord);			break;
	case VT_DWORD:		val = dwordToFloat(scope, t->dDword);		break;
	case VT_SHORT:		val = shortToFloat(scope, t->dShort);		break;
	case VT_INT:		val = intToFloat(scope, t->dInt);			break;
	case VT_DECIMAL:	val = decimalToFloat(scope, t->dDecimal);	break;
	case VT_FLOAT:		val = t->dFloat;		break;
	case VT_BYTES:		val = bytesToFloat(scope, &t->dBytes);		break;
	case VT_STRING:		val = stringToFloat(scope, &t->dString);	break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	return true;
}

FLOAT TypeManager::boolToFloat(const IScope *scope, bool d) const
{
	return d ? 1.0 : 0.0;
}

FLOAT TypeManager::charToFloat(const IScope *scope, CHAR d) const
{
	return d;
}

FLOAT TypeManager::byteToFloat(const IScope *scope, BYTE d) const
{
	return d;
}

FLOAT TypeManager::wordToFloat(const IScope *scope, WORD d) const
{
	return d;
}

FLOAT TypeManager::dwordToFloat(const IScope *scope, DWORD d) const
{
	return d;
}

FLOAT TypeManager::shortToFloat(const IScope *scope, SHORT d) const
{
	return d;
}

FLOAT TypeManager::intToFloat(const IScope *scope, INT d) const
{
	return d;
}

FLOAT TypeManager::decimalToFloat(const IScope *scope, DECIMAL d) const
{
	if( scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("decimal"), ES_T("float") );
	}
	return es::decimalToFloat(d);
}

FLOAT TypeManager::bytesToFloat(const IScope *scope, const BYTES * d) const
{
	FLOAT v = 0;
	SIZE n = getBytesSize(d);
	if( n != sizeof(v) && scope ) {
		IReporter *rep = scope->getContext()->getReporter();
		if( rep ) rep->report(NOTE_DATALOST, ES_T("bytes"), ES_T("float") );
	}
	const BYTE *ptr = getBytesReadPtr(d);
	BYTE *dsp = (BYTE*)&v;
	for( SIZE i=0; i<sizeof(v) && n>0; i++, n-- ) {
		*dsp++ = *ptr++;
	}
	return v;
}

FLOAT TypeManager::stringToFloat(const IScope *scope, const STRING * d) const
{
	FLOAT vi=0;
	FLOAT vf=0;
	FLOAT vp=1;
	const CHAR *p = getStringReadPtr(d);
	if( p ) {
		while(*p) {
			if( *p < ES_T('0') || ES_T('9') < *p ) break;
			vi = vi * 10 + (*p - ES_T('0'));
			p++;
		}
		if( *p == ES_T('.') ) {
			p++;
			while(*p) {
				if( *p < ES_T('0') || ES_T('9') < *p ) break;
				vf = vf * 10 + (*p - ES_T('0'));
				vp *= 10;
				p++;
			}
		}
		if( *p  && scope ) {
			IReporter *rep = scope->getContext()->getReporter();
			if( rep ) rep->report(NOTE_DATALOST, ES_T("string"), ES_T("float") );
		}
	}
	return vi+vf/vp;
}

bool TypeManager::toBytes(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toBytes(v, scope, p);
	}
	VariantPtr val(v);
	Bytes bv;
	bool r=false;
	switch( t->vt ) {
	case VT_NULL:		r = true;	val.clear();					break;
	case VT_BOOL:		r = boolToBytes(scope, t->dBool, bv);		break;
	case VT_CHAR:		r = toBytes(sizeof(t->dChar),	(const BYTE*)&t->dChar, bv);	break;
	case VT_BYTE:		r = toBytes(sizeof(t->dByte),	(const BYTE*)&t->dByte, bv);	break;
	case VT_WORD:		r = toBytes(sizeof(t->dWord),	(const BYTE*)&t->dWord, bv);	break;
	case VT_DWORD:		r = toBytes(sizeof(t->dDword),	(const BYTE*)&t->dDword, bv);	break;
	case VT_SHORT:		r = toBytes(sizeof(t->dShort),	(const BYTE*)&t->dShort, bv);	break;
	case VT_INT:		r = toBytes(sizeof(t->dInt),	(const BYTE*)&t->dInt, bv);		break;
	case VT_DECIMAL:	r = toBytes(sizeof(t->dDecimal),(const BYTE*)&t->dDecimal, bv);	break;
	case VT_FLOAT:		r = toBytes(sizeof(t->dFloat),	(const BYTE*)&t->dFloat, bv);	break;
	case VT_BYTES:		r = true; bv = &t->dBytes;		break;
	case VT_STRING:		r = stringToBytes(scope, &t->dString, bv);	break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	if( !r ) return false;
	val = bv;
	return true;
}

bool TypeManager::toBytes(SIZE size, const BYTE *d, BYTES * ptr) const
{
	if( !resizeBytes(ptr, size) ) return false;
	BYTE *p = getBytesWritePtr(ptr);
	while(size) {
		*p++ = *d++;
		size--;
	}
	return true;
}

bool TypeManager::boolToBytes(const IScope *scope, bool d, BYTES * ptr) const
{
	WORD v = d ? 1 : 0;
	return toBytes(sizeof(v), (const BYTE*)&v, ptr);
}

bool TypeManager::stringToBytes(const IScope *scope, const STRING * d, BYTES * ptr) const
{
	return copyBytes(ptr, (const BYTES*)d);
}

bool TypeManager::toString(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	if( !isPrimitive(t) ) {
		Variant p;
		if( !toPrimitive(p, scope, t) ) return false;
		return toString(v, scope, p);
	}
	VariantPtr val(v);
	String bv;
	bool r=false;
	switch( t->vt ) {
	case VT_NULL:		r = true;	bv = ES_T("null");				break;
	case VT_BOOL:		r = boolToString(scope, t->dBool, bv);		break;
	case VT_CHAR:		r = charToString(scope, t->dChar, bv);		break;
	case VT_BYTE:		r = byteToString(scope, t->dByte, bv);		break;
	case VT_WORD:		r = wordToString(scope, t->dWord, bv);		break;
	case VT_DWORD:		r = dwordToString(scope, t->dDword, bv);	break;
	case VT_SHORT:		r = shortToString(scope, t->dShort, bv);	break;
	case VT_INT:		r = intToString(scope, t->dInt, bv);		break;
	case VT_DECIMAL:	r = decimalToString(scope, t->dDecimal, bv);break;
	case VT_FLOAT:		r = floatToString(scope, t->dFloat, bv);	break;
	case VT_BYTES:		r = bytesToString(scope, &t->dBytes, bv);	break;
	case VT_STRING:		r = true; bv = t->dString;					break;
	default:
		es->throwException(scope, ERROR_TYPE, NULL);
		return false;
	}
	val = bv;
	return true;
}

bool TypeManager::boolToString(const IScope *scope, bool d, STRING * ptr) const
{
	return es::boolToString(ptr, d);
}

bool TypeManager::charToString(const IScope *scope, CHAR d, STRING * ptr) const
{
	CHAR v[2];
	v[0] = d;
	v[1] = 0;
	return setChars(ptr, v);
}

bool TypeManager::byteToString(const IScope *scope, BYTE d, STRING * ptr) const
{
	return es::byteToString(ptr, d);
}

bool TypeManager::wordToString(const IScope *scope, WORD d, STRING * ptr) const
{
	return es::wordToString(ptr, d);
}

bool TypeManager::dwordToString(const IScope *scope, DWORD d, STRING * ptr) const
{
	return es::dwordToString(ptr, d);
}

bool TypeManager::shortToString(const IScope *scope, SHORT d, STRING * ptr) const
{
	return es::shortToString(ptr, d);
}

bool TypeManager::intToString(const IScope *scope, INT d, STRING * ptr) const
{
	return es::intToString(ptr, d);
}

bool TypeManager::decimalToString(const IScope *scope, DECIMAL d, STRING * ptr) const
{
	return es::decimalToString(ptr, d);
}

bool TypeManager::floatToString(const IScope *scope, FLOAT d, STRING * ptr) const
{
	return es::floatToString(ptr, d);
}

bool TypeManager::bytesToString(const IScope *scope, const BYTES * d, STRING * ptr) const
{
	return es::bytesToString(ptr, *d);
}

bool TypeManager::toPrimitive(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	return true;
}

bool TypeManager::toPrimitive(VARIANT * v, const IScope *scope, const REF *t ) const
{
	return true;
}

bool TypeManager::toObject(VARIANT * v, const IScope *scope, const VARIANT * t) const
{
	return true;
}


} // es
