/*
 * variant.h
 *
 *  Created on: 2012/01/10
 *      Author: tanaka
 */

#ifndef VARIANT_H_
#define VARIANT_H_

#include "../include/elise.h"
#include "error.h"
#include "string.h"

namespace es {
class Variant {
public:
	Variant() {
		v.vt		= VT_NULL;
	}
	Variant(const Variant &d) {
		copyVariant(&v, d);
	}
	Variant(const VARIANT *d) {
		copyVariant(&v, d);
	}
	Variant(bool d) {
		v.vt = VT_BOOL;
		v.dBool	= d;
	}
	Variant(CHAR d) {
		v.vt = VT_CHAR;
		v.dChar = d;
	}
	Variant(BYTE d) {
		v.vt		= VT_BYTE;
		v.dByte		= d;
	}
	Variant(WORD d) {
		v.vt		= VT_WORD;
		v.dWord		= d;
	}
	Variant(DWORD d) {
		v.vt		= VT_DWORD;
		v.dDword	= d;
	}
	Variant(SHORT d) {
		v.vt		= VT_SHORT;
		v.dShort	= d;
	}
	Variant(INT d) {
		v.vt		= VT_INT;
		v.dInt		= d;
	}
	Variant(DECIMAL d) {
		v.vt		= VT_DECIMAL;
		v.dDecimal	= d;
	}
	Variant(FLOAT d) {
		v.vt		= VT_FLOAT;
		v.dFloat	= d;
	}
	Variant(const BYTES *d) {
		v.vt	= VT_NULL;
		setVariantBytes(&v, d);
	}
	Variant(const CHAR *d) {
		v.vt	= VT_NULL;
		setVariantChars(&v, d);
	}
	Variant(const STRING *d) {
		v.vt	= VT_NULL;
		setVariantString(&v, d);
	}
	Variant(const REF *d) {
		v.vt	= VT_NULL;
		if( !setVariantReference(&v, d) ) {
			throw Error();
		}
	}
	~Variant() {
		clearVariant(&v);
	}
	Variant &operator=(const Variant &d) {
		if( this != &d ) {
			copyVariant(&v, &d.v);
		}
		return *this;
	}

	void clear() {
		clearVariant(&v);
	}
	void copy(const VARIANT *d) {
		copyVariant(&v, d);
	}
	Variant &copy(const Variant &d) {
		copyVariant(&v, &d.v);
		return *this;
	}
	bool isNull() const {
		return v.vt == VT_NULL;
	}
	bool isReference() const {
		return v.vt == VT_REF;
	}
	bool isPrimitive() const {
		return v.vt != VT_REF;
	}
	operator VARIANT *() {
		return &v;
	}
	operator const VARIANT *() const {
		return &v;
	}
	VARIANT * operator ->() {
		return &v;
	}
	const VARIANT * operator ->() const {
		return &v;
	}
	Variant &operator=(bool d) {
		clearVariant(&v);
		v.vt		= VT_BOOL;
		v.dBool		= d;
		return *this;
	}
	Variant &operator=(CHAR d) {
		clearVariant(&v);
		v.vt		= VT_CHAR;
		v.dChar		= d;
		return *this;
	}
	Variant &operator=(BYTE d) {
		clearVariant(&v);
		v.vt		= VT_BYTE;
		v.dByte		= d;
		return *this;
	}
	Variant &operator=(WORD d) {
		clearVariant(&v);
		v.vt		= VT_WORD;
		v.dWord		= d;
		return *this;
	}
	Variant &operator=(DWORD d) {
		clearVariant(&v);
		v.vt		= VT_DWORD;
		v.dDword	= d;
		return *this;
	}
	Variant &operator=(SHORT d) {
		clearVariant(&v);
		v.vt		= VT_SHORT;
		v.dShort	= d;
		return *this;
	}
	Variant &operator=(INT d) {
		clearVariant(&v);
		v.vt		= VT_INT;
		v.dInt		= d;
		return *this;
	}
	Variant &operator=(DECIMAL d) {
		clearVariant(&v);
		v.vt		= VT_DECIMAL;
		v.dDecimal	= d;
		return *this;
	}
	Variant &operator=(FLOAT d) {
		clearVariant(&v);
		v.vt		= VT_FLOAT;
		v.dFloat	= d;
		return *this;
	}
	Variant &operator=(const BYTES *d) {
		setVariantBytes(&v, d);
		return *this;
	}
	Variant &operator=(const CHAR *d) {
		setVariantChars(&v, d);
		return *this;
	}
	Variant &operator=(const STRING *d) {
		setVariantString(&v, d);
		return *this;
	}
	Variant &operator=(const REF *d) {
		if( !setVariantReference(&v, d) ) {
			throw Error();
		}
		return *this;
	}
	const Variant &operator=(const VARIANT *d) {
		if( !copyVariant(&v, d) ) {
			throw Error();
		}
		return *this;
	}
	Variant &operator=(VARIANT *d) {
		if( !copyVariant(&v, d) ) {
			throw Error();
		}
		return *this;
	}

protected:
	VARIANT 	v;
};

class VariantPtr {
public:
	~VariantPtr() {
	}

private:
	VariantPtr();

public:
	VariantPtr(VARIANT *d) {
		v = d;
	}
	bool isNull() const {
		return v == NULL;
	}
	void init() {
		initVariant(v);
	}
	void clear() {
		clearVariant(v);
	}
	void copy(const VARIANT *d) {
		if( !copyVariant(v, d) )
			throw Error();
	}
	bool	isReference() const {
		return (v && v->vt == VT_REF);
	}
	bool	isPrimitive() const {
		return (!v || v->vt != VT_REF);
	}
	operator VARIANT *() {
		return v;
	}
	operator const VARIANT *() const {
		return v;
	}
	VARIANT *operator ->() {
		return v;
	}
	const VARIANT *operator ->() const {
		return v;
	}

	VariantPtr &operator=(bool d) {
		clearVariant(v);
		v->vt		= VT_BOOL;
		v->dBool	= d;
		return *this;
	}
	VariantPtr &operator=(CHAR d) {
		clearVariant(v);
		v->vt		= VT_CHAR;
		v->dChar	= d;
		return *this;
	}
	VariantPtr &operator=(BYTE d) {
		clearVariant(v);
		v->vt		= VT_BYTE;
		v->dByte	= d;
		return *this;
	}
	VariantPtr &operator=(WORD d) {
		clearVariant(v);
		v->vt		= VT_WORD;
		v->dWord	= d;
		return *this;
	}
	VariantPtr &operator=(DWORD d) {
		clearVariant(v);
		v->vt		= VT_DWORD;
		v->dDword	= d;
		return *this;
	}
	VariantPtr &operator=(SHORT d) {
		clearVariant(v);
		v->vt		= VT_SHORT;
		v->dShort	= d;
		return *this;
	}
	VariantPtr &operator=(INT d) {
		clearVariant(v);
		v->vt		= VT_INT;
		v->dInt		= d;
		return *this;
	}
	VariantPtr &operator=(DECIMAL d) {
		clearVariant(v);
		v->vt		= VT_DECIMAL;
		v->dDecimal	= d;
		return *this;
	}
	VariantPtr &operator=(FLOAT d) {
		clearVariant(v);
		v->vt		= VT_FLOAT;
		v->dFloat	= d;
		return *this;
	}
	VariantPtr &operator=(const BYTES *d) {
		if( !setVariantBytes(v, d) ) {
			throw Error();
		}
		return *this;
	}
	VariantPtr &operator=(const CHAR *d) {
		if( !setVariantChars(v, d) ) {
			throw Error();
		}
		return *this;
	}
	VariantPtr &operator=(const STRING *d) {
		if( !setVariantString(v, d) ) {
			throw Error();
		}
		return *this;
	}
	VariantPtr &operator=(REF *d) {
		if( !setVariantReference(v, d) ) {
			throw Error();
		}
		return *this;
	}
	const VariantPtr &operator=(const REF *d) {
		if( !setVariantReference(v, d) ) {
			throw Error();
		}
		return *this;
	}
	VariantPtr &operator=(const VariantPtr &d) {
		if( !copyVariant(v, d.v) )
			throw Error();
		return *this;
	}
	VariantPtr &operator=(VARIANT *d) {
		if( !copyVariant(v, d) )
			throw Error();
		return *this;
	}
	const VariantPtr &operator=(const VARIANT *d) {
		if( !copyVariant(v, d) )
			throw Error();
		return *this;
	}
protected:
	VARIANT * 	v;
};

class VariantArray
{
public:
	VariantArray() {
		bNode	= 0;
		nNode	= 0;
		pNode	= NULL;
	}
	VariantArray(SIZE size) {
		bNode	= size;
		nNode	= size;
		pNode	= static_cast<VARIANT*>(allocMemory(sizeof(VARIANT)*size));
		SIZE p;
		for( p = 0; p < nNode; p++ ) {
			initVariant(&pNode[p]);
		}
	}
	VariantArray(const VariantArray &d) {
		bNode	= 0;
		nNode	= 0;
		pNode	= NULL;
		copy(d);
	}
	~VariantArray() {
		clear();
	}
	void	clear() {
		SIZE p;
		for( p = 0; p < nNode; p++ ) {
			clearVariant(&pNode[p]);
		}
		freeMemory(pNode);
		bNode	= 0;
		nNode	= 0;
		pNode	= NULL;
	}
	void add(const VARIANT *p) {
		if( nNode + 1 >= bNode ) {
			SIZE bNew = bNode + 8;
			VARIANT *pNew = (VARIANT*)resizeMemory(pNode, sizeof(VARIANT)*bNew);
			if( !pNew ) {
				throw Error();
			}
			fillMemory((BYTE*)&pNew[nNode], 0, sizeof(VARIANT)*(bNew-nNode));
			pNode	= pNew;
			bNode	= bNew;
		}
		VariantPtr t(&pNode[nNode]);
		t = p;
		nNode++;
	}
	void push(const VARIANT *p) {
		add(p);
	}
	void push(SIZE n, const VARIANT *p) {
		if( nNode + n >= bNode ) {
			SIZE bNew = bNode + (n/8+1)*8;
			VARIANT * pNew = (VARIANT*)resizeMemory(pNode, sizeof(VARIANT)*bNew);
			if( !pNew ) throw Error();
			fillMemory((BYTE*)&pNew[nNode], 0, sizeof(VARIANT)*(bNew-nNode));
			pNode	= pNew;
			bNode	= bNew;
		}
		for(SIZE i=0; i<n; i++ ) {
			VariantPtr t(&pNode[nNode]);
			t = p++;
			nNode++;
		}
	}
	bool pop(VARIANT *p) {
		if( nNode == 0 ) return false;
		VariantPtr r(p);
		nNode--;
		r	= &pNode[nNode];
		clearVariant(&pNode[nNode]);
		return true;
	}
	void erase(INT index) {
		clearVariant(&pNode[index]);
		SIZE i;
		for( i = index; i < nNode-1; i++ ) {
			pNode[i] = pNode[i+1];
		}
		initVariant(&pNode[i]);
		nNode--;
	}
	VariantPtr get(SIZE index) const{
		return VariantPtr(&pNode[index]);
	}
	SIZE size() const {
		return nNode;
	}
	void resize(SIZE nNewSize) {
		if( nNewSize > nNode ) {
			if( nNewSize > bNode ) {
				SIZE bNew = nNewSize;
				VARIANT *pNew = (VARIANT*)resizeMemory(pNode, sizeof(VARIANT)*bNew);
				if( !pNew ) throw Error();
				fillMemory((BYTE*)&pNew[nNode], 0, sizeof(VARIANT)*(bNew-nNode));
				pNode	= pNew;
				bNode	= bNew;
			}
		} else if( nNewSize < nNode ) {
			for(SIZE i = nNewSize; i < nNode; i++ ) {
				clearVariant(&pNode[i]);
			}
		}
		nNode	= nNewSize;
	}
	VariantPtr operator[](INT index) {
		return VariantPtr(&pNode[index]);
	}
	operator const VARIANT *() const {
		return pNode;
	}
	VariantArray &operator=(const VariantArray &d) {
		copy(d);
		return *this;
	}
	void copy(const VariantArray &d) {
		if( this != &d ) {
			clear();
			pNode	= (VARIANT*)resizeMemory(pNode, sizeof(VARIANT)*d.bNode);
			if( !pNode ) throw Error();
			bNode	= d.bNode;
			nNode	= d.nNode;
			for( SIZE i = 0; i < nNode; i++ ) {
				if( !copyVariant(&pNode[i], &d.pNode[i]) ) throw Error();
			}
		}
	}

protected:
	SIZE		bNode;
	SIZE		nNode;
	VARIANT	*	pNode;
};

class VariantMap
{
	class Node {
	public:
		Node() {
		}
		~Node() {
		}
		String	key;
		Variant	val;
	};
	class MapNode {
	public:
		enum {
			MapNodes = 5,
		};
		MapNode() : nNodes(0), nKeys(0), nMaps(1), parent(NULL), left(NULL), right(NULL) {
		}
		~MapNode() {
			delete left;
			delete right;
		}
		Node	node[MapNodes];
		INT	nNodes;
		INT	nKeys;
		INT	nMaps;
		MapNode *parent;
		MapNode *left;
		MapNode *right;
	};
protected:
	VariantMap(const VariantMap &d);
	VariantMap &operator=(const VariantMap &d);

public:
	VariantMap()
	{
		top	= NULL;
		endKeyNum = 0;
		endKeyStr = ES_T("[[end]]");
	}
	~VariantMap() {
		delete top;
	}
	class iterator {
		friend class VariantMap;
	public:
		iterator() {
			this->map		= NULL;
			this->n			= 0;
		}
		iterator(const CHAR *endKey) {
			this->endKey	= endKey;
			this->map		= NULL;
			this->n			= 0;
		}
		iterator(MapNode *map, INT n, const CHAR * endKey) {
			this->endKey	= endKey;
			this->map		= map;
			this->n			= n;
		}
		iterator(const iterator &d) {
			endKey	= d.endKey;
			map		= d.map;
			n		= d.n;
		}
		~iterator() {
		}
		bool Empty() const {
			return map == NULL;
		}
		const CHAR *key() const {
			if( !map ) return endKey.ptr();
			return map->node[n].key.ptr();
		}
		bool operator==(const iterator &d) {
			return d.map == map && d.n == n && d.endKey.equals(endKey);
		}
		bool operator!=(const iterator &d) {
			return d.map != map || d.n != n || !d.endKey.equals(endKey);
		}
		VARIANT *v() {
			return map->node[n].val;
		}
		VARIANT *operator->() {
			return v();
		}
		operator VARIANT*() {
			return v();
		}
		iterator& operator++(int) {
			if( !map ) return *this;

			if( n+1 < map->nNodes ) {
				n++;
				return *this;
			}
			n	= 0;
			MapNode *p = map;
			while( p ) {
				if( p->right ) {
					p = p->right;
					while( p->left ) {
						p = p->left;
					}
				} else {
					while( p ) {
						MapNode *o = p;
						p = p->parent;
						if( p && p->right != o ) break;
					}
				}
				if( p && p->nNodes ) {
					map = p;
					return *this;
				}
			}
			map	= NULL;
			return *this;
		}
	protected:
		String		endKey;
		MapNode *	map;
		INT			n;
	};
protected:
	iterator findKey(MapNode *root, const CHAR *key) const {
		MapNode *k = root;
		while( k ) {
			if( k->nNodes ) {
				INT cond = k->node[0].key.compare(key);
				if( cond > 0 ) {
					k = k->left;
					continue;
				} else if( cond == 0 ) {
					return iterator(k,0, endKey());
				} else {
					if( k->nNodes > 1 ) {
						cond = k->node[k->nNodes-1].key.compare(key);
					}
					if( cond < 0 ) {
						k = k->right;
						continue;
					} else if( cond == 0 ) {
						return iterator(k,k->nNodes-1, endKey());
					} else {
						INT l=0;
						INT r=k->nNodes-1;
						INT c=(r-l)/2+l;
						while(l < c && c < r) {
							cond = k->node[c].key.compare(key);
							if( cond == 0 ) {
								return iterator(k, c, endKey());
							}
							if( cond < 0 ) {
								l = c;
							} else {
								r = c;
							}
							c=(r-l)/2+l;
						}
					}
				}
			}
			break;
		}
		return iterator(endKey());
	}

	iterator  eraseKey(iterator t) {
		if( !t.map ) return t;

		MapNode *k = t.map;
		INT c = t.n;

		MapNode *rcnode = NULL;
		String nextKey;
		if( k->nNodes <= 1 ) {
			t++;
			MapNode *rep = NULL;
			MapNode *ladd = NULL;
			MapNode *radd = NULL;
			if( k->left && k->right ) {
				if( k->left->nMaps > k->right->nMaps ) {
					rep = k->left;
					radd = k->right;
				} else {
					rep	= k->right;
					ladd = k->left;
				}
			} else if( k->left ) {
				rep	= k->left;
			} else {
				rep = k->right;
			}

			if( k->parent ) {
				if( k->parent->left == k ) {
					k->parent->left = rep;
				} else {
					k->parent->right = rep;
				}
				if( rep ) {
					rep->parent = k->parent;
					rcnode	= rep;
				}
			} else if( rep ) {
				INT i;
				for( i = 0; i < rep->nNodes; i++ ) {
					k->node[i] = rep->node[i];
					rep->node[i].val.clear();
				}
				for( ; i < k->nNodes; i++ ) {
					k->node[i].val.clear();
				}
				k->nKeys	= rep->nKeys;
				k->nMaps	= rep->nMaps;
				k->nNodes	= rep->nNodes;
				k->left		= rep->left;
				k->right	= rep->right;
				if( k->left )	k->left->parent		= k;
				if( k->right )	k->right->parent	= k;
				if( t.map == rep ) t.map = k;
				k = rep;
			}
			if( ladd ) {
				MapNode *rp = rep;
				while(rp->left) {
					rp = rp->left;
				}
				rp->left = ladd;
				ladd->parent = rp;
				rcnode = ladd;
			}
			if( radd ) {
				MapNode *rp = rep;
				while(rp->right) {
					rp = rp->right;
				}
				rp->right = radd;
				radd->parent = rp;
				rcnode = radd;
			}
			if( k->parent )	{
				k->left = NULL;
				k->right = NULL;
				delete k;
			} else {
				for( INT i = c; i < k->nNodes-1; i++ ) {
					k->node[i] = k->node[i+1];
				}
				k->node[k->nNodes-1].val.clear();
				k->nNodes--;
				k->nKeys--;
				rcnode = k;
			}
		} else {
			for( INT i = c; i < k->nNodes-1; i++ ) {
				k->node[i] = k->node[i+1];
			}
			k->node[k->nNodes-1].val.clear();
			k->nNodes--;
			k->nKeys--;
			rcnode = k;
		}
		if( rcnode ) {
			rcnode = rcnode->parent;
			while( rcnode ) {
				INT nKeys = rcnode->nNodes;
				INT nMaps = 1;
				if( rcnode->left ) {
					nKeys += rcnode->left->nKeys;
					nMaps += rcnode->left->nMaps;
				}
				if( rcnode->right ) {
					nKeys += rcnode->right->nKeys;
					nMaps += rcnode->right->nMaps;
				}
				rcnode->nKeys = nKeys;
				rcnode->nMaps = nMaps;
				rcnode = rcnode->parent;
			}
		}
		return t;
	}

	void newEndKey(const CHAR *key) {
		if( isEqualCharsLen(key, endKeyStr.ptr(), 7) ) {
			const CHAR *keyNum = &key[7];
			if( *keyNum ) {
				INT n = parseInt(keyNum);
				if( endKeyNum == 0 || endKeyNum < n ) {
					endKeyNum = n+1;
				}
			}
			endKeyStr.format(ES_T("[[end]](%d)"), endKeyNum);
		}
	}

	iterator addKey(MapNode *root, const CHAR * key) {
		MapNode *p = NULL;
		MapNode *k = root;
		INT c = 0;
		while( k ) {
			p = k;
			if( k->nNodes ) {
				INT cond = k->node[0].key.compare(key);
				if( cond > 0 ) {
					c = 0;
					k = k->left;
					continue;
				} else if( cond == 0 ) {
					return iterator(k,0,endKey());
				} else {
					if( k->nNodes > 1 ) {
						cond = k->node[k->nNodes-1].key.compare(key);
					}
					if( cond < 0 ) {
						c = k->nNodes;
						k = k->right;
						continue;
					} else if( cond == 0 ) {
						return iterator(k,k->nNodes-1, endKey());
					} else {
						for( c = 1; c < k->nNodes-1; c++ ) {
							cond = k->node[c].key.compare(key);
							if( cond == 0 ) return iterator(k,c, endKey());
							if( cond > 0 ) break;
						}
					}
				}
			}
			break;
		}
		if( p->nNodes < MapNode::MapNodes ) {
			for(INT i=p->nNodes-1; c<=i; i--) {
				p->node[i+1] = p->node[i];
			}
			Node *n = &p->node[c];
			n->key = key;
			p->nNodes++;
			p->nKeys++;
			MapNode *rcnode = p->parent;
			while(rcnode) {
				rcnode->nKeys++;
				rcnode = rcnode->parent;
			}
			return iterator(p,c, endKey());
		} else {
			new_map(p);
			return addKey(root, key);
		}
	}

	bool new_map(MapNode *p) {
		INT spn = (p->nNodes-1)/2;
		INT max = MapNode::MapNodes - spn;
		INT am = 0;
		if( !p->left ) {
			p->left = new MapNode();
			if( !p->left ) return false;
			p->left->parent = p;
			p->nMaps++;
			am++;
		} else if( (MapNode::MapNodes - p->left->nNodes) > max ) {
			new_map(p->left);
		}
		if( !p->right ) {
			p->right = new MapNode();
			if( !p->right ) return false;
			p->right->parent = p;
			p->nMaps++;
			am++;
		} else if( (MapNode::MapNodes - p->right->nNodes) > max ) {
			new_map(p->right);
		}

		INT si=0, i, m;
		for( i=0; i<spn; i++) {
			p->left->node[p->left->nNodes++] = p->node[si++];
			p->left->nKeys++;
		}
		for( m=0; m<p->nNodes-(spn*2); m++) {
			p->node[m] = p->node[si++];
		}

		for( i=0; i<spn; i++) {
			p->right->node[p->right->nNodes++] = p->node[si++];
			p->right->nKeys++;
		}

		for( ; m < p->nNodes; m++) {
			p->node[m].val.clear();
		}
		p->nNodes = p->nNodes-(spn*2);

		if( am ) {
			MapNode *mp = p->parent;
			while(mp) {
				mp->nMaps += am;
				mp = mp->parent;
			}
		}
		return true;
	}
public:

	iterator set(const CHAR * key, const VARIANT *v) {
		newEndKey(key);
		iterator t(endKey());
		if( !top ) {
			top = new MapNode();
			top->nNodes = 1;
			top->nKeys  = 1;
			top->nMaps	= 1;
			top->node[0].key = key;
			t.map	= top;
			t.n		= 0;
		} else {
			t = addKey(top, key);
		}
		if( t.map ) {
			t.map->node[t.n].val = v;
		}
		return t;
	}
	VARIANT *get(const CHAR *key) const {
		if( !top ) return NULL;
		iterator t = findKey(top, key);
		if( end() == t ) return NULL;
		return t.v();
	}
	iterator find(const CHAR *key) const {
		return findKey(top, key);
	}
	iterator erase(iterator t) {
		t = eraseKey(t);
		if( top && top->nNodes == 0 ) {
			delete top;
			top = NULL;
		}
		return t;
	}
	iterator replaceKey(iterator t, const CHAR *key) {
		newEndKey(key);
		Variant v = t.v();
		t = eraseKey(t);
		if( top && top->nNodes == 0 ) {
			delete top;
			top = NULL;
		}
		return set(key, v);
	}
	bool clear() {
		delete top;
		return true;
	}
	SIZE size() const {
		if( !top ) return 0;
		return top->nKeys;
	}
	iterator begin() const {
		iterator i(endKey());
		if( !top || top->nKeys == 0 ) {
			i.map	= NULL;
			i.n		= 0;
			return i;
		}
		MapNode *p = top;
		while( p->left ) {
			p = p->left;
		}
		i.map	= p;
		i.n		= 0;
		if( p && p->nNodes == 0 ) {
			i++;
		}
		return i;
	}
	iterator end() const {
		return iterator(endKey());
	}
	const CHAR *endKey() const {
		return endKeyStr.ptr();
	}
	INT endKeyNum;
	String endKeyStr;
	MapNode *top;
};

} // es


#endif /* VARIANT_H_ */
