/*
 * buffer.cpp
 *
 *  Created on: 2012/07/05
 *      Author: yasuoki
 */

#include "buffer.h"
#include <stdio.h>
#include <stdarg.h>
#include <memory.h>
#include <malloc.h>
#include <assert.h>

namespace SST {

bool CyclicBuffer::ptrTest(size_t a, size_t r, size_t w)
{
	return (a==0||nAlloc == a) && nWrite == w && nRead == r;
}

CyclicBuffer::CyclicBuffer()
{
	nAlloc	= 0;
	nWrite	= 0;
	nRead	= 0;
	ptr		= NULL;
}

CyclicBuffer::~CyclicBuffer()
{
	clear();
}

size_t CyclicBuffer::getFreeSpace(size_t &nLeft, size_t &nRight) const
{
	if( nAlloc == 0 ) {
		nLeft	= 0;
		nRight	= 0;
		return 0;
	}
	if( nRead == nWrite ) {
		// empty
		// Zooooooooooooooo nLeft = 0 nRight = 16
		// oooooooooZoooooo nLeft = 0 nRight = 16
		if( nRead == 0 ) {
			nLeft	= 0;
			nRight	= nAlloc - 1;
		} else {
			nLeft	= nRead - 1;
			nRight	= nAlloc - nWrite;
		}
	} else if( nRead < nWrite ) {
		// RxxxxxxxxWoooooo nLeft = 0 nRight = 7
		// ooooRxxxxWoooooo nLeft = 4 nRight = 7
		if( nRead == 0 ) {
			nLeft	= nRead;
			nRight	= nAlloc - nWrite - 1;
		} else {
			nLeft	= nRead - 1;
			nRight	= nAlloc - nWrite;
		}
	} else {
		// xxxxWoooRxxxxxxxx nLeft = 0 nRight = 3
		nLeft	= 0;
		nRight	= nRead - nWrite - 1;
	}
	return nLeft + nRight;
}

size_t CyclicBuffer::getUsingSpace(size_t &nLeft, size_t &nRight) const
{
	if( nRead == nWrite ) {
		// empty
		// Zooooooooooooooo nLeft = 0 nRight = 16
		// oooooooooZoooooo nLeft = 0 nRight = 16
		nLeft	= 0;
		nRight	= 0;
	} else if( nRead < nWrite ){
		nRight	= nRead - nWrite;
		// RxxxxxxxxWoooooo nLeft = 0 nRight = 7
		// ooooRxxxxWoooooo nLeft = 4 nRight = 7
		nLeft	= 0;
		nRight	= nWrite - nRead;
	} else {
		// xxxxWoooRxxxxxxx nLeft = 0 nRight = 4
		nLeft	= nWrite;
		nRight	= nAlloc - nRead;
	}
	return nLeft + nRight;
}

size_t CyclicBuffer::size() const
{
	size_t nRight;
	size_t nLeft;
	size_t nUse = getUsingSpace(nLeft, nRight);
	return nUse;
}

bool CyclicBuffer::add(const char *data, ssize_t dataSize)
{
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::add(%d) A=%d R=%d W=%d\n", dataSize, nAlloc, nRead, nWrite);
#endif
	if( dataSize == -1 ) {
		dataSize  = strlen(data);
	}
	size_t size = (size_t)dataSize;
	size_t nRight;
	size_t nLeft;
	size_t nFree = getFreeSpace(nLeft, nRight);
	if( nFree < size ) {
		size_t need	= size - nFree;
		size_t newAlloc = (((nAlloc + need)/SST_IOBUFFER_BLOCKSIZE)+1)*SST_IOBUFFER_BLOCKSIZE;
		char * newPtr = (char*)malloc(newAlloc);
		if( !newPtr ) return false;
		size_t nUse = 0;
		if( nAlloc ) {
			nUse = getUsingSpace(nLeft, nRight);
			memcpy(newPtr, &ptr[nRead], nRight);
			memcpy(&newPtr[nRight], ptr, nLeft);
			free(ptr);
		}
		nAlloc	= newAlloc;
		ptr		= newPtr;
		nRead	= 0;
		nWrite	= nUse;
		nLeft	= 0;
		nRight	= nAlloc - nUse - 1;
	}
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::add(%d) A=%d R=%d W=%d Left=%d Right=%d\n", size, nAlloc, nRead, nWrite, nLeft, nRight);
#endif

	size_t as=0;
	size_t cs;
	if( size && nRight ) {
		if( size > nRight )
			cs = nRight;
		else
			cs = size;
		memcpy(&ptr[nWrite], data, cs);
		size 	-= cs;
		nWrite	+= cs;
		as		+= cs;
		if( nWrite == nAlloc ) nWrite = 0;
	}
	if( size && nLeft ) {
		if( size > nLeft )
			cs = nLeft;
		else
			cs = size;
		memcpy(ptr, &data[as], cs);
		size	-= cs;
		nWrite	+= cs;
		if( nWrite == nAlloc ) nWrite = 0;
	}
	ptr[nWrite] = 0;
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::add(%d) A=%d R=%d W=%d ok\n", size, nAlloc, nRead, nWrite);
#endif
	return true;
}

bool  CyclicBuffer::addfmt(const char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    char buf[1024];
    char *out = buf;
    int outSize = sizeof(buf);
    int size = vsnprintf(out, outSize, fmt, ap);
    if( size > outSize ) {
    	outSize = size + 10;
    	out = (char*)malloc(outSize);
    	if( !out ) return false;
    	va_end(ap);
        va_start(ap, fmt);
        size = vsnprintf(out, outSize, fmt, ap);
    	if( size > outSize ) {
    		free(out);
    		return false;
    	}
    }

#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::adfmt(%s) str=%s size=%d\n", fmt, out, size);
#endif

	bool ret = add(out,size);
   	if( out != buf )
    	free(out);
   	return ret;
}

size_t CyclicBuffer::get(char *buff, size_t buffSize)
{
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::get(%d) A=%d R=%d W=%d\n", buffSize, nAlloc, nRead, nWrite);
#endif
	size_t rs = 0;
	size_t cs;

	size_t nRight;
	size_t nLeft;
	size_t nUse = getUsingSpace(nLeft, nRight);

	if( nUse == 0 ) return 0;
	if( buffSize ) {
		if( nRight < buffSize )
			cs = nRight;
		else
			cs = buffSize;
		if( buff ) memcpy(buff, &ptr[nRead], cs);
		nRead += cs;
		if( nRead >= nAlloc ) nRead = 0;
		rs += cs;
	}
	if( rs < buffSize ) {
		if( nLeft < buffSize-rs )
			cs = nLeft;
		else
			cs = buffSize-rs;
		if( buff ) memcpy(&buff[rs], ptr, cs);
		nRead += cs;
		if( nRead >= nAlloc ) nRead = 0;
		rs += cs;
	}
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::get(%d) A=%d R=%d W=%d ok=%d\n", buffSize, nAlloc, nRead, nWrite, rs);
#endif
	return rs;
}

size_t CyclicBuffer::getLine(char *buff, size_t buffSize)
{
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::getLine(%d) A=%d R=%d W=%d\n", buffSize, nAlloc, nRead, nWrite);
#endif

	size_t bs = buffSize - 1;
	size_t rs = 0;
	size_t cs;

	size_t nRight;
	size_t nLeft;
	size_t nUse = getUsingSpace(nLeft, nRight);

	if( nUse == 0 ) return 0;
	if( bs ) {
		if( nRight < bs )
			cs = nRight;
		else
			cs = bs;
		size_t nc = 0;
		char *p0 = buff;
		char *p1 = &ptr[nRead];
		while(*p1 != '\r' && *p1 != '\n' && nc < cs ) {
			*p0++ = *p1++;
			nc++;
		}
		rs += nc;
		if( nc < cs ) {
			if( *p1 == '\r' || *p1 == '\n' ) {
				if( *p1 == '\r' ) { p1++; nc++; }
				if( *p1 == '\n' ) { p1++; nc++; }
			}
		}
		nRead += nc;
		if( nRead >= nAlloc ) nRead = 0;
	}
	if( rs < bs ) {
		if( nLeft < bs-rs )
			cs = nLeft;
		else
			cs = bs-rs;

		size_t nc = 0;
		char *p0 = &buff[rs];
		char *p1 = ptr;
		while(*p1 != '\r' && *p1 != '\n' && nc < cs ) {
			*p0++ = *p1++;
			nc++;
		}
		rs += nc;
		if( nc < cs ) {
			if( *p1 == '\r' || *p1 == '\n' ) {
				if( *p1 == '\r' ) { p1++; nc++; }
				if( *p1 == '\n' ) { p1++; nc++; }
			}
		}
		nRead += cs;
		if( nRead >= nAlloc ) nRead = 0;
		rs += cs;
	}
	buff[rs] = 0;
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::getLine(%d) A=%d R=%d W=%d ok=%d\n", buffSize, nAlloc, nRead, nWrite, rs);
#endif
	return rs;
}

size_t CyclicBuffer::getAt(char *buff, size_t start, size_t len)
{
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::getAt(%d, %d) A=%d R=%d W=%d\n", start, len, nAlloc, nRead, nWrite);
#endif
	CyclicBuffer t;
	t.nAlloc	= nAlloc;
	t.nWrite	= nWrite;
	t.nRead		= nRead;
	t.ptr		= ptr;

	size_t ss = t.get(NULL, start);
	if( ss != start ) {
		t.ptr	= NULL;
		return 0;
	}
	size_t rs = t.get(buff, len-1);
	buff[rs] = 0;
	t.ptr	= NULL;
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::getAt(%d, %d) A=%d R=%d W=%d ok=%d\n", start, len, nAlloc, nRead, nWrite, rs);
#endif
	return rs;
}

const char *CyclicBuffer::getPtr()
{
	const char *r = NULL;
	size_t nRight;
	size_t nLeft;
	size_t nUse = getUsingSpace(nLeft, nRight);
	if( nUse == 0 ) return NULL;
	if( nLeft == 0 && nRead < nWrite ) {
		ptr[nWrite] = 0;
		r = &ptr[nRead];
	} else {
		char * tmpPtr = (char*)malloc(nUse+1);
		if( !tmpPtr ) return false;
		memcpy(tmpPtr, &ptr[nRead], nRight);
		memcpy(&tmpPtr[nRight], ptr, nLeft);
		memcpy(ptr, tmpPtr, nUse);
		ptr[nUse] = 0;
		free(tmpPtr);
		r = ptr;
	}
	nRead	= 0;
	nWrite	= 0;
	return r;
}

void CyclicBuffer::clear()
{
	if( ptr ) free(ptr);
	nAlloc	= 0;
	nWrite	= 0;
	nRead	= 0;
	ptr		= NULL;
}
//////////////////////////////////////////////
#define STATIC_BLOCK_SIZE 256
StaticBuffer::StaticBuffer()
{
	next		= NULL;
	nAlloc		= 0;
	nUse		= 0;
	ptr			= NULL;
	prevStore	= NULL;
}

StaticBuffer::~StaticBuffer()
{
	clear();
}

bool StaticBuffer::add(const char *data, ssize_t size)
{
	return store(data, size) != NULL;
}

bool StaticBuffer::addfmt(const char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    char buf[1024];
    char *out = buf;
    int outSize = sizeof(buf);
    int size = vsnprintf(out, outSize, fmt, ap);
    if( size > outSize ) {
    	outSize = size + 10;
    	out = (char*)malloc(outSize);
    	if( !out ) return false;
    	va_end(ap);
        va_start(ap, fmt);
        size = vsnprintf(out, outSize, fmt, ap);
    	if( size > outSize ) {
    		free(out);
    		return false;
    	}
    }
	bool ret = add(out,size);
   	if( out != buf )
    	free(out);
   	return ret;
}

size_t StaticBuffer::size() const
{
	size_t nSize = nUse;
	StaticBuffer *p = next;
	while(p) {
		nSize += p->nUse;
		p = p->next;
	}
	return nSize;
}

size_t StaticBuffer::get(char *buff, size_t buffSize)
{
	StaticBuffer *p = this;
	size_t getSize = 0;
	while(p && buffSize) {
		size_t nc = p->nUse < buffSize ? p->nUse : buffSize;
		if( nc ) {
			memcpy(buff, p->ptr, nc);
			buff += nc;
			buffSize -= nc;
			getSize += nc;
		}
		p = p->next;
	}
	return getSize;
}

size_t StaticBuffer::getLine(char *buff, size_t buffSize)
{
	StaticBuffer *p = this;
	size_t getSize = 0;
	while(p && buffSize>1) {
		char *dp = p->ptr;
		size_t nc = 0;
		while( nc < p->nUse && nc < buffSize-1 ) {
			if( *dp == '\r' || *dp == '\n' ) {
				*buff++ = '\n';
				nc++;
				break;
			}
			*buff++ = *dp++;
			nc++;
		}
		buffSize -= nc;
		getSize += nc;
		p = p->next;
	}
	*buff = 0;
	return getSize;
}

size_t StaticBuffer::getAt(char *buff, size_t start, size_t len)
{
	StaticBuffer *p = this;
	while( p ) {
		if( p->nUse < start ) {
			start -= p->nUse;
			p = p->next;
		} else {
			break;
		}
	}
	size_t getSize = 0;
	if( p ) {
		while(p && len) {
			size_t nc = p->nUse-start < len ? p->nUse-start : len;
			if( nc ) {
				memcpy(buff, &p->ptr[start], nc);
				buff += nc;
				len	-= nc;
				getSize += nc;
				start = 0;
			}
			p = p->next;
		}
	}
	return getSize;
}

const char * StaticBuffer::getPtr()
{
	return ptr;
}

void StaticBuffer::clear()
{
	if( next ) delete next;
	if( ptr ) free(ptr);
	next		= NULL;
	nAlloc		= 0;
	nUse		= 0;
	ptr			= NULL;
	prevStore	= NULL;
}

bool StaticBuffer::empty() const
{
	return size() == 0;
}

const char * StaticBuffer::store(const char *data, size_t size)
{
	char *sp = NULL;
	StaticBuffer *p = this;
	StaticBuffer *last = NULL;
	while( p ) {
		size_t free = p->nAlloc - p->nUse;
		if( free >= size ) break;
		last	= p;
		p = p->next;
	}
	char *buff = NULL;
	if( !p ) {
		size_t ns = STATIC_BLOCK_SIZE;
		if( ns < size ) {
			ns = (size/STATIC_BLOCK_SIZE+1)*STATIC_BLOCK_SIZE;
		}
		buff = (char*)malloc(ns);
		if( !ptr ) return NULL;
		if( nAlloc == 0 ) {
			p		= this;
		} else {
			p = new StaticBuffer();
			if( !p ) {
				free(buff);
				return NULL;
			}
			last->next	= p;
		}
		p->ptr		= buff;
		p->nAlloc	= ns;
	}
	memcpy(&p->ptr[p->nUse], data, size);
	prevStore	= &p->ptr[p->nUse];
	p->nUse += size;
	return prevStore;
}

const char * StaticBuffer::storePtr() const
{
	return prevStore;
}


}

#if 0

int _tmain(int argc, _TCHAR* argv[])
{

	SST::CyclicBuffer b0;

	_ASSERT(b0.size()==0);
	_ASSERT(b0.add("1234",4));
	_ASSERT(b0.ptrTest(0, 0, 4));

	char buff[257];
	_ASSERT(b0.get(&buff[0], 1)==1);
	_ASSERT(b0.ptrTest(0, 1, 4));
	_ASSERT(b0.get(&buff[1], 1)==1);
	_ASSERT(b0.ptrTest(0, 2, 4));
	_ASSERT(b0.get(&buff[2], 1)==1);
	_ASSERT(b0.ptrTest(0, 3, 4));
	_ASSERT(b0.get(&buff[3], 1)==1);
	_ASSERT(b0.ptrTest(0, 4, 4));
	_ASSERT(b0.get(&buff[4], 1)==0);
	_ASSERT(b0.ptrTest(0, 4, 4));
	buff[4] = 0;
	_ASSERT(strcmp(buff, "1234")==0);

	int i;
	for( i = 0; i < 256; i++ ) {
		if( i == 12 ) {
			int a=0;
		}
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.get(buff, 6) ==6);
		buff[6] = 0;
		_ASSERT(strcmp(buff, "123456")==0);
		_ASSERT(b0.get(buff, 6) ==4);
		buff[4] = 0;
		_ASSERT(strcmp(buff, "7890")==0);
	}
	for( i = 0; i < 256; i++ ) {
		if( i == 12 ) {
			int a=0;
		}
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.get(buff, 17) ==17);
		buff[17] = 0;
		_ASSERT(strcmp(buff, "12345678901234567")==0);
		_ASSERT(b0.get(buff, 3) ==3);
		buff[3] = 0;
		_ASSERT(strcmp(buff, "890")==0);
	}

	int n=0;
	for( i = 0; i < 256; i++ ) {
		if( i == 7 ) {
			int a=0;
		}
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.get(buff, 13) ==13);
		buff[13] = 0;
		_ASSERT(strncmp(buff, &"1234567890123456789012345678901234567890"[n],13)==0);
		n += 13;
		n = n % 10;
	}
	size_t bs;
	for( i = 0; i < 7; i++ ) {
		bs = b0.size();
		_ASSERT(bs == (7-i)*256);
		_ASSERT(b0.get(buff, 256) == 256);

		buff[256] = 0;
		for(int m=0; m<16; m++) {
			const char *pat = &"1234567890123456789012345678901234567890"[n];
			_ASSERT(strncmp(&buff[m*16], pat,16)==0);
			n += 16;
			n = n % 10;
		}
	}

	bs = b0.size();
	_ASSERT(bs == 0);

	const char *bp;
	for( i = 0; i < 256; i++ ) {
		_ASSERT(b0.add("1234567890", 10));
		bp = b0.getPtr();
		_ASSERT(strcmp(bp, "1234567890")==0);
	}
	for( i = 0; i < 256; i++ ) {
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.add("1234567890", 10));
		bp = b0.getPtr();
		_ASSERT(strcmp(bp, "12345678901234567890")==0);
	}


	bs = b0.size();
	_ASSERT(bs == 0);

	return 0;
}


#endif

