/*
 * database.cpp
 *
 *  Created on: 2012/06/08
 *      Author: yasuoki
 */
#include <stdlib.h>
#include <syslog.h>
#include <memory.h>
#include <time.h>
#include <stdarg.h>
#include <hiredis/hiredis.h>
#include "database.h"
#include "sentinel.h"

#define REPLY_CHECK(R,T) \
	if( R==NULL ) { \
		sentinel->log(LOG_ERR, "redis error %s", conn->errstr); \
		return DBErrorSystem; \
	} else if( R->type != T) {\
		sentinel->log(LOG_ERR, "redis error type not match"); \
		freeReplyObject(R); \
		return DBErrorSystem; \
	}

#define REPLY_CHECK_TYPE_OR_NULL(R,T) \
	if( R==NULL ) { \
		sentinel->log(LOG_ERR, "redis error %s", conn->errstr); \
		return DBErrorSystem; \
	} else if( R->type != REDIS_REPLY_NIL && R->type != T) {\
		sentinel->log(LOG_ERR, "redis error type not match"); \
		freeReplyObject(R); \
		return DBErrorSystem; \
	}

#define REPLY_STATUS(R) \
	REPLY_CHECK(R,REDIS_REPLY_STATUS)

#define REPLY_TYPE_IS(R,T) \
	(R != NULL && R->type == T )

#define IS_STATUS(R,S) \
	(strcmp(R->str, S) == 0)

#define IS_STATUS_OK(R) IS_STATUS(R,"OK")

#define LOCK_ACCOUNT(ID, TIMEOUT) \
	{ \
		time_t ct = time(NULL) + TIMEOUT; \
		char timeout[12]; \
		sprintf(timeout, "%ld", ct); \
		redisReply *r = (redisReply*)redisCommand(conn, "setnx lock:%s %s", ID, timeout); \
		REPLY_CHECK(r, REDIS_REPLY_INTEGER); \
		if( r->integer != 1 ) { \
			freeReplyObject(r); \
			r = (redisReply*)redisCommand(conn, "get lock:%s", ID); \
			REPLY_CHECK(r, REDIS_REPLY_INTEGER); \
			if( r->integer > ct) { \
				freeReplyObject(r); \
				return DBErrorLocked; \
			} \
		} \
		freeReplyObject(r); \
		r = (redisReply*)redisCommand(conn, "expire lock:%s %d", ID, TIMEOUT); \
	}
#define UNLOCK_ACCOUNT(ID) \
	{ \
		redisReply *r = (redisReply*)redisCommand(conn, "del lock:%s", ID); \
		if( r ) freeReplyObject(r); \
	}

#define STATUS_CHECK_OK(R, ID) \
	REPLY_STATUS(R); \
	if( !IS_STATUS(R,"OK") ) { \
		freeReplyObject(R); \
		UNLOCK_ACCOUNT(ID); \
		sentinel->log(LOG_ERR, "database error, %s", conn->errstr); \
		return DBErrorSystem; \
	} \
	freeReplyObject(R);

#define STATUS_CHECK_QUEUED(R, ID) \
	REPLY_STATUS(R); \
	if( !IS_STATUS(R,"QUEUED") ) { \
		freeReplyObject(R); \
		R = (redisReply*)redisCommand(conn, "DISCARD"); \
		freeReplyObject(R); \
		UNLOCK_ACCOUNT(ID); \
		sentinel->log(LOG_ERR, "database error, %s", conn->errstr); \
		return DBErrorSystem; \
	} \
	freeReplyObject(R);

namespace SST {

Database::Database()
{
	sentinel	= NULL;
	server[0]	= 0;
	port		= 0;
	threads		= 1;
	auth[0]		= 0;
}

Database::~Database()
{
}

bool Database::configure(Sentinel *obj, Conf *conf)
{
	sentinel	= obj;
	strcpy(server, conf->dbServer);
	port = conf->dbPort;
	threads	= conf->dbThreads;
	if( conf->dbAuth )
		strcpy( auth, conf->dbAuth);
	return true;
}

bool Database::checkAccountName(const char *name)
{
	return true;
}

///////////////////////////
RedisDB::RedisDB()
{
	conn	= NULL;
}

RedisDB::~RedisDB()
{
	if( conn )
		redisFree(conn);
}

bool RedisDB::getInt(long long int &v, bool &null, const char *msg, const char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
	redisReply *rep = (redisReply*)redisvCommand(conn, fmt, ap);
	va_end(ap);
	if( rep == NULL ) {
		sentinel->log(LOG_ERR, "redis error. %s", conn->errstr);
		return false;
	}
	if( rep->type == REDIS_REPLY_NIL ) {
		freeReplyObject(rep);
		if( null ) {
			return true;
		}
		if( msg )
			sentinel->log(LOG_ERR, "redis data is nil. %s", msg);
		else
			sentinel->log(LOG_ERR, "redis data is nil.");
		return false;
	}
	null = false;
	if( rep->type != REDIS_REPLY_INTEGER ) {
		if( msg )
			sentinel->log(LOG_ERR, "redis type error. %s", msg);
		else
			sentinel->log(LOG_ERR, "redis type error.");
		freeReplyObject(rep);
		return false;
	}
	v = rep->integer;
	return true;
}

bool RedisDB::getStr(redisReply **rep, bool &null, const char *msg, const char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
	*rep = (redisReply*)redisvCommand(conn, fmt, ap);
	va_end(ap);
	if( *rep == NULL ) {
		sentinel->log(LOG_ERR, "redis error. %s", conn->errstr);
		return false;
	}
	if( (*rep)->type == REDIS_REPLY_NIL ) {
		freeReplyObject(*rep);
		*rep	= NULL;
		if( null ) {
			return true;
		}
		if( msg )
			sentinel->log(LOG_ERR, "redis data is nil. %s", msg);
		else
			sentinel->log(LOG_ERR, "redis data is nil.");
		return false;
	}
	null = false;
	if( (*rep)->type != REDIS_REPLY_STRING ) {
		if( msg )
			sentinel->log(LOG_ERR, "redis type error. %s", msg);
		else
			sentinel->log(LOG_ERR, "redis type error.");
		freeReplyObject(*rep);
		*rep	= NULL;
		return false;
	}
	return true;
}


bool RedisDB::connect()
{
	redisContext *c = redisConnect(server, port);
	if( c->err != 0 ) {
		//c->errstr
		sentinel->log(LOG_ERR, "Database connect error. %s", c->errstr );
		redisFree(c);
		return false;
	}
	if( conn ) redisFree(conn);
	conn = c;
	return true;
}

void RedisDB::close()
{
	if( conn ) {
		redisFree(conn);
		conn = NULL;
	}
}

DBError RedisDB::getUserList(size_t nFrom, RecList *pList)
{
	size_t i;
	size_t idSize = sizeof(char)*(SST_USERIDLEN+1)*pList->nCount;
	size_t ptSize = sizeof(char*)*pList->nCount;
	if( pList->nSize < sizeof(RecList) + idSize + ptSize ) {
		return DBErrorNoSpace;
	}
	char *pId = (char*)&pList->pValue[pList->nCount];
	for( i = 0; i < pList->nCount; i++ ) {
		pList->pValue[i]	= pId;
		pId += sizeof(char)*(SST_USERIDLEN+1);
	}

	redisReply *reply = (redisReply*)redisCommand(conn, "LRANGE account %d %d", nFrom, nFrom+pList->nCount);
	REPLY_CHECK(reply, REDIS_REPLY_ARRAY);

	pList->nValues	= 0;
	size_t nAdd = 0;
	for( i = 0; i < reply->elements; i++ ) {
		if( reply->element[i]->type == REDIS_REPLY_STRING ) {
			strcpy(pList->pValue[nAdd], reply->element[i]->str);
			nAdd++;
		}
	}
	pList->nValues	= nAdd;
	freeReplyObject(reply);
	return DBErrorOk;
}

DBError RedisDB::getGroupList(size_t nFrom, RecList *pList)
{
	sentinel->log(LOG_DEBUG, "RedisDB::getGroupList begin");
	size_t i;
	size_t idSize = sizeof(char)*(SST_USERIDLEN+1)*pList->nCount;
	size_t ptSize = sizeof(char*)*pList->nCount;
	if( pList->nSize < sizeof(RecList) + idSize + ptSize ) {
		sentinel->log(LOG_ERR, "RedisDB::getGroupList no space");
		return DBErrorNoSpace;
	}
	char *pId = (char*)&pList->pValue[pList->nCount];
	for( i = 0; i < pList->nCount; i++ ) {
		pList->pValue[i]	= pId;
		pId += sizeof(char)*(SST_USERIDLEN+1);
	}

	redisReply *reply = (redisReply*)redisCommand(conn, "LRANGE group %d %d", nFrom, nFrom+pList->nCount);
	REPLY_CHECK(reply, REDIS_REPLY_ARRAY);

	pList->nValues	= 0;
	size_t nAdd = 0;
	for( i = 0; i < reply->elements; i++ ) {
		if( reply->element[i]->type == REDIS_REPLY_STRING ) {
			strcpy(pList->pValue[nAdd], reply->element[i]->str);
			nAdd++;
		}
	}
	pList->nValues	= nAdd;
	freeReplyObject(reply);
	return DBErrorOk;
}

DBError RedisDB::addAcocunt(Account *account)
{
	char com[4096];

	if( !checkAccountName(account->userName) ) {
		return DBErrorUserName;
	}
	redisReply *reply = (redisReply*)redisCommand(conn, "get an:%s:aid", account->userName );
	if( reply == NULL ) {
		sentinel->log(LOG_ERR, "redis error %s", conn->errstr);
		return DBErrorSystem;
	}
	if( reply->type != REDIS_REPLY_NIL ) {
		return DBErrorDuplication;
	}
	freeReplyObject(reply);

	// account id
	reply = (redisReply*)redisCommand(conn, "incr aidseq" );
	REPLY_CHECK(reply, REDIS_REPLY_INTEGER);
	char aid[10];
	sprintf(aid, "%8.8lld", reply->integer);
	freeReplyObject(reply);

	// lock
	LOCK_ACCOUNT(aid, 10);

	// account list
	reply = (redisReply*)redisCommand(conn, "lpush account %s", aid);
	REPLY_CHECK(reply, REDIS_REPLY_INTEGER);
	freeReplyObject(reply);

	time_t ct = time(NULL);
	struct tm t;
	gmtime_r(&ct, &t);
	char createDate[100];
	strftime(createDate, sizeof(createDate), "%Y/%m/%d-%H:%M:%S", &t);

	// account information
	reply = (redisReply*)redisCommand(conn, "set an:%s:aid %s", account->userName, aid);
	REPLY_STATUS(reply);
	STATUS_CHECK_OK(reply, aid);

	reply = (redisReply*)redisCommand(conn, "set aid:%s:an %s", aid, account->userName);
	REPLY_STATUS(reply);
	STATUS_CHECK_OK(reply, aid);

	sprintf(com, "%d", account->level );
	reply = (redisReply*)redisCommand(conn, "set aid:%s:level %s", aid, com);
	REPLY_STATUS(reply);
	STATUS_CHECK_OK(reply, aid);
	reply = (redisReply*)redisCommand(conn, "set aid:%s:create %s", aid, createDate);
	REPLY_STATUS(reply);
	STATUS_CHECK_OK(reply, aid);

	if( account->auth ) {
		reply = (redisReply*)redisCommand(conn, "set aid:%s:auth %s", aid, account->auth);
		STATUS_CHECK_OK(reply, aid);
	}
	if( account->firstName ) {
		reply = (redisReply*)redisCommand(conn, "set aid:%s:firstName %s", aid, account->firstName);
		STATUS_CHECK_OK(reply, aid);
	}
	if( account->secondName ) {
		reply = (redisReply*)redisCommand(conn, "set aid:%s:secondName %s", aid, account->secondName);
		STATUS_CHECK_OK(reply, aid);
	}
	if( account->lastName ) {
		reply = (redisReply*)redisCommand(conn, "set aid:%s:lastName %s", aid, account->lastName);
		STATUS_CHECK_OK(reply, aid);
	}
	if( account->email ) {
		reply = (redisReply*)redisCommand(conn, "set aid:%s:email %s", aid, account->email);
		STATUS_CHECK_OK(reply, aid);
	}
	if( account->addr1 ) {
		reply = (redisReply*)redisCommand(conn, "set aid:%s:addr1 %s", aid, account->addr1);
		STATUS_CHECK_OK(reply, aid);
	}
	if( account->addr2 ) {
		reply = (redisReply*)redisCommand(conn, "set aid:%s:addr2 %s", aid, account->addr2);
		STATUS_CHECK_OK(reply, aid);
	}
	if( account->addr3 ) {
		reply = (redisReply*)redisCommand(conn, "set aid:%s:addr3 %s", aid, account->addr3);
		STATUS_CHECK_OK(reply, aid);
	}
	if( account->phone ) {
		reply = (redisReply*)redisCommand(conn, "set aid:%s:phone %s", aid, account->phone);
		STATUS_CHECK_OK(reply, aid);
	}

	UNLOCK_ACCOUNT(aid);
	return DBErrorOk;
}

DBError RedisDB::removeAcocunt(const char *aid)
{
	redisReply *reply = (redisReply*)redisCommand(conn, "get aid:%s:an", aid);
	REPLY_CHECK(reply, REDIS_REPLY_STRING);
	char userName[256];
	strcpy(userName, reply->str);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "lrem account %s", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del an:%s:id", userName);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:level", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:create", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:lastDate", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:maxStorage", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:useStorage", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:auth", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:firstName", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:secondName", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:lastName", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:email", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:addr1", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:addr2", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:addr3", aid);
	freeReplyObject(reply);

	reply = (redisReply*)redisCommand(conn, "del aid:%s:phone", aid);
	freeReplyObject(reply);

	return DBErrorOk;
}

DBError RedisDB::getAccountId(const char *name, char *aid)
{
	redisReply *reply = (redisReply*)redisCommand(conn, "get an:%s:aid", name);
	REPLY_CHECK_TYPE_OR_NULL(reply, REDIS_REPLY_STRING);
	if( reply->type == REDIS_REPLY_NIL ) {
		freeReplyObject(reply);
		return DBErrorNotFound;
	}
	strcpy(aid, reply->str);
	freeReplyObject(reply);
	return DBErrorOk;
}

DBError RedisDB::getAccountInfo(const char *aid, AccountField f, Account **account)
{
	redisReply	*pRep[17];
	size_t	nRep = 0;
	size_t 	i;
	for( i = 0; i < 17; i++ ) {
		pRep[i]	= NULL;
	}

	size_t	nUserName;
	size_t	nFirstName;
	size_t	nSecondName;
	size_t	nLastName;
	size_t	nAuth;
	size_t	nMail;
	size_t	nAddr1;
	size_t	nAddr2;
	size_t	nAddr3;
	size_t	nPhone;
	size_t	nLevel;
	size_t	nCreate;
	size_t	nLastDate;
	size_t	nMaxStorage;
	size_t	nUseStorage;

//	GET_REPLY_STRING(rUserName, pUserName, nUserName);

	if( f & AccountName ) {
		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:an", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nUserName	= nRep++;

		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:firstName", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nFirstName	= nRep++;

		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:secondName", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nSecondName	= nRep++;

		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:lastName", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nLastName	= nRep++;
	}
	if( f & AccountAuth ) {
		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:auth", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nAuth	= nRep++;
	}
	if( f & AccountMail ) {
		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:email", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nMail	= nRep++;
	}
	if( f & AccountAddr ) {
		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:addr1", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nAddr1	= nRep++;

		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:addr2", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nAddr2	= nRep++;

		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:addr3", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nAddr3	= nRep++;
	}
	if( f & AccountPhone ) {
		pRep[nRep] = (redisReply*)redisCommand(conn, "get aid:%s:phone", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nPhone	= nRep++;
	}
	if( f & AccountStat ) {
		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:level", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_INTEGER);
		nLevel	= nRep++;

		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:create", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nCreate	= nRep++;

		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:lastDate", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_STRING);
		nLastDate	= nRep++;

		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:maxStorage", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_INTEGER);
		nMaxStorage	= nRep++;

		pRep[nRep]	= (redisReply*)redisCommand(conn, "get aid:%s:useStrage", aid);
		REPLY_CHECK_TYPE_OR_NULL(pRep[nRep], REDIS_REPLY_INTEGER);
		nUseStorage	= nRep++;
	}

	size_t nSize = 0;
	for( i = 0; i < nRep; i++ ) {
		if( pRep[i]->type == REDIS_REPLY_STRING ) {
			nSize += pRep[i]->len + 1;
		}
	}
	char *ptrBase = (char*)malloc(sizeof(Account)+nSize);
	if( ptrBase == NULL ) {
		for( i = 0; i < nRep; i++ ) {
			freeReplyObject(pRep[i]);
		}
		return DBErrorNoSpace;
	}
	Account *pAcc = (Account*)ptrBase;
	memset(pAcc, 0, sizeof(Account));
	char *ptr = ptrBase;
	ptr += sizeof(Account);

	if( f & AccountName ) {
		if( pRep[nUserName]->type == REDIS_REPLY_STRING ) {
			pAcc->userName	= ptr;
			strcpy(ptr, pRep[nUserName]->str);
			ptr += pRep[nUserName]->len+1;
		} else {
			pAcc->userName	= NULL;
		}
		if( pRep[nFirstName]->type == REDIS_REPLY_STRING ) {
			pAcc->firstName	= ptr;
			strcpy(ptr, pRep[nFirstName]->str);
			ptr += pRep[nFirstName]->len+1;
		} else {
			pAcc->firstName	= NULL;
		}
		if( pRep[nSecondName]->type == REDIS_REPLY_STRING ) {
			pAcc->secondName	= ptr;
			strcpy(ptr, pRep[nSecondName]->str);
			ptr += pRep[nSecondName]->len+1;
		} else {
			pAcc->secondName = NULL;
		}
		if( pRep[nLastName]->type == REDIS_REPLY_STRING ) {
			pAcc->lastName	= ptr;
			strcpy(ptr, pRep[nLastName]->str);
			ptr += pRep[nLastName]->len+1;
		} else {
			pAcc->lastName	= NULL;
		}
	}
	if( f & AccountAuth ) {
		if( pRep[nAuth]->type == REDIS_REPLY_STRING ) {
			pAcc->auth	= ptr;
			strcpy(ptr, pRep[nAuth]->str);
			ptr += pRep[nAuth]->len+1;
		} else {
			pAcc->auth	= NULL;
		}
	}
	if( f & AccountMail ) {
		if( pRep[nMail]->type == REDIS_REPLY_STRING ) {
			pAcc->email	= ptr;
			strcpy(ptr, pRep[nMail]->str);
			ptr += pRep[nMail]->len+1;
		} else {
			pAcc->email	= NULL;
		}
	}
	if( f & AccountAddr ) {
		if( pRep[nAddr1]->type == REDIS_REPLY_STRING ) {
			pAcc->addr1	= ptr;
			strcpy(ptr, pRep[nAddr1]->str);
			ptr += pRep[nAddr1]->len+1;
		} else {
			pAcc->addr1 = NULL;
		}
		if( pRep[nAddr2]->type == REDIS_REPLY_STRING ) {
			pAcc->addr2	= ptr;
			strcpy(ptr, pRep[nAddr2]->str);
			ptr += pRep[nAddr2]->len+1;
		} else {
			pAcc->addr2 = NULL;
		}
		if( pRep[nAddr3]->type == REDIS_REPLY_STRING ) {
			pAcc->addr3	= ptr;
			strcpy(ptr, pRep[nAddr3]->str);
			ptr += pRep[nAddr3]->len+1;
		} else {
			pAcc->addr2 = NULL;
		}
	}
	if( f & AccountPhone ) {
		if( pRep[nPhone]->type == REDIS_REPLY_STRING ) {
			pAcc->phone	= ptr;
			strcpy(ptr, pRep[nPhone]->str);
			ptr += pRep[nPhone]->len+1;
		} else {
			pAcc->phone	= NULL;
		}
	}
	if( f & AccountStat ) {
		if( pRep[nLevel]->type == REDIS_REPLY_INTEGER ) {
			pAcc->level	= pRep[nLevel]->integer;
		} else {
			pAcc->level	= 0;
		}

		if( pRep[nCreate]->type == REDIS_REPLY_STRING ) {
			struct tm t;
			strptime(pRep[nCreate]->str, "%Y/%m/%d-%H:%M:%S", &t);
			pAcc->createDate	= timegm(&t);
		} else {
			pAcc->createDate	= NULL;
		}
		if( pRep[nLastDate]->type == REDIS_REPLY_STRING ) {
			struct tm t;
			strptime(pRep[nLastDate]->str, "%Y/%m/%d-%H:%M:%S", &t);
			pAcc->lastDate	= timegm(&t);
		} else {
			pAcc->lastDate	= NULL;
		}
		if( pRep[nMaxStorage]->type == REDIS_REPLY_INTEGER ) {
			pAcc->maxStorage	= pRep[nMaxStorage]->integer;
		} else {
			pAcc->maxStorage	= 0;
		}
		if( pRep[nUseStorage]->type == REDIS_REPLY_INTEGER ) {
			pAcc->useStorage	= pRep[nUseStorage]->integer;
		} else {
			pAcc->useStorage	= 0;
		}
	}
	for( i = 0; i < nRep; i++ ) {
		freeReplyObject(pRep[i]);
	}
	*account	= pAcc;
	return DBErrorOk;
}

DBError	RedisDB::setAccountStat(const char *id, time_t lastLogin)
{
	struct tm t;
	gmtime_r(&lastLogin, &t);
	char dateStr[100];
	strftime(dateStr, sizeof(dateStr), "%Y/%m/%d-%H:%M:%S", &t);

	redisReply *reply = (redisReply*)redisCommand(conn, "set aid:%s:lastDate %s", id, dateStr);
	REPLY_STATUS(reply);
	if(!IS_STATUS_OK(reply) ) {
		freeReplyObject(reply);
		return DBErrorSystem;
	}
	freeReplyObject(reply);
	return DBErrorOk;
}

DBError	RedisDB::getParticipationGroupList(const char *aid, size_t nFrom, RecList *pList)
{
	size_t i;
	size_t idSize = sizeof(char)*(SST_USERIDLEN+1)*pList->nCount;
	size_t ptSize = sizeof(char*)*pList->nCount;
	if( pList->nSize < sizeof(RecList) + idSize + ptSize ) {
		return DBErrorNoSpace;
	}
	char *pId = (char*)&pList->pValue[pList->nCount];
	for( i = 0; i < pList->nCount; i++ ) {
		pList->pValue[i]	= pId;
		pId += sizeof(char)*(SST_USERIDLEN+1);
	}

	redisReply *reply = (redisReply*)redisCommand(conn, "LRANGE aid:%s:group %d %d", aid, nFrom, nFrom+pList->nCount);
	REPLY_CHECK(reply, REDIS_REPLY_ARRAY);

	pList->nValues	= 0;
	size_t nAdd = 0;
	for( i = 0; i < reply->elements; i++ ) {
		if( reply->element[i]->type == REDIS_REPLY_STRING ) {
			strcpy(pList->pValue[nAdd], reply->element[i]->str);
			nAdd++;
		}
	}
	pList->nValues	= nAdd;
	freeReplyObject(reply);
	return DBErrorOk;
}

DBError RedisDB::participationGroupList(const char *aid, const char *gid)
{
	// lock
	LOCK_ACCOUNT(aid, 10);

	redisReply *reply = (redisReply*)redisCommand(conn, "multi");
	STATUS_CHECK_OK(reply, aid);

	reply = (redisReply*)redisCommand(conn, "sadd aid:%s:group %s", aid, gid);
	STATUS_CHECK_QUEUED(reply, aid);

	reply = (redisReply*)redisCommand(conn, "sadd gid:%s:member %s", gid, aid);
	STATUS_CHECK_QUEUED(reply, aid);

	reply = (redisReply*)redisCommand(conn, "exec");
	REPLY_CHECK(reply, REDIS_REPLY_ARRAY);
	freeReplyObject(reply);

	UNLOCK_ACCOUNT(aid);
	return DBErrorOk;
}

DBError RedisDB::getDeviceInfo(const char *id, Device *device)
{
	return DBErrorSystem;
}

DBError RedisDB::getServiceCount(size_t &n)
{
	bool null=false;
	long long int ret;
	if( !getInt(ret, null, NULL, "LLEN service") ) {
		return DBErrorSystem;
	}
	n = ret;
	return DBErrorOk;
}

DBError	RedisDB::getServiceList(size_t nFrom, RecList *pList)
{
	sentinel->log(LOG_DEBUG, "RedisDB::getServiceList 1");
	size_t i;
	size_t idSize = sizeof(char)*(SST_SERVICEIDLEN+1)*pList->nCount;
	size_t ptSize = sizeof(char*)*pList->nCount;
	if( pList->nSize < sizeof(RecList) + idSize + ptSize ) {
		sentinel->log(LOG_ERR, "RedisDB::getServiceList: parameter error. no space for service list. nSize=%d nCount=%d need=%d", pList->nSize, pList->nCount, sizeof(RecList) + idSize + ptSize);
		return DBErrorNoSpace;
	}
	char *pId = (char*)&pList->pValue[pList->nCount];
	for( i = 0; i < pList->nCount; i++ ) {
		pList->pValue[i]	= pId;
		pId += sizeof(char)*(SST_SERVICEIDLEN+1);
	}

	redisReply *reply = (redisReply*)redisCommand(conn, "LRANGE service %d %d", nFrom, nFrom+pList->nCount);
#ifdef _DEBUG
	sentinel->log(LOG_DEBUG, "r->type = %d", reply->type );
#endif
	REPLY_CHECK(reply, REDIS_REPLY_ARRAY);

	pList->nValues	= 0;
	size_t nAdd = 0;
	for( i = 0; i < reply->elements; i++ ) {
		if( reply->element[i]->type == REDIS_REPLY_STRING ) {
			strcpy(pList->pValue[nAdd], reply->element[i]->str);
			nAdd++;
		}
	}
	pList->nValues	= nAdd;
	freeReplyObject(reply);
	sentinel->log(LOG_DEBUG, "RedisDB::getServiceList ok");
	return DBErrorOk;
}

DBError RedisDB::getServiceInfo(const char *id, ServiceInfo **info)
{
	sentinel->log(LOG_DEBUG, "RedisDB::getServiceInfo 1");
	bool null[2] = {false, false};
	redisReply *pRep[2];

	if( !getStr(&pRep[0], null[0], "sid:x:name", "get sid:%s:name", id) ||
		!getStr(&pRep[1], null[1], "sid:x:owner", "get sid:%s:owner", id) ) {
		return DBErrorSystem;
	}

	sentinel->log(LOG_DEBUG, "RedisDB::getServiceInfo 2");
	size_t s = sizeof(ServiceInfo);
	s += sizeof(char)*(SST_SERVICEIDLEN+1);
	s += sizeof(char)*(pRep[0]->len+1);
	s += sizeof(char)*(pRep[1]->len+1);
	char *pBuff = (char*)malloc(s);
	if( !pBuff ) {
		size_t i;
		for( i = 0; i < 2; i++ ) {
			freeReplyObject(pRep[i]);
		}
		return DBErrorNoSpace;
	}
	sentinel->log(LOG_DEBUG, "RedisDB::getServiceInfo 3");
	memset(pBuff, 0, s);
	ServiceInfo *pInfo = (ServiceInfo*)pBuff;
	pBuff += sizeof(ServiceInfo);
	pInfo->serviceId	= pBuff;
	pBuff += SST_SERVICEIDLEN+1;
	pInfo->serviceName	= pBuff;
	pBuff += pRep[0]->len+1;
	pInfo->ownerId	= pBuff;
	pBuff += pRep[1]->len+1;
	strcpy(pInfo->serviceId, id);
	strcpy(pInfo->serviceName, pRep[0]->str);
	strcpy(pInfo->ownerId, pRep[1]->str);
	size_t i;
	for( i = 0; i < 2; i++ ) {
		freeReplyObject(pRep[i]);
	}
	sentinel->log(LOG_DEBUG, "RedisDB::getServiceInfo 4");

	bool nu=false;
	redisReply *v;
	if( !getStr(&v, nu, "sid:x:status", "get sid:%s:status", id) ) {
		free(pInfo);
		return DBErrorSystem;
	}
	pInfo->status	= (ServiceStatus)atoi(v->str);
	freeReplyObject(v);
	sentinel->log(LOG_DEBUG, "RedisDB::getServiceInfo 5");

	long long int lv;
	if( !getInt(lv, nu, "sid:x:group", "llen sid:%s:group", id) ) {
		free(pInfo);
		return DBErrorSystem;
	}
	pInfo->participationGroupCount	= (size_t)lv;
	sentinel->log(LOG_DEBUG, "RedisDB::getServiceInfo 6");

	if( !getInt(lv, nu, "sid:x:resource", "llen sid:%s:resource", id) ) {
		free(pInfo);
		return DBErrorSystem;
	}
	pInfo->resourceCount	= (size_t)lv;
	*info = pInfo;
	sentinel->log(LOG_DEBUG, "RedisDB::getServiceInfo ok");

	return DBErrorOk;
}

DBError RedisDB::getServiceStatus(const char *id, ServiceStatus &st)
{
	bool nu=false;
	long long int v;
	if( !getInt(v, nu, NULL, "get sid:%s:status", id) ) {
		return DBErrorSystem;
	}
	st	= (ServiceStatus)v;
	return DBErrorOk;
}

DBError RedisDB::getServiceGroupList(const char *sid, size_t nFrom, RecList *pList)
{
	size_t i;
	size_t idSize = sizeof(char)*(SST_USERIDLEN+1)*pList->nCount;
	size_t ptSize = sizeof(char*)*pList->nCount;
	if( pList->nSize < sizeof(RecList) + idSize + ptSize ) {
		return DBErrorNoSpace;
	}
	char *pId = (char*)&pList->pValue[pList->nCount];
	for( i = 0; i < pList->nCount; i++ ) {
		pList->pValue[i]	= pId;
		pId += sizeof(char)*(SST_USERIDLEN+1);
	}

	redisReply *reply = (redisReply*)redisCommand(conn, "LRANGE sid:%s:group %d %d", sid, nFrom, nFrom+pList->nCount);
	REPLY_CHECK(reply, REDIS_REPLY_ARRAY);

	pList->nValues	= 0;
	size_t nAdd = 0;
	for( i = 0; i < reply->elements; i++ ) {
		if( reply->element[i]->type == REDIS_REPLY_STRING ) {
			strcpy(pList->pValue[nAdd], reply->element[i]->str);
			nAdd++;
		}
	}
	pList->nValues	= nAdd;
	freeReplyObject(reply);
	return DBErrorOk;
}

DBError	RedisDB::getResourceList(const char *sid, size_t nFrom, RecList *pList)
{
	size_t i;
	size_t idSize = sizeof(char)*(SST_RESOURCEIDLEN+1)*pList->nCount;
	size_t ptSize = sizeof(char*)*pList->nCount;
	if( pList->nSize < sizeof(RecList) + idSize + ptSize ) {
		return DBErrorNoSpace;
	}
	char *pId = (char*)&pList->pValue[pList->nCount];
	for( i = 0; i < pList->nCount; i++ ) {
		pList->pValue[i]	= pId;
		pId += sizeof(char)*(SST_RESOURCEIDLEN+1);
	}

	redisReply *reply = (redisReply*)redisCommand(conn, "LRANGE sid:%s:resource %d %d", sid, nFrom, nFrom+pList->nCount);
	REPLY_CHECK(reply, REDIS_REPLY_ARRAY);

	pList->nValues	= 0;
	size_t nAdd = 0;
	for( i = 0; i < reply->elements; i++ ) {
		if( reply->element[i]->type == REDIS_REPLY_STRING ) {
			strcpy(pList->pValue[nAdd], reply->element[i]->str);
			nAdd++;
		}
	}
	pList->nValues	= nAdd;
	freeReplyObject(reply);
	return DBErrorOk;
}

DBError RedisDB::getResourceInfo(const char *rid, ResourceInfo **info)
{
	sentinel->log(LOG_DEBUG, "RedisDB::getResourceInfo 1");

	redisReply *pSid = NULL;
	bool		bSid = false;
	redisReply *pName = NULL;
	bool		bName = false;
	redisReply *pURL = NULL;
	bool		bURL = false;
	redisReply *pMethod = NULL;
	bool		bMethod  = false;
	redisReply *pParam = NULL;
	bool		bParam = false;
	redisReply *pHeader = NULL;
	bool		bHeader = false;
	redisReply *pStatus = NULL;
	bool		bStatus = false;

	if( !getStr(&pSid, bSid, 		"rid:@rid:sid", "get rid:%s:sid", rid) ||
		!getStr(&pName, bName, 		"rid:@rid:name", "get rid:%s:name", rid) ||
		!getStr(&pURL, bURL, 		"rid:@rid:URL", "get rid:%s:URL", rid) ||
		!getStr(&pMethod, bMethod,	"rid:@rid:method", "get rid:%s:method", rid) ||
		!getStr(&pParam, bParam, 	"rid:@rid:paramPattern", "get rid:%s:paramPattern", rid) ||
		!getStr(&pHeader, bHeader, 	"rid:@rid:headerPattern", "get rid:%s:headerPattern", rid) ||
		!getStr(&pStatus, bStatus, 	"rid:@rid:status", "get rid:%s:status", rid) ) {
		if( pSid ) freeReplyObject(pSid);
		if( pName ) freeReplyObject(pName);
		if( pURL ) freeReplyObject(pURL);
		if( pMethod ) freeReplyObject(pMethod);
		if( pParam) freeReplyObject(pParam);
		if( pHeader ) freeReplyObject(pHeader);
		if( pStatus) freeReplyObject(pStatus);
		return DBErrorSystem;
	}
	sentinel->log(LOG_DEBUG, "RedisDB::getResourceInfo 2");

	size_t s = sizeof(ResourceInfo);
	s += sizeof(char)* pSid->len+1;
	s += sizeof(char)* strlen(rid)+1;
	s += sizeof(char)* pName->len+1;
	s += sizeof(char)* pURL->len+1;
	s += sizeof(char)* pMethod->len+1;
	s += sizeof(char)* pParam->len+1;
	s += sizeof(char)* pHeader->len+1;
	char *pBuff = (char*)malloc(s);
	if( !pBuff ) {
		freeReplyObject(pSid);
		freeReplyObject(pName);
		freeReplyObject(pURL);
		freeReplyObject(pMethod);
		freeReplyObject(pParam);
		freeReplyObject(pHeader);
		freeReplyObject(pStatus);
		return DBErrorNoSpace;
	}

	sentinel->log(LOG_DEBUG, "RedisDB::getResourceInfo 3");
	memset(pBuff, 0, s);
	ResourceInfo *pInfo = (ResourceInfo*)pBuff;
	pBuff += sizeof(ResourceInfo);
	pInfo->serviceId	= pBuff;	pBuff += pSid->len+1;
	pInfo->resourceId	= pBuff;	pBuff += strlen(rid)+1;
	pInfo->resourceName	= pBuff;	pBuff += pName->len+1;
	pInfo->URL			= pBuff;	pBuff += pURL->len+1;
	pInfo->method		= pBuff;	pBuff += pMethod->len+1;
	pInfo->paramPattern	= pBuff;	pBuff += pParam->len+1;
	pInfo->headerPattern= pBuff;	pBuff += pHeader->len+1;

	strcpy(pInfo->serviceId, pSid->str);
	strcpy(pInfo->resourceId, rid);
	strcpy(pInfo->resourceName, pName->str);
	strcpy(pInfo->URL, pURL->str);
	strcpy(pInfo->method, pMethod->str);
	strcpy(pInfo->paramPattern, pParam->str);
	strcpy(pInfo->headerPattern, pHeader->str);
	pInfo->status	= (ResourceStatus)atoi(pStatus->str);

	freeReplyObject(pSid);
	freeReplyObject(pName);
	freeReplyObject(pURL);
	freeReplyObject(pMethod);
	freeReplyObject(pParam);
	freeReplyObject(pHeader);
	freeReplyObject(pStatus);

	*info = pInfo;
	sentinel->log(LOG_DEBUG, "RedisDB::getResourceInfo ok");

	return DBErrorOk;
}

}
