/*
 * processor.cpp
 *
 *  Created on: 2012/07/04
 *      Author: Yasuoki
 */

#include "processor.h"
#include "sentinel.h"
#include "queue.h"
#include "commands.h"
#include "buffer.h"
#include "database.h"
#include "pjson.h"
#include <syslog.h>
#include <errno.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <curl/curl.h>

namespace SST {

ProcessorPool::Processor::Processor()
{
	pool		= NULL;
	doStop		= false;
	threadHandlePtr	= NULL;
}

ProcessorPool::Processor::~Processor()
{
}

bool ProcessorPool::Processor::run(ProcessorPool *obj)
{
	pool	= obj;

	if( pthread_create( &threadHandle, NULL, threadEntry, this) != 0 ) {
		pool->sentinel->log(LOG_ERR, "start queue process thread is failed");
		return false;
	}
	threadHandlePtr	= &threadHandle;
	return true;
}

void ProcessorPool::Processor::stop()
{
	doStop	= true;
	if( threadHandlePtr ) {
		pool->sentinel->getQueue().put(new QnEnd());
	}
}

void ProcessorPool::Processor::wait()
{
	void *ret = NULL;
	if( threadHandlePtr ) {
		pthread_join(*threadHandlePtr, &ret);
		threadHandlePtr	= NULL;
	}
	doStop	= false;
}

void *ProcessorPool::Processor::threadEntry(void *ptr)
{
	void *ret = ((Processor*)ptr)->mainLoop();
	((Processor*)ptr)->threadHandlePtr	= NULL;
	return ret;
}

void *ProcessorPool::Processor::mainLoop()
{
	pool->sentinel->log(LOG_INFO, "queue processor thread %x is start", threadHandle);

	while( !doStop ) {
		pool->sentinel->log(LOG_DEBUG, "queue processor %x wait in", threadHandle);
		QueueNode *q = pool->sentinel->getQueue().get();
		pool->sentinel->log(LOG_DEBUG, "queue processor %x wait out", threadHandle);
		if( q == NULL ) {
			pool->sentinel->log(LOG_ERR, "queue error");
		}
		switch( q->getType() ) {
		case QueueEnd:
			delete q;
			break;
		case QueueAccept:
			acceptProc(q);
			break;
		case QueueSessionData:
			sessionDataProc(q);
			break;
		case QueueSessionCommand:
			if( !commandProc((QnSessionCommand*)q) ) {
				Session *con = ((QnSessionCommand*)q)->con;
				if( con->getStatus() != SessionStatusLogin ) {
					pool->sentinel->log(LOG_DEBUG, "login failed. close connection" );
					pool->sentinel->getSessionPool().freeSession(con);
				}
			}
			break;
		case QueueResponse:
			if( !responseProc((QnResponse*)q) ) {

			}
			break;
		default:
			pool->sentinel->log(LOG_ERR, "queue type error %d", q->getType() );
			break;
		}
		delete q;
	}

	pool->sentinel->log(LOG_ERR, "queue processor thread %x is exit", threadHandle);
	return NULL;
}

void  ProcessorPool::Processor::cmdError(Session *con, CMDStatus code, const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	char buf[1024];
	char *out = buf;
	int outSize = sizeof(buf);
	int size = vsnprintf(out, outSize, fmt, ap);
	if( size > outSize ) {
		outSize = size + 10;
		out = (char*)malloc(outSize);
		if( !out ) {
			pool->sentinel->log(LOG_ERR, "processor::cmdError no more memory");
			return;
		}
		va_end(ap);
		va_start(ap, fmt);
		size = vsnprintf(out, outSize, fmt, ap);
		if( size > outSize ) {
			pool->sentinel->log(LOG_ERR, "processor::cmdError no more memory");
			free(out);
			return;
		}
	}

	pjson::builder jb;
	if( !jb.init(256) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result", 6) ||
		!jb.valueInt(code) ||
		!jb.addObjectProp("message", 7) ||
		!jb.valueString(out) ||
		!jb.endObject() ) {
		pool->sentinel->log(LOG_ERR, "processor::cmdError json builder failed code=%d", jb.getError());
		if( out != buf ) {
			free(out);
		}
		return;
	}
	if( out != buf ) {
		free(out);
	}
	response(con, jb);
}

bool  ProcessorPool::Processor::response(Session *con, pjson::builder &jb)
{
	pjson::json *js = jb.detouch();
	if( !js ) {
		cmdError(con, CMDErrRuntime, "JSON Processor error. code = %d", jb.getError());
		pool->sentinel->log(LOG_INFO, "processor::response json builder error %d", jb.getError());
		return false;
	}
	CyclicBuffer b;
	if( !js->getText(&b) ) {
		pool->sentinel->log(LOG_DEBUG, "processor::response json builder error %d", jb.getError());
		delete js;
		return false;
	}
	delete js;
	b.add("\r\n", 2);
	int fd = con->getDiscriptor();
	size_t bs = b.size();
	write(fd, b.getPtr(), bs);
	return true;
}

bool ProcessorPool::Processor::loginPrompt(Session *con)
{
	pool->sentinel->log(LOG_DEBUG, "processor::loginPrompt %x begin", threadHandle );

	char *nonce = con->createNonce();
	pjson::builder jb;
	if( !jb.init(256) ||
		!jb.beginObject() ) {
		pool->sentinel->log(LOG_ERR, "processor::loginPrompt json builder failed. code=%d", jb.getError());
		return false;
	}
	if(	!jb.addObjectProp("auth", 4) ||
		!jb.beginObject() ||
		!jb.addObjectProp("nonce", 5) ||
		!jb.valueString(nonce) ||
		!jb.addObjectProp("algorithm", 9) ||
		!jb.valueString("MD5") ||
		!jb.endObject() ) {
		pool->sentinel->log(LOG_ERR, "processor::loginPrompt json builder failed. code=%d", jb.getError());
		return false;
	}

	Conf *cf = pool->sentinel->getConf();
	if( cf->serverName ) {
		if( !jb.addObjectProp("server", 6) ||
			!jb.beginObject() ||
			!jb.addObjectProp("host", 4) ||
			!jb.valueString(cf->serverName) ||
			!jb.addObjectProp("port", 4) ||
			!jb.valueInt(cf->port) ||
			!jb.addObjectProp("load", 4) ||
			!jb.valueInt(pool->sentinel->getLoad()) ||
			!jb.endObject() ) {
			pool->sentinel->log(LOG_ERR, "processor::loginPrompt json builder failed. code=%d", jb.getError());
			return false;
		}
	}
	if( !jb.endObject() ) {
		pool->sentinel->log(LOG_ERR, "processor::loginPrompt json builder failed. code=%d", jb.getError());
		return false;
	}
	if( !response(con, jb) ) {
		pool->sentinel->log(LOG_ERR, "processor::loginPrompt json response failed.");
		return false;
	}
	pool->sentinel->log(LOG_DEBUG, "processor::loginPrompt %x end", threadHandle );
	return true;
}

bool ProcessorPool::Processor::acceptProc(QueueNode *q)
{
	pool->sentinel->log(LOG_DEBUG, "processor::acceptProc %x %d begin", threadHandle, ((QnAccept*)q)->fd );
	int cfd = ((QnAccept*)q)->fd;
	sockaddr_in addr = ((QnAccept*)q)->addr;

	Session *con = pool->sentinel->getSessionPool().getSession();
	if( con == NULL ) {
		::write(cfd, "500\n", 4);
		pool->sentinel->log(LOG_WARNING, "no more connection");
		pool->sentinel->getSessionPool().freeSession(con);
		return false;
	}
	if( !con->open(pool->sentinel, cfd, &addr) ) {
		::write(cfd, "500\n", 4);
		pool->sentinel->log(LOG_ERR, "can not connection open");
		pool->sentinel->getSessionPool().freeSession(con);
		return false;
	}

	if( !loginPrompt(con) ) {
		::write(cfd, "500\n", 4);
		pool->sentinel->log(LOG_ERR, "can not connection open");
		pool->sentinel->getSessionPool().freeSession(con);
		return false;
	}

	con->setStatus(SessionStatusConnect);

	ListenerPool &listener = pool->sentinel->getListener();
	if( !listener.waitData(con) ) {
		::write(cfd, "500\n", 4);
		pool->sentinel->getSessionPool().freeSession(con);
		return false;
	}
	pool->sentinel->log(LOG_DEBUG, "processor::acceptProc %x con=%x %d ok", threadHandle, con, ((QnAccept*)q)->fd );
	return true;
}

bool ProcessorPool::Processor::sessionDataProc(QueueNode *q)
{
	pool->sentinel->log(LOG_DEBUG, "processor::sessionDataProc %x begin", threadHandle );
	Session *con = ((QnSessionData*)q)->con;

	Buffer *readBuff = con->getReadBuffer();
	char buff[256];
	int fd = con->getDiscriptor();
	while(1) {
		ssize_t len = ::read(fd, buff, sizeof(buff));
		if( len == -1 ) {
			if( errno == EAGAIN || errno == EWOULDBLOCK ) break;
			pool->sentinel->log(LOG_ERR, "i/o error %d %s", errno, strerror(errno) );
			pool->sentinel->getSessionPool().freeSession(con);
			return false;
		}
		if( len == 0 ) {
			if( con->getUserId() ) {
				fprintf(stderr, "user %s logout\n", con->getUserId());
			}
			fprintf(stderr, "connection closed.\n");
			pool->sentinel->getSessionPool().freeSession(con);
			return false;
		}
		if( !readBuff->add(buff, len) ) {
			pool->sentinel->log(LOG_ERR, "buffer full");
			pool->sentinel->getSessionPool().freeSession(con);
			return false;
		}
		if( len < (ssize_t)sizeof(buff) ) break;
	}
	pool->sentinel->getListener().waitData(con);

	// empty line?
	bool cmdEnd=false;
	size_t s = readBuff->size();
	if( s >= 2 ) {
		char buff[10];
		size_t rs = readBuff->getAt(buff, s-2, 3);
#ifdef _DEBUG
		pool->sentinel->log(LOG_DEBUG, "command parse 1 %d/%d", rs,s);
		for(size_t n=0; n < rs; n++) {
			pool->sentinel->log(LOG_DEBUG, "command parse %d:[%x]", n, buff[n]);
		}
#endif
		if( rs == 2 && strcmp("\r\n", buff) == 0 ) {
			if( s == 2 ) {
				cmdEnd = true;
			} else if( s >= 4 ) {
				rs = readBuff->getAt(buff, s-4, 3);
				if( rs == 2 && strcmp("\r\n", buff) == 0 ) {
					cmdEnd = true;
				}
			}
		}
	}
#ifdef _DEBUG
	if( !cmdEnd ) {
		pool->sentinel->log(LOG_DEBUG, "command continue");
	}
#endif
	if( cmdEnd ) {
		QueueNode *qn = new QnSessionCommand(con);
		if( qn == NULL ) {
			pool->sentinel->getSessionPool().freeSession(con);
			pool->sentinel->getSessionPool().freeSession(con);
			return false;
		}
		if( !pool->sentinel->getQueue().put(qn) ) {
			pool->sentinel->getSessionPool().freeSession(con);
			pool->sentinel->getSessionPool().freeSession(con);
			return false;
		}
	}

	pool->sentinel->log(LOG_DEBUG, "processor::sessionDataProc %x end", threadHandle );
	return true;
}

bool ProcessorPool::Processor::commandProc(QnSessionCommand *q)
{
	pool->sentinel->log(LOG_DEBUG, "processor::commandProc %x begin", threadHandle );
	Session *con = q->con;

	Buffer *r = con->getReadBuffer();
	size_t readSize = r->size();
	const char *readData = r->getPtr();

	char msg[1024];
	const char *room = con->getChatRoom();
	if( room ) {
		const char *pUser = con->getUserId();
		SessionPool &s = pool->sentinel->getSessionPool();
		Session *session = s.getUse();
		while(session) {
			if( session != con ) {
				const char *room2 = session->getChatRoom();
				if( room2 && strcmp(room, room2)== 0 ) {
					int fd = session->getDiscriptor();
					sprintf(msg, "%s: %s", pUser, readData);
					write(fd, msg, strlen(msg));
				}
			}
			session = s.getNext(session);
		}
		if( strcmp(readData,"bye\n") == 0 ||
			strcmp(readData,"bye\r") == 0 ||
			strcmp(readData,"bye\r\n") == 0 ) {
			con->exitChatRoom();
		}
		pool->sentinel->log(LOG_DEBUG, "processor::commandProc %x end", threadHandle );
		return true;
	}

	pjson::parser ps;
	pjson::json *js = ps.parse(readData,readSize);

	if( js == NULL ) {
		cmdError(con, CMDErrMessage, "processor::commandProc command format error. ps=%d", ps.getErrorPos());
		delete js;
		return false;
	}

	const char *cmd = NULL;

	pjson::value *v = js->getValue();
	if( v->vt == pjson::vt_string ) {
		cmd = v->vString;
	} else if( v->vt == pjson::vt_object ) {
		cmd = v->vObject->props[0]->name;
	} else {
		cmdError(con, CMDErrParameter, "processor::commandProc command parameter error.");
		delete js;
		return false;
	}

	if( con->getStatus() == SessionStatusConnect ) {
		if( strcmp(cmd, "login") == 0 ) {
			if( !cmdLoginProc(q, js) ) {
				delete js;
				return false;
			}
		} else {
			cmdError(con, CMDErrNotLogin, "not login");
			delete js;
			return false;
		}
	} else if( con->getStatus() == SessionStatusLogin ) {

		if( strcmp(cmd, "chat") == 0 ) {
			if( v->vObject->n <= 1 || v->vObject->props[1]->val.vt != pjson::vt_string ) {
				pool->sentinel->log(LOG_INFO, "processor::waitProc chat command parametor error" );
				delete js;
				return false;
			}
			const char *pRoom = v->vObject->props[1]->val.vString;
			con->setChatRoom(pRoom);
		} else if( strcmp(cmd, "user") == 0 ) {
			if( !cmdUserProc(q, js) ) {
				delete js;
				return false;
			}
		} else if( strcmp(cmd, "users") == 0 ) {
			if( !cmdUsersProc(q, js)) {
				delete js;
				return false;
			}
		} else if( strcmp(cmd, "groups") == 0 ) {
			if( !cmdGroupsProc(q, js)) {
				delete js;
				return false;
			}
		} else if( strcmp(cmd, "services") == 0 ) {
			if( !cmdServicesProc(q, js) ) {
				delete js;
				return false;
			}
		} else if( strcmp(cmd, "resources") == 0 ) {
			if( !cmdResourcesProc(q, js) ) {
				delete js;
				return false;
			}
		} else if( strcmp(cmd, "resource") == 0 ) {
			if( !cmdResourceInfoProc(q, js) ) {
				delete js;
				return false;
			}
		} else if( strcmp(cmd, "joingroup") == 0 ) {
			if( !cmdJoinGroupProc(q, js) ) {
				delete js;
				return false;
			}
		} else if( strcmp(cmd, "test") == 0 ) {
			if( !cmdTestProc(q, js) ) {
				delete js;
				return false;
			}
		} else {
			cmdError(con, CMDErrCommand, "unknown command. %s", cmd);
			delete js;
			return false;
		}
	} else {
		cmdError(con, CMDErrNotLogin, "not login");
		delete js;
		return false;
	}
	delete js;
	pool->sentinel->log(LOG_DEBUG, "processor::commandProc %x end", threadHandle );
	return true;
}

bool ProcessorPool::Processor::cmdLoginProc(QnSessionCommand *q, pjson::json *js)
{
	pool->sentinel->log(LOG_DEBUG, "processor::cmdLoginProc %x begin", threadHandle );
	Session *con = q->con;

	if( con->getStatus() == SessionStatusWait ) {
		cmdError(con, CMDErrHierarchy, "already login");
		return false;
	}

	pool->sentinel->log(LOG_DEBUG, "processor::cmdLoginProc json parse ok" );

	pjson::value *vUser = NULL;
	pjson::value *vAuth = NULL;
	pjson::value *vResponse = NULL;
	pjson::value *vNonce = NULL;
	pjson::value *vCnonce = NULL;
	pjson::value *vAlgo = NULL;

	pjson::value *vLogin = js->get("login");
	if( !vLogin ) {
		cmdError(con, CMDErrCommand, "login");
		return false;
	}
	vUser = js->get("user", vLogin);
	if( !vUser || vUser->vt != pjson::vt_string ) {
		cmdError(con, CMDErrParameter, "login.user type");
		return false;
	}

	pool->sentinel->log(LOG_DEBUG, "processor::cmdLoginProc json auth data ok" );

	Database *db = pool->sentinel->getDatabase();
	char aid[SST_ACCOUNID_LEN+1];
	DBError e = db->getAccountId(vUser->vString, aid);
	pool->sentinel->log(LOG_DEBUG, "processor::cmdLoginProc getAccountId error code=%d", e);
	if( e == DBErrorNotFound ) {
		cmdError(con, CMDErrNotFound, "user not found");
		return false;
	}
	if( e != DBErrorOk ) {
		cmdError(con, CMDErrDatabase, "code = %d", e);
		return false;
	}
	pool->sentinel->log(LOG_DEBUG, "processor::cmdLoginProc getAccountId = %s", aid);

	vAuth	= js->get("auth", vLogin);
	if( vAuth ) {
		if( vAuth->vt != pjson::vt_object ) {
			cmdError(con, CMDErrParameter, "login.auth type");
			return false;
		}
		vResponse 	= js->get("response", vAuth);
		if( !vResponse || vResponse->vt != pjson::vt_string ) {
			cmdError(con, CMDErrParameter, "login.auth.response type");
			return false;
		}
		vNonce		= js->get("nonce", vAuth);
		if( !vNonce || vNonce->vt != pjson::vt_string ) {
			cmdError(con, CMDErrParameter, "login.auth.nonce type");
			return false;
		}
		vCnonce		= js->get("cnonce", vAuth);
		if( !vCnonce || vCnonce->vt != pjson::vt_string ) {
			cmdError(con, CMDErrParameter, "login.auth.cnonce type");
			return false;
		}
		vAlgo		= js->get("algorithm", vAuth);
		if( !vAlgo || vAlgo->vt != pjson::vt_string ) {
			cmdError(con, CMDErrParameter, "login.auth.algorithm type");
			return false;
		}
		if( strcmp(con->getLastNonce(), vNonce->vString) != 0 ) {
			cmdError(con, CMDErrPermission, "login nonce error");
			return false;
		}
		if( strcmp(vAlgo->vString, "MD5") != 0 ) {
			cmdError(con, CMDErrPermission, "login algorithm error");
			return false;
		}
	}

	Account *pAcc;
	e = db->getAccountInfo(aid, (AccountField)(AccountAuth|AccountName|AccountStat), &pAcc);
	if( e != DBErrorOk ) {
		cmdError(con, CMDErrDatabase, "code = %d", e);
		return false;
	}
	if( pAcc->auth != NULL ) {
		if( vAuth == NULL ) {
			pool->sentinel->log(LOG_INFO, "processor::cmdLoginProc permission error" );
#ifndef _DEBUG
			free(pAcc);
			cmdError(con, CMDErrPermission, "user or password is unmatched");
			return false;
#endif
		}
		if( !vResponse || strcmp(pAcc->auth, vResponse->vString) != 0 ) {
			pool->sentinel->log(LOG_INFO, "processor::cmdLoginProc permission error" );
#ifndef _DEBUG
			cmdError(con, CMDErrPermission, "user or password is unmatched");
			free(pAcc);
			return false;
#endif
		}
	}
	char lastDate[100];
	if( pAcc->lastDate ) {
		struct tm t;
		gmtime_r(&pAcc->lastDate, &t);
		strftime(lastDate, sizeof(lastDate), "%Y/%m/%d-%H:%M:%S", &t);
	} else {
		lastDate[0] = 0;
	}

	pjson::builder jb;
	if( !jb.init(1024) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result") ||
		!jb.valueInt(0) ||
		!jb.addObjectProp("userName") ||
		!jb.valueString(pAcc->userName) ||
		!jb.addObjectProp("lastDate") ||
		!jb.valueString(lastDate) ||
		!jb.endObject()) {
		cmdError(con, CMDErrRuntime, "json builder code=%d", jb.getError());
		free(pAcc);
		return false;
	}
	free(pAcc);

	time_t ct = time(NULL);
	db->setAccountStat(aid, ct);
	con->setUserId((char*)vUser->vString);
	con->setAccountId(aid);
	con->setStatus(SessionStatusLogin);
	pool->sentinel->log(LOG_INFO, "processor::cmdLoginProc user %s is logged in", vUser->vString );

	if( !response(con, jb) ) return false;
	pool->sentinel->log(LOG_DEBUG, "processor::cmdLoginProc %x end", threadHandle );
	return true;
}

bool ProcessorPool::Processor::cmdUserProc(QnSessionCommand *q, pjson::json *js)
{
	pool->sentinel->log(LOG_DEBUG, "processor::cmdUserProc begin" );
	Session *con = q->con;
	Database *db = pool->sentinel->getDatabase();
	DBError e;

	pjson::value *vCmd = js->get("user");

	char aidBuff[SST_ACCOUNID_LEN+1];
	const char *aid = NULL;
	if( vCmd ) {
		pjson::value *vAcc = js->get("account", vCmd);
		if( !vAcc ) {
			pjson::value *vUsr = js->get("user", vCmd);
			if( !vUsr ) {
				aid = con->getAccountId();
			} else {
				if( vUsr->vt != pjson::vt_string ) {
					cmdError(con, CMDErrParameter, "user.user type");
					return false;
				}
				if( strcmp(vUsr->vString, "self") == 0 ) {
					aid = con->getAccountId();
				} else {
					e = db->getAccountId(vUsr->vString, aidBuff);
					if( e != DBErrorOk ) {
						cmdError(con, CMDErrDatabase, "code=%d", e);
						return false;
					}
					aid = aidBuff;
				}
			}
		} else {
			if( vAcc->vt != pjson::vt_string ) {
				cmdError(con, CMDErrParameter, "user.account type");
				return false;
			}
			aid = vAcc->vString;
		}
	} else {
		aid = con->getAccountId();
	}

	pool->sentinel->log(LOG_DEBUG, "processor::cmdUserProc aid=%s", aid );

	int field = AccountAll;
	pjson::value *vField = js->get("field", vCmd);
	if( vField ) {
		if( vField->vt != pjson::vt_array ) {
			cmdError(con, CMDErrParameter, "user.field type");
			return false;
		}
		field = 0;
		size_t i;
		for( i = 0; i < vField->vArray->n; i++ ){
			if( vField->vArray->values[i]->vt != pjson::vt_string ) {
				cmdError(con, CMDErrParameter, "user.filed[%d] type", i);
				return false;
			}
			if( strcmp(vField->vArray->values[i]->vString, "name") == 0 )
				field |= AccountName;
			else if( strcmp(vField->vArray->values[i]->vString, "auth") == 0 )
				field |= AccountAuth;
			else if( strcmp(vField->vArray->values[i]->vString, "mail") == 0 )
				field |= AccountMail;
			else if( strcmp(vField->vArray->values[i]->vString, "addr") == 0 )
				field |= AccountAddr;
			else if( strcmp(vField->vArray->values[i]->vString, "phone") == 0 )
				field |= AccountPhone;
			else if( strcmp(vField->vArray->values[i]->vString, "stat") == 0 )
				field |= AccountStat;
			else if( strcmp(vField->vArray->values[i]->vString, "all") == 0 )
				field |= AccountAll;
			else {
				cmdError(con, CMDErrParameter, "unknown value user.field[%d]", i);
				return false;
			}
		}
	}

	Account *acc;
	e = db->getAccountInfo(aid, (AccountField)field, &acc);
	if( e != DBErrorOk ) {
		cmdError(con, CMDErrDatabase, "code=%d", e);
		return false;
	}

	pjson::builder jb;
	if( !jb.init(1024) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result") ||
		!jb.valueInt(0) ||
		!jb.addObjectProp("account") ||
		!jb.valueString(aid) ) {
		cmdError(con, CMDErrRuntime, "json builder initalize failed. code=%d", jb.getError());
		free(acc);
		return false;
	}
	if( acc->userName ) {
		if( !jb.addObjectProp("userName") ||
			!jb.valueString(acc->userName) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->auth ) {
		if( !jb.addObjectProp("auth") ||
			!jb.valueString(acc->auth) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->firstName ) {
		if( !jb.addObjectProp("firstName") ||
			jb.valueString(acc->firstName) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->secondName ) {
		if( !jb.addObjectProp("secondName") ||
			!jb.valueString(acc->secondName) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->lastName ) {
		if( !jb.addObjectProp("lastName") ||
			!jb.valueString(acc->lastName) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->email ) {
		if( !jb.addObjectProp("email") ||
			!jb.valueString(acc->email) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->addr1 ) {
		if( !jb.addObjectProp("addr1") ||
			!jb.valueString(acc->addr1) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->addr2 ) {
		if( !jb.addObjectProp("addr2") ||
			!jb.valueString(acc->addr2) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->addr3 ) {
		if( !jb.addObjectProp("addr2") ||
			!jb.valueString(acc->addr2) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->phone ) {
		if( !jb.addObjectProp("phone") ||
			!jb.valueString(acc->phone) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->level ) {
		if( !jb.addObjectProp("level") ||
			!jb.valueInt(acc->level) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->maxStorage ) {
		if( !jb.addObjectProp("maxStorage") ||
			!jb.valueInt(acc->maxStorage) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->useStorage ) {
		if( !jb.addObjectProp("useStorage") ||
			!jb.valueInt(acc->useStorage) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->createDate ) {
		struct tm t;
		gmtime_r(&acc->createDate, &t);
		char tmp[100];
		strftime(tmp, sizeof(tmp), "%Y/%m/%d-%H:%M:%S", &t);
		if( !jb.addObjectProp("createDate") ||
			!jb.valueString(tmp) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	if( acc->lastDate ) {
		struct tm t;
		gmtime_r(&acc->lastDate, &t);
		char tmp[100];
		strftime(tmp, sizeof(tmp), "%Y/%m/%d-%H:%M:%S", &t);
		if( !jb.addObjectProp("lastDate") ||
			!jb.valueString(tmp) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(acc);
			return false;
		}
	}
	jb.endObject();
	free(acc);
	if( !response(con, jb) ) return false;
	pool->sentinel->log(LOG_DEBUG, "processor::cmdUserProc end" );
	return true;
}

bool ProcessorPool::Processor::cmdUsersProc(QnSessionCommand *q, pjson::json *js)
{
	pool->sentinel->log(LOG_DEBUG, "processor::cmdUsersProc begin" );
	Session *con = q->con;

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result",6) ||
		!jb.valueInt(0) ||
		!jb.addObjectProp("users",5)  ) {
		cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
		return false;
	}
	SessionPool &s = pool->sentinel->getSessionPool();
	Session *session = s.getUse();
	int n=0;
	while(session) {
		if( n == 0 ) {
			if( !jb.beginArray() ) {
				cmdError(con, CMDErrRuntime, "json builder begin array failed. code=%d", jb.getError());
				return false;
			}
		}
		if( !jb.addArrayContent() ||
			!jb.valueString(session->getUserId()) ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			return false;
		}
		session = s.getNext(session);
		n++;
	}
	if( n ) {
		if( !jb.endArray() ) {
			cmdError(con, CMDErrRuntime, "json builder end array failed. code=%d", jb.getError());
			return false;
		}
	} else {
		if( !jb.valueString("empty") ) {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			return false;
		}
	}
	if( !jb.endObject() ) {
		cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
		return false;
	}
	if( !response(con, jb) ) return false;
	pool->sentinel->log(LOG_DEBUG, "processor::cmdUsersProc end" );
	return true;
}

bool ProcessorPool::Processor::cmdGroupsProc(QnSessionCommand *q, pjson::json *js)
{
	pool->sentinel->log(LOG_DEBUG, "processor::cmdGroupsProc begin" );
	Session *con = q->con;
	Database *db = pool->sentinel->getDatabase();

	RecList *pList = (RecList*)malloc(sizeof(RecList)+sizeof(char)*(SST_USERIDLEN+1)*32+sizeof(char*)*32);
	pList->nCount	= 32;
	pList->nSize	= sizeof(RecList)+sizeof(char)*(SST_USERIDLEN+1)*32+sizeof(char*)*32;
	pList->nValues	= 0;
	size_t nFrom = 0;

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result",6) ||
		!jb.valueInt(0) ||
		!jb.addObjectProp("groups", 6) ) {
		cmdError(con, CMDErrRuntime, "json builder init failed. code=%d", jb.getError());
		free(pList);
		return false;
	}
	DBError e;
	do {
		e = db->getGroupList(nFrom, pList);
		if( e != DBErrorOk ) {
			cmdError(con, CMDErrDatabase, "code=%d", e);
			free(pList);
			return false;
		}
		size_t i;
		for( i = 0; i < pList->nValues; i++ ) {
			if( nFrom == 0 ) {
				if( !jb.beginArray() ) {
					cmdError(con, CMDErrRuntime, "json builder begin array failed. code=%d", jb.getError());
					free(pList);
					return false;
				}
			}
			if( !jb.addArrayContent() ||
				!jb.valueString(pList->pValue[i]) ) {
				cmdError(con, CMDErrRuntime, "json builder add array failed. code=%d", jb.getError());
				free(pList);
				return false;
			}
		}
		nFrom += pList->nValues;
	} while(pList->nValues == pList->nCount);
	free(pList);

	if( nFrom ) {
		if( !jb.endArray()  ) {
			cmdError(con, CMDErrRuntime, "json builder end array failed. code=%d", jb.getError());
			return false;
		}
	} else {
		if( !jb.valueString("empty") ) {
			cmdError(con, CMDErrRuntime, "json builder end array failed. code=%d", jb.getError());
			return false;
		}
	}
	if( !jb.endObject() ) {
		cmdError(con, CMDErrRuntime, "json builder end array failed. code=%d", jb.getError());
		return false;
	}
	if( !response(con, jb) ) return false;
	pool->sentinel->log(LOG_DEBUG, "processor::cmdGroupsProc end" );
	return true;
}

bool ProcessorPool::Processor::cmdServicesProc(QnSessionCommand *q, pjson::json *js)
{
	pool->sentinel->log(LOG_DEBUG, "processor::cmdServiceProc begin" );
	Session *con = q->con;
	Services &sv = pool->sentinel->getServices();

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result",6) ||
		!jb.valueInt(0) ) {
		cmdError(con, CMDErrRuntime, "json builder init failed. code=%d", jb.getError());
		return false;
	}
	size_t n = sv.getServiceCount();
	size_t i;
	for( i = 0; i < n; i++ ) {
		Service *s = sv.getService(i);
		ServiceInfo *info = s->getServiceInfo();
		if( i == 0 ) {
			if( !jb.addObjectProp("services", 6) ||
				!jb.beginArray() ) {
				cmdError(con, CMDErrRuntime, "json builder begin array failed. code=%d", jb.getError());
				return false;
			}
		}
		if( !jb.addArrayContent() ||
			!jb.beginObject() ||
			!jb.addObjectProp("serviceId", 9) ||
			!jb.valueString(info->serviceId) ||
			!jb.addObjectProp("serviceName", 11) ||
			!jb.valueString(info->serviceName) ||
			!jb.addObjectProp("ownerId", 7) ||
			!jb.valueString(info->ownerId) ||
			!jb.addObjectProp("resourceCount", 13) ||
			!jb.valueInt(info->resourceCount) ||
			!jb.addObjectProp("participationGroupCount", 23) ||
			!jb.valueInt(info->participationGroupCount) ||
			!jb.addObjectProp("status", 6) ||
			!jb.valueInt(info->status) ||
			!jb.endObject() ) {
			cmdError(con, CMDErrRuntime, "json builder property set failed. code=%d", jb.getError());
			return false;
		}
	}
	if( i ) {
		if( !jb.endArray() ) {
			cmdError(con, CMDErrRuntime, "json builder end array failed. code=%d", jb.getError());
			return false;
		}
	}
	if( !jb.endObject() ) {
		cmdError(con, CMDErrRuntime, "json builder end object failed. code=%d", jb.getError());
		return false;
	}
	if( !response(con, jb) ) return false;
	pool->sentinel->log(LOG_DEBUG, "processor::cmdServiceProc end" );
	return true;
}

bool ProcessorPool::Processor::cmdResourcesProc(QnSessionCommand *q, pjson::json *js)
{
	pool->sentinel->log(LOG_DEBUG, "processor::cmdResourcesProc begin" );
	Session *con = q->con;

	Database *db = pool->sentinel->getDatabase();
	DBError e;

	pjson::value *vCmd = js->get("resources");
	if( !vCmd ) {
		cmdError(con, CMDErrParameter, "resources");
		return false;
	}
	pjson::value *vSvc = js->get("serviceId", vCmd);
	if( !vSvc || vSvc->vt != pjson::vt_string ) {
		cmdError(con, CMDErrParameter, "resources.serviceId");
		return false;
	}

	const char *aid = NULL;
	pjson::value *vAcc = js->get("accountId", vCmd);
	if( vAcc ) {
		if( vAcc->vt != pjson::vt_string ) {
			cmdError(con, CMDErrParameter, "resources.serviceId");
			return false;
		}
		Account *acc;
		e = db->getAccountInfo(vAcc->vString, AccountStat, &acc);
		if( e != DBErrorOk ) {
			cmdError(con, CMDErrDatabase, "code=%d", e);
			return false;
		}
		free(acc);
		aid	= vAcc->vString;
	} else {
		aid	= con->getAccountId();
	}

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result",6) ||
		!jb.valueInt(0) ||
		!jb.addObjectProp("resources", 9) ) {
		cmdError(con, CMDErrRuntime, "json builder init failed. code=%d", jb.getError());
		return false;
	}

	RecList *pList = (RecList*)malloc(sizeof(RecList)+sizeof(char)*(SST_RESOURCEIDLEN+1)*32+sizeof(char*)*32);
	pList->nCount	= 32;
	pList->nSize	= sizeof(RecList)+sizeof(char)*(SST_RESOURCEIDLEN+1)*32+sizeof(char*)*32;
	pList->nValues	= 0;
	size_t nFrom = 0;

	do {
		e = db->getResourceList(vSvc->vString, nFrom, pList);
		if( e != DBErrorOk ) {
			cmdError(con, CMDErrDatabase, "code=%d", e);
			free(pList);
			return false;
		}
		size_t i;
		for( i = 0; i < pList->nValues; i++ ) {
			if( nFrom == 0 ) {
				if( !jb.beginArray() ) {
					cmdError(con, CMDErrRuntime, "json builder begin array failed. code=%d", jb.getError());
					free(pList);
					return false;
				}
			}
			if( !jb.addArrayContent() ||
				!jb.valueString(pList->pValue[i]) ) {
				cmdError(con, CMDErrRuntime, "json builder add array failed. code=%d", jb.getError());
				free(pList);
				return false;
			}
		}
		nFrom += pList->nValues;
	} while(pList->nValues == pList->nCount);
	free(pList);

	if( nFrom ) {
		if( !jb.endArray() ) {
			cmdError(con, CMDErrRuntime, "json builder end array failed. code=%d", jb.getError());
			free(pList);
			return false;
		}
	} else {
		if( !jb.valueString("empty") )  {
			cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
			free(pList);
			return false;
		}
	}

	if( !jb.endObject() ) {
		cmdError(con, CMDErrRuntime, "json builder end object failed. code=%d", jb.getError());
		return false;
	}
	if( !response(con, jb) ) return false;
	pool->sentinel->log(LOG_DEBUG, "processor::cmdResourcesProc end" );
	return true;
}

bool ProcessorPool::Processor::cmdResourceInfoProc(QnSessionCommand *q, pjson::json *js)
{
	pool->sentinel->log(LOG_DEBUG, "processor::cmdResourceInfoProc begin" );
	Session *con = q->con;
	Database *db = pool->sentinel->getDatabase();
	DBError e;

	pjson::value *vCmd = js->get("resource");
	if( !vCmd ) {
		cmdError(con, CMDErrParameter, "resource");
		return false;
	}
	pjson::value *vRid = js->get("rid", vCmd);
	if( !vRid || vRid->vt != pjson::vt_string ) {
		cmdError(con, CMDErrParameter, "resource.rid");
		return false;
	}

	ResourceInfo *pInfo = NULL;
	e = db->getResourceInfo(vRid->vString, &pInfo);
	if( e != DBErrorOk ) {
		cmdError(con, CMDErrDatabase, "code=%d", e);
		return false;
	}

	pjson::builder jb;
	if( !jb.init(1024) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result") ||
		!jb.valueInt(0) ||
		!jb.addObjectProp("serviceId") ||
		!jb.valueString(pInfo->serviceId) ||
		!jb.addObjectProp("resourceId") ||
		!jb.valueString(pInfo->resourceId) ||
		!jb.addObjectProp("resourceName") ||
		!jb.valueString(pInfo->resourceName) ||
		!jb.addObjectProp("URL") ||
		!jb.valueString(pInfo->URL) ||
		!jb.addObjectProp("method") ||
		!jb.valueString(pInfo->method) ||
		!jb.addObjectProp("param") ||
		!jb.valueString(pInfo->paramPattern) ||
		!jb.addObjectProp("header") ||
		!jb.valueString(pInfo->headerPattern) ||
		!jb.addObjectProp("status") ||
		!jb.valueInt(pInfo->status) ||
		!jb.endObject() ) {
		cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
		free(pInfo);
		return false;
	}
	free(pInfo);
	if( !response(con, jb) ) return false;
	pool->sentinel->log(LOG_DEBUG, "processor::cmdUserProc end" );
	return true;
}


bool ProcessorPool::Processor::cmdJoinGroupProc(QnSessionCommand *q, pjson::json *js)
{
	pool->sentinel->log(LOG_DEBUG, "processor::cmdJoinGroupProc begin" );

	Session *con = q->con;
	char *aid = con->getAccountId();
//		Buffer &wb = con->getWriteBuffer();
	if( v->vObject->n <= 1 ) {
		pool->sentinel->log(LOG_INFO, "processor::cmdJoinGroupProc invalid parameter count." );
		return false;
	}
	if( strcmp(v->vObject->props[1]->name, "gid") != 0 ) {
		pool->sentinel->log(LOG_INFO, "processor::cmdJoinGroupProc invalid parameter %s.", v->vObject->props[1]->name );
		return false;
	}

	const char *gid = v->vObject->props[1]->val.vString;

	Database *db = pool->sentinel->getDatabase();

	RecList *pList = (RecList*)malloc(sizeof(RecList)+sizeof(char)*(SST_USERIDLEN+1)*32+sizeof(char*)*32);
	pList->nCount	= 32;
	pList->nSize	= sizeof(RecList)+sizeof(char)*(SST_USERIDLEN+1)*32+sizeof(char*)*32;
	pList->nValues	= 0;
	size_t nFrom = 0;
	do {
		if( db->getGroupList(nFrom, pList) != DBErrorOk ) {
			pool->sentinel->log(LOG_DEBUG, "processor::cmdJoinGroupProc getGroupList failed");
			return false;
		}
		size_t i;
		for( i = 0; i < pList->nValues; i++ ) {
			if( strcmp( pList->pValue[i], gid) == 0 ) {
				pool->sentinel->log(LOG_INFO, "processor::cmdJoinGroupProc %s is already joined %s", aid, gid);
				return false;
			}
		}
		nFrom += pList->nValues;
	} while(pList->nValues == pList->nCount);

	if( db->participationGroupList(aid, gid) != DBErrorOk ) {
		pool->sentinel->log(LOG_INFO, "processor::cmdJoinGroupProc %s joined to %s is failed", aid, gid);
		return false;
	}
	pool->sentinel->log(LOG_INFO, "processor::cmdJoinGroupProc %s was joined %s", aid, gid);
	pool->sentinel->log(LOG_DEBUG, "processor::cmdJoinGroupProc end" );
	*/
	return true;
}

size_t callBackFunk(char* ptr, size_t size, size_t nmemb, void* data)
{
	ProcessorPool::Processor *obj = (ProcessorPool::Processor*)data;
	return obj->testCb(ptr, size, nmemb);
}

size_t ProcessorPool::Processor::testCb(char* ptr, size_t size, size_t nmemb)
{
	pool->sentinel->log(LOG_DEBUG, "curl cb ptr=%x size=%u, nmemb=%u", ptr, size, nmemb);
	char m[1024];
	size_t s;
	s = nmemb < 1023 ? nmemb : 1023;
	strncpy(m, ptr, s);
	m[s] = 0;
	pool->sentinel->log(LOG_DEBUG, "%s", m);
    return size * nmemb;
}

bool ProcessorPool::Processor::cmdTestProc(QnSessionCommand *q, pjson::json *js)
{
	pool->sentinel->log(LOG_DEBUG, "processor::cmdTestProc 1" );
	Session *con = q->con;

	pjson::value *vCmd = js->get("test");
	if( !vCmd ) {
		cmdError(con, CMDErrParameter, "test");
		return false;
	}
	pjson::value *vURL= js->get("url", vCmd);
	if( !vURL || vURL->vt != pjson::vt_string ) {
		cmdError(con, CMDErrParameter, "test.URL");
		return false;
	}
	pool->sentinel->log(LOG_DEBUG, "processor::cmdTestProc 2 url=%s", vURL->vString );

	CURL *curl;
	CURLcode res;
	curl = curl_easy_init();
	if( !curl) {
		cmdError(con, CMDErrRuntime, "curl init error");
		return false;
	}
	pool->sentinel->log(LOG_DEBUG, "processor::cmdTestProc curl_init ok");
	curl_easy_setopt(curl, CURLOPT_URL, vURL->vString);
	pool->sentinel->log(LOG_DEBUG, "processor::cmdTestProc CURLOPT_URL ok");
	curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, callBackFunk);
	pool->sentinel->log(LOG_DEBUG, "processor::cmdTestProc CURLOPT_WRITEFUNCTION ok");
	curl_easy_setopt(curl, CURLOPT_WRITEDATA, this);
	pool->sentinel->log(LOG_DEBUG, "processor::cmdTestProc CURLOPT_WRITEDATA ok");
	res = curl_easy_perform(curl);
	pool->sentinel->log(LOG_DEBUG, "processor::cmdTestProc curl_easy_perform ok");
	curl_easy_cleanup(curl);
	if (res != CURLE_OK) {
		cmdError(con, CMDErrRuntime, "curl error");
		return false;
	}
	pool->sentinel->log(LOG_DEBUG, "processor::cmdTestProc curl_cleanup ok");

	pjson::builder jb;
	if( !jb.init(1024) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result") ||
		!jb.valueInt(0) ||
		!jb.endObject() ) {
		cmdError(con, CMDErrRuntime, "json builder failed. code=%d", jb.getError());
		return false;
	}
	if( !response(con, jb) ) return false;
	pool->sentinel->log(LOG_DEBUG, "processor::cmdUserProc end" );
	return true;
}

bool ProcessorPool::Processor::responseProc(QnResponse *q)
{
	pool->sentinel->log(LOG_DEBUG, "Processor::responseProc begin" );
	Session *con = q->con;



	pool->sentinel->log(LOG_DEBUG, "Processor::responseProc end" );
	return true;
}

////////////////////////////////////
ProcessorPool::ProcessorPool()
{
	sentinel		= NULL;
	nProcessor		= 0;
	pProcessor		= NULL;
}

ProcessorPool::~ProcessorPool()
{
	clear();
}

void ProcessorPool::clear()
{

}

bool ProcessorPool::run(Sentinel *obj)
{
	sentinel	= obj;
	Conf *conf = sentinel->getConf();
	int defProcessor = conf->numProcessor;
	if( defProcessor <= 0 ) defProcessor = 1;
	pProcessor	= new Processor[defProcessor];
	if( pProcessor == NULL ) {
		sentinel->log(LOG_ERR, "Out of memory for processor thread.");
		return false;
	}
	nProcessor	= defProcessor;
	size_t i;
	for( i = 0; i < nProcessor; i++ ) {
		if( !pProcessor[i].run(this) ) {
			return false;
		}
	}
	return true;
}

void ProcessorPool::stop()
{
	size_t i;
	for( i = 0; i < nProcessor; i++ ) {
		pProcessor[i].stop();
	}
	for( i = 0; i < nProcessor; i++ ) {
		pProcessor[i].wait();
	}
}

}
