//============================================================================
// Name        : sentinel.cpp
// Author      : Yasuoki
// Version     : 0.1
// Copyright   : LGPL
// Description : sentinel server
//============================================================================

#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <stdarg.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include "sentinel.h"
#include "listener.h"
#include "processor.h"
#include "database.h"

namespace SST {

Sentinel::Sentinel()
{
	conf		= NULL;
	pDevices	= NULL;
	pDatabase	= NULL;
	eventBase	= NULL;
	sigterm		= NULL;
	sighup		= NULL;
	sigint		= NULL;

	evthread_use_pthreads();
}

Sentinel::~Sentinel()
{
	if( pDevices ) {
		delete pDevices;
	}
	if( pDatabase ) {
		delete pDatabase;
	}
	if( conf ) {
		freeConf(conf);
	}
}

void Sentinel::log(int logLevel, const char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    if( conf ) {
    	if( logLevel == LOG_DEBUG )
    		if( conf->logLevel > LogLevelDebug ) return;
    	if( logLevel == LOG_INFO )
    		if( conf->logLevel > LogLevelVerbose ) return;
    	if( logLevel == LOG_NOTICE )
    		if( conf->logLevel > LogLevelNotice ) return;
    	if( logLevel == LOG_WARNING )
    		if( conf->logLevel > LogLevelWarning ) return;
#ifdef _DEBUG
    	vfprintf(stderr, fmt, ap);
    	fprintf(stderr, "\n");
#else
    	if( conf->logIdent ) {
            vsyslog(logLevel, fmt, ap);
    	} else {
        	vfprintf(stderr, fmt, ap);
        	fprintf(stderr, "\n");
        }
#endif
    } else {
#ifdef _DEBUG
    	vfprintf(stderr, fmt, ap);
    	fprintf(stderr, "\n");
#else
    	if( logLevel >= LOG_WARNING ) {
			vfprintf(stderr, fmt, ap);
			fprintf(stderr, "\n");
    	}
#endif
    }
}

bool Sentinel::loadConfig(const char *file)
{
	log(LOG_DEBUG, "Loading configure file. %s", file);
	if( file == NULL ) {
		if( conf == NULL ) {
			log(LOG_ERR, "Configure file is not specified.");
			return false;
		}
		file = conf->confFile;
	}
	log(LOG_INFO, "loading %s", file);

	FILE *fp = fopen(file, "rt");
	if( fp == NULL ) {
		log(LOG_ERR, "%s can't open.", file);
		return false;
	}
	log(LOG_DEBUG, "open configure file %s ok.", file);

	Conf *new_conf = allocConf(file);
	if( new_conf == NULL ) {
		log(LOG_ERR, "Out of memory at loading %s.", file);
		return false;
	}
	log(LOG_DEBUG, "memory alloc ok.");

	bool hasError=false;
	int line=1;
	char buff[4096];
	char key[4096];
	char value[4096];
	while( !feof(fp) ) {
		if( fgets(buff, sizeof(buff), fp) == NULL ) break;
		char *p = buff;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p == 0 || *p == '\r' || *p == '\n' || *p == '#') {
			line++;
			continue;
		}
		char *pWord = p;
		while( *p != 0 && *p != '\r' && *p != '\n' && *p != '\t' && *p != ' ' && *p != '=') p++;
		if( size_t(p-pWord) > sizeof(key)-1 ) {
			log(LOG_ERR, "%s:%d keyword length is too long.", file, line);
			hasError = true;
			line++;
			continue;
		}
		strncpy(key, pWord, p-pWord);
		key[p-pWord] = 0;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p != '=' ) {
			log(LOG_ERR, "%s:%d syntax error.", file, line);
			hasError = true;
			line++;
			continue;
		}
		p++;
		while( *p == ' ' || *p == '\t' ) p++;
		char em = 0;
		if( *p == '"' ) {
			em = '"';
			p++;
		}
		pWord = p;
		while( *p != 0 ) {
			if( em ) {
				if( *p == em ) break;
			} else {
				if( *p == '\t' || *p == ' ' || *p == '\r' || *p == '\n' ) break;
			}
			p++;
		}
		if( size_t(p-pWord) > sizeof(value)-1 ) {
			log(LOG_ERR, "%s:%d value length is too long.", file, line);
			hasError = true;
			line++;
			continue;
		}
		strncpy(value, pWord, p-pWord);
		value[p-pWord] = 0;
		log(LOG_DEBUG, "Line %d Key=%s Value=%s", line, key, value);

		if( strcmp(key, "pidFile") == 0 ) {
			if( new_conf->pidFile ) free(new_conf->pidFile);
			new_conf->pidFile	= strdup(value);
		} else if( strcmp(key, "daemonize") == 0 ) {
			if( strcmp(value, "yes") == 0 )
				new_conf->daemonaize	= true;
			else if(strcmp(value, "no") == 0 )
				new_conf->daemonaize	= false;
			else {
				log(LOG_ERR, "%s:%d bad parameter. (yes or no)", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "initProcesses") == 0 ) {
			new_conf->initProcesses	= atoi(value);
		} else if( strcmp(key, "numListenThreads") == 0 ) {
			new_conf->numListener	= atoi(value);
		} else if( strcmp(key, "numProcessThreads") == 0 ) {
			new_conf->numProcessor	= atoi(value);
		} else if( strcmp(key, "maxConnections") == 0 ) {
			new_conf->maxConnections	= atoi(value);
		} else if( strcmp(key, "maxDevices") == 0 ) {
			new_conf->maxDevices	= atoi(value);
		} else if( strcmp(key, "maxUsers") == 0 ) {
			new_conf->maxUsers	= atoi(value);
		} else if( strcmp(key, "maxGroups") == 0 ) {
			new_conf->maxGroups	= atoi(value);
		} else if( strcmp(key, "maxServices") == 0 ) {
			new_conf->maxServices	= atoi(value);
		} else if( strcmp(key, "maxResources") == 0 ) {
			new_conf->maxResources	= atoi(value);
		} else if( strcmp(key, "serverName") == 0 ) {
			if( new_conf->serverName) free(new_conf->serverName);
			new_conf->serverName = strdup(value);
		} else if( strcmp(key, "port") == 0 ) {
			new_conf->port		= atoi(value);
		} else if( strcmp(key, "bind") == 0 ) {
			if( new_conf->bind ) free(new_conf->bind);
			new_conf->bind	= strdup(value);
		} else if( strcmp(key, "authPassword") == 0 ) {
			if( new_conf->authPasswd ) free(new_conf->authPasswd);
			new_conf->authPasswd	= strdup(value);
		} else if( strcmp(key, "authMethod") == 0 ) {
			if( new_conf->authMethod ) free(new_conf->authMethod);
			new_conf->authMethod	= strdup(value);
		} else if( strcmp(key, "timeout") == 0 ) {
			new_conf->timeout	= atoi(value);
		} else if( strcmp(key, "maxWaitTime") == 0 ) {
			new_conf->maxWaitTime	= atoi(value);
		} else if( strcmp(key, "connThreads") == 0 ) {
			new_conf->connThreads	= atoi(value);
		} else if( strcmp(key, "logLevel") == 0 ) {
			if( strcmp(value, "debug") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "verbose") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "notice") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "warning") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "error") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else {
				log(LOG_ERR, "%s:%d bad parameter. (debug, verbose, notice, warning or error)", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "logIdent") == 0 ) {
			if( new_conf->logIdent ) free(new_conf->logIdent);
			new_conf->logIdent	= NULL;
			if( strcmp(value, SST_STDERRIDENT) != 0 )
				new_conf->logIdent	= strdup(value);
		} else if( strcmp(key, "dbServer") == 0 ) {
			if( new_conf->dbServer ) free(new_conf->dbServer);
			new_conf->dbServer	= strdup(value);
		} else if( strcmp(key, "dbPort") == 0 ) {
			new_conf->dbPort	= atoi(value);
		} else if( strcmp(key, "dbAuth") == 0 ) {
			if( new_conf->dbAuth ) free(new_conf->dbAuth);
			new_conf->dbAuth	= strdup(value);
		} else if( strcmp(key, "dbThreads") == 0 ) {
			new_conf->dbThreads	= atoi(value);
		} else if( strcmp(key, "masterServer") == 0 ) {
			if( new_conf->masterServer ) free(new_conf->masterServer);
			new_conf->masterServer	= strdup(value);
		} else if( strcmp(key, "masterPort") == 0 ) {
			new_conf->masterPort	= atoi(value);
		} else if( strcmp(key, "masterAuth") == 0 ) {
			if( new_conf->masterAuth ) free(new_conf->masterAuth);
			new_conf->masterAuth	= strdup(value);
		} else if( strcmp(key, "cloneMaster") == 0 ) {
		} else {
			log(LOG_ERR, "%s:%d unknown keyword error. (%s)", file, line, key);
			hasError = true;
			line++;
			continue;
		}
		line++;
	}
	fclose(fp);

	if( new_conf->initProcesses != 0 &&
		(new_conf->initProcesses < 0 ||
		new_conf->initProcesses > 32) ) {
		log(LOG_ERR, "%s bad range. (0 or 1 to 32)", file);
		hasError = true;
	}

	if( hasError ) {
		log(LOG_DEBUG, "configure file has error.");
		freeConf(new_conf);
		return false;
	}

	if( this->conf ) freeConf(this->conf);
	this->conf = new_conf;
	log(LOG_DEBUG,"confFile: %s", conf->confFile);
	log(LOG_DEBUG,"pidFile: %s", conf->pidFile);
	log(LOG_DEBUG,"daemonaize: %s", conf->daemonaize ? "yes" : "no");
	log(LOG_DEBUG,"initProcesses: %d", conf->initProcesses);
	log(LOG_DEBUG,"maxConnections: %d", conf->maxConnections);
	log(LOG_DEBUG,"maxDevices: %d", conf->maxDevices);
	log(LOG_DEBUG,"maxUsers: %d", conf->maxUsers);
	log(LOG_DEBUG,"maxGroups: %d", conf->maxGroups);
	log(LOG_DEBUG,"maxServices: %d", conf->maxServices);
	log(LOG_DEBUG,"maxResources: %d", conf->maxResources);
	log(LOG_DEBUG,"port: %d", conf->port);
	log(LOG_DEBUG,"bind: %s", conf->bind);
	log(LOG_DEBUG,"authPasswd: %s", conf->authPasswd ? conf->authPasswd : "(empty)");
	log(LOG_DEBUG,"authMethod: %s", conf->authMethod ? conf->authMethod : "(empty)");
	log(LOG_DEBUG,"timeout: %d", conf->timeout);
	log(LOG_DEBUG,"maxWaitTime: %d", conf->maxWaitTime);
	log(LOG_DEBUG,"connThreads: %d", conf->connThreads);
	log(LOG_DEBUG,"logLevel: %d", conf->logLevel);
	log(LOG_DEBUG,"logIdent: %s", conf->logIdent);
	log(LOG_DEBUG,"dbServer: %s", conf->dbServer);
	log(LOG_DEBUG,"dbPort: %d", conf->dbPort);
	log(LOG_DEBUG,"dbAuth: %s", conf->dbAuth ? conf->dbAuth : "(empty)");
	log(LOG_DEBUG,"dbThreads: %d", conf->dbThreads);
	log(LOG_DEBUG,"masterServer: %s", conf->masterServer ? conf->masterServer : "(empty)");
	log(LOG_DEBUG,"masterPort: %d", conf->masterPort);
	log(LOG_DEBUG,"masterAuth: %s", conf->masterAuth ? conf->masterAuth : "(empty)");
	log(LOG_DEBUG,"cloneMaster: %s", conf->cloneMaster ? "yes" : "no");
	log(LOG_DEBUG, "Loading configure file complete.");
	return true;
}

Conf * Sentinel::getConf()
{
	return conf;
}

bool Sentinel::configure()
{
	log(LOG_DEBUG, "Sentinel::configure");
	conList.clear();
	if( pDevices ) delete pDevices;
	if( pDatabase ) delete pDatabase;

	pDevices = new DeviceSlotList(this);
	pDatabase = new RedisDB();
	log(LOG_DEBUG, "Sentinel::configure: configure database");
	if( !pDatabase->configure(this, conf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure device slot");
	if( !pDevices->allocSlot(conf->maxDevices) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure session pool");
	if( !conList.configure(this, conf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure resuqster");
	if( !requester.configure(this, conf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure services");
	if( !services.configure(this, conf) ) {
		return false;
	}

	log(LOG_DEBUG, "Sentinel::configure: connect database");
	int nRetry=6;
	int i;
	for(i = 0; i < nRetry; i++ ) {
		if( pDatabase->connect() ) break;
		log(LOG_WARNING, "wait for database connection. %d/%d", i + 1, nRetry);
		::sleep(10);
	}
	if( i == nRetry ) {
		log(LOG_ERR, "can't connect to database.");
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: complete");
	return true;
}

void Sentinel::sigtermProc(int fd, short event, void *args)
{
	Sentinel *obj = (Sentinel*)args;
	obj->log(LOG_DEBUG, "accept SIGTERM");
	event_base_loopbreak(obj->eventBase);
	obj->waitThreads();
}

void Sentinel::sighupProc(int fd, short event, void *args)
{
	Sentinel *obj = (Sentinel*)args;
	obj->log(LOG_DEBUG, "accept SIGHUP");
	event_base_loopbreak(obj->eventBase);
	obj->waitThreads();
	if( !obj->loadConfig(NULL) ) {
		obj->log(LOG_ERR, "load configure error");
		return;
	}
	if( !obj->startThreads() ) {
		return;
	}
	obj->log(LOG_INFO, "restart");}

void Sentinel::sigintProc(int fd, short event, void *args)
{
	Sentinel *obj = (Sentinel*)args;
	obj->log(LOG_DEBUG, "accept SIGINT");
	event_base_loopbreak(obj->eventBase);
	obj->waitThreads();
}

bool Sentinel::run()
{
	log(LOG_INFO, "sentinel start");
	if( !configure() ) {
		return false;
	}
	log(LOG_DEBUG, "sentinel configure");

	eventBase = event_base_new();
	sigterm = evsignal_new(eventBase, SIGTERM, sigtermProc, this);
	evsignal_add(sigterm, NULL);
	sighup = evsignal_new(eventBase, SIGHUP, sighupProc, this);
	evsignal_add(sighup, NULL);
	sigint = evsignal_new(eventBase, SIGINT, sigintProc, this);
	evsignal_add(sigint, NULL);
/*
	sigset_t ss;
	sigemptyset(&ss);
    sigfillset(&ss);
    pthread_sigmask(SIG_BLOCK, &ss, NULL);
*/
	if( !startThreads() ) {
		return false;
	}
	event_base_dispatch(eventBase);
	/*
	int sn;
	struct timespec ts={5,0};
	while(1) {
		log(LOG_DEBUG, "watch signal");
		siginfo_t info;
		if( sigtimedwait(&ss, &info, &ts) > 0) {
			if( info.si_signo == SIGTERM ) {
				sig_term	= 1;
				log(LOG_DEBUG, "accept SIGTERM");
				waitThreads();
				sig_term	= 0;
				break;
			}
			if( info.si_signo == SIGINT ) {
				sig_int		= 1;
				log(LOG_DEBUG, "accept SIGINT");
				waitThreads();
				sig_int		= 0;
				break;
			}
			if( info.si_signo == SIGHUP ) {
				sig_hup		= 1;
				log(LOG_DEBUG, "accept SIGHUP");
				waitThreads();
				sig_hup		= 0;
				if( !loadConfig(NULL) ) {
					log(LOG_ERR, "load configure error");
					break;
				}
				if( !startThreads() ) {
					return false;
				}
				log(LOG_INFO, "restart");
			}
		}
	}
	*/

    evsignal_del(sigterm);
    evsignal_del(sighup);
    evsignal_del(sigint);
    event_free(sigterm);
    event_free(sighup);
    event_free(sigint);
    event_base_free(eventBase);
//    pthread_sigmask(SIG_UNBLOCK, &ss, NULL);
	log(LOG_INFO, "sentinel terminate");
	return true;
}

Conf * Sentinel::allocConf(const char *file)
{
	Conf *new_conf = (Conf*)malloc(sizeof(Conf));
	if( new_conf == NULL ) {
		return NULL;
	}

	if( file )
		new_conf->confFile	= strdup(file);
	else
		new_conf->confFile	= NULL;
	new_conf->pidFile		= strdup(SST_PIDFILE);
	new_conf->daemonaize	= false;
	new_conf->initProcesses	= 1;
	new_conf->numListener	= 1;
	new_conf->numProcessor	= 2;
	new_conf->maxConnections= 1000;
	new_conf->maxDevices	= 1000;
	new_conf->maxUsers		= 1000;
	new_conf->maxGroups		= 0;
	new_conf->maxServices	= 0;
	new_conf->maxResources	= 0;
	new_conf->serverName	= NULL;
	new_conf->port			= SST_PORT;
	new_conf->bind			= NULL;
	new_conf->authPasswd	= NULL;
	new_conf->authMethod	= NULL;
	new_conf->timeout		= SST_TIMEOUT;
	new_conf->maxWaitTime	= SST_MAXWAITTIME;
	new_conf->connThreads	= SST_CONNTHREADS;
	new_conf->logLevel		= LogLevelError;
	new_conf->logIdent		= strdup(SST_SYSLOGIDENT);
	new_conf->dbServer		= strdup("localhost");
	new_conf->dbPort		= 6379;
	new_conf->dbAuth		= NULL;
	new_conf->dbThreads 	= SST_DBTHREADS;
	new_conf->masterServer	= NULL;
	new_conf->masterPort	= 0;
	new_conf->masterAuth	= NULL;
	new_conf->cloneMaster	= false;

	return new_conf;
}

void Sentinel::freeConf(Conf *conf)
{
	log(LOG_DEBUG, "freeConf");
	if( conf ) {
		if( conf->confFile ) free(conf->confFile);
		if( conf->pidFile ) free(conf->pidFile);
		if( conf->serverName) free(conf->serverName);
		if( conf->bind ) free(conf->bind);
		if( conf->authPasswd ) free(conf->authPasswd);
		if( conf->authMethod ) free(conf->authMethod);
		if( conf->logIdent ) free(conf->logIdent);
		if( conf->dbServer ) free(conf->dbServer);
		if( conf->dbAuth ) free(conf->dbAuth);
		if( conf->masterServer ) free(conf->masterServer);
		if( conf->masterAuth ) free(conf->masterAuth);
		free(conf);
	}
}

SessionPool &Sentinel::getSessionPool()
{
	return conList;
}

ListenerPool & Sentinel::getListener()
{
	return listener;
}

int	Sentinel::getLoad() const
{
	size_t a = conList.getActiveSessions();
	size_t n = conList.getCapacity();
	return (a*100/n);
}

bool Sentinel::startThreads()
{
	log(LOG_DEBUG, "sentinel processor run");
	if( !processor.run(this) ) {
		return false;
	}
	log(LOG_DEBUG, "sentinel listener run");
	if( !listener.run(this) ) {
		return false;
	}
	log(LOG_DEBUG, "sentinel requester run");
	if( !requester.run(this) ) {
		return false;
	}
	log(LOG_DEBUG, "sentinel services run");
	if( !services.run() ) {
		return false;
	}
	return true;
}

void Sentinel::waitThreads()
{
	listener.stop();
	processor.stop();
}

}
