/*
 * db_port.h
 *
 *  Created on: 2012/12/11
 *      Author: yasuoki
 */

#ifndef DB_PORT_H_
#define DB_PORT_H_

#include "../include/sst_types.h"
#include "lock.h"
#include "buffer.h"
#include <string.h>
#include <openssl/sha.h>

#define HASH_KEY_LEN	((SHA224_DIGEST_LENGTH)*2+16)

namespace SST {

static char b64char[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

/////////////////////////////////////////////////////////////////////
class DBUtil
{
public:

	static void b64enc(unsigned char *s, size_t size, char *out)
	{
		unsigned char c0;
		unsigned char c1;
		unsigned char c2;
		while(size >= 3) {
			c0 = *s++;
			c1 = *s++;
			c2 = *s++;
			*out++ = b64char[(c0>>2)&0x3f];						// xxxxxx00 00000000 00000000
			*out++ = b64char[((c0<<4)&0x30)|((c1>>4)&0x0f)]; 	// 000000xx xxxx0000 00000000
			*out++ = b64char[((c1<<2)&0x3c)|((c2>>6)&0x03)]; 	// 00000000 0000xxxx xx000000
			*out++ = b64char[c2&0x3f];							// 00000000 00000000 00xxxxxx
			size -= 3;
		}
		if( size == 2 ) {
			c0 = *s++;
			c1 = *s++;
			c2 = 0;
			*out++ = b64char[(c0>>2)&0x3f];						// xxxxxx00 00000000 00000000
			*out++ = b64char[((c0<<4)&0x30)|((c1>>4)&0x0f)]; 	// 000000xx xxxx0000 00000000
			*out++ = b64char[((c1<<2)&0x3c)|((c2>>6)&0x03)]; 	// 00000000 0000xxxx xx000000
		} else if( size == 1 ) {
			c0 = *s++;
			c1 = 0;
			c2 = 0;
			*out++ = b64char[(c0>>2)&0x3f];						// xxxxxx00 00000000 00000000
			*out++ = b64char[((c0<<4)&0x30)|((c1>>4)&0x0f)]; 	// 000000xx xxxx0000 00000000
		}
		*out = 0;
	}

	static ErrorCode keyHash(const char *key, char *hash)
	{
		unsigned char hash_out[SHA224_DIGEST_LENGTH+1];
		SHA224((const unsigned char*)key, strlen(key), hash_out);
		b64enc(hash_out, SHA224_DIGEST_LENGTH, hash);
		return ErrorOk;
	}

	static ErrorCode keyService(int sid, Buffer &key)
	{
		key.clear();
		key.addfmt("S%x", sid);
		return ErrorOk;
	}

	static ErrorCode keyServiceRev(const char *sname, Buffer &key)
	{
		char hash[HASH_KEY_LEN+1];
		keyHash(sname, hash);
		key.clear();
		key.addfmt("KS:%s", hash);
		return ErrorOk;
	}

	static ErrorCode keyServiceWritableList(const char *sid, Buffer &key)
	{
		key.clear();
		key.addfmt("HPW%s", sid);
		return ErrorOk;
	}

	static ErrorCode keyServiceReadableList(const char *sid, Buffer &key)
	{
		key.clear();
		key.addfmt("HPR%s", sid);
		return ErrorOk;
	}

	static ErrorCode keyServiceRefuseList(const char *sid, Buffer &key)
	{
		key.clear();
		key.addfmt("HRU%s", sid);
		return ErrorOk;
	}

	static ErrorCode keyServiceResourceList(const char *sid, Buffer &key)
	{
		key.clear();
		key.addfmt("HRS%s", sid);
		return ErrorOk;
	}

	static ErrorCode keyServiceGroupList(const char *sid, Buffer &key)
	{
		key.clear();
		key.addfmt("LGR%s", sid);
		return ErrorOk;
	}

	static ErrorCode keyAccount(int aid, Buffer &key)
	{
		key.clear();
		key.addfmt("A%x", aid);
		return ErrorOk;
	}

	static ErrorCode keyAccountRev(const char *userId, Buffer &key)
	{
		key.clear();
		key.addfmt("KA:%s", userId);
		return ErrorOk;
	}

	static ErrorCode keyAccountDeviceList(const char *aid, Buffer &key)
	{
		key.clear();
		key.addfmt("HDV%s", aid);
		return ErrorOk;
	}

	static ErrorCode keyAccountGroupList(const char *aid, Buffer &key)
	{
		key.clear();
		key.addfmt("HGR%s", aid);
		return ErrorOk;
	}

	static ErrorCode keyDevice(int did, Buffer &key)
	{
		key.clear();
		key.addfmt("D%x", did);
		return ErrorOk;
	}

	static ErrorCode keyDeviceRev(const char *udid, Buffer &key)
	{
		key.clear();
		key.addfmt("KD:%s", udid);
		return ErrorOk;
	}

	static ErrorCode keyResource(int rid, Buffer &key)
	{
		key.clear();
		key.addfmt("R%x", rid);
		return ErrorOk;
	}

	static ErrorCode keyResourceRev(const char *sid, const char *rname, Buffer &key)
	{
		char hash[HASH_KEY_LEN+1];
		keyHash(rname, hash);
		key.clear();
		key.addfmt("KR:%s:%s", sid, hash);
		return ErrorOk;
	}

	static ErrorCode keyResourceGadgetList(const char *rid, Buffer &key)
	{
		key.clear();
		key.addfmt("LGD%s", rid);
		return ErrorOk;
	}

	static ErrorCode keyGroup(int gid, Buffer &key)
	{
		key.clear();
		key.addfmt("G%x", gid);
		return ErrorOk;
	}

	static ErrorCode keyGroupRev(const char *sid, const char *gname, Buffer &key)
	{
		char hash[HASH_KEY_LEN+1];
		keyHash(gname, hash);
		key.clear();
		key.addfmt("KG:%s:%s", sid, hash);
		return ErrorOk;
	}

	static ErrorCode keyGroupMemberList(const char *gid, Buffer &key)
	{
		key.clear();
		key.addfmt("HAC%s", gid);
		return ErrorOk;
	}

	static ErrorCode keyGroupShareList(const char *gid, Buffer &key)
	{
		key.clear();
		key.addfmt("LSH%s", gid);
		return ErrorOk;
	}

	static ErrorCode keyShare(const char *rid, size_t index, Buffer &key)
	{
		key.clear();
		key.addfmt("M%s:%d", rid, index);
		return ErrorOk;
	}

	static ErrorCode keyDeviceShareList(const char *did, Buffer &key)
	{
		key.clear();
		key.addfmt("LSH%s", did);
		return ErrorOk;
	}

	static ErrorCode keyDeviceShare(const char *did, const char *shareId, Buffer &key)
	{
		key.clear();
		key.addfmt("C%s%s", did, shareId);
		return ErrorOk;
	}

	static ErrorCode keyShareRid(const char *mid, Buffer &rid, int &index) {
		if(*mid != 'M' || *(mid+1) != 'R' ) return ErrorData;
		const char *p = &mid[2];
		while( *p && *p != ':' ) {
			if( (*p < '0' || '9' < *p) &&
				(*p < 'a' || 'f' < *p) &&
				(*p < 'A' || 'F' < *p) ) return ErrorData;
			p++;
		}
		if( *p != ':' ) return ErrorData;
		// MR1223:
		//       ^
		rid.add(&mid[1], p-mid-1);
		p++;
		while( *p ) {
			if( *p < '0' || '9' < *p ) return ErrorData;
			p++;
		}
		index = atoi(p+1);
		return ErrorOk;
	}

	static ErrorCode keySyncQueue(const char *sid, Buffer &key)
	{
		key.clear();
		key.addfmt("Q%s", sid);
		return ErrorOk;
	}

	static size_t sizeofKey(const QueryKeyType *key) {
		size_t size = sizeof(QueryKeyType);
		if( key ) {
			if( key->mId ) {
				size += strlen(key->mId) + 1;
			}
			if( key->mName ) {
				size += strlen(key->mName) + 1;
			}
		}
		return size;
	}
	static char * writeString(char **ptr, char *ext, const char *src) {
		if( src ) {
			*ptr	= ext;
			while(*src) *ext++ = *src++;
			*ext++ = 0;
		} else {
			*ptr	= NULL;
		}
		return ext;
	}
	static char * writeKey(QueryKeyType *ptr, char *ext, const QueryKeyType *src) {
		if( src && src->mId ) {
			ext = writeString((char **)(&ptr->mId), ext, src->mId);
		} else {
			ptr->mId	= NULL;
		}
		if( src && src->mName ) {
			ext = writeString((char **)(&ptr->mName), ext, src->mName);
		} else {
			ptr->mName	= NULL;
		}
		return ext;
	}
	static size_t sizeofRange(const QueryRangeType *range) {
		return sizeof(QueryRangeType);
	}
	static char * writeRange(QueryRangeType *ptr, char *ext, const QueryRangeType *src) {
		if( src ) {
			ptr->mStart	= src->mStart;
			ptr->mCount	= src->mCount;
		} else {
			ptr->mStart	= 0;
			ptr->mCount	= -1;
		}
		return ext;
	}

	static size_t sizeofQBase(const QueryKeyType *key) {
		size_t size = sizeof(QueryBase);
		if( key )
			size += sizeofKey(key);
		return size;
	}
	static char * writeQBase(QueryBase *ptr, char *ext, const QueryKeyType * sref) {
		//ptr->mQueryCmd	= cmd;
		return writeKey(&ptr->mSREF, ext, sref);
	}
	static size_t sizeofSBase(const QueryKeyType *key) {
		size_t size = sizeof(RegistBase);
		if( key )
			size += sizeofKey(key);
		return size;
	}
	static char * writeSBase(RegistBase *ptr, char *ext, const QueryKeyType * sref) {
		//ptr->mRegistCmd	= cmd;
		return writeKey(&ptr->mSREF, ext, sref);
	}
	static size_t sizeofRBase() {
		return sizeof(ResultBase);
	}

	static size_t sizeofList(size_t count, size_t elmSize) {
		size_t	size = sizeof(RecList) + (elmSize+1+sizeof(char*)) *count;
		return size;
	}

	static size_t sizeofAccount(const Account *src) {
		size_t	size = sizeof(Account);
		if( src ) {
			if( src->id ) size += strlen(src->id)+1;
			if( src->uid ) size += strlen(src->uid)+1;
			if( src->name1 ) size += strlen(src->name1)+1;
			if( src->name2 ) size += strlen(src->name2)+1;
			if( src->locale ) size += strlen(src->locale)+1;
			if( src->authData ) size += strlen(src->authData)+1;
		}
		return size;
	}
	static char * writeAccount(Account *ptr, char *ext, const Account *src) {
		if( src ) {
			ext = writeString(&ptr->id, ext, src->id);
			ext = writeString(&ptr->uid, ext, src->uid);
			ext = writeString(&ptr->name1, ext, src->name1);
			ext = writeString(&ptr->name2, ext, src->name2);
			ext = writeString(&ptr->locale, ext, src->locale);
			ext = writeString(&ptr->authData, ext, src->authData);

			ptr->authType	= src->authType;
			ptr->status		= src->status;
			ptr->level		= src->level;
			ptr->maxStorage	= src->maxStorage;
			ptr->useStorage	= src->useStorage;
			ptr->createTime	= src->createTime;
			ptr->updateTime	= src->updateTime;
			ptr->loginTime	= src->loginTime;
		} else {
			ptr->id			= NULL;
			ptr->uid		= NULL;
			ptr->name1		= NULL;
			ptr->name2		= NULL;
			ptr->locale		= NULL;
			ptr->authData	= NULL;

			ptr->authType	= AuthNone;
			ptr->status		= AccountNormal;
			ptr->level		= AccountUser;
			ptr->maxStorage	= 0;
			ptr->useStorage	= 0;
			ptr->createTime	= 0;
			ptr->updateTime	= 0;
			ptr->loginTime	= 0;
		}
		return ext;
	}
	static size_t	sizeofDevice(const DeviceInfo *src) {
		size_t	size = sizeof(DeviceInfo);
		if( src ) {
			if( src->id )		size += strlen(src->id) + 1;
			if( src->udid )		size += strlen(src->udid) + 1;
			if( src->OS )		size += strlen(src->OS) + 1;
			if( src->OSVersion)	size += strlen(src->OSVersion) + 1;
			if( src->carrier )	size += strlen(src->carrier) + 1;
			if( src->model )	size += strlen(src->model) + 1;
			if( src->signature)	size += strlen(src->signature) + 1;
			if( src->accountId)	size += strlen(src->accountId) + 1;
		}
		return size;
	}
	static char *writeDevice(DeviceInfo *ptr, char *ext, const DeviceInfo *src) {
		if( src ) {
			ext = writeString(&ptr->id, ext, src->id);
			ext = writeString(&ptr->udid, ext, src->udid);
			ext = writeString(&ptr->OS, ext, src->OS);
			ext = writeString(&ptr->OSVersion, ext, src->OSVersion);
			ext = writeString(&ptr->carrier, ext, src->carrier);
			ext = writeString(&ptr->model, ext, src->model);
			ext = writeString(&ptr->signature, ext, src->signature);
			ext = writeString(&ptr->accountId, ext, src->accountId);

			ptr->storageCrypt	= src->storageCrypt;
			ptr->storageBlockSize	= src->storageBlockSize;
			ptr->storageBlockCount	= src->storageBlockCount;
			ptr->storageUse		= src->storageUse;
			ptr->status			= src->status;
			ptr->loginTime		= src->loginTime;
			ptr->createTime		= src->createTime;
			ptr->updateTime		= src->updateTime;
		} else {
			ptr->id				= NULL;
			ptr->udid			= NULL;
			ptr->OS				= NULL;
			ptr->OSVersion		= NULL;
			ptr->carrier		= NULL;
			ptr->model			= NULL;
			ptr->signature		= NULL;
			ptr->accountId	= NULL;

			ptr->storageCrypt	= CryptAlgorithmNone;
			ptr->storageBlockSize	= 0;
			ptr->storageBlockCount	= 0;
			ptr->storageUse		= 0;
			ptr->status			= DeviceNull;
			ptr->loginTime		= 0;
			ptr->createTime		= 0;
			ptr->updateTime		= 0;
		}
		return ext;
	}

	static size_t	sizeofDeviceShare(const DeviceShareInfo *src) {
		size_t	size = sizeof(DeviceShareInfo);
		if( src ) {
			if( src->serviceId )	size += strlen(src->serviceId) + 1;
			if( src->shareId )		size += strlen(src->shareId) + 1;
			if( src->deviceId )		size += strlen(src->deviceId) + 1;
			if( src->accountId)		size += strlen(src->accountId) + 1;
		}
		return size;
	}
	static char *writeDeviceShare(DeviceShareInfo *ptr, char *ext, const DeviceShareInfo *src) {
		if( src ) {
			ext = writeString(&ptr->serviceId, ext, src->serviceId);
			ext = writeString(&ptr->shareId, ext, src->shareId);
			ext = writeString(&ptr->deviceId, ext, src->deviceId);
			ext = writeString(&ptr->accountId, ext, src->accountId);
			ptr->deviceTime	= src->deviceTime;
			ptr->deviceRevision	= src->deviceRevision;
			ptr->status	= src->status;
			ptr->shareTime		= src->shareTime;
		} else {
			ptr->serviceId		= NULL;
			ptr->shareId		= NULL;
			ptr->deviceId		= NULL;
			ptr->accountId		= NULL;
			ptr->deviceTime		= 0;
			ptr->deviceRevision	= 0;
			ptr->status			= DeviceShareNone;
			ptr->shareTime		= 0;
		}
		return ext;
	}

	static size_t	sizeofService(const ServiceInfo *src) {
		size_t	size = sizeof(ServiceInfo);
		if( src ) {
			if( src->id )		size += strlen(src->id) + 1;
			if( src->name )		size += strlen(src->name) + 1;
			if( src->title )	size += strlen(src->title) + 1;
			if( src->desc)		size += strlen(src->desc) + 1;
			if( src->version)	size += strlen(src->version) + 1;
			if( src->adminId )	size += strlen(src->adminId) + 1;
			if( src->refURL )	size += strlen(src->refURL) + 1;
			if( src->defaultGroup )	size += strlen(src->defaultGroup) + 1;
			if( src->authData )	size += strlen(src->authData) + 1;
		}
		return size;
	}
	static char *writeService(ServiceInfo *ptr, char *ext, const ServiceInfo *src) {
		if( src ) {
			ext = writeString(&ptr->id, ext, src->id);
			ext = writeString(&ptr->name, ext, src->name);
			ext = writeString(&ptr->title, ext, src->title);
			ext = writeString(&ptr->desc, ext, src->desc);
			ext = writeString(&ptr->version, ext, src->version);
			ext = writeString(&ptr->adminId, ext, src->adminId);
			ext = writeString(&ptr->refURL, ext, src->refURL);
			ext = writeString(&ptr->defaultGroup, ext, src->defaultGroup);
			ext = writeString(&ptr->authData, ext, src->authData);

			ptr->scope				= src->scope;
			ptr->storageBlockSize	= src->storageBlockSize;
			ptr->storageBlockCount	= src->storageBlockCount;
			ptr->storageCrypt		= src->storageCrypt;
			ptr->authType			= src->authType;
			ptr->status				= src->status;
			ptr->createTime			= src->createTime;
			ptr->updateTime			= src->updateTime;
		} else {
			ptr->id					= NULL;
			ptr->name				= NULL;
			ptr->title				= NULL;
			ptr->desc				= NULL;
			ptr->version			= NULL;
			ptr->adminId			= NULL;
			ptr->refURL				= NULL;
			ptr->defaultGroup		= NULL;
			ptr->authData			= NULL;

			ptr->scope				= ScopePublic;
			ptr->storageBlockSize	= 0;
			ptr->storageBlockCount	= 0;
			ptr->storageCrypt		= CryptAlgorithmNone;
			ptr->authType			= AuthNone;
			ptr->status				= ServiceStop;
			ptr->createTime			= 0;
			ptr->updateTime			= 0;
		}
		return ext;
	}

	static size_t	sizeofGadget(const GadgetInfo *src) {
		size_t	size = 0;
		if( src ) {
//			if( src->id )			size += strlen(src->id) + 1;
			if( src->name )			size += strlen(src->name) + 1;
			if( src->tag )			size += strlen(src->tag) + 1;
		}
		return size;
	}

	static char *writeGadget(GadgetInfo *ptr, char *ext, const GadgetInfo *src) {
		if( src ) {
//			ext = writeString(&ptr->id, ext, src->id);
			ext = writeString(&ptr->name, ext, src->name);
			ext = writeString(&ptr->tag, ext, src->tag);
			ptr->refTime	= src->refTime;
			ptr->revision	= src->revision;
			ptr->status		= src->status;
		} else {
//			ptr->id			= NULL;
			ptr->name		= NULL;
			ptr->tag		= NULL;
			ptr->refTime	= 0;
			ptr->revision	= 0;
			ptr->status		= ResourceStop;
		}
		return ext;
	}

	static size_t	sizeofResource(const ResourceInfo *src, const GadgetInfo *pGadget, size_t nGadget) {
		size_t	size = 0;
		if( src ) {
			if( src->id )			size += strlen(src->id) + 1;
			if( src->name )			size += strlen(src->name) + 1;
			if( src->tag )			size += strlen(src->tag) + 1;
			if( src->serviceId )	size += strlen(src->serviceId) + 1;
			if( src->refURL )		size += strlen(src->refURL) + 1;
		}
		if( nGadget > 0 ) {
			size += sizeof(GadgetInfo) * nGadget;
			size_t i;
			for( i = 0; i < nGadget; i++ ) {
				size += sizeofGadget(&pGadget[i]);
			}
		}
		return size;
	}

	static char *writeResource(ResourceInfo *ptr, char *ext, const ResourceInfo *src, const GadgetInfo *pGadget, size_t nGadget) {
		if( src ) {
			ext = writeString(&ptr->id, ext, src->id);
			ext = writeString(&ptr->name, ext, src->name);
			ext = writeString(&ptr->tag, ext, src->tag);
			ext = writeString(&ptr->serviceId, ext, src->serviceId);
			ext = writeString(&ptr->refURL, ext, src->refURL);
			ptr->refTime	= src->refTime;
			ptr->revision	= src->revision;
			ptr->status		= src->status;
			ptr->shareCount	= src->shareCount;
			ptr->createTime	= src->createTime;
			ptr->updateTime	= src->updateTime;
		} else {
			ptr->id			= NULL;
			ptr->name		= NULL;
			ptr->tag		= NULL;
			ptr->serviceId	= NULL;
			ptr->refURL		= NULL;
			ptr->refTime	= 0;
			ptr->revision	= 0;
			ptr->status		= ResourceStop;
			ptr->shareCount	= 0;
			ptr->createTime	= 0;
			ptr->updateTime	= 0;
		}
		if( nGadget ) {
			size_t i;
			ptr->gadgetCount	= nGadget;
			ptr->gadgetArray	= (GadgetInfo*)ext;
			ext += sizeof(GadgetInfo)*nGadget;
			for( i = 0; i < nGadget; i++ ) {
				ext = writeGadget(&ptr->gadgetArray[i], ext, &pGadget[i]);
			}
		} else {
			ptr->gadgetCount	= 0;
			ptr->gadgetArray	= NULL;
		}
		return ext;
	}

	static size_t	sizeofShare(const ShareInfo *src) {
		size_t	size = 0;
		if( src ) {
			if( src->id )			size += strlen(src->id) + 1;
			if( src->resourceId )	size += strlen(src->resourceId) + 1;
			if( src->serviceId )	size += strlen(src->serviceId) + 1;
			if( src->groupCount > 0 ) {
				size += sizeof(QueryKeyType) * src->groupCount;
				size_t i;
				for( i = 0; i <  src->groupCount; i++ ) {
					size += sizeofKey(&src->groupArray[i]);
				}
			}
		}
		return size;
	}

	static char *writeShare(ShareInfo *ptr, char *ext, const ShareInfo *src) {
		if( src ) {
			ext = writeString(&ptr->id, ext, src->id);
			ext = writeString(&ptr->resourceId, ext, src->resourceId);
			ext = writeString(&ptr->serviceId, ext, src->serviceId);
			ptr->index				= src->index;
			ptr->dataMode			= src->dataMode;
			ptr->accessMode			= src->accessMode;
			ptr->expireTime			= src->expireTime;
			ptr->fromTime			= src->fromTime;
			ptr->endTime			= src->endTime;
			ptr->geoLocationCount	= src->geoLocationCount;
			ptr->geoLocation[0]		= src->geoLocation[0];
			ptr->geoLocation[1]		= src->geoLocation[1];
			ptr->gpsGrace			= src->gpsGrace;
			ptr->offlineGrace		= src->offlineGrace;
			ptr->groupCount			= src->groupCount;
			ptr->createTime			= src->createTime;
			ptr->groupArray			= (QueryKeyType*)ext;
			ext += sizeof(QueryKeyType) * src->groupCount;
			size_t i;
			for( i = 0; i < src->groupCount; i++ ) {
				ext = writeKey(&ptr->groupArray[i], ext, &src->groupArray[i]);
			}

		} else {
			ptr->resourceId			= NULL;
			ptr->index				= 0;
			ptr->serviceId			= NULL;
			ptr->dataMode			= ShareDataIndex;
			ptr->accessMode			= ShareAccessRead;
			ptr->expireTime			= 0;
			ptr->fromTime			= 0;
			ptr->endTime			= 0;
			ptr->geoLocationCount	= 0;
			ptr->geoLocation[0].dis	= 0.0;
			ptr->geoLocation[0].lat	= 0.0;
			ptr->geoLocation[0].lon	= 0.0;
			ptr->geoLocation[1].dis	= 0.0;
			ptr->geoLocation[1].lat	= 0.0;
			ptr->geoLocation[1].lon	= 0.0;
			ptr->gpsGrace			= 0;
			ptr->offlineGrace		= 0;
			ptr->groupCount			= 0;
			ptr->groupArray			= NULL;
			ptr->createTime			= 0;
		}
		return ext;
	}

	static size_t	sizeofGroupInfo(const GroupInfo *src) {
		size_t	size = sizeof(GroupInfo);
		if( src ) {
			if( src->id )			size += strlen(src->id) + 1;
			if( src->serviceId )	size += strlen(src->serviceId) + 1;
			if( src->name)			size += strlen(src->name) + 1;
			if( src->adminId )		size += strlen(src->adminId) + 1;
			if( src->description )	size += strlen(src->description) + 1;
		}
		return size;
	}
	static char *writeGroupInfo(GroupInfo *ptr, char *ext, const GroupInfo *src) {
		if( src ) {
			ext = writeString(&ptr->id, ext, src->id);
			ext = writeString(&ptr->name, ext, src->name);
			ext = writeString(&ptr->serviceId, ext, src->serviceId);
			ext = writeString(&ptr->adminId, ext, src->adminId);
			ext = writeString(&ptr->description, ext, src->description);

			ptr->status			= src->status;
			ptr->createTime		= src->createTime;
			ptr->updateTime		= src->updateTime;
		} else {
			ptr->id				= NULL;
			ptr->name			= NULL;
			ptr->serviceId		= NULL;
			ptr->adminId		= NULL;
			ptr->description	= NULL;

			ptr->status			= GroupStop;
			ptr->createTime		= 0;
			ptr->updateTime		= 0;
		}
		return ext;
	}

	static size_t	sizeofSessionInfo(const SessionInfo *src) {
		size_t	size = sizeof(SessionInfo);
		if( src ) {
			if( src->id )			size += strlen(src->id) + 1;
			if( src->servName )		size += strlen(src->servName) + 1;
			if( src->serviceId )	size += strlen(src->serviceId) + 1;
			if( src->accountId )	size += strlen(src->accountId) + 1;
			if( src->deviceId )		size += strlen(src->deviceId) + 1;
		}
		return size;
	}
	static char * writeSessionInfo(SessionInfo *ptr, char *ext, const SessionInfo *src) {
		if( src ) {
			ext = writeString(&ptr->id, ext, src->id);
			ext = writeString(&ptr->servName, ext, src->servName);
			ext = writeString(&ptr->serviceId, ext, src->serviceId);
			ext = writeString(&ptr->accountId, ext, src->accountId);
			ext = writeString(&ptr->deviceId, ext, src->deviceId);

			ptr->servName	= src->servName;
			ptr->peerAddr	= src->peerAddr;
			ptr->status		= src->status;
			ptr->createTime	= src->createTime;
			ptr->updateTime	= src->updateTime;
		} else {
			ptr->id			= NULL;
			ptr->servName	= NULL;
			ptr->serviceId	= NULL;
			ptr->accountId	= NULL;
			ptr->deviceId	= NULL;

			memset(&ptr->servName, 0, sizeof(ptr->servName));
			memset(&ptr->peerAddr, 0, sizeof(ptr->peerAddr));
			ptr->status		= SessionStatusWait;
			ptr->createTime	= 0;
			ptr->updateTime	= 0;
		}
		return ext;
	}

	static size_t	sizeofNodeInfo(const NodeInfo *src) {
		size_t	size = sizeof(NodeInfo);
		if( src ) {
			if( src->name )	size += strlen(src->name);
		}
		return size;
	}
	static char * writeNodeInfo(NodeInfo *ptr, char *ext, const NodeInfo *src) {
		if( src ) {
			ext = writeString(&ptr->name, ext, src->name);

			ptr->addr	= src->addr;
			ptr->time	= src->time;
			ptr->load	= src->load;
		} else {
			ptr->name	= NULL;

			memset(&ptr->addr, 0, sizeof(ptr->addr));
			ptr->time	= 0;
			ptr->load	= 0;
		}
		return ext;
	}
};


class DBPort {
public:
	DBPort();
	virtual ~DBPort();

	virtual bool	configure(Sentinel *obj, Conf *conf);
	virtual bool	checkAccountName(const char *name);
	virtual bool	connect() = 0;
	virtual void	close() = 0;

	bool	isProcess();
	void	setProcess();
	void	endProcess();

	virtual ErrorCode getAccountId(const char *uid, char *id_ret) = 0;
	virtual ErrorCode setAccountInfo(Account *account, char *id_ret) = 0;
	virtual ErrorCode getAccountInfo(const char *id, ResultAccount **account) = 0;
	virtual ErrorCode remAcocuntInfo(const char *id) = 0;
	virtual ErrorCode getAccountCount(size_t &nRec) = 0;
	virtual ErrorCode getAccountList(size_t nFrom, size_t nCount, ResultList **pList) = 0;
	virtual ErrorCode getAccountDeviceCount(const char *aid, size_t &nRec) = 0;
	virtual ErrorCode getAccountDeviceList(const char *aid, size_t nFrom, size_t nCount, ResultList **pList) = 0;
	virtual ErrorCode getAccountGroupCount(const char *aid, size_t &nRec) = 0;
	virtual ErrorCode getAccountGroupList(const char *aid, size_t nFrom, size_t nCount, ResultList **pList) = 0;

	virtual ErrorCode getDeviceId(const char *udid, char *id_ret) = 0;
	virtual ErrorCode setDeviceInfo(DeviceInfo *device, char *id_ret) = 0;
	virtual ErrorCode getDeviceInfo(const char *id, ResultDevice **device) = 0;
	virtual ErrorCode remDeviceInfo(const char *id) = 0;
	virtual ErrorCode getDeviceCount(size_t &nRec) = 0;
	virtual ErrorCode getDeviceList(size_t nFrom, size_t nCount, ResultList **pList) = 0;
	virtual ErrorCode setDeviceShareInfo(const char *did, const DeviceShareInfo *shareInfo) = 0;
	virtual ErrorCode remDeviceShareInfo(const char *did, const char *mid) = 0;
	virtual ErrorCode getDeviceShareInfo(const char *cid, ResultDeviceShare **deviceShare) = 0;
	virtual ErrorCode getDeviceShareCount(const char *did, size_t &nRec) = 0;
	virtual ErrorCode getDeviceShareList(const char *did, size_t nFrom, size_t nCount, ResultList **pList) = 0;

	virtual ErrorCode getServiceId(const char *name, char *id_ret) = 0;
	virtual ErrorCode setServiceInfo(ServiceInfo *si, char *id_ret) = 0;
	virtual ErrorCode getServiceInfo(const char *id, ResultService **res) = 0;
	virtual ErrorCode getServicePWList(const char *id, size_t nFrom, size_t nCount, ResultList **pList) = 0;
	virtual ErrorCode isServiceWritable(const char *sid, const char *aid) = 0;
	virtual ErrorCode getServicePRList(const char *id, size_t nFrom, size_t nCount, ResultList **pList) = 0;
	virtual ErrorCode isServiceReadable(const char *sid, const char *aid) = 0;
	virtual ErrorCode remServiceInfo(const char *id) = 0;
	virtual ErrorCode setServiceRefuseList(const char *sid, const char *aid) = 0;
	virtual ErrorCode remServiceRefuseList(const char *sid, const char *aid) = 0;
	virtual ErrorCode isServiceRefuse(const char *sid, const char *aid) = 0;
	virtual ErrorCode getServiceCount(size_t &nRec) = 0;
	virtual ErrorCode getServiceList(size_t nFrom, size_t nCount, ResultList **pList) = 0;
//	virtual DBError	getServiceSessionCount(const char *rid, size_t &nRec) = 0;
//	virtual DBError	getServiceSessionList(const char *rid, size_t nFrom, size_t nCount, ResultList **pList) = 0;
	virtual ErrorCode getServiceGroupCount(const char *sid, size_t &nRec) = 0;
	virtual ErrorCode getServiceGroupList(const char *sid, size_t nFrom, size_t nCount, ResultList **pList) = 0;
	virtual ErrorCode getServiceResourceCount(const char *sid, size_t &nRec) = 0;
	virtual ErrorCode getServiceResourceList(const char *sid, size_t nFrom, size_t nCount, ResultList **pList) = 0;

	virtual ErrorCode getGroupId(const char *sid, const char *name, char *id_ret) = 0;
	virtual ErrorCode setGroup(GroupInfo *g, char *id_ret) = 0;
	virtual ErrorCode getGroup(const char *gid, ResultGroup **res) = 0;
	virtual ErrorCode remGroup(const char *gid) = 0;
	virtual ErrorCode getGroupAccountCount(const char *gid, size_t &nRec) = 0;
	virtual ErrorCode getGroupAccountList(const char *gid, size_t nFrom, size_t nCount, ResultList **pList) = 0;
	virtual ErrorCode getGroupShareCount(const char *gid, size_t &nRec) = 0;
	virtual ErrorCode getGroupShareList(const char *gid, size_t nFrom, size_t nCount, ResultList **pList) = 0;

	virtual ErrorCode isGroupAccount(const char *gid, const char *aid) = 0;
	virtual ErrorCode joinGroupAccount(const char *gid, const char *aid) = 0;
	virtual ErrorCode leaveGroupAccount(const char *gid, const char *aid) = 0;

	virtual ErrorCode getResourceId(const char *sid, const char *name, char *id_ret) = 0;
	virtual ErrorCode setResourceInfo(ResourceInfo *r, char *id_ret) = 0;
	virtual ErrorCode getResourceInfo(const char *rid, ResultResource **res) = 0;
	virtual ErrorCode remResourceInfo(const char *rid) = 0;
//	virtual ErrorCode getGadgetId(const char *rid, const char *name, char *id_ret) = 0;
	virtual ErrorCode getResourceShareCount(const char *rid, size_t &nRec) = 0;
	virtual ErrorCode getResourceShareList(const char *rid, size_t nFrom, size_t nCount, ResultList **pList) = 0;

	virtual ErrorCode getShareInfo(const char *mid, ResultShare **res) = 0;
	virtual ErrorCode setShareInfo(ShareInfo *shareInfo, char *id_ret) = 0;
	virtual ErrorCode remShareInfo(const char *mid) = 0;

	virtual ErrorCode pushSyncQueue(const char *sid, const char *gid) = 0;
	virtual ErrorCode popSyncQueue(const char *sid, char *idRet) = 0;

	/*
	virtual DBError getSessionId(const char *peerAddr, char *id_ret) = 0;
	virtual DBError	setSession(const SessionInfo *session, char *id_ret) = 0;
	virtual DBError getSession(const char *ssid, ResultSession **res) = 0;
	virtual DBError	remSession(const char *ssid) = 0;
	virtual DBError	getSessionCount(const char *rid, size_t &nRec) = 0;
	virtual DBError	getSessionList(const char *rid, size_t nFrom, size_t nCount, ResultList **pList) = 0;
*/
protected:
	Sentinel *	mSentinel;
	bool		mProcess;
};

class DBPool
{
public:

	DBPool();
	~DBPool();

	bool		addPort(DBPort *port);
	void		clear();
	DBPort *	assignPort();
	void		freePort(DBPort *port);

protected:

	class PortList {
	public:
		PortList();
		~PortList();
		void	clear();
		bool	add(DBPort *port);
		DBPort *	get(size_t pos);
		size_t	size() const;
		bool	empty() const;

		size_t		mSize;
		size_t		mBlock;
		DBPort **	mPtr;
	};

protected:
	LockObject	mLock;
	PortList	mPortList;
};


}


#endif /* DB_PORT_H_ */
