/*
 * pjson.h
 *
 *  Created on: 2012/07/05
 *      Author: yasuoki
 */

#ifndef PJSON_H_
#define PJSON_H_

#define PJSON_MIN_BLOCKSIZE 256
#define PJSON_MAX_BLOCKSIZE 8192
#define PJSON_CHUNK 4

#include "buffer.h"
#include <malloc.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

namespace pjson {

enum vt_type {
	vt_na		= 0,
	vt_null		= 1,
	vt_bool		= 2,
	vt_int		= 3,
	vt_number	= 4,
	vt_string	= 5,
	vt_object	= 6,
	vt_array	= 7,
};

enum {
	ERR_SYNTAX		= 1,
	ERR_MEMORY		= 2,
	ERR_NULL		= 3,
	ERR_BOOL		= 4,
	ERR_NUMBER		= 5,
	ERR_STRING		= 6,
	ERR_OBJECT		= 7,
	ERR_ARRAY		= 8,
	ERR_HIERARCHY	= 9,
	ERR_EMPTYOBJECT	= 10,
	ERR_EMPTYARRAY	= 11
};

typedef struct _buffer buffer;
typedef struct _object object;
typedef struct _array array;
typedef struct _value value;
typedef struct _prop prop;

struct _value {
	vt_type	vt;
	union {
		bool		vBool;
		const char *vString;
		long long	vInt;
		double	*	vNumber;
		object	*	vObject;
		array	*	vArray;
	};
};

struct _prop {
	const char *name;
	value 	val;
};

struct _object {
	size_t	n;
	object *next;
	prop *	props[1];
};

struct _array {
	size_t	n;
	array * next;
	value *	values[1];
};

struct _buffer {
	size_t	nAlloc;
	size_t	nUse;
	buffer *next;
	char 	data[1];
};

size_t getArraySize(value *arr);
value *getArrayContent(value *arr, size_t index);

class json
{
	friend class parser;
	friend class builder;
public:
	json();
	~json();
	value *	getValue() const;
	value * get(const char *path, value *from=NULL );
	bool	getText(SST::Buffer *b, bool wf=true) const;
	static value * getFrom(pjson::value *from, const char *path);
	static void dumpjson(const char *tab, pjson::json *js);

protected:
	static bool	dump_na(SST::Buffer *b, bool wf, const char *tab, pjson::value *v);
	static bool dump_null(SST::Buffer *b, bool wf, const char *tab, pjson::value *v);
	static bool dump_bool(SST::Buffer *b, bool wf, const char *tab, pjson::value *v);
	static bool dump_str(SST::Buffer *b, bool wf, const char *tab, pjson::value *v);
	static bool dump_int(SST::Buffer *b, bool wf, const char *tab, pjson::value *v);
	static bool dump_num(SST::Buffer *b, bool wf, const char *tab, pjson::value *v);
	static bool dump_obj(SST::Buffer *b, bool wf, const char *tab, pjson::value *v);
	static bool dump_arr(SST::Buffer *b, bool wf, const char *tab, pjson::value *v);
	static bool dump_val(SST::Buffer *b, bool wf, const char *tab, pjson::value *v);

	buffer *allocBufferSeg(size_t size);
	bool	allocBuffer(size_t size);
	void *	getBuffer(size_t size);
	buffer *pBuff;
	buffer *pFree;
	value *	pTop;
};

typedef struct _CHUNK CHUNK;
struct _CHUNK {
	CHUNK *	nStack;
	size_t	n;
	value * pc;
	union {
		object *	pObject;
		array *		pArray;
	};
	union {
		prop *	pchunk[PJSON_CHUNK];
		value *	vchunk[PJSON_CHUNK];
	};
};

class builder
{
public:
	builder();
	~builder();
	bool	init(size_t size);
	void	clear();
	json *	detouch();
	void *	getBuffer(size_t size);
	int		getError();
	void	setError(int code);
	void	setErrorMsg(const char *p);
	const char *getErrorMsg();

	bool	beginObject();
	bool	addObjectProp(const char *name, size_t len=0);
	bool	endObject();
	bool	beginArray();
	bool	addArrayContent();
	bool	endArray();
	bool	valueNull();
	bool	valueBool(bool val);
	bool	valueInt(long long val);
	bool	valueNumber(double val);
	bool	valueBufString(const char *val);
	bool	valueString(const char *val);

	size_t	escapeSize(const char *val);
	size_t	escapeText(char *dest, size_t dsize, const char *val);

protected:

	char *	errMsg;
	int		err;
	value *	pc;
	json *	bld;
	json *	js;
	CHUNK *	st;
};
class parser
{
public:
	parser();
	~parser();
	json *parse(const char *text, size_t textSize);
	int		getErrorPos();
	int 	getErrorCode();
//	const char *getErrorMessage();
protected:
	char skipwsp();
	char getc();
	void ungetc();
	bool match(const char *pat);
	bool parseValue();
	bool parseNull();
	bool parseBool();
	bool parseString();
	bool parseNumber();
	bool parseObject();
	bool parseArray();

	builder	bd;
	const char *ptr;
	const char *src;
};

}

#endif /* PJSON_H_ */
