/*
 * listener.cpp
 *
 *  Created on: 2012/07/04
 *      Author: yasuoki
 */

#include "sentinel.h"
#include "proc_listener.h"
#include "commands.h"
#include "scheduler.h"
#include "task.h"
#include "lock.h"
#include <errno.h>
#include <memory.h>
#include <pthread.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <assert.h>

namespace SST {

QueueCommSend::QueueCommSend(Task *task, Session *session) : QueueNode(task)
{
	mType		= (int)QueueTypeComSend;
	mProc		= &task->getSentinel()->getListener();
	mFunc		= (int)CMDComSend;
	mSession	= session;
}
QueueCommSend::~QueueCommSend()
{
}

QueueCommRecvRaw::QueueCommRecvRaw(Task *task, size_t size) : QueueNode(task)
{
	mType		= (int)QueueTypeComRecvRaw;
	mProc		= &task->getSentinel()->getListener();
	mFunc		= (int)CMDComRecvRaw;
	mSize		= size;
}

QueueCommRecvRaw::~QueueCommRecvRaw()
{
}

QnCommRecvResult::QnCommRecvResult(Task *task) : QnResult(task)
{
	mType		= (int)QueueTypeComRecvResult;
	mData		= NULL;
}

QnCommRecvResult::~QnCommRecvResult()
{
	if( mData ) delete mData;
}

//////////////////////////////////////////////////////////////////////////

void Listener::connectCB(int sFd, short event, void *arg)
{
	((Listener*)arg)->connectProc(sFd, event);
}

void Listener::connectProc(int sFd, short event)
{
	taskLog(NULL, LOG_DEBUG, "Listener:connectProc: begin. fd=%d", sFd);

	sockaddr_in caddr;
	socklen_t addrLen=sizeof(caddr);
	int cfd = accept(sFd, (sockaddr *)&caddr, &addrLen);
	int val = 1;
	ioctl(cfd, FIONBIO, &val);

	Session *con = mSentinel->getSessionPool().getSession();
	if( con == NULL ) {
		taskLog(NULL, LOG_WARNING, "Listener:connectProc: No more connection. fd=%d addr=%s:%d", cfd, inet_ntoa(caddr.sin_addr), ntohs(caddr.sin_port));
		close(cfd);

		Task *task = mSentinel->getTaskManager().createTaskFromSession(NULL);
		QueueNode *q = new QueueNode(task, this, CMDComPurge);
		task->startTask(q);

		return;
	}
	if( !con->open(mSentinel, cfd, &caddr) ) {
		mSentinel->getSessionPool().freeSession(con);
		close(cfd);
		return;
	}
	event_add(sListenEvent, NULL);

	Task *task = mSentinel->getTaskManager().createTaskFromSession(con);
	QueueNode *q = new QueueNode(task, this, CMDComAccept);
	task->startTask(q);
	taskLog(q, LOG_DEBUG, "end task=%d fd=%d", task->getTaskId(), sFd);
	return;
}

void Listener::reciveCB(int sFd, short event, void *arg)
{
	Session *con	= (Session*)arg;
	Sentinel *sentinel = con->getSentinel();
	sentinel->getListener().reciveProc(con, event);
}

void Listener::reciveProc(Session *con, short event)
{
	Task *task = NULL;
	{
		READ_LOCK(mLock);
		if( con->getStatus() == SessionStatusWait ) {
			return;
		}
		task = con->getReciveTask();
	}

	QueueNode *q = NULL;
	if( task == NULL ) {
		task = mSentinel->getTaskManager().createTaskFromSession(con);
		taskLog(NULL, LOG_DEBUG, "Listener:reciveProc: new task=%d fd=%d", task->getTaskId(), con->getDiscriptor());
		q = new QueueNode(task, this, CMDComRecive);//QnComRecive(task);
		task->startReciveTask(q);
	} else {
		taskLog(NULL, LOG_DEBUG, "Listener:reciveProc: continue task=%d fd=%d", task->getTaskId(), con->getDiscriptor());
		q = new QueueNode(task, this, CMDComRecive);//QnComRecive(task);
		task->putQueue(q);
	}

	taskLog(q, LOG_DEBUG, "end. task=%d fd=%d", task->getTaskId(), con->getDiscriptor());
	return;
}

Listener::Listener()
{
	mMaxThread		= 32;
	mMinThread		= 6;
	sFd = 0;
	sListenBase = NULL;
	sListenEvent = NULL;
}

Listener::~Listener()
{
}

bool Listener::configure(Sentinel *obj, Conf *conf)
{
	if( !Process::configure(obj, conf) )
		return false;

	mMaxThread	= conf->maxDeviceProcess;
	mMinThread	= conf->minDeviceProcess;

	if( sFd == 0 ) {
		struct sockaddr_in srcAddr;
		memset(&srcAddr, 0, sizeof(srcAddr));
		srcAddr.sin_port = htons(conf->port);
		srcAddr.sin_family = AF_INET;
		srcAddr.sin_addr.s_addr = htonl(INADDR_ANY);

		int s = socket(AF_INET, SOCK_STREAM, 0);
		if( s == -1 ) {
			taskLog(NULL, LOG_ERR, "tcp socket can't open");
			return false;
		}
		char optval = 1/*1*/;
		::setsockopt( s, SOL_SOCKET, SO_REUSEADDR, (char*)&optval, sizeof(char) );
		if( bind(s, (struct sockaddr *) &srcAddr, sizeof(srcAddr)) == -1 ) {
			taskLog(NULL, LOG_ERR, "tcp socket can't bind");
			return false;
		}

		if( listen(s, 1) == -1 ) {
			taskLog(NULL, LOG_ERR, "tcp socket can't listen");
			return false;
		}

		sFd = s;
		sListenBase	= event_base_new();
		sListenEvent = event_new(sListenBase, sFd, EV_READ, Listener::connectCB, this);
		event_add(sListenEvent, NULL);
	}
	return true;
}

void Listener::clear()
{
	WRITE_LOCK(mQueue.getLockObject());
 	if( sListenEvent ) {
 		event_del(sListenEvent);
 		event_free(sListenEvent);
 		sListenEvent	= NULL;
 	}
 	if( sListenBase ) {
 		event_base_free(sListenBase);
 		sListenBase	= NULL;
 	}
 	if( sFd ) {
 		::close(sFd);
 	}
}

const char *Listener::getFuncName(int id)
{
 	switch(id) {
	case CMDComListen:		return "Listen";
	case CMDComAccept:		return "Accept";
	case CMDComRecive:		return "Recive";
	case CMDComResponse:	return "Response";
	case CMDComSend:		return "Send";
	case CMDComPurge:		return "Purge";
	case CMDComRecvRaw:		return "ReciveRaw";
	}
 	return Process::getFuncName(id);
}

bool Listener::exec(QueueNode *q)
{
 	int func = q->getFunction();
 	switch(func) {
	case CMDComListen:		return cmdListen(q);
	case CMDComAccept:		return cmdAccept(q);
	case CMDComRecive:		return cmdRecive(q);
	case CMDComResponse:	return cmdResponse(q);
	case CMDComSend:		return cmdSend(q);
	case CMDComPurge:		return cmdPurge(q);
	case CMDComRecvRaw:		return cmdRecvRaw(q);
 	default:
	 	taskLog(q, LOG_WARNING, "unknown command . q=%x func=%d step=%d.", q, q->getFunction(), q->getStep());
	 	return false;
	}
 	return true;
}

void Listener::stop(QueueNode *q)
{
	WRITE_LOCK(mQueue.getLockObject());
 	if( sListenBase ) {
 		event_base_loopbreak(sListenBase);
 	}
}

bool Listener::waitData(Session *con)
{
	struct event *ev = con->assignEvent(sListenBase, Listener::reciveCB);
    event_add(ev, NULL);
    return true;
}

bool  Listener::response(QueueNode *q, pjson::json *js)
{
	Task *task = q->getTask();
	TaskBuffer *tb = task->getTaskBuffer();
	if( tb ) delete tb;
	task->setTaskBuffer(NULL);

	QueueNode *nq = new QueueNode(task);
	nq->setProcess(this, CMDComResponse, 0);
	nq->setJSONMessage(js);
	task->putQueue(nq);
	return true;
}

bool Listener::taskResponseLogVa(int logLevel, QueueNode *q, ErrorCode code, const char *fmt, va_list ap)
{
	StdBuffer buf;

	if( logLevel <= LOG_ERR ) {
		va_list ap2;
		va_copy(ap2, ap);
		mSentinel->logv(logLevel, fmt, ap2);
		va_end(ap2);
	}
	int seq = 0;
	if( q && q->getTask() ) {
		seq = q->getTask()->getSequence();
	}

	buf.addvafmt(fmt, ap);

	pjson::builder jb;
	if( !jb.init(256) ||
		!jb.beginObject() ||
		!jb.addObjectProp("result", 6) ||
		!jb.valueInt(code) ||
		!jb.addObjectProp("sequence", 8) ||
		!jb.valueInt(seq) ||
		!jb.addObjectProp("message", 7) ||
		!jb.valueString(buf.getPtr()) ||
		!jb.endObject() ) {
		taskLog(q, LOG_ERR, "json builder failed code=%d", jb.getError());
		return false;
	}
	return response(q, jb.detouch());
}


bool Listener::cmdListen(QueueNode *q)
{
	taskLog(q, LOG_DEBUG, "begin event dispatch.");
	event_base_dispatch(sListenBase);
	q->getTask()->endTask(NULL);
	taskLog(q, LOG_DEBUG, "end event dispatch.");
	return true;
}

bool Listener::cmdAccept(QueueNode *q)
{
	Task *task = q->getTask();
	Session *con = task->getSession();
	sockaddr_in addr;
	con->getPeerAddress(&addr);

	pjson::builder jb;
	jb.init(256);

	Conf *cf = mSentinel->getConf();

//	char *nonce = con->createNonce();
	if( !jb.beginObject() ) {
		taskLog(q, LOG_ERR, "json builder failed(1). code=%d addr=%s:%d.", jb.getError(), inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
		return false;
	}
	if( !jb.addObjectProp("sentinel",8) ||
		!jb.beginObject() ||
		!jb.addObjectProp("version",7) ||
		!jb.beginArray() ||
		!jb.addArrayContent() ||
		!jb.valueString("1.0") ||
		!jb.endArray() ) {
		taskLog(q, LOG_ERR, "json builder failed(2). code=%d addr=%s:%d.", jb.getError(), inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
		return false;
	}

	if( cf->serverName ) {
		if( !jb.addObjectProp("server", 6) ||
			!jb.beginArray() ||
			!jb.addArrayContent() ||
			!jb.beginObject() ||
			!jb.addObjectProp("host", 4) ||
			!jb.valueString(cf->serverName) ||
			!jb.addObjectProp("port", 4) ||
			!jb.valueInt(cf->port) ||
			!jb.addObjectProp("load", 4) ||
			!jb.valueInt(mSentinel->getLoad()) ||
			!jb.endObject() ||
			!jb.endArray()  ) {
			taskLog(q, LOG_ERR, "json builder failed(3). code=%d addr=%s:%d.", jb.getError(), inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
			return false;
		}
	}
	if( !jb.endObject() || !jb.endObject() ) {
		taskLog(q, LOG_ERR, "json builder failed(4). code=%d.", jb.getError());
		return false;
	}

//	if( !waitData(con) ) {
//		taskLog(q, LOG_ERR, "waitData error addr=%s:%d.", jb.getError(), inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
//		return false;
//	}
	con->setStatus(SessionStatusConnect);

	Task *purgeTask = mSentinel->getTaskManager().createTaskFromBaseTask(task);
	QueueNode *purgeCmd = new QueueNode(purgeTask, this, CMDComPurge);
	purgeTask->startTask(purgeCmd);


	StdBuffer jsonBuff;
	pjson::json *js = jb.detouch();
	return response(q, js);
	/*
	js->getText(&jsonBuff, true);

	size_t size = jsonBuff.size();
	const char * ptr = jsonBuff.getPtr();
	ssize_t ws = write(fd, ptr, size);
	int wt_err = errno;
	*/
/*
	if( ws > 0 ) {
		buff->seek(ws);
	}
	if( ws < (ssize_t)size ) {
		if( wt_err != EAGAIN && wt_err != EWOULDBLOCK ) {
			taskLog(q, LOG_ERR, "Listener: write to device socket error. q=%x code=%d", q, wt_err);
			return false;
		}
		taskLog(q, LOG_INFO, "Listener: continue write. q=%x, data=%u.", q, size-ws);
		mSentinel->getScheduler().setQueueTimer(this, q, 1);
		return true;
	}
	*/
}

class TBRecive : public TaskBuffer {
public:
	TBRecive() : TaskBuffer() {
	}
	virtual ~TBRecive() {
	}
	StdBuffer	mRecvBuff;
};

bool Listener::cmdRecive(QueueNode *q)
{
	Task *task = q->getTask();
	Session *con = task->getSession();

	TBRecive *tb = (TBRecive*)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBRecive();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	char buff[4+4096+1];
//	int fd = con->getDiscriptor();
	bool	cmdEnd = false;
	ssize_t	cmdEndPos = 0;
	ssize_t cmdEndSkip = 0;

	ssize_t lead_cr=0;
	buff[0] = 0;
	if( !tb->mRecvBuff.empty() ) {
		size_t bs = tb->mRecvBuff.size();
		const char *tailPtr;
		if( bs <= 4) {
			tailPtr = tb->mRecvBuff.getPtr();
			memcpy(buff, tailPtr, bs);
			lead_cr = bs;
		} else {
			tailPtr = tb->mRecvBuff.getPtr() + bs - 4;
			memcpy(buff, tailPtr, 4);
			lead_cr = 4;
		}
		buff[lead_cr] = 0;
	}

	char deb[512];
	{
		int i;
		int n=0;
		for(i = 0; i < lead_cr && n < (int)(sizeof(deb)-1); i++ ) {
			char c = buff[i];
			if( c < ' ') {
				char c0 = (c >> 4) &0x0f;
				char c1 = c &0x0f;
				deb[n++] = 'x';
				deb[n++] = c0 >= 10 ? (c0-10+'a') : (c0+'0');
				deb[n++] = c1 >= 10 ? (c1-10+'a') : (c1+'0');
				deb[n++] = ' ';
			} else {
				deb[n++] = c;
			}
		}
		deb[n] = 0;
	}

	taskLog(q, LOG_DEBUG, "tailSize = %d <%s>", lead_cr, deb);
	while(1) {
		ssize_t len = con->read(&buff[lead_cr], 4096);
		if( len == -1 ) {
			if( errno == EAGAIN || errno == EWOULDBLOCK ) {
				break;
			}
			taskLog(q, LOG_ERR, "i/o error %d %s", errno, strerror(errno) );
			return false;
		}
		if( len == 0 ) {
			if( con->getUserId() ) {
				taskLog(q, LOG_INFO, "user %s logout.", con->getUserId());
			}
			if( tb->mRecvBuff.size() != 0 )
				taskLog(q, LOG_INFO, "connection closed. recive data is ommited. size=%d", tb->mRecvBuff.size());
			else
				taskLog(q, LOG_INFO, "connection closed.");
			// TODO: logout process
			mSentinel->getSessionPool().freeSession(con);
			task->endTask(NULL);
			return true;
		}

		{
			int i;
			int n=0;
			for(i = 0; i < len && n <(int)( sizeof(deb)-1); i++ ) {
				char c = buff[i+lead_cr];
				if( c < ' ') {
					char c0 = (c >> 4) &0x0f;
					char c1 = c &0x0f;
					deb[n++] = 'x';
					deb[n++] = c0 >= 10 ? (c0-10+'a') : (c0+'0');
					deb[n++] = c1 >= 10 ? (c1-10+'a') : (c1+'0');
					deb[n++] = ' ';
				} else {
					deb[n++] = c;
				}
			}
			deb[n] = 0;
			taskLog(q, LOG_DEBUG, "recive %d bytes <%s>", len, deb);
		}

		buff[lead_cr+len] = 0;
		const char *ptr = buff;
		for( ssize_t i = 0; i < lead_cr+len; i++ ) {
			if( *ptr == '\r' || *ptr == '\n' ) {
				size_t rest = lead_cr+len-i;
				if( rest >= 4 && *ptr == '\r' && *(ptr+1) == '\n' && *(ptr+2) == '\r' && *(ptr+3) == '\n' ) {
					cmdEnd	= true;
					cmdEndPos	= i+4;
					cmdEndSkip	= 4;
					break;
				}
				if( rest >= 2 && *ptr == '\n' && *(ptr+1) == '\n' ) {
					cmdEnd	= true;
					cmdEndPos	= i+2;
					cmdEndSkip	= 2;
					break;
				}
			}
			ptr++;
		}

		if( cmdEnd ) {
			// xxxxc..next
			//    ^   ^   ^
			//    |   |   +- len(7)
			//    |   +----- cmdLen(7)
			//    +--------- lead_cr(4)
			if( cmdEndPos > lead_cr ) {
				if( !tb->mRecvBuff.add(&buff[lead_cr], cmdEndPos-lead_cr) ) {
					taskLog(q, LOG_ERR, "recive buffer full q=%x len=%d", q, len);
					return false;
				}
			}
			if( lead_cr+len > cmdEndPos) {
				con->unread(&buff[cmdEndPos], (lead_cr+len)-cmdEndPos);
				{
					int urlen=(lead_cr+len)-cmdEndPos;
					char *urptr = &buff[cmdEndPos];
					int i;
					int n=0;
					for(i = 0; i < urlen && n < (int)(sizeof(deb)-1); i++ ) {
						char c = urptr[i];
						if( c < ' ') {
							char c0 = (c >> 4) &0x0f;
							char c1 = c &0x0f;
							deb[n++] = 'x';
							deb[n++] = c0 >= 10 ? (c0-10+'a') : (c0+'0');
							deb[n++] = c1 >= 10 ? (c1-10+'a') : (c1+'0');
							deb[n++] = ' ';
						} else {
							deb[n++] = c;
						}
					}
					deb[n] = 0;
					taskLog(q, LOG_DEBUG, "unread %d bytes <%s>", urlen, deb);
				}
			}
			break;
		} else {
			if( !tb->mRecvBuff.add(&buff[lead_cr], len) ) {
				taskLog(q, LOG_ERR, "recive buffer full q=%x len=%d", q, len);
				return false;
			}
			ssize_t tail;
			if( len < 4 )
				tail = len;
			else
				tail = 4;

			ssize_t head_shift = (lead_cr + tail) - 4;
			ssize_t head;
			if( head_shift <= 0 ) {
				head = lead_cr;
			} else {
				if( lead_cr <= head_shift ) {
					head = 0;
				} else {
					memmove(buff, &buff[head_shift], lead_cr - head_shift);
					head = lead_cr - head_shift;
				}
			}
			memcpy(&buff[head], &buff[lead_cr+len-tail], tail);
			lead_cr = head+tail;
		}
	}

	if( !cmdEnd ) {
//		mSentinel->getListener().waitData(con);
		waitData(con);
		taskLog(q, LOG_DEBUG, "not end");
		return true;
	}

	pjson::json *js = NULL;
	size_t readSize = tb->mRecvBuff.size();
	if( readSize <= (size_t)cmdEndSkip ) {
//		mSentinel->getListener().waitData(con);
		waitData(con);
		taskLog(q, LOG_DEBUG, "empty");
		tb->mRecvBuff.clear();
		return true;
	}
	const char *readData = tb->mRecvBuff.getPtr();

	pjson::parser ps;
	js = ps.parse(readData, readSize);

	if( js == NULL ) {
		taskLog(q, LOG_ERR, "json parse error. session=%x task=%d q=%x pos=%d code=%d.", con, task->getTaskId(), q, ps.getErrorPos(), ps.getErrorCode());
		return false;
	}

	const char *cmd = NULL;
	int seq=0;

	pjson::value *v = js->getValue();
	if( v->vt == pjson::vt_string ) {
		cmd = v->vString;
	} else if( v->vt == pjson::vt_object ) {
		cmd = v->vObject->props[0]->name;
		pjson::value *vSeq = js->get("sequence", &v->vObject->props[0]->val);
		if( vSeq && vSeq->vt == pjson::vt_int ) {
			seq = vSeq->vInt;
		}
	} else {
		taskLog(q, LOG_ERR, "command format error. q=%x.", q);
		delete js;
		return false;
	}

	size_t cmdLen = strlen(cmd);
	Command cmdId = CMDNull;
	switch(*cmd) {
	case 'a':
		if( cmdLen == 8 && strcmp(cmd, "addGroup") == 0 )
			cmdId	= CMDAddGroup;
		break;
	case 'd':
		if( cmdLen == 8 && strcmp(cmd, "downSync") == 0 )
			cmdId	= CMDDownSync;
		break;
//	case 'e':
//		if( cmdLen == 12 && strcmp(cmd, "eraseAccount") == 0 )
//			cmdId	= CMDEraseAccount;
//		break;
	case 'j':
		if( cmdLen == 9 && strcmp(cmd, "joinGroup") == 0 )
			cmdId	= CMDJoinGroup;
		break;
	case 'l':
		if( cmdLen == 5 && strcmp(cmd, "login") == 0 )
			cmdId	= CMDLogin;
		else if( cmdLen == 7 && strcmp(cmd, "lsGroup") == 0 )
			cmdId	= CMDLsGroup;
		else if( cmdLen == 8 && strcmp(cmd, "lsDevice") == 0 )
			cmdId	= CMDLsDevice;
		else if( cmdLen == 9 && strcmp(cmd, "lsAccount") == 0 )
			cmdId	= CMDLsAccount;
		else if( cmdLen == 9 && strcmp(cmd, "lsService") == 0 )
			cmdId	= CMDLsService;
		else if( cmdLen == 10 && strcmp(cmd, "lsResource") == 0 )
			cmdId	= CMDLsResource;
		else if( cmdLen == 10 && strcmp(cmd, "leaveGroup") == 0 )
			cmdId	= CMDLeaveGroup;
		else if( cmdLen == 11 && strcmp(cmd, "lsGroupUser") == 0 )
			cmdId	= CMDLsGroupUser;
		else if( cmdLen == 11 && strcmp(cmd, "lockAccount") == 0 )
			cmdId	= CMDLockAccount;
		else if( cmdLen == 12 && strcmp(cmd, "lsGroupShare") == 0 )
			cmdId	= CMDLsGroupShare;
		else if( cmdLen == 13 && strcmp(cmd, "lsDeviceShare") == 0 )
			cmdId	= CMDLsDeviceShare;
		else if( cmdLen == 14 && strcmp(cmd, "lsServiceGroup") == 0 )
			cmdId	= CMDLsServiceGroup;
		else if( cmdLen == 15 && strcmp(cmd, "lsResourceShare") == 0 )
			cmdId	= CMDLsResourceShare;
		break;
	case 'r':
		if( cmdLen == 5 && strcmp(cmd, "rmService") == 0 )
			cmdId	= CMDRmService;
		else if( cmdLen == 7 && strcmp(cmd, "rmGroup") == 0 )
			cmdId	= CMDRmGroup;
		else if( cmdLen == 8 && strcmp(cmd, "rmDevice") == 0 )
			cmdId	= CMDRmAccount;
		else if( cmdLen == 9 && strcmp(cmd, "rmAccount") == 0 )
			cmdId	= CMDRmAccount;
		else if( cmdLen == 10 ) {
			if( strcmp(cmd, "regAccount") == 0 )
				cmdId	= CMDRegAccount;
			else if( strcmp(cmd, "rmResource") == 0 )
				cmdId	= CMDRmResource;
			else if( strcmp(cmd, "regService") == 0 )
				cmdId	= CMDRegService;
		}
		break;
	case 's':
		if( cmdLen == 4 && strcmp(cmd, "sync") == 0 )
			cmdId	= CMDSync;
		else if( cmdLen == 5 && strcmp(cmd, "share") == 0 )
			cmdId	= CMDShare;
		else if( cmdLen == 8 ) {
			if(  strcmp(cmd, "syncStat") == 0 )
				cmdId	= CMDSyncStat;
			else if( strcmp(cmd, "setGroup") == 0 )
				cmdId	= CMDSetGroup;
		} else if( cmdLen == 9 ) {
			if(  strcmp(cmd, "showShare") == 0 )
				cmdId	= CMDShowShare;
			else if( strcmp(cmd, "showGroup") == 0 )
				cmdId	= CMDShowGroup;
			else if( strcmp(cmd, "syncQueue") == 0 )
				cmdId	= CMDSyncQueue;
		} else if( cmdLen == 10 ) {
			if( strcmp(cmd, "setAccount") == 0 )
				cmdId	= CMDSetAccount;
			else if( strcmp(cmd, "showDevice") == 0 )
				cmdId	= CMDShowDevice;
			else if( strcmp(cmd, "setService") == 0 )
				cmdId	= CMDSetService;
		} else if( cmdLen == 11 ) {
			if( strcmp(cmd, "showAccount") == 0 )
				cmdId	= CMDShowAccount;
			else if( strcmp(cmd, "showService") == 0 )
				cmdId	= CMDShowService;
			else if( strcmp(cmd, "syncAccount") == 0 )
				cmdId	= CMDSyncAccount;
		} else if( cmdLen == 12 && strcmp(cmd, "showResource") == 0 )
				cmdId	= CMDShowResource;
		break;
	case 't':
		if( cmdLen == 4 ) {
			if(  strcmp(cmd, "test") == 0 )
				cmdId	= CMDDebugTest;
		}
		break;

	case 'u':
		if( cmdLen == 6 ) {
			if(  strcmp(cmd, "update") == 0 )
				cmdId	= CMDUpdate;
			else if( strcmp(cmd, "upSync") == 0 )
				cmdId	= CMDUpSync;
		} else if( cmdLen == 7 && strcmp(cmd, "unshare") == 0 )
			cmdId	= CMDUnshare;
		else if( cmdLen == 13 && strcmp(cmd, "unlockAccount") == 0 )
			cmdId	= CMDUnlockAccount;

		break;
	}

	// diag command
	if( cmdId == 0 ) {
		if( strcmp(cmd, "@session") == 0 )
			cmdId	= CMDDiagSession;
		else if( strcmp(cmd, "@task") == 0 )
			cmdId	= CMDDiagTask;
		else if( strcmp(cmd, "@queue") == 0 )
			cmdId	= CMDDiagQueue;
		else if( strcmp(cmd, "@process") == 0 )
			cmdId	= CMDDiagProcess;
		else if( strcmp(cmd, "@thread") == 0 )
			cmdId	= CMDDiagThread;
		else if( strcmp(cmd, "@test") == 0 )
			cmdId	= CMDDebugTest;
	}

	delete tb;
	task->setTaskBuffer(NULL);

	if( cmdId == 0 ) {
		if( strcmp(cmd, "@purge") == 0 ) {
			QueueNode *purgeCmd = new QueueNode(task, this, CMDComPurge);
			return task->call(purgeCmd, 0, CMDComResponse);
		}
		taskLog(q, LOG_ERR, "unknown command error. q=%x cmd=%s.", q, cmd);
		delete js;
		return false;
	}

	//Task *cmdTask = mSentinel->getTaskManager().createTaskFromSession(con);
	//QueueNode *cmd_q = new QueueNode(cmdTask, &mSentinel->getCommand(), cmdId);
	//cmd_q->setJSONMessage(js);
	//cmdTask->setSequence(seq);
	//return cmdTask->startTask(cmd_q);

	QueueNode *cmd_q = new QueueNode(task, &mSentinel->getCommand(), cmdId);
	cmd_q->setJSONMessage(js);
	cmd_q->setSequence(seq);
	task->setSequence(seq);

	if( con->isCommandQueueEmpty() && task->getTaskStackSize() == 0 ) {
		return task->call(cmd_q, 0, CMDComResponse);
	}
	con->putNextCommand(cmd_q);
	return true;
}

bool Listener::makeResponseMessage(QueueNode *q, Buffer *buff)
{
	const pjson::json *jsr = q->getJSONMessage();
	const pjson::json *js = NULL;
	if( jsr == NULL ) {
		int seq = 0;
		if( q && q->getTask() ) {
			seq = q->getTask()->getSequence();
		}
		ErrorCode code = ErrorOk;
		if( q->getType() == QueueTypeResult ){
			code = ((QnResult*)q)->getResultCode();
		}
		Buffer *msg = q->getMessage();
		pjson::builder jb;
		if( !jb.init(1024) ||
			!jb.beginObject() ||
			!jb.addObjectProp("result",6) ||
			!jb.valueInt(code) ||
			!jb.addObjectProp("sequence",8) ||
			!jb.valueInt(seq) ) {
			return false;
		}
		if( msg && !msg->empty() ) {
			if( !jb.addObjectProp("message",7) ||
				!jb.valueString(msg->getPtr()) ) {
				return false;
			}
		}
		Task *t = q->getTask();
		if( t ) {
			const char *sessin_id = t->getUserId();
			const char *sessin_sv = t->getServiceId();
			if( sessin_id ) {
				if( !jb.addObjectProp("sessionUser",11) ||
					!jb.valueString(sessin_id) ) {
					return false;
				}
			}
			if( sessin_sv ) {
				if( !jb.addObjectProp("sessionService",14) ||
					!jb.valueString(sessin_sv) ) {
					return false;
				}
			}
		}
		if( !jb.endObject() ) {
			return false;
		}
		js = jb.detouch();
		if( !js ) return false;
		jsr	= js;
	}
	if( !jsr->getText(buff, true) ) {
		if( js ) delete js;
		return false;
	}
	if( js ) delete js;
	if( !buff->add("\r\n\r\n", 4) ) {
		return false;
	}
	return true;
}

class TBResponse :  public TaskBuffer {
public:
	TBResponse() {

	}
	virtual ~TBResponse() {

	}
	StdBuffer mSendBuff;
};

bool Listener::cmdResponse(QueueNode *q)
{
	Task *task = q->getTask();
	Session *con = task->getSession();

	TBResponse *tb = (TBResponse*)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBResponse();
		taskLog(q, LOG_DEBUG, "new task buffer task=%d tb=%x", task->getTaskId(), tb);
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( q->getType() == QueueTypeRawResult ) {
			tb->mSendBuff.clear();
			tb->mSendBuff.add(q->mMsg->getPtr(), q->mMsg->size());
		} else {
			if( !makeResponseMessage(q, &tb->mSendBuff)) {
				tb->mSendBuff.clear();
				tb->mSendBuff.addfmt("{\"result\":999}\r\n\r\n");
			}
			taskLog(q, LOG_DEBUG, "%s", tb->mSendBuff.getPtr());
		}
	} else {
		taskLog(q, LOG_DEBUG, "load task buffer task=%d tb=%x", task->getTaskId(), tb);
	}

	if( con ) {
		int fd = con->getDiscriptor();
		if( fd == 0 ) {
			taskLog(q, LOG_INFO, "output socket is closed. q=%x.", q);
		} else {
			size_t total_write = 0;
			size_t total_size  = tb->mSendBuff.size();
			while( total_write < total_size ) {
				const char * ptr = tb->mSendBuff.getPtr();
				size_t psize = 4096;
				if( psize > total_size - total_write ) {
					psize = total_size - total_write;
				}
				ssize_t ws = write(fd, ptr, psize);
				int wt_err = errno;
				if( ws < 0 ) {
					taskLog(q, LOG_DEBUG, "write to device socket error. task=%d q=%lld, data=%u/%u.", task->getTaskId(), q->getId(), total_write, total_size);
					if( wt_err != EAGAIN && wt_err != EWOULDBLOCK ) {
						taskLog(q, LOG_ERR, "write to device socket error. task=%d q=%lld code=%d.", task->getTaskId(), q->getId(), wt_err);
						return false;
					}
					QueueNode *qn = new QueueNode(task, this, q->getFunction());
					taskLog(q, LOG_DEBUG, "continue write. task=%d q=%lld next q=%lld next data=%u.", task->getTaskId(), q->getId(), qn->getId(), tb->mSendBuff.size());
					return mSentinel->getScheduler().setQueueTimer(qn,1);
				}
				if( ws > 0 ) {
					total_write += (size_t)ws;
					tb->mSendBuff.seek(ws);
					taskLog(q, LOG_DEBUG, "write to device socket task=%d q=%lld, data=%u/%u.", task->getTaskId(), q->getId(), total_write, total_size);
				}
			}
		}
	}

	taskLog(q, LOG_DEBUG, "write to device end.");

	QueueNode *nextq = NULL;
	if( con ) {
		nextq = con->getNextCommand();
	}

	taskLog(q, LOG_DEBUG, "next queue = %x.", nextq);
	task->endTask(NULL);
	if( nextq ) {
		Task *cmdTask = mSentinel->getTaskManager().createTaskFromSession(con);
		nextq->mTask	= cmdTask;
		cmdTask->setSequence(nextq->getSequence());
		cmdTask->startTask(nextq);
	} else {
		waitData(con);
		taskLog(q, LOG_DEBUG, "wait data start.");
	}

//	mSentinel->getListener().waitData(con);

	return true;
}

bool Listener::cmdSend(QueueNode *q)
{
	Task *task = q->getTask();
	QueueCommSend *qs = (QueueCommSend*)q;
	Session *sendCon = qs->mSession;

	TBResponse *tb = (TBResponse*)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBResponse();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}

		if( !makeResponseMessage(q, &tb->mSendBuff)) {
			tb->mSendBuff.clear();
			tb->mSendBuff.addfmt("{\"result\":999}\r\n\r\n");
		}
 	}

	if( sendCon && (sendCon->getStatus() == SessionStatusConnect || sendCon->getStatus() == SessionStatusLogin) ) {
		int fd = sendCon->getDiscriptor();
		if( fd == 0 ) {
			taskLog(q, LOG_INFO, "output socket is closed. q=%x.", q);
		} else {
			size_t total_write = 0;
			size_t total_size  = tb->mSendBuff.size();
			while( total_write < total_size ) {
				const char * ptr = tb->mSendBuff.getPtr();
				size_t psize = 4096;
				if( psize > total_size - total_write ) {
					psize = total_size - total_write;
				}
				ssize_t ws = write(fd, ptr, psize);
				int wt_err = errno;
				if( ws < 0 ) {
					taskLog(q, LOG_DEBUG, "write to device socket error. task=%d q=%lld, data=%u/%u.", task->getTaskId(), q->getId(), total_write, total_size);
					if( wt_err != EAGAIN && wt_err != EWOULDBLOCK ) {
						taskLog(q, LOG_ERR, "write to device socket error. task=%d q=%lld code=%d.", task->getTaskId(), q->getId(), wt_err);
						return false;
					}
					QueueCommSend *qn = new QueueCommSend(task,sendCon);
					taskLog(q, LOG_DEBUG, "continue write. task=%d q=%lld next q=%lld next data=%u.", task->getTaskId(), q->getId(), qn->getId(), tb->mSendBuff.size());
					return mSentinel->getScheduler().setQueueTimer(qn,1);
				}
				if( ws > 0 ) {
					total_write += (size_t)ws;
					tb->mSendBuff.seek(ws);
					taskLog(q, LOG_DEBUG, "write to device socket task=%d q=%lld, data=%u/%u.", task->getTaskId(), q->getId(), total_write, total_size);
				}
			}
		}

	}

	QnResult *ret = new QnResult(task);
	ret->setResult(CMDComSend, ErrorOk);
	return task->endTask(ret);
}

// TODO: 接続時間の長いセッションを切断して空きを作る
bool Listener::cmdPurge(QueueNode *q)
{
	Task *task = q->getTask();

	int hb	= mSentinel->getConf()->heartBeetInterval;
	int wt	= mSentinel->getConf()->maxWaitTime;

	{
		WRITE_LOCK(mLock);
		Session *dead_conn	= mSentinel->getSessionPool().getUse();
		while(dead_conn) {
			Session *n = mSentinel->getSessionPool().getNext(dead_conn);
			time_t last_time = dead_conn->getLastTime();
			time_t cur_time = time(NULL);
			if( last_time + hb < cur_time ) {
				struct sockaddr_in addr;
				dead_conn->getPeerAddress(&addr);
				taskLog(q, LOG_DEBUG, "session purged. interval=%d addr=%s:%d",
						(int)(cur_time-last_time),
						inet_ntoa(addr.sin_addr),
						ntohs(addr.sin_port));
				if( dead_conn->getUserId() ) {
					taskLog(q, LOG_INFO, "user %s logout.", dead_conn->getUserId());
				}
				mSentinel->getSessionPool().freeSession(dead_conn);
			}
			dead_conn = n;
		}
	}

	{
		WRITE_LOCK(mLock);
		size_t cap = mSentinel->getSessionPool().getCapacity();
		size_t use = mSentinel->getSessionPool().getActiveSessions();
		size_t max = cap * 0.9;
		Session *s;
		Session *n;
		if( use > max ) {
			taskLog(q, LOG_INFO, "connection soft purge %d/%d -> %d/%d.", use, cap, max, cap);
			while( use > max ) {
				time_t	old_time 	= 0;
				Session *old_conn	= NULL;
				s = mSentinel->getSessionPool().getUse();
				time_t last_time;
				while(s) {
					time_t cur_time = time(NULL);
					n = mSentinel->getSessionPool().getNext(s);
					time_t chk_time = s->getStartTime();
					last_time = s->getLastTime();
					if( last_time + wt < cur_time ) {
						if( s->getCommandQueueCount() == 0 && s->getRunningTaskCount() == 0 ) {
							if( old_time == 0 || chk_time < old_time ) {
								old_time	= chk_time;
								old_conn	= s;
							}
						}
					}
					s = n;
				}
				if( old_conn == NULL ) {
					break;
				}

				struct sockaddr_in addr;
				old_conn->getPeerAddress(&addr);
				char startTime[256];
				strftime(startTime, sizeof(startTime), "%Y/%m/%d-%H:%M:%S", gmtime(&last_time));
				taskLog(q, LOG_DEBUG, "session purged. start=%s addr=%s:%d",
						startTime,
						inet_ntoa(addr.sin_addr),
						ntohs(addr.sin_port));
				if( old_conn->getUserId() ) {
					taskLog(q, LOG_INFO, "user %s logout.", old_conn->getUserId());
				}
				mSentinel->getSessionPool().freeSession(old_conn);
				max--;
			}
		}
		if( use > max ) {
			taskLog(q, LOG_WARNING, "connection hard purge %d/%d -> %d/%d.", use, cap, max, cap);
			while( use > max ) {
				time_t	old_time 	= 0;
				Session *old_conn	= NULL;
				s = mSentinel->getSessionPool().getUse();
				time_t last_time;
				while(s) {
					time_t cur_time = time(NULL);
					n = mSentinel->getSessionPool().getNext(s);
					time_t chk_time = s->getStartTime();
					last_time = s->getLastTime();
					if( last_time + 60 < cur_time ) {
						if( s->getCommandQueueCount() == 0 && s->getRunningTaskCount() == 0 ) {
							if( old_time == 0 || chk_time < old_time ) {
								old_time	= chk_time;
								old_conn	= s;
							}
						}
					}
					s = n;
				}
				if( old_conn == NULL ) {
					break;
				}

				struct sockaddr_in addr;
				old_conn->getPeerAddress(&addr);
				char startTime[256];
				strftime(startTime, sizeof(startTime), "%Y/%m/%d-%H:%M:%S", gmtime(&last_time));
				taskLog(q, LOG_DEBUG, "session purged. start=%s addr=%s:%d",
						startTime,
						inet_ntoa(addr.sin_addr),
						ntohs(addr.sin_port));
				if( old_conn->getUserId() ) {
					taskLog(q, LOG_INFO, "user %s logout.", old_conn->getUserId());
				}
				mSentinel->getSessionPool().freeSession(old_conn);
				max--;
			}
		}
	}

	QnResult *ret = new QnResult(task);
	ret->setResult(CMDComPurge, ErrorOk);
	task->endTask(ret);
	return true;
}

class TBReciveRaw:  public TaskBuffer {
public:
	TBReciveRaw() {
		mLastRecvTime	= time(NULL);
		mNeedSize		= 0;
		mRecvSize		= 0;
		mRecvBuff		= NULL;
	}
	virtual ~TBReciveRaw() {
		if( mRecvBuff )	delete mRecvBuff;
	}
	time_t		mLastRecvTime;
	size_t		mNeedSize;
	size_t		mRecvSize;
	StdBuffer *	mRecvBuff;
};

bool Listener::cmdRecvRaw(QueueNode *q)
{
	Task *task = q->getTask();
	Session *con = task->getSession();

	TBReciveRaw *tb = (TBReciveRaw*)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBReciveRaw();
		taskLog(q, LOG_DEBUG, "new task buffer task=%d tb=%x", task->getTaskId(), tb);
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		tb->mNeedSize	= ((QueueCommRecvRaw*)q)->mSize;
	} else {
		taskLog(q, LOG_DEBUG, "load task buffer task=%d tb=%x", task->getTaskId(), tb);
	}
	taskLog(q, LOG_DEBUG, "requestSize=%ld reciveSize=%ld", tb->mNeedSize, tb->mRecvSize);

	size_t curSize = tb->mRecvSize;

	if( con ) {
		int fd = con->getDiscriptor();
		if( fd == 0 ) {
			taskLog(q, LOG_INFO, "input socket is closed. q=%x.", q);
			return taskResponse(q, ErrorComm, "input socket is closed.");
		} else {
			while( tb->mRecvSize < tb->mNeedSize ) {
				char buff[4096];
				size_t chunkSize = 4096;
				if( tb->mNeedSize - tb->mRecvSize < chunkSize ) {
					chunkSize = tb->mNeedSize - tb->mRecvSize;
				}
				ssize_t rs = con->read(buff, chunkSize);
				int rd_err = errno;
				if( rs == -1 ) {
					if( rd_err != EAGAIN && rd_err !=  EWOULDBLOCK ) {
						taskLog(q, LOG_ERR, "read from device socket error. task=%d q=%lld code=%d.", task->getTaskId(), q->getId(), rd_err);
						return false;
					}
					time_t now = time(NULL);
					if( curSize == tb->mRecvSize ) {
						if( tb->mLastRecvTime + 30 < now ) {
							return taskResponse(q, ErrorTimeout, "%s. recive data timeout.", codeToMessgae(ErrorTimeout));
						}
					} else {
						tb->mLastRecvTime = now;
					}
					QueueNode *qn = new QueueNode(task, this, q->getFunction());
					taskLog(q, LOG_DEBUG, "continue read. task=%d q=%lld next q=%lld next data=%u.", task->getTaskId(), q->getId(), qn->getId(), tb->mNeedSize-tb->mRecvSize);
					return mSentinel->getScheduler().setQueueTimer(qn,1);
				}
				if( rs > 0 ) {
					if( tb->mRecvBuff == NULL ) {
						tb->mRecvBuff	= new StdBuffer();
						tb->mRecvBuff->addCapacity(tb->mNeedSize);
					}
					tb->mRecvBuff->add(buff, rs);
					tb->mRecvSize	+= rs;
					taskLog(q, LOG_DEBUG, "read from device socket task=%d q=%lld, data=%u/%u.", task->getTaskId(), q->getId(), tb->mRecvSize, tb->mNeedSize);
				}
			}
		}
	}
	taskLog(q, LOG_DEBUG, "read from device end.");

	QnCommRecvResult *qn = new QnCommRecvResult(task);
	StdBuffer *buffPtr = tb->mRecvBuff;
	tb->mRecvBuff	= NULL;
	qn->setResult(buffPtr);
	task->endTask(qn);
	return true;
}

}




