/*
 * process.cpp
 *
 *  Created on: 2012/12/08
 *      Author: yasuoki
 */

#include "process.h"
#include "queue.h"
#include "sentinel.h"
#include <syslog.h>
#include <string.h>
#include <assert.h>

namespace SST {

Process::Process()
{
	mSentinel	= NULL;
	mMaxThread	= 1;
	mMinThread	= 0;
	mReserveThread	= 0;
	mRunThread	= 0;
	mPutQueue	= 0;
	mGetQueue	= 0;
}

Process::~Process()
{

}


bool Process::taskResponse(QueueNode *q, ErrorCode code, const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	int logLevel = -1;
	switch(code) {
	case ErrorOk:			logLevel	= -1;			break;
	case ErrorNoMemory:		logLevel	= LOG_ERR;		break;
	case ErrorRuntime:		logLevel	= LOG_ERR;		break;
	case ErrorConnection:	logLevel	= LOG_ERR;		break;
	case ErrorDatabase:		logLevel	= LOG_ERR;		break;
	case ErrorComm:			logLevel	= LOG_ERR;		break;
	case ErrorMessage:		logLevel	= LOG_ERR;		break;
	case ErrorCommand:		logLevel	= LOG_WARNING;	break;
	case ErrorNotLogin:		logLevel	= LOG_INFO;		break;
	case ErrorPermission:	logLevel	= LOG_INFO;		break;
	case ErrorNotFound:		logLevel	= LOG_INFO;		break;
	case ErrorDataFull:		logLevel	= LOG_ERR;		break;
	case ErrorDataLost:		logLevel	= LOG_ERR;		break;
	case ErrorDataConflict:	logLevel	= LOG_ERR;		break;
	case ErrorParameter:	logLevel	= LOG_WARNING;	break;
	case ErrorHierarchy:	logLevel	= LOG_WARNING;	break;
	case ErrorDuplicate:	logLevel	= LOG_INFO;		break;
	case ErrorAuth:			logLevel	= LOG_INFO;		break;
	case ErrorHTTP:			logLevel	= LOG_WARNING;	break;
	case ErrorSystem:		logLevel	= LOG_ERR;		break;
	case ErrorLocked:		logLevel	= LOG_INFO;		break;
	case ErrorUser:			logLevel	= LOG_INFO;		break;
	case ErrorNoSpace:		logLevel	= LOG_ERR;		break;
	case ErrorDataType:		logLevel	= LOG_WARNING;	break;
	case ErrorData:			logLevel	= LOG_WARNING;	break;
	case ErrorNotImpl:		logLevel	= LOG_ERR;		break;
	case ErrorNotEmpty:		logLevel	= LOG_INFO;		break;
	case ErrorRestriction:	logLevel	= LOG_INFO;		break;
	case ErrorTimeout:		logLevel	= LOG_WARNING;	break;
	}

	bool r = taskResponseLogVa(logLevel, q, code, fmt, ap);
	va_end(ap);
	return r;
}

bool Process::taskResponseLog(int logLevel, QueueNode *q, ErrorCode code, const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	bool r = taskResponseLogVa(logLevel, q, code, fmt, ap);
	va_end(ap);
	return r;
}

bool Process::taskResponseLogVa(int logLevel, QueueNode *q, ErrorCode code, const char *fmt, va_list ap)
{
#ifndef _DEBUG
	if( logLevel <= LOG_ERR )
#endif
	{
		if( fmt ) {
			StdBuffer logmsg;
			va_list ap2;
			va_copy(ap2, ap);
			logmsg.addvafmt(fmt, ap);
			va_end(ap2);
			const char *pos = "";
			const char *acc = "";
			StdBuffer _pos;
			StdBuffer _acc;
			if( q ) {
				Task *task = q->getTask();
				if( task ) {
					const char *sv	= task->getServiceId();
					const char *id	= task->getUserId();
					if( sv || id ) {
						if( !sv ) sv = "unknown";
						if( !id ) id = "unknown";
						_acc.addfmt("[%s:%s] ", id, sv);
						acc = _acc.getPtr();
					}
				}
				Process *p = q->getProcess();
				if( p ) {
					const char *_proc = p->getName();
					if( _proc ) {
						const char *_func = p->getFuncName(q->getFunction());
						_pos.addfmt("%s:%s(%d) ", _proc, _func, q->getStep());
						pos = _pos.getPtr();
					}
				}
			}
			mSentinel->log(logLevel, "%s%s%s", acc, pos, logmsg.getPtr());
		}
	}
	Task *task = q->getTask();
	QnResult *r = new QnResult(task);
	r->setResult(q->getFunction(), code, fmt, ap);
	va_end(ap);
	task->endTask(r);
	return true;
}

void Process::taskLog(QueueNode *q, int logLevel, const char *fmt, ...)
{
	if( fmt ) {
		va_list ap;
		va_start(ap, fmt);

		StdBuffer logmsg;
		logmsg.addvafmt(fmt, ap);

		va_end(ap);

		const char *pos = "";
		const char *acc = "";
		StdBuffer _pos;
		StdBuffer _acc;
		if( q ) {
			Task *task = q->getTask();
			if( task ) {
				const char *sv	= task->getServiceId();
				const char *id	= task->getUserId();
				if( sv || id ) {
					if( !sv ) sv = "unknown";
					if( !id ) id = "unknown";
					_acc.addfmt("[%s:%s] ", id, sv);
					acc = _acc.getPtr();
				}
			}
			Process *p = q->getProcess();
			if( p ) {
				const char *_proc = p->getName();
				if( _proc ) {
					const char *_func = p->getFuncName(q->getFunction());
					_pos.addfmt("%s:%s(%d) ", _proc, _func, q->getStep());
					pos = _pos.getPtr();
				}
			}
		}
		mSentinel->log(logLevel, "%s%s%s", acc, pos, logmsg.getPtr());

	}
}

int Process::getMaxThread() const
{
	return mMaxThread;
}

int Process::getMinThread() const
{
	return mMinThread;
}

int Process::getRunThread() const
{
	READ_LOCK(const_cast<Process*>(this)->mQueue.getLockObject());
	int r = mRunThread;
	return r;
}

bool Process::reserveThread()
{
	WRITE_LOCK(mQueue.getLockObject());
	if( mReserveThread < mMaxThread ) {
		mReserveThread++;
		return true;
	}
	return false;
}

void Process::cancelThread()
{
	WRITE_LOCK(mQueue.getLockObject());
	mReserveThread--;
}

bool Process::queuIsEmpty() const
{
	return mQueue.empty();
}
unsigned long long  Process::getWaitTime() const
{
	return mQueue.getNextId();
}

QueueNode * Process::nextQueue()
{
	return mQueue.next();
}

QueueNode * Process::getQueue()
{
	QueueNode * q = mQueue.get();
 	if( q ) {
 		WRITE_LOCK(mQueue.getLockObject());
 		mGetQueue++;
 	}
	return q;
}

bool Process::putQueue(QueueNode *q)
{
	if( !mQueue.put(q) )
		return false;
 	{
 		WRITE_LOCK(mQueue.getLockObject());
 		mPutQueue++;
 	}
//	mSentinel->log(LOG_DEBUG, "%s::putQueue: task=%x queue=%x type=%d func=%d step=%d", getName(), q->getTask(), q, q->getType(), q->getFunction(), q->getStep());
	mSentinel->getScheduler().schedule();
	return true;
}

void  Process::clear()
{
	mQueue.clear();
}

bool Process::configure(Sentinel *obj, Conf *conf)
{
	mSentinel	= obj;
	return true;
}
bool Process::doProcess(QueueNode *q)
{
	taskLog(q, LOG_DEBUG, "start task=%d qid=%lld",
			q->getTask()->getTaskId(), q->mId);
 	{
 		WRITE_LOCK(mQueue.getLockObject());
 		mRunThread++;
 	}
 	bool r = true;
 	if( q != NULL ) {
 		r = exec(q);
 	}
 	{
 		WRITE_LOCK(mQueue.getLockObject());
 		mRunThread--;
 		mReserveThread--;
 	}
 	taskLog(q, LOG_DEBUG, "end task=%d qid=%lld",
			q->getTask()->getTaskId(), q->mId);

 	return r;
}

void Process::doAbort()
{
	stop(NULL);
}

bool  Process::diag(pjson::builder &jb)
{
	if( !jb.addObjectProp(getName(), strlen(getName())) ||
		!jb.beginObject() ||
		!jb.addObjectProp("maxThread",9) ||
		!jb.valueInt(mMaxThread) ||
		!jb.addObjectProp("minThread",9) ||
		!jb.valueInt(mMinThread) ||
		!jb.addObjectProp("runningThread",13) ||
		!jb.valueInt(mRunThread) ||
		!jb.addObjectProp("waitingQueue",12) ||
		!jb.valueInt(mQueue.size()) ||
		!jb.addObjectProp("processedQueue",14) ||
		!jb.valueInt(mGetQueue) ||
		!jb.addObjectProp("totalQueue",10) ||
		!jb.valueInt(mPutQueue) ||
		!jb.endObject() ) {
		mSentinel->log(LOG_ERR, "TaskManager::diag json error 1. code=%d", jb.getError());
		return false;
	}
	return true;
}

bool Process::exec(QueueNode *q)
{
	assert( mSentinel != NULL );
	mSentinel->log(LOG_ERR, "%s::doProcess is not implement", getName());
	return false;
}

void Process::stop(QueueNode *q)
{
	assert( mSentinel != NULL );
	mSentinel->log(LOG_ERR, "%s::doAbort is not implement", getName());
}


}

