/*
 * cmd_account.cpp
 *
 *  Created on: 2013/01/05
 *      Author: yasuoki
 */
#include "commands.h"
#include "proc_command.h"
#include "proc_database.h"
#include "task.h"
#include "sentinel.h"
#include "buffer.h"
#include "pjson.h"
#include <syslog.h>
#include <errno.h>
#include <netdb.h>

namespace SST {

class TBRegAccount : public TaskBuffer {
public:
	TBRegAccount() {
		mJsCommand		= NULL;
		mJsKeyAID		= NULL;
		mJsKeyUserId	= NULL;
		mJsUserId		= NULL;
		mJsName1		= NULL;
		mJsName2		= NULL;
		mJsLocale		= NULL;
		mJsAuthType		= NULL;
		mJsAuthData		= NULL;
		mJsLevel		= NULL;
		mLevel 			= AccountUser;
	}
	virtual ~TBRegAccount() {
		if( mJsCommand )	delete mJsCommand;
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsKeyUserId;
	pjson::value *	mJsKeyAID;
	pjson::value *	mJsUserId;
	pjson::value *	mJsName1;
	pjson::value *	mJsName2;
	pjson::value *	mJsLocale;
	pjson::value *	mJsAuthType;
	pjson::value *	mJsAuthData;
	pjson::value *	mJsLevel;
	AccountLevel 	mLevel;
	AuthType		mAuthType;
};

bool CommandProc::cmdRegAccount(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBRegAccount *tb = (TBRegAccount *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBRegAccount();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s::cmdRegAccount: no more memory. q=%x", getName(), q);
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s::cmdRegAccount: no more memory. q=%x", getName(), q);
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("regAccount");
			if( !vCmd ) {
				return taskResponse(q, ErrorParameter, "%s. \"regAccount\" is not object.",
						codeToMessgae(ErrorParameter));
			}
			tb->mJsUserId	= tb->mJsCommand->get("userId",  vCmd);
			tb->mJsName1	= tb->mJsCommand->get("name1", vCmd);
			tb->mJsName2	= tb->mJsCommand->get("name2", vCmd);
			tb->mJsLocale	= tb->mJsCommand->get("locale", vCmd);
			tb->mJsAuthType	= tb->mJsCommand->get("authType", vCmd);
			tb->mJsAuthData	= tb->mJsCommand->get("authData", vCmd);
			tb->mJsLevel	= tb->mJsCommand->get("level", vCmd);

			if( !tb->mJsUserId ) {
				return taskResponse(q, ErrorParameter, "%s. \"userId\" is not specified.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsUserId->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
			}
			tb->mAuthType = AuthNone;
			if( tb->mJsAuthType ) {
				if( tb->mJsAuthType->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"authType\" type.", codeToMessgae(ErrorParameter));
				}
				if( strcmp(tb->mJsAuthType->vString, "none") == 0 ) {

				} else if( strcmp(tb->mJsAuthType->vString, "password") == 0 ) {
					if( !tb->mJsAuthData )
						return taskResponse(q, ErrorParameter, "%s. \"authData\" is not specified.", codeToMessgae(ErrorParameter));
					if( tb->mJsAuthData->vt != pjson::vt_string )
						return taskResponse(q, ErrorParameter, "%s. \"authData\" type.", codeToMessgae(ErrorParameter));
					tb->mAuthType = AuthPasswd;
				} else if( strcmp(tb->mJsAuthType->vString, "external") == 0 ) {
					if( !tb->mJsAuthData )
						return taskResponse(q, ErrorParameter, "%s. \"authData\" is nto specified.", codeToMessgae(ErrorParameter));
					if( tb->mJsAuthData->vt != pjson::vt_string )
						return taskResponse(q, ErrorParameter, "%s. \"authData\" type.", codeToMessgae(ErrorParameter));
					tb->mAuthType = AuthExternURL;
				} else {
					return taskResponse(q, ErrorParameter, "%s. \"authType\" value.", codeToMessgae(ErrorParameter));
				}
			}
			if( tb->mJsName1 && tb->mJsName1->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"name1\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsName2 && tb->mJsName2->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"name2\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsLocale && tb->mJsLocale->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"locale\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsLevel && tb->mJsLevel->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"level\" type.", codeToMessgae(ErrorParameter));
			}

			if( tb->mJsLevel ) {
				if( strcmp(tb->mJsLevel->vString, "SystemAdministrator") == 0 ) {
					tb->mLevel	= AccountSystemAdmin;
				} else if( strcmp(tb->mJsLevel->vString, "user") == 0 ) {
					tb->mLevel	= AccountUser;
				} else {
					return taskResponse(q, ErrorParameter, "%s. \"level\" value.", codeToMessgae(ErrorParameter));
				}
			}
			QueryKeyType 	sref;
			Account		 	acc;

			sref.mId	= NULL;
			sref.mName	= NULL;

			memset(&acc, 0, sizeof(Account));
			acc.id			= NULL;
			acc.uid			= (char*)(tb->mJsUserId->vString);
			acc.name1		= (char*)(tb->mJsName1 ? tb->mJsName1->vString : NULL);
			acc.name2		= (char*)(tb->mJsName2 ? tb->mJsName2->vString : NULL);
			acc.locale		= (char*)(tb->mJsLocale ? tb->mJsLocale->vString : NULL);
			acc.authType	= tb->mAuthType;
			acc.authData	= (char*)(tb->mJsAuthData ? tb->mJsAuthData->vString : NULL);
			acc.level		= tb->mLevel;

			QnDBRegister *qn = new QnDBRegister(task);
			qn->registAccount(&sref, &acc);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, res->getMessagePtr());
			}
			ResultId *rs = (ResultId*)(res->mResult);
			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("aid", 3) ||
				!jb.valueString(rs->mId) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "%s. json builder failed 3. account=%s. code=%d",
						codeToMessgae(ErrorRuntime),
						tb->mJsUserId->vString,
						jb.getError());
				return false;
			}
			return response(q, jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

bool CommandProc::cmdSetAccount(QueueNode *q)
{
	Task *task = q->getTask();

	TBRegAccount *tb = (TBRegAccount *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBRegAccount();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("setAccount");
			if( !vCmd ) {
				return taskResponse(q, ErrorParameter, "%s. \"setAccount\" is not object.", codeToMessgae(ErrorParameter));
			}
			tb->mJsKeyAID		= tb->mJsCommand->get("aid",  vCmd);
			tb->mJsKeyUserId	= tb->mJsCommand->get("userId",  vCmd);

			if( !tb->mJsKeyUserId && !tb->mJsKeyAID ) {
				return taskResponse(q, ErrorParameter, "%s. account is not specified.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsKeyUserId && tb->mJsKeyUserId->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsKeyAID && tb->mJsKeyAID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"aid\" type.", codeToMessgae(ErrorParameter));
			}

			pjson::value *vSet = tb->mJsCommand->get("set", vCmd);
			if( !vSet ) {
				return taskResponse(q, ErrorParameter, "%s. \"set\" is not specified.", codeToMessgae(ErrorParameter));
			}

			tb->mJsUserId	= tb->mJsCommand->get("userId", vSet);
			tb->mJsName1	= tb->mJsCommand->get("name1", vSet);
			tb->mJsName2	= tb->mJsCommand->get("name2", vSet);
			tb->mJsLocale	= tb->mJsCommand->get("locale", vSet);
			tb->mJsAuthType	= tb->mJsCommand->get("authType", vSet);
			tb->mJsAuthData	= tb->mJsCommand->get("authData", vSet);

			if( tb->mJsUserId && tb->mJsUserId->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsAuthType ) {
				if( tb->mJsAuthType->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"authType\" type.", codeToMessgae(ErrorParameter));
				}
				if( strcmp(tb->mJsAuthType->vString, "none") == 0 ) {
					tb->mAuthType = AuthNone;
				} else if( strcmp(tb->mJsAuthType->vString, "password") == 0 ) {
					if( !tb->mJsAuthData )
						return taskResponse(q, ErrorParameter, "%s. \"authData\" is not specified.", codeToMessgae(ErrorParameter));
					if( tb->mJsAuthData->vt != pjson::vt_string )
						return taskResponse(q, ErrorParameter, "%s. \"authData\" type.", codeToMessgae(ErrorParameter));
					tb->mAuthType = AuthPasswd;
				} else if( strcmp(tb->mJsAuthType->vString, "external") == 0 ) {
					if( !tb->mJsAuthData )
						return taskResponse(q, ErrorParameter, "%s. \"authData\" is not specified.", codeToMessgae(ErrorParameter));
					if( tb->mJsAuthData->vt != pjson::vt_string )
						return taskResponse(q, ErrorParameter, "%s. \"authData\" type.", codeToMessgae(ErrorParameter));
					tb->mAuthType = AuthExternURL;
				} else {
					return taskResponse(q, ErrorParameter, "%s. \"authType\" value.", codeToMessgae(ErrorParameter));
				}
			}
			if( tb->mJsName1 && tb->mJsName1->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"name1\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsName2 && tb->mJsName2->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"name2\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsLocale && tb->mJsLocale->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"locale\" type.", codeToMessgae(ErrorParameter));
			}

			QueryKeyType aref;
			aref.mId	= tb->mJsKeyAID ? tb->mJsKeyAID->vString : NULL;
			aref.mName	= tb->mJsKeyUserId ? tb->mJsKeyUserId->vString : NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryAccount(&aref);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Account info load faild.", res->getMessagePtr());
			}
			ResultAccount *rs = (ResultAccount*)(res->mResult);

			QueryKeyType 	sref;
			Account		 	acc;

			sref.mId	= NULL;
			sref.mName	= NULL;

			acc		= rs->mAccount;

			if( tb->mJsUserId )
				acc.uid		= (char*)tb->mJsUserId->vString;
			if( tb->mJsName1 )
				acc.name1	= (char*)tb->mJsName1->vString;
			if( tb->mJsName2 )
				acc.name2	= (char*)tb->mJsName2->vString;
			if( tb->mJsLocale )
				acc.locale	= (char*)tb->mJsLocale->vString;
			if( tb->mJsAuthType )
				acc.authType	= tb->mAuthType;
			if( tb->mJsAuthData )
				acc.authData	= (char*)tb->mJsAuthData->vString;

			QnDBRegister *qn = new QnDBRegister(task);
			qn->registAccount(&sref, &acc);
			return task->call(qn, step+1);
		}
		break;
	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Account info registration failed. %s", res->getMessagePtr());
			}
			ResultId *rs = (ResultId*)(res->mResult);
			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("aid", 3) ||
				!jb.valueString(rs->mId) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
				return false;
			}
			return response(q, jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}


class TBRmAccount : public TaskBuffer {
public:
	TBRmAccount() {
		mJsCommand		= NULL;
		mJsKeyAID		= NULL;
		mJsKeyUserId	= NULL;
		mAccount		= NULL;
		mGroupList		= NULL;
		mGroupIndex		= 0;
		mDeviceList		= NULL;
		mDeviceIndex	= 0;
	}
	virtual ~TBRmAccount() {
		if( mJsCommand )	delete mJsCommand;
		if( mAccount )		free(mAccount);
		if( mGroupList )	free(mGroupList);
		if( mDeviceList )	free(mDeviceList);
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsKeyUserId;
	pjson::value *	mJsKeyAID;
	ResultAccount	*mAccount;
	ResultList		*mGroupList;
	size_t			mGroupIndex;
	ResultList		*mDeviceList;
	size_t			mDeviceIndex;
};

bool CommandProc::cmdRmAccount(QueueNode *q)
{
	Task *task = q->getTask();

	TBRmAccount *tb = (TBRmAccount *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBRmAccount();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("rmAccount");
			if( vCmd ) {
				tb->mJsKeyAID		= tb->mJsCommand->get("aid",  vCmd);
				tb->mJsKeyUserId	= tb->mJsCommand->get("userId",  vCmd);
				if( tb->mJsKeyUserId && tb->mJsKeyUserId->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
				}
				if( tb->mJsKeyAID && tb->mJsKeyAID->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"aid\" type", codeToMessgae(ErrorParameter));
				}
			}

			QueryKeyType aref;
			aref.mId	= tb->mJsKeyAID ? tb->mJsKeyAID->vString : NULL;
			aref.mName	= tb->mJsKeyUserId ? tb->mJsKeyUserId->vString : NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryAccount(&aref);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Account info load failed. %s", res->getMessagePtr());
			}
			tb->mAccount = (ResultAccount*)(res->mResult);
			res->mResult = NULL;

			tb->mAccount->mAccount.status = AccountStop;
			QueryKeyType 	sref;
			sref.mId	= NULL;
			sref.mName	= NULL;
			QnDBRegister *qn = new QnDBRegister(task);
			qn->registAccount(&sref, &tb->mAccount->mAccount);
			return task->call(qn, step+1);
		}
	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Account info registration failed. %s", res->getMessagePtr());
			}

			QueryKeyType 	aref;
			QueryRangeType	range;
			aref.mId		= tb->mAccount->mAccount.id;
			aref.mName		= NULL;
			range.mStart	= 0;
			range.mCount	= -1;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryGroupList(&aref, &range);
			return task->call(qn, step+1);
		}
		break;
	case 3:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				if( code != ErrorNotFound ) {
					return taskResponse(q, code, "Group list load failed. %s", res->getMessagePtr());
				}
			}

			if( code == ErrorOk ) {
				tb->mGroupList	= (ResultList*)(res->mResult);
				res->mResult	= NULL;
			}
		}
	case 4:
		{
			if( step == 4 ) {
				QnResult *res	= (QnResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != 0 ) {
					return taskResponse(q, code, "leaveGroup failed. %s", res->getMessagePtr());
				}
			}
			step	= 4;

			if( tb->mGroupList != NULL && tb->mGroupIndex < tb->mGroupList->mList.values ) {
				pjson::builder jb;
				if( !jb.init(256) ||
					!jb.beginObject() ||
					!jb.addObjectProp("leaveGroup", 10) ||
					!jb.beginObject() ||
					!jb.addObjectProp("aid", 3) ||
					!jb.valueString(tb->mAccount->mAccount.id) ||
					!jb.addObjectProp("gid", 3) ||
					!jb.valueString(tb->mGroupList->mList.value[tb->mGroupIndex++]) ||
					!jb.endObject() ||
					!jb.endObject() ) {
					taskLog(q, LOG_ERR, "json builder failed(1). code=%d", jb.getError());
					return false;
				}
				pjson::json *js = jb.detouch();
				if( !js ) {
					taskLog(q, LOG_ERR, "json builder failed(2).");
					return false;
				}
				QueueNode *cmd = new QueueNode(task, this, CMDLeaveGroup);
				cmd->setJSONMessage(js);
				return task->call(cmd, step);
			}

			QueryKeyType 	aref;
			QueryRangeType	range;
			aref.mId		= tb->mAccount->mAccount.id;
			aref.mName		= NULL;
			range.mStart	= 0;
			range.mCount	= -1;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryDeviceList(&aref, &range);
			return task->call(qn, step+1);
		}
		break;
	case 5:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				if( code != ErrorNotFound ) {
					return taskResponse(q, code, "Device list load failed. %s", res->getMessagePtr());
				}
			}

			if( code == ErrorOk ) {
				tb->mDeviceList = (ResultList*)(res->mResult);
				res->mResult	= NULL;
			}
		}
	case 6:
		{
			if( step == 6 ) {
				QnResult *res	= (QnResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != 0 ) {
					return taskResponse(q, code, "Device info remove failed. %s", res->getMessagePtr());
				}
			}
			step	= 6;

			if( tb->mDeviceList != NULL && tb->mDeviceIndex < tb->mDeviceList->mList.values ) {
				QueryKeyType key;
				key.mId		= tb->mDeviceList->mList.value[tb->mDeviceIndex++];
				key.mName	= NULL;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->removeDevice(&key);
				return task->call(qn, step);
			}

			QueryKeyType key;
			key.mId		= tb->mAccount->mAccount.id;
			key.mName	= NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->removeAccount(&key);
			return task->call(qn, step+1);
		}
		break;
	case 7:
		{
			QnResult *res	= (QnResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != 0 ) {
				return taskResponse(q, code, "Account remove failed. %s", res->getMessagePtr());
			}
			taskLog(q, LOG_INFO, "Account %s,%s removed.", tb->mAccount->mAccount.id, tb->mAccount->mAccount.uid);
			SessionPool &pool = mSentinel->getSessionPool();
			Session *userSession = pool.findAcountSession(tb->mAccount->mAccount.id);
			while( userSession != NULL ) {
				userSession->close();
				userSession = pool.findAcountSession(tb->mAccount->mAccount.id);
			}
			return taskResponse(q, ErrorOk, NULL);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBLsAccount : public TaskBuffer {
public:
	TBLsAccount() {
		mJsCommand	= NULL;
		mJsFrom		= NULL;
		mJsCount	= NULL;
	}
	virtual ~TBLsAccount() {
		if( mJsCommand )	delete mJsCommand;
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsFrom;
	pjson::value *	mJsCount;
};

bool CommandProc::cmdLsAccount(QueueNode *q)
{
	Task *task = q->getTask();

	TBLsAccount *tb = (TBLsAccount *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBLsAccount();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("lsAccount");
			if( vCmd ) {
				tb->mJsFrom		= tb->mJsCommand->get("from",  vCmd);
				tb->mJsCount	= tb->mJsCommand->get("count", vCmd);

				if( tb->mJsFrom && tb->mJsFrom->vt != pjson::vt_int ) {
					return taskResponse(q, ErrorParameter, "%s. \"from\" type.", codeToMessgae(ErrorParameter));
				}
				if( tb->mJsCount && tb->mJsCount->vt != pjson::vt_int ) {
					return taskResponse(q, ErrorParameter, "%s. \"count\" type.", codeToMessgae(ErrorParameter));
				}
			}

			QueryKeyType 	sref;
			QueryKeyType 	gref;
			QueryRangeType	key;
			sref.mId	= NULL;
			sref.mName	= NULL;
			gref.mId	= NULL;
			gref.mName	= NULL;
			key.mStart	= tb->mJsFrom ? tb->mJsFrom->vInt : 0;
			key.mCount	= tb->mJsCount ? tb->mJsCount->vInt : -1;

			if( task->getAccountLevel() != AccountSystemAdmin ) {
				return taskResponse(q, ErrorPermission, "%s.", codeToMessgae(ErrorPermission));
			}

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryAccountList(&sref, &gref, &key);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Account list load failed. %s", res->getMessagePtr());
			}
			ResultList *rs = (ResultList*)(res->mResult);
			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("from", 4) ||
				!jb.valueInt(tb->mJsFrom?tb->mJsFrom->vInt:0) ||
				!jb.addObjectProp("accounts", 8) ||
				!jb.beginArray() ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
				return false;
			}

			for( size_t i = 0; i < rs->mList.count; i++ ) {
				if( !jb.addArrayContent() ||
					!jb.valueString(rs->mList.value[i]) ) {
					taskLog(q, LOG_ERR, "json builder failed(2). code=%d.", jb.getError());
					return false;
				}
			}

			if( !jb.endArray() ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(3). code=%d.", jb.getError());
				return false;
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBShowAccount : public TaskBuffer {
public:
	TBShowAccount() {
		mARef.mId		= NULL;
		mARef.mName		= NULL;
		mAccount		= NULL;
		mGroupCount		= NULL;
		mDeviceCount	= NULL;
		mRestricted		= false;
	}
	virtual ~TBShowAccount() {
		if( mAccount )		free(mAccount);
		if( mGroupCount )	free(mGroupCount);
		if( mDeviceCount )	free(mDeviceCount);
	}
	ResultAccount	*mAccount;
	QueryKeyType	mARef;
	ResultCount		*mGroupCount;
	ResultCount		*mDeviceCount;
	bool			mRestricted;
};

bool  CommandProc::cmdShowAccount(QueueNode *q)
{
	Task *task = q->getTask();

	TBShowAccount *tb = (TBShowAccount *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBShowAccount();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			pjson::json * cmd = q->detouchJSONMessage();

			pjson::value *vCmd = cmd->get("showAccount");
			pjson::value *vUID = NULL;
			pjson::value *vAID = NULL;
			if( vCmd ) {
				vAID	= cmd->get("aid",  vCmd);
				vUID	= cmd->get("userId",  vCmd);
				if( vAID && vAID->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"aid\" type.", codeToMessgae(ErrorParameter));
				}
				if( vUID && vUID->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
				}
				tb->mARef.mId	= vAID ? vAID->vString : NULL;
				tb->mARef.mName	= vUID ? vUID->vString : NULL;
			}

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryAccount(&tb->mARef);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Account info load failed. %s", res->getMessagePtr());
			}
			tb->mAccount 	= (ResultAccount*)(res->mResult);
			res->mResult	= NULL;
			tb->mARef.mId	= tb->mAccount->mAccount.id;


			ErrorCode e = task->allowShowAccount(&tb->mAccount->mAccount);
			if( e != ErrorOk ) {
				if( e != ErrorRestriction ) {
					return taskResponse(q, e, "%s. account=%s,%s.",
							codeToMessgae(ErrorPermission),
							tb->mARef.mId, tb->mARef.mName);
				}
				tb->mRestricted	= true;
			}

			if( !tb->mRestricted ) {
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryGroupCount(&tb->mARef);
				return task->call(qn, step+1);
			}
		}

	case 2:
		{
			if( step == 2 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_WARNING, "Group count failed. %s", res->getMessagePtr());
				} else {
					tb->mGroupCount	= (ResultCount*)(res->mResult);
					res->mResult	= NULL;
				}
			}
			step = 2;

			if( !tb->mRestricted ) {
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryDeviceCount(&tb->mARef);
				return task->call(qn, step+1);
			}
		}

	case 3:
		{
			if( step == 3 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_WARNING, "Device count failed. %s", res->getMessagePtr());
				} else {
					tb->mDeviceCount= (ResultCount*)(res->mResult);
					res->mResult	= NULL;
				}
			}
			step = 3;

			char _createTime[256];
			const char *createTime = NULL;
			if( tb->mAccount->mAccount.createTime != 0 ) {
				struct tm *tmp = gmtime(&tb->mAccount->mAccount.createTime);
				strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", tmp);
				createTime = _createTime;
			}
			char _updateTime[256];
			const char *updateTime = NULL;
			if( tb->mAccount->mAccount.updateTime != 0 ) {
				struct tm *tmp = gmtime(&tb->mAccount->mAccount.updateTime);
				strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", tmp);
				updateTime = _updateTime;
			}
			char _loginTime[256];
			const char *loginTime = NULL;
			if( tb->mAccount->mAccount.loginTime != 0 ) {
				struct tm *tmp = gmtime(&tb->mAccount->mAccount.loginTime);
				strftime(_loginTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", tmp);
				loginTime = _loginTime;
			}

			const char *authType = NULL;
			const char *authData = NULL;
			switch( tb->mAccount->mAccount.authType ) {
			case AuthNone:
				authType	= "none";
				break;
			case AuthPasswd:
				authType	= "password";
				break;
			case AuthExternURL:
				authType	= "external";
				authData	= tb->mAccount->mAccount.authData;
				break;
			default:
				authType	= "unknown";
			}

			const char *accountLevel = NULL;
			AccountLevel level = tb->mAccount->mAccount.level;
			if( task->getAccountId() && strcmp(task->getAccountId(), tb->mAccount->mAccount.id) == 0 ) {
				level	= task->getAccountLevel();
			}
			switch(level) {
			case AccountUser:
				accountLevel	= "user";
				break;
			case AccountServiceAdmin:
				accountLevel	= "ServiceAndminstrator";
				break;
			case AccountSystemAdmin:
				accountLevel	= "SystemAndminstrator";
				break;
			case AccountSystemService:
				accountLevel	= "SystemService";
				break;
			default:
				accountLevel	= "unknown";
				break;
			}

			const char *accountStatus = NULL;
			switch(tb->mAccount->mAccount.status) {
			case AccountNormal:
				accountStatus	= "normal";
				break;
			case AccountStop:
				accountStatus	= "stop";
				break;
			default:
				accountStatus	= "unknown";
				break;
			}
			pjson::builder jb;
			if( tb->mRestricted ) {
				if( !jb.init(512) ||
					!jb.beginObject() ||
					!jb.addObjectProp("result", 6) ||
					!jb.valueInt(0) ||
					!jb.addObjectProp("sequence", 8) ||
					!jb.valueInt(task->getSequence()) ||
					!jb.addObjectProp("userId", 6) ||
					!jb.valueString(tb->mAccount->mAccount.uid) ||
					!jb.addObjectProp("name1", 5) ||
					!jb.valueString(tb->mAccount->mAccount.name1) ||
					!jb.addObjectProp("name2", 5) ||
					!jb.valueString(tb->mAccount->mAccount.name2) ||
					!jb.addObjectProp("locale", 6) ||
					!jb.valueString(tb->mAccount->mAccount.locale) ||
					!jb.endObject() ) {
					taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
					return false;
				}
			} else {
				if( !jb.init(512) ||
					!jb.beginObject() ||
					!jb.addObjectProp("result", 6) ||
					!jb.valueInt(0) ||
					!jb.addObjectProp("sequence", 8) ||
					!jb.valueInt(task->getSequence()) ||
					!jb.addObjectProp("aid", 3) ||
					!jb.valueString(tb->mAccount->mAccount.id) ||
					!jb.addObjectProp("userId", 6) ||
					!jb.valueString(tb->mAccount->mAccount.uid) ||
					!jb.addObjectProp("name1", 5) ||
					!jb.valueString(tb->mAccount->mAccount.name1) ||
					!jb.addObjectProp("name2", 5) ||
					!jb.valueString(tb->mAccount->mAccount.name2) ||
					!jb.addObjectProp("locale", 6) ||
					!jb.valueString(tb->mAccount->mAccount.locale) ||
					!jb.addObjectProp("authType", 8) ||
					!jb.valueString(authType) ||
					!jb.addObjectProp("authData", 8) ||
					!jb.valueString(authData) ||
					!jb.addObjectProp("status", 6) ||
					!jb.valueString(accountStatus) ||
					!jb.addObjectProp("level", 5) ||
					!jb.valueString(accountLevel) ||
					!jb.addObjectProp("groups", 6) ||
					!jb.valueInt(tb->mGroupCount ? tb->mGroupCount->mCount : 0) ||
					!jb.addObjectProp("devices", 7) ||
					!jb.valueInt(tb->mDeviceCount ? tb->mDeviceCount->mCount : 0) ||
					!jb.addObjectProp("createTime", 10) ||
					!jb.valueString(createTime) ||
					!jb.addObjectProp("updateTime", 10) ||
					!jb.valueString(updateTime) ||
					!jb.addObjectProp("loginTime", 9) ||
					!jb.valueString(loginTime) ||
					!jb.endObject() ) {
					taskLog(q, LOG_ERR, "json builder failed(2). code=%d.", jb.getError());
					return false;
				}
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBLockAccount : public TaskBuffer {
public:
	TBLockAccount() {
		mJsCommand	= NULL;
		mSRef.mId	= NULL;
		mSRef.mName	= NULL;
		mARef.mId	= NULL;
		mARef.mName	= NULL;
		mDeviceIndex	= 0;
		mDeviceList	= NULL;
	}
	virtual ~TBLockAccount() {
		if( mJsCommand )	delete mJsCommand;
		if( mDeviceList )	free(mDeviceList);
	}
	pjson::json *	mJsCommand;
	QueryKeyType	mSRef;
	QueryKeyType	mARef;
	size_t			mDeviceIndex;
	ResultList *	mDeviceList;
	size_t			mShareIndex;
	ResultList *	mShareList;
};

bool CommandProc::cmdLockAccount( QueueNode *q)
{
	Task *task = q->getTask();

	TBLockAccount *tb = (TBLockAccount *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBLockAccount();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("lockAccount");
			if( !vCmd ) {
				return taskResponse(q, ErrorParameter, "%s. \"lockAccount\" is not object.",
						codeToMessgae(ErrorParameter));
			}
			pjson::value * vSID		= tb->mJsCommand->get("sid", vCmd);
			pjson::value * vSName	= tb->mJsCommand->get("serviceName", vCmd);
			pjson::value * vAID		= tb->mJsCommand->get("aid", vCmd);
			pjson::value * vUID		= tb->mJsCommand->get("userId", vCmd);

			if( vAID && vAID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
			}
			if( vUID && vUID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"aid\" type.", codeToMessgae(ErrorParameter));
			}
			if( vSID && vSID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
			}
			if( vSName && vSName->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
			}

			tb->mSRef.mId	= vSID   ? vSID->vString : NULL;
			tb->mSRef.mName	= vSName ? vSName->vString : NULL;
			tb->mARef.mId	= vAID   ? vAID->vString : NULL;
			tb->mARef.mName	= vUID   ? vUID->vString : NULL;

			QnDBRegister *qn = new QnDBRegister(task);
			qn->registAccountLock(&tb->mSRef, &tb->mARef);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				if( code != ErrorDuplicate ) {
					return taskResponse(q, code, "Register lock failed. %s", res->getMessagePtr());
				}
			}

			// syncronize account
			QueryKeyType gref;
			gref.mId	= "all";
			gref.mName	= NULL;
			QnDBRegister *qn = new QnDBRegister(task);
			qn->registSyncQueue(SyncCMDRemove, &tb->mSRef, &gref, &tb->mARef);
			return task->call(qn, step+1);
		}
		break;
	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != 0 ) {
				return taskResponse(q, code, "Sync queue add error. %s",
							res->getMessagePtr());
			}
			pjson::builder jb;

			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("syncQueue",9) ||
				!jb.beginObject() ) {
				return taskResponse(q, ErrorRuntime, "Json builder failed(1). code=%d service=%s,%s account=%s,%s.",
						jb.getError(),
						tb->mSRef.mId, tb->mSRef.mName,
						tb->mARef.mId, tb->mARef.mName);
			}
			if( tb->mSRef.mId ) {
				if( !jb.addObjectProp("sid", 3) ||
					!jb.valueString(tb->mSRef.mId) ) {
					return taskResponse(q, ErrorRuntime, "Json builder failed(2). code=%d service=%s,%s account=%s,%s.",
							jb.getError(),
							tb->mSRef.mId, tb->mSRef.mName,
							tb->mARef.mId, tb->mARef.mName);
				}
			}
			if( tb->mSRef.mName ) {
				if( !jb.addObjectProp("serviceName", 11) ||
					!jb.valueString(tb->mSRef.mName) ) {
					return taskResponse(q, ErrorRuntime, "Json builder failed(3). code=%d service=%s,%s account=%s,%s.",
							jb.getError(),
							tb->mSRef.mId, tb->mSRef.mName,
							tb->mARef.mId, tb->mARef.mName);
				}
			}
			if( !jb.endObject() ||
				!jb.endObject() ) {
				return taskResponse(q, ErrorRuntime, "Json builder failed(4). code=%d service=%s,%s account=%s,%s.",
						jb.getError(),
						tb->mSRef.mId, tb->mSRef.mName,
						tb->mARef.mId, tb->mARef.mName);
			}
			Task *st = mSentinel->getTaskManager().createTaskFromBaseTask(task);
			QueueNode *qn = new QueueNode(st, this, CMDSyncQueue);
			qn->setJSONMessage(jb.detouch());
			st->startTask(qn);

			return taskResponse(q, ErrorOk, NULL);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

bool CommandProc::cmdUnlockAccount( QueueNode *q)
{
	Task *task = q->getTask();

	TBLockAccount *tb = (TBLockAccount *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBLockAccount();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("unlockAccount");
			if( !vCmd ) {
				return taskResponse(q, ErrorParameter, "%s. \"lockAccount\" is not object.",
						codeToMessgae(ErrorParameter));
			}
			pjson::value * vSID		= tb->mJsCommand->get("sid", vCmd);
			pjson::value * vSName	= tb->mJsCommand->get("serviceName", vCmd);
			pjson::value * vAID		= tb->mJsCommand->get("aid", vCmd);
			pjson::value * vUID		= tb->mJsCommand->get("userId", vCmd);

			if( vAID && vAID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
			}
			if( vUID && vUID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"aid\" type.", codeToMessgae(ErrorParameter));
			}
			if( vSID && vSID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
			}
			if( vSName && vSName->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
			}

			tb->mSRef.mId	= vSID   ? vSID->vString : NULL;
			tb->mSRef.mName	= vSName ? vSName->vString : NULL;
			tb->mARef.mId	= vAID   ? vAID->vString : NULL;
			tb->mARef.mName	= vUID   ? vUID->vString : NULL;

			QnDBRegister *qn = new QnDBRegister(task);
			qn->registAccountUnlock(&tb->mSRef, &tb->mARef);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				if( code != ErrorDuplicate ) {
					return taskResponse(q, code, "Register lock failed. %s", res->getMessagePtr());
				}
			}

			// syncronize account
			QueryKeyType gref;
			gref.mId	= "all";
			gref.mName	= NULL;
			QnDBRegister *qn = new QnDBRegister(task);
			qn->registSyncQueue(SyncCMDUpdate, &tb->mSRef, &gref, &tb->mARef);
			return task->call(qn, step+1);
		}
		break;
	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != 0 ) {
				return taskResponse(q, code, "Sync queue add error. %s",
							res->getMessagePtr());
			}

			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("syncQueue",9) ||
				!jb.beginObject() ) {
				return taskResponse(q, ErrorRuntime, "Json builder failed(1). code=%d service=%s,%s account=%s,%s.",
						jb.getError(),
						tb->mSRef.mId, tb->mSRef.mName,
						tb->mARef.mId, tb->mARef.mName);
			}
			if( tb->mSRef.mId ) {
				if( !jb.addObjectProp("sid", 3) ||
					!jb.valueString(tb->mSRef.mId) ) {
					return taskResponse(q, ErrorRuntime, "Json builder failed(2). code=%d service=%s,%s account=%s,%s.",
							jb.getError(),
							tb->mSRef.mId, tb->mSRef.mName,
							tb->mARef.mId, tb->mARef.mName);
				}
			}
			if( tb->mSRef.mName ) {
				if( !jb.addObjectProp("serviceName", 11) ||
					!jb.valueString(tb->mSRef.mName) ) {
					return taskResponse(q, ErrorRuntime, "Json builder failed(3). code=%d service=%s,%s account=%s,%s.",
							jb.getError(),
							tb->mSRef.mId, tb->mSRef.mName,
							tb->mARef.mId, tb->mARef.mName);
				}
			}
			if( !jb.endObject() ||
				!jb.endObject() ) {
				return taskResponse(q, ErrorRuntime, "Json builder failed(4). code=%d service=%s,%s account=%s,%s.",
						jb.getError(),
						tb->mSRef.mId, tb->mSRef.mName,
						tb->mARef.mId, tb->mARef.mName);
			}

			Task *st = mSentinel->getTaskManager().createTaskFromBaseTask(task);
			QueueNode *qn = new QueueNode(st, this, CMDSyncQueue);
			qn->setJSONMessage(jb.detouch());
			st->startTask(qn);

			return taskResponse(q, ErrorOk, NULL);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

}







