/*
 * cmd_account.cpp
 *
 *  Created on: 2013/01/05
 *      Author: yasuoki
 */
#include "commands.h"
#include "proc_command.h"
#include "proc_database.h"
#include "task.h"
#include "sentinel.h"
#include "buffer.h"
#include "pjson.h"
#include <syslog.h>
#include <errno.h>
#include <netdb.h>

namespace SST {

class TBRmDevice : public TaskBuffer {
public:
	TBRmDevice() {
		mJsCommand		= NULL;
		mJsDID			= NULL;
		mJsUDID			= NULL;
		mDevice			= NULL;
	}
	virtual ~TBRmDevice() {
		if( mJsCommand )	delete mJsCommand;
		if( mDevice )		free(mDevice);
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsDID;
	pjson::value *	mJsUDID;
	ResultDevice *	mDevice;
};

bool CommandProc::cmdRmDevice(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBRmDevice *tb = (TBRmDevice *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBRmDevice();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();
			pjson::value *vCmd = tb->mJsCommand->get("rmDevice");
			if( !vCmd || vCmd->vt != pjson::vt_object ) {
				return taskResponse(q, ErrorParameter, "%s. \"rmDevice\" is not object.", codeToMessgae(ErrorParameter));
			}
			tb->mJsDID	= tb->mJsCommand->get("did", vCmd);
			tb->mJsUDID	= tb->mJsCommand->get("udid", vCmd);
			if( tb->mJsDID && tb->mJsDID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"did\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsUDID && tb->mJsUDID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"udid\" type.", codeToMessgae(ErrorParameter));
			}
			QueryKeyType dref;
			dref.mId	= tb->mJsDID ? tb->mJsDID->vString : NULL;
			dref.mName	= tb->mJsUDID ? tb->mJsUDID->vString : NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryDevice(&dref);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Device info load error. %s", res->getMessagePtr());
			}
			tb->mDevice	= (ResultDevice*)res->mResult;
			res->mResult	= NULL;

			SessionPool &pool = mSentinel->getSessionPool();
			Session *session = pool.findDeviceSession(tb->mDevice->mDevice.id);
			if( session != NULL ) {
				return taskResponse(q, ErrorPermission, "%s. Device is connecting system.", codeToMessgae(ErrorPermission));
			}

			QueryKeyType dref;
			dref.mId	= tb->mDevice->mDevice.id;
			dref.mName	= NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->removeDevice(&dref);
			return task->call(qn, step+1);
		}
		break;
	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "%s. rmDevice failed.", res->getMessagePtr());
			}
			return taskResponse(q, ErrorOk, "device %s removed.", tb->mDevice->mDevice.id);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

class TBLsDevice : public TaskBuffer {
public:
	TBLsDevice() {
		mJsCommand	= NULL;
		mJsAID		= NULL;
		mJsUID		= NULL;
		mJsFrom		= NULL;
		mJsCount	= NULL;
	}
	virtual ~TBLsDevice() {
		if( mJsCommand )	delete mJsCommand;
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsAID;
	pjson::value *	mJsUID;
	pjson::value *	mJsFrom;
	pjson::value *	mJsCount;
};

bool CommandProc::cmdLsDevice(QueueNode *q)
{
	Task *task = q->getTask();

	TBLsDevice *tb = (TBLsDevice *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBLsDevice();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("lsDevice");
			if( vCmd ) {
				tb->mJsAID		= tb->mJsCommand->get("aid",  vCmd);
				tb->mJsUID		= tb->mJsCommand->get("userId", vCmd);
				tb->mJsFrom		= tb->mJsCommand->get("from",  vCmd);
				tb->mJsCount	= tb->mJsCommand->get("count", vCmd);
				if( tb->mJsAID && tb->mJsAID->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"aid\" type.", codeToMessgae(ErrorParameter));
				}
				if( tb->mJsUID && tb->mJsUID->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
				}
				if( tb->mJsFrom && tb->mJsFrom->vt != pjson::vt_int ) {
					return taskResponse(q, ErrorParameter, "%s. \"from\" type.", codeToMessgae(ErrorParameter));
				}
				if( tb->mJsCount && tb->mJsCount->vt != pjson::vt_int ) {
					return taskResponse(q, ErrorParameter, "%s. \"count\" type.", codeToMessgae(ErrorParameter));
				}
			}

			QueryKeyType 	aref;
			QueryRangeType	range;
			aref.mId	= tb->mJsAID ? tb->mJsAID->vString : NULL;
			aref.mName	= tb->mJsUID ? tb->mJsUID->vString : NULL;
			range.mStart	= tb->mJsFrom ? tb->mJsFrom->vInt : 0;
			range.mCount	= tb->mJsCount ? tb->mJsCount->vInt : -1;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryDeviceList(&aref, &range);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Device list load failed. %s", res->getMessagePtr());
			}
			ResultList *rs = (ResultList*)(res->mResult);
			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("from", 4) ||
				!jb.valueInt(tb->mJsFrom?tb->mJsFrom->vInt:0) ||
				!jb.addObjectProp("devices", 7) ||
				!jb.beginArray() ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
				return false;
			}

			for( size_t i = 0; i < rs->mList.count; i++ ) {
				if( !jb.addArrayContent() ||
					!jb.valueString(rs->mList.value[i]) ) {
					taskLog(q, LOG_ERR, "json builder failed(2). code=%d.", jb.getError());
					return false;
				}
			}

			if( !jb.endArray() ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(3). code=%d.", jb.getError());
				return false;
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBShowDevice : public TaskBuffer {
public:
	TBShowDevice() {
		mJsCommand		= NULL;
		mJsDID			= NULL;
		mJsUDID			= NULL;
	}
	virtual ~TBShowDevice() {
		if( mJsCommand )	delete mJsCommand;
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsDID;
	pjson::value *	mJsUDID;
};

bool CommandProc::cmdShowDevice(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBShowDevice *tb = (TBShowDevice *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBShowDevice();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();
			pjson::value *vCmd = tb->mJsCommand->get("showDevice");
			if( !vCmd || vCmd->vt != pjson::vt_object ) {
				return taskResponse(q, ErrorParameter, "%s. \"showDevice\" is not object.", codeToMessgae(ErrorParameter));
			}
			tb->mJsDID	= tb->mJsCommand->get("did", vCmd);
			tb->mJsUDID	= tb->mJsCommand->get("udid", vCmd);
			if( tb->mJsDID && tb->mJsDID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"did\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsUDID && tb->mJsUDID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"udid\" type.", codeToMessgae(ErrorParameter));
			}
			QueryKeyType dref;
			dref.mId	= tb->mJsDID ? tb->mJsDID->vString : NULL;
			dref.mName	= tb->mJsUDID ? tb->mJsUDID->vString : NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryDevice(&dref);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Device info load error. %s", res->getMessagePtr());
			}
			DeviceInfo *device	= &((ResultDevice*)res->mResult)->mDevice;

			const char *cryptAlgo = "unknown";
			switch( device->storageCrypt ) {
			case CryptAlgorithmNone:	cryptAlgo = "none";		break;
			case CryptAlgorithm3DES:	cryptAlgo = "3des";		break;
			case CryptAlgorithmAES192:	cryptAlgo = "aes192";	break;
			}

			SessionPool &pool = mSentinel->getSessionPool();
			Session *session = pool.findDeviceSession(device->id);
			const char *devStatus = "offline";
			char _startTime[256];
			const char *startTime = NULL;
			char _lastTime[256];
			const char *lastTime = NULL;
			if( session ) {
				switch(session->getStatus()) {
				case SessionStatusAccept:	devStatus	= "accept";		break;
				case SessionStatusConnect:	devStatus	= "connect";	break;
				case SessionStatusLogin:	devStatus	= "login";		break;
				case SessionStatusWait:		devStatus	= "wait";		break;
				}
				time_t tm = session->getStartTime();
				if( tm != 0 ) {
					struct tm *tmp = gmtime(&tm);
					strftime(_startTime, sizeof(_startTime), "%Y/%m/%d-%H:%M:%S", tmp);
					startTime = _startTime;
				}
				tm = session->getLastTime();
				if( tm != 0 ) {
					struct tm *tmp = gmtime(&tm);
					strftime(_lastTime, sizeof(_lastTime), "%Y/%m/%d-%H:%M:%S", tmp);
					lastTime = _lastTime;
				}
			}

			char _createTime[256];
			const char *createTime = NULL;
			if( device->createTime != 0 ) {
				struct tm *tmp = gmtime(&device->createTime);
				strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", tmp);
				createTime = _createTime;
			}
			char _updateTime[256];
			const char *updateTime = NULL;
			if( device->updateTime != 0 ) {
				struct tm *tmp = gmtime(&device->updateTime);
				strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", tmp);
				updateTime = _updateTime;
			}
			char _loginTime[256];
			const char *loginTime = NULL;
			if( device->loginTime != 0 ) {
				struct tm *tmp = gmtime(&device->loginTime);
				strftime(_loginTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", tmp);
				loginTime = _loginTime;
			}

			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("did", 3) ||
				!jb.valueString(device->id) ||
				!jb.addObjectProp("udid", 4) ||
				!jb.valueString(device->udid) ||
				!jb.addObjectProp("os", 2) ||
				!jb.valueString(device->OS?device->OS:"") ||
				!jb.addObjectProp("osVersion", 9) ||
				!jb.valueString(device->OSVersion?device->OSVersion:"") ||
				!jb.addObjectProp("carrier", 7) ||
				!jb.valueString(device->carrier?device->carrier:"") ||
				!jb.addObjectProp("model", 5) ||
				!jb.valueString(device->model?device->model:"") ||
				!jb.addObjectProp("signature", 9) ||
				!jb.valueString(device->signature?device->signature:"") ||
				!jb.addObjectProp("userAccount", 11) ||
				!jb.valueString(device->accountId?device->accountId:"") ||
				!jb.addObjectProp("storageBlockCount", 17) ||
				!jb.valueInt(device->storageBlockCount) ||
				!jb.addObjectProp("storageBlockSize", 16) ||
				!jb.valueInt(device->storageBlockSize) ||
				!jb.addObjectProp("storageUse", 10) ||
				!jb.valueInt(device->storageUse) ||
				!jb.addObjectProp("storageCrypt", 12) ||
				!jb.valueString(cryptAlgo) ||
				!jb.addObjectProp("status", 6) ||
				!jb.valueString(devStatus) ||
				!jb.addObjectProp("loginTime", 9) ||
				!jb.valueString(loginTime) ||
				!jb.addObjectProp("createTime", 10) ||
				!jb.valueString(createTime) ||
				!jb.addObjectProp("updateTime", 10) ||
				!jb.valueString(updateTime) ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
				return false;
			}
			if( session != NULL ) {
				if( !jb.addObjectProp("sessionStartTime", 16) ||
					!jb.valueString(startTime) ||
					!jb.addObjectProp("sessionLastTime", 15) ||
					!jb.valueString(lastTime) ) {
					taskLog(q, LOG_ERR, "json builder failed(2). code=%d.", jb.getError());
					return false;
				}
			}
			if(	!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(3). code=%d.", jb.getError());
				return false;
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

}

