/*
 * redis_db.cpp
 *
 *  Created on: 2012/12/11
 *      Author: yasuoki
 */

#include <stdlib.h>
#include <syslog.h>
#include <memory.h>
#include <time.h>
#include <stdarg.h>
#include <hiredis/hiredis.h>
#include <unistd.h>
#include <assert.h>

#include "redis_db.h"
#include "sentinel.h"

#define LOCK_TIMEOUT	5
namespace SST {

RedisDB::RedisDB()
{
	mConn	= NULL;

}

RedisDB::~RedisDB()
{
	if( mConn )
		redisFree(mConn);
}

bool RedisDB::configure(Sentinel *obj, Conf *conf)
{
	if( !DBPort::configure(obj, conf) ) return false;
	strcpy(mServer, conf->dbServer);
	mPort = conf->dbPort;
	if( conf->dbAuth )
		strcpy(mAuth, conf->dbAuth);
	return true;
}

void	clear()
{

}

bool RedisDB::connect()
{
	redisContext *c = redisConnect(mServer, mPort);
	if( c->err != 0 ) {
		//c->errstr
		mSentinel->log(LOG_ERR, "Database connect error. %s", c->errstr );
		redisFree(c);
		return false;
	}
	if( mConn ) redisFree(mConn);
	mConn = c;
	mSentinel->log(LOG_INFO, "Database connect ok. inst=%x", this );
	return true;
}

void RedisDB::close()
{
	if( mConn ) {
		redisFree(mConn);
		mConn = NULL;
	}
}

ErrorCode	RedisDB::getInt(long long int &v, const char *msg, const char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
	redisReply *rep = (redisReply*)redisvCommand(mConn, fmt, ap);
	va_end(ap);
	if( rep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB:%s: redis error. %s", msg, mConn->errstr);
		return ErrorSystem;
	}
	if( rep->type == REDIS_REPLY_NIL ) {
		freeReplyObject(rep);
		return ErrorNotFound;
	}
	if( rep->type != REDIS_REPLY_INTEGER ) {
		freeReplyObject(rep);
		return ErrorDataType;
	}
	v = rep->integer;
	return ErrorOk;
}

ErrorCode RedisDB::getStr(redisReply **rep, const char *msg, const char *fmt, ...)
{
	va_list ap;
    va_start(ap, fmt);
	*rep = (redisReply*)redisvCommand(mConn, fmt, ap);
	va_end(ap);
	if( *rep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB:%s: redis error. %s", msg, mConn->errstr);
		return ErrorSystem;
	}
	if( (*rep)->type == REDIS_REPLY_NIL ) {
		mSentinel->log(LOG_ERR, "RedisDB:%s: redis error. %s", msg, fmt);
		freeReplyObject(*rep);
		*rep	= NULL;
		return ErrorNotFound;
	}
	if( (*rep)->type != REDIS_REPLY_STRING ) {
		mSentinel->log(LOG_ERR, "RedisDB:%s: redis type error. type=%d", msg, (*rep)->type);
		freeReplyObject(*rep);
		*rep	= NULL;
		return ErrorDataType;
	}
	return ErrorOk;
}

ErrorCode	RedisDB::getJson(pjson::json **js, const char *msg, const char *cmd, ...)
{
	va_list ap;
    va_start(ap, cmd);
    redisReply *rep = (redisReply*)redisvCommand(mConn, cmd, ap);
	va_end(ap);
	*js = NULL;

	if( rep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB:%s: redis error. %s", msg, mConn->errstr);
		return ErrorSystem;
	}
	if( rep->type == REDIS_REPLY_NIL ) {
		freeReplyObject(rep);
		return ErrorNotFound;
	}
	if( rep->type != REDIS_REPLY_STRING ) {
		mSentinel->log(LOG_ERR, "RedisDB:%s: redis type error. type=%d", msg, rep->type);
		freeReplyObject(rep);
		return ErrorDataType;
	}

	pjson::parser ps;
	pjson::json *_js = ps.parse(rep->str, rep->len);
	if( _js == NULL ) {
		sprintf(mConn->errstr, "json parse error. position=%d", ps.getErrorPos());
		mSentinel->log(LOG_ERR, "RedisDB:%s: json parse error. pos=%d", msg, ps.getErrorPos());
		freeReplyObject(rep);
		return ErrorData;
	}
	freeReplyObject(rep);
	*js	= _js;
	return ErrorOk;
}

ErrorCode RedisDB::execStatusCommand(const char *msg, const char *fmt, ...)
{
	va_list ap;
    va_start(ap, fmt);
    redisReply *rep = (redisReply*)redisvCommand(mConn, fmt, ap);
	va_end(ap);

	if( rep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB:%s: redis error. %s", msg, mConn->errstr);
		return ErrorSystem;
	}
	if( rep->type != REDIS_REPLY_STATUS ) {
		mSentinel->log(LOG_ERR, "RedisDB:%s: redis type error. type=%d", msg, rep->type);
		freeReplyObject(rep);
		return ErrorDataType;
	}
	if( strcmp(rep->str,"OK") != 0 ) {
		mSentinel->log(LOG_ERR, "%s: redis set command error. reply=%s", msg, rep->str);
		freeReplyObject(rep);
		return ErrorSystem;
	}
	freeReplyObject(rep);
	return ErrorOk;
}

ResultList * RedisDB::createListResult(const redisReply *pRep, size_t nFrom, size_t nCount)
{
	size_t i, n;
	size_t maxLen = 0;
	size_t elmCount = 0;

	for( i = nFrom; i < pRep->elements && elmCount < nCount; i++ ) {
		if( pRep->element[i]->type == REDIS_REPLY_STRING ) {
			size_t len = pRep->element[i]->len;
			if( len > maxLen ) maxLen = len;
			elmCount++;
		}
	}

	size_t size = sizeof(ResultList)
				+ DBUtil::sizeofRBase()
				+ DBUtil::sizeofList(elmCount, maxLen);
	char *ptr = (char *)malloc(size);
	if( ptr == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::createListResult: no more memory. count=%d len=%d size=%d", elmCount, maxLen, size);
		return NULL;
	}
	ResultList *res = (ResultList*)ptr;
	ptr += sizeof(ResultList) + sizeof(char*) * elmCount;

	res->mList.size		= size - sizeof(ResultList);
	res->mList.count	= elmCount;
	res->mList.values	= 0;
	for( i = nFrom, n = 0; i <  pRep->elements && n < nCount; i++ ) {
		if( pRep->element[i]->type == REDIS_REPLY_STRING ) {
			res->mList.value[n]	= ptr;
			strcpy(res->mList.value[n], pRep->element[i]->str);
			ptr += pRep->element[i]->len+1;
			n++;
		}
	}
	res->mList.values	= n;

	mSentinel->log(LOG_DEBUG, "RedisDB::createListResult: from=%d count=%d list=%d out=%d", nFrom, nCount, pRep->elements, n);

	return res;
}

ErrorCode RedisDB::getListCount(const char *key, size_t &nRec)
{
	nRec = 0;

	long long int nLen;
	ErrorCode e = getInt(nLen, "getCount", "LLEN %s", key);
	if( e != ErrorOk ) {
		return e;
	}
	nRec = (size_t)nLen;
	return ErrorOk;
}

ErrorCode RedisDB::getListList(const char *key, size_t nFrom, size_t nCount, ResultList **pList)
{
	*pList	= NULL;

	if( nCount == (size_t)-1 ) {
		ErrorCode e = getListCount(key, nCount);
		if( e != ErrorOk ) {
			return e;
		}
		nCount = nCount - nFrom;
	}

	redisReply *pRep = (redisReply*)redisCommand(mConn, "LRANGE %s %d %d", key, nFrom, nFrom+nCount-1);
	if( pRep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::getList: redis error %s", mConn->errstr);
		return ErrorSystem;
	}
	if( pRep->type == REDIS_REPLY_NIL ) {
		freeReplyObject(pRep);
		return ErrorOk;
	}
	if( pRep->type != REDIS_REPLY_ARRAY ) {
		mSentinel->log(LOG_ERR, "RedisDB::getList: data type error. type=%d", pRep->type);
		freeReplyObject(pRep);
		return ErrorDataType;
	}

	*pList = createListResult(pRep, 0, pRep->elements);
	if( *pList == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::getList: no more memory. count=%d", nCount);
		freeReplyObject(pRep);
		return ErrorNoMemory;
	}
	freeReplyObject(pRep);
	return ErrorOk;
}

ErrorCode RedisDB::getKeyCount(const char *key, size_t &nRec)
{
	nRec = 0;
	// MEMO:これはO(N)で遅いよ
	redisReply *pRep = (redisReply*)redisCommand(mConn, "KEYS %s*", key);
	if( pRep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::getKeyCount: redis error. %s", mConn->errstr);
		return ErrorData;
	}
	if( pRep->type == REDIS_REPLY_NIL ) {
		freeReplyObject(pRep);
		return ErrorOk;
	}
	if( pRep->type != REDIS_REPLY_ARRAY ) {
		mSentinel->log(LOG_ERR, "RedisDB::getKeyCount: data type error. type=%d", pRep->type);
		freeReplyObject(pRep);
		return ErrorDataType;
	}
	nRec = pRep->elements;
	freeReplyObject(pRep);
	return ErrorOk;
}

ErrorCode RedisDB::getKeyList(const char *key, size_t nFrom, size_t nCount, ResultList **pList)
{
	*pList = NULL;

	// MEMO:これはO(N)で遅いよ
	redisReply *pRep = (redisReply*)redisCommand(mConn, "keys %s*", key);
	if( pRep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::getKeyList: redis error. %s", mConn->errstr);
		return ErrorData;
	}
	if( pRep->type == REDIS_REPLY_NIL ) {
		(*pList)->mList.values = 0;
		freeReplyObject(pRep);
		return ErrorOk;
	}
	if( pRep->type != REDIS_REPLY_ARRAY ) {
		mSentinel->log(LOG_ERR, "RedisDB::getKeyList: data type error. type=%d", pRep->type);
		freeReplyObject(pRep);
		return ErrorDataType;
	}

	if( nCount == (size_t)-1 || nCount > pRep->elements )
		nCount = pRep->elements;

	*pList = createListResult(pRep, nFrom, nCount);
	if( *pList == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::getKeyList: no more memory. counts=%d", nCount);
		freeReplyObject(pRep);
		return ErrorNoMemory;
	}
	freeReplyObject(pRep);
	return ErrorOk;
}


ErrorCode RedisDB::getSetCount(const char *key, size_t &nRec)
{
	nRec = 0;
	redisReply *pRep = (redisReply*)redisCommand(mConn, "SCARD %s", key);
	if( pRep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::getSetCount: redis error. %s", mConn->errstr);
		return ErrorData;
	}
	if( pRep->type == REDIS_REPLY_NIL ) {
		freeReplyObject(pRep);
		return ErrorOk;
	}
	if( pRep->type != REDIS_REPLY_INTEGER ) {
		mSentinel->log(LOG_ERR, "RedisDB::getSetCount: data type error. type=%d", pRep->type);
		freeReplyObject(pRep);
		return ErrorDataType;
	}
	nRec = pRep->integer;
	freeReplyObject(pRep);
	return ErrorOk;
}

ErrorCode	RedisDB::isSetMember(const char *key, const char *member, bool *res)
{
	*res = false;
	redisReply *pRep = (redisReply*)redisCommand(mConn, "ISSETMEMBER %s, %s", key, member);
	if( pRep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::isSetMember: redis error. %s", mConn->errstr);
		return ErrorData;
	}
	if( pRep->type == REDIS_REPLY_NIL ) {
		freeReplyObject(pRep);
		return ErrorOk;
	}
	if( pRep->type != REDIS_REPLY_INTEGER ) {
		mSentinel->log(LOG_ERR, "RedisDB::isSetMember: data type error. type=%d", pRep->type);
		freeReplyObject(pRep);
		return ErrorDataType;
	}
	if( pRep->integer ) {
		*res	= true;
	} else {
		*res	= false;
	}
	freeReplyObject(pRep);
	return ErrorOk;
}

ErrorCode RedisDB::getSetList(const char *key, size_t nFrom, size_t nCount, ResultList **pList)
{
	*pList = NULL;

	redisReply *pRep = (redisReply*)redisCommand(mConn, "SMEMBERS %s", key);
	if( pRep == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::getSetList: redis error. %s", mConn->errstr);
		return ErrorData;
	}
	if( pRep->type == REDIS_REPLY_NIL ) {
		(*pList)->mList.values = 0;
		freeReplyObject(pRep);
		return ErrorOk;
	}
	if( pRep->type != REDIS_REPLY_ARRAY ) {
		mSentinel->log(LOG_ERR, "RedisDB::getSetList: data type error. type=%d", pRep->type);
		freeReplyObject(pRep);
		return ErrorDataType;
	}

	if( nCount == (size_t)-1 || nCount > pRep->elements )
		nCount = pRep->elements;

	*pList = createListResult(pRep, nFrom, nCount);
	if( *pList == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::getSetList: no more memory. counts=%d", nCount);
		freeReplyObject(pRep);
		return ErrorNoMemory;
	}
	freeReplyObject(pRep);
	return ErrorOk;
}

/////////////////////////////////////////////////////////////////////////////
// account
ErrorCode RedisDB::getAccountId(const char *uid, char *id_ret)
{
	StdBuffer key;
	DBUtil::keyAccountRev(uid, key);
	redisReply *pRep = NULL;

	ErrorCode e = getStr(&pRep, "getAccountId", "get %s", key.getPtr());
	if( e != ErrorOk ) {
		if( pRep ) freeReplyObject(pRep);
		return e;
	}
	strcpy(id_ret, pRep->str);
	freeReplyObject(pRep);
	return ErrorOk;
}

ErrorCode	RedisDB::setAccountInfo(Account *acc, char *id_ret)
{
	bool newId = false;
	bool newRevKey = false;
	StdBuffer keyAid;
	const char *aid;
	StdBuffer keyRev;

	if( acc->uid == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo: userid is not specfied.");
		return ErrorData;
	}

	if( acc->id == NULL ) {
		// account id
		long long int aid_seq;
		ErrorCode e = getInt(aid_seq, "setAccountInfo", "INCR IDSEQ_A");
		if( e != ErrorOk ) {
			return e;
		}
		DBUtil::keyAccount(aid_seq, keyAid);
		aid		= keyAid.getPtr();
		newId	= true;
		newRevKey = true;
	} else {
		aid		= acc->id;
	}

	DBUtil::keyAccountRev(acc->uid, keyRev);

	time_t now = time(NULL);
	char _createTime[256];
	char _updateTime[256];
	char _loginTime[256];
	_createTime[0] = 0;
	_updateTime[0] = 0;
	_loginTime[0] = 0;
	if( newId ) {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));
	} else {
		if( acc->createTime )
			strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&acc->createTime));
		if( acc->loginTime )
			strftime(_loginTime, sizeof(_loginTime), "%Y/%m/%d-%H:%M:%S", gmtime(&acc->loginTime));
	}
	strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:1: json builder failed %d. aid=%s", jb.getError(), aid);
		return ErrorSystem;
	}

	if( !jb.addObjectProp("userId", 6) ||
		!jb.valueString(acc->uid) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:2: json builder failed %d. aid=%s", jb.getError(), aid);
		return ErrorSystem;
	}
	if( acc->name1 ) {
		if( !jb.addObjectProp("name1", 5) ||
			!jb.valueString(acc->name1) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:2: json builder failed %d. aid=%s", jb.getError(), aid);
			return ErrorSystem;
		}
	}
	if( acc->name2 ) {
		if( !jb.addObjectProp("name2", 5) ||
		!jb.valueString(acc->name2) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:3: json builder failed %d. aid=%s", jb.getError(), aid);
			return ErrorSystem;
		}
	}
	if( acc->locale ) {
		if( !jb.addObjectProp("locale", 6) ||
		!jb.valueString(acc->locale) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:5: json builder failed %d. aid=%s", jb.getError(), aid);
			return ErrorSystem;
		}
	}
	if( acc->authData ) {
		if( !jb.addObjectProp("authData", 8) ||
			!jb.valueString(acc->authData) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:4: json builder failed %d. aid=%s", jb.getError(), aid);
			return ErrorSystem;
		}
	}
	if( !jb.addObjectProp("authType", 8) ||
		!jb.valueInt(acc->authType) ||
		!jb.addObjectProp("level", 5) ||
		!jb.valueInt(acc->level) ||
		!jb.addObjectProp("maxStorate", 10) ||
		!jb.valueInt(acc->maxStorage) ||
		!jb.addObjectProp("useStorage", 10) ||
		!jb.valueInt(acc->useStorage) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:6: json builder failed %d. aid=%s", jb.getError(), aid);
		return ErrorSystem;
	}
	if( _createTime[0] ) {
		if( !jb.addObjectProp("createTime", 10) ||
			!jb.valueString(_createTime) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:7: json builder failed %d. aid=%s", jb.getError(), aid);
			return ErrorSystem;
		}
	}
	if( !jb.addObjectProp("updateTime", 10) ||
		!jb.valueString(_updateTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:8: json builder failed %d. aid=%s", jb.getError(), aid);
		return ErrorSystem;
	}
	if( _loginTime[0] ) {
		if( !jb.addObjectProp("loginTime", 10) ||
			!jb.valueString(_loginTime) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:8: json builder failed %d. aid=%s", jb.getError(), aid);
			return ErrorSystem;
		}
	}
	if( !jb.endObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:9: json builder failed %d. aid=%s", jb.getError(), aid);
		return ErrorSystem;
	}

	pjson::json *js = jb.detouch();
	StdBuffer b;
	if( !js->getText(&b,false) ) {
		delete js;
		mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:10: json builder failed %d. aid=%s", jb.getError(), aid);
		return ErrorSystem;
	}
	delete js;

	ResultAccount *curAcc = NULL;
	if( !newId ) {
		ErrorCode e = getAccountInfo(aid, &curAcc);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo: account error. aid=%s, code=%d", aid, e);
			return e;
		}
		if( strcmp(curAcc->mAccount.uid, acc->uid) != 0 )
			newRevKey = true;
	}

	if( newRevKey ) {
		long long int r;
		ErrorCode e = getInt(r, "setAccountInfo", "SETNX %s %s", keyRev.getPtr(), aid);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:1: reverse key set failed. aid=%s", aid);
			if( curAcc )	free(curAcc);
			return e;
		}
		if( r != 1 ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo: reverse key duplication. aid=%s", aid);
			if( curAcc )	free(curAcc);
			return ErrorDuplicate;
		}
	}

	ErrorCode e  = execStatusCommand("RedisDB::setAccountInfo:: set", "SET %s %s", aid, b.getPtr());
	if( e != ErrorOk ) {
		if( newRevKey ) {
			long long int r;
			ErrorCode e = getInt(r, "setAccountInfo", "DEL %s", keyRev.getPtr());
			if( e != ErrorOk ) {
				mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:1: reverse key delete failed. aid=%s", aid);
			}
		}
		if( curAcc )	free(curAcc);
		return e;
	}

	if( !newId && newRevKey ) {
		DBUtil::keyAccountRev(curAcc->mAccount.uid, keyRev);
		long long int r;
		ErrorCode e = getInt(r, "setAccountInfo", "DEL %s", keyRev.getPtr());
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setAccountInfo:1: reverse key delete failed. aid=%s", aid);
		}
	}
	if( curAcc )	free(curAcc);
	if( id_ret )
		strcpy(id_ret, aid);

	return ErrorOk;
}

ErrorCode RedisDB::getAccountInfo(const char *id, ResultAccount **res)
{
	*res = NULL;
	pjson::json *js;
	assert(id[0] == 'A');
	ErrorCode e = getJson(&js, "getAccountInfo", "GET %s", id);
	if( e != ErrorOk ) {
		if( e == ErrorData )
			mSentinel->log(LOG_WARNING, "RedisDB::getAccountInfo: data error.");
		return e;
	}

	pjson::value	*v	= js->getValue();
	if( v->vt != pjson::vt_object ) {
		delete js;
		return ErrorData;
	}

	pjson::value *vUid			= js->get("userId", v);
	pjson::value *vName1		= js->get("name1", v);
	pjson::value *vName2		= js->get("name2", v);
	pjson::value *vLocale		= js->get("locale", v);
	pjson::value *vAuthType		= js->get("authType", v);
	pjson::value *vAuthData		= js->get("authData", v);
	pjson::value *vLevel		= js->get("level", v);
	pjson::value *vMaxStorage	= js->get("maxStorage", v);
	pjson::value *vUseStorage	= js->get("useStorage", v);
	pjson::value *vCreateTime	= js->get("createTime", v);
	pjson::value *vUpdateTime	= js->get("updateTime", v);
	pjson::value *vLoginTime	= js->get("loginTime", v);

	Account info;
	memset(&info, 0, sizeof(info));
	info.id			= (char*)id;
	info.uid		= (char*)((vUid && vUid->vt == pjson::vt_string) ? vUid->vString : NULL);
	info.name1		= (char*)((vName1 && vName1->vt == pjson::vt_string) ? vName1->vString : NULL);
	info.name2		= (char*)((vName2 && vName2->vt == pjson::vt_string) ? vName2->vString : NULL);
	info.locale		= (char*)((vLocale && vLocale->vt == pjson::vt_string) ? vLocale->vString : 0);
	info.authType	= (AuthType)((vAuthType && vAuthType->vt == pjson::vt_int) ? vAuthType->vInt : 0);
	info.authData	= (char*)((vAuthData && vAuthData->vt == pjson::vt_string) ? vAuthData->vString : NULL);
	info.level		= (AccountLevel)((vLevel && vLevel->vt == pjson::vt_int) ? vLevel->vInt : AccountUser);
	info.maxStorage	= (vMaxStorage && vMaxStorage->vt == pjson::vt_int) ? vMaxStorage->vInt : 0;
	info.useStorage	= (vUseStorage && vUseStorage->vt == pjson::vt_int) ? vUseStorage->vInt : 0;
	struct tm tm_create;
	if( vCreateTime && vCreateTime->vt == pjson::vt_string) {
		strptime(vCreateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_create);
		info.createTime	= timegm(&tm_create);
	} else {
		info.createTime	= 0;
	}
	struct tm tm_update;
	if( vUpdateTime && vUpdateTime->vt == pjson::vt_string) {
		strptime(vUpdateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_update);
		info.updateTime	= timegm(&tm_update);
	} else {
		info.updateTime	= 0;
	}
	struct tm tm_login;
	if( vLoginTime && vLoginTime->vt == pjson::vt_string) {
		strptime(vUpdateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_login);
		info.loginTime	= timegm(&tm_login);
	} else {
		info.loginTime	= 0;
	}

	size_t size = sizeof(ResultAccount)
				+ DBUtil::sizeofAccount(&info);
	char *ptr = (char*)malloc(size);
	ResultAccount *ret = (ResultAccount*)ptr;
	ptr	+= sizeof(ResultAccount);
	ptr	= DBUtil::writeAccount(&ret->mAccount, ptr, &info);
	delete js;

	*res = ret;
	return ErrorOk;
}

ErrorCode RedisDB::remAcocuntInfo(const char *id)
{
	// TODO: 端末内のデータは全て削除されていること
	// TODO: 共有情報が何もないこと＝グループに参加していないこと
	// TODO: ログオフし、ログイン禁止状態となっていること

	long long int rem;
	ErrorCode e;
	StdBuffer key;

	// delete reverse key
	{
		ResultAccount *acc = NULL;
		e = getAccountInfo(id, &acc);
		if( e != ErrorOk ) {
			return e;
		}
		DBUtil::keyAccountRev(acc->mAccount.uid, key);
		free(acc);
	}

	e = getInt(rem, "remAcocuntInfo", "DEL %s", key.getPtr());
	if( e != ErrorOk ) {
		mSentinel->log(LOG_WARNING, "RedisDB::remAcocuntInfo: delete accountReverseKey error. code=%d aid=%s key=%s", e, id, key.getPtr());
	}

	// delete accountGroupList
	DBUtil::keyAccountGroupList(id, key);
	e = getInt(rem, "remAcocuntInfo", "DEL %s", key.getPtr());
	if( e != ErrorOk && e != ErrorNotFound ) {
		mSentinel->log(LOG_WARNING, "RedisDB::remAcocuntInfo: delete accountGroupList error. code=%d aid=%s key=%s", e, id, key.getPtr());
	}
	// delete accountDeviceList
	DBUtil::keyAccountDeviceList(id, key);
	e = getInt(rem, "remAcocuntInfo", "DEL %s", key.getPtr());
	if( e != ErrorOk && e != ErrorNotFound ) {
		mSentinel->log(LOG_WARNING, "RedisDB::remAcocuntInfo: delete accountDeviceList error. code=%d aid=%s key=%s", e, id, key.getPtr());
	}
	// delete account info
	e = getInt(rem, "remAcocuntInfo", "DEL %s", id);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_WARNING, "RedisDB::remAcocuntInfo: delete accountInfo error. code=%d aid=%s key=%s", e, id, key.getPtr());
	}
	return ErrorOk;
}

ErrorCode	RedisDB::getAccountCount(size_t &nRec)
{
	ErrorCode e = getKeyCount("A", nRec);
	return e;
}

ErrorCode RedisDB::getAccountList(size_t nFrom, size_t nCount, ResultList **pList)
{
	ErrorCode e = getKeyList("A", nFrom, nCount, pList);
	return e;
}

ErrorCode RedisDB::getAccountDeviceCount(const char *aid, size_t &nRec)
{
	StdBuffer key;
	DBUtil::keyAccountDeviceList(aid, key);
	ErrorCode e = getSetCount(key.getPtr(), nRec);
	return e;
}

ErrorCode RedisDB::getAccountDeviceList(const char *aid, size_t nFrom, size_t nCount, ResultList **pList)
{
	StdBuffer key;
	DBUtil::keyAccountDeviceList(aid, key);
	ErrorCode e = getSetList(key.getPtr(), nFrom, nCount, pList);
	return e;
}
/////////////////////////////////////////////////////////////////////////////
// device
ErrorCode RedisDB::getDeviceId(const char *udid, char *id_ret)
{
	StdBuffer key;
	DBUtil::keyDeviceRev(udid, key);
	redisReply *pRep = NULL;

	ErrorCode e = getStr(&pRep, "getDeviceId", "GET %s", key.getPtr());
	if( e != ErrorOk ) {
		if( pRep ) freeReplyObject(pRep);
		return e;
	}
	strcpy(id_ret, pRep->str);
	freeReplyObject(pRep);
	return ErrorOk;
}

ErrorCode RedisDB::setDeviceInfo(DeviceInfo *dev, char *id_ret)
{
	bool newId = false;
	StdBuffer key;
	const char *did = NULL;
	StdBuffer keyRev;

	if( dev->udid == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo: udid is not specfied.");
		return ErrorData;
	}
	DBUtil::keyDeviceRev(dev->udid, keyRev);

	if( dev->id == NULL ) {
		mSentinel->log(LOG_DEBUG, "RedisDB::setDeviceInfo: dev->id is null");
		// exist ?
		if( dev->accountId ) {
			redisReply *pRep = NULL;
			ErrorCode e = getStr(&pRep,"setDeviceInfo","GET %s", keyRev.getPtr());
			if( e == ErrorOk ) {
				mSentinel->log(LOG_DEBUG, "RedisDB::setDeviceInfo: %s is found", keyRev.getPtr());
				ResultDevice *pDev = NULL;
				e = getDeviceInfo(pRep->str, &pDev);
				if( e == ErrorOk ) {
					mSentinel->log(LOG_DEBUG, "RedisDB::setDeviceInfo: device info load ok %s", pRep->str);
					if( strcmp(pDev->mDevice.accountId, dev->accountId) == 0 ) {
						mSentinel->log(LOG_DEBUG, "RedisDB::setDeviceInfo: device account match %s", pDev->mDevice.accountId);
						key.add(pDev->mDevice.id, -1);
						did = key.getPtr();
						mSentinel->log(LOG_DEBUG, "RedisDB::setDeviceInfo: did set to %s", did);
					}
				}
				if( pDev ) free(pDev);
			}
			if( pRep ) freeReplyObject(pRep);
		}
		if( did == NULL ) {
			// dev id
			long long int did_seq;
			ErrorCode e = getInt(did_seq, "setDeviceInfo", "INCR IDSEQ_D");
			if( e != ErrorOk ) {
				return e;
			}
			DBUtil::keyDevice(did_seq, key);
			did		= key.getPtr();
			newId	= true;
			mSentinel->log(LOG_DEBUG, "RedisDB::setDeviceInfo: new did set to %s", did);
		}
	} else {
		did		= dev->id;
		mSentinel->log(LOG_DEBUG, "RedisDB::setDeviceInfo: did set to %s", did);
	}

	if( newId ) {
		long long int r;
		ErrorCode e = getInt(r, "setDeviceInfo", "SETNX %s %s", keyRev.getPtr(), did);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:1: reverse key set failed. did=%s", did);
			return e;
		}
		if( r != 1 ) {
			mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo: reverse key duplication. did=%s", did);
			return ErrorDuplicate;
		}
	}

	time_t now = time(NULL);
	char _loginTime[256];
	char _createTime[256];
	char _updateTime[256];
	if( dev->loginTime != 0 ) {
		strftime(_loginTime, sizeof(_loginTime), "%Y/%m/%d-%H:%M:%S", gmtime(&dev->loginTime));
	} else {
		_loginTime[0] = 0;
	}
	if( newId ) {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));
	} else {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&dev->createTime));
	}
	strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:1: json builder failed %d. did=%s", jb.getError(), did);
		return ErrorSystem;
	}

	if( !jb.addObjectProp("udid", 4) ||
		!jb.valueString(dev->udid) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:2: json builder failed %d. did=%s", jb.getError(), did);
		return ErrorSystem;
	}
	if( dev->OS ) {
		if( !jb.addObjectProp("os", 2) ||
			!jb.valueString(dev->OS) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:2: json builder failed %d. did=%s", jb.getError(), did);
			return ErrorSystem;
		}
	}
	if( dev->OSVersion ) {
		if( !jb.addObjectProp("osVersion", 9) ||
			!jb.valueString(dev->OSVersion) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:2: json builder failed %d. did=%s", jb.getError(), did);
			return ErrorSystem;
		}
	}
	if( dev->carrier ) {
		if( !jb.addObjectProp("carrier", 7) ||
		!jb.valueString(dev->carrier) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:3: json builder failed %d. did=%s", jb.getError(), did);
			return ErrorSystem;
		}
	}
	if( dev->model ) {
		if( !jb.addObjectProp("model", 5) ||
			!jb.valueString(dev->model) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:4: json builder failed %d. did=%s", jb.getError(), did);
			return ErrorSystem;
		}
	}
	if( dev->signature ) {
		if( !jb.addObjectProp("signature", 9) ||
		!jb.valueString(dev->signature) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:5: json builder failed %d. did=%s", jb.getError(), did);
			return ErrorSystem;
		}
	}
	if( dev->accountId ) {
		if( !jb.addObjectProp("accountId", 11) ||
		!jb.valueString(dev->accountId) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:5: json builder failed %d. did=%s", jb.getError(), did);
			return ErrorSystem;
		}
	}
	if( !jb.addObjectProp("storageCrypt", 12) ||
		!jb.valueInt(dev->storageCrypt) ||
		!jb.addObjectProp("storageBlockSize", 16) ||
		!jb.valueInt(dev->storageBlockSize) ||
		!jb.addObjectProp("storageBlockCount", 17) ||
		!jb.valueInt(dev->storageBlockCount) ||
		!jb.addObjectProp("storageUse", 10) ||
		!jb.valueInt(dev->storageUse) ||
		!jb.addObjectProp("status", 6) ||
		!jb.valueInt(dev->status) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:6: json builder failed %d. did=%s", jb.getError(), did);
		return ErrorSystem;
	}
	if( _loginTime[0] != 0 ) {
		if( !jb.addObjectProp("loginTime", 9) ||
			!jb.valueString(_loginTime) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:7: json builder failed %d. did=%s", jb.getError(), did);
			return ErrorSystem;
		}
	}
	if( !jb.addObjectProp("createTime", 10) ||
		!jb.valueString(_createTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:8: json builder failed %d. did=%s", jb.getError(), did);
		return ErrorSystem;
	}
	if( !jb.addObjectProp("updateTime", 10) ||
		!jb.valueString(_updateTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:9: json builder failed %d. did=%s", jb.getError(), did);
		return ErrorSystem;
	}
	if( !jb.endObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:10: json builder failed %d. did=%s", jb.getError(), did);
		return ErrorSystem;
	}

	pjson::json *js = jb.detouch();
	StdBuffer b;
	if( !js->getText(&b,false) ) {
		delete js;
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceInfo:10: json builder failed %d. did=%s", jb.getError(), did);
		return ErrorSystem;
	}
	delete js;

	ErrorCode e = execStatusCommand("RedisDB::setDeviceInfo:: set", "SET %s %s", did, b.getPtr());
	if( e != ErrorOk ) {
		if( newId ) {
			long long int r;
			getInt(r, "setDeviceInfo", "DEL %s", keyRev.getPtr());
		}
		return e;
	}

	long long int r;
	StdBuffer keyList;
	DBUtil::keyAccountDeviceList(dev->accountId, keyList);
	e = getInt(r, "setDeviceInfo", "SADD %s %s", keyList.getPtr(), did);
	if( e != ErrorOk ) {
		if( newId ) {
			long long int r;
			getInt(r, "setDeviceInfo", "DEL %s", did);
			getInt(r, "setDeviceInfo", "DEL %s", keyRev.getPtr());
		}
		return e;
	}

	if( id_ret )
		strcpy(id_ret, did);
	return ErrorOk;
}

ErrorCode RedisDB::getDeviceInfo(const char *id, ResultDevice **res)
{
	*res = NULL;
	pjson::json *js;

	assert(id[0] == 'D');

	ErrorCode e = getJson(&js, "getDeviceInfo", "GET %s", id);
	if( e != ErrorOk ) {
		if( e == ErrorData )
			mSentinel->log(LOG_DEBUG, "RedisDB::getDeviceInfo: data error.");
		return e;
	}

	pjson::value	*v	= js->getValue();
	if( v->vt != pjson::vt_object ) {
		delete js;
		return ErrorData;
	}

	pjson::value *vUdid			= js->get("udid", v);
	pjson::value *vOS			= js->get("os", v);
	pjson::value *vOSVer		= js->get("osVersion", v);
	pjson::value *vCarrier		= js->get("carrier", v);
	pjson::value *vModel		= js->get("model", v);
	pjson::value *vSignature	= js->get("signature", v);
	pjson::value *vAccountId	= js->get("accountId", v);
	pjson::value *vStorageCR	= js->get("storageCrypt", v);
	pjson::value *vBlockSize	= js->get("storageBlockSize", v);
	pjson::value *vBlockCount	= js->get("storageBlockCount", v);
	pjson::value *vStorageUse	= js->get("storageUse", v);
	pjson::value *vStatus		= js->get("status", v);
	pjson::value *vLoginTime	= js->get("loginTime", v);
	pjson::value *vCreateTime	= js->get("createTime", v);
	pjson::value *vUpdateTime	= js->get("updateTime", v);

	DeviceInfo info;
	memset(&info, 0, sizeof(info));
	info.id				= (char*)id;
	info.udid			= (char*)((vUdid && vUdid->vt == pjson::vt_string) ? vUdid->vString : NULL);
	info.OS				= (char*)((vOS && vOS->vt == pjson::vt_string) ? vOS->vString : NULL);
	info.OSVersion		= (char*)((vOSVer && vOSVer->vt == pjson::vt_string) ? vOSVer->vString : NULL);
	info.carrier		= (char*)((vCarrier && vCarrier->vt == pjson::vt_string) ? vCarrier->vString : NULL);
	info.model			= (char*)((vModel && vModel->vt == pjson::vt_string) ? vModel->vString : NULL);
	info.signature		= (char*)((vSignature&& vSignature->vt == pjson::vt_string) ? vSignature->vString : NULL);
	info.accountId		= (char*)((vAccountId && vAccountId->vt == pjson::vt_string) ? vAccountId->vString : NULL);
	info.storageCrypt	= (CryptAlgorithm)((vStorageCR && vStorageCR->vt == pjson::vt_int) ? vStorageCR->vInt : 0);
	info.storageBlockSize	= (size_t)((vBlockSize && vBlockSize->vt == pjson::vt_int) ? vBlockSize->vInt : 0);
	info.storageBlockCount	= (size_t)((vBlockCount && vBlockCount->vt == pjson::vt_int) ? vBlockCount->vInt : 0);
	info.storageUse		= (size_t)((vStorageUse && vStorageUse->vt == pjson::vt_int) ? vStorageUse->vInt : 0);
	info.status			= (DeviceStatus)((vStatus && vStatus->vt == pjson::vt_int) ? vStatus->vInt : 0);
	struct tm tm_login;
	if( vLoginTime && vLoginTime->vt == pjson::vt_string) {
		strptime(vLoginTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_login);
		info.loginTime	= timegm(&tm_login);
	} else {
		info.loginTime	= 0;
	}
	struct tm tm_create;
	if( vCreateTime && vCreateTime->vt == pjson::vt_string) {
		strptime(vCreateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_create);
		info.createTime	= timegm(&tm_create);
	} else {
		info.createTime	= 0;
	}
	struct tm tm_last;
	if( vUpdateTime && vUpdateTime->vt == pjson::vt_string) {
		strptime(vUpdateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_last);
		info.updateTime	= timegm(&tm_last);
	} else {
		info.updateTime	= 0;
	}

	size_t size = sizeof(ResultDevice)
				+ DBUtil::sizeofRBase()
				+ DBUtil::sizeofDevice(&info);
	char *ptr = (char*)malloc(size);
	ResultDevice *ret = (ResultDevice*)ptr;
	ptr	+= sizeof(ResultDevice);
	ptr	= DBUtil::writeDevice(&ret->mDevice, ptr, &info);

	delete js;

	*res = ret;
	return ErrorOk;
}

ErrorCode  RedisDB::remDeviceInfo(const char *id)
{
	// TODO: 端末内のデータは全て削除されていること
	// TODO: 所有者から切り離されていること
	// TODO: ログオフし、ログイン禁止状態となっていること

	ResultDevice *dev = NULL;
	ErrorCode e = getDeviceInfo(id, &dev);
	if( e != ErrorOk ) {
		return e;
	}

	StdBuffer keyRev;
	DBUtil::keyDeviceRev(dev->mDevice.udid, keyRev);

	long long int rem;

	// delete device share info
	{
		long long int share_count=0;
		e = getInt(share_count, "remDeviceInfo", "LLEN LM%s", id);
		while( e == ErrorOk && share_count > 0 ) {
			redisReply *pShareList = (redisReply*)redisCommand(mConn, "LRANGE LM%s 0 %d", id, 50);
			if( pShareList == NULL ) {
				mSentinel->log(LOG_ERR, "RedisDB::remDeviceInfo. redis error. %s",  mConn->errstr);
				break;
			}
			if( pShareList->type == REDIS_REPLY_ARRAY && pShareList->elements > 0 ) {
				size_t i;
				for( i = 0; i < pShareList->elements; i++) {
					redisReply *sKey = pShareList->element[i];
					if( sKey && sKey->type == REDIS_REPLY_STRING ) {
						e = getInt(rem, "remAcocuntInfo", "DEL %s", sKey->str);
						e = getInt(rem, "remAcocuntInfo", "LREM LM%s %s 1", id, sKey->str);
					}
				}
			}
			e = getInt(share_count, "remDeviceInfo", "LLEN LM%s", id);
		}
		e = getInt(share_count, "remDeviceInfo", "DEL LM%s", id);
	}

	// delete reverse key
	e = getInt(rem, "remDeviceInfo rev", "DEL %s", keyRev.getPtr());
	if( e != ErrorOk ) {
		return e;
	}

	// delete device info
	e = getInt(rem, "remDeviceInfo", "DEL %s", id);
	if( e != ErrorOk ) {
		return e;
	}
	return ErrorOk;
}

ErrorCode RedisDB::getDeviceCount(size_t &nRec)
{
	ErrorCode e = getSetCount("D", nRec);
	return e;
}

ErrorCode RedisDB::getDeviceList(size_t nFrom, size_t nCount, ResultList **pList)
{
	ErrorCode e = getSetList("D", nFrom, nCount, pList);
	return e;
}

ErrorCode RedisDB::setDeviceShareInfo(const char *did, const DeviceShareInfo *shareInfo)
{
	assert(did[0] == 'D');

	StdBuffer keyList;
	DBUtil::keyDeviceShareList(did, keyList);

	StdBuffer keyShare;
	DBUtil::keyDeviceShare(did, shareInfo->shareId, keyShare);

	time_t now = time(NULL);
	char _shareTime[256];
	strftime(_shareTime, sizeof(_shareTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));
	char _deviceTime[256];
	char *deviceTime = NULL;
	if( shareInfo->deviceTime != 0 ) {
		strftime(_deviceTime, sizeof(_deviceTime), "%Y/%m/%d-%H:%M:%S", gmtime(&shareInfo->deviceTime));
		deviceTime	= _deviceTime;
	}

	typedef struct _DeviceShareInfo {
		char *				serviceId;
		char *				shareId;
		time_t				serverTime;
		revision_t			serverRevision;
		time_t				deviceTime;
		revision_t			deviceRevision;
		DeviceShareStatus	status;
		time_t				shareTime;
	} DeviceShareInfo;

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ||
		!jb.addObjectProp("sid", 3) ||
		!jb.valueString(shareInfo->serviceId) ||
		!jb.addObjectProp("shareId", 7) ||
		!jb.valueString(shareInfo->shareId) ||
		!jb.addObjectProp("deviceTime", 10) ||
		!jb.valueString(deviceTime) ||
		!jb.addObjectProp("deviceRevision", 14) ||
		!jb.valueInt(shareInfo->deviceRevision) ||
		!jb.addObjectProp("status", 6) ||
		!jb.valueInt(shareInfo->status) ||
		!jb.addObjectProp("shareTime", 9) ||
		!jb.valueString(_shareTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceShareInfo: json builder failed %d. did=%s", jb.getError(), did);
		return ErrorSystem;
	}

	pjson::json *js = jb.detouch();
	StdBuffer b;
	if( !js->getText(&b,false) ) {
		delete js;
		mSentinel->log(LOG_ERR, "RedisDB::setDeviceShareInfo: json builder failed %d. did=%s", jb.getError(), did);
		return ErrorSystem;
	}
	delete js;

	ErrorCode e = execStatusCommand("RedisDB::setDeviceShareInfo:: set", "SET %s %s", keyShare.getPtr(), b.getPtr());
	if( e != ErrorOk ) {
		return e;
	}

	long long int r;
	e = getInt(r, "setDeviceShareInfo", "SREM %s 0 %s", keyList.getPtr(), keyShare.getPtr());
	e = getInt(r, "setDeviceShareInfo", "SADD %s %s", keyList.getPtr(), keyShare.getPtr());
	if( e != ErrorOk ) {
		return e;
	}
	return ErrorOk;
}

ErrorCode RedisDB::remDeviceShareInfo(const char *did, const char *mid)
{
	return ErrorNotImpl;
}

ErrorCode RedisDB::getDeviceShareCount(const char *did, size_t &nRec)
{
	assert(did[0] == 'D');
	StdBuffer key;
	DBUtil::keyDeviceShareList(did, key);
	ErrorCode e = getListCount(key.getPtr(), nRec);
	return e;
}

ErrorCode RedisDB::getDeviceShareList(const char *did, size_t nFrom, size_t nCount, ResultList **pList)
{
	assert(did[0] == 'D');
	StdBuffer key;
	DBUtil::keyDeviceShareList(did, key);
	ErrorCode e = getListList(key.getPtr(), nFrom, nCount, pList);
	return e;
}

ErrorCode RedisDB::getAccountGroupCount(const char *aid, size_t &nRec)
{
	assert(aid[0] == 'A');
	StdBuffer key;
	DBUtil::keyAccountGroupList(aid, key);
	ErrorCode e = getSetCount(key.getPtr(), nRec);
	return e;
}

ErrorCode	RedisDB::getAccountGroupList(const char *aid, size_t nFrom, size_t nCount, ResultList **pList)
{
	assert(aid[0] == 'A');
	StdBuffer key;
	DBUtil::keyAccountGroupList(aid, key);
	ErrorCode e = getSetList(key.getPtr(), nFrom, nCount, pList);
	return e;
}

/////////////////////////////////////////////////////////////////////////////
// service
ErrorCode RedisDB::getServiceId(const char *name, char *ret_id)
{
	StdBuffer keyRev;
	DBUtil::keyServiceRev(name, keyRev);
	redisReply *pRep = NULL;
	ErrorCode e = getStr(&pRep, "getServiceId", "GET %s", keyRev.getPtr());
	if( e != ErrorOk ) {
		if( pRep )
			freeReplyObject(pRep);
		return e;
	}
	strcpy(ret_id, pRep->str);
	freeReplyObject(pRep);
	return ErrorOk;
}

ErrorCode RedisDB::setServiceInfo(ServiceInfo *si, char *sid_ret)
{
	if( si->name == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo: name is can't specfied. sid=%s", si->id);
		return ErrorData;
	}

	bool newId = false;
	bool newRevKey = false;

	const char *sid;
	StdBuffer key;
	StdBuffer keyRev;
	if( si->id == NULL ) {
		// service id
		long long int sid_seq;
		ErrorCode e = getInt(sid_seq, "setServiceInfo", "INCR IDSEQ_S");
		if( e != ErrorOk ) {
			return e;
		}
		DBUtil::keyService(sid_seq, key);
		sid	= key.getPtr();
		newId		= true;
		newRevKey	= true;
	} else {
		sid = si->id;
	}

	DBUtil::keyServiceRev(si->name, keyRev);

	time_t now = time(NULL);
	char _createTime[256];
	char _updateTime[256];
	if( newId ) {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));
	} else {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&si->createTime));
	}
	strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:1: json builder failed %d. sid=%s", jb.getError(), sid);
		return ErrorSystem;
	}

	if( !jb.addObjectProp("name", 4) ||
		!jb.valueString(si->name) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:2: json builder failed %d. sid=%s", jb.getError(), sid);
		return ErrorSystem;
	}
	if( si->title ) {
		if( !jb.addObjectProp("title", 5) ||
			!jb.valueString(si->title) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:2: json builder failed %d. sid=%s", jb.getError(), sid);
			return ErrorSystem;
		}
	}
	if( si->desc ) {
		if( !jb.addObjectProp("desc", 11) ||
			!jb.valueString(si->desc) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:2: json builder failed %d. sid=%s", jb.getError(), sid);
			return ErrorSystem;
		}
	}
	if( si->version) {
		if( !jb.addObjectProp("version", 7) ||
			!jb.valueString(si->version) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:3: json builder failed %d. sid=%s", jb.getError(), sid);
			return ErrorSystem;
		}
	}
	if( si->adminId ) {
		if( !jb.addObjectProp("adminId", 7) ||
			!jb.valueString(si->adminId) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:4: json builder failed %d. sid=%s", jb.getError(), sid);
			return ErrorSystem;
		}
	}
	if( si->refURL ) {
		if( !jb.addObjectProp("refURL", 6) ||
			!jb.valueString(si->refURL) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:4: json builder failed %d. sid=%s", jb.getError(), sid);
			return ErrorSystem;
		}
	}
	if( si->defaultGroup ) {
		if( !jb.addObjectProp("defaultGroup", 12) ||
			!jb.valueString(si->defaultGroup) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:4: json builder failed %d. sid=%s", jb.getError(), sid);
			return ErrorSystem;
		}
	}
	if( si->authData ) {
		if( !jb.addObjectProp("authData", 8) ||
			!jb.valueString(si->authData) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:6: json builder failed %d. sid=%s", jb.getError(), sid);
			return ErrorSystem;
		}
	}
	if( !jb.addObjectProp("scope", 5) ||
		!jb.valueInt(si->scope) ||
		!jb.addObjectProp("storageBlockSize", 16) ||
		!jb.valueInt(si->storageBlockSize) ||
		!jb.addObjectProp("storageBlockCount", 17) ||
		!jb.valueInt(si->storageBlockCount) ||
		!jb.addObjectProp("storageCrypt", 12) ||
		!jb.valueInt(si->storageCrypt) ||
		!jb.addObjectProp("authType", 8) ||
		!jb.valueInt(si->authType) ||
		!jb.addObjectProp("status", 6) ||
		!jb.valueInt(si->status) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:5: json builder failed %d. sid=%s", jb.getError(), sid);
		return ErrorSystem;
	}
	if( !jb.addObjectProp("createTime", 10) ||
		!jb.valueString(_createTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:7: json builder failed %d. sid=%s", jb.getError(), sid);
		return ErrorSystem;
	}
	if( !jb.addObjectProp("updateTime", 10) ||
		!jb.valueString(_updateTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:8: json builder failed %d. sid=%s", jb.getError(), sid);
		return ErrorSystem;
	}
	if( !jb.endObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:9: json builder failed %d. sid=%s", jb.getError(), sid);
		return ErrorSystem;
	}

	pjson::json *js = jb.detouch();
	StdBuffer b;
	if( !js->getText(&b,false) ) {
		delete js;
		mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:10: json builder failed %d. sid=%s", jb.getError(), sid);
		return ErrorSystem;
	}
	delete js;

	ResultService *curSvc = NULL;
	if( !newId ) {
		ErrorCode e = getServiceInfo(sid, &curSvc);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo: service error. sid=%s, code=%d", sid, e);
			return e;
		}
		if( strcmp(curSvc->mService.name, si->name) != 0 )
			newRevKey = true;
	}

	if( newRevKey ) {
		long long int r;
		ErrorCode e = getInt(r, "setServiceInfo", "SETNX %s %s", keyRev.getPtr(), sid);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo:1: reverse key set failed. sid=%s", sid);
			if( curSvc )	free(curSvc);
			return e;
		}
		if( r != 1 ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo: reverse key duplication. sid=%s", sid);
			if( curSvc )	free(curSvc);
			return ErrorDuplicate;
		}
	}

	ErrorCode e = execStatusCommand("RedisDB::setServiceInfo", "SET %s %s", sid, b.getPtr());
	if( e != ErrorOk ) {
		if( newRevKey ) {
			long long int r;
			ErrorCode e = getInt(r, "setServiceInfo", "DEL %s", keyRev.getPtr());
			if( e != ErrorOk ) {
				mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo: reverse key delete error. sid=%s code=%d", sid, e);
			}
		}
		if( curSvc )	free(curSvc);
		return e;
	}

	if( !newId && newRevKey ) {
		DBUtil::keyServiceRev(curSvc->mService.name, keyRev);
		long long int r;
		ErrorCode e = getInt(r, "setServiceInfo", "DEL %s", keyRev.getPtr());
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setServiceInfo: reverse key delete failed. sid=%s code=%d", sid, e);
		}
	}
	if( curSvc )	free(curSvc);

	if( sid_ret )
		strcpy(sid_ret, sid);
	return ErrorOk;
}

ErrorCode RedisDB::getServiceInfo(const char *id, ResultService **res)
{
	assert(id[0] == 'S');
	*res = NULL;
	pjson::json *js;
	ErrorCode e = getJson(&js, "getServiceInfo", "GET %s", id);
	if( e != ErrorOk ) {
		if( e == ErrorData )
			mSentinel->log(LOG_DEBUG, "RedisDB::getServiceInfo: data error.");
		return e;
	}

	pjson::value	*v	= js->getValue();
	if( v->vt != pjson::vt_object ) {
		delete js;
		return ErrorData;
	}

	pjson::value *vName			= js->get("name", v);
	pjson::value *vTitle		= js->get("title", v);
	pjson::value *vDescription	= js->get("desc", v);
	pjson::value *vVersion		= js->get("version", v);
	pjson::value *vAdminId		= js->get("adminId", v);
	pjson::value *vRef			= js->get("refURL", v);
	pjson::value *vScope		= js->get("scope", v);
	pjson::value *vDefGroup		= js->get("defaultGroup", v);
	pjson::value *vStorageBS	= js->get("storageBlockSize", v);
	pjson::value *vStorageBC	= js->get("storageBlockCount", v);
	pjson::value *vStorageCR	= js->get("storageCrypt", v);
	pjson::value *vAuthType		= js->get("authType", v);
	pjson::value *vAuthData		= js->get("authData", v);
	pjson::value *vStatus		= js->get("status", v);
	pjson::value *vCreateTime	= js->get("createTime", v);
	pjson::value *vUpdateTime	= js->get("updateTime", v);

	ServiceInfo info;
	memset(&info, 0, sizeof(info));
	info.id				= (char*)id;
	info.name			= (char*)((vName && vName->vt == pjson::vt_string) ? vName->vString : NULL);
	info.title			= (char*)((vTitle && vTitle->vt == pjson::vt_string) ? vTitle->vString : NULL);
	info.desc			= (char*)((vDescription && vDescription->vt == pjson::vt_string) ? vDescription->vString : NULL);
	info.version		= (char*)((vVersion && vVersion->vt == pjson::vt_string) ? vVersion->vString : NULL);
	info.adminId		= (char*)((vAdminId && vAdminId->vt == pjson::vt_string) ? vAdminId->vString : NULL);
	info.refURL			= (char*)((vRef && vRef->vt == pjson::vt_string) ? vRef->vString : NULL);
	info.scope			= (ServiceScope)((vScope && vScope->vt == pjson::vt_int) ? vScope->vInt : ScopePublic);
	info.defaultGroup	= (char*)((vDefGroup && vDefGroup->vt == pjson::vt_string) ? vDefGroup->vString : NULL);
	info.storageBlockSize = (size_t)((vStorageBS && vStorageBS->vt == pjson::vt_int) ? vStorageBS->vInt : 0);
	info.storageBlockCount= (size_t)((vStorageBC && vStorageBC->vt == pjson::vt_int) ? vStorageBC->vInt : 0);
	info.storageCrypt	= (CryptAlgorithm)((vStorageCR && vStorageCR->vt == pjson::vt_int) ? vStorageCR->vInt : 0);
	info.authType 		= (AuthType)((vAuthType && vAuthType->vt == pjson::vt_int) ? vAuthType->vInt : 0);
	info.authData		= (char*)((vAuthData && vAuthData->vt == pjson::vt_string) ? vAuthData->vString : NULL);
	info.status			= (ServiceStatus)((vStatus && vStatus->vt == pjson::vt_int) ? vStatus->vInt : 0);
	struct tm tm_create;
	if( vCreateTime && vCreateTime->vt == pjson::vt_string) {
		strptime(vCreateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_create);
		info.createTime	= timegm(&tm_create);
	} else {
		info.createTime	= 0;
	}
	struct tm tm_last;
	if( vUpdateTime && vUpdateTime->vt == pjson::vt_string) {
		strptime(vUpdateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_last);
		info.updateTime	= timegm(&tm_last);
	} else {
		info.updateTime	= 0;
	}

	size_t size = sizeof(ResultService)
				+ DBUtil::sizeofRBase()
				+ DBUtil::sizeofService(&info);
	char *ptr = (char*)malloc(size);
	if( ptr == NULL ) {
		delete js;
		return ErrorNoMemory;
	}
	ResultService *ret = (ResultService*)ptr;
	ptr	+= sizeof(ResultService);
	ptr	= DBUtil::writeService(&ret->mService, ptr, &info);
	delete js;
	*res = ret;
	return ErrorOk;
}

ErrorCode RedisDB::getServicePWList(const char *sid, size_t nFrom, size_t nCount, ResultList **pList)
{
	assert(sid[0] == 'S');
	StdBuffer key;
	DBUtil::keyServiceWritableList(sid, key);
	ErrorCode e = getSetList(key.getPtr(), nFrom, nCount, pList);
	return e;
}

ErrorCode RedisDB::isServiceWritable(const char *sid, const char *aid)
{
	assert(sid[0] == 'S');
	long long int nLen;
	StdBuffer key;
	DBUtil::keyServiceWritableList(sid, key);
	ErrorCode e = getInt(nLen, "isServiceWritable", "SISMEMBER %s %s", key.getPtr(), aid);
	if( e != ErrorOk ) {
		return e;
	}
	if( nLen == 0 ) {
		return ErrorNotFound;
	}
	return ErrorOk;
}

ErrorCode RedisDB::getServicePRList(const char *sid, size_t nFrom, size_t nCount, ResultList **pList)
{
	assert(sid[0] == 'S');
	StdBuffer key;
	DBUtil::keyServiceReadableList(sid, key);
	ErrorCode e = getSetList(key.getPtr(), nFrom, nCount, pList);
	return e;
}

ErrorCode RedisDB::isServiceReadable(const char *sid, const char *aid)
{
	assert(sid[0] == 'S');
	long long int nLen;
	StdBuffer key;
	DBUtil::keyServiceReadableList(sid, key);
	ErrorCode e = getInt(nLen, "isServiceReadable", "SISMEMBER %s %s", key.getPtr(), aid);
	if( e != ErrorOk ) {
		return e;
	}
	if( nLen == 0 ) {
		return ErrorNotFound;
	}
	return ErrorOk;
}

ErrorCode RedisDB::remServiceInfo(const char *id)
{
	// TODO: serviceの削除
	// 前提
	//  ステータスが削除済み
	//  デバイス内データが全削除（これは次回ログイン時に消えるので気にしなくて良いかも）
	// 処理
	//  関連データの削除
	return ErrorNotImpl;
}

ErrorCode RedisDB::setServiceRefuseList(const char *sid, const char *aid)
{
	assert(sid[0] == 'S');
	long long int nLen;
	StdBuffer key;
	DBUtil::keyServiceRefuseList(sid, key);
	ErrorCode e = getInt(nLen, "setServiceRefuseList", "SADD %s %s", key.getPtr(), aid);
	if( e != ErrorOk ) {
		return e;
	}
	if( nLen == 0 ) {
		return ErrorDuplicate;
	}
	return ErrorOk;
}

ErrorCode RedisDB::remServiceRefuseList(const char *sid, const char *aid)
{
	assert(sid[0] == 'S');
	long long int nLen;
	StdBuffer key;
	DBUtil::keyServiceRefuseList(sid, key);
	ErrorCode e = getInt(nLen, "setServiceRefuseList", "SREM %s %s", key.getPtr(), aid);
	if( e != ErrorOk ) {
		return e;
	}
	if( nLen == 0 ) {
		return ErrorNotFound;
	}
	return ErrorOk;
}

ErrorCode RedisDB::isServiceRefuse(const char *sid, const char *aid)
{
	assert(sid[0] == 'S');
	long long int nLen;
	StdBuffer key;
	DBUtil::keyServiceRefuseList(sid, key);
	ErrorCode e = getInt(nLen, "setServiceRefuseList", "SISMEMBER %s %s", key.getPtr(), aid);

	mSentinel->log(LOG_DEBUG, "RedisDB::isServiceRefuse code=%d nLen=%d", e, nLen);

	if( e != ErrorOk ) {
		return e;
	}
	if( nLen == 0 ) {
		return ErrorNotFound;
	}
	return ErrorOk;
}

ErrorCode RedisDB::getServiceCount(size_t &nRec)
{
	ErrorCode e = getKeyCount("S", nRec);
	return e;
}

ErrorCode	RedisDB::getServiceList(size_t nFrom, size_t nCount, ResultList **pList)
{
	ErrorCode e = getKeyList("S", nFrom, nCount, pList);
	return e;
}
/*
DBError RedisDB::getServiceSessionCount(const char *sid, size_t &nRec)
{
	mSentinel->log(LOG_DEBUG, "RedisDB::getServiceSessionCount: begin");
	char key[256];
	sprintf(key, "LC%s", sid);
	DBError e = getListCount(key, nRec);
	nRec = 0;
	mSentinel->log(LOG_DEBUG, "RedisDB::getServiceSessionCount: end");
	return e;
}

DBError RedisDB::getServiceSessionList(const char *sid, size_t nFrom, size_t nCount, ResultList **pList)
{
	mSentinel->log(LOG_DEBUG, "RedisDB::getServiceSessionList: begin.");
	char key[256];
	sprintf(key, "LC%s", sid);
	DBError e = getListList(key, nFrom, nCount, pList);
	mSentinel->log(LOG_DEBUG, "RedisDB::getServiceSessionList: end");
	return e;
}
*/
ErrorCode RedisDB::getServiceGroupCount(const char *sid, size_t &nRec)
{
	assert(sid[0] == 'S');
	StdBuffer key;
	DBUtil::keyServiceGroupList(sid, key);
	ErrorCode e = getListCount(key.getPtr(), nRec);
	return e;
}

ErrorCode RedisDB::getServiceGroupList(const char *sid, size_t nFrom, size_t nCount, ResultList **pList)
{
	assert(sid[0] == 'S');
	StdBuffer key;
	DBUtil::keyServiceGroupList(sid, key);
	ErrorCode e = getListList(key.getPtr(), nFrom, nCount, pList);
	return e;
}

ErrorCode RedisDB::getServiceResourceCount(const char *sid, size_t &nRec)
{
	assert(sid[0] == 'S');
	StdBuffer key;
	DBUtil::keyServiceResourceList(sid, key);
	ErrorCode e = getListCount(key.getPtr(), nRec);
	return e;
}

ErrorCode RedisDB::getServiceResourceList(const char *sid, size_t nFrom, size_t nCount, ResultList **pList)
{
	assert(sid[0] == 'S');
	StdBuffer key;
	DBUtil::keyServiceResourceList(sid, key);
	ErrorCode e = getSetList(key.getPtr(), nFrom, nCount, pList);
	return e;
}

/////////////////////////////////////////////////////////////////////////////
// group
ErrorCode RedisDB::getGroupId(const char *sid, const char *name, char *id_ret)
{
	assert(sid[0] == 'S');
	StdBuffer keyRev;
	DBUtil::keyGroupRev(sid, name, keyRev);
	redisReply *pRep = NULL;
	ErrorCode e = getStr(&pRep, "getServiceId", "GET %s", keyRev.getPtr());
	if( e != ErrorOk ) {
		if( pRep )
			freeReplyObject(pRep);
		return e;
	}
	strcpy(id_ret, pRep->str);
	freeReplyObject(pRep);
	return ErrorOk;
}

ErrorCode RedisDB::setGroup(GroupInfo *g, char *id_ret)
{
	if( g->name == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setGroup: name is not specfied. gid=%s", g->id);
		return ErrorData;
	}
	if( g->serviceId == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setGroup: serviceName is not specfied. gid=%s", g->id);
		return ErrorData;
	}

	bool newId = false;
	StdBuffer key;
	const char *gid;

	if( g->id == NULL ) {
		// service id
		long long int id_seq;
		ErrorCode e = getInt(id_seq, "RedisDB::setGroup", "INCR IDSEQ_G");
		if( e != ErrorOk ) {
			return e;
		}
		DBUtil::keyGroup(id_seq, key);
		gid	= key.getPtr();
		newId	= true;

	} else {
		gid = g->id;
	}

	time_t now = time(NULL);
	char _createTime[256];
	char _updateTime[256];
	if( newId ) {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));
	} else {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&g->createTime));
	}
	strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setGroup:1: json builder failed %d. gid=%s", jb.getError(), gid);
		return ErrorSystem;
	}

	if( !jb.addObjectProp("name", 4) ||
		!jb.valueString(g->name) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setGroup:2: json builder failed %d. gid=%s", jb.getError(), gid);
		return ErrorSystem;
	}
	if( !jb.addObjectProp("serviceId", 9) ||
		!jb.valueString(g->serviceId) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setGroup:3: json builder failed %d. gid=%s", jb.getError(), gid);
		return ErrorSystem;
	}
	if( !jb.addObjectProp("status", 6) ||
		!jb.valueInt(g->status) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setGroup:4: json builder failed %d. gid=%s", jb.getError(), gid);
		return ErrorSystem;
	}
	if( g->adminId ) {
		if( !jb.addObjectProp("adminId", 7) ||
			!jb.valueString(g->adminId) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setGroup:5: json builder failed %d. gid=%s", jb.getError(), gid);
			return ErrorSystem;
		}
	}
	if( g->description ) {
		if( !jb.addObjectProp("description", 11) ||
			!jb.valueString(g->description) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setGroup:6: json builder failed %d. gid=%s", jb.getError(), gid);
			return ErrorSystem;
		}
	}
	if( !jb.addObjectProp("createTime", 10) ||
		!jb.valueString(_createTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setGroup:7: json builder failed %d. gid=%s", jb.getError(), gid);
		return ErrorSystem;
	}
	if( !jb.addObjectProp("updateTime", 10) ||
		!jb.valueString(_updateTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setGroup:8: json builder failed %d. gid=%s", jb.getError(), gid);
		return ErrorSystem;
	}
	if( !jb.endObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setGroup:9: json builder failed %d. gid=%s", jb.getError(), gid);
		return ErrorSystem;
	}

	pjson::json *js = jb.detouch();
	StdBuffer b;
	if( !js->getText(&b,false) ) {
		delete js;
		mSentinel->log(LOG_ERR, "RedisDB::setGroup:10: json builder failed %d. gid=%s", jb.getError(), gid);
		return ErrorSystem;
	}
	delete js;

	StdBuffer keyRevNew;
	StdBuffer keyRevOld;
	bool newRevKey = false;
	bool oldRevKey = false;

	if( newId ) {
		newRevKey = true;
	} else {
		ResultGroup *curRes = NULL;
		ErrorCode e = getGroup(gid, &curRes);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setGroup: group load error. gid=%s code=%d",  gid, e);
			return e;
		}
		if( strcmp(curRes->mGroup.name, g->name) != 0 ) {
			newRevKey = true;
			oldRevKey = true;
			DBUtil::keyGroupRev(g->serviceId, curRes->mGroup.name, keyRevOld);
		}
		free(curRes);
	}

	if( newRevKey ) {
		DBUtil::keyGroupRev(g->serviceId, g->name, keyRevNew);
		long long int r;
		ErrorCode e = getInt(r, "RedisDB::setGroup", "SETNX %s %s", keyRevNew.getPtr(), gid);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setGroup: reverse key set failed. gid=%s", gid);
			return e;
		}
		if( r != 1 ) {
			mSentinel->log(LOG_ERR, "RedisDB::setGroup: reverse key duplication. gid=%s", gid);
			return ErrorDuplicate;
		}
	}
	ErrorCode e = execStatusCommand("RedisDB::setGroup", "SET %s %s", gid, b.getPtr());
	if( e != ErrorOk ) {
		if( newRevKey ) {
			long long int nDummy;
			getInt(nDummy, "RedisDB::setGroup", "DEL %s", keyRevNew.getPtr());
		}
		return e;
	}

	if( id_ret )
		strcpy(id_ret, gid);

	if( oldRevKey ) {
		long long int nDummy;
		getInt(nDummy, "RedisDB::setGroup", "DEL %s", keyRevOld.getPtr());
	}

	// service group list
	if( newId ){
		StdBuffer keySG;
		DBUtil::keyServiceGroupList(g->serviceId, keySG);
		long long int nDummy;
		ErrorCode e;
		e = getInt(nDummy, "RedisDB::setGroup", "LREM %s 0 %s", keySG.getPtr(), gid);
		e = getInt(nDummy, "RedisDB::setGroup", "RPUSH %s %s", keySG.getPtr(), gid);
		if( e != ErrorOk ) {
			getInt(nDummy, "RedisDB::setGroup", "DEL %s", keyRevNew.getPtr());
			getInt(nDummy, "RedisDB::setGroup", "DEL %s", gid);
			return e;
		}
	}
	return ErrorOk;
}

ErrorCode RedisDB::getGroup(const char *gid, ResultGroup **res)
{
	assert(gid[0] == 'G');
	*res = NULL;
	pjson::json *js;
	ErrorCode e = getJson(&js, "RedisDB::getGroup", "GET %s", gid);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_DEBUG, "RedisDB::getGroup: data error.");
		return e;
	}

	pjson::value	*v	= js->getValue();
	if( v->vt != pjson::vt_object ) {
		delete js;
		return ErrorData;
	}
	pjson::value *vName		= js->get("name", v);
	pjson::value *vServiceId= js->get("serviceId", v);
	pjson::value *vStatus	= js->get("status", v);
	pjson::value *vAdminId	= js->get("adminId", v);
	pjson::value *vDesc		= js->get("description", v);
	pjson::value *vCreateTime	= js->get("createTime", v);
	pjson::value *vUpdateTime	= js->get("updateTime", v);

	GroupInfo info;
	memset(&info, 0, sizeof(info));
	info.id			= (char*)gid;
	info.serviceId	= (char*)((vServiceId && vServiceId->vt == pjson::vt_string) ? vServiceId->vString : NULL);
	info.name		= (char*)((vName && vName->vt == pjson::vt_string) ? vName->vString : NULL);
	info.status		= (GroupStatus)((vStatus && vStatus->vt == pjson::vt_int) ? vStatus->vInt : 0);
	info.adminId	= (char*)((vAdminId && vAdminId->vt == pjson::vt_string) ? vAdminId->vString : NULL);
	info.description= (char*)((vDesc && vDesc->vt == pjson::vt_string) ? vDesc->vString : NULL);
	struct tm tm_create;
	if( vCreateTime && vCreateTime->vt == pjson::vt_string) {
		strptime(vCreateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_create);
		info.createTime	= timegm(&tm_create);
	} else {
		info.createTime	= 0;
	}
	struct tm tm_last;
	if( vUpdateTime && vUpdateTime->vt == pjson::vt_string) {
		strptime(vUpdateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_last);
		info.updateTime	= timegm(&tm_last);
	} else {
		info.updateTime	= 0;
	}

	size_t size = sizeof(ResultGroup)
				+ DBUtil::sizeofRBase()
				+ DBUtil::sizeofGroupInfo(&info);
	char *ptr = (char*)malloc(size);
	ResultGroup *ret = (ResultGroup*)ptr;
	ptr	+= sizeof(ResultGroup);
	ptr	= DBUtil::writeGroupInfo(&ret->mGroup, ptr, &info);
	*res	= ret;

	return ErrorOk;
}

ErrorCode RedisDB::remGroup(const char *gid)
{
	assert(gid[0] == 'G');
	// TODO: アカウントから切り離されていること
	// TODO: サービスから切り離されていること
	// TODO: 停止状態であること
	long long int rem;

	ResultGroup *grp = NULL;
	ErrorCode e = getGroup(gid, &grp);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_ERR, "remGroup: group info load error. group=%s. %s.", gid, codeToMessgae(e));
	}

	StdBuffer keyRev;
	if( grp ) {
		DBUtil::keyGroupRev(grp->mGroup.serviceId, grp->mGroup.name, keyRev);
		// delete reverse key
		e = getInt(rem, "remAcocuntInfo", "DEL %s", keyRev.getPtr());
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::remServiceGroup: redis error. cmd=\"DEL %s\" code=%d msg=%s", keyRev.getPtr(), e, mConn->errstr);
		}
	}

	// delete service group list
	StdBuffer key;
	DBUtil::keyServiceGroupList(grp->mGroup.serviceId, key);
	mSentinel->log(LOG_DEBUG, "RedisDB::remServiceGroup: remove service group list. key=%s", key.getPtr());
	e = getInt(rem, "remAcocuntInfo", "LREM %s 0 %s", key.getPtr(), gid);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::remServiceGroup: redis error. cmd=\"LREM %s 0 %s\" code=%d msg=%s", key.getPtr(), gid, e, mConn->errstr);
	}

	// TODO: delete share control list

	// TODO: delete group member list

	// delete group info
	mSentinel->log(LOG_DEBUG, "RedisDB::remServiceGroup. remove device key. key=%s", gid);
	e = getInt(rem, "remAcocuntInfo", "DEL %s", gid);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::remServiceGroup: redis error. cmd=\"DEL %s\" code=%d msg=%s", gid, e, mConn->errstr);
	}
	return ErrorOk;
}

ErrorCode RedisDB::isGroupAccount(const char *gid, const char *aid)
{
	assert(gid[0] == 'G');
	assert(aid[0] == 'A');
	long long int num;
	StdBuffer key;
	DBUtil::keyGroupMemberList(gid, key);

	ErrorCode e = getInt(num, "isGroupAccount", "SISMEMBER %s %s", key.getPtr(), aid);
	if( e != ErrorOk ) {
		return e;
	}
	if( num == 0 ) {
		return ErrorNotFound;
	}
	return ErrorOk;
}

ErrorCode	RedisDB::joinGroupAccount(const char *gid, const char *aid)
{
	assert(gid[0] == 'G');
	assert(aid[0] == 'A');
	ErrorCode e;

	StdBuffer keyGM;
	DBUtil::keyGroupMemberList(gid, keyGM);

	StdBuffer keyAG;
	DBUtil::keyAccountGroupList(aid, keyAG);

	long long int gmNum = 0;
	e = getInt(gmNum, "joinGroupAccount", "SADD %s %s", keyGM.getPtr(), aid);
	if( e != ErrorOk ) {
		return e;
	}

	long long int agNum = 0;
	e = getInt(agNum, "joinGroupAccount", "SADD %s %s", keyAG.getPtr(), gid);
	if( e != ErrorOk ) {
		if( gmNum == 1 ) {
			getInt(gmNum, "joinGroupAccount", "SREM %s %s", keyGM.getPtr(), aid);
		}
		return e;
	}

	return ErrorOk;
}

ErrorCode	RedisDB::leaveGroupAccount(const char *gid, const char *aid)
{
	assert(gid[0] == 'G');
	assert(aid[0] == 'A');
	ErrorCode e;

	StdBuffer keyGM;
	DBUtil::keyGroupMemberList(gid, keyGM);

	StdBuffer keyAG;
	DBUtil::keyAccountGroupList(aid, keyAG);

	long long int numGM = 0;
	e = getInt(numGM, "joinGroupAccount", "SREM %s %s", keyGM.getPtr(), aid);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::leaveGroupAccount: redis error. cmd=\"SREM %s %s\" code=%d msg=%s", keyGM.getPtr(), aid, e, mConn->errstr);
	}

	long long int numAG = 0;
	e = getInt(numAG, "joinGroupAccount", "SREM %s %s", keyAG.getPtr(), gid);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::leaveGroupAccount: redis error. cmd=\"SREM %s %s\" code=%d msg=%s", keyAG.getPtr(), gid, e, mConn->errstr);
	}

	return ErrorOk;
}

ErrorCode	RedisDB::getGroupAccountCount(const char *gid, size_t &nRec)
{
	assert(gid[0] == 'G');
	StdBuffer keyGM;
	DBUtil::keyGroupMemberList(gid, keyGM);
	ErrorCode e = getSetCount(keyGM.getPtr(), nRec);
	return e;
}

ErrorCode	RedisDB::getGroupAccountList(const char *gid, size_t nFrom, size_t nCount, ResultList **pList)
{
	assert(gid[0] == 'G');
	StdBuffer keyGM;
	DBUtil::keyGroupMemberList(gid, keyGM);
	*pList	= NULL;
	ErrorCode e = getSetList(keyGM.getPtr(), nFrom, nCount, pList);
	return e;
}

ErrorCode RedisDB::getGroupShareCount(const char *gid, size_t &nRec)
{
	assert(gid[0] == 'G');
	StdBuffer keyGS;
	DBUtil::keyGroupShareList(gid, keyGS);
	ErrorCode e = getListCount(keyGS.getPtr(), nRec);
	return e;
}

ErrorCode RedisDB::getGroupShareList(const char *gid, size_t nFrom, size_t nCount, ResultList **pList)
{
	assert(gid[0] == 'G');
	StdBuffer keyGS;
	DBUtil::keyGroupShareList(gid, keyGS);
	ErrorCode e = getListList(keyGS.getPtr(), nFrom, nCount, pList);
	return e;
}

/////////////////////////////////////////////////////////////////////////////
// resource
ErrorCode RedisDB::getResourceId(const char *sid, const char *name, char *id_ret)
{
	assert(sid[0] == 'S');
	StdBuffer keyRev;
	DBUtil::keyResourceRev(sid, name, keyRev);
	redisReply *pRep = NULL;
	ErrorCode e = getStr(&pRep, "getServiceId", "GET %s", keyRev.getPtr());
	if( e != ErrorOk ) {
		if( pRep )
			freeReplyObject(pRep);
		return e;
	}
	strcpy(id_ret, pRep->str);
	freeReplyObject(pRep);
	return ErrorOk;
}

ErrorCode RedisDB::setResourceInfo(ResourceInfo *rsc, char *id_ret)
{
	bool newId = false;
	StdBuffer key;
	const char *rid;

	if( rsc->serviceId == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo: serviceId is not specfied. resource=%s", rsc->name);
		return ErrorData;
	}
	if( rsc->name == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo: name is not specfied. resource=%s", rsc->refURL);
		return ErrorData;
	}
	/*
	if( res->refURL == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo: refURL is not specfied. resource=%s", res->name);
		return ErrorData;
	}
	if( res->refTime == 0 ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo: refURL is not specfied. resource=%s", res->name);
		return ErrorData;
	}
	*/

	if( rsc->id == NULL ) {
		// resource id
		long long int rid_seq;
		ErrorCode e = getInt(rid_seq, "setResourceInfo", "INCR IDSEQ_R");
		if( e != ErrorOk ) {
			return e;
		}
		DBUtil::keyResource(rid_seq, key);
		rid		= key.getPtr();
		newId	= true;
	} else {
		rid		= rsc->id;
	}

	time_t now = time(NULL);
	char _createTime[256];
	char _updateTime[256];
	if( newId ) {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));
	} else {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&rsc->createTime));
	}
	strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:1: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
		return ErrorSystem;
	}

	if( !jb.addObjectProp("name", 4) ||
		!jb.valueString(rsc->name) ||
		!jb.addObjectProp("serviceId", 9) ||
		!jb.valueString(rsc->serviceId) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:2: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
		return ErrorSystem;
	}
	if( rsc->tag ) {
		if( !jb.addObjectProp("tag", 3) ||
		    !jb.valueString(rsc->tag) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:3: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
			return ErrorSystem;
		}
	}
	if( rsc->refURL ) {
		if( !jb.addObjectProp("refURL", 6) ||
			!jb.valueString(rsc->refURL) ) {
			mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:4: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
			return ErrorSystem;
		}
	}
	if( rsc->refTime != 0 ) {
		char _refTime[256];
		strftime(_refTime, sizeof(_refTime), "%Y/%m/%d-%H:%M:%S", gmtime(&rsc->refTime));
		if( !jb.addObjectProp("refTime", 7) ||
		    !jb.valueString(_refTime)) {
			mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:5: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
			return ErrorSystem;
		}
	}

	if( !jb.addObjectProp("status", 6) ||
		!jb.valueInt(rsc->status) ||
		!jb.addObjectProp("revision", 8) ||
		!jb.valueInt(rsc->revision) ||
		!jb.addObjectProp("shareCount", 10) ||
		!jb.valueInt(rsc->shareCount) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:6: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
		return ErrorSystem;
	}

	if( rsc->gadgetCount > 0 ) {
		if( !jb.addObjectProp("gadgets", 7) ||
			!jb.beginArray() ) {
			mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:9: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
			return ErrorSystem;
		}
		size_t i;
		for( i = 0; i < rsc->gadgetCount; i++) {
			if( !jb.addArrayContent() ||
				!jb.beginObject() ||
				!jb.addObjectProp("id", 2) ||
				!jb.valueString(rsc->gadgetArray[i].id) ||
				!jb.addObjectProp("name", 4) ||
				!jb.valueString(rsc->gadgetArray[i].name))  {
				mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:10: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
				return ErrorSystem;
			}
			if( rsc->gadgetArray[i].tag ) {
				if( !jb.addObjectProp("tag", 3) ||
					!jb.valueString(rsc->gadgetArray[i].tag))  {
					mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:11: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
					return ErrorSystem;
				}
			}
			if( rsc->gadgetArray[i].refTime != 0) {
				char _refTime[256];
				strftime(_refTime, sizeof(_refTime), "%Y/%m/%d-%H:%M:%S", gmtime(&rsc->gadgetArray[i].refTime));
				if( !jb.addObjectProp("refTime", 7) ||
					!jb.valueString(_refTime)) {
					mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:12: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
					return ErrorSystem;
				}
			}
			if( !jb.addObjectProp("status", 6) ||
				!jb.valueInt(rsc->gadgetArray[i].status) ||
				!jb.addObjectProp("revision", 8) ||
				!jb.valueInt(rsc->gadgetArray[i].revision)  ) {
				mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:13: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
				return ErrorSystem;
			}
			if( !jb.endObject() ) {
				mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:14: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
				return ErrorSystem;
			}
		}
		if( !jb.endArray() ) {
			mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:15: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
			return ErrorSystem;
		}
	}
	if( !jb.addObjectProp("createTime", 10) ||
		!jb.valueString(_createTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:7: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
		return ErrorSystem;
	}
	if( !jb.addObjectProp("updateTime", 10) ||
		!jb.valueString(_updateTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:8: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
		return ErrorSystem;
	}

	if( !jb.endObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:16: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
		return ErrorSystem;
	}

	pjson::json *js = jb.detouch();
	StdBuffer b;
	if( !js->getText(&b,false) ) {
		delete js;
		mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo:17: json builder failed %d. rid=%s code=%d", jb.getError(), rsc->name, jb.getError());
		return ErrorSystem;
	}
	delete js;

	StdBuffer keyRevNew;
	StdBuffer keyRevOld;
	bool newRevKey = false;
	bool oldRevKey = false;

	if( newId ) {
		newRevKey = true;
	} else {
		ResultResource *curRes = NULL;
		ErrorCode e = getResourceInfo(rid, &curRes);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo: resource error. rid=%s code=%d",  rid, e);
			return e;
		}
		if( strcmp(curRes->mResource.name, rsc->name) != 0 ) {
			newRevKey = true;
			oldRevKey = true;
			DBUtil::keyResourceRev(rsc->serviceId, curRes->mResource.name, keyRevOld);
		}
		free(curRes);
	}

	if( newRevKey ) {
		DBUtil::keyResourceRev(rsc->serviceId, rsc->name, keyRevNew);
		long long int r;
		ErrorCode e = getInt(r, "setResourceInfo", "SETNX %s %s", keyRevNew.getPtr(), rid);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo: reverse key set failed. rid=%s", rid);
			return e;
		}
		if( r != 1 ) {
			mSentinel->log(LOG_ERR, "RedisDB::setResourceInfo: reverse key duplication. rid=%s", rid);
			return ErrorDuplicate;
		}
	}

	ErrorCode e  = execStatusCommand("setResourceInfo", "SET %s %s", rid, b.getPtr());
	if( e != ErrorOk ) {
		if( newRevKey ) {
			long long int nDummy;
			getInt(nDummy, "setResourceInfo", "DEL %s", keyRevNew.getPtr());
		}
		return e;
	}

	// add ServiceResourceList
	{
		StdBuffer keySR;
		DBUtil::keyServiceResourceList(rsc->serviceId, keySR);
		long long int nDummy;
		ErrorCode e = getInt(nDummy, "setResourceInfo", "SADD %s %s", keySR.getPtr(), rid);
		if( e != ErrorOk ) {
			if( newRevKey ) {
				getInt(nDummy, "setResourceInfo", "DEL %s", keyRevNew.getPtr());
			}
			if( newId ) {
				getInt(nDummy, "setResourceInfo", "DEL %s", rid);
			}
			return e;
		}
	}

	if( oldRevKey ) {
		long long int nDummy;
		getInt(nDummy, "setResourceInfo", "DEL %s", keyRevOld.getPtr());
	}
	if( id_ret )
		strcpy(id_ret, rid);

	return ErrorOk;
}

ErrorCode RedisDB::getResourceInfo(const char *id, ResultResource **res)
{
	assert(id[0] == 'R');
	*res = NULL;
	pjson::json *js;
	ErrorCode e = getJson(&js, "getResourceInfo", "GET %s", id);
	if( e != ErrorOk ) {
		if( e == ErrorData )
			mSentinel->log(LOG_DEBUG, "RedisDB::getResourceInfo: data error. rid=%s", id);
		return e;
	}

	pjson::value	*v	= js->getValue();
	if( v->vt != pjson::vt_object ) {
		delete js;
		mSentinel->log(LOG_DEBUG, "RedisDB::getResourceInfo: data error.");
		return ErrorData;
	}

	pjson::value *vName			= js->get("name", v);
	pjson::value *vTag			= js->get("tag", v);
	pjson::value *vServiceId	= js->get("serviceId", v);
	pjson::value *vURL			= js->get("refURL", v);
	pjson::value *vRefTime		= js->get("refTime", v);
	pjson::value *vRevision		= js->get("revision", v);
	pjson::value *vShare		= js->get("shareCount", v);
	pjson::value *vStatus		= js->get("status", v);
	pjson::value *vGadget		= js->get("gadgets", v);
	pjson::value *vCreateTime	= js->get("createTime", v);
	pjson::value *vUpdateTime	= js->get("updateTime", v);

	ResourceInfo info;
	memset(&info, 0, sizeof(info));
	info.id	= (char*)id;
	info.name		= (char*)((vName && vName->vt == pjson::vt_string) ? vName->vString : NULL);
	info.tag		= (char*)((vTag && vTag->vt == pjson::vt_string) ? vTag->vString : NULL);
	info.serviceId	= (char*)((vServiceId && vServiceId->vt == pjson::vt_string) ? vServiceId->vString : NULL);
	info.refURL		= (char*)((vURL && vURL->vt == pjson::vt_string) ? vURL->vString : NULL);
	info.revision	= (revision_t)(vRevision && vRevision->vt == pjson::vt_int) ? vRevision->vInt : 0;
	info.shareCount	= (vShare && vShare->vt == pjson::vt_int ) ? vShare->vInt : 0;
	info.status		= (ResourceStatus)((vStatus && vStatus->vt == pjson::vt_int) ? vStatus->vInt: NULL);
	struct tm tm_ref;
	if( vRefTime && vRefTime->vt == pjson::vt_string) {
		strptime(vRefTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_ref);
		info.refTime	= timegm(&tm_ref);
	}
	struct tm tm_create;
	if( vCreateTime && vCreateTime->vt == pjson::vt_string) {
		strptime(vCreateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_create);
		info.createTime	= timegm(&tm_create);
	}
	struct tm tm_last;
	if( vUpdateTime && vUpdateTime->vt == pjson::vt_string) {
		strptime(vUpdateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_last);
		info.updateTime	= timegm(&tm_last);
	}

	size_t nGadget = 0;
	GadgetInfo *pGadget = NULL;
	if( vGadget && vGadget->vt == pjson::vt_array ) {
		nGadget	= pjson::getArraySize(vGadget);
		pGadget	= (GadgetInfo*)malloc(sizeof(GadgetInfo)*nGadget);
		if( pGadget == NULL ) {
			delete js;
			return ErrorNoMemory;
		}
		pjson::value *arr;
		for( size_t i = 0; i < nGadget; i++ ) {
			arr = pjson::getArrayContent(vGadget, i);
			pjson::value *vName		= js->get("name", arr);
			pjson::value *vTag		= js->get("tag", arr);
			pjson::value *vRefTime	= js->get("refTime", arr);
			pjson::value *vRevision	= js->get("revision", arr);
			pjson::value *vStatus	= js->get("status", arr);
			memset(&pGadget[i], 0, sizeof(GadgetInfo));
			pGadget[i].id		= (char*)id;
			pGadget[i].name		= (char*)((vName && vName->vt == pjson::vt_string) ? vName->vString : NULL);
			pGadget[i].tag		= (char*)((vTag && vTag->vt == pjson::vt_string) ? vTag->vString : NULL);
			pGadget[i].revision	= (revision_t)(vRevision && vRevision->vt == pjson::vt_int) ? vRevision->vInt : 0;
			pGadget[i].status	= (ResourceStatus)((vStatus && vStatus->vt == pjson::vt_int) ? vStatus->vInt: NULL);
			struct tm _refTime;
			if( vRefTime && vRefTime->vt == pjson::vt_string) {
				strptime(vRefTime->vString, "%Y/%m/%d-%H:%M:%S", &_refTime);
				pGadget[i].refTime	= timegm(&_refTime);
			} else {
				pGadget[i].refTime	= 0;
			}
		}
	}

	size_t size = sizeof(ResultResource)
				+ DBUtil::sizeofRBase()
				+ DBUtil::sizeofResource(&info, pGadget, nGadget);
	char *ptr = (char*)malloc(size);
	if( ptr == NULL ) {
		if( pGadget) free(pGadget);
		delete js;
		return ErrorNoMemory;
	}
	ResultResource *ret = (ResultResource*)ptr;
	ptr	+= sizeof(ResultResource);
	ptr	= DBUtil::writeResource(&ret->mResource, ptr, &info, pGadget, nGadget);

	if( pGadget) free(pGadget);
	delete js;

	*res = ret;
	return ErrorOk;
}

ErrorCode RedisDB::remResourceInfo(const char *rid)
{
	assert(rid[0] == 'R');
	// TODO: 共有解除されていること
	// TODO: 停止状態であること

	ResultResource *rsc = NULL;
	ErrorCode e = getResourceInfo(rid, &rsc);
	if( e != ErrorOk ) {
		return e;
	}

	StdBuffer key;
	long long int rem;

	// delete ResourceKeyReverse
	StdBuffer keyRev;
	DBUtil::keyResourceRev(rsc->mResource.serviceId, rsc->mResource.name, keyRev);
	e = getInt(rem, "remAcocuntInfo", "DEL %s", keyRev.getPtr());
	if( e != ErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::remResourceInfo: redis error. cmd=\"DEL %s\" code=%d msg=%s", keyRev.getPtr(), e, mConn->errstr);
	}

	// delete ServiceResourceList
	DBUtil::keyServiceResourceList(rsc->mResource.serviceId, key);
	e = getInt(rem, "remAcocuntInfo", "LREM %s 0 %s", key.getPtr(), rid);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::remResourceInfo: redis error. cmd=\"LREM %s 0 %s\" code=%d msg=%s", key.getPtr(), e, mConn->errstr);
	}

	// delete ResourceInfo
	e = getInt(rem, "remAcocuntInfo", "DEL %s", rid);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::remResourceInfo: redis error. cmd=\"DEL %s\" code=%d msg=%s", rid, e, mConn->errstr);
	}

	return ErrorOk;
}
/*
ErrorCode RedisDB::getGadgetId(const char *rid, const char *name, char *id_ret)
{
	StdBuffer keyRev;
	keyGadgetRev(rid, name, keyRev);
	redisReply *pRep = NULL;
	ErrorCode e = getStr(&pRep, "getGadgetId", "GET %s", keyRev.getPtr());
	if( e != ErrorOk ) {
		if( pRep )
			freeReplyObject(pRep);
		return e;
	}
	strcpy(id_ret, pRep->str);
	freeReplyObject(pRep);
	return ErrorOk;
}
*/

ErrorCode	RedisDB::getResourceShareCount(const char *rid, size_t &nRec)
{
	return ErrorNotImpl;
}

ErrorCode	RedisDB::getResourceShareList(const char *rid, size_t nFrom, size_t nCount, ResultList **pList)
{
	return ErrorNotImpl;
}


/////////////////////////////////////////////////////////////////////////////
// share
ErrorCode RedisDB::getShareInfo(const char *mid, ResultShare **res)
{
	assert(mid[0] == 'M');

	pjson::json *js;
	ErrorCode e = getJson(&js, "getShareInfo", "GET %s", mid);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_DEBUG, "RedisDB::getShareInfo: %s key=%s", codeToMessgae(e), mid);
		if( e == ErrorData )
			mSentinel->log(LOG_DEBUG, "RedisDB::getShareInfo: data error.");
		return e;
	}

	pjson::value	*v	= js->getValue();
	if( v->vt != pjson::vt_object ) {
		delete js;
		return ErrorData;
	}

	pjson::value *vServiceId	= js->get("serviceId", v);
	pjson::value *vResourceId	= js->get("resourceId", v);
	pjson::value *vIndex		= js->get("index", v);
	pjson::value *vDataMode		= js->get("dataMode", v);
	pjson::value *vAccessMode	= js->get("accessMode", v);
	pjson::value *vExpireTime	= js->get("expireTime", v);
	pjson::value *vFromTime		= js->get("fromTime", v);
	pjson::value *vEndTime		= js->get("endTime", v);
	pjson::value *vGeoLocation	= js->get("geoLocation", v);
	pjson::value *vGpsGrace		= js->get("gpsGrace", v);
	pjson::value *vOfflineGrace	= js->get("offlineGrace", v);
	pjson::value *vGroup		= js->get("group", v);
	pjson::value *vCreateTime	= js->get("createTime", v);

	ShareInfo info;
	memset(&info, 0, sizeof(info));
	info.id				= (char*)mid;
	info.resourceId		= (char*)((vResourceId && vResourceId->vt == pjson::vt_string) ? vResourceId->vString : NULL);
	info.index			= (vIndex && vIndex->vt == pjson::vt_int) ? vIndex->vInt : 0;
	info.serviceId		= (char*)((vServiceId && vServiceId->vt == pjson::vt_string) ? vServiceId->vString : NULL);
	info.dataMode		= (ShareDataMode)((vDataMode && vDataMode->vt == pjson::vt_int) ? vDataMode->vInt : ShareDataFull);
	info.accessMode		= (ShareAccessMode)((vAccessMode && vAccessMode->vt == pjson::vt_int) ? vAccessMode->vInt : ShareAccessRead);
	if( vExpireTime && vExpireTime->vt == pjson::vt_string) {
		struct tm tmv;
		char *r = strptime(vExpireTime->vString, "%Y/%m/%d-%H:%M:%S", &tmv);
		if( r == NULL || *r != 0 ) {
			return ErrorData;
		}
		info.expireTime		= timegm(&tmv);
	}
	if( vFromTime && vFromTime->vt == pjson::vt_string) {
		struct tm tmv;
		strptime(vFromTime->vString, "%Y/%m/%d-%H:%M:%S", &tmv);
		char *r = strptime(vCreateTime->vString, "%Y/%m/%d-%H:%M:%S", &tmv);
		if( r == NULL || *r != 0 ) {
			return ErrorData;
		}
		info.fromTime		= timegm(&tmv);
	}
	if( vEndTime && vEndTime->vt == pjson::vt_string) {
		struct tm tmv;
		char *r = strptime(vEndTime->vString, "%Y/%m/%d-%H:%M:%S", &tmv);
		if( r == NULL || *r != 0 ) {
			return ErrorData;
		}
		info.endTime		= timegm(&tmv);
	}
	if( vGeoLocation && vGeoLocation->vt == pjson::vt_array ) {
		size_t n = pjson::getArraySize(vGeoLocation);
		if( n > 2 ) {
			return ErrorData;
		}
		info.geoLocationCount	= n;
		size_t i;
		for( i = 0; i < n; i++ ) {
			pjson::value *arr = pjson::getArrayContent(vGeoLocation, i);
			pjson::value *vDis	= js->get("dis", arr);
			pjson::value *vLat	= js->get("lat", arr);
			pjson::value *vLon	= js->get("lon", arr);
			if( vDis && (vDis->vt == pjson::vt_int || vDis->vt == pjson::vt_number) )
				info.geoLocation[i].dis	= vDis->vt == pjson::vt_int ? vDis->vInt : *vDis->vNumber;
			else
				info.geoLocation[i].dis	= 0;
			if( vLat && (vLat->vt == pjson::vt_int || vLat->vt == pjson::vt_number) )
				info.geoLocation[i].lat	= vLat->vt == pjson::vt_int ? vLat->vInt : *vLat->vNumber;
			else
				info.geoLocation[i].lat	= 0;
			if( vLon && (vLon->vt == pjson::vt_int || vLon->vt == pjson::vt_number) )
				info.geoLocation[i].lon	= vLon->vt == pjson::vt_int ? vLon->vInt : *vLon->vNumber;
			else
				info.geoLocation[i].lon	= 0;
		}
	}
	info.gpsGrace		= (int)(vGpsGrace ? vGpsGrace->vInt : 0);
	info.offlineGrace	= (int)(vOfflineGrace ? vOfflineGrace->vInt : 0);
	if( vGroup && vGroup->vt == pjson::vt_array ) {
		size_t n = pjson::getArraySize(vGroup);
		if( n > 0 ) {
			info.groupCount	= n;
			info.groupArray	= (QueryKeyType*)malloc(sizeof(QueryKeyType)*n);
			if( info.groupArray == NULL )
				return ErrorNoMemory;
			size_t i;
			for( i = 0; i < n; i++ ) {
				pjson::value *arr = pjson::getArrayContent(vGroup, i);
				pjson::value *vGID = js->get("gid", arr);
				pjson::value *vGName = js->get("groupName", arr);
				info.groupArray[i].mId		= (char*)((vGID && vGID->vt == pjson::vt_string) ? vGID->vString : NULL);
				info.groupArray[i].mName	= (char*)((vGName && vGName->vt == pjson::vt_string) ? vGName->vString : NULL);
			}
		}
	}
	if( vCreateTime && vCreateTime->vt == pjson::vt_string) {
		struct tm tmv;
		strptime(vCreateTime->vString, "%Y/%m/%d-%H:%M:%S", &tmv);
		char *r = strptime(vCreateTime->vString, "%Y/%m/%d-%H:%M:%S", &tmv);
		if( r == NULL || *r != 0 ) {
			if( info.groupArray	) free(info.groupArray);
			return ErrorData;
		}
		info.createTime		= timegm(&tmv);
	}


	size_t size = sizeof(ResultShare)
				+ DBUtil::sizeofRBase()
				+ DBUtil::sizeofShare(&info);
	char *ptr = (char*)malloc(size);
	if( ptr == NULL ) {
		delete js;
		if( info.groupArray	) free(info.groupArray);
		return ErrorNoMemory;
	}
	ResultShare *ret = (ResultShare*)ptr;
	ptr	+= sizeof(ResultShare);
	ptr	= DBUtil::writeShare(&ret->mShare, ptr, &info);
	if( info.groupArray	) free(info.groupArray);

	*res	= ret;
	return ErrorOk;
}

static int qsCompareGid(const void *a, const void *b)
{
	char **ap = (char**)a;
	char **bp = (char**)b;
	return strcmp(*ap, *bp);
}

ErrorCode	RedisDB::setShareInfo(ShareInfo *share, char *id_ret)
{
	StdBuffer keyShare;
	DBUtil::keyShare(share->resourceId, share->index, keyShare);

	time_t now = time(NULL);
	char _createTime[256];
	strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));

	if( share->serviceId == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setShareInfo: serviceId is not specfied.");
		return ErrorData;
	}
	if( share->resourceId == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setShareInfo: resourceId is not specfied.");
		return ErrorData;
	}

	ResultShare *cur = NULL;
	ErrorCode e = getShareInfo(keyShare.getPtr(), &cur);
	if( e != ErrorOk ) {
		cur = NULL;
	}

	e	= ErrorOk;
	char *newGidPtr = NULL;
	char **newGidArr = NULL;
	if( share->groupCount > 0 ) {
		newGidPtr = (char*)malloc((SST_GROUPID_LEN+1+sizeof(char*))*share->groupCount);
		if( newGidPtr == NULL ) {
			mSentinel->log(LOG_ERR, "RedisDB::setShareInfo: group id array error. code=%d msg=%s sid=%s rid=&s groups=%d",
					ErrorNoMemory, codeToMessgae(ErrorNoMemory),
					share->serviceId, share->resourceId, share->groupCount);
			return ErrorNoMemory;
		}
		char *newGidCur = newGidPtr;
		newGidArr	= (char **)newGidCur;
		newGidCur	+= sizeof(char*) * share->groupCount;
		size_t i;
		for( i = 0; i < share->groupCount; i++) {
			newGidArr[i] = newGidCur;
			newGidCur += SST_GROUPID_LEN+1;
			char _gid[SST_GROUPID_LEN+1];
			const char *gid = NULL;
			QueryKeyType *gref = &share->groupArray[i];
			if( gref->mId ) {
				gid = gref->mId;
			} else if( gref->mName != NULL ) {
				e = getGroupId(share->serviceId, gref->mName, _gid);
				if( e != ErrorOk ) {
					mSentinel->log(LOG_ERR, "RedisDB::setShareInfo: getGroupId error. sid=%s groupName=%s", share->serviceId, gref->mName);
					free(newGidPtr);
					return e;
				}
				gid	= _gid;
			}
			if( gid == NULL || *gid == 0 ) {
				mSentinel->log(LOG_ERR, "RedisDB::setShareInfo: group id is not specified error. sid=%s rid=%s", share->serviceId, share->resourceId);
				free(newGidPtr);
				return ErrorData;
			}
			strcpy(newGidArr[i], gid);
		}
		qsort(newGidArr, share->groupCount, sizeof(char*), qsCompareGid);
	}

	char *curGidPtr = NULL;
	char **curGidArr = NULL;
	if( cur && cur->mShare.groupCount > 0 ) {
		curGidPtr = (char*)malloc((SST_GROUPID_LEN+1+sizeof(char*))*cur->mShare.groupCount);
		if( curGidPtr == NULL ) {
			mSentinel->log(LOG_ERR, "RedisDB::setShareInfo: group id array error. code=%d msg=%s sid=%s rid=&s groups=%d",
					ErrorNoMemory, codeToMessgae(ErrorNoMemory),
					share->serviceId, share->resourceId, cur->mShare.groupCount);
			return ErrorNoMemory;
		}
		char *curGidCur = curGidPtr;
		curGidArr	= (char **)curGidCur;
		curGidCur	+= sizeof(char*) * cur->mShare.groupCount;
		size_t i;
		for( i = 0; i < cur->mShare.groupCount; i++) {
			curGidArr[i] = curGidCur;
			curGidCur += SST_GROUPID_LEN+1;
			char _gid[SST_GROUPID_LEN+1];
			const char *gid = NULL;
			QueryKeyType *gref = &cur->mShare.groupArray[i];
			if( gref->mId ) {
				gid = gref->mId;
			} else if( gref->mName != NULL ) {
				e = getGroupId(share->serviceId, gref->mName, _gid);
				if( e != ErrorOk ) {
					mSentinel->log(LOG_ERR, "RedisDB::setShareInfo: getGroupId error. sid=%s groupName=%s", share->serviceId, gref->mName);
					if( newGidPtr ) free(newGidPtr);
					free(curGidPtr);
					return e;
				}
				gid	= _gid;
			}
			if( gid == NULL || *gid == 0 ) {
				mSentinel->log(LOG_ERR, "RedisDB::setShareInfo: group id is not specified error. sid=%s rid=%s", share->serviceId, share->resourceId);
				if( newGidPtr ) free(newGidPtr);
				free(curGidPtr);
				return ErrorData;
			}
			strcpy(curGidArr[i], gid);
		}
		qsort(curGidArr, cur->mShare.groupCount, sizeof(char*), qsCompareGid);
	}

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:1: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
		if( newGidPtr ) free(newGidPtr);
		if( curGidPtr ) free(curGidPtr);
		return ErrorSystem;
	}

	if( !jb.addObjectProp("id", 2) ||
		!jb.valueString(keyShare.getPtr()) ||
		!jb.addObjectProp("serviceId", 9) ||
		!jb.valueString(share->serviceId) ||
		!jb.addObjectProp("resourceId", 10) ||
		!jb.valueString(share->resourceId) ||
		!jb.addObjectProp("index", 5) ||
		!jb.valueInt(share->index) ||
		!jb.addObjectProp("dataMode", 8) ||
		!jb.valueInt((int)share->dataMode) ||
		!jb.addObjectProp("accessMode", 10) ||
		!jb.valueInt((int)share->accessMode)) {
		mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
		if( newGidPtr ) free(newGidPtr);
		if( curGidPtr ) free(curGidPtr);
		return ErrorSystem;
	}
	if( share->expireTime != 0 ) {
		char _time[256];
		strftime(_time, sizeof(_time), "%Y/%m/%d-%H:%M:%S", gmtime(&share->expireTime));
		if( !jb.addObjectProp("expireTime", 10) ||
		    !jb.valueString(_time)) {
			mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
			if( newGidPtr ) free(newGidPtr);
			if( curGidPtr ) free(curGidPtr);
			return ErrorSystem;
		}
	}
	if( share->fromTime != 0 ) {
		char _time[256];
		strftime(_time, sizeof(_time), "%Y/%m/%d-%H:%M:%S", gmtime(&share->fromTime));
		if( !jb.addObjectProp("fromTime", 8) ||
		    !jb.valueString(_time)) {
			mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
			if( newGidPtr ) free(newGidPtr);
			if( curGidPtr ) free(curGidPtr);
			return ErrorSystem;
		}
	}
	if( share->endTime != 0 ) {
		char _time[256];
		strftime(_time, sizeof(_time), "%Y/%m/%d-%H:%M:%S", gmtime(&share->endTime));
		if( !jb.addObjectProp("endTime", 7) ||
		    !jb.valueString(_time)) {
			mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
			if( newGidPtr ) free(newGidPtr);
			if( curGidPtr ) free(curGidPtr);
			return ErrorSystem;
		}
	}

	if( share->geoLocationCount > 0 ) {
		if( !jb.addObjectProp("geoLocation", 11) ||
			!jb.beginArray() ) {
			mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
			if( newGidPtr ) free(newGidPtr);
			if( curGidPtr ) free(curGidPtr);
			return ErrorSystem;
		}
		size_t i;
		for( i = 0; i < share->geoLocationCount; i++) {
			if( !jb.addArrayContent() ||
				!jb.beginObject() ||
				!jb.addObjectProp("dis", 3) ||
				!jb.valueNumber(share->geoLocation[i].dis) ||
				!jb.addObjectProp("lat", 3) ||
				!jb.valueNumber(share->geoLocation[i].lat) ||
				!jb.addObjectProp("lon", 3) ||
				!jb.valueNumber(share->geoLocation[i].lon) ||
				!jb.endObject() ) {
				mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
				if( newGidPtr ) free(newGidPtr);
				if( curGidPtr ) free(curGidPtr);
				return ErrorSystem;
			}

		}
		if( !jb.endArray() ) {
			mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
			if( newGidPtr ) free(newGidPtr);
			if( curGidPtr ) free(curGidPtr);
			return ErrorSystem;
		}
	}

	if( !jb.addObjectProp("gpsGrace", 8) ||
		!jb.valueInt(share->gpsGrace) ||
		!jb.addObjectProp("offlineGrace", 12) ||
		!jb.valueInt(share->offlineGrace)  ) {
		mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:6: json builder failed %d. rid=%s", jb.getError(),share->resourceId);
		if( newGidPtr ) free(newGidPtr);
		if( curGidPtr ) free(curGidPtr);
		return ErrorSystem;
	}

	if( share->groupCount > 0 ) {
		if( !jb.addObjectProp("group", 5) ||
			!jb.beginArray() ) {
			mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
			return ErrorSystem;
		}
		size_t i;
		for( i = 0; i < share->groupCount; i++) {
			if( !jb.addArrayContent() ||
				!jb.beginObject() ||
				!jb.addObjectProp("gid", 3) ||
				!jb.valueString(newGidArr[i]) ||
				!jb.endObject() ) {
				mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
				if( newGidPtr ) free(newGidPtr);
				if( curGidPtr ) free(curGidPtr);
				return ErrorSystem;
				break;
			}
		}
		if( !jb.endArray() ) {
			mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:2: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
			if( newGidPtr ) free(newGidPtr);
			if( curGidPtr ) free(curGidPtr);
			return ErrorSystem;
		}
	}
	if( !jb.addObjectProp("createTime", 10) ||
		!jb.valueString(_createTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:7: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
		if( newGidPtr ) free(newGidPtr);
		if( curGidPtr ) free(curGidPtr);
		return ErrorSystem;
	}

	if( !jb.endObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:9: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
		if( newGidPtr ) free(newGidPtr);
		if( curGidPtr ) free(curGidPtr);
		return ErrorSystem;
	}

	pjson::json *js = jb.detouch();
	StdBuffer b;
	if( !js->getText(&b,false) ) {
		delete js;
		mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:10: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
		if( newGidPtr ) free(newGidPtr);
		if( curGidPtr ) free(curGidPtr);
		return ErrorSystem;
	}
	delete js;

	e  = execStatusCommand("setShareInfo", "SET %s %s", keyShare.getPtr(), b.getPtr());
	if( e != ErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::setShareInfo:10: json builder failed %d. rid=%s", jb.getError(), share->resourceId);
		if( newGidPtr ) free(newGidPtr);
		if( curGidPtr ) free(curGidPtr);
		return e;
	}
	size_t ci = 0;
	size_t ni = 0;
	while( cur && ci < cur->mShare.groupCount && ni < share->groupCount ) {
		long long int rem;
		int cond = strcmp(curGidArr[ci], newGidArr[ni]);
		if( cond == 0 ) {
			ci++;
			ni++;
		} else if( cond < 0 ) {
			StdBuffer key;
			DBUtil::keyGroupShareList(curGidArr[ci], key);
			e = getInt(rem, "setShareInfo", "LREM %s 0 %s", key.getPtr(), keyShare.getPtr());
			if( e != ErrorOk ) {
				mSentinel->log(LOG_WARNING, "RedisDB::setShareInfo: redis error. cmd=\"LREM %s 0 %s\" code=%d msg=%s", key.getPtr(),  keyShare.getPtr(), e, mConn->errstr);
			}
			ci++;
		} else if( cond > 0 ) {
			StdBuffer key;
			DBUtil::keyGroupShareList(newGidArr[ni], key);
			e = getInt(rem, "setShareInfo", "RPUSH %s %s", key.getPtr(), keyShare.getPtr());
			if( e != ErrorOk ) {
				mSentinel->log(LOG_WARNING, "RedisDB::setShareInfo: redis error. cmd=\"RPUSH %s %s\" code=%d msg=%s", key.getPtr(),  keyShare.getPtr(), e, mConn->errstr);
			}
			ni++;
		}
	}
	while( cur && ci < cur->mShare.groupCount ) {
		long long int rem;
		StdBuffer key;
		DBUtil::keyGroupShareList(curGidArr[ci], key);
		e = getInt(rem, "setShareInfo", "LREM %s 0 %s", key.getPtr(), keyShare.getPtr());
		if( e != ErrorOk ) {
			mSentinel->log(LOG_WARNING, "RedisDB::setShareInfo: redis error. cmd=\"LREM %s 0 %s\" code=%d msg=%s", key.getPtr(),  keyShare.getPtr(), e, mConn->errstr);
		}
		ci++;
	}
	while( ni < share->groupCount ) {
		long long int rem;
		StdBuffer key;
		DBUtil::keyGroupShareList(newGidArr[ni], key);
		e = getInt(rem, "setShareInfo", "RPUSH %s %s", key.getPtr(), keyShare.getPtr());
		if( e != ErrorOk ) {
			mSentinel->log(LOG_WARNING, "RedisDB::setShareInfo: redis error. cmd=\"RPUSH %s %s\" code=%d msg=%s", key.getPtr(),  keyShare.getPtr(), e, mConn->errstr);
		}
		ni++;
	}

	if( id_ret )
		strcpy(id_ret, keyShare.getPtr());

	return ErrorOk;
}

ErrorCode	RedisDB::remShareInfo(const char *mid)
{
	ResultShare *cur = NULL;
	ErrorCode e = getShareInfo(mid, &cur);
	if( e != ErrorOk ) {
		return e;
	}

	long long int rem;

	// delete service group list
	size_t i;
	for( i = 0; i < cur->mShare.groupCount; i++ ) {
		StdBuffer key;
		DBUtil::keyGroupShareList(cur->mShare.groupArray[i].mId, key);
		mSentinel->log(LOG_DEBUG, "RedisDB::remShareInfo: remove group share list. key=%s", key.getPtr());
		e = getInt(rem, "remShareInfo", "LREM %s 0 %s", key.getPtr(), mid);
		if( e != ErrorOk ) {
			mSentinel->log(LOG_WARNING, "RedisDB::remShareInfo: redis error. cmd=\"LREM %s 0 %s\" code=%d msg=%s", key.getPtr(),  mid, e, mConn->errstr);
		}
	}

	// delete shareInfo
	e = getInt(rem, "remAcocuntInfo", "DEL %s", mid);
	if( e != ErrorOk ) {
		mSentinel->log(LOG_WARNING, "RedisDB::remShareInfo: redis error. cmd=\"DEL %s\" code=%d msg=%s", mid, e, mConn->errstr);
	}
	return ErrorOk;
}

ErrorCode RedisDB::pushSyncQueue(const char *sid, const char *gid)
{
	StdBuffer key;
	DBUtil::keySyncQueue(sid, key);

	long long int nDummy;
	ErrorCode e;
	e = getInt(nDummy, "pushSyncQueue", "LREM %s 0 %s", key.getPtr(), gid);
	e = getInt(nDummy, "pushSyncQueue", "RPUSH %s %s", key.getPtr(), gid);
	return e;
}

ErrorCode RedisDB::popSyncQueue(const char *sid, char *id_ret)
{
	StdBuffer key;
	DBUtil::keySyncQueue(sid, key);

	redisReply *pRep = NULL;
	ErrorCode e = getStr(&pRep, "popSyncQueue", "LPOP %s", key.getPtr());
	if( e != ErrorOk ) {
		if( pRep ) freeReplyObject(pRep);
		return e;
	}
	strcpy(id_ret, pRep->str);
	freeReplyObject(pRep);
	return ErrorOk;
}

/////////////////////////////////////////////////////////////////////////////
// session
/*
DBError	RedisDB::getSessionId(const char *peerAddr, char *id_ret)
{
	mSentinel->log(LOG_DEBUG, "RedisDB::getSessionId: start");

	// TODO: addrのハッシュを作る
	char _addr_hash[256];
	strcpy(_addr_hash, peerAddr);
	redisReply *pRep = NULL;
	DBError e = getStr(&pRep, "getSessionId", "GET KC:%s", _addr_hash);
	if( e != DBErrorOk ) {
		if( pRep )
			freeReplyObject(pRep);
		return e;
	}
	strcpy(id_ret, pRep->str);
	freeReplyObject(pRep);

	mSentinel->log(LOG_DEBUG, "RedisDB::getSessionId: end");
	return DBErrorOk;
}

DBError	RedisDB::setSession(const SessionInfo *ss, char *id_ret)
{
	mSentinel->log(LOG_DEBUG, "RedisDB::setSession: start");

	char peer[SST_ADDR_LEN+1];
	bool newId = false;
	char _ssid[SST_SESSIONID_LEN+1];
	const char *ssid;
	char _addr_hash[256];

	sprintf(peer, "%s:%d", inet_ntoa(ss->peerAddr.sin_addr), ntohs(ss->peerAddr.sin_port));
	if(strlen(peer) < 10 ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession: peerAddr is not specfied.");
		return DBErrorData;
	}
	if( ss->servName == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession: servName is not specfied.");
		return DBErrorData;
	}
	if( ss->serviceId == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession: serviceId is not specfied.");
		return DBErrorData;
	}
	if( ss->accountId == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession: accountId is not specfied.");
		return DBErrorData;
	}
	if( ss->deviceId == NULL ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession: deviceId is not specfied.");
		return DBErrorData;
	}

	if( ss->sessionId == NULL ) {
		// session id
		long long int cid_seq;
		DBError e = getInt(cid_seq, "setSession", "INCR IDSEQ_C");
		if( e != DBErrorOk ) {
			return e;
		}
		sprintf(_ssid, "C%8.8llx", cid_seq);
		ssid		= _ssid;
		newId	= true;

		// TODO: uidの文字種チェックを行い、key convert用にハッシュを作る
		strcpy(_addr_hash, peer);
	} else {
		ssid	= ss->sessionId;
	}

	time_t now = time(NULL);
	char _createTime[256];
	char _updateTime[256];
	if( newId ) {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));
	} else {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", gmtime(&ss->createTime));
	}
	strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", gmtime(&now));

	if( newId ) {
		long long int r;
		DBError e = getInt(r, "setSession", "SETNX KC:%s %s", _addr_hash, ssid);
		if( e != DBErrorOk ) {
			mSentinel->log(LOG_ERR, "RedisDB::setSession:1: reverse key set failed. rid=%s", ssid);
			return e;
		}
		if( r != 1 ) {
			mSentinel->log(LOG_ERR, "RedisDB::setSession: reverse key duplication. rid=%s", ssid);
			return DBErrorDuplication;
		}
	}

	pjson::builder jb;
	if( !jb.init(512) ||
		!jb.beginObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession:1: json builder failed %d. rid=%s", jb.getError(), peer);
		return DBErrorSystem;
	}

	if( !jb.addObjectProp("servName", 8) ||
		!jb.valueString(ss->servName) ||
		!jb.addObjectProp("peerAddr", 8) ||
		!jb.valueString(peer) ||
		!jb.addObjectProp("serviceId", 9) ||
		!jb.valueString(ss->serviceId) ||
		!jb.addObjectProp("accountId", 9) ||
		!jb.valueString(ss->accountId) ||
		!jb.addObjectProp("deviceId", 8) ||
		!jb.valueString(ss->deviceId) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession:2: json builder failed %d. rid=%s", jb.getError(), peer);
		return DBErrorSystem;
	}

	if( !jb.addObjectProp("status", 6) ||
		!jb.valueInt(ss->status)  ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession:6: json builder failed %d. rid=%s", jb.getError(), peer);
		return DBErrorSystem;
	}
	if( !jb.addObjectProp("createTime", 10) ||
		!jb.valueString(_createTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession:7: json builder failed %d. rid=%s", jb.getError(), peer);
		return DBErrorSystem;
	}
	if( !jb.addObjectProp("updateTime", 10) ||
		!jb.valueString(_updateTime) ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession:8: json builder failed %d. rid=%s", jb.getError(), peer);
		return DBErrorSystem;
	}

	if( !jb.endObject() ) {
		mSentinel->log(LOG_ERR, "RedisDB::setSession:9: json builder failed %d. rid=%s", jb.getError(), peer);
		return DBErrorSystem;
	}

	pjson::json *js = jb.detouch();
	StdBuffer b;
	if( !js->getText(&b,false) ) {
		delete js;
		mSentinel->log(LOG_ERR, "RedisDB::setSession:10: json builder failed %d. rid=%s", jb.getError(), peer);
		return DBErrorSystem;
	}
	delete js;

	DBError e  = execStatusCommand("setSession", "SET %s %s", ssid, b.getPtr());
	if( e != DBErrorOk ) {
		return e;
	}

	// add ServiceSessionList
	{
		long long int nDummy;
		DBError e = getInt(nDummy, "setSession", "RPUSH LC%s %s", ss->serviceId, ssid);
		if( e != DBErrorOk ) {
			if( newId ) {
				getInt(nDummy, "setSession", "DEL KC:%s:%s", ss->serviceId, _addr_hash);
				getInt(nDummy, "setSession", "DEL %s", ssid);
			}
			return e;
		}
	}

	if( id_ret )
		strcpy(id_ret, ssid);

	mSentinel->log(LOG_DEBUG, "RedisDB::setSession: end");
	return DBErrorOk;
}

DBError	RedisDB::getSession(const char *ssid, ResultSession **res)
{
	mSentinel->log(LOG_DEBUG, "RedisDB::getSession. begin");

	*res = NULL;
	pjson::json *js;
	DBError e = getJson(&js, "getSession", "GET %s", ssid);
	if( e != DBErrorOk ) {
		if( e == DBErrorData )
			mSentinel->log(LOG_DEBUG, "RedisDB::getSession: data error.");
		return e;
	}

	pjson::value	*v	= js->getValue();
	if( v->vt != pjson::vt_object ) {
		delete js;
		return DBErrorData;
	}

	pjson::value *vServName		= js->get("servName", v);
	pjson::value *vPeerAddr		= js->get("peerAddr", v);
	pjson::value *vServiceId	= js->get("serviceId", v);
	pjson::value *vAccountId	= js->get("accountId", v);
	pjson::value *vDeviceId		= js->get("deviceId", v);
	pjson::value *vStatus		= js->get("status", v);
	pjson::value *vCreateTime	= js->get("createTime", v);
	pjson::value *vUpdateTime	= js->get("updateTime", v);

	SessionInfo info;
	memset(&info, 0, sizeof(info));
	info.sessionId	= (char*)ssid;
	info.servName	= (char*)((vServName && vServName->vt == pjson::vt_string) ? vServName->vString : NULL);
	if( vPeerAddr && vPeerAddr->vt == REDIS_REPLY_STRING ) {
		char *t = strdup(vPeerAddr->vString);
		char *p = t;
		while(*p) {
			if( *p == ':' ) {
				*p++ = 0;
				break;
			}
			p++;
		}
		inet_aton(t, &info.peerAddr.sin_addr);
		info.peerAddr.sin_port		= atoi(t);
		info.peerAddr.sin_family	= AF_INET;
	} else {
		memset(&info.peerAddr, 0, sizeof(info.peerAddr));
	}
	info.serviceId	= (char*)((vServiceId && vServiceId->vt == pjson::vt_string) ? vServiceId->vString : NULL);
	info.accountId	= (char*)((vAccountId && vAccountId->vt == pjson::vt_string) ? vAccountId->vString : NULL);
	info.deviceId	= (char*)((vDeviceId && vDeviceId->vt == pjson::vt_string) ? vDeviceId->vString : NULL);
	info.status		= (SessionStatus)((vStatus && vStatus->vt == pjson::vt_int) ? vStatus->vInt : 0);
	struct tm tm_create;
	if( vCreateTime && vCreateTime->vt == pjson::vt_string) {
		strptime(vCreateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_create);
		info.createTime	= timegm(&tm_create);
	} else {
		info.createTime	= 0;
	}
	struct tm tm_last;
	if( vUpdateTime && vUpdateTime->vt == pjson::vt_string) {
		strptime(vUpdateTime->vString, "%Y/%m/%d-%H:%M:%S", &tm_last);
		info.updateTime	= timegm(&tm_last);
	} else {
		info.updateTime	= 0;
	}

	size_t size = sizeof(ResultSession)
				+ DBUtil::sizeofRBase()
				+ DBUtil::sizeofSessionInfo(&info);
	char *ptr = (char*)malloc(size);
	ResultSession *ret = (ResultSession*)ptr;
	ptr	+= sizeof(ResultAccount);
	ptr	= DBUtil::writeSessionInfo(&ret->mSession, ptr, &info);

	delete js;

	*res = ret;

#ifdef _DEBUG
	char _peerAddr[SST_ADDR_LEN+1];
	sprintf(_peerAddr, "%s:%d", inet_ntoa(ret->mSession.peerAddr.sin_addr), ntohs(ret->mSession.peerAddr.sin_port));

	char _createTime[256];
	if( info.createTime != 0 ) {
		strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", &tm_create);
	} else {
		strcpy(_createTime, "not set");
	}
	char _updateTime[256];
	if( info.updateTime != 0 ) {
		strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", &tm_last);
	} else {
		strcpy(_updateTime, "not set");
	}

	mSentinel->log(LOG_DEBUG, "*** RedisDB::getSession ***");
	mSentinel->log(LOG_DEBUG, "* mSession.sessionId=%s", ret->mSession.sessionId);
	mSentinel->log(LOG_DEBUG, "* mSession.servName=%s", ret->mSession.servName);
	mSentinel->log(LOG_DEBUG, "* mSession.peerAddr=%s", _peerAddr);
	mSentinel->log(LOG_DEBUG, "* mSession.serviceId=%s", ret->mSession.serviceId);
	mSentinel->log(LOG_DEBUG, "* mSession.accountId=%d", ret->mSession.accountId);
	mSentinel->log(LOG_DEBUG, "* mSession.deviceId=%d", ret->mSession.deviceId);
	mSentinel->log(LOG_DEBUG, "* mSession.status=%d", ret->mSession.status);
	mSentinel->log(LOG_DEBUG, "* mSession.createTime=%s", _createTime);
	mSentinel->log(LOG_DEBUG, "* mSession.updateTime=%s", _updateTime);
#endif

	mSentinel->log(LOG_DEBUG, "RedisDB::getSession: end");
	return DBErrorOk;
}

DBError	RedisDB::remSession(const char *ssid)
{
	mSentinel->log(LOG_DEBUG, "RedisDB::remSession. begin");

	// TODO: TASKが空であること

	ResultSession *ss = NULL;
	DBError e = getSession(ssid, &ss);
	if( e != DBErrorOk ) {
		return e;
	}

	// TODO: uidの文字種チェックを行い、key convert用にハッシュを作る
	char _addr_hash[256];
	char peer[SST_ADDR_LEN+1];
	sprintf(peer, "%s:%d", inet_ntoa(ss->mSession.peerAddr.sin_addr), ntohs(ss->mSession.peerAddr.sin_port));
	strcpy(_addr_hash, peer);	strcpy(_addr_hash, peer);

	long long int rem;

	// delete SessionKeyReverse
	mSentinel->log(LOG_DEBUG, "RedisDB::remSession: remove SessionKeyReverse. key=KC:%s", _addr_hash);
	DBError e1 = getInt(rem, "remAcocuntInfo", "DEL KC:%s", _addr_hash);
	if( e1 != DBErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::remSession: redis error. cmd=\"DEL KC:%s\" code=%d msg=%s", _addr_hash, e1, mConn->errstr);
	}

	// delete ServiceSessionList
	mSentinel->log(LOG_DEBUG, "RedisDB::remSession: remove ServiceSessionList.");
	DBError e2 = getInt(rem, "remAcocuntInfo", "LREM LC%s 0 %s", ss->mSession.serviceId, ssid);
	if( e2 != DBErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::remSession: redis error. cmd=\"LREM LC%s 0 %s\" code=%d msg=%s", ss->mSession.serviceId, ssid, e2, mConn->errstr);
	}

	// delete SessionInfo
	mSentinel->log(LOG_DEBUG, "RedisDB::remSession. remove SessionInfo. key=%s", ssid);
	DBError e3 = getInt(rem, "remAcocuntInfo", "DEL %s", ssid);
	if( e3 != DBErrorOk ) {
		mSentinel->log(LOG_ERR, "RedisDB::remSession: redis error. cmd=\"DEL %s\" code=%d msg=%s", ssid, e3, mConn->errstr);
	}

	mSentinel->log(LOG_DEBUG, "RedisDB::remSession. end");

	if(e1 != DBErrorOk || e2 != DBErrorOk || e3 != DBErrorOk )
		return DBErrorSystem;
	return DBErrorOk;
}

DBError	RedisDB::getSessionCount(const char *rid, size_t &nRec)
{
	mSentinel->log(LOG_DEBUG, "RedisDB::getSessionCount: begin");
	DBError e = getKeyCount("C", nRec);
	mSentinel->log(LOG_DEBUG, "RedisDB::getSessionCount: end");
	return e;
}

DBError	RedisDB::getSessionList(const char *rid, size_t nFrom, size_t nCount, ResultList **pList)
{
	mSentinel->log(LOG_DEBUG, "RedisDB::getSessionList: begin");
	DBError e = getKeyList("C", nFrom, nCount, pList);
	mSentinel->log(LOG_DEBUG, "RedisDB::getSessionList: end");
	return e;
}
*/

}

