/*
 * worker.cpp
 *
 *  Created on: 2012/12/10
 *      Author: yasuoki
 */


#include "sentinel.h"
#include "scheduler.h"
#include "buffer.h"
#include "queue.h"
#include "lock.h"
#include <memory.h>
#include <pthread.h>
#include <syslog.h>
#include <errno.h>
#include <unistd.h>
#include <time.h>

namespace SST {

WaitQueue::WaitQueue(QueueNode *q, int waitSec)
{
	clock_gettime(CLOCK_REALTIME, &mWaitSec);
	mQueue		= q;
	mWaitSec.tv_sec += waitSec;
}

WaitQueue::~WaitQueue()
{
}
/////////////////////////////////////////////////////////////////////////
Scheduler::Scheduler()
{
	mSentinel	= NULL;
	mWorkerPtr	= NULL;
	mWorkerCnt	= 0;
	mDoStop		= false;
	mThreadHandlePtr	= NULL;
}

Scheduler::~Scheduler()
{
	clear();
}

bool Scheduler::configure(Sentinel *obj, Conf *conf)
{
	mSentinel	= obj;
	int numWorker = conf->numThreads;
	mWorkerPtr	= new ProcThread[numWorker];
	if( mWorkerPtr == NULL ) {
		mSentinel->log(LOG_ERR, "Scheduler::configure: Out of memory for scheduler thread.");
		return false;
	}
	mWorkerCnt	= numWorker;
	int i;
	for( i = 0; i < numWorker; i++ ) {
		mWorkerPtr[i].init(this);
	}
	if( sem_init(&mSem, 0, 0) == -1 ) {
		mSentinel->log(LOG_ERR, "Scheduler::configure: semaphore init error code=%d", errno);
		return false;
	}

	return true;
}

ProcThread * Scheduler::assign() const
{
	ProcThread *thread = mWorkerPtr;
	int i;
	for( i = 0; i < mWorkerCnt; i++ ) {
		if( thread->assignIfIdle() ) break;
		thread++;
	}
	if( i == mWorkerCnt ) {
//		mSentinel->log(LOG_DEBUG, "Scheduler::assign not found 1.");
		thread = mWorkerPtr;
		for( i = 0; i < mWorkerCnt; i++ ) {
			if( thread->assignIfStop() ) break;
			thread++;
		}
		if( i == mWorkerCnt ) {
//			mSentinel->log(LOG_DEBUG, "Scheduler::assign not found 2.");
			return NULL;
		}
//		mSentinel->log(LOG_DEBUG, "Scheduler::assign run thread.");
		if( !thread->run() ) {
			return NULL;
		}
	}
	return thread;
}

void Scheduler::schedule()
{
	if( mThreadHandlePtr == NULL ) {
		if( pthread_create( &mThreadHandle, NULL, scheduleThreadEntry, this) != 0 ) {
			mSentinel->log(LOG_ERR, "Scheduler::setQueueTimer: start timer thread is failed");
			return;
		}
		mThreadHandlePtr	= &mThreadHandle;
	}
	sem_post(&mSem);
}

bool Scheduler::setQueueTimer(QueueNode *q, int waitSec)
{
	WaitQueue *wq = new WaitQueue(q, waitSec);
	mQueueTimerList.addBottom(wq);
	schedule();
	return true;
}

void Scheduler::stop()
{
	mDoStop	= true;
	ProcThread *thread = mWorkerPtr;
	int i;
	for( i = 0; i < mWorkerCnt; i++ ) {
		thread->stop();
		thread++;
	}
}

void Scheduler::clear()
{
	if( mWorkerCnt ) {
		delete [] mWorkerPtr;
		mWorkerPtr	= NULL;
		mWorkerCnt	= 0;
	}
}

void * Scheduler::scheduleThreadEntry(void *ptr)
{
	Scheduler * pThis = (Scheduler*)ptr;
	void *ret = pThis->scheduleLoop();
	pThis->mThreadHandlePtr	= NULL;
	return ret;
}

void * Scheduler::scheduleLoop()
{
	int cnt=0;
	while(1) {
		if( mDoStop) break;

		struct timespec ts;
		if( clock_gettime(CLOCK_REALTIME, &ts) == -1 ) {
			mSentinel->log(LOG_ERR, "Scheduler::scheduleLoop: gettime error. code=%d.", errno);
			break;
		}
		if( mQueueTimerList.empty() )
			ts.tv_sec += 10;
		else
			ts.tv_sec += 1;
		int waitcode = sem_timedwait(&mSem, &ts);
		if( mDoStop ) {
			mSentinel->log(LOG_INFO, "Scheduler::scheduleLoop: accept stop command.");
			break;
		}

		if( waitcode == -1 ) {
			int e = errno;
			cnt++;
			if( e == EINTR ) {
				mSentinel->log(LOG_INFO, "Scheduler::scheduleLoop: accept interrupt.");
				break;
			} else if( e == EINVAL ) {
				mSentinel->log(LOG_ERR, "Scheduler::scheduleLoop: wait invalid error.");
				if( cnt <= 5 ) {
					mSentinel->log(LOG_INFO, "Scheduler::scheduleLoop: x continued.");
					continue;
				}
				break;
			} else if( e == ETIMEDOUT ) {
				scheduleTimer();
			} else {
				mSentinel->log(LOG_ERR, "Scheduler::scheduleLoop: wait unkown error. code=%d", e);
				break;
			}
			continue;
		}
		scheduleQueue();
	}

	mSentinel->log(LOG_DEBUG, "Scheduler::scheduleLoop: end");

	return NULL;
}

void Scheduler::scheduleQueue()
{
	Process *proc;
	unsigned long long wait;
	Process *exec_proc = NULL;
	unsigned long long exec_wait = NULL;
	bool	reserveThread;

	proc = &mSentinel->getDatabase();
	if( !proc->queuIsEmpty() ) {
		reserveThread	= proc->reserveThread();
		if( reserveThread ) {
			wait = proc->getWaitTime();
			if( exec_proc == NULL || exec_wait < wait ) {
				if( exec_proc )
					exec_proc->cancelThread();
				exec_proc = proc;
				exec_wait = wait;
			} else {
				proc->cancelThread();
			}
		}
	}
	proc = &mSentinel->getRequester();
	if( !proc->queuIsEmpty() ) {
		reserveThread	= proc->reserveThread();
		if( reserveThread ) {
			wait = proc->getWaitTime();
			if( exec_proc == NULL || exec_wait < wait ) {
				if( exec_proc )
					exec_proc->cancelThread();
				exec_proc = proc;
				exec_wait = wait;
			} else {
				proc->cancelThread();
			}
		}
	}
	proc = &mSentinel->getListener();
	if( !proc->queuIsEmpty() ) {
		reserveThread	= proc->reserveThread();
		if( reserveThread ) {
			wait = proc->getWaitTime();
			if( exec_proc == NULL || exec_wait < wait ) {
				if( exec_proc )
					exec_proc->cancelThread();
				exec_proc = proc;
				exec_wait = wait;
			} else {
				proc->cancelThread();
			}
		}
	}
	proc = &mSentinel->getCommand();
	if( !proc->queuIsEmpty() ) {
		reserveThread	= proc->reserveThread();
		if( reserveThread ) {
			wait = proc->getWaitTime();
			if( exec_proc == NULL || exec_wait < wait ) {
				if( exec_proc )
					exec_proc->cancelThread();
				exec_proc = proc;
				exec_wait = wait;
			} else {
				proc->cancelThread();
			}
		}
	}

	if( exec_proc ) {
		ProcThread *thread = assign();
		if( thread ) {
			if( !thread->execute(exec_proc) ) {
				//return false;
			}
		} else {
			exec_proc->cancelThread();
		}
	}
}

void Scheduler::scheduleTimer()
{
	WaitQueue *wq = (WaitQueue*)mQueueTimerList.getTop();
	if( wq == NULL ) return;
	timespec ts;
	clock_gettime(CLOCK_REALTIME, &ts);
	while(wq) {
		if( mDoStop) break;
		WaitQueue *nq = (WaitQueue*)mQueueTimerList.getNext(wq);
		if( wq->mWaitSec.tv_sec < ts.tv_sec || (wq->mWaitSec.tv_sec == ts.tv_sec && wq->mWaitSec.tv_nsec <= ts.tv_nsec) ) {
			Process *proc = wq->mQueue->getProcess();
			proc->putQueue(wq->mQueue);
			mQueueTimerList.removeNode(wq);
		}
		wq = nq;
	}
}

bool Scheduler::diag(pjson::builder &jb)
{
	READ_LOCK(mLock);
	if( !jb.beginObject() ||
		!jb.addObjectProp("thread",6) ||
		!jb.beginArray() ) {
		mSentinel->log(LOG_ERR, "Scheduler::diag json error 1. code=%d", jb.getError());
		return false;
	}

	ProcThread *s = mWorkerPtr;
	int i;
	for( i = 0; i < mWorkerCnt; i++ ) {
		char buf[256];
		sprintf(buf, "%8.8X", (unsigned int)s->getThreadHandle());
		if( !jb.addArrayContent() ||
			!jb.beginObject() ||
			!jb.addObjectProp("addr",4) ||
			!jb.valueString(buf) ) {
			mSentinel->log(LOG_ERR, "Scheduler::diag json error 2. code=%d", jb.getError());
			return false;
		}

		Process *proc = s->getProcess();

		if( !jb.addObjectProp("assign", 6) ||
			!jb.valueString( s->isAssigned() ? "yes":"no") ||
			!jb.addObjectProp("exec", 4) ||
			!jb.valueString( s->isExecuting()? "yes":"no") ||
			!jb.addObjectProp("proc", 4) ||
			!jb.valueString( proc ? proc->getName() : "-") ) {
			mSentinel->log(LOG_ERR, "Scheduler::diag json error 3. code=%d", jb.getError());
			return false;
		}
		if( !jb.endObject() ) {
			mSentinel->log(LOG_ERR, "Scheduler::diag json error 2. code=%d", jb.getError());
			return false;
		}

		s++;
	}
	if( !jb.endArray() ||
		!jb.endObject() ) {
		mSentinel->log(LOG_ERR, "Scheduler::diag json error 6. code=%d", jb.getError());
		return false;
	}
	return true;
}

/////////////////////////////////////////////////////////////////

ProcThread::ProcThread()
{
	mDoStop				= false;
	mPool				= NULL;
	mThreadHandlePtr	= NULL;
	mProc				= NULL;
	mAssign				= false;
}

ProcThread::~ProcThread()
{
}

bool ProcThread::init(Scheduler *pool)
{
	mPool	= pool;
	if( sem_init(&mSem, 0, 0) == -1 ) {
		mPool->mSentinel->log(LOG_ERR, "ProcThread::init: semaphore init error code=%d", errno);
		return false;
	}
	return true;
}

bool ProcThread::run()
{
	if( pthread_create( &mThreadHandle, NULL, threadEntry, this) != 0 ) {
		mPool->mSentinel->log(LOG_ERR, "ProcThread::run: start scheduler thread is failed");
		return false;
	}
	mThreadHandlePtr	= &mThreadHandle;
	return true;
}

void * ProcThread::threadEntry(void *ptr)
{
	void *ret = (( ProcThread*)ptr)->mainLoop();
	return ret;
}

void * ProcThread::mainLoop()
{
	int cnt=0;
	bool doProc=false;
	while( !mDoStop ) {
		struct timespec ts;
		if( clock_gettime(CLOCK_REALTIME, &ts) == -1 ) {
			mPool->mSentinel->log(LOG_ERR, "ProcThread::mainLoop: gettime error. code=%d.", errno);
			break;
		}
		ts.tv_sec += 5;
		int waitcode = sem_timedwait(&mSem, &ts);
		if( mDoStop ) {
			mPool->mSentinel->log(LOG_INFO, "ProcThread::mainLoop: accept stop command.");
			break;
		}
		if( waitcode == -1 ) {
			int e = errno;
			doProc = false;
			cnt++;
			if( e == EINTR ) {
				mPool->mSentinel->log(LOG_INFO, "ProcThread::mainLoop: accept interrupt.");
			} else if( e == EINVAL ) {
				mPool->mSentinel->log(LOG_ERR, "ProcThread::mainLoop: wait invalid error.");
				bool assign;
				{
					WRITE_LOCK(mPool->mLock);
					assign = mAssign;
				}
				if( assign ) {
					if( cnt <= 5 ) {
						mPool->mSentinel->log(LOG_INFO, "ProcThread::mainLoop: x continued.");
						doProc = true;
					}
				}
			} else if( e == ETIMEDOUT ) {
				WRITE_LOCK(mPool->mLock);
				if( mAssign ) {
					doProc = true;
				}
			} else {
				mPool->mSentinel->log(LOG_ERR, "ProcThread::mainLoop: wait unkown error. code=%d", e);
			}
			if( !doProc ) {
				WRITE_LOCK(mPool->mLock);
				mThreadHandlePtr	= NULL;
				break;
			}
		}
		cnt = 0;
		Process *proc = NULL;
		{
			READ_LOCK(mPool->mLock);
			proc = mProc;
		}
		if( proc ) {
			QueueNode *q = proc->getQueue();
			if( q != NULL ) {
				Task *task = q->getTask();
				task->beginProcess(q);
				q->mWaitResult = false;
				bool result = proc->doProcess(q);
				task->endProcess(q);
				Session *con = task->getSession();
				if( !result ) {
					// TODO: taskスタック中のdoProcessがエラー終了すると停止する
					mPool->mSentinel->log(LOG_ERR, "scheduler: task=%x proc=%s func=%s q=%x waitq=%d failed.",
							task, proc->getName(), proc->getFuncName(q->getFunction()), q, task->getWaitingQueue());
					delete q;
					task->endTask(NULL);
					if( con )
						mPool->mSentinel->getSessionPool().freeSession(con);
				} else {
					if( con && con->getStatus() == SessionStatusWait ) {
						// close connection
						if( task->getTaskStatus() != TaskIdle && task->getWaitingQueue() == 0 )
							task->endTask(NULL);
						mPool->mSentinel->getSessionPool().freeSession(con);
						delete q;
					} else {
						delete q;
					}
				}
			}
		}
		{
			WRITE_LOCK(mPool->mLock);
			mAssign	= false;
			mProc	= NULL;
		}
 		mPool->schedule();
	}
	{
		WRITE_LOCK(mPool->mLock);
		mAssign	= false;
		mProc	= NULL;
	}
	return NULL;
}

bool ProcThread::assignIfIdle()
{
	WRITE_LOCK(mPool->mLock);
	if( !mAssign && mThreadHandlePtr != NULL ) {
		mAssign	= true;
		return true;
	}
	return false;
}

bool ProcThread::assignIfStop()
{
	WRITE_LOCK(mPool->mLock);
	if( !mAssign && mThreadHandlePtr == NULL ) {
		mAssign	= true;
		return true;
	}
	return false;
}

bool ProcThread::execute(Process *proc)
{
	READ_LOCK(mPool->mLock);
	if( !mAssign )
		return false;
	if( !mThreadHandlePtr )
		return false;
	mProc	= proc;
	sem_post(&mSem);
	return true;
}

void ProcThread::stop()
{
	mDoStop	= true;
	if( mThreadHandlePtr ) {
		if( mProc )
			mProc->doAbort();
		sem_post(&mSem);
		void *ret = NULL;
		pthread_join(mThreadHandle, &ret);
		mThreadHandlePtr = NULL;
	}
	mDoStop	= false;
}

bool ProcThread::isAssigned() const
{
	return mAssign;
}

bool ProcThread::isExecuting() const
{
	return mAssign && mThreadHandlePtr != NULL;
}


}
