/*
 * cmd_service.cpp
 *
 *  Created on: 2013/01/05
 *      Author: yasuoki
 */

#include "commands.h"
#include "proc_command.h"
#include "proc_database.h"
#include "task.h"
#include "sentinel.h"
#include "buffer.h"
#include "pjson.h"
#include <syslog.h>
#include <errno.h>
#include <netdb.h>

namespace SST {


class TBRegService : public TaskBuffer {
public:
	TBRegService() {
		mCommand		= NULL;
		mJsKeyId		= NULL;
		mJsKeyName		= NULL;
		mJsName			= NULL;
		mJsTitle		= NULL;
		mJsDesc			= NULL;
		mJsVersion		= NULL;
		mJsAdminAID		= NULL;
		mJsAdminUID		= NULL;
		mJsRefURL		= NULL;
		mJsScope		= NULL;
		mJsGroupName	= NULL;
		mJsBlockSize	= NULL;
		mJsBlockCount	= NULL;
		mJsStorageCrypt	= NULL;
		mJsAuthType		= NULL;
		mJsAuthData		= NULL;
		mCrypt			= CryptAlgorithmNone;
		mScope			= ScopePublic;
		mAuthType		= AuthNone;
		mAccount		= NULL;
		mServiceId		= NULL;
		mService		= NULL;
		mGroup			= NULL;
		mGroupId		= NULL;
	}
	virtual ~TBRegService() {
		if( mCommand )		delete mCommand;
		if( mAccount )		free(mAccount);
		if( mServiceId )	free(mServiceId);
		if( mService )		free(mService);
		if( mGroup )		free(mGroup);
		if( mGroupId )		free(mGroupId);
	}

	pjson::json *	mCommand;
	pjson::value *	mJsKeyId;
	pjson::value *	mJsKeyName;
	pjson::value *	mJsName;
	pjson::value *	mJsTitle;
	pjson::value *	mJsDesc;
	pjson::value *	mJsVersion;
	pjson::value *	mJsAdminAID;
	pjson::value *	mJsAdminUID;
	pjson::value *	mJsRefURL;
	pjson::value *	mJsScope;
	pjson::value *	mJsGroupName;
	pjson::value *	mJsBlockSize;
	pjson::value *	mJsBlockCount;
	pjson::value *	mJsStorageCrypt;
	pjson::value *	mJsAuthType;
	pjson::value *	mJsAuthData;
	CryptAlgorithm	mCrypt;
	ServiceScope	mScope;
	AuthType		mAuthType;
	ResultAccount *	mAccount;
	ResultId *		mServiceId;
	ResultService *	mService;
	ResultGroup*	mGroup;
	ResultId *		mGroupId;
};

bool CommandProc::cmdRegService(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBRegService *tb = (TBRegService *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBRegService();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			pjson::json *jsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = jsCommand->get("regService");
			if( !vCmd ) {
				return taskResponse(q, ErrorParameter, "%s. \"regService\" is not object.", codeToMessgae(ErrorParameter));
			}

			tb->mJsName			= jsCommand->get("serviceName", vCmd);
			tb->mJsTitle		= jsCommand->get("title", vCmd);
			tb->mJsDesc			= jsCommand->get("description", vCmd);
			tb->mJsVersion		= jsCommand->get("version", vCmd);
			pjson::value *vAdmin = jsCommand->get("admin", vCmd);
			if( vAdmin ) {
				if( vAdmin->vt != pjson::vt_object )
					return taskResponse(q, ErrorParameter, "%s. \"admin\" type.", codeToMessgae(ErrorParameter));
				tb->mJsAdminAID	= jsCommand->get("aid", vAdmin);
				tb->mJsAdminUID	= jsCommand->get("userId", vAdmin);
			}
			tb->mJsRefURL		= jsCommand->get("refURL", vCmd);
			tb->mJsScope		= jsCommand->get("scope", vCmd);
			tb->mJsGroupName	= jsCommand->get("defaultGroup", vCmd);
			tb->mJsBlockSize	= jsCommand->get("storageBlockSize", vCmd);
			tb->mJsBlockCount	= jsCommand->get("storageBlockCount", vCmd);
			tb->mJsStorageCrypt	= jsCommand->get("storageCrypt", vCmd);
			tb->mJsAuthType		= jsCommand->get("authType", vCmd);
			tb->mJsAuthData		= jsCommand->get("authData", vCmd);

			if( !tb->mJsName )
				return taskResponse(q, ErrorParameter, "%s. \"serviceName\" is not specified.", codeToMessgae(ErrorParameter));
			if( tb->mJsName->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"name\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsTitle && tb->mJsTitle->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"title\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsDesc && tb->mJsDesc->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"description\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsVersion && tb->mJsVersion->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"version\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAdminAID && tb->mJsAdminAID->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"admin.aid\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAdminUID && tb->mJsAdminUID->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"admin.userId\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsRefURL && tb->mJsRefURL->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"refURL\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsScope && tb->mJsScope->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"scope\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsGroupName && tb->mJsGroupName->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"defaultGroup\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsBlockSize && tb->mJsBlockSize->vt != pjson::vt_int)
				return taskResponse(q, ErrorParameter, "%s. \"storageBlockSize\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsBlockCount && tb->mJsBlockCount->vt != pjson::vt_int )
				return taskResponse(q, ErrorParameter, "%s. \"storageBlockCount\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsStorageCrypt && tb->mJsStorageCrypt->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"storageCrypt\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAuthType && tb->mJsAuthType->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"authType\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAuthData && tb->mJsAuthData->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"authData\" type.", codeToMessgae(ErrorParameter));
			taskLog(q, LOG_DEBUG, "json data ok");

			if( !tb->mJsAdminAID && !tb->mJsAdminUID )
				return taskResponse(q, ErrorParameter, "%s. administrator account is not specified", codeToMessgae(ErrorParameter));
			if( tb->mJsScope ) {
				if( strcmp(tb->mJsScope->vString, "private") == 0 )
					tb->mScope	= ScopePrivate;
				else if( strcmp(tb->mJsScope->vString, "public") == 0 )
					tb->mScope	= ScopePublic;
				else
					return taskResponse(q, ErrorParameter, "%s. scope value", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsAuthType ) {
				if( strcmp(tb->mJsAuthType->vString, "none") == 0 )
					tb->mAuthType	= AuthNone;
				else if( strcmp(tb->mJsAuthType->vString, "user") == 0 )
					tb->mAuthType	= AuthUser;
				else if( strcmp(tb->mJsAuthType->vString, "password") == 0 )
					tb->mAuthType	= AuthPasswd;
				else if( strcmp(tb->mJsAuthType->vString, "external") == 0 )
					tb->mAuthType	= AuthExternURL;
				else
					return taskResponse(q, ErrorParameter, "%s. authType value", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsStorageCrypt ) {
				if( strcmp(tb->mJsStorageCrypt->vString, "none") == 0 )
					tb->mCrypt	= CryptAlgorithmNone;
				else if( strcmp(tb->mJsStorageCrypt->vString, "3des") == 0 )
					tb->mCrypt	= CryptAlgorithm3DES;
				else if( strcmp(tb->mJsStorageCrypt->vString, "aes192") == 0 )
					tb->mCrypt	= CryptAlgorithmAES192;
			}
			switch(tb->mAuthType) {
			case AuthNone:
				if( tb->mJsAuthData )
					return taskResponse(q, ErrorParameter, "%s. authNone authData value", codeToMessgae(ErrorParameter));
				break;
			case AuthUser:
				if( tb->mJsAuthData )
					return taskResponse(q, ErrorParameter, "%s. authUser authData value", codeToMessgae(ErrorParameter));
				break;
			case AuthPasswd:
				if( !tb->mJsAuthData )
					return taskResponse(q, ErrorParameter, "%s. authPassword authData is not specified", codeToMessgae(ErrorParameter));
				break;
			case AuthExternURL:
				if( !tb->mJsAuthData )
					return taskResponse(q, ErrorParameter, "%s. authExternal authData is not specified", codeToMessgae(ErrorParameter));
				break;
			}

			if( tb->mJsBlockSize ) {
				if( tb->mJsBlockSize->vInt < SST_MINIMUM_BLOCKSIZE)
					return taskResponse(q, ErrorParameter, "%s. storageBlockSize is too small.", codeToMessgae(ErrorParameter));
				if( tb->mJsBlockSize->vInt < SST_MAXIMUM_BLOCKSIZE)
					return taskResponse(q, ErrorParameter, "%s. storageBlockSize is too large.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsBlockCount ) {
				if( tb->mJsBlockCount->vInt < SST_MINIMUM_BLOCKCOUNT)
					return taskResponse(q, ErrorParameter, "%s. storageBlockCount is too small.", codeToMessgae(ErrorParameter));
				if( tb->mJsBlockCount->vInt < SST_MAXIMUM_BLOCKCOUNT)
					return taskResponse(q, ErrorParameter, "%s. storageBlockCount is too large.", codeToMessgae(ErrorParameter));
			}

			QueryKeyType aref;
			aref.mId	= tb->mJsAdminAID ? tb->mJsAdminAID->vString : NULL;
			aref.mName	= tb->mJsAdminUID ? tb->mJsAdminUID->vString : NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryAccount(&aref);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "administrator account load faild. %s", res->getMessagePtr());
			}
			tb->mAccount = (ResultAccount*)res->mResult;
			res->mResult = NULL;

			ServiceInfo srv;
			memset(&srv, 0, sizeof(srv));

			srv.id			= NULL;
			srv.name		= (char*)(tb->mJsName->vString);
			srv.title		= (char*)(tb->mJsTitle ? tb->mJsTitle->vString : NULL);
			srv.desc		= (char*)(tb->mJsDesc ? tb->mJsDesc->vString : NULL);
			srv.version		= (char*)(tb->mJsVersion ? tb->mJsVersion->vString : NULL);
			srv.adminId		= (char*)(tb->mAccount->mAccount.id);
			srv.refURL		= (char*)(tb->mJsRefURL ? tb->mJsRefURL->vString : NULL);
			srv.scope		= tb->mScope;
			srv.defaultGroup		= NULL;
			srv.storageBlockSize	= tb->mJsBlockSize  ? tb->mJsBlockSize->vInt : SST_DEFAULT_BLOCKSIZE;
			srv.storageBlockCount	= tb->mJsBlockCount ? tb->mJsBlockCount->vInt : SST_DEFAULT_BLOCKCOUNT;
			srv.storageCrypt		= tb->mCrypt;
			srv.authType	= tb->mAuthType;
			srv.authData	= (char*)(tb->mJsAuthData ? tb->mJsAuthData->vString : NULL);

			QnDBRegister *qn = new QnDBRegister(task);
			qn->registService(&srv);
			return task->call(qn, step+1);
		}
		break;

	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "service info registration failed. %s", res->getMessagePtr());
			}

			tb->mServiceId	= (ResultId*)res->mResult;
			res->mResult	= NULL;

			if( tb->mJsGroupName ) {
				QueryKeyType sref;
				sref.mId	= tb->mServiceId->mId;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryService(&sref);
				return task->call(qn, step+1);
			}
		}
	case 3:
		{
			if( step == 3 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "service info load failed. %s", res->getMessagePtr());
				}

				tb->mService	= (ResultService*)res->mResult;
				res->mResult	= NULL;
			}
			step	= 3;

			if( tb->mJsGroupName ) {
				QueryKeyType sref;
				sref.mId	= tb->mServiceId->mId;
				sref.mName	= NULL;
				GroupInfo g;
				memset(&g, 0, sizeof(g));
				g.id			= NULL;
				g.serviceId		= tb->mServiceId->mId;
				g.name			= (char*)tb->mJsGroupName->vString;
				g.status		= GroupNormal;
				g.adminId		= (char*)tb->mAccount->mAccount.id;
				g.description	= (char*)"default group";
				QnDBRegister *qn = new QnDBRegister(task);
				qn->registGroup(&sref, &g);
				return task->call(qn, step+1);
			}
		}
	case 4:
		{
			if( step == 4 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "default group registration failed. %s", res->getMessagePtr());
				}

				tb->mGroupId	= (ResultId*)res->mResult;
				res->mResult	= NULL;
			}
			step = 4;

			if( tb->mGroupId ) {
				ServiceInfo srv;
				srv = tb->mService->mService;
				srv.defaultGroup	= tb->mGroupId->mId;

				QnDBRegister *qn = new QnDBRegister(task);
				qn->registService(&srv);
				return task->call(qn, step+1);
			}
		}
	case 5:
		{
			if( step == 5 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "service info update failed. %s", res->getMessagePtr());
				}
			}

			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("sid", 3) ||
				!jb.valueString(tb->mServiceId->mId) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
				return false;
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

bool CommandProc::cmdSetService(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBRegService *tb = (TBRegService *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBRegService();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			pjson::json *jsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = jsCommand->get("setService");
			if( !vCmd ) {
				return taskResponse(q, ErrorParameter, "%s.\"setService\" is not object.", codeToMessgae(ErrorParameter));
			}

			tb->mJsKeyId		= jsCommand->get("sid", vCmd);
			tb->mJsKeyName		= jsCommand->get("serviceName", vCmd);
			pjson::value *vSet	= jsCommand->get("set", vCmd);
			if( vSet ) {
				if( vSet->vt != pjson::vt_object )
					return taskResponse(q, ErrorParameter, "%s. \"set\" type.", codeToMessgae(ErrorParameter));
				tb->mJsName			= jsCommand->get("serviceName", vSet);
				tb->mJsTitle		= jsCommand->get("title", vSet);
				tb->mJsDesc			= jsCommand->get("description", vSet);
				tb->mJsVersion		= jsCommand->get("version", vSet);
				pjson::value *vAdmin = jsCommand->get("admin", vSet);
				if( vAdmin ) {
					if( vAdmin->vt != pjson::vt_object )
						return taskResponse(q, ErrorParameter, "%s. \"admin\" type.", codeToMessgae(ErrorParameter));
					tb->mJsAdminAID	= jsCommand->get("aid", vAdmin);
					tb->mJsAdminUID	= jsCommand->get("userId", vAdmin);
				}
				tb->mJsRefURL		= jsCommand->get("refURL", vSet);
				tb->mJsScope		= jsCommand->get("scope", vSet);
				tb->mJsGroupName	= jsCommand->get("defaultGroup", vSet);
				tb->mJsBlockSize	= jsCommand->get("storageBlockSize", vSet);
				tb->mJsBlockCount	= jsCommand->get("storageBlockCount", vSet);
				tb->mJsStorageCrypt	= jsCommand->get("storageCrypt", vSet);
				tb->mJsAuthType		= jsCommand->get("authType", vSet);
				tb->mJsAuthData		= jsCommand->get("authData", vSet);
			}
			if( tb->mJsKeyId && tb->mJsKeyId->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsKeyName && tb->mJsKeyName->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"name\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsName && tb->mJsName->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"name\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsTitle && tb->mJsTitle->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"title\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsDesc && tb->mJsDesc->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"description\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsVersion && tb->mJsVersion->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"version\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAdminAID && tb->mJsAdminAID->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"admin.aid\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAdminUID && tb->mJsAdminUID->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"admin.userId\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsRefURL && tb->mJsRefURL->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"refURL\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsScope && tb->mJsScope->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"scope\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsGroupName && tb->mJsGroupName->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"defaultGroup\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsBlockSize && tb->mJsBlockSize->vt != pjson::vt_int)
				return taskResponse(q, ErrorParameter, "%s. \"storageBlockSize\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsBlockCount && tb->mJsBlockCount->vt != pjson::vt_int )
				return taskResponse(q, ErrorParameter, "%s. \"storageBlockCount\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsStorageCrypt && tb->mJsStorageCrypt->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"storageCrypt\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAuthType && tb->mJsAuthType->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"authType\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAuthData && tb->mJsAuthData->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"authData\" type.", codeToMessgae(ErrorParameter));
			taskLog(q, LOG_DEBUG, "json data ok");

			if( !tb->mJsKeyName && !tb->mJsKeyId )
				return taskResponse(q, ErrorParameter, "%s. service is not specified.", codeToMessgae(ErrorParameter));
			if( tb->mJsScope ) {
				if( strcmp(tb->mJsScope->vString, "private") == 0 )
					tb->mScope	= ScopePrivate;
				else if( strcmp(tb->mJsScope->vString, "public") == 0 )
					tb->mScope	= ScopePublic;
				else
					return taskResponse(q, ErrorParameter, "%s. \"scope\" value.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsAuthType ) {
				if( strcmp(tb->mJsAuthType->vString, "none") == 0 )
					tb->mAuthType	= AuthNone;
				else if( strcmp(tb->mJsAuthType->vString, "user") == 0 )
					tb->mAuthType	= AuthUser;
				else if( strcmp(tb->mJsAuthType->vString, "password") == 0 )
					tb->mAuthType	= AuthPasswd;
				else if( strcmp(tb->mJsAuthType->vString, "external") == 0 )
					tb->mAuthType	= AuthExternURL;
				else
					return taskResponse(q, ErrorParameter, "%s. \"authType\" value.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsStorageCrypt ) {
				if( strcmp(tb->mJsStorageCrypt->vString, "none") == 0 )
					tb->mCrypt	= CryptAlgorithmNone;
				else if( strcmp(tb->mJsStorageCrypt->vString, "3des") == 0 )
					tb->mCrypt	= CryptAlgorithm3DES;
				else if( strcmp(tb->mJsStorageCrypt->vString, "aes192") == 0 )
					tb->mCrypt	= CryptAlgorithmAES192;
				else
					return taskResponse(q, ErrorParameter, "%s. \"storageCrypt\" value.", codeToMessgae(ErrorParameter));
			}

			if( tb->mJsBlockSize ) {
				if( tb->mJsBlockSize->vInt < SST_MINIMUM_BLOCKSIZE)
					return taskResponse(q, ErrorParameter, "%s. storageBlockSize is too small.", codeToMessgae(ErrorParameter));
				if( tb->mJsBlockSize->vInt < SST_MAXIMUM_BLOCKSIZE)
					return taskResponse(q, ErrorParameter, "%s. storageBlockSize is too large.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsBlockCount ) {
				if( tb->mJsBlockCount->vInt < SST_MINIMUM_BLOCKCOUNT)
					return taskResponse(q, ErrorParameter, "%s. storageBlockCount is too small.", codeToMessgae(ErrorParameter));
				if( tb->mJsBlockCount->vInt < SST_MAXIMUM_BLOCKCOUNT)
					return taskResponse(q, ErrorParameter, "%s. storageBlockCount is too large.", codeToMessgae(ErrorParameter));
			}

			switch(tb->mAuthType) {
			case AuthNone:
				if( tb->mJsAuthData )
					return taskResponse(q, ErrorParameter, "%s. authNone authData value.", codeToMessgae(ErrorParameter));
				break;
			case AuthUser:
				if( tb->mJsAuthData )
					return taskResponse(q, ErrorParameter, "%s. authUser authData value.", codeToMessgae(ErrorParameter));
				break;
			case AuthPasswd:
				if( !tb->mJsAuthData )
					return taskResponse(q, ErrorParameter, "%s. authPassword authData is not specified.", codeToMessgae(ErrorParameter));
				break;
			case AuthExternURL:
				if( !tb->mJsAuthData )
					return taskResponse(q, ErrorParameter, "%s. authExternal authData is not specified.", codeToMessgae(ErrorParameter));
				break;
			}

			if( tb->mJsAdminAID || tb->mJsAdminUID ) {
				QueryKeyType aref;
				aref.mId	= tb->mJsAdminAID ? tb->mJsAdminAID->vString : NULL;
				aref.mName	= tb->mJsAdminUID ? tb->mJsAdminUID->vString : NULL;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryAccount(&aref);
				return task->call(qn, step+1);
			}
		}
	case 1:
		{
			if( step == 1 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "Administrator account load failed. %s", res->getMessagePtr());
				}
				tb->mAccount = (ResultAccount*)res->mResult;
				res->mResult = NULL;
			}
			step	= 1;

			QueryKeyType sref;
			sref.mId	= tb->mJsKeyId   ? tb->mJsKeyId->vString   : NULL;
			sref.mName	= tb->mJsKeyName ? tb->mJsKeyName->vString : NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryService(&sref);
			return task->call(qn, step+1);
		}
		break;
	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Service info load failed. %s", res->getMessagePtr());
			}
			tb->mService	= (ResultService*)res->mResult;
			res->mResult = NULL;

			if( tb->mJsGroupName ) {
				QueryKeyType sref;
				QueryKeyType gref;
				sref.mId	= tb->mService->mService.id;
				sref.mName	= NULL;
				gref.mId	= NULL;
				gref.mName	= tb->mJsGroupName->vString;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryGroup(&sref, &gref);
				return task->call(qn, step+1);
			}
		}
	case 3:
		{
			if( step == 3 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "Group info load failed. %s", res->getMessagePtr());
				}
				tb->mGroup	= (ResultGroup*)res->mResult;
				res->mResult = NULL;
			}
			step	= 3;

			ServiceInfo srv = tb->mService->mService;

			if( tb->mJsName )
				srv.name		= (char*)(tb->mJsName->vString);
			if( tb->mJsTitle )
				srv.title		= (char*)(tb->mJsTitle ? tb->mJsTitle->vString : NULL);
			if( tb->mJsDesc )
				srv.desc		= (char*)(tb->mJsDesc ? tb->mJsDesc->vString : NULL);
			if( tb->mJsVersion )
				srv.version		= (char*)(tb->mJsVersion ? tb->mJsVersion->vString : NULL);
			if( tb->mAccount )
				srv.adminId		= (char*)(tb->mAccount->mAccount.id);
			if( tb->mJsRefURL )
				srv.refURL		= (char*)(tb->mJsRefURL ? tb->mJsRefURL->vString : NULL);
			if( tb->mJsScope )
				srv.scope		= tb->mScope;
			if( tb->mGroup ) {
				if( srv.defaultGroup ) {
					if( strcmp(srv.defaultGroup, tb->mGroup->mGroup.id) != 0 ) {
						return taskResponse(q, ErrorParameter, "%s. \"defaultGroup\" is can not change.", codeToMessgae(ErrorParameter));
					}
				}
			}
			if( tb->mJsBlockSize )
				srv.storageBlockSize	= tb->mJsBlockSize->vInt;
			if( tb->mJsBlockCount )
				srv.storageBlockCount	= tb->mJsBlockCount->vInt;
			if( tb->mJsStorageCrypt )
				srv.storageCrypt		= tb->mCrypt;
			if( tb->mJsAuthType )
				srv.authType			= tb->mAuthType;
			if( tb->mJsAuthData )
				srv.authData			= (char*)(tb->mJsAuthData->vString);

			QnDBRegister *qn = new QnDBRegister(task);
			qn->registService(&srv);
			return task->call(qn, step+1);
		}
		break;

	case 4:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Service info update failed. %s", res->getMessagePtr());
			}

			tb->mServiceId	= (ResultId*)res->mResult;
			res->mResult	= NULL;

			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("sid", 3) ||
				!jb.valueString(tb->mServiceId->mId) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d", jb.getError());
				return false;
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

bool CommandProc::cmdRmService(QueueNode *q)
{
	return taskResponse(q, ErrorNotImpl, "%s.", codeToMessgae(ErrorNotImpl));
}

class TBShowService : public TaskBuffer {
public:
	TBShowService() {
		mResService	= NULL;
		mResPWList	= NULL;
		mResPRList	= NULL;
	}
	virtual ~TBShowService() {
		if( mResService )	free(mResService);
		if( mResPWList )	free(mResPWList);
		if( mResPRList )	free(mResPRList);
	}
	ResultService *	mResService;
	ResultList	  * mResPWList;
	ResultList	  * mResPRList;
};

bool CommandProc::cmdShowService(QueueNode *q)
{
	Task *task = q->getTask();

	TBShowService *tb = (TBShowService *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBShowService();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			pjson::json *jsCommand	= q->detouchJSONMessage();

			QueryKeyType sref;
			sref.mId	= NULL;
			sref.mName	= NULL;
			pjson::value *vCmd = jsCommand->get("showService");
			if( vCmd ) {
				pjson::value *vSID	= jsCommand->get("sid", vCmd);
				pjson::value *vName	= jsCommand->get("serviceName", vCmd);
				if( vSID && vSID->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
				}
				if( vName && vName->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
				}
				if( vSID )
					sref.mId	= vSID->vString;
				if( vName )
					sref.mName	= vName->vString;
			}

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryService(&sref);
			return task->call(qn, step+1);
		}
		break;

	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Service info load failed. %s'", res->getMessagePtr());
			}

			tb->mResService	= (ResultService*)(res->mResult);
			res->mResult	= NULL;

			QueryKeyType 	sref;
			QueryRangeType	key;
			sref.mId	= tb->mResService->mService.id;
			sref.mName	= NULL;
			key.mStart	= 0;
			key.mCount	= -1;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryServicePWList(&sref, &key);
			return task->call(qn, step+1);
		}
		break;

	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				if( code != ErrorNotFound ) {
					return taskResponse(q, code, "Service PWList load failed. %s", res->getMessagePtr());
				}
				tb->mResPWList = NULL;
			} else {
				tb->mResPWList = (ResultList*)(res->mResult);
				res->mResult	= NULL;
			}

			QueryKeyType 	sref;
			QueryRangeType	key;
			sref.mId	= tb->mResService->mService.id;
			sref.mName	= NULL;
			key.mStart	= 0;
			key.mCount	= -1;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryServicePRList(&sref, &key);
			return task->call(qn, step+1);
		}
		break;

	case 3:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				if( code != ErrorNotFound ) {
					return taskResponse(q, code, "%Service PRList load failed. %s", res->getMessagePtr());
				}
				tb->mResPRList = NULL;
			} else {
				tb->mResPRList = (ResultList*)(res->mResult);
				res->mResult	= NULL;
			}

			const char *authType = NULL;
			const char *authData = NULL;
			switch(tb->mResService->mService.authType) {
			case AuthNone:
				authType = "none";
				authData = "";
				break;
			case AuthPasswd:
				authType = "password";
				authData = "****";
				break;
			case AuthExternURL:
				authType = "external";
				authData = tb->mResService->mService.authData;
				break;
			default:
				authType = "unknown";
				authData = "";
				break;
			}
			const char *storageCrypt;
			switch(tb->mResService->mService.storageCrypt) {
			case CryptAlgorithmNone:	storageCrypt = "none";		break;
			case CryptAlgorithm3DES:	storageCrypt = "3des";		break;
			case CryptAlgorithmAES192:	storageCrypt = "aes192";	break;
			default:					storageCrypt = "unknown";	break;
			}
			const char *scope;
			switch(tb->mResService->mService.scope) {
			case ScopePrivate:
				scope	= "private";
				break;
			case ScopePublic:
				scope	= "public";
				break;
			default:
				scope	= "unknown";
				break;
			}
			char _createTime[256];
			const char *createTime = "";
			if( tb->mResService->mService.createTime != 0 ) {
				struct tm *tmp = gmtime(&tb->mResService->mService.createTime);
				strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", tmp);
				createTime = _createTime;
			}
			char _updateTime[256];
			const char *updateTime = "";
			if( tb->mResService->mService.updateTime != 0 ) {
				struct tm *tmp = gmtime(&tb->mResService->mService.updateTime);
				strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", tmp);
				updateTime = _updateTime;
			}

			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("sid", 3) ||
				!jb.valueString(tb->mResService->mService.id) ||
				!jb.addObjectProp("name", 4) ||
				!jb.valueString(tb->mResService->mService.name ? tb->mResService->mService.name : "") ||
				!jb.addObjectProp("title", 5) ||
				!jb.valueString(tb->mResService->mService.title ? tb->mResService->mService.title : "") ||
				!jb.addObjectProp("description", 11) ||
				!jb.valueString(tb->mResService->mService.desc ? tb->mResService->mService.desc : "") ||
				!jb.addObjectProp("version", 7) ||
				!jb.valueString(tb->mResService->mService.version ? tb->mResService->mService.version : "") ||
				!jb.addObjectProp("adminId", 7) ||
				!jb.valueString(tb->mResService->mService.adminId ? tb->mResService->mService.adminId : "") ||
				!jb.addObjectProp("status", 6) ||
				!jb.valueInt(tb->mResService->mService.status) ||
				!jb.addObjectProp("authType", 8) ||
				!jb.valueString(authType) ||
				!jb.addObjectProp("authData", 8) ||
				!jb.valueString(authData) ||
				!jb.addObjectProp("ref", 3) ||
				!jb.valueString(tb->mResService->mService.refURL ? tb->mResService->mService.refURL : "") ||
				!jb.addObjectProp("scope", 5) ||
				!jb.valueString(scope) ||
				!jb.addObjectProp("storage", 7) ||
				!jb.beginObject() ||
				!jb.addObjectProp("blockSize", 9) ||
				!jb.valueInt(tb->mResService->mService.storageBlockSize) ||
				!jb.addObjectProp("blockCount", 10) ||
				!jb.valueInt(tb->mResService->mService.storageBlockCount) ||
				!jb.addObjectProp("crypt", 5) ||
				!jb.valueString(storageCrypt) ||
				!jb.endObject() ||
				!jb.addObjectProp("defaultGroup", 12) ||
				!jb.valueString(tb->mResService->mService.defaultGroup) ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
				return false;
			}

			if( (tb->mResPWList && tb->mResPWList->mList.values > 0) ||
				(tb->mResPWList && tb->mResPWList->mList.values > 0) ) {
				if(	!jb.addObjectProp("permission",10) ||
					!jb.beginObject() ) {
					taskLog(q, LOG_ERR, "json builder failed(2). code=%d.", jb.getError());
					return false;
				}
				if( tb->mResPWList && tb->mResPWList->mList.values > 0 ) {
					if(	!jb.addObjectProp("writable",8) ||
						!jb.beginArray() ) {
						taskLog(q, LOG_ERR, "json builder failed(3). code=%d.", jb.getError());
						return false;
					}
					size_t i;
					for( i = 0; i < tb->mResPWList->mList.values; i++ ) {
						if( !jb.addArrayContent() ||
							!jb.valueString(tb->mResPWList->mList.value[i]) ) {
							taskLog(q, LOG_ERR, "json builder failed(4). code=%d.", jb.getError());
							return false;
						}
					}
					if( !jb.endArray() ) {
						taskLog(q, LOG_ERR, "json builder failed(5). code=%d.", jb.getError());
						return false;
					}
				}
				if( tb->mResPRList && tb->mResPRList->mList.values > 0 ) {
					if(	!jb.addObjectProp("readable",8) ||
						!jb.beginArray() ) {
						taskLog(q, LOG_ERR, "json builder failed(6). code=%d.", jb.getError());
						return false;
					}
					size_t i;
					for( i = 0; i < tb->mResPRList->mList.values; i++ ) {
						if( !jb.addArrayContent() ||
							!jb.valueString(tb->mResPRList->mList.value[i]) ) {
							taskLog(q, LOG_ERR, "json builder failed(7). code=%d.", jb.getError());
							return false;
						}
					}
					if( !jb.endArray() ) {
						taskLog(q, LOG_ERR, "json builder failed(8). code=%d.", jb.getError());
						return false;
					}
				}
				if(	!jb.endObject() ) {
					taskLog(q, LOG_ERR, "json builder failed(9). code=%d.", jb.getError());
					return false;
				}
			}
			if(	!jb.addObjectProp("createTime", 10) ||
				!jb.valueString(createTime) ||
				!jb.addObjectProp("updateTime", 10) ||
				!jb.valueString(updateTime) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(10). code=%d.", jb.getError());
				return false;
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

class TBLsService : public TaskBuffer {
public:
	TBLsService() {
		mJsCommand	= NULL;
		mJsFrom		= NULL;
		mJsCount	= NULL;
	}
	virtual ~TBLsService() {
		if( mJsCommand )	delete mJsCommand;
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsFrom;
	pjson::value *	mJsCount;
};

bool CommandProc::cmdLsService(QueueNode *q)
{
	Task *task = q->getTask();

	TBLsService *tb = (TBLsService *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBLsService();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("lsService");
			if( vCmd ) {
				tb->mJsFrom		= tb->mJsCommand->get("from", 	vCmd);
				tb->mJsCount	= tb->mJsCommand->get("count", vCmd);

				if( tb->mJsFrom && tb->mJsFrom->vt != pjson::vt_int ) {
					return taskResponse(q, ErrorParameter, "%s. \"from\" type.",codeToMessgae(ErrorParameter));
				}
				if( tb->mJsCount && tb->mJsCount->vt != pjson::vt_int ) {
					return taskResponse(q, ErrorParameter, "%s. \"count\" type.",codeToMessgae(ErrorParameter));
				}
			}
			QueryKeyType 	sref;
			QueryRangeType	key;
			sref.mId	= NULL;
			sref.mName	= NULL;
			key.mStart	= tb->mJsFrom ? tb->mJsFrom->vInt : 0;
			key.mCount	= tb->mJsCount ? tb->mJsCount->vInt : -1;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryServiceList(&sref, &key);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Service list load failed. %s", res->getMessagePtr());
			}
			ResultList *rs = (ResultList*)(res->mResult);
			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("from", 4) ||
				!jb.valueInt(tb->mJsFrom?tb->mJsFrom->vInt:0) ||
				!jb.addObjectProp("services", 8) ||
				!jb.beginArray() ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
				return false;
			}

			for( size_t i = 0; i < rs->mList.count; i++ ) {
				if( !jb.addArrayContent() ||
					!jb.valueString(rs->mList.value[i]) ) {
					taskLog(q, LOG_ERR, "json builder failed(2). code=%d.", jb.getError());
					return false;
				}
			}

			if( !jb.endArray() ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(3). code=%d.", jb.getError());
				return false;
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

class TBLsServiceGroup : public TaskBuffer {
public:
	TBLsServiceGroup() {
		mJsCommand	= NULL;
		mJsSID		= NULL;
		mJsSName	= NULL;
		mJsFrom		= NULL;
		mJsCount	= NULL;
	}
	virtual ~TBLsServiceGroup() {
		if( mJsCommand )	delete mJsCommand;
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsSID;
	pjson::value *	mJsSName;
	pjson::value *	mJsFrom;
	pjson::value *	mJsCount;
};

bool CommandProc::cmdLsServiceGroup(QueueNode *q)
{
	Task *task = q->getTask();

	TBLsServiceGroup *tb = (TBLsServiceGroup *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBLsServiceGroup();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("lsServiceGroup");
			if( vCmd ) {
				tb->mJsSID		= tb->mJsCommand->get("sid", vCmd);
				tb->mJsSName	= tb->mJsCommand->get("serviceName", vCmd);
				tb->mJsFrom		= tb->mJsCommand->get("from", vCmd);
				tb->mJsCount	= tb->mJsCommand->get("count", vCmd);

				if( tb->mJsSID && tb->mJsSID->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
				if( tb->mJsSName && tb->mJsSName->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
				if( tb->mJsFrom && tb->mJsFrom->vt != pjson::vt_int )
					return taskResponse(q, ErrorParameter, "%s. \"from\" type.", codeToMessgae(ErrorParameter));
				if( tb->mJsCount && tb->mJsCount->vt != pjson::vt_int )
					return taskResponse(q, ErrorParameter, "%s. \"count\" type.", codeToMessgae(ErrorParameter));
			}

			QueryKeyType 	sref;
			QueryRangeType	key;
			sref.mId	= (char*)(tb->mJsSID ? tb->mJsSID->vString : NULL);
			sref.mName	= (char*)(tb->mJsSName ? tb->mJsSName->vString : NULL);
			key.mStart	= tb->mJsFrom ? tb->mJsFrom->vInt : 0;
			key.mCount	= tb->mJsCount ? tb->mJsCount->vInt : -1;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryServiceGroupList(&sref, &key);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Service group list load failed. %s", res->getMessagePtr());
			}
			ResultList *rs = (ResultList*)(res->mResult);
			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("from", 4) ||
				!jb.valueInt(tb->mJsFrom?tb->mJsFrom->vInt:0) ||
				!jb.addObjectProp("groups", 6) ||
				!jb.beginArray() ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
				return false;
			}

			for( size_t i = 0; i < rs->mList.values; i++ ) {
				if( !jb.addArrayContent() ||
					!jb.valueString(rs->mList.value[i]) ) {
					taskLog(q, LOG_ERR, "json builder failed(2). code=%d.", jb.getError());
					return false;
				}
			}

			if( !jb.endArray() ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(3). code=%d.", jb.getError());
				return false;
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}


}




