/*
 * connection.cpp
 *
 *  Created on: 2012/07/03
 *      Author: yasuoki
 */
#include <memory.h>
#include <malloc.h>
#include <syslog.h>

#include <unistd.h>
#include <crypt.h>
#include <assert.h>
#include <time.h>

#include "session.h"
#include "sentinel.h"
#include "lock.h"

namespace SST {

Session::Session()
{
	mInUse			= false;
	mFd				= 0;
	mAccountId[0]	= 0;
	mUserId[0]		= 0;
	mDeviceId[0]	= 0;
	mServiceId[0]	= 0;
	mRecvEvent		= NULL;
	mNextList		= NULL;
	mRecvTask		= NULL;
	mSendTask		= NULL;
	mTaskCnt		= 0;
	mStatus			= SessionStatusWait;
	srand((unsigned int)time(NULL));
}

Session::~Session()
{
}

bool  Session::open(Sentinel *sentinel, int fd, sockaddr_in *addr)
{
	mSentinel		= sentinel;
	mInUse			= true;
	mFd				= fd;
	mPeerAddr		= *addr;
	mNonce[0]		= 0;
	mAccountId[0]	= 0;
	mAccountLevel	= AccountSystemService;
	mUserId[0]		= 0;
	mDeviceId[0]	= 0;
	mServiceId[0]	= 0;
	mSessionId[0]	= 0;
	mRecvTask		= NULL;
	mSendTask		= NULL;
	mTaskCnt		= 0;
	mStatus			= SessionStatusAccept;
	mStart			= time(NULL);
	mLastReciveTime	= mStart;

	mSentinel->log(LOG_INFO, "Session::open fd=%d addr=%s:%d", mFd, inet_ntoa(mPeerAddr.sin_addr), ntohs(mPeerAddr.sin_port));
	return true;
}

void Session::close()
{
	mSentinel->log(LOG_INFO, "Session::close fd=%d addr=%s:%d", mFd, inet_ntoa(mPeerAddr.sin_addr), ntohs(mPeerAddr.sin_port));

	if( mFd ) ::close(mFd);
	mFd	= 0;
	mInUse	= false;
	if( mRecvEvent ) {
		struct event *e = mRecvEvent;
		mRecvEvent	= NULL;
		event_free(e);
	}
	mNonce[0]		= 0;
	mAccountId[0]	= 0;
	mAccountLevel	= AccountSystemService;
	mUserId[0]		= 0;
	mDeviceId[0]	= 0;
	mServiceId[0]	= 0;
	mSessionId[0]	= 0;
	mRecvTask		= NULL;
	mSendTask		= NULL;
//	mTaskCnt		= 0;


	mStatus			= SessionStatusWait;
}

static const char *hex64="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
char *base64(char *src, int len, char *dest, int destLen)
{
	char *p1 = dest;
	while( len >= 3 ) {
		if( destLen < 5 ) return NULL;
		char c0 = *src++;
		char c1 = *src++;
		char c2 = *src++;
		*p1++ = hex64[(c0>>2)&0x3f];
		*p1++ = hex64[((c0<<4)&0x30)|((c1>>4)&0xf)];
		*p1++ = hex64[((c1<<2)&0x3c)|((c2>>6)&0x3)];
		*p1++ = hex64[c2&0x3f];
		len 	-= 3;
		destLen -= 4;
	}
	if( len == 1 ) {
		if( destLen < 5 ) return NULL;
		char c0 = *src++;
		*p1++ = hex64[(c0>>2)&0x3f];
		*p1++ = hex64[(c0<<4)&0x30];
		*p1++ = '=';
		*p1++ = '=';
	}
	if( len == 2 ) {
		if( destLen < 5 ) return NULL;
		char c0 = *src++;
		char c1 = *src++;
		*p1++ = hex64[(c0>>2)&0x3f];
		*p1++ = hex64[((c0<<4)&0x30)|((c1>>4)&0xf)];
		*p1++ = hex64[(c1<<2)&0x3c];
		*p1++ = '=';
	}
	*p1 = 0;
	return dest;
}

char * Session::createNonce()
{
	int hex[8];
	for( int i=0; i< 8; i++ ) {
		hex[i] = rand();
	}
	char b64hex[sizeof(hex)*2];
	base64((char*)hex, sizeof(hex), b64hex, sizeof(b64hex));
	char *key = crypt(b64hex, "$1$z4$zz4p");
	mSentinel->log(LOG_DEBUG, "nonce=%s %s\n", key, b64hex);
	strcpy(mNonce, &key[6]);
	return mNonce;
}

struct event *Session::assignEvent(struct event_base *base, event_callback_fn fn)
{
	if( mRecvEvent ) {
		event_assign(mRecvEvent, base, mFd, EV_READ, fn, this);
	} else {
		mRecvEvent = event_new(base, mFd, EV_READ, fn, this);
	}
	return mRecvEvent;
}

SessionStatus Session::getStatus() const
{
	READ_LOCK(const_cast<Session*>(this)->mLock);
	return mStatus;
}

void Session::setStatus(SessionStatus s)
{
	WRITE_LOCK(mLock);
	mStatus = s;
}

Task * Session::getReciveTask()
{
	READ_LOCK(const_cast<Session*>(this)->mLock);
	mLastReciveTime	= time(NULL);
	return mRecvTask;
}

Task * Session::getSendTask() const
{
	READ_LOCK(const_cast<Session*>(this)->mLock);
	return mSendTask;
}


int Session::getRunningTaskCount() const
{
	READ_LOCK(const_cast<Session*>(this)->mLock);
	int r = mTaskCnt;
	return r;
}

void Session::beginTask(Task *task)
{
	WRITE_LOCK(mLock);
	mTaskCnt++;
//	mSentinel->log(LOG_DEBUG, "Session::beginTask: session=%x task=%x running count=%d recv task=%x send task=%x", this, task, mTaskCnt, mRecvTask, mSendTask);
}

void Session::beginReciveTask(Task *task)
{
	assert(mRecvTask==NULL);
	WRITE_LOCK(mLock);
	mTaskCnt++;
	mRecvTask = task;
//	mSentinel->log(LOG_DEBUG, "Session::beginReciveTask: session=%x task=%x running count=%d recv task=%x send task=%x", this, task, mTaskCnt, mRecvTask, mSendTask);
}

void Session::beginSendTask(Task *task)
{
	assert(mRecvTask==NULL);
	WRITE_LOCK(mLock);
	mTaskCnt++;
	mSendTask = task;
//	mSentinel->log(LOG_DEBUG, "Session::beginSendTask: session=%x task=%x running count=%d recv task=%x send task=%x", this, task, mTaskCnt, mRecvTask, mSendTask);
}


void Session::endTask(Task *task)
{
	WRITE_LOCK(mLock);

	assert( mTaskCnt > 0 );
	mTaskCnt--;
	if( task == mSendTask )
		mSendTask = NULL;
	if( task == mRecvTask )
		mRecvTask = NULL;
//	mSentinel->log(LOG_DEBUG, "Session::endTask: session=%x task=%x running count=%d recv task=%x send task=%x", this, task, mTaskCnt, mRecvTask, mSendTask);
}

/////////////////////////////////////////////
SessionPool::SessionPool()
{
	mUseList		= NULL;
	mFreeList	= NULL;
	nSize		= 0;
	nUse		= 0;
	nFree		= 0;
}

SessionPool::~SessionPool()
{
	clear();
}

void  SessionPool::clear()
{
	while( mUseList ) {
		freeSession(mUseList);
	}
	Session *p = mFreeList;
	while(p) {
		Session *n = p->mNextList;
		delete p;
		p = n;
	}
	nSize	= 0;
	nUse	= 0;
	nFree	= 0;
	mUseList		= NULL;
	mFreeList	= NULL;
}

bool SessionPool::configure(Sentinel *obj, Conf *conf)
{
	mSentinel	= obj;
	nSize	= conf->maxConnections;
	if( nSize < nUse + nFree ) {
		while(nSize < nUse + nFree && nFree > 0) {
			Session *p = mFreeList;
			Session *n = p->mNextList;
			mFreeList	= n;
			nFree--;
			delete p;
		}
	}
	size_t nAdd = nSize - ( nUse + nFree);
	while( nAdd > 0 ) {
		Session *p = new Session();
		if( p == NULL ) {
			return false;
		}
		p->mNextList	= mFreeList;
		mFreeList	= p;
		nFree++;
		nAdd--;
	}
	return true;
}

void SessionPool::freeSession(Session *con)
{
	con->close();

	WRITE_LOCK(mLock);

	Session *c = mUseList;
	Session *p = NULL;
	while( c ) {
		if( c == con ) break;
		p = c;
		c = c->mNextList;
	}
	if( c == NULL ) return;

	if( p )
		p->mNextList = c->mNextList;
	else
		mUseList	 = c->mNextList;

	c->mNextList	= mFreeList;
	mFreeList = c;

	nUse--;
	nFree++;
}

Session *SessionPool::getSession()
{
	WRITE_LOCK(mLock);
	Session *a = mFreeList;

	mFreeList = a->mNextList;
	a->mNextList	= mUseList;
	mUseList		= a;

	nUse++;
	nFree--;
	return a;
}

Session * SessionPool::getUse()
{
	READ_LOCK(mLock);
	return mUseList;
}

Session * SessionPool::getNext(Session *con)
{
	READ_LOCK(mLock);
	return con->mNextList;
}

Session * SessionPool::findDeviceSession(const char *deviceId)
{
	READ_LOCK(mLock);
	Session *s = mUseList;
	while(s) {
		const char *sid = s->getDeviceId();
		if( sid && deviceId && strcmp(sid, deviceId) == 0 ) break;
		s = s->mNextList;
	}
	return s;
}

Session * SessionPool::findAcountSession(const char *accountId)
{
	READ_LOCK(mLock);
	Session *s = mUseList;
	while(s) {
		const char *sid = s->getAccountId();
		if( sid && strcmp(sid, accountId) == 0 ) break;
		s = s->mNextList;
	}
	return s;
}

bool SessionPool::diag(pjson::builder &jb)
{
	mSentinel->log(LOG_DEBUG, "SessionPool::diag begin");

	READ_LOCK(mLock);

	if( !jb.beginObject() ||
		!jb.addObjectProp("session",7) ||
		!jb.beginArray() ) {
		mSentinel->log(LOG_ERR, "SessionPool::diag json error 1. code=%d", jb.getError());
		return false;
	}

	Session *s = mUseList;
	while( s != NULL ) {
		const char *v;
		if( !jb.addArrayContent() ||
			!jb.beginObject() ||
//			!jb.addObjectProp("addr",4) ||
//			!jb.valueInt(s) ||
			!jb.addObjectProp("sessionId",9) ||
			!jb.valueString(s->getSessionId()) ) {
			mSentinel->log(LOG_ERR, "SessionPool::diag json error 2. code=%d", jb.getError());
			return false;
		}

		sockaddr_in caddr;
		s->getPeerAddress(&caddr);
		char buf[256];
		sprintf(buf, "%s:%d", inet_ntoa(caddr.sin_addr), ntohs(caddr.sin_port));
		if( !jb.addObjectProp("peer_addr", 9) ||
			!jb.valueString(buf) ) {
			mSentinel->log(LOG_ERR, "SessionPool::diag json error 4. code=%d", jb.getError());
			return false;
		}
		jb.addObjectProp("status",6);
		SessionStatus st = s->getStatus();
		switch(st) {
		case SessionStatusWait:			v = "wait";		break;
		case SessionStatusAccept:		v = "accept";	break;
		case SessionStatusConnect:		v = "connect";	break;
		case SessionStatusLogin:		v = "login";	break;
		default:						v = "unknown";	break;
		}
		if( !jb.valueString(v) ||
			!jb.addObjectProp("account",7) ||
			!jb.valueString( s->getAccountId() ) ||
			!jb.addObjectProp("accountLevel",12) ||
			!jb.valueInt(s->getAccountLevel()) ||
			!jb.addObjectProp("device",6) ||
			!jb.valueString( s->getDeviceId() ) ||
			!jb.addObjectProp("service",7) ||
			!jb.valueString( s->getServiceId() ) ||
			!jb.addObjectProp("user",4) ||
			!jb.valueString( s->getUserId() ) ||
			!jb.addObjectProp("task",4) ||
			!jb.valueInt( s->getRunningTaskCount() ) ||
			!jb.addObjectProp("commandQueue",12) ||
			!jb.valueInt( s->getCommandQueueCount() ) ||
			!jb.endObject() ) {
			mSentinel->log(LOG_ERR, "SessionPool::diag json error 5. code=%d", jb.getError());
			return false;
		}
		s = s->mNextList;
	}
	if( !jb.endArray() ||
		!jb.endObject() ) {
		mSentinel->log(LOG_ERR, "SessionPool::diag json error 6. code=%d", jb.getError());
		return false;
	}

	mSentinel->log(LOG_DEBUG, "SessionPool::diag end");

	return true;
}


}


