/*
 * staticBuffer.cpp
 *
 *  Created on: 2012/07/05
 *      Author: yasuoki
 */

#include "staticBuffer.h"
#include <stdio.h>
#include <stdarg.h>
#include <memory.h>
#include <malloc.h>
#include <assert.h>

#define STATIC_BLOCK_SIZE 256

namespace SST {

StaticBuffer::StaticBuffer()
{
	next		= NULL;
	nAlloc		= 0;
	nUse		= 0;
	ptr			= NULL;
	prevStore	= NULL;
}

StaticBuffer::~StaticBuffer()
{
	clear();
}

bool StaticBuffer::add(const char *data, ssize_t size)
{
	if( size == -1 ) {
		size  = data ? strlen(data) : 0;
	}
	return store(data, size) != NULL;
}

size_t StaticBuffer::size() const
{
	size_t nSize = nUse;
	StaticBuffer *p = next;
	while(p) {
		nSize += p->nUse;
		p = p->next;
	}
	return nSize;
}

size_t StaticBuffer::get(char *buff, size_t buffSize)
{
	StaticBuffer *p = this;
	size_t getSize = 0;
	while(p && buffSize) {
		size_t nc = p->nUse < buffSize ? p->nUse : buffSize;
		if( nc ) {
			memcpy(buff, p->ptr, nc);
			buff += nc;
			buffSize -= nc;
			getSize += nc;
		}
		p = p->next;
	}
	return getSize;
}

size_t StaticBuffer::getLine(char *buff, size_t buffSize)
{
	StaticBuffer *p = this;
	size_t getSize = 0;
	while(p && buffSize>1) {
		char *dp = p->ptr;
		size_t nc = 0;
		while( nc < p->nUse && nc < buffSize-1 ) {
			if( *dp == '\r' || *dp == '\n' ) {
				*buff++ = '\n';
				nc++;
				break;
			}
			*buff++ = *dp++;
			nc++;
		}
		buffSize -= nc;
		getSize += nc;
		p = p->next;
	}
	*buff = 0;
	return getSize;
}

size_t StaticBuffer::getAt(char *buff, size_t start, size_t len) const
{
	const StaticBuffer *p = this;
	while( p ) {
		if( p->nUse < start ) {
			start -= p->nUse;
			p = p->next;
		} else {
			break;
		}
	}
	size_t getSize = 0;
	if( p ) {
		while(p && len) {
			size_t nc = p->nUse-start < len ? p->nUse-start : len;
			if( nc ) {
				memcpy(buff, &p->ptr[start], nc);
				buff += nc;
				len	-= nc;
				getSize += nc;
				start = 0;
			}
			p = p->next;
		}
	}
	buff[getSize] = 0;
	return getSize;
}

const char * StaticBuffer::getPtr() const
{
	return ptr;
}

void StaticBuffer::clear()
{
	if( next ) delete next;
	if( ptr ) free(ptr);
	next		= NULL;
	nAlloc		= 0;
	nUse		= 0;
	ptr			= NULL;
	prevStore	= NULL;
}

bool StaticBuffer::empty() const
{
	return size() == 0;
}

const char * StaticBuffer::store(const char *data, size_t size)
{
	StaticBuffer *p = this;
	StaticBuffer *last = NULL;
	while( p ) {
		size_t free = p->nAlloc - p->nUse;
		if( free > size ) break;
		last	= p;
		p = p->next;
	}
	char *buff = NULL;
	if( !p ) {
		size_t ns = STATIC_BLOCK_SIZE;
		if( ns < size+1 ) {
			ns = ((size+1)/STATIC_BLOCK_SIZE+1)*STATIC_BLOCK_SIZE;
		}
		buff = (char*)malloc(ns);
		if( !ptr ) return NULL;
		if( nAlloc == 0 ) {
			p		= this;
		} else {
			p = new StaticBuffer();
			if( !p ) {
				free(buff);
				return NULL;
			}
			last->next	= p;
		}
		p->ptr		= buff;
		p->nAlloc	= ns;
	}
	memcpy(&p->ptr[p->nUse], data, size);
	prevStore	= &p->ptr[p->nUse];
	p->nUse += size;
	p->ptr[p->nUse] = 0;
	return prevStore;
}

const char * StaticBuffer::storePtr() const
{
	return prevStore;
}


}

#if 0

int _tmain(int argc, _TCHAR* argv[])
{

	SST::CyclicBuffer b0;

	_ASSERT(b0.size()==0);
	_ASSERT(b0.add("1234",4));
	_ASSERT(b0.ptrTest(0, 0, 4));

	char buff[257];
	_ASSERT(b0.get(&buff[0], 1)==1);
	_ASSERT(b0.ptrTest(0, 1, 4));
	_ASSERT(b0.get(&buff[1], 1)==1);
	_ASSERT(b0.ptrTest(0, 2, 4));
	_ASSERT(b0.get(&buff[2], 1)==1);
	_ASSERT(b0.ptrTest(0, 3, 4));
	_ASSERT(b0.get(&buff[3], 1)==1);
	_ASSERT(b0.ptrTest(0, 4, 4));
	_ASSERT(b0.get(&buff[4], 1)==0);
	_ASSERT(b0.ptrTest(0, 4, 4));
	buff[4] = 0;
	_ASSERT(strcmp(buff, "1234")==0);

	int i;
	for( i = 0; i < 256; i++ ) {
		if( i == 12 ) {
			int a=0;
		}
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.get(buff, 6) ==6);
		buff[6] = 0;
		_ASSERT(strcmp(buff, "123456")==0);
		_ASSERT(b0.get(buff, 6) ==4);
		buff[4] = 0;
		_ASSERT(strcmp(buff, "7890")==0);
	}
	for( i = 0; i < 256; i++ ) {
		if( i == 12 ) {
			int a=0;
		}
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.get(buff, 17) ==17);
		buff[17] = 0;
		_ASSERT(strcmp(buff, "12345678901234567")==0);
		_ASSERT(b0.get(buff, 3) ==3);
		buff[3] = 0;
		_ASSERT(strcmp(buff, "890")==0);
	}

	int n=0;
	for( i = 0; i < 256; i++ ) {
		if( i == 7 ) {
			int a=0;
		}
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.get(buff, 13) ==13);
		buff[13] = 0;
		_ASSERT(strncmp(buff, &"1234567890123456789012345678901234567890"[n],13)==0);
		n += 13;
		n = n % 10;
	}
	size_t bs;
	for( i = 0; i < 7; i++ ) {
		bs = b0.size();
		_ASSERT(bs == (7-i)*256);
		_ASSERT(b0.get(buff, 256) == 256);

		buff[256] = 0;
		for(int m=0; m<16; m++) {
			const char *pat = &"1234567890123456789012345678901234567890"[n];
			_ASSERT(strncmp(&buff[m*16], pat,16)==0);
			n += 16;
			n = n % 10;
		}
	}

	bs = b0.size();
	_ASSERT(bs == 0);

	const char *bp;
	for( i = 0; i < 256; i++ ) {
		_ASSERT(b0.add("1234567890", 10));
		bp = b0.getPtr();
		_ASSERT(strcmp(bp, "1234567890")==0);
	}
	for( i = 0; i < 256; i++ ) {
		_ASSERT(b0.add("1234567890", 10));
		_ASSERT(b0.add("1234567890", 10));
		bp = b0.getPtr();
		_ASSERT(strcmp(bp, "12345678901234567890")==0);
	}


	bs = b0.size();
	_ASSERT(bs == 0);

	return 0;
}


#endif

