/*
 * task.cpp
 *
 *  Created on: 2012/12/08
 *      Author: yasuoki
 */


#include "sentinel.h"
#include "task.h"
#include "db_port.h"
#include <memory.h>
#include <malloc.h>
#include <assert.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

namespace SST {

int Task::mTaskIdSeq = 0;

Task::Task(Sentinel *obj)
{
	mSentinel		= obj;
	mId				= 0;
	mConn			= NULL;
	mStatus			= TaskInit;
	mExecQueue		= NULL;
	mWaitQueue		= 0;
	mSubTasks		= 0;
	mCurStack		= NULL;
}

Task::~Task()
{
	mTaskStack.clear(true);
	mTaskBuffer.clear(true);
}

void Task::initTask(Session *conn, Task *baseTask, Process *baseProc, int baseFunc, int baseStep)
{
	if( mCurStack ) delete mCurStack;
	mConn			= conn;
	mId				= mTaskIdSeq++;
	mStatus			= TaskIdle;
	mExecQueue		= NULL;
	mWaitQueue		= 0;
	mSubTasks		= 0;
	mCurStack		= NULL;
	mDeviceId[0]	= 0;
	mServiceId[0]	= 0;
	mAccountId[0]	= 0;
	mAccountLevel	= AccountSystemService;
	mSequence		= 0;
	mUserId[0]		= 0;
	mPasswd[0]		= 0;
	if( baseTask ) {
		const char * deviceId	= baseTask->getDeviceId();
		const char * serviceId	= baseTask->getServiceId();
		const char * accountId	= baseTask->getAccountId();
		AccountLevel level		= baseTask->getAccountLevel();
		const char * userId		= baseTask->getUserId();
		const char * passwd		= baseTask->getPasswd();
		if( deviceId )	strcpy(mDeviceId, deviceId);
		if( serviceId )	strcpy(mServiceId, serviceId);
		if( accountId )	strcpy(mAccountId, accountId);
		if( userId )	strcpy(mUserId, userId);
		if( passwd )	strcpy(mPasswd, passwd);
		mAccountLevel	= level;
	}
	mTaskStack.clear(true);
	mTaskBuffer.clear(true);
}
const char * Task::getDeviceId() const
{
	if( mConn ) return mConn->getDeviceId();
	if( mDeviceId[0] )	return mDeviceId;
	return NULL;
}

const char * Task::getServiceId() const
{
	if( mConn ) return mConn->getServiceId();
	if( mServiceId[0] )	return mServiceId;
	return NULL;
}

const char * Task::getAccountId() const
{
	if( mConn ) return mConn->getAccountId();
	if( mAccountId[0] )	return mAccountId;
	return NULL;
}

AccountLevel Task::getAccountLevel() const
{
	if( mConn ) return mConn->getAccountLevel();
	return mAccountLevel;
}

const char * Task::getUserId() const
{
	if( mConn ) return mConn->getUserId();
	if( mUserId[0] )	return mUserId;
	return NULL;
}

const char * Task::getPasswd() const
{
	if( mConn ) return mConn->getPasswd();
	if( mPasswd[0] )	return mPasswd;
	return NULL;
}

bool Task::startTask(QueueNode *q)
{
//	mSentinel->log(LOG_DEBUG, "start task task=%x", this);
	assert(mStatus == TaskIdle);
	mStatus			= TaskActive;
	if( mConn ) {
		mConn->beginTask(this);
	}
	mCurStack	= new TaskStack(q->getProcess(), q->getFunction());
	putQueue(q);
	return true;
}

bool Task::startReciveTask(QueueNode *q)
{
//	mSentinel->log(LOG_DEBUG, "start revice task task=%x", this);
	assert(mStatus == TaskIdle);
	mStatus			= TaskActive;
	if( mConn ) {
		mConn->beginReciveTask(this);
	}
	mCurStack	= new TaskStack(q->getProcess(), q->getFunction());
	putQueue(q);
	return true;
}

bool Task::putQueue(QueueNode *q)
{
	Process *proc = q->getProcess();
	{
		WRITE_LOCK(mLock);
		mWaitQueue++;
	}
	if( !proc->putQueue(q) ) {
		{
			WRITE_LOCK(mLock);
			mWaitQueue--;
		}
		return false;
	}
	return true;
}

bool Task::call(QueueNode *q, int retStep, int retFunc, Process *retProc)
{
	mCurStack->mBaseStep	= retStep;
	if( retFunc )
		mCurStack->mBaseFunc	= retFunc;
	if( retProc )
		mCurStack->mBaseProc	= retProc;

	mTaskStack.addTop(mCurStack);
	mCurStack	= new TaskStack(q->getProcess(), q->getFunction());
	return putQueue(q);
}

void Task::beginProcess(QueueNode *q)
{
	mStatus			= TaskProcess;
	mExecQueue		= q;
}

void Task::endProcess(QueueNode *q)
{
	mStatus			= TaskActive;
	mExecQueue		= NULL;
	{
		WRITE_LOCK(mLock);
		mWaitQueue--;
	}
}

bool Task::endTask(QueueNode *q)
{
//	if( mConn && mConn->getReciveTask() == this ) {
//		mSentinel->log(LOG_DEBUG, "Task::endTask: end recive task task=%x", this);
//	} else {
//		mSentinel->log(LOG_DEBUG, "Task::endTask: end task task=%x", this);
//	}

	delete mCurStack;
	mCurStack = (TaskStack*)mTaskStack.removeTop();
	if( mCurStack ) {
		if( !q ) {
//			mSentinel->log(LOG_DEBUG, "Task:endTask: stack but q is null task=%x", this);
			q = new QnResult(this);
			((QnResult*)q)->setResult(0,ErrorOk);
		}
		q->setProcess(mCurStack->mBaseProc, mCurStack->mBaseFunc, mCurStack->mBaseStep);
		return putQueue(q);
	}

	if( q ) {
		mSentinel->log(LOG_INFO, "Task:endTask: q but stack is empty task=%d", mId);
		delete q;
	}

	if( mStatus != TaskIdle ) {
		{
			WRITE_LOCK(mLock);
			mStatus	= TaskIdle;
		}
		if( mConn ) {
			mConn->endTask(this);
		}
	}
	mConn			= NULL;
	mExecQueue		= NULL;
	mWaitQueue		= 0;
	mSubTasks		= 0;
	mCurStack 		= NULL;
	mTaskStack.clear(true);
	mTaskBuffer.clear(true);

	mSentinel->getTaskManager().removeTask(this);
	return true;
}

bool Task::setTaskBuffer(TaskBuffer *tb)
{
	mCurStack->mTaskBuffer	= tb;
	return true;
}

TaskBuffer * Task::getTaskBuffer()
{
	return mCurStack->mTaskBuffer;
}

ErrorCode Task::allowShowAccount(const Account *acc)
{
	const char * session_id = getAccountId();
	AccountLevel session_lv	= getAccountLevel();
	if( session_lv != AccountSystemAdmin ) {
		if( session_id && strcmp(session_id, acc->id) != 0 ) {
			if( acc->loginTime != 0) {
				return ErrorRestriction;
			}
		}
	}
	return ErrorOk;
}

ErrorCode Task::allowAddAccount(DBPort *port, const Account *acc)
{
	AccountLevel session_lv	= getAccountLevel();
	if( session_lv == AccountSystemService ) {
		if( acc->level == AccountSystemService ) {
			return ErrorPermission;
		}
		if( acc->level == AccountServiceAdmin ) {
			return ErrorPermission;
		}
		if( acc->level == AccountSystemAdmin ) {
			ResultAccount *resultAccount;
			ErrorCode e = port->getAccountInfo("A1", &resultAccount);
			if( e == ErrorOk ) {
				free(resultAccount);
				return ErrorPermission;
			}
		}
	}
	if( session_lv == AccountUser ) {
		if( acc->level != AccountUser ) {
			return ErrorPermission;
		}
	}
	if( session_lv == AccountServiceAdmin ) {
		if( acc->level != AccountUser ) {
			return ErrorPermission;
		}
	}
	if( acc->level != AccountSystemAdmin ) {
		if( acc->level != AccountUser ) {
			return ErrorPermission;
		}
	}
	return ErrorOk;
}

ErrorCode Task::allowSetAccount(DBPort *port, const Account *acc)
{
	const char * session_id = getAccountId();
	AccountLevel session_lv	= getAccountLevel();

	if( session_lv == AccountSystemService ) {
		return ErrorPermission;
	}
	if( session_lv == AccountUser ) {
		if( !session_id || strcmp(session_id, acc->id) != 0 ) {
			if( acc->loginTime != 0) {
				return ErrorPermission;
			}
		}
	}
	if( session_lv == AccountServiceAdmin ) {
		if( !session_id || strcmp(session_id, acc->id) != 0 ) {
			if( acc->loginTime != 0) {
				return ErrorPermission;
			}
		}
	}
	return ErrorOk;
}

ErrorCode Task::allowRemAccount(DBPort *port, const Account *acc)
{
	const char * session_id = getAccountId();
	AccountLevel session_lv	= getAccountLevel();

	if( session_lv == AccountSystemService ) {
		return ErrorPermission;
	}
	if( session_lv == AccountUser ) {
		if( !session_id || strcmp(session_id, acc->id) != 0 ) {
			if( acc->loginTime != 0) {
				return ErrorPermission;
			}
		}
	}
	if( session_lv == AccountServiceAdmin ) {
		if( !session_id || strcmp(session_id, acc->id) != 0 ) {
			if( acc->loginTime != 0) {
				return ErrorPermission;
			}
		}
	}
	return ErrorOk;
}
ErrorCode Task::allowShowService(const ServiceInfo *svc)
{
	return ErrorOk;
}

ErrorCode Task::allowAddService(DBPort *port, const ServiceInfo *svc)
{
	AccountLevel session_lv	= getAccountLevel();

	if( session_lv == AccountSystemService ) {
		return ErrorPermission;
	}
	if( session_lv == AccountUser ) {
		return ErrorPermission;
	}
	if( session_lv == AccountServiceAdmin ) {
		return ErrorPermission;
	}
	return ErrorOk;
}

ErrorCode Task::allowSetService(DBPort *port, const ServiceInfo *svc)
{
	AccountLevel session_lv	= getAccountLevel();
	const char * session_id = getAccountId();

	if( session_lv == AccountSystemService ) {
		return ErrorPermission;
	}
	if( session_lv == AccountUser ) {
		return ErrorPermission;
	}
	if( session_lv == AccountServiceAdmin ) {
		if( !session_id || !svc->adminId || strcmp(session_id, svc->adminId) != 0 ) {
			return ErrorPermission;
		}
	}
	return ErrorOk;
}

ErrorCode Task::allowRemService(DBPort *port, const ServiceInfo *svc)
{
	AccountLevel session_lv	= getAccountLevel();

	if( session_lv == AccountSystemService ) {
		return ErrorPermission;
	}
	if( session_lv == AccountUser ) {
		return ErrorPermission;
	}
	if( session_lv == AccountServiceAdmin ) {
		return ErrorPermission;
	}
	return ErrorOk;
}


ErrorCode Task::allowAddGroup(DBPort *port, const ServiceInfo *svc)
{
	AccountLevel session_lv	= getAccountLevel();
	const char *session_id	= getAccountId();
	const char *session_sv	= getServiceId();
	if( session_lv == AccountSystemService ||
		session_lv == AccountUser ) {
		return ErrorPermission;
	}
	if( session_lv == AccountServiceAdmin ) {
		if( !session_id || strcmp(session_sv, svc->id) != 0 )  {
			return ErrorPermission;
		}
	}
	return ErrorOk;
}

ErrorCode Task::allowSetGroup(DBPort *port, const ServiceInfo *svc, const GroupInfo *group)
{
	AccountLevel session_lv	= getAccountLevel();
	const char *session_id	= getAccountId();
	const char *session_sv	= getServiceId();

	if( session_lv == AccountSystemService ) {
		return ErrorPermission;
	}
	if( session_lv == AccountUser ) {
		if( !session_id || !group->adminId || strcmp(session_id,group->adminId) != 0 )  {
			return ErrorPermission;
		}
	}
	if( session_lv == AccountServiceAdmin ) {
		if( !session_id || !group->adminId || strcmp(session_id,group->adminId) != 0 )  {
			if( !session_sv || strcmp(session_sv, svc->id) != 0 )  {
				return ErrorPermission;
			}
		}
	}
	return ErrorOk;
}

ErrorCode Task::allowRemGroup(DBPort *port, const ServiceInfo *svc, const GroupInfo *group)
{
	AccountLevel session_lv	= getAccountLevel();
	const char * session_id	= getAccountId();
	const char * session_sv	= getServiceId();

	if( session_lv == AccountSystemService ) {
		return ErrorPermission;
	}
	if( session_lv == AccountUser ) {
		if( !session_id || strcmp(session_id, group->adminId) != 0 )  {
			return ErrorPermission;
		}
	}
	if( session_lv == AccountServiceAdmin ) {
		if( !session_id || !group->adminId || strcmp(session_id,group->adminId) != 0 )  {
			if( !session_sv || strcmp(session_sv, svc->id) != 0 )  {
				return ErrorPermission;
			}
		}
	}
	return ErrorOk;
}

ErrorCode Task::allowAddGroupMember(DBPort *port, const ServiceInfo *svc, const GroupInfo *group, const Account *acc)
{
	AccountLevel session_lv	= getAccountLevel();
	const char * session_id	= getAccountId();
	const char * session_sv	= getServiceId();

	if( session_lv == AccountSystemService ) {
		return ErrorPermission;
	}
	if( session_lv == AccountUser ) {
		if( session_sv && strcmp(session_sv, svc->id) == 0 ) {
			if( session_id && group->adminId && strcmp(session_id, group->adminId) == 0 ) {
				return ErrorOk;
			}
		}
		return ErrorPermission;
	}
	if( session_lv == AccountServiceAdmin ) {
		if( session_sv && strcmp(session_sv, svc->id) == 0 ) {
			return ErrorOk;
		}
		return ErrorPermission;
	}
	return ErrorOk;
}

ErrorCode Task::allowRemGroupMember(DBPort *port, const ServiceInfo *svc, const GroupInfo *group, const char *acc)
{
	AccountLevel session_lv	= getAccountLevel();
	const char * session_id	= getAccountId();
	const char * session_sv	= getServiceId();

	if( session_lv == AccountSystemService ) {
		return ErrorPermission;
	}
	if( session_lv == AccountUser ) {
		if( !session_id || strcmp(session_id, acc) != 0 ) {
			if( session_id && group->adminId && strcmp(session_id, group->adminId) == 0 &&
				session_sv && strcmp(session_sv, svc->id) == 0 ) {
				return ErrorOk;
			}
			return ErrorPermission;
		}
	}
	if( session_lv == AccountServiceAdmin ) {
		if( !session_id || strcmp(session_id, acc) != 0 ) {
			if( session_sv || strcmp(session_sv, svc->id) == 0 ) {
				return ErrorOk;
			}
			return ErrorPermission;
		}
	}
	return ErrorOk;
}


/////////////////////////////////////////////////////////////////////////////////////
#define TASK_BLOCK_SIZE	64
TaskManager::TaskList::TaskList()
{
	mSize	= 0;
	mBlock	= 0;
	mPtr	= NULL;
}

TaskManager::TaskList::~TaskList()
{
	clear();
}

void TaskManager::TaskList::clear()
{
	if( mBlock != 0 ) {
		size_t i;
		for( i = 0; i < mSize; i++ )
			delete mPtr[i];
		free(mPtr);
	}
	mSize	= 0;
	mBlock	= 0;
	mPtr	= NULL;
}

bool TaskManager::TaskList::add(Task *t)
{
	if( mBlock - mSize < 1 ) {
		size_t	new_block	= mBlock + TASK_BLOCK_SIZE;
		Task ** new_ptr = (Task**)realloc(mPtr, sizeof(Task*)*new_block);
		if( new_ptr == NULL ) {
			return false;
		}
		mPtr	= new_ptr;
		mBlock	= new_block;
	}
	mPtr[mSize++] = t;
	return true;
}

bool TaskManager::TaskList::remove(Task *t)
{
	size_t i;
	for( i = 0; i < mSize; i++ ) {
		if( mPtr[i] == t ) break;
	}
	if( i == mSize ) return false;

	if( i < mSize-1 )
		memcpy(&mPtr[i], &mPtr[i+1], sizeof(Task*)*(mSize-i-1));
	mSize--;
	return true;
}

Task * TaskManager::TaskList::get(size_t pos)
{
	return mPtr[pos];
}

size_t TaskManager::TaskList::size() const
{
	return mSize;
}

bool TaskManager::TaskList::empty() const
{
	return mSize == 0;
}

/////////////////////////////////////////////////////////////////////////////////////

TaskManager::TaskManager()
{
	mSentinel	= NULL;
}

TaskManager::~TaskManager()
{
}

bool TaskManager::configure(Sentinel *obj, Conf *conf)
{
	mSentinel	= obj;
	return true;
}

Task * TaskManager::createTaskFromSession(Session *conn)
{
	Task *t = NULL;
	{
		WRITE_LOCK(mLock);
		if( !mPoolList.empty() ) {
			t = mPoolList.get(0);
			mPoolList.remove(t);
		} else {
			t = new Task(mSentinel);
		}
		mTaskList.add(t);
	}
	t->initTask(conn, NULL, NULL, 0, 0);
	return t;
}

Task * TaskManager::createTaskFromBaseTask(Task *task)
{
	Task *t = NULL;
	{
		WRITE_LOCK(mLock);
		if( !mPoolList.empty() ) {
			t = mPoolList.get(0);
			mPoolList.remove(t);
		} else {
			t = new Task(mSentinel);
		}
		mTaskList.add(t);
	}
	t->initTask(NULL, task, NULL, 0, 0);
	return t;
}

bool TaskManager::removeTask(Task *t)
{
	WRITE_LOCK(mLock);
	if( !mTaskList.remove(t) )
		return false;
	mPoolList.add(t);
	mSentinel->log(LOG_DEBUG, "TaskManager: task=%d closed.", t->mId);

	return true;
}

bool TaskManager::diag(pjson::builder &jb)
{
	mSentinel->log(LOG_DEBUG, "TaskManager::diag begin");

	READ_LOCK(mLock);

	if( !jb.beginObject() ||
		!jb.addObjectProp("task",4) ||
		!jb.beginArray() ) {
		mSentinel->log(LOG_ERR, "TaskManager::diag json error 1. code=%d", jb.getError());
		return false;
	}

	size_t s = mTaskList.size();
	size_t i;
	for( i = 0; i < s; i++ ) {
		const char *v;
		Task *task = mTaskList.get(i);
		if( !jb.addArrayContent() ||
			!jb.beginObject() ||
//			!jb.addObjectProp("taskAddr",8) ||
//			!jb.valueInt((int)task) ||
			!jb.addObjectProp("taskId",6) ||
			!jb.valueInt(task->mId) ) {
			mSentinel->log(LOG_ERR, "TaskManager::diag json error 2. code=%d", jb.getError());
			return false;
		}

		if( task->mCurStack || !task->mTaskStack.empty() ) {
			if( !jb.addObjectProp("stack",5) ||
				!jb.beginArray() ) {
				mSentinel->log(LOG_ERR, "TaskManager::diag json error 1. code=%d", jb.getError());
				return false;
			}
			if( task->mCurStack  ) {
				if( !jb.addArrayContent() ||
					!jb.beginObject() ||
					!jb.addObjectProp("proc",4) ||
					!jb.valueString(task->mCurStack->mBaseProc->getName()) ||
					!jb.addObjectProp("func",4) ||
					!jb.valueInt(task->mCurStack->mBaseFunc) ||
					!jb.addObjectProp("funcName",8) ||
					!jb.valueString(task->mCurStack->mBaseProc->getFuncName(task->mCurStack->mBaseFunc)) ||
					!jb.addObjectProp("step",4) ||
					!jb.valueInt(task->mCurStack->mBaseStep) ||
					!jb.endObject() ) {
					mSentinel->log(LOG_ERR, "TaskManager::diag json error 2. code=%d", jb.getError());
					return false;
				}
			}
			TaskStack *sp = (TaskStack*)task->mTaskStack.getTop();
			while(sp) {
				if( !jb.addArrayContent() ||
					!jb.beginObject() ||
					!jb.addObjectProp("proc",4) ||
					!jb.valueString(sp->mBaseProc->getName()) ||
					!jb.addObjectProp("func",4) ||
					!jb.valueInt(sp->mBaseFunc) ||
					!jb.addObjectProp("funcName",8) ||
					!jb.valueString(sp->mBaseProc->getFuncName(sp->mBaseFunc)) ||
					!jb.addObjectProp("step",4) ||
					!jb.valueInt(sp->mBaseStep) ||
					!jb.endObject() ) {
					mSentinel->log(LOG_ERR, "TaskManager::diag json error 2. code=%d", jb.getError());
					return false;
				}
				sp = (TaskStack*)task->mTaskStack.getNext(sp);
			}
			if( !jb.endArray() ) {
				mSentinel->log(LOG_ERR, "TaskManager::diag json error 10. code=%d", jb.getError());
				return false;
			}
		}

		Session *con = task->getSession();
		if( !jb.addObjectProp("session", 7) ||
			!jb.valueInt( (long long)con ) ) {
			mSentinel->log(LOG_ERR, "TaskManager::diag json error 3. code=%d", jb.getError());
			return false;
		}
		if( con ) {
			sockaddr_in caddr;
			con->getPeerAddress(&caddr);
			char buf[256];
			sprintf(buf, "%s:%d", inet_ntoa(caddr.sin_addr), ntohs(caddr.sin_port));
			if( !jb.addObjectProp("addr", 4) ||
				!jb.valueString(buf) ) {
				mSentinel->log(LOG_ERR, "TaskManager::diag json error 4. code=%d", jb.getError());
				return false;
			}
		}
		jb.addObjectProp("status",6);
		TaskStatus st = task->getTaskStatus();
		switch(st) {
		case TaskInit:			v = "Init"; break;
		case TaskIdle:			v = "Idle"; break;
		case TaskActive:		v = "Active"; break;
		case TaskProcess:		v = "Process"; break;
		default:				v = "Unknown"; break;
		}
		if( !jb.endObject() ) {
			mSentinel->log(LOG_ERR, "TaskManager::diag json error 9. code=%d", jb.getError());
			return false;
		}
	}
	if( !jb.endArray() ||
		!jb.endObject() ) {
		mSentinel->log(LOG_ERR, "TaskManager::diag json error 10. code=%d", jb.getError());
		return false;
	}

	mSentinel->log(LOG_DEBUG, "TaskManager::diag end");

	return true;
}


}

