/*
 * bitmap.cpp
 *
 *  Created on: 2012/02/03
 *      Author: tanaka
 */
#include "bitmap.h"
#include "bitmap_h1.h"
#include "bitmap_v1.h"
#include "bitmap_g2.h"
#include "bitmap_g4.h"

#include <stdlib.h>
#include <malloc.h>
#include <memory.h>

namespace ssg {

Bitmap::Private::Private() {
	width	= 0;
	height	= 0;
	fb_width	= 0;
	fb_height	= 0;
	mb_width	= 0;
	mb_height	= 0;
	pixformat	= (Pixformat)0;
	bmp_bytes	= 0;
	bmp			= NULL;
}

Bitmap::Private::~Private()
{
	if(bmp) free(bmp);
}

bool Bitmap::Private::create(INT width, INT height, Pixformat pf)
{
	INT mbw, mbh;
	INT fbw, fbh;
	if( pf == SC_HORZ ) {
		mbw	= width / 8;
		if( width % 8 != 0 ) mbw++;
		mbh	= height;
		fbw = mbw * 8;
		fbh = mbh;
	} else if( pf == SC_VERT ){
		mbw	= width;
		mbh	= height / 8;
		if( height % 8 != 0 ) mbh++;
		fbw = mbw;
		fbh = mbh * 8;
	} else if( pf == SC_HORZG2 ) {
		mbw	= width / 4;
		if( width % 4 != 0 ) mbw++;
		mbh	= height;
		fbw = mbw * 4;
		fbh = mbh;
	} else if( pf == SC_HORZG4 ) {
		mbw	= width / 2;
		if( width % 2 != 0 ) mbw++;
		mbh	= height;
		fbw = mbw * 2;
		fbh = mbh;
	} else {
		return false;
	}
	SIZE bmp_bytes = mbw * mbh;
	this->width		= width;
	this->height	= height;
	this->fb_width	= fbw;
	this->fb_height	= fbh;
	this->mb_width	= mbw;
	this->mb_height	= mbh;
	this->bmp_bytes	= bmp_bytes;
	this->bmp		= malloc(bmp_bytes);
	clear();
	return true;
}

Bitmap::Bitmap()
{
	ptr = NULL;
}

Bitmap::~Bitmap()
{
	delete ptr;
}

Bitmap * Bitmap::create(INT width, INT height, Pixformat pf)
{
	Bitmap *bmp = new Bitmap();
	Private *p = NULL;
	if( pf == SC_HORZ ) {
		p = new HorzBitmap(bmp);
	} else if( pf == SC_VERT ){
		p = new VertBitmap(bmp);
	} else if( pf == SC_HORZG2 ) {
		p = new HorzG2Bitmap(bmp);
	} else if( pf == SC_HORZG4 ) {
		p = new HorzG4Bitmap(bmp);
	} else {
		return NULL;
	}
	p->create(width, height, pf);
	bmp->ptr	= p;
	return bmp;
}

INT Bitmap::getWidth() const
{
	return ptr->width;
}
INT Bitmap::getHeight() const
{
	return ptr->height;
}
INT Bitmap::getLineSize() const
{
	return ptr->mb_width;
}
INT Bitmap::getHorzSegments() const
{
	return ptr->mb_width;
}
INT Bitmap::getVertSegments() const
{
	return ptr->height / 8;
}
Pixformat Bitmap::getPixformat() const
{
	return ptr->pixformat;
}

void *Bitmap::getBitmap()
{
	return ptr->bmp;
}

Bitmap::PixIterator *Bitmap::createHorzPixIterator() const
{
	return ptr->createHorzPixIterator();
}

Bitmap::PixIterator *Bitmap::createVertPixIterator() const
{
	return ptr->createVertPixIterator();
}

Bitmap::SegmentIterator *Bitmap::createSegIterator() const
{
	return ptr->createSegIterator();
}

Bitmap::PixMapper *	Bitmap::createPixMapper(Pixformat pf) const
{
	int spf = ptr->pixformat;
	int sdp = 0;
	switch(spf) {
	case SC_HORZ:
	case SC_VERT:
		sdp	= 1;
		break;
	case SC_HORZG2:
		sdp	= 2;
		break;
	case SC_HORZG4:
		sdp	= 4;
		break;
	}
	int ddp = 0;
	switch(pf) {
	case SC_HORZ:
	case SC_VERT:
		ddp	= 1;
		break;
	case SC_HORZG2:
		ddp	= 2;
		break;
	case SC_HORZG4:
		ddp	= 4;
		break;
	}
	switch(sdp) {
	case 1:
		switch(ddp) {
		case 1:	return new PixMapper1x1();
		case 2:	return new PixMapper1x2();
		case 4:	return new PixMapper1x4();
		}
		break;
	case 2:
		switch(ddp) {
		case 1:	return new PixMapper2x1();
		case 2:	return new PixMapper1x1();
		case 4:	return new PixMapper2x4();
		}
		break;
	case 4:
		switch(ddp) {
		case 1:	return new PixMapper4x1();
		case 2:	return new PixMapper4x2();
		case 4:	return new PixMapper1x1();
		}
		break;
	}
	return new PixMapper1x1();
}

Bitmap::Private *Bitmap::getPrivate() const
{
	return ptr;
}

} // ssg



