/**************************************************
OpengateM - MAC address authentication system 

 module to control management database that includes following tables
  list of mac addresses and their owners
  list of usage log
 this database is implemented with MySql 

Copyright (C) 2011 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include "opengatemmng.h"
#include <mysql.h>

MYSQL mysql;

/*****************************/
/* initialize management db  */
/*****************************/
int initMngDb(void){

  /* set parameters read from conf file */
  char *server = GetConfValue("MySqlDb/Server");
  char *user =  GetConfValue("MySqlDb/User");
  char *password = GetConfValue("MySqlDb/Password");
  char *database = GetConfValue("MySqlDb/Database");
  my_bool reconnect;

/* initialize mysql */
  mysql_library_init(-1,NULL,NULL);
  if(mysql_init(&mysql)==NULL){
     err_msg("ERR at %s#%d: mysql init: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     terminateProg(0);
  }

  /* Connect to database */
  if (!mysql_real_connect(&mysql, server,
			   user, password, database, 0, NULL, 0)) {
    err_msg("ERR at %s#%d: mysql connect: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
    terminateProg(0);
  }

  /* set auto-reconnect true */
  reconnect = TRUE;
  mysql_options(&mysql, MYSQL_OPT_RECONNECT, &reconnect);  

  return TRUE;
}

/**********************************/
/*close management db             */
/**********************************/
void closeMngDb(void){
  mysql_close(&mysql);
  mysql_library_end();
}


/*********************************************************/
/* get nic vendor from table nicvendors in management db */
/* macAddress:(in),vendor:(out),bufferLength:(in)        */ 
/*********************************************************/
int getNicVendorFromMngDb(char* macAddress, char* vendor, int bufferLength){

  MYSQL_RES *res;
  MYSQL_ROW row;
  int found=FALSE;
  char queryStr[BUFFMAXLN];
  char macHeader[ADDRMAXLN];

  /* set default values (="?") */
  vendor[0]='?';
  vendor[1]='\0';
 
  /* prepare query string */
  /* macHeader is first 3 bytes of macAddess that represents vender code */
  /* macAddress="11:22:33:44:55:66" -> macHeader="11:22:33" */
  strlcpy(macHeader, macAddress, ADDRMAXLN);
  macHeader[8]='\0';
  snprintf(queryStr, BUFFMAXLN, 
	   "select org from nicvendors where oui='%s'",
	   macHeader);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
     err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     return FALSE;
  }
  res = mysql_use_result(&mysql);
  
  /* get a row from query result */
  row = mysql_fetch_row(res);

  /* if not found, return false */
  if(row==NULL)  found=FALSE;

  /* if found, get the vendor name */
  else {
    strlcpy(vendor, row[0], bufferLength);
    found=TRUE;
  }

  /* free memory area */
  mysql_free_result(res);
  return found;
}

/*************************************************/
/* The macAddr is already registered or not      */
/* the row having status="D"(deleted) is ignored */
/* macAddr:(input)                               */
/*************************************************/
int isMacAddrFoundInMngDb(char* macAddr){

  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  int ret;

  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "select * from macaddrs "
	   " where macAddress='%s' and status!='D'",
	   macAddr);
  
  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  
  /* store query result */
  res = mysql_store_result(&mysql);

  /* get a row from query result */
  row = mysql_fetch_row(res);
  
  /* if found, return true */
  if(row!=NULL) ret=TRUE;

  /* if not found, return false */
  else ret=FALSE;

  /* free memory area */
  mysql_free_result(res);
  
  return ret;
}

/******************************************************/
/* The count of registered mac addresses for the user */
/* the row having status="D"(deleted) is ignored      */
/* all-args:(input)                                   */
/******************************************************/
int countMacAddrsForUserInMngDb(char* userId, char* extraId){

  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  char countStr[WORDMAXLN];
  int count=10000;

  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "select count(*) from macaddrs "
	   " where userId='%s' and extraId='%s' and status!='D'",
	   userId, extraId);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return count;
  }

  /* store query result */
  res = mysql_store_result(&mysql);

  /* get a row from query result */
  row = mysql_fetch_row(res);

  /* if found, return the count */
  if(row!=NULL){
    strlcpy(countStr, row[0], WORDMAXLN);
    count=atoi(countStr);
  }

  /* free memory area */
  mysql_free_result(res);

  return count;
}

/***********************************************************/
/* register mac address and the owner to the management db */
/* all-args:(input)                                        */
/***********************************************************/
int registMacAddrToMngDb(char* macAddr, char* deviceName, char* userId, char* extraId, char* mailAddr){

  char queryStr[BUFFMAXLN];
  int count;

   /* if alreay registered, return false */
  if(IsMacAddrFoundInMngDb(macAddr)){
    SetMessage(ExistentMacAddr);
    return FALSE;
  }
 
  /* if the count of devices for a user exceeds the limit(in conf), return false */
  count=CountMacAddrsForUserInMngDb(userId,extraId);
  if(count >= atoi(GetConfValue("MaxDevices"))){
    SetMessage(DeviceCountOver);
    return FALSE;
  }
 
  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "insert into macaddrs "
	   "(macAddress, status, device, userId, extraId, "
	   " entryDate, renewDate, limitDate, mailAddress) "
	   "values ('%s','A','%s', '%s', '%s', "
	   " now(), now(), %s, '%s')",
	   macAddr, deviceName, userId, extraId, 
	   GetConfValue("LimitDate"), mailAddr);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
     err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     return FALSE;
  }

  return TRUE;
}

/********************************************************/
/* get next mac address for the user from management db */
/* at first call, execute query and return first row    */
/* from second call, return next row of previous query  */
/* repeat until return=false (end of list, clear work)  */
/* userId,extraId:(input), others:(output)              */
/********************************************************/
int getNextMacAddrFromMngDb(char* userId, char* extraId, char* macAddress, char* deviceName, char* entryDate, char* limitDate, char* status, char* mailAddress){

  static MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];

  /* set default values */
  macAddress[0]=deviceName[0]=entryDate[0]=limitDate[0]='\0';

  /* if do not get result yet */
  if(res==NULL){

    /* prepare query string */
    snprintf(queryStr, BUFFMAXLN, 
      "select macAddress, device, entryDate, limitDate, status, mailAddress "
      "from macaddrs where userId='%s' and extraId='%s' and status!='D'",
      userId, extraId);

    /* send SQL query */
    if (mysql_query(&mysql, queryStr)){
      err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	      mysql_error(&mysql));
      return FALSE;
    }
    
    /* store result */
    res = mysql_store_result(&mysql);
  }

  /* output table name */
  row = mysql_fetch_row(res);

  /* if found, return values */
  if(row!=NULL){
    strlcpy(macAddress, row[0],ADDRMAXLN);
    strlcpy(deviceName,row[1],WORDMAXLN);
    strlcpy(entryDate,row[2],WORDMAXLN);
    strlcpy(limitDate,row[3],WORDMAXLN);
    strlcpy(status,row[4],WORDMAXLN);
    strlcpy(mailAddress,row[5],BUFFMAXLN);
    return TRUE;
  }
  /* if not found, free memory area */
  else{
    mysql_free_result(res);
    return FALSE;
  }
}


/***************************************************/
/* write mac modification log to the management db */
/* all-args:(input)                                */
/***************************************************/
int putMacModifyLogToMngDb(char* userId, char* extraId, char* macAddr, char modifyType){

  char queryStr[BUFFMAXLN];

  /*** insert modify log ***/
  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "insert into macmodify "
	   "(userId, extraId, macAddress, modifyType, modifyDate) "
	   "values ('%s', '%s', '%s', '%c', now())",
	   userId, extraId, macAddr, modifyType);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
     err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     return FALSE;
  }
  /*** remove old log ***/ 
  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "delete from macmodify "
	   " where modifyDate < adddate(now(), interval -1 day) ");

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
     err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     return FALSE;
  }

  return TRUE;
}

/************************************************************/
/* The count of mac address modification in last 24 hours   */
/* (the modification count is checked to cope with abuse)   */
/* all-args:(input)                                         */
/* indicate either of userID&extraId or macAddress to count */ 
/************************************************************/
int countMacModifyPerDayInMngDb(char* userId, char* extraId, char* macAddress){

  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  char countStr[WORDMAXLN];
  int count=10000;

  /* prepare query string */
  /* count for one macAddress if not null, elsecount for one userID)*/
  if(!isNull(macAddress)){
    snprintf(queryStr, BUFFMAXLN, 
	   "select count(*) from macmodify "
	   " where macAddress='%s' and "
	   " modifyDate > adddate(now(), interval -1 day) ",
	   macAddress);
  }else{
    snprintf(queryStr, BUFFMAXLN, 
	   "select count(*) from macmodify "
	   " where userId='%s' and extraId='%s' and "
	   " modifyDate > adddate(now(), interval -1 day) ",
	   userId, extraId);
  }
  
  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return count;
  }

  /* store query result */
  res = mysql_store_result(&mysql);

  /* get a row from query result */
  row = mysql_fetch_row(res);

  /* if found, return count (if not, return large value for safe) */
  if(row!=NULL){
    strlcpy(countStr, row[0], WORDMAXLN);
    count=atoi(countStr);
  }

  /* free memory area */
  mysql_free_result(res);

  return count;
}

/*********************************************************/
/* delete a mac address registered in the management db  */
/* actuality, not delete row, but set the status="D"     */
/* macAddr:(input)                                       */
/*********************************************************/
int delMacAddrFromMngDb(char* macAddr){

  char queryStr[BUFFMAXLN];

    /* prepare query string */
    /* don't touch device of status="I"(set Inactive by admin) */
    snprintf(queryStr, BUFFMAXLN, 
	     "update macaddrs set status='D',limitDate=now() "
	     " where macAddress='%s' and (status='A' or status='P')", macAddr);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  return TRUE;
}

/******************************************/
/* renew the limit date for a mac address */
/*  in the management db                  */
/* macAddr:(input)                        */
/* LimitDate in conf file is a SQL string */
/* to make a future date                  */
/* eg, "adddate(now(),interval 1 month)"  */
/******************************************/
int renewMacAddrInMngDb(char* macAddr){

  char queryStr[BUFFMAXLN];

  /* prepare query string */
  /* don't touch device set as inactive by admin */
  snprintf(queryStr, BUFFMAXLN, 
	   "update macaddrs set status='A', renewDate=now(), limitDate=%s "
	   " where (status='A' or status='P') and macAddress='%s'", 
	   GetConfValue("LimitDate"),macAddr);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  return TRUE;
}

/******************************************
pause the usage of the device for a mac address
 registered in the management db
******************************************/
int pauseMacAddrInMngDb(char* macAddr){

  char queryStr[BUFFMAXLN];

  /* prepare query string */
  /* don't touch device set as inactive by admin */
  snprintf(queryStr, BUFFMAXLN, 
	   "update macaddrs set status='P' "
	   " where status='A' and macAddress='%s' and limitDate>now()", 
	   macAddr);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  return TRUE;
}

/*******************************************
 get next next usage log from management db
 if end of list, return false
*******************************************/
int getNextUsageLogFromMngDb(char* userId, char* extraId, char* macAddr, char* deviceName, char* openTime, char* closeTime, char* gatewayName){
  static MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];

  /* set default values */
  macAddr[0]=deviceName[0]=openTime[0]=gatewayName[0]='\0';

  /* if session table is not found in management db, return */
  if(!IsTableFoundInMngDb("sessionmd")) return FALSE;

  /* if do not get result yet (if first call) */
  if(res==NULL){

    /* prepare query string */
    /* get log where addr.entry < session.open < addr.limit */
    /*  (the same device may be registered by other users in old days) */
    snprintf(queryStr, BUFFMAXLN, 
	     "select macaddrs.macAddress, device, openTime, closeTime, "
	     " gatewayName "
	     " from macaddrs, sessionmd "
	     " where macaddrs.macAddress=sessionmd.macAddress "
	     " and entryDate <= openTime and openTime <= limitDate "
	     " and (%s<closeTime or closeTime=0) "
	     " and userId='%s' and extraId='%s'",
	     GetConfValue("ShowLogAfter"), userId, extraId);

    /* send SQL query */
    if (mysql_query(&mysql, queryStr)){

      /* query error */
      err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	      mysql_error(&mysql));
      return FALSE; 
    }
    
    /* store result */
    res = mysql_store_result(&mysql);
  }

  /* output table name */
  row = mysql_fetch_row(res);

  /* if found, return values */
  if(row!=NULL){
    strlcpy(macAddr, row[0],ADDRMAXLN);
    strlcpy(deviceName,row[1],WORDMAXLN);
    strlcpy(openTime,row[2],WORDMAXLN);
    strlcpy(closeTime,row[3],WORDMAXLN);
    strlcpy(gatewayName,row[4],WORDMAXLN);
    return TRUE;
  }
  /* if not found, free memory area */
  else{
    mysql_free_result(res);
    return FALSE;
  }
}


/*******************************************
 get next mail address near limit date
from management db 
*******************************************/
int getNextMailAddressFromMngDb(char* mailAddress, char* limitDate, char* device){

  static MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  char* mailTiming=NULL;

  /* set default values */
  mailAddress[0]=limitDate[0]='\0';

  /* if do not get result yet */
  if(res==NULL){

    /* get mail send timing from conf file */
    mailTiming = GetConfValue("Mail/Timing");
    if(isNull(mailTiming)) return FALSE;

    /* prepare query string */
    /* [select mailAddress,limitDate,device from macaddrs 
        where (  date(now())=date(adddate(limitDate, interval -1 day)) 
              OR date(now())=date(adddate(limitDate, interval -7 day))  ) 
             AND (status='A' OR status='P') order by mailAddress,limitDate] */
    snprintf(queryStr, BUFFMAXLN, 
	     "select mailAddress,limitDate,device from macaddrs "
	     " where (%s) and (status='A' or status='P') "
	     " order by mailAddress,limitDate", mailTiming);
    /*       mailTiming is a condition string defined in conf file */ 

    /* send SQL query */
    if (mysql_query(&mysql, queryStr)){
      err_msg("ERR at %s#%d: mysql query[%s]: %s",__FILE__,__LINE__,queryStr,
	      mysql_error(&mysql));
      return FALSE;
    }
    
    /* store result */
    res = mysql_store_result(&mysql);
  }

  /* output table name */
  row = mysql_fetch_row(res);

  /* if found, return values */
  if(row!=NULL){
    strlcpy(mailAddress, row[0], BUFFMAXLN);
    strlcpy(limitDate, row[1], WORDMAXLN);
    strlcpy(device, row[2], WORDMAXLN);
    return TRUE;
  }
  /* if not found, free memory area */
  else{
    mysql_free_result(res);
    return FALSE;
  }
}

/*******************************************
 get the date corresponding to ShowLogAfter in conf file
*******************************************/
int getTimeRangeToShowLog(char* beginTime, char* endTime, int* dateCount){

  static MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  char* showLogAfter;
  char countStr[WORDMAXLN];

  /* get conf value for beginning day for listing */
  showLogAfter=GetConfValue("ShowLogAfter");
  if(isNull(showLogAfter)) return FALSE;

  /* set default values */
  beginTime[0]='\0';
  endTime[0]='\0';
  *dateCount=0;

  /* if do not get result yet */
  if(res==NULL){

    /* prepare query string */
    /* select (adddate(now(), interval -1 month)), 
       now(), 
       datediff(now(),(adddate(now(), interval -1 month))) + 1; */
    snprintf(queryStr, BUFFMAXLN, 
	     "select (%s), now(), datediff(now(),(%s)) + 1", 
	     showLogAfter, showLogAfter);

    /* send SQL query */
    if (mysql_query(&mysql, queryStr)){
      err_msg("ERR at %s#%d: mysql query[%s]: %s",__FILE__,__LINE__,queryStr,
	      mysql_error(&mysql));
      return FALSE;
    }
    
    /* store result */
    res = mysql_store_result(&mysql);
  }

  /* output table name */
  row = mysql_fetch_row(res);

  /* if found, return values */
  if(row!=NULL){
    strlcpy(beginTime, row[0], WORDMAXLN);
    strlcpy(endTime, row[1], WORDMAXLN);
    strlcpy(countStr, row[2], WORDMAXLN);
    *dateCount=atoi(countStr);
    return TRUE;
  }
  /* if not found, free memory area */
  else{
    mysql_free_result(res);
    return FALSE;
  }
}

/********************************************
rename the device name for indicated mac address
********************************************/
int renameDeviceNameInMngDb(char* macAddr, char* nameStr){

  char queryStr[BUFFMAXLN];

  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "update macaddrs set device='%s' "
	   " where (status='A' or status='P') and macAddress='%s'", 
	   nameStr, macAddr);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  return TRUE;
}

/********************************************
rename the mail address for indicated mac address
********************************************/
int renameMailAddressInMngDb(char* macAddr, char* mailStr){

  char queryStr[BUFFMAXLN];

  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "update macaddrs set mailAddress='%s' "
	   " where (status='A' or status='P') and macAddress='%s'", 
	   mailStr, macAddr);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  return TRUE;
}

/********************************************
Register or Update MacAddress which is unlinked to user
********************************************/
int regOrUpNobodyMacAddr(char* macAddress){

  int modified=FALSE;

  /* macAddress in inactive status cannot be modified */
  if(IsMacAddrStatusInactiveInMngDb(macAddress)) return FALSE;
  
  /* if mac is already registered, update it */
  if(IsMacAddrFoundInMngDb(macAddress)){
    if(RenewMacAddrInMngDb(macAddress)){
      PutMacModifyLogToMngDb("?", "", macAddress, 'E'); /*(userId,extraId,.)*/
      modified=TRUE;
    }
  }

  /* if mac is not yet registered, register it */ 
  else{
    if(RegistMacAddrToMngDb(macAddress,"?","?","","")){ /*(dev,user,ext,mail) */
	PutMacModifyLogToMngDb("?","", macAddress, 'R'); /* (userId,extraId,.) */
      modified=TRUE;
    }
  }
  return modified;
}

/******************************************
 Is the macAddr  in INACTIVE('I') status
******************************************/
int isMacAddrStatusInactiveInMngDb(char* macAddr){

  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  int ret;

  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "select * from macaddrs "
	   " where macAddress='%s' and status='I'",
	   macAddr);
  
  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  
  /* store result */
  res = mysql_store_result(&mysql);

  /* output table name */
  row = mysql_fetch_row(res);
  
  /* if found, return true */
  if(row!=NULL) ret=TRUE;

  /* if not found, return false */
  else ret=FALSE;

  mysql_free_result(res);
  return ret;
}

/******************************************************/
/* Does the mac address belong to the userid@extraid  */
/* all-args:(input)                                   */
/* return value: true(1)=YES, false(0)=NO             */
/******************************************************/
int doesMacAddrBelongToUser(char* macAddr, char* userId, char* extraId){

  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  char existsStr[WORDMAXLN];
  int exists=FALSE;

  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "select exists(select * from macaddrs "
	   " where macAddress='%s' and userId='%s' "
	   " and extraId='%s' and status!='D') ",
	   macAddr, userId, extraId);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }

  /* store query result */
  res = mysql_store_result(&mysql);

  /* get a row from query result */
  row = mysql_fetch_row(res);

  /* if found, return the count */
  if(row!=NULL){
    strlcpy(existsStr, row[0], WORDMAXLN);
    exists=atoi(existsStr);
  }

  /* free memory area */
  mysql_free_result(res);

  return exists;
}

/******************************************
is found table in management db
input=table name
return value: 1=found,0=not found or error
******************************************/
int isTableFoundInMngDb(char* table){
  MYSQL_RES *res;
  MYSQL_ROW row;
  int found=FALSE;
  char queryStr[BUFFMAXLN];

  /* if no table, return false */
  if(isNull(table)) return FALSE;
  
  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "show tables like '%s'", table);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
     err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     return ERROR;
  }
  res = mysql_use_result(&mysql);
  
  /* output table row */
  row = mysql_fetch_row(res);

  /* row==NULL means getting no record */
  if(row==NULL) found=FALSE;
  else          found=TRUE;

  mysql_free_result(res);
  return found;
}


/********************************************
 routines for debugging output
********************************************/

int InitMngDb(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>initMngDb()");
  ret=initMngDb();
  if(debug>1) err_msg("DEBUG:(%d)<=initMngDb()",ret);
  return ret;
}
void CloseMngDb(void){
  if(debug>1) err_msg("DEBUG:=>closeMngDb()");
  closeMngDb();
  if(debug>1) err_msg("DEBUG:<=closeMngDb()");
}

int GetNextMacAddrFromMngDb(char* userId, char* extraId, char* macAddress, char* deviceName, char* entryDate, char* limitDate, char* status, char* mailAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getNextMacAddrFromMngDb(%s,%s)",userId,extraId);
  ret=getNextMacAddrFromMngDb(userId,extraId,macAddress,deviceName,
			      entryDate,limitDate, status,mailAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=getNextMacAddrFromMngDb(,,%s,%s,%s,%s,%s,%s)",
		      ret,macAddress,deviceName,entryDate,limitDate,status,mailAddress);
  return ret;
}

int RegistMacAddrToMngDb(char* macAddr, char* deviceName, char* userId, char* extraId, char* mailAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>registMacAddrToMngDb(%s,%s,%s,%s,%s)",
		      macAddr,deviceName,userId,extraId,mailAddress);
  ret=registMacAddrToMngDb(macAddr,deviceName,userId,extraId, mailAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=registMacAddrToMngDb( )",ret);
  return ret;
}

int IsMacAddrFoundInMngDb(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isMacAddrFoundInMngDb(%s)", macAddr);
  ret=isMacAddrFoundInMngDb(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=isMacAddrFoundInMngDb( )",ret);
  return ret;
}

int CountMacAddrsForUserInMngDb(char* userId, char* extraId){
  int ret;
  if(debug>1) err_msg("DEBUG:=>countMacAddrsForUserInMngDb(%s,%s)",
		      userId,extraId);
  ret=countMacAddrsForUserInMngDb(userId,extraId);
  if(debug>1) err_msg("DEBUG:(%d)<=countMacAddrsForUserInMngDb( )",ret);
  return ret;
}

int PutMacModifyLogToMngDb(char* userId, char* extraId, char* macAddr, char modifyType){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putMacModifyLogToMngDb(%s,%s,%s,%c)",
		      userId,extraId,macAddr,modifyType);
  ret=putMacModifyLogToMngDb(userId,extraId,macAddr,modifyType);
  if(debug>1) err_msg("DEBUG:(%d)<=putMacModifyLogToMngDb( )",ret);
  return ret;
}

int GetNicVendorFromMngDb(char* macAddress, char* vendor, int bufferLength){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getNicVendorFromMngDb(%s,,%d)",
		      macAddress, bufferLength);
  ret=getNicVendorFromMngDb(macAddress, vendor, bufferLength);
  if(debug>1) err_msg("DEBUG:(%d)<=getNicVendorFromMngDb(%s)",ret, vendor);
  return ret;
}


int CountMacModifyPerDayInMngDb(char* userId, char* extraId, char* macAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>countMacModifyPerDayInMngDb(%s,%s,%s)",
		      userId,extraId,macAddress);
  ret=countMacModifyPerDayInMngDb(userId,extraId,macAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=countMacModifyPerDayInMngDb( )",ret);
  return ret;
}

int DelMacAddrFromMngDb(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>delMacAddrFromMngDb(%s)",macAddr);
  ret=delMacAddrFromMngDb(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=delMacAddrFromMngDb( )",ret);
  return ret;
}

int RenewMacAddrInMngDb(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>renewMacAddrInMngDb(%s)",macAddr);
  ret=renewMacAddrInMngDb(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=renewMacAddrinMngDb( )",ret);
  return ret;
}

int PauseMacAddrInMngDb(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>pauseMacAddrInMngDb(%s)",macAddr);
  ret=pauseMacAddrInMngDb(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=pauseMacAddrinMngDb( )",ret);
  return ret;
}

int GetNextUsageLogFromMngDb(char* userId, char* extraId, char* macAddr, 
 char* deviceName, char* openTime, char* closeTime, char* gatewayName){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getNextUsageLogFromMngDb(%s,%s)",userId,extraId);
  ret=getNextUsageLogFromMngDb(userId,extraId,macAddr,deviceName,
			      openTime,closeTime,gatewayName);
  if(debug>1) err_msg("DEBUG:(%d)<=getNextUsageLogFromMngDb(,,%s,%s,%s,%s,%s)",
	         ret,macAddr,deviceName,openTime,closeTime,gatewayName);
  return ret;
}

int GetNextMailAddressFromMngDb(char* mailAddress, char* limitDate, char*device){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getnextMailAddressFromMngDb( )");
  ret=getNextMailAddressFromMngDb(mailAddress, limitDate, device);
  if(debug>1) err_msg("DEBUG:(%d)<=getNextMailAddressFromMngDb(%s,%s,%s)",
		      ret,mailAddress,limitDate,device);
  return ret;
}

int GetTimeRangeToShowLog(char* beginTime, char* endTime, int* dateCount){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getTimeRangeToShowLog( )");
  ret=getTimeRangeToShowLog(beginTime, endTime, dateCount);
  if(debug>1) err_msg("DEBUG:(%d)<=getTimeRangeToShowLog(%s,%s,%d)",
		      ret,beginTime,endTime,*dateCount);
  return ret;
}

int RenameDeviceNameInMngDb(char* macAddr, char* nameStr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>renameDevideNameInMngDb(%s,%s)",macAddr,nameStr);
  ret=renameDeviceNameInMngDb(macAddr, nameStr);
  if(debug>1) err_msg("DEBUG:(%d)<=renameDeviceNameInMngDb()",ret);
  return ret;
}

int RenameMailAddressInMngDb(char* macAddr, char* mailStr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>renameMailAddressInMngDb(%s,%s)",macAddr,mailStr);
  ret=renameMailAddressInMngDb(macAddr, mailStr);
  if(debug>1) err_msg("DEBUG:(%d)<=renameMailAddressInMngDb()",ret);
  return ret;
}

int RegOrUpNobodyMacAddr(char* macAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>regOrUpNobodyMacAddr(%s)",macAddress);
  ret=regOrUpNobodyMacAddr(macAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=regOrUpNobodyMacAddr()",ret);
  return ret;
}


int IsMacAddrStatusInactiveInMngDb(char* macAddress){
    int ret;
  if(debug>1) err_msg("DEBUG:=>isMacAddrStatusInactiveInMngDb(%s)",macAddress);
  ret=isMacAddrStatusInactiveInMngDb(macAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=isMacAddrStatusInactiveInMngDb()",ret);
  return ret;
}

int DoesMacAddrBelongToUser(char* macAddr, char* userId, char* extraId){
  int ret;
  if(debug>1) err_msg("DEBUG:=>doesMacAddrBelongToUser(%s,%s,%s)",
		      macAddr,userId,extraId);
  ret=doesMacAddrBelongToUser(macAddr,userId,extraId);
  if(debug>1) err_msg("DEBUG:(%d)<=doesMacAddrBelongToUser( )",ret);
  return ret;
}

int IsTableFoundInMngDb(char* table){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isTableFoundInMngDb(%s)", table);
  ret = isTableFoundInMngDb(table);
  if(debug>1) err_msg("DEBUG:(%d)<=isTableFoundInMngDb()", ret);
  return ret;
}

