/*
 * Copyright 2004-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.extension.jdbc;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.Statement;

/**
 * {@link JdbcManager}が管理しているトランザクション中の状態を持つインターフェースです。
 * トランザクション外で使われる場合は、SQLの実行後すぐに破棄されます。
 * 
 * @author higa
 * 
 */
public interface JdbcContext {

    /**
     * 状態を破棄します。
     */
    void destroy();

    /**
     * トランザクション中に作成されたかどうかを返します。
     * 
     * @return トランザクション中に作成されたかどうか
     */
    boolean isTransactional();

    /**
     * キャッシュしているステートメントを返します。
     * 
     * @return キャッシュしているステートメント
     */
    Statement getStatement();

    /**
     * キャッシュしている準備されたステートメントを返します。
     * 
     * @param sql
     *            SQL
     * 
     * @return キャッシュしている準備されたステートメント
     */
    PreparedStatement getPreparedStatement(String sql);

    /**
     * 準備されたステートメントを使用します。
     * <p>
     * ステートメントはもしキャッシュにあれば取り出され、なければ新たに作成されます。 ステートメントは使用後にキャッシュに追加されます。
     * </p>
     * 
     * @param <T>
     *            戻り値の型
     * @param sql
     *            SQL
     * @param handler
     *            準備されたステートメントを使用するハンドラ
     * @return ハンドラの戻り値
     */
    <T> T usingPreparedStatement(String sql,
            StatementHandler<T, ? super PreparedStatement> handler);

    /**
     * キャッシュしている準備されたステートメントを返します。
     * 
     * @param sql
     *            SQL
     * @param autoGeneratedKeys
     *            自動生成キーを返すかどうかを示すフラグ。{@link Statement#RETURN_GENERATED_KEYS}または
     *            {@link Statement#NO_GENERATED_KEYS}
     * 
     * @return キャッシュしている準備されたステートメント
     */
    PreparedStatement getPreparedStatement(String sql, int autoGeneratedKeys);

    /**
     * 準備されたステートメントを使用します。
     * <p>
     * ステートメントはもしキャッシュにあれば取り出され、なければ新たに作成されます。 ステートメントは使用後にキャッシュに追加されます。
     * </p>
     * 
     * @param <T>
     *            戻り値の型
     * @param sql
     *            SQL
     * @param autoGeneratedKeys
     *            自動生成キーを返すかどうかを示すフラグ。{@link Statement#RETURN_GENERATED_KEYS}または
     *            {@link Statement#NO_GENERATED_KEYS}
     * @param handler
     *            準備されたステートメントを使用するハンドラ
     * @return ハンドラの戻り値
     */
    <T> T usingPreparedStatement(String sql, int autoGeneratedKeys,
            StatementHandler<T, ? super PreparedStatement> handler);

    /**
     * キャッシュしているカーソル用の準備されたステートメントを返します。
     * 
     * @param sql
     *            SQL
     * 
     * @return キャッシュしている準備されたステートメント
     */
    PreparedStatement getCursorPreparedStatement(String sql);

    /**
     * カーソル用の準備されたステートメントを使用します。
     * <p>
     * ステートメントはもしキャッシュにあれば取り出され、なければ新たに作成されます。 ステートメントは使用後にキャッシュに追加されます。
     * </p>
     * 
     * @param <T>
     *            戻り値の型
     * @param sql
     *            SQL
     * @param handler
     *            準備されたカーソル用のステートメントを使用するハンドラ
     * @return ハンドラの戻り値
     */
    <T> T usingCursorPreparedStatement(String sql,
            StatementHandler<T, ? super PreparedStatement> handler);

    /**
     * 呼び出し可能なステートメントを返します。
     * <p>
     * ステートメントはもしキャッシュにあれば取り出され、なければ新たに作成されます。 ステートメントは使用後にキャッシュに追加されます。
     * </p>
     * 
     * @param sql
     *            SQL
     * 
     * @return キャッシュしている呼び出し可能なステートメント
     */
    CallableStatement getCallableStatement(String sql);

    /**
     * 呼び出し可能なステートメントを使用します。
     * <p>
     * ステートメントはもしキャッシュにあれば取り出され、なければ新たに作成されます。 ステートメントは使用後にキャッシュに追加されます。
     * </p>
     * 
     * @param <T>
     *            戻り値の型
     * @param sql
     *            SQL
     * @param handler
     *            準備された呼び出し可能なステートメントを使用するハンドラ
     * @return ハンドラの戻り値
     */
    <T> T usingCallableStatement(String sql,
            StatementHandler<T, ? super CallableStatement> handler);

}
