﻿using CleanAuLait.Core.Log;
using CleanAuLait.ObservableCollectionsMod;
using NLog;
using Prism.Navigation;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using System.ComponentModel;
using System.Diagnostics;
using System.Reactive.Disposables;
using TestNarou3.Adaptor.Boundary.Controller;
using TestNarou3.Adaptor.Boundary.Gateway.ViewModel;
using TestNarou3.Adaptor.Boundary.Gateway.ViewModel.Child;
using TestNarou3.Adaptor.Translator;
using TestNarou3.Domain.Model.Entity;
using TestNarou3.Domain.Model.Entity.Child;

namespace TestNarou3.Adaptor.Gateway.ViewModel
{
    internal class BookmarkDetailListViewModel
            : IBookmarkDetailListViewModel, INotifyPropertyChanged, IDestructible, IDisposable
    {
        private static readonly ILogger logger = LogManager.GetCurrentClassLogger();

#pragma warning disable CS0067
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore CS0067

        public ReactiveProperty<IBookmarkDetailListRowViewModel> SelectedItem { get; }

        public INotifyCollectionChangedListSynchronizedSingleView<
                BookmarkDetailListRow, IBookmarkDetailListRowViewModel> Rows { get; private set; } = null;
        private ISortableSynchronizedCoupleView<
                BookmarkDetailListRow, string, IBookmarkDetailListRowViewModel> SortableRows { get; set; }

        public ReactiveCommand CommandOpenBookmark { get; }

        public BookmarkDetailList Source { get; private set; }

        private IAppWindowController WindowController { get; }

        private IBookmarkDetailListRowViewModel selectedRow;
        private readonly IBookmarkDetailListRowViewModelTranslator vmTranslator;

        private readonly CompositeDisposable disposables = new();
        private bool disposedValue;

        public BookmarkDetailListViewModel(
                IAppWindowController wc,
                IBookmarkDetailListRowViewModelTranslator vmTranslator)
        {
            this.WindowController = wc;
            this.vmTranslator = vmTranslator;

            this.SelectedItem = new ReactiveProperty<IBookmarkDetailListRowViewModel>().AddTo(disposables);
            this.SelectedItem.Subscribe(OnSelectedItemChanged).AddTo(disposables);

            this.CommandOpenBookmark = new ReactiveCommand()
                    .WithSubscribe(OpenBookmark).AddTo(disposables);
        }

        #region event handler

        private void OnSelectedItemChanged(IBookmarkDetailListRowViewModel selectedRow)
        {
            this.selectedRow = selectedRow;
        }

        private void OpenBookmark()
        {
            logger.Trace("OpenBookmark: {0}", this.selectedRow?.NCode ?? "item not selected");

            if (this.selectedRow == null)
            {
                return;
            }

            // TODO Service

            string url = "https://ncode.syosetu.com/" + this.selectedRow.NCode.ToLower();
            Process.Start(new ProcessStartInfo(url) { UseShellExecute = true });
        }

        #endregion

        #region vm operation

        public void SynchronizeWith(BookmarkDetailList source)
        {
            if (this.Source == null)
            {
                this.SortableRows = source.Rows
                        .ToSortableSynchronizedCoupleView(
                                r => r.NCode, r => this.vmTranslator.Translate(r, this),
                                new DetailListViewComparer<string>(r => r.Title.Value));

                this.Rows = this.SortableRows
                        .ToSynchronizedSingleView()
                        .WithINotifyCollectionChangedList()
                        .AddTo(disposables);

                logger.Trace("BookmarkDetailListViewModel.SynchronizeWith: source [{0}], collection view [{1}]",
                        source.ToHashString(), this.Rows.GetType().FullName);

                this.Source = source;
            }
            else
            {
                logger.Trace("BookmarkDetailListViewModel.SynchronizeWith: viewmodel already synchronized to model [{0}]",
                        this.Source.ToHashString());
            }
        }

        public void SortByViewColumn<TViewColumn>(
                string tag, Func<IBookmarkDetailListRowViewModel, TViewColumn> selector, bool ascending)
        {
            logger.Trace("Sort: {0}, ascending = {1}", tag, ascending);
            this.SortableRows.Sort(this.SortableRows.CreateViewComparer(selector, ascending));
        }

        public void SortByValueColumn<TValueColumn>(
                string tag, Func<BookmarkDetailListRow, TValueColumn> selector, bool ascending)
        {
            logger.Trace("Sort: {0}, ascending = {1}", tag, ascending);
            this.SortableRows.Sort(this.SortableRows.CreateValueComparer(selector, ascending));
        }

        private class DetailListViewComparer<TViewColumn>
                : SortableSynchronizedCoupleViewViewComparer<
                        string, BookmarkDetailListRow, IBookmarkDetailListRowViewModel, TViewColumn>
        {
            public DetailListViewComparer(
                    Func<IBookmarkDetailListRowViewModel, TViewColumn> selector, bool ascending = true
            ) : base(selector, ascending)
            {
            }
        }

        private class DetailListValueComparer<TValueColumn>
                : SortableSynchronizedCoupleViewValueComparer<
                        string, BookmarkDetailListRow, IBookmarkDetailListRowViewModel, TValueColumn>
        {
            public DetailListValueComparer(
                    Func<BookmarkDetailListRow, TValueColumn> selector, bool ascending = true
            ) : base(selector, ascending)
            {
            }
        }

        #endregion

        #region IDestructible, IDisposable

        public void Destroy()
        {
            this.Dispose();
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    this.disposables.Dispose();

                    logger.Trace("BookmarkDetailListViewModel disposed.");
                }

                disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }

        #endregion
    }
}
