/**************************************************************************
 Moenizer - Allow to set background image for OmegaT.
 
 Copyright (C) 2013 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.plugin.moenizer;

import java.io.File;
import java.io.FileFilter;
import java.net.URL;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.omegat.util.FileUtil;
import org.omegat.util.Log;

/**
 *
 * @author Yu-Tang
 */
public class MoeConfig {
    private static final String CONFIG_FILE_NAME = "moenizer.conf";
    private static final String SCRIPT_FILE_NAME = "moenizer.groovy";
    
    private static final float DEFAULT_OPACITY = 0.5f;
    private static final int MAX_OPACITY_IN_PERCENTAGE = 100;  // Opaque
    private static final int MIN_OPACITY_IN_PERCENTAGE = 0;  // Transparent

    private static final long DEFAULT_INTERVAL = 60L * 1000L; // 1 minute in milliseconds
    private static final long MAX_INTERVAL = 86400L * 1000L;  // 24 hours
    private static final long MIN_INTERVAL = 3L * 1000L;      // 3 seconds

    private static final Pattern RE_LINEBREAK = Pattern.compile("[\\n\\r]+");
    private static final Pattern RE_OPACITY = Pattern.compile("(\\d+)\\s*%$");
    private static final Pattern RE_INTERVAL = Pattern.compile("(\\d+)\\s*([msh])", Pattern.CASE_INSENSITIVE);
    private static final Pattern RE_SCRIPT = Pattern.compile("(?:/|\\\\\\\\[^\\\\]|[a-zA-Z]:\\\\).+\\.groovy");
    private static final Pattern RE_SOURCE = Pattern.compile("(?:http://|https://|/|\\\\\\\\[^\\\\]|[a-zA-Z]:\\\\).+");
    private static final Pattern RE_HAS_IMAGE_EXTENSION = Pattern.compile(".+\\.(?:bmp|png|jpg|jpeg|gif)");
    private static final Pattern RE_HAS_INTERNET_SHORTCUT_EXTENSION = Pattern.compile(".+\\.(?:url|website|webloc)");
    private static final Pattern RE_HTTP_OR_HTTPS_PROTOCOL = Pattern.compile("(?:http://|https://).+");
    private static final Pattern RE_SOURCE_IN_PLUGIN_DIR = Pattern.compile(".+\\.(?:bmp|png|jpg|jpeg|gif|url|website|webloc)");
    
    private static ConfigData data;
    private static HashSet<Object> sources;
    
    static {
        // load config from file
        ConfigData cd = loadConfig();

        // flatten source dir to files (images only)
        HashSet<Object> flattenSources = getFlattenSources(cd.source);
        
        // scan plugin dir if both sources and script are empty 
        if (flattenSources.isEmpty() && cd.script.isEmpty()) {
            cd.script = getScriptInPluginDir();
            flattenSources = getSourcesInPluginDir();
        }

        // set default interval if current interval is 0 and has multiple sources
        if (cd.interval == 0 && flattenSources.size() > 1) {
            cd.interval = DEFAULT_INTERVAL;
        }
        
        data = cd;
        sources = flattenSources;
    }
    
    public static long getInterval() {
        return data.interval;
    }
    
	public static float getOpacity() {
        return data.opacity;
    }
	    
	public static File getScript() {
        return data.script.isEmpty() ? null : new File(data.script);
    }
	
    public static Object[] getSources() {
        return sources.toArray();
    }


    private static ConfigData loadConfig() {
        ConfigData data = getDefaultConfig();
        try {
            File file = new File(MoeUtil.getPluginJarDir(), CONFIG_FILE_NAME);
            if ( file.isFile() ) {
                String text = FileUtil.readTextFile(file);
                String[] lines = RE_LINEBREAK.split(text);
                for (String line: lines) {
                    if ( ! line.isEmpty() ) {
                        if (setOpacity(data, line) == true) {
                            // probably opacity value is changed from default
                        } else if (setInterval(data, line) == true) {
                            // probably interval value is changed from default
                        } else if (setScript(data, line) == true) {
                            // script URI
                        } else if (addSource(data, line) == true) {
                            // source URI (image)
                        } else {
                            // ignore
                        }
                    }
                }
            } else {
            }
        } catch (Exception ex) {
            Log.log(ex);
        }
        return data;
    }
    
    private static ConfigData getDefaultConfig() {
        ConfigData cd = new ConfigData();
        cd.opacity = DEFAULT_OPACITY;
        cd.interval = 0L;
        cd.script = "";
        cd.source = new HashSet<String>(); // should be unique
        return cd;
    }
    
    /**
     * set new opacity value to ConfigData
     * @param data
     * @param line
     * @return true if line contains opacity value
     */
    private static boolean setOpacity(ConfigData data, String line) {
        Matcher matcher = RE_OPACITY.matcher(line);
        if ( matcher.matches() ) {
            String num = matcher.group(1);
            int i = Integer.parseInt(num); // in percent
            if (i > MAX_OPACITY_IN_PERCENTAGE) {
                data.opacity = MAX_OPACITY_IN_PERCENTAGE;
            } else if (i < MIN_OPACITY_IN_PERCENTAGE) {
                data.opacity = MIN_OPACITY_IN_PERCENTAGE;
            } else {
                float f = ((float) i / MAX_OPACITY_IN_PERCENTAGE);
                data.opacity = f;
            }
            return true;
        }
        return false;
    }

    /**
     * set new interval value to ConfigData
     * @param data
     * @param line
     * @return true if line contains interval value
     */
    private static boolean setInterval(ConfigData data, String line) {
        Matcher matcher = RE_INTERVAL.matcher(line);
        if ( matcher.matches() ) {
            String num = matcher.group(1);
            String unit = matcher.group(2).toLowerCase(); // s | m | h
            
            long i = Long.parseLong(num);
            if (unit.equals("s")) {
                i = i * 1000;
            } else if (unit.equals("m")) {
                i = i * 60 * 1000;
            } else if (unit.equals("h")) {
                i = i * 60 * 60 * 1000;
            }
            
            // validate range
            if (i > MAX_INTERVAL) {
                data.interval = MAX_INTERVAL;
            } else if (i < MIN_INTERVAL) {
                data.interval = MIN_INTERVAL;
            } else {
                data.interval = i;
            }
            return true;
        }
        return false;
    }

    /**
     * set new script URI to ConfigData
     * @param data
     * @param line
     * @return true if line contains script URI
     */
    private static boolean setScript(ConfigData data, String line) {
        Matcher matcher = RE_SCRIPT.matcher(line);
        if ( matcher.matches() ) {
            data.script = line;
            return true;
        }
        return false;
    }

    /**
     * add new source URI to ConfigData
     * @param data
     * @param line
     * @return true if line contains source URI
     */
    private static boolean addSource(ConfigData data, String line) {
        Matcher matcher = RE_SOURCE.matcher(line);
        if ( matcher.matches() ) {
            data.source.add(line);
            return true;
        }
        return false;
    }

    /**
     * 
     * @param source
     * @return URL object or File object (file only)
     */
    private static HashSet<Object> getFlattenSources(Set<String> source) {
        HashSet<Object> newSource = new HashSet<Object>(); // should be unique
        for (String s: source) {
            try {
                if (RE_HTTP_OR_HTTPS_PROTOCOL.matcher(s.toLowerCase()).matches()) {
                    newSource.add(new URL(s));
                } else {
                    File file = new File(s);
                    // Single file and image file ?
                    if (file.isFile() && RE_HAS_IMAGE_EXTENSION.matcher(s.toLowerCase()).matches()) {
                        newSource.add(file);
                    } else if (file.isDirectory()) {
                        File[] listFiles = file.listFiles(new FileFilter() {
                            @Override
                            public boolean accept(File f) {
                                if (f.isFile()) {
                                    String fileName = f.getName().toLowerCase();
                                    if (RE_HAS_IMAGE_EXTENSION.matcher(fileName).matches() || 
                                            RE_HAS_INTERNET_SHORTCUT_EXTENSION.matcher(fileName).matches()) {
                                        return true;
                                    }
                                }
                                return false;
                            }
                        });
                        for (File f: listFiles) {
                            String fileName = f.getName().toLowerCase();
                            // image file
                            if (RE_HAS_IMAGE_EXTENSION.matcher(fileName).matches()) {
                                newSource.add(file);
                            // internet shortcut
                            } else if (RE_HAS_INTERNET_SHORTCUT_EXTENSION.matcher(fileName).matches()) {
                                URL url = MoeUtil.getURL(file);
                                if (url != null) {
                                    newSource.add(url);
                                }
                            }
                        }
                    }
                }
            } catch (Exception ex) {
                Log.log(ex);
            }
        }
        return newSource;
    }
    
    private static String getScriptInPluginDir() {
        try {
            File file = new File(MoeUtil.getPluginJarDir(), SCRIPT_FILE_NAME);
            if (file.isFile()) {
                return file.getCanonicalPath();
            }
        } catch (Exception ex) { /* ignore */ }
        return "";
    }
    
    private static HashSet<Object> getSourcesInPluginDir() {
        HashSet<Object> newSource = new HashSet<Object>(); // should be unique
        try {
            File dir = MoeUtil.getPluginJarDir();
            File[] listFiles = dir.listFiles(new FileFilter() {
                @Override
                public boolean accept(File f) {
                    if (f.isFile()) {
                        String fileName = f.getName().toLowerCase();
                        if (RE_SOURCE_IN_PLUGIN_DIR.matcher(fileName).matches()) {
                            return true;
                        }
                    }
                    return false;
                }
            });
            for (File f: listFiles) {
                String fileName = f.getName().toLowerCase();
                // image file
                if (RE_HAS_IMAGE_EXTENSION.matcher(fileName).matches()) {
                    newSource.add(f);
                // internet shortcut
                } else if (RE_HAS_INTERNET_SHORTCUT_EXTENSION.matcher(fileName).matches()) {
                    URL url = MoeUtil.getURL(f);
                    if (url != null) {
                        newSource.add(url);
                    }
                }
            }
        } catch (Exception ex) { /* ignore */ }
        return newSource;
    }

    private static class ConfigData {
        protected float opacity; // 0.0f <--> 1.0f
        protected long interval;  // by milliseconds
        protected String script;
        protected Set<String> source;
    }
}

