/**************************************************************************
 Moenizer - Allow to set background image for OmegaT.
 
 Copyright (C) 2013 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.plugin.moenizer.imageloader;

import java.awt.image.BufferedImage;
import java.io.File;
import java.net.URL;
import java.util.regex.Pattern;
import javax.imageio.ImageIO;
import static org.omegat.plugin.moenizer.MoeConfig.*;
import org.omegat.util.Log;

/**
 * Default image loader.
 *
 * calling order is:
 * 1. availableNext() ... exit if returned false
 * 2. readyForNextImage()
 * 3. getNextInterval() ... skipped first time
 * 4. getNextImage()
 * ... loop back to step 1
 * 
 * @author Yu-Tang
 */
public class DefaultImageLoader implements IImageLoader {

    private static final Pattern RE_HAS_IMAGE_EXTENSION = Pattern.compile(".+\\.(?:bmp|png|jpg|jpeg|gif)");
    private long interval;
    private Object[] sources;
    private Object currentSource;
    private int sourceIndex;
    
    protected BufferedImage image;
    
    public DefaultImageLoader() {
        interval = getInterval();
        sources = getSources();
        currentSource = null;
        image = null;
        sourceIndex = -1;
    }
    
    /**
     * get if source is acceptable.
     * @return bool. 
     */
    public static boolean isAcceptable(Object source) {
        if (source instanceof URL) {
            return true;
        } else if (source instanceof File) {
            String name = ((File) source).getName().toLowerCase();
            if (RE_HAS_IMAGE_EXTENSION.matcher(name).matches()) {
                return true;
            }
        }
        return false;
    }
    
    @Override
    public boolean availableNext() {
        boolean ret = false;
        if (sources.length == 0) {
            // it's over!
        } else if (sources.length == 1) {
            if (currentSource == null || interval > 0) { // first time or refresh
                currentSource = sources[0];
                ret = true;
            }
        } else {
            currentSource = getCircularNextSource();
            ret = true;
        }
        return ret;
    }
    
    /**
     * get next image.
     * @return BufferedImage object. Can be null if image is not available.
     */
    @Override
    public BufferedImage getNextImage() {
        return image;
    }

    /**
     * Do anything should be done for the next image. 
     * Calling before getNextImage().
     */
    @Override
    public void readyForNextImage() {
        //TODO use cache
        image = null;
        try {
            if (currentSource != null) {
                if (currentSource instanceof File) {
                    image = ImageIO.read((File) currentSource);
                } else if (currentSource instanceof URL) {
                    image = ImageIO.read((URL) currentSource);
                } else {
                    removeCurrentSource();
                }
            }
        } catch (Exception ex) {
            Log.log(ex);
            removeCurrentSource();
        }
    }

    /**
     * get next interval.
     * This method is not called at first time loop.
     * @return long by milliseconds
     */
    @Override
    public long getNextInterval() {
        return interval;
    }

    private Object getCircularNextSource() {
        sourceIndex++;
        if (sourceIndex >= sources.length) {
            sourceIndex = 0;
        }
        return sources[sourceIndex];
    }
    
    private void removeCurrentSource() {
        if (sourceIndex >= 0 && sourceIndex < sources.length) {
            sourceIndex = 0; // something is wrong. rewind index.
        } else {
            Object[] newSources = new Object[sources.length - 1];
            for (int i = 0; i < sources.length; i++) {
                if ( i == sourceIndex ) {
                    // skip
                } else if ( i < sourceIndex ) {
                    newSources[i] = sources[i];
                } else {
                    newSources[i - 1] = sources[i];
                }
            }
            sources = newSources;
            if (sourceIndex >= sources.length) {
                sourceIndex = 0;
            }
        }
    }

}
