package hayashi.yuu.tools.logger;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.logging.LogManager;
import java.util.logging.Logger;

/**
 * ロギングファイルに動作ログを出力する簡単なモデル
 * "log"+日時+".log"ファイルに出力される。
 * 利用例：
 * (1) インスタンスを取得する。
 * 		Logger logger = LoggerFactory.getInstance();
 * (2) ログ出力例
 * 		logger.finest("[finest] 詳細レベル（高）");
 * 		logger.finer("[finer] 詳細レベル（中）");
 * 		logger.fine("[fine] 詳細レベル（小）");
 * 		logger.config("[config] 設定");
 * 		logger.info("[info] 情報");
 * 		logger.warning("[warning] 警告");
 * 		logger.severe("[severe] 致命的");
 * @author yuu
 * @version 2010/02/07
 * @since 2010/02/07
 */
public abstract class LoggerFactory 
{
	public static void main(String[] args) {
		/*
		 * （設定例）
		 * 		handlers=java.util.logging.ConsoleHandler, java.util.logging.FileHandler
		 * 		.level=FINEST
		 * 
		 * 		java.util.logging.ConsoleHandler.level=FINEST
		 * 		java.util.logging.ConsoleHandler.formatter=hayashi.yuu.tools.logger.YuuLogFormatter
		 * 
		 * 		java.util.logging.FileHandler.level=WARNING
		 * 		java.util.logging.FileHandler.pattern=SampleLogging%u.%g.log
		 * 		java.util.logging.FileHandler.formatter=hayashi.yuu.tools.logger.YuuLogFormatter
		 * 		java.util.logging.FileHandler.count=10
		 * 		
		 * 
		 * 標準設定時でのログ出力。
		 * info、warning、severeの３つのレベルのみ標準エラー出力に出力されます。
		 * また、同時にファイルへも出力します。
		 * 出力先ファイルは「Logging%u.%g.txt」。ログファイルは10個でローテーションする。
		 * 
		 * 情報: [info] 情報
		 * 警告: [warning] 警告
		 * 致命的: [severe] 致命的
		 */
		Logger logger = LoggerFactory.getInstance();
		logger.finest("[finest] 詳細レベル（高）");
		logger.finer("[finer] 詳細レベル（中）");
		logger.fine("[fine] 詳細レベル（小）");
		logger.config("[config] 設定");
		logger.info("[info] 情報");
		logger.warning("[warning] 警告");
		logger.severe("[severe] 致命的");
	}
	
    public static Logger logger;
    
    /**
     * 簡単な標準ロガーを得る
     * @return	標準ロガー
     */
    public static Logger getInstance() {
    	if (logger == null) {
            //FileHandler handler = new FileHandler("GuardixMonitor.log", 1000, 3);	// ハンドラオブジェクトの生成
            //handler.setFormatter(new SimpleFormatter());		// 出力フォーマットの指定
            logger = Logger.getLogger("log");		// Loggerオブジェクトの生成
            //GuardixMonitor.logger.addHandler(handler);		// Loggerオブジェクトにハンドラを追加
    	}
		return logger;
    }
	
    /**
	 * ログ設定プロパティファイルのファイル内容
	 * 
	 */
	protected static final String LOGGING_PROPERTIES_DATA
	    = "handlers=java.util.logging.ConsoleHandler, java.util.logging.FileHandler\n"
	    + ".level=INFO\n"
	    + "java.util.logging.ConsoleHandler.level=FINEST\n"
	    + "java.util.logging.ConsoleHandler.formatter=hayashi.yuu.tools.logger.YuuLogFormatter\n"
	    + "java.util.logging.FileHandler.level=INFO\n"
	    + "java.util.logging.FileHandler.pattern=Logging%u.%g.txt\n"
	    + "java.util.logging.FileHandler.formatter=hayashi.yuu.tools.logger.YuuLogFormatter\n"
	    + "java.util.logging.FileHandler.count=10";

    
    /**
	 * static initializer によるログ設定の初期化
	 */
    static {
        final Logger logger = Logger.getLogger("SampleLogging");
        InputStream inStream = null;
        try {
            inStream = new ByteArrayInputStream(LOGGING_PROPERTIES_DATA.getBytes("UTF-8"));
            try {
                LogManager.getLogManager().readConfiguration(inStream);
                logger.config("ログ設定: LogManagerを設定しました。");
            }
            catch (IOException e) {
                logger.warning("ログ設定: LogManager設定の際に例外が発生しました。:" + e.toString());
            }
        }
        catch (UnsupportedEncodingException e) {
            logger.severe("ログ設定: UTF-8エンコーディングがサポートされていません。:" + e.toString());
        }
        finally {
            try {
                if (inStream != null) inStream.close();
            }
            catch (IOException e) {
                logger.warning("ログ設定: ログ設定プロパティファイルのストリームクローズ時に例外が発生しました。:" + e.toString());
            }
        } 
    }
}
