﻿using System;
using System.Diagnostics;

namespace Framework
{
    sealed class ExpectedType
    {
        delegate string ErrorMessageDelegate(string name, object val, string expectedName, object[] expecteds);

        readonly ErrorMessageDelegate errorMessage;

        ExpectedType(ErrorMessageDelegate errorMessage)
        {
            this.errorMessage = errorMessage;
        }

        /// <summary>
        /// エラー文字列を構築します。
        /// </summary>
        internal string ErrorMessage(string name, object val, string expectedName, object[] expecteds)
        {
            return errorMessage(name, val, expectedName, expecteds);
        }

        static string OriginalFormat(string expectedName, object expected, string name, object val)
        {
            var f = expectedName == null ? "expected value is {1}. but {2} is {3}."
                                         : "expected value({0}) is {1}. but {2} is {3}.";
            return string.Format(f, expectedName, expected, name, val);
        }

        static string BetweenAandB(object[] expecteds)
        {
            return "between " + expecteds[0] + " and " + expecteds[1];
        }

        static string[] AsStrArray(object[] objs)
        {
            return Array.ConvertAll(objs, src => (src ?? "null").ToString());
        }

        internal static readonly ExpectedType None = new ExpectedType(
            (name, val, _1, _2) => string.Format("{0} is {1}.", name, val));

        internal static readonly ExpectedType FromTo = new ExpectedType(
            (name, val, expectedName, expecteds) =>
                OriginalFormat(expectedName, BetweenAandB(expecteds), name, val));

        internal static readonly ExpectedType Any = new ExpectedType(
            (name, val, expectedName, expecteds) =>
            {
                if (expecteds.Length == 0)
                    return OriginalFormat(expectedName, "nothing", name, val);
                if (expecteds.Length == 1)
                    return OriginalFormat(expectedName, expecteds[0], name, val);

                var joinedExpected = string.Join(", ", AsStrArray(expecteds));
                var expectedStr = "one of { " + joinedExpected + " }";
                return OriginalFormat(expectedName, expectedStr, name, val);
            });

        internal static readonly ExpectedType GreaterThan = new ExpectedType(
            (name, val, expectedName, expecteds) =>
                OriginalFormat(expectedName, "greater than " + expecteds[0], name, val));

        internal static readonly ExpectedType GreaterThanEq = new ExpectedType(
            (name, val, expectedName, expecteds) =>
                OriginalFormat(expectedName, "greater than " + expecteds[0] + " or equals", name, val));

        internal static readonly ExpectedType LessThan = new ExpectedType(
            (name, val, expectedName, expecteds) =>
                OriginalFormat(expectedName, "less than " + expecteds[0], name, val));

        internal static readonly ExpectedType LessThanEq = new ExpectedType(
            (name, val, expectedName, expecteds) =>
                OriginalFormat(expectedName, "less than " + expecteds[0] + " or equals", name, val));

        internal static readonly ExpectedType NotNull = new ExpectedType(
            (name, _1, _2, _3) => string.Format("expected value is not null. but {0} is null.", name));

        internal static readonly ExpectedType Unexpected = new ExpectedType(
            (name, val, _, unexpecteds) => string.Format("unexpected {0}. but {1} is {2}.", unexpecteds[0], name, val));

        internal static readonly ExpectedType UnexpectedEq = new ExpectedType(
            (name, val, _, unexpecteds) => string.Format("unexpected {0}({2}). but {1} is {0}.", unexpecteds[0], name, val));

        internal static readonly ExpectedType Format = new ExpectedType(
            (name, val, format, _2) => string.Format("expected format is {2}. but {0} is {1}.", name, val, format));
    }
}
