﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web;
using System.Web.UI;

namespace Framework.Web
{
    /// <summary>
    /// Session管理用クラス
    /// </summary>
    public class SessionMgr
    {
        private SessionMgr()
        {
        }

        private static System.Web.SessionState.HttpSessionState _session
        {
            get { return HttpContext.Current.Session; }
        }

        /// <summary>
        /// Sessionに保存します。
        /// </summary>
        public static void Save(Page page, string key, object value)
        {
            SessionMgr.Save(SessionMgr.GetGroupName(page), key, value, false);
        }

        /// <summary>
        /// Sessionに保存します。
        /// </summary>
        /// <param name="groupName"></param>
        /// <param name="key"></param>
        /// <param name="value"></param>
        /// <param name="permanent">trueを指定した場合は、通常のRemoveでは削除されません</param>
        public static void Save(string groupName, string key, object value, bool permanent)
        {
            _session[SessionMgr.GetSessionKey(groupName, key, permanent)] = value;
        }

        /// <summary>
        /// 指定のキーでSessionに保存されたデータを返します。
        /// </summary>
        /// <param name="page"></param>
        /// <param name="key"></param>
        /// <param name="valueIfEmpty">セッションに指定のキーで保存されていない場合に、保存する値</param>
        /// <returns></returns>
        public static T Load<T>(Page page, string key, Func<T> valueIfEmpty)
        {
            return Load(SessionMgr.GetGroupName(page), key, false, valueIfEmpty);
        }
        
        /// <summary>
        /// 指定のキーでSessionに保存されたデータを返します。
        /// </summary>
        /// <param name="groupName"></param>
        /// <param name="key"></param>
        /// <param name="valueIfEmpty">セッションに指定のキーで保存されていない場合に、保存する値</param>
        /// <returns></returns>
        public static T Load<T>(string groupName, string key, bool permanent, Func<T> valueIfEmpty)
        {
            var obj = _session[SessionMgr.GetSessionKey(groupName, key, permanent)];

            if (obj == null)
            {
                if (valueIfEmpty != null)
                {
                    obj = valueIfEmpty();
                    SessionMgr.Save(groupName, key, obj, permanent);
                }
            }

            return (T)obj;
        }

        /// <summary>
        /// セッションから削除します。
        /// </summary>
        /// <param name="withPermanent"></param>
        public static void RemoveAll(bool withPermanent)
        {
            _session.Keys.AsEnumerableT<string>()
                .Where(t =>
                {
                    var b = true;

                    if (withPermanent == false)
                    {
                        if (t.EndsWith("#") == true)
                        {
                            b = false;
                        }
                    }

                    return b;
                })
                .ToArray()
                .Foreach(t => _session.Remove(t));
        }

        /// <summary>
        /// 指定ページのSessionを削除する
        /// </summary>
        public static void Remove(Page page)
        {
            SessionMgr.Remove(SessionMgr.GetGroupName(page), false);
        }

        /// <summary>
        /// 指定グループのSessionを削除する
        /// </summary>
        /// <param name="groupName"></param>
        public static void Remove(string groupName, bool withPermanent)
        {
            _session.Keys.AsEnumerableT<string>()
                .Where(t =>
                    {
                        var b = t.StartsWith(groupName + ".") == true;

                        if (b == true && withPermanent == false)
                        {
                            if (t.EndsWith("#") == true)
                            {
                                b = false;
                            }
                        }

                        return b;
                    })
                .ToArray()
                .Foreach(t => _session.Remove(t));
        }

        /// <summary>
        /// 指定のセッションを削除する
        /// </summary>
        /// <param name="sessionKey"></param>
        public static void Remove(string sessionKey)
        {
            _session.Remove(sessionKey);
        }

        public static string GetSessionKey(string groupName, string key, bool permanent)
        {
            var sessionKey = "{0}.{1}".Fmt(groupName, key);
            if (permanent == true)
            {
                sessionKey = sessionKey + "#";
            }
            return sessionKey;
        }

        public static string GetGroupName(Page page)
        {
            return SessionMgr.GetGroupName(page.GetType().BaseType);
        }

        public static string GetGroupName(Type type)
        {
            return type.FullName;
        }
    }
}
