﻿using System;
using System.ComponentModel;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Linq;
using System.Collections;

namespace Framework.Web.UI
{
    /// <summary>
    /// GridViewの機能拡張版。
    /// </summary>
    [DefaultProperty("Text")]
    [ToolboxData("<{0}:LGridView runat=server></{0}:LGridView>")]
    public class LGridView : GridView, ILocalizeControl, IMappingControl
    {
        /// <summary>
        /// 行のスタイル（CssClass）を各セルにコピーするか。デフォルトはfalse
        /// </summary>
        [DefaultValue("false")]
        public bool CopyRowCssClassToCell { get; set; }

        /// <summary>
        /// 0行データをデータバインドしたときにヘッダ行を表示するか。デフォルトはTrue
        /// </summary>
        [DefaultValue("true")]
        public bool ShowEmptyTable { get; set; }

        /// <summary>
        /// データソースをリクエストするときのキー
        /// </summary>
        public string DataSrcKey { get; set; }

        /// <summary>
        /// データソースのリクエストイベント
        /// </summary>
        public event LGridViewSrcRequestEventHandler DataSrcRequest;

        /// <summary>
        /// 編集行を返します。
        /// </summary>
        public GridViewRow EditRow
        {
            get { return (this.EditIndex < 0) ? null : this.Rows[this.EditIndex]; }
        }

        /// <summary>
        /// 選択行のデータキーの値を","で結合した文字列を返します。
        /// </summary>
        public string SelectedDataKeyValues
        {
            get { return (this.SelectedIndex < 0) ? null : string.Join(",", this.SelectedDataKey.Values.AsEnumerableT<DictionaryEntry>().Select(t => t.Value.ToString()).ToArray()); }
        }


        /// <summary>
        /// 依存コントロールのID。
        /// 現在指定できるのはMDropDownListだけです。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string ControlsToDepend { get; set; }
        private MDropDownList[] _dependDDLs;

        /// <summary>
        /// 依存コントロールのID。
        /// 現在指定できるのはLGridViewだけです。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string GridViewsToDepend { get; set; }
        private LGridView[] _dependGVWs;

        /// <summary>
        /// 依存コントロールのID。
        /// 現在指定できるのはMTextBoxだけです。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string TextBoxesToDepend { get; set; }
        private MTextBox[] _dependTXTs;

        /// <summary>
        /// 行追加用のフッターを表示するか？デフォルトはfalse。
        /// ※これをtrueにする場合は必ずLEditField列を追加すること。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("false")]
        public bool ShowInsertFooter { get; set; }

        /// <summary>
        /// キャプションのローカライズキー
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string CaptionLTextKey { get; set; }

        /// <summary>
        /// ヘッダ行
        /// </summary>
        public override GridViewRow HeaderRow
        {
            get
            {
                var row = base.HeaderRow;

                if (row == null && this.ShowEmptyTable && this.ShowHeader)
                {
                    //データ行がないときに無理やりヘッダーを表示させている場合は、その行を返す。
                    var tbl = this.FindControl("EmptyTable") as Table;
                    row = (GridViewRow)tbl.Rows[0];
                }

                return row;
            }
        }

        /// <summary>
        /// フッタ行
        /// </summary>
        public override GridViewRow FooterRow
        {
            get
            {
                var row = base.FooterRow;

                if (row == null && this.ShowEmptyTable && this.ShowFooter)
                {
                    //データ行がないときに無理やりフッターを表示させている場合は、その行を返す。
                    var tbl = this.FindControl("EmptyTable") as Table;
                    row = (GridViewRow)(this.ShowHeader ? tbl.Rows[1] : tbl.Rows[0]);
                }

                return row;
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LGridView()
        {
            this.CopyRowCssClassToCell = false;
            this.ShowEmptyTable = true;
            this.ShowInsertFooter = false;
        }

        /// <summary>
        /// CreateChildControls
        /// </summary>
        /// <param name="dataSource"></param>
        /// <param name="dataBinding"></param>
        /// <returns></returns>
        protected override int CreateChildControls(System.Collections.IEnumerable dataSource, bool dataBinding)
        {
            var rowNum = base.CreateChildControls(dataSource, dataBinding);

            if (rowNum == 0 && this.ShowEmptyTable == true && (base.ShowHeader == true || base.ShowFooter == true))
            {
                //データ行がない場合でも、ヘッダーとフッターを表示させる。

                Table table = new Table();
                table.ID = "EmptyTable";

                DataControlField[] fields = new DataControlField[this.Columns.Count];
                this.Columns.CopyTo(fields, 0);

                if (base.ShowHeader == true)
                {
                    GridViewRow row = base.CreateRow(-1, -1, DataControlRowType.Header, DataControlRowState.Normal);
                    this.InitializeRow(row, fields);
                    table.Rows.Add(row);
                }
                if (base.ShowFooter == true)
                {
                    GridViewRow row = base.CreateRow(-1, -1, DataControlRowType.Footer, DataControlRowState.Normal);
                    this.InitializeRow(row, fields);
                    table.Rows.Add(row);
                }


                this.Controls.Add(table);
            }

            return rowNum;
        }

        /// <summary>
        /// OnInit
        /// </summary>
        /// <param name="e"></param>
        protected override void OnInit(EventArgs e)
        {
            //依存コントロールがある場合は、依存コントロールのChangeイベントを受け取る。
            if (this.ControlsToDepend.IsNotEmpty())
            {
                _dependDDLs = this.ControlsToDepend.SplitByConma()
                    .Select(t => this.NamingContainer.FindControl(t.Trim()) as MDropDownList)
                    .ToArray();

                _dependDDLs.Foreach(ddl =>
                {
                    ddl.AutoPostBack = true;
                    ddl.SelectedIndexChanged += new EventHandler(dependDDL_SelectedIndexChanged);
                });
            }

            if (this.GridViewsToDepend.IsNotEmpty())
            {
                _dependGVWs = this.GridViewsToDepend.SplitByConma()
                    .Select(t => this.NamingContainer.FindControl(t.Trim()) as LGridView)
                    .ToArray();

                _dependGVWs.Foreach(gvw =>
                {
                    gvw.SelectedIndexChanged += new EventHandler(dependDDL_SelectedIndexChanged);
                });
            }

            //行追加用フッターを表示する場合は、フッタを表示する。
            if (this.ShowInsertFooter == true)
            {
                base.ShowFooter = true;
            }

            base.OnInit(e);
        }


        /// <summary>
        /// OnPreRender
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPreRender(EventArgs e)
        {
            if (this.CopyRowCssClassToCell)
            {
                for (int i = 0; i < base.Columns.Count; i++)
                {
                    var col = base.Columns[i];
                    if (base.HeaderStyle.CssClass.IsNotEmpty() && col.HeaderStyle.CssClass.IsEmpty())
                    {
                        col.HeaderStyle.CssClass = base.HeaderStyle.CssClass;
                    }

                    if (base.RowStyle.CssClass.IsNotEmpty() && col.ItemStyle.CssClass.IsEmpty())
                    {
                        col.ItemStyle.CssClass = base.RowStyle.CssClass;
                    }

                    if (base.ShowFooter && base.FooterStyle.CssClass.IsNotEmpty() && col.FooterStyle.CssClass.IsEmpty())
                    {
                        col.FooterStyle.CssClass = base.FooterStyle.CssClass;
                    }
                }
            }

            base.OnPreRender(e);
        }

        /// <summary>
        /// 依存しているDDLの選択が変更されたときに呼び出されるイベントです。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void dependDDL_SelectedIndexChanged(object sender, EventArgs e)
        {
            //リストソースを取得し、データバインド
            var arg = (string)null;
            if (sender is ListControl)
            {
                arg = ((ListControl)sender).SelectedValue;
            }
            else if (sender is LGridView)
            {
                arg = ((LGridView)sender).SelectedDataKeyValues;
            }
            this.DataSrcDataBind(arg);

            //自分が依存コントロールに指定されている可能性があるので、変更を通知する。
            this.OnSelectedIndexChanged(new EventArgs());
        }

        /// <summary>
        /// データソースをリクエストし、その戻りのデータをバインドする。
        /// </summary>
        public void DataSrcDataBind()
        {
            DataSrcDataBind(null);
        }

        /// <summary>
        /// データソースをリクエストし、その戻りのデータをバインドする。
        /// </summary>
        /// <param name="arg"></param>
        public void DataSrcDataBind(string arg)
        {
            if (this.DataSrcRequest == null) return;

            //データソースを取得し、データバインド
            var e = new LGridViewSrcRequestEventArgs()
            {
                DataSrcKey = this.DataSrcKey,
                Argument = arg
            };
            this.DataSrcRequest(this, e);

            base.SelectedIndex = e.SelectedIndex;
            base.EditIndex = e.EditIndex;
            base.DataSource = e.DataSource;
            base.DataBind();
        }

        #region ILocalizeControl メンバ

        public void Localize(System.Globalization.CultureInfo lang)
        {
            if (this.CaptionLTextKey.IsNotEmpty())
            {
                base.Caption = LTextMgr.GetText(lang, this.CaptionLTextKey);
            }
        }

        #endregion

        #region IMappingControl メンバ

        public string MappingName { get; set; }

        /// <summary>
        /// RequestMappingData
        /// </summary>
        /// <param name="data"></param>
        public void RequestMappingData(Framework.Data.MappingData data)
        {
            if (this.MappingName.IsEmpty() || base.DataKeyNames.Any() == false)
            {
                return;
            }

            //選択行のデータキーの値を返す
            var mappingNames = MappingName.SplitByConma();
            if (base.SelectedIndex < 0)
            {
                mappingNames.Foreach(t => data.Add(t, string.Empty));
            }
            else
            {
                for (int i = 0; i < mappingNames.Length && i < base.DataKeyNames.Length; i++)
                {
                    data.Add(mappingNames[i], this.DataKeys[base.SelectedIndex][i].ToString());
                }
            }
        }

        /// <summary>
        /// SetMappingData
        /// </summary>
        /// <param name="data"></param>
        public void SetMappingData(Framework.Data.MappingData data)
        {
            //依存するコントロールがある場合は、それをさきにマッピングする。
            string arg = null;
            if (_dependDDLs != null)
            {
                _dependDDLs.Foreach(ddl =>
                {
                    ddl.SetMappingData(data);
                    arg = ddl.SelectedValue;
                });
            }
            if (_dependGVWs != null)
            {
                _dependGVWs.Foreach(gvw =>
                {
                    gvw.SetMappingData(data);
                    arg = gvw.SelectedDataKeyValues;
                });
            }
            if (this.TextBoxesToDepend.IsNotEmpty())
            {
                this.TextBoxesToDepend.SplitByConma().Foreach(id =>
                {
                    var txt = this.Page.FindControl(id.Trim()) as MTextBox;
                    txt.SetMappingData(data);
                    arg = txt.Text;
                });
            }

            //データソースを取得し、データバインド
            this.DataSrcDataBind(arg);



            //データキーが一致する行を選択行にする。
            if (this.MappingName.IsNotEmpty() && base.DataKeyNames.Any())
            {
                var mappingNames = MappingName.SplitByConma();
                for (var rowIndex = 0; rowIndex < base.DataKeys.Count; rowIndex++)
                {
                    var dataKey = base.DataKeys[rowIndex];

                    bool match = true;
                    for (int i = 0; i < mappingNames.Length && i < base.DataKeyNames.Length; i++)
                    {
                        if (data[mappingNames[i]] != dataKey[i].ToString())
                        {
                            match = false;
                            break;
                        }
                    }

                    if (match)
                    {
                        base.SelectedIndex = rowIndex;
                        break;
                    }
                }
            }
        }

        #endregion
    }
}
