﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Framework.Web.UI
{
    public class PageNav : LLinkButton
    {
        private string _className;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PageNav()
        {
            _className = this.GetType().FullName;

            this.LTextKey = "{0}.{1}".Fmt(_className, "Back");
            this.BackMethod = UI.PageNavMethod.Redirect;
            this.IsRoot = false;
        }

        /// <summary>
        /// 前ページに戻るときの画面遷移の方法
        /// </summary>
        public PageNavMethod BackMethod { get; set; }

        /// <summary>
        /// 現在ページの履歴のインデックス
        /// </summary>
        private int CurrentHistoryIndex
        {
            get
            {
                return SessionMgr.Load<int>(_className, "CurrentHistoryIndex", true, () => -1);
            }
            set
            {
                SessionMgr.Save(_className, "CurrentHistoryIndex", value, true);
            }
        }

        /// <summary>
        /// 画面遷移履歴
        /// </summary>
        private List<PageNavInfo> History
        {
            get
            {
                return SessionMgr.Load<List<PageNavInfo>>(_className, "History", true, () => new List<PageNavInfo>());
            }
        }

        /// <summary>
        /// 画面遷移を履歴に残すかどうか
        /// </summary>
        private bool? SaveInHistory
        {
            get
            {
                return SessionMgr.Load<bool?>(_className, "SaveInHistory", false, null);
            }
            set
            {
                SessionMgr.Save(_className, "SaveInHistory", value, false);
            }
        }

        /// <summary>
        /// 画面遷移の起点となるページかどうか
        /// </summary>
        public bool IsRoot { get; set; }

        /// <summary>
        /// 戻り先ページを返します。ない場合はnullを返します。
        /// </summary>
        public PageNavInfo BackPage
        {
            get
            {
                var curIdx = this.CurrentHistoryIndex;
                if (curIdx >= 1)
                {
                    return this.History[curIdx - 1];
                }
                return null;
            }
        }

        /// <summary>
        /// 遷移前ページを返します。ない場合はnullを返します。
        /// </summary>
        public PageNavInfo PrevPage
        {
            get
            {
                var curIdx = this.CurrentHistoryIndex;
                var hist = this.History;
                if (hist.Count > curIdx + 1)
                {
                    return hist[curIdx + 1];
                }
                return null;
            }
        }

        /// <summary>
        /// OnInit
        /// </summary>
        /// <param name="e"></param>
        protected override void OnInit(EventArgs e)
        {
            if (this.Page.IsPostBack == false)
            {
                //許可された画面遷移かどうかの確認
                if (this.SaveInHistory == null && this.IsRoot == false)
                {
                    var msg = LTextMgr.GetText("{0}.{1}".Fmt(_className, "NotAllowedNavigation"));
                    throw new PageNavException(msg);
                }
                
                //起点ページの場合は、履歴をすべて削除
                if (this.IsRoot == true)
                {
                    this.RemoveHistory();
                }

                //現在ページを履歴に追加
                if (this.IsRoot == true || this.SaveInHistory == true) 
                {
                    var curIdx = this.CurrentHistoryIndex;
                    var hist = this.History;
                    if (curIdx < hist.Count - 1)
                    {
                        hist.RemoveRange(curIdx + 1, hist.Count - 1 - curIdx);
                    }
                    hist.Add(new PageNavInfo(this.Page.Request.Url));
                    this.CurrentHistoryIndex = curIdx + 1;
                }

                SessionMgr.Remove(SessionMgr.GetSessionKey(_className, "SaveInHistory", false));
            }

            //OnInit
            base.OnInit(e);
        }

        /// <summary>
        /// OnClick
        /// </summary>
        /// <param name="e"></param>
        protected override void OnClick(EventArgs e)
        {
            //戻る
            var backIdx = this.CurrentHistoryIndex - 1;
            this.CurrentHistoryIndex = backIdx;
            var url = this.History[backIdx].Url;
            this.Navigate(this.BackMethod, url, false);

            //OnClick
            base.OnClick(e);
        }

        /// <summary>
        /// OnPreRender
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPreRender(EventArgs e)
        {
            if (this.BackPage == null)
            {
                this.Enabled = false;
            }

            //OnPreRender
            base.OnPreRender(e);
        }

        /// <summary>
        /// Response.Redirectで画面遷移します。
        /// </summary>
        /// <param name="url"></param>
        public void Redirect(string url, bool saveInHistory)
        {
            this.SaveInHistory = saveInHistory;

            this.Page.Response.Redirect(url);
        }

        /// <summary>
        /// Server.Transferで画面遷移します。
        /// </summary>
        /// <param name="url"></param>
        public void Transfer(string url, bool saveInHistory)
        {
            this.SaveInHistory = saveInHistory;

            this.Page.Server.Transfer(url, true);
        }

        /// <summary>
        /// 画面遷移します
        /// </summary>
        /// <param name="method"></param>
        /// <param name="url"></param>
        /// <param name="saveInHistory"></param>
        public void Navigate(PageNavMethod method, string url, bool saveInHistory)
        {
            if (method == PageNavMethod.Redirect)
            {
                this.Redirect(url, saveInHistory);
            }
            else if (method == PageNavMethod.Transfer)
            {
                this.Transfer(url, saveInHistory);
            }
        }

        /// <summary>
        /// 画面遷移の履歴を削除します。
        /// </summary>
        public void RemoveHistory()
        {
            SessionMgr.Remove(_className, true);
        }
    }

    /// <summary>
    /// 前ページに戻るときの画面遷移の方法
    /// </summary>
    public enum PageNavMethod
    {
        /// <summary>
        /// Response.Redirect
        /// </summary>
        Redirect,

        /// <summary>
        /// Server.Transfer
        /// </summary>
        Transfer
    }

    /// <summary>
    /// 遷移画面情報
    /// </summary>
    public class PageNavInfo
    {
        public PageNavInfo(Uri uri)
        {
            this.Url = uri.PathAndQuery;
        }

        public string Url { get; private set; }
    }

    public class PageNavException : ApplicationException
    {
        public PageNavException(string msg) : base(msg)
        {
        }
    }
}
